/*!
 * UI development toolkit for HTML5 (OpenUI5)
 * (c) Copyright 2009-2025 SAP SE or an SAP affiliate company.
 * Licensed under the Apache License, Version 2.0 - see LICENSE.txt.
 */

sap.ui.define([],
	function() {
	"use strict";


	/**
	 * Calendar renderer.
	 * @namespace
	 */
	var CalendarRenderer = {
	};

	/**
	 * Renders the HTML for the given control, using the provided {@link sap.ui.core.RenderManager}.
	 *
	 * @param {sap.ui.core.RenderManager} oRm the RenderManager that can be used for writing to the render output buffer
	 * @param {sap.ui.unified.Calendar} oCal an object representation of the control that should be rendered
	 */
	CalendarRenderer.render = function(oRm, oCal){

		oCal._iMode = 0; // it's rendered always as DayPicker

		var sId = oCal.getId();
		var sTooltip = oCal.getTooltip_AsString();
		var aMonths = oCal.getAggregation("month");
		var sWidth = oCal.getWidth();

		oRm.write("<div");
		oRm.writeControlData(oCal);
		oRm.addClass("sapUiCal");
		if (aMonths.length > 1) {
			oRm.addClass("sapUiCalMulti");
		}
		// This makes the calendar focusable and therefore
		// the white empty areas can be clicked without closing the calendar
		// by accident.
		oRm.writeAttribute("tabindex", "-1");

		var rb = sap.ui.getCore().getLibraryResourceBundle("sap.ui.unified");
		var mAccProps = {labelledby: {value: "", append: false}}; // render on Month
		if (oCal._bPoupupMode) {
			mAccProps["role"] = "dialog";
		}
		oRm.writeAccessibilityState(oCal, mAccProps);

		if (sTooltip) {
			oRm.writeAttributeEscaped('title', sTooltip);
		}

		if (sWidth) {
			oRm.addClass("sapUiCalWidth");
			oRm.addStyle("width", sWidth);
			oRm.writeStyles();
		}

		if (oCal._getSecondaryCalendarType()) {
			oRm.addClass("sapUiCalSecType");
		}

		if (this.addAttributes) {
			// additional stuff by inherited controls
			this.addAttributes(oRm, oCal);
		}
		oRm.writeClasses();
		oRm.write(">"); // div element

		var oHeader = oCal.getAggregation("header");
		oRm.renderControl(oHeader);

		oRm.write("<div id=\"" + sId + "-content\" class=\"sapUiCalContent");
		// SAS Edit Start
		this.writeAdditionalStyleClasses(oRm, oCal);
		// SAS Edit End
		oRm.write("\">");
		for (var i = 0; i < aMonths.length; i++) {
			var oMonth = aMonths[i];
			oRm.renderControl(oMonth);
			//SAS Edit Start
			//HTMLCOMMONS-8634 - Add divider line for multiple months
			if (i < aMonths.length - 1) {
				oRm.write("<hr class=\"sasUiCalDivider\">");
			}
			//SAS Edit End
		}

		// SAS Edit Start
		this.writeAdditionalControls(oRm, oCal);
		// SAS Edit End


		this.renderCalContentOverlay(oRm, oCal, sId);

		if (!oCal._bNamesLengthChecked) {
			// render MonthPicker to check month names length
			var oMonthPicker = oCal.getAggregation("monthPicker");
			oRm.renderControl(oMonthPicker);
		}

		oRm.write("</div>");

		oRm.write("<button id=\"" + sId + "-cancel\" class=\"sapUiCalCancel\" tabindex=\"-1\">");
		oRm.write(rb.getText("CALENDAR_CANCEL"));
		oRm.write("</button>");

		// dummy element to catch tabbing in from next element
		oRm.write("<div id=\"" + sId + "-end\" tabindex=\"0\" style=\"width:0;height:0;position:absolute;right:0;bottom:0;\"></div>");

		this.renderCalContentAndArrowsOverlay(oRm, oCal, sId);

		oRm.write("</div>");
	};

	CalendarRenderer.renderCalContentOverlay = function(oRm, oCal, sId) {
		oRm.write("<div id=\"" + sId + "-contentOver\" class=\"sapUiCalContentOver\" style=\"display:none;\"></div>");
	};

	CalendarRenderer.renderCalContentAndArrowsOverlay = function(oRm, oCal, sId) {
	};

	// SAS Edit Start
	/**
	 * This method is reserved for derived class to add extra controls to Calendar.
	 *
	 * @param {sap.ui.core.RenderManager} oRm The RenderManager that can be used for writing to the render output buffer.
	 * @param {sap.ui.core.Control} oControl An object representation of the control that should be rendered.
	 */
	CalendarRenderer.writeAdditionalControls = function(oRm, oCal) {};

	/**
	 * This method is reserved for derived class to add style classes to Calendar content.
	 *
	 * @param {sap.ui.core.RenderManager} oRm The RenderManager that can be used for writing to the render output buffer.
	 * @param {sap.ui.core.Control} oControl An object representation of the control that should be rendered.
	 */
	CalendarRenderer.writeAdditionalStyleClasses = function(oRm, oCal) {};
	// SAS Edit End

	return CalendarRenderer;

}, /* bExport= */ true);
