/*!
 * UI development toolkit for HTML5 (OpenUI5)
 * (c) Copyright 2009-2025 SAP SE or an SAP affiliate company.
 * Licensed under the Apache License, Version 2.0 - see LICENSE.txt.
 */

// Provides utility class sap.ui.core.BusyIndicatorUtils
sap.ui.define(['jquery.sap.global'], //require of sap/ui/core/library not possible due to cyclic dependencies
	function(jQuery) {
	"use strict";

	// Static class

	/**
	 * @alias sap.ui.core.BusyIndicatorUtils
	 * @namespace
	 * @public
	 */
	var BusyIndicatorUtils = function() {};

	/**
	 * Returns the HTML content for the busy indicator
	 * styling + animation: LocalBusyIndicator.less
	 *
	 * @param {string} sSize either "Large" or "Medium". Other sizes will be mapped to "Medium"
	 * @param {string} sBusyIndicatorText text to be displayed below the busy indicator.
	 * 						It can be used to inform the user of the current operation.
	 * @returns {DOM.element} the element for the busy indicator
	 */
	BusyIndicatorUtils.getElement = function(sSize, sBusyIndicatorText /* SAS Edit */ ) {
		//default size is medium
		var sSizeClass = "sapUiLocalBusyIndicatorSizeMedium";

		//SAS Edit Start
		var oResBundle = sap.ui.getCore().getLibraryResourceBundle("sap.ui.core");
		var sTitle = sBusyIndicatorText || oResBundle.getText("BUSY_TEXT");

		if (sSize === "Large") {
			sSizeClass = "sapUiLocalBusyIndicatorSizeBig";
		}

		var oContainer = document.createElement("div");
		oContainer.className = "sapUiLocalBusyIndicator " + sSizeClass + " sapUiLocalBusyIndicatorFade";
		oContainer.setAttribute("tabIndex", "0");
		oContainer.setAttribute("aria-label", sTitle);
		oContainer.setAttribute("title", sTitle);

		var oParentElementsDiv = document.createElement("div");
		oParentElementsDiv.className = "sasBusyIndicatorElements";
		oParentElementsDiv.setAttribute("aria-busy", true);

		// determine automation size class
		var oAnimation = document.createElement("div");
		oAnimation.className = "sapUiLocalBusyIndicatorAnimation sapUiLocalBusyIndicatorAnimStandard";
		oAnimation.appendChild(document.createElement("div"));
		oAnimation.appendChild(document.createElement("div"));
		oAnimation.appendChild(document.createElement("div"));

		oParentElementsDiv.appendChild(oAnimation);

		oContainer.appendChild(oParentElementsDiv);

		if (sBusyIndicatorText && sBusyIndicatorText !== "") {
			var oLabel = document.createElement("label");
			oLabel.className = "sapMLabel sapMLabelMaxWidth sapUiSelectable";
			oLabel.setAttribute("style", "text-align:center");
			oLabel.innerHTML = sBusyIndicatorText;

			oContainer.appendChild(oLabel);
		}
		//SAS Edit End

		return oContainer;
	};

	/**
	 * Adds the BusyIndicator to the given control.
	 *
	 * @param {jQuery} $control a jQuery DOM instance to which the busy
	 *                     indicator is added
	 * @param {string} sBusyIndicatorId the actual DOM ID which will be used for
	 *                     the busy indicator content
	 * @param {sap.ui.core.BusyIndicatorSize} sSize either "Auto", "Large", "Medium" or "Small", determines the size of the
	 *                     indicator, default is "Medium"
	 * @param {string} sBusyIndicatorText text to be displayed below the busy indicator.
	 * 						It can be used to inform the user of the current operation.
	 * @returns {object} a jQuery object for the busy indicator
	 */
	BusyIndicatorUtils.addHTML = function ($control, sBusyIndicatorId, sSize, sBusyIndicatorText /* SAS Edit */) {
		//SAS Edit Start - add busy indicator text
		var oElement = BusyIndicatorUtils.getElement(sSize, sBusyIndicatorText),
			//SAS Edit End
			sSizeClass = "sapUiLocalBusyIndicatorAnimation sapUiLocalBusyIndicatorAnimStandard",
			oAnimation = oElement.children[0].children[0] || oElement.children[0];  /* SAS Edit - S1433199 */

		oElement.id = sBusyIndicatorId;

		var oDomRef = $control.get(0);
		oDomRef.appendChild(oElement);
		oDomRef.className += " sapUiLocalBusy";

		// handle animation size
		if (sSize === sap.ui.core.BusyIndicatorSize.Small) {
			sSizeClass = "sapUiLocalBusyIndicatorAnimation sapUiLocalBusyIndicatorAnimSmall";
		} else if (sSize === sap.ui.core.BusyIndicatorSize.Auto) {
			//set standard animation for width calculation
			oAnimation.className = sSizeClass;
			var iWidth = oAnimation.offsetWidth;

			// We can only determine the actual animation after the browser has
			// calculated the size of the indicator we need to know the pixel-size of
			// 3rem, under which the indicator will animate differently
			if ($control[0].offsetWidth < iWidth) {
				sSizeClass = "sapUiLocalBusyIndicatorAnimation sapUiLocalBusyIndicatorAnimSmall";
			}
		}
		oAnimation.className = sSizeClass;

		return jQuery(oElement);
	};

	/**
	 * Obsolete IE9 support, kept for some more time to avoid issues with custom controls,
	 * start/stop now are 'noop's.
	 */
	BusyIndicatorUtils.animateIE9 = {
		start: function () {},
		stop: function () {}
	};

	return BusyIndicatorUtils;

}, /* bExport= */ true);
