/*!
 * UI development toolkit for HTML5 (OpenUI5)
 * (c) Copyright 2009-2025 SAP SE or an SAP affiliate company.
 * Licensed under the Apache License, Version 2.0 - see LICENSE.txt.
 */

sap.ui.define(['sap/m/library'],
	function(library) {
	"use strict";

	// shortcut for sap.m.IconTabFilterDesign
	var IconTabFilterDesign = library.IconTabFilterDesign;

	/**
		 * HBox renderer.
		 * @namespace
		 */
	var IconTabHeaderRenderer = {
	};

	/**
	 * Renders the HTML for the given control, using the provided {@link sap.ui.core.RenderManager}.
	 *
	 * @param {sap.ui.core.RenderManager} oRM the RenderManager that can be used for writing to the render output buffer
	 * @param {sap.ui.core.Control} oControl an object representation of the control that should be rendered
	 */
	IconTabHeaderRenderer.render = function(oRM, oControl){
		// return immediately if control is not visible
		if (!oControl.getVisible()) {
			return;
		}

        // SAS Edit Start
        /**
         * For HTMLCOMMONS-8161: We have to assign aria-label for tabs according to movable property. So we create a
         * invisible text here that holds the description as "Fixed". This will be used by tabs when they are "Fixed".
         */
        this._addAriaLabelForFixedTab(oRM, oControl);
        // SAS Edit End

		var aItems = oControl.getItems(),
			iItemsCount = aItems.length,
			// SAS Edit Start
			//aVisibleTabFilters = oControl.getVisibleTabFilters(),
			//iVisibleTabFiltersCount = aVisibleTabFilters.length,
			// SAS Edit End
			iVisibleTabFilterIndex = 0,
			bTextOnly = oControl._checkTextOnly(aItems),
			bNoText = oControl._checkNoText(aItems),
			bInLine = oControl._checkInLine(aItems) || oControl.isInlineMode(),
			bShowOverflowSelectList = oControl.getShowOverflowSelectList(),
			oItem,
			bIsHorizontalDesign,
			bHasHorizontalDesign;

		// SAS Edit Start
		//var oIconTabBar = oControl.getParent();
		//var bUpperCase = oIconTabBar && oIconTabBar instanceof sap.m.IconTabBar && oIconTabBar.getUpperCase();
		var	bUpperCase = oControl._getUpperCase();
		// SAS Edit End

		// render wrapper div
		oRM.write("<div role='tablist' ");
		oRM.addClass("sapMITH");
		oRM.addClass("sapContrastPlus");
		oRM.addClass("sapMITHBackgroundDesign" + oControl.getBackgroundDesign());

		// SAS Edit Start
		this._addInnerHeaderClass(oRM, oControl);
		// SAS Edit Start

		if (bShowOverflowSelectList) {
			oRM.addClass("sapMITHOverflowList");
		}

		if (oControl._scrollable) {
			oRM.addClass("sapMITBScrollable");
			if (oControl._bPreviousScrollForward) {
				oRM.addClass("sapMITBScrollForward");
			} else {
				oRM.addClass("sapMITBNoScrollForward");
			}
			if (oControl._bPreviousScrollBack) {
				oRM.addClass("sapMITBScrollBack");
			} else {
				oRM.addClass("sapMITBNoScrollBack");
			}
		} else {
			oRM.addClass("sapMITBNotScrollable");
		}
		// Check for upperCase property on IconTabBar
		if (bUpperCase) {
			oRM.addClass("sapMITBTextUpperCase");
		}
		oRM.writeControlData(oControl);
		oRM.writeClasses();
		oRM.write(">");

		// render left scroll arrow
		// SAS Edit Start
		this._renderLeftScrollButton(oRM, oControl);
		// SAS Edit End

		// render scroll container on touch devices
        // SAS Edit Start
        //S1470098, this fix is copied from upcoming sdk 1.58
		//oRM.write("<div id='" + oControl.getId() + "-scrollContainer' aria-hidden='true' class='sapMITBScrollContainer'>");
        oRM.write("<div id='" + oControl.getId() + "-scrollContainer' class='sapMITBScrollContainer'>");

		//oRM.write("<div id='" + oControl.getId() + "-head' aria-hidden='true' ");
        oRM.write("<div id='" + oControl.getId() + "-head'");
        // SAS Edit End
		oRM.addClass("sapMITBHead");

		// SAS Edit Start
		this._addInnerHeadClass(oRM, oControl);
		// SAS Edit End

		if (bTextOnly) {
			oRM.addClass("sapMITBTextOnly");
		}

		if (bNoText) {
			oRM.addClass("sapMITBNoText");
		}

		if (bInLine) {
			oRM.addClass("sapMITBInLine");
		}

		oRM.writeClasses();
		oRM.write(">");

		for (var i = 0; i < iItemsCount; i++) {
			oItem = aItems[i];
			//SAS Edit Start
			//oItem.render(oRM, iVisibleTabFilterIndex, iVisibleTabFiltersCount);
			// since PropertyPane extends from IconTabFilter, renderControl need to be call so that onBeforeRendering and onAfterRendering of PropertyPane will get called
			// additional arguments iVisibleTabFilterIndex, iVisibleTabFiltersCount are now
			oItem._visibleIndex = iVisibleTabFilterIndex;
			if (oItem instanceof sap.m.IconTabSeparator) {
                oItem.render(oRM);
			} else {
                oRM.renderControl(oItem);
			}

			// for HTMLCOMMONS-8161: insert a separator between fixed and movable tabs:
			this._renderToolbarSeparator(oRM, oControl, oItem, i);
			// SAS Edit End

			if (oItem instanceof sap.m.IconTabFilter) {
				bIsHorizontalDesign = oItem.getDesign() === IconTabFilterDesign.Horizontal;
				if (bIsHorizontalDesign) {
					bHasHorizontalDesign = true;
				}

				if (oItem.getVisible()) {
					iVisibleTabFilterIndex++;
				}
			}
		}

		oRM.write("</div>");

		oRM.write("</div>"); //scrollContainer

		// render right scroll arrow
		// SAS Edit Start
		this._renderRightScrollButton(oRM, oControl);
		// render new tab button
		this._addTabButton(oRM, oControl);
		// render overflow button
		this._addOverflowButton(oRM, oControl);
		// render responsive toolbar at teh end of tab list
		this._addToolbar(oRM, oControl);
		// SAS Edit End

		// render overflow button
		if (bShowOverflowSelectList) {
			var oOverflowButton = oControl._getOverflowButton();
			if (bInLine) {
				oOverflowButton.addStyleClass('sapMBtnInline');
			} else if (bTextOnly) {
				oOverflowButton.addStyleClass('sapMBtnTextOnly');
			} else if (bNoText || bHasHorizontalDesign) {
				oOverflowButton.addStyleClass('sapMBtnNoText');
			}

			oRM.renderControl(oOverflowButton);
		}

		// end wrapper div
		oRM.write("</div>");
	};

	// SAS Edit Start
	IconTabHeaderRenderer._addInnerHeaderClass = function (oRM, oControl) {
	};

	IconTabHeaderRenderer._addInnerHeadClass = function (oRM, oControl) {
	};

	IconTabHeaderRenderer._addAriaLabelForFixedTab = function (oRM, oControl) {

	};

	IconTabHeaderRenderer._renderToolbarSeparator = function (oRM, oControl, oItem, iIndex) {

	};

	IconTabHeaderRenderer._renderLeftScrollButton = function (oRM, oControl) {
		oRM.renderControl(oControl._getScrollingArrow("left"));
	};

	IconTabHeaderRenderer._renderRightScrollButton = function (oRM, oControl) {
		oRM.renderControl(oControl._getScrollingArrow("right"));
	};

	IconTabHeaderRenderer._addTabButton = function (oRM, oControl) {

	};

	IconTabHeaderRenderer._addOverflowButton = function (oRM, oControl) {

	};

	IconTabHeaderRenderer._addToolbar = function (oRM, oControl) {

	};
	// SAS Edit End


	return IconTabHeaderRenderer;

}, /* bExport= */ true);
