// Copyright (c) 2013 - 2025, SAS Institute Inc., Cary, NC, USA. All Rights Reserved.
package pgupdate

import groovy.io.FileType
import groovy.lang.GroovyObject
import groovy.text.SimpleTemplateEngine

import java.io.File
import java.net.InetAddress
import java.util.regex.*
import java.text.SimpleDateFormat
import java.util.Properties

import org.apache.commons.io.*

public class UpdatePostgres {
    boolean continueProcessing = true
    boolean isTLSenabled = false
    def newline = System.getProperty("line.separator")
    def working_folder = System.getProperty("user.dir") + File.separator + "classes"
    String certFileName, keyFileName, caFileName = ""

    //used to call UpdateLogger.groovy
    File UpdateLoggerClassFile = new File(working_folder, "UpdateLogger.groovy")
    Class UpdateLoggerClass = new GroovyClassLoader(getClass().getClassLoader()).parseClass(UpdateLoggerClassFile)
    GroovyObject updatelogger = (GroovyObject) UpdateLoggerClass.newInstance()

    /**
     * Used to call into the custom logger class
     * @return returns a new instance of the logger class
     */
    def log(){
        return UpdateLoggerClass.newInstance()
    }

    /**
     * Used to execute system commands. Will fail is the RC is not 0
     * @param cmd | command we want to execute
     * @return rc (return code of the executed command)
     */
    def String launchCommandwEnv(cmd){
        if (continueProcessing){
            log().debug("Running the command $cmd")
            ProcessBuilder pb = new ProcessBuilder(cmd.split('  '))

            //set environment variables
            Map<String, String> env = pb.environment()
            env.put("LD_LIBRARY_PATH", System.properties['LD_LIBRARY_PATH'])
            env.put("LIBPATH", System.properties['LIBPATH'])
            env.put("PGUSER", System.properties['PGUSER'])
            env.put("PGPASSWORD", System.properties['PGPASSWORD'])
            env.put("PGPORT", System.properties['PGPORT'])

            pb.redirectErrorStream(true)
            //execute the command w/ parameters
            Process p = pb.start()
            //get output from the command
            InputStream stdout = p.getInputStream()
            BufferedReader reader = new BufferedReader (new InputStreamReader(stdout))
            def line = ""

            while ((line = reader.readLine()) != null){
                log().trace(line)
            }

            //print RC
            int rc = p.waitFor()
            log().debug("rc: " + rc)
            if(rc.toString() != "0"){
                log().error("Script has received a non-zero return code when executing $cmd. Upgrade process will be stopped.")
                System.exit(2)
            }
            return rc.toString()
        }
    }

    /**
     * Used to execute system commands. Will  fail is the RC is not 0. NOTE: This is specific to the windows start up process. We do not try to read the output from the process because it will hang and never return.
     * @param cmd | command we want to execute
     * @return rc (return code of the executed command)
     */
    def String launchCommandwEnvWin(cmd, cmdoutputfile){
        if (continueProcessing){
            log().debug("Running the command $cmd")

            //piping the output to this text file in order to get around hanging issue with the bufferedReader
            cmd = cmd + "  >  " + cmdoutputfile
            def cmdlist = cmd.split('  ')
            cmdlist.plus(0, '/c')
            cmdlist.plus(0, 'cmd')
            ProcessBuilder pb = new ProcessBuilder(cmdlist)

            //set environment variables
            Map<String, String> env = pb.environment()
            env.put("LD_LIBRARY_PATH", System.properties['LD_LIBRARY_PATH'])
            env.put("LIBPATH", System.properties['LIBPATH'])
            env.put("PGUSER", System.properties['PGUSER'])
            env.put("PGPASSWORD", System.properties['PGPASSWORD'])
            env.put("PGPORT", System.properties['PGPORT'])
            log().debug("PGPORT is " +  System.properties['PGPORT'])
            log().debug("LD_LIBRARY_PATH is " +  System.properties['LD_LIBRARY_PATH'])

            pb.redirectErrorStream(true)
            //execute the command w/ parameters
            Process p = pb.start()

            //get return code
            int rc = p.waitFor()

            //read output back in from file
            File outputfile = new File(cmdoutputfile)
            String console_output = outputfile.text

            if (console_output.contains("Permission denied")){
                log().error(console_output)
            }
            else{
                log().trace(console_output)
            }

            //remove temp file
            outputfile.delete()
            outputfile.deleteOnExit()

            log().debug("rc: " + rc)
            if(rc.toString() != "0"){
                log().error("Script has received a non-zero return code executing the command $cmd. Upgrade process will be stopped.")
                System.exit(2)
            }
            return rc.toString()
        }
    }

    /**
     * Used to execute system commands when we don't care about the output (only the rc). Will NOT fail is the RC is not 0
     * @param cmd | command we want to execute
     * @return
     */
    def String sendCommandwEnv(cmd, addLogEntry){
        if (continueProcessing){
            if (addLogEntry) {
                log().debug("Running the command $cmd")
            }

            ProcessBuilder pb = new ProcessBuilder(cmd.split('  '))

            //set environment variables
            Map<String, String> env = pb.environment()
            env.put("LD_LIBRARY_PATH", System.properties['LD_LIBRARY_PATH'])
            env.put("LIBPATH", System.properties['LIBPATH'])
            env.put("PGUSER", System.properties['PGUSER'])
            env.put("PGPASSWORD", System.properties['PGPASSWORD'])
            env.put("PGPORT", System.properties['PGPORT'])
            log().debug("PGPORT is " + System.properties['PGPORT'])
            log().debug("LD_LIBRARY_PATH is " + System.properties['LD_LIBRARY_PATH'])

            pb.redirectErrorStream(true)
            //execute the command w/ parameters
            Process p = pb.start()

            InputStream stdout = p.getInputStream()
            BufferedReader reader = new BufferedReader (new InputStreamReader(stdout))
            def line = ""
            while ((line = reader.readLine()) != null){
                log().trace(line)
            }

            //print RC
            int rc = p.waitFor();
            log().debug("rc: " + rc)
            return rc.toString()
        }
    }

    /**
     * Used to execute system commands when we care about the output
     * @param cmd | command we want to execute
     * @param withNewLines | Optional.  print new lines out in return string.
     * @return
     */

    def String sendCommandgetOutput(cmd, withNewLines = false){
        log().debug("Running the following command: " + cmd)
        ProcessBuilder pb = new ProcessBuilder(cmd.split('  '))

        Map<String, String> env = pb.environment()
        env.put("LD_LIBRARY_PATH", System.properties['LD_LIBRARY_PATH'])
        env.put("LIBPATH", System.properties['LIBPATH'])
        //These propreties will only be set in the context of Data Servers.
        //If not in the context of data servers, just let them be blank,
        //because they won't be used.
        env.put("PGUSER", System.properties['PGUSER'] ?: "")
        env.put("PGPASSWORD", System.properties['PGPASSWORD'] ?: "")
        env.put("PGPORT", System.properties['PGPORT'] ?: "")

        pb.redirectErrorStream(true)
        //execute the command w/ parameters
        Process p = pb.start()

        InputStream stdout = p.getInputStream()
        BufferedReader reader = new BufferedReader (new InputStreamReader(stdout))
        def line = ""
        def output = ""
        while ((line = reader.readLine()) != null){
            output = output + line + (withNewLines ? newline : "")
            log().trace(line)
        }
        int rc = p.waitFor();

        return output

    }

    /**
     * Starts the specified server and verifies that it is running prior to returning
     * @param script | location of the script we are starting
     * @param binloc | location of the corresponding bin directory for verification purposes
     * @return
     */
    def startServiceForCurrent(script, binloc){
        if (continueProcessing){
            log().info("Starting the Data Server...")
            if(determineOS() == "WIN"){
                launchCommandwEnvWin(script +"  start", "startService_output.temp")
            }
            else{
                launchCommandwEnv(script +"  start")
            }

            //check if server is running
            String checkconnection = binloc + File.separator + "psql  -d  postgres  -c  select version()"
            int rc = sendCommandwEnv(checkconnection, true)
            def timeout = 6
            while(rc != "0"){
                //wait for 5 secs before trying to connect again
                log().debug("Data Server is not yet accepting connections. Will retry in 5 seconds...")
                sleep(5000)
                rc = sendCommandwEnv(checkconnection, true)
                timeout = timeout - 1

                if (timeout == 0){
                    log().error("Unable to connect to the Data Server, check the Data Server log. Upgrade process will be stopped.")
                    System.exit(rc)
                }
            }
        }
    }

    /**
     * Starts the specified server and verifies that it is running prior to returning
     * @param script | location of the script we are starting
     * @param binloc | location of the corresponding bin directory for verification purposes 
     * @param isTLSenabled 
     * @return 
     */
    def startServiceForTarget(script, binloc, isTLSenabled){
        if (continueProcessing){
            log().info("Starting the Data Server...")
            if(determineOS() == "WIN"){
                launchCommandwEnvWin(script +"  start", "startService_output.temp")
            }
            else{
                launchCommandwEnv(script +"  start")
            }

            // Alter the newly created PGUSER with a new PG Authentication method if the server is TLS enabled
            //if(isTLSenabled){
            alterPGDataServerUser(binloc)
            //}

            //check if server is running
            String checkconnection = binloc + File.separator + "psql  -d  postgres  -c  select version()"
            int rc = sendCommandwEnv(checkconnection, true)
            def timeout = 6
            while(rc != "0"){
                //wait for 5 secs before trying to connect again
                log().debug("Data Server is not yet accepting connections. Will retry in 5 seconds...")
                sleep(5000)
                rc = sendCommandwEnv(checkconnection, true)
                timeout = timeout - 1

                if (timeout == 0){
                    log().error("Unable to connect to the Data Server, check the Data Server log. Upgrade process will be stopped.")
                    System.exit(rc)
                }
            }
        }
    }

    /**
     * Starts the specified server and verifies that it is running prior to returning
     * @param script | location of the script we are stopping
     * @return
     */
    def stopService(script){
        if (continueProcessing){
            if(new File(script).exists()){
                log().info("Stopping the Data Server...")
                launchCommandwEnv(script +"  stop")
            }
        }
    }

    /**
     * @param binloc | location of the corresponding bin directory for verification purposes
     * @return
     */
    def alterPGDataServerUser(binloc){
        if (continueProcessing){
            log().info("Altering the DataServer User credentials")
            String pguser = System.properties['PGUSER']
            String pguserpwd = System.properties['PGPASSWORD']
            String hostname = InetAddress.localHost.hostName.trim()
            String pgportnum =  System.properties['PGPORT']
            String alterpgusercmd = binloc + File.separator + "psql  -h  $hostname  -p  $pgportnum  -U  $pguser  -d  postgres  -c  alter user \"$pguser\" password '$pguserpwd'"
            //log().debug("alterpgusercmd is $alterpgusercmd.")
            int rc = sendCommandwEnv(alterpgusercmd, false)
            def timeout = 6
            while(rc != "0"){
                //wait for 5 secs before trying to connect again
                log().debug("Data Server is not yet accepting connections. Will retry in 5 seconds...")
                sleep(5000)
                rc = sendCommandwEnv(alterpgusercmd, false)
                timeout = timeout - 1

                if (timeout == 0){
                    log().error("Unable to alter the DataServer User credentials. Check the Data Server log file. Upgrade process will be stopped.")
                    System.exit(rc)
                }
            }
        }
    }


    /**
     * Copies a file from one location to another
     * @param source | file we are copying
     * @param dest | location we are copying to
     * @return
     */
    def copyFile(source, dest){
        if (continueProcessing){
            log().debug("Copying file $source to $dest")
            new File(dest) << new File(source).text
        }

    }

    /**
     * Comment the "host * * * *"" line from pg_hba.conf, if it exists.
     * Also, add the temporary "host * * * trust" line
     * @param fileName             | file to modify (pg_hba.conf)
     * @param lineToInsert         | line to temporarily add 
     * @param prepForMD5Conversion | true for disabling security, false for restoring 
     */
    static void modifyPGHBAconfigfile(String fileName, String lineToInsert, boolean prepForMD5Conversion) {
        // Read the file lines into a list
        File confFile = new File(fileName)
        String commentString = "# TEMP COMMENT FOR UPGRADE : "

        String stringToWrite = confFile.text
        if (prepForMD5Conversion){
            stringToWrite = stringToWrite.replaceAll(/(?im)^\s*(host.*(reject|md5|password|gss|sspi|krb5|ident|peer|pam|ldap|radius|cert)\s*$)/, commentString + '$1')
            stringToWrite = stringToWrite + "\n" + lineToInsert
        } else {
            stringToWrite = stringToWrite.replaceAll(/(?im)^${commentString}/, '')
            stringToWrite = stringToWrite.replaceAll(/(?im)^${lineToInsert}/, '')
        }
           
        confFile.write(stringToWrite)
    }

    /**
     * Copies a file from one location to another while replacing 9.4 with 9.4_PREVIOUS as well as updating the permission on the resulting file. Definition also updates the servertype for windows
     * @param source | file we are copying
     * @param dest | location we are copying to
     * @return
     */
    def updateExecutionScript(script, target, datadir, datadirback){
        if (continueProcessing){
            log().info("Updating $target")
            File file = new File(script)
            File newfile = new File(target)
            String line, databackupdir
            String minus_string = "SASWebInfrastructurePlatformDataServer" + File.separator + "9.4"
            String plus_string  = "SASWebInfrastructurePlatformDataServer" + File.separator + "9.4_PREVIOUS"
            //delete existing temp file if exists
            newfile.delete()
            File datadirbackfile = new File(datadirback)
            databackupdir = datadirbackfile.getName()

            def newline
            file.withReader { reader ->
                while ((line = reader.readLine())!=null) {
                    if (line.contains(minus_string)|| (line.contains("SERVERTYPE=services") && determineOS() == "WIN") || line.contains(datadir)) {
                        newline = line.replace(minus_string, plus_string).replace("SERVERTYPE=services", "SERVERTYPE=scripts")

                        if (determineOS() == "WIN") {
                            String windatadir = findDataDirFromBatchFile(script)
                            newline = newline.replace(windatadir, datadirback).replace("/data", "\\" + databackupdir)
                            newfile.append(newline + '\r\n')
                        } else {
                            newline = newline.replace(datadir, datadirback)
                            newfile.append(newline + '\n')
                        }

                    } else {
                        if (determineOS() == "WIN") {
                            newfile.append(line + '\r\n')
                        } else {
                            newfile.append(line + '\n')
                        }
                    }
                }
            }
            //update permissions on unix
            if(determineOS() == "UNIX"){
                String chmodcom = "chmod  775  " + target
                launchCommandwEnv(chmodcom)
            }
        }
    }

    def String findDataDirFromBatchFile(newfile){
        String datadir = "Unavailable"
        String line
        try {
            File file = new File(newfile)
            file.withReader { reader ->
                while ((line = reader.readLine())!=null) {
                    if (line.contains(" -D ")){
                        def holder = line.split()
                        holder.eachWithIndex {var, index ->
                            if(var.equals("-D")){
                                if (!holder[index.value + 1].contains('data')) {
                                    datadir = holder[index.value + 1] + " " + holder[index.value + 2]
                                    datadir = datadir.replaceAll('"', "")
                                }
                                else{
                                    datadir = holder[index.value + 1].replaceAll('"', "")
                                }
                                //log().debug("Data Directory $datadir was found.")
                            }
                        }
                    }
                }
            }
        }
        catch (FileNotFoundException e) {
        }
        return datadir
    }

    /**
     * Creates a directory
     * @param directory | directory location to be created
     * @return
     */

    def createDir(directory){
        if (continueProcessing){
            log().debug("Attempting to create the directory $directory")
            new File(directory).mkdir()

            if(new File(directory).exists()){
                log().info("Directory $directory was successfully created.")
            }
            else{
                log().error("Directory was not created. Processing will not continue for this Data Server.")
                continueProcessing = false
            }
        }
    }

    /**
     * Creates a backup of a specified folder. The folder backup
     * will have the naming convention of <folder>_backup_<version>
     * @param directory | folder to be backed up
     * @param version | version of the older content
     * @return the path/name of the backup
     */

    def String renameToBackupFolder(directory, version){
        String backupdirname = directory +"_backup_" + version
        if (continueProcessing){
            if(new File(directory).exists()){
                def backup = new File(backupdirname)
                if(backup.exists()){
                    //if there is already a backup, delete it and recreate it.
                    log().info("Deleting the existing backup directory $backupdirname.")
                    backup.deleteDir()
                    if(backup.exists()){
                        log().error("Backup directory was not deleted.")
                    }
                }
                log().info("Data directory $directory will be renamed to $backupdirname.")

                new File(directory).renameTo(backup)

                if(new File(backupdirname).exists()){
                    log().debug("Data directory was successfully backed up.")
                }
                else{
                    log().error("Backup directory was not created. Processing will not continue for this Data Serverr.")
                    continueProcessing = false
                }
            }
            else{
                if(new File(directory+"_backup_" + version).exists()){
                    log().info("Backup directory already created. Proceeding.")
                }
                else{
                    log().error("Unable to back up the data directory $directory so processing will not continue for this dataserver.")
                    continueProcessing = false
                }
            }
        }
        return backupdirname
    }


    /**
     * Creates a backup of a specified file. The file backup will
     * have the naming convention of <file>_backup_<version>
     * @param file to be backed up
     * @param version | version of the file
     * @return the path/name of the backup
     */

    def String renameToBackupFile(file, version){
        String backupFilename = file + "_backup_" + version
        if (continueProcessing){
            if(new File(file).exists()){
                def backup = new File(backupFilename)
                if(backup.exists()){
                    //if there is already a backup, delete it and recreate it.
                    log().info("Deleting the existing backup file $backupFilenamee.")
                    backup.deleteDir()
                    if(backup.exists()){
                        log().error("Backup file was not deleted.")
                    }
                }
                log().info("File $file will be renamed to $backupFilename.")

                new File(file).renameTo(backup)

                if(new File(backupFilename).exists()){
                    log().debug("File $file was successfully backed up.")
                }
                else{
                    log().error("Backup file was not created. Processing will not continue for this Data Serverr.")
                    continueProcessing = false
                }
            }
            else{
                if(new File(file+"_backup_" + version).exists()){
                    log().info("Backup file already created. Proceeding.")
                }
                else{
                    log().error("Unable to back up the file $file so processing will not continue for this dataserver.")
                    continueProcessing = false
                }
            }
        }
        return backupFilename
    }

    def String copyToBackup(directory, sourcever, targetver){
        String backupsourcedirname = directory + "_backup_" + sourcever
        if (continueProcessing){
            log().info("Backup data directory data_backup_$sourcever will be created.")
            def source = new File(directory)
            def backup = new File(backupsourcedirname)
            boolean reentry = false

            if(source.exists()){
                if(backup.exists()){
                    //if there is already a backup, delete it and recreate it. ONLY DO THIS IS THE CURRENT PG_VERSION OF THE DATADIR IS NOT AT THE TARGET VERSION
                    if(getDatasrvVersiononDisk(directory).toString().minus(".").trim() != targetver){
                        log().info("Deleting an existing backup directory.")
                        backup.deleteDir()

                        if(new File(backupsourcedirname).exists()){
                            log().error("Existing backup directory not deleted.")
                        }
                    }
                    else{
                        log().info("Existing backup data directory is already at the target version. The script will delete the existing data directory and continue the upgrade process.")
                        source.deleteDir()
                        reentry = true
                    }
                }

                if(!reentry){
                    try{
                        if(determineOS() == "WIN"){
                            log().info("Renaming the data directory $directory to $backupsourcedirname")
                            def mvcmd = "cmd /c rename " + directory +  " " + backupsourcedirname
                            mvcmd.execute()
                            //still need to call the rename to change the system references. Even though we moved the file on disk it will still seem like the backup doesn't exist.
                            source.renameTo(backup)
                        }
                        else{
                            source.renameTo(backup)
                        }
                    }
                    catch(IOException e){
                        e.printStackTrace()
                    }

                    if(backup.exists()){
                        log().info("Backup $backup was successfully created.")
                        //new File(directory).deleteDir()
                    }
                    else{
                        log().error("Backup directory $backupsourcedirname was not created. Processing will not continue for this dataserver.")
                        continueProcessing = false
                    }
                }
            }
            else{
                //source.renameTo(backup)
                //the data directory is not found so check for a valid backup
                if(backup.exists()){
                    log().warn("NOTE: Script detected no data directory but found a previously created backup. Proceeding")
                }
                else{
                    log().error("Unable to find an existing back up $backup so processing will not continue for this dataserver.")
                    continueProcessing = false
                }
            }
        }
        return backupsourcedirname
    }

    // Define the property filename and new property
    def modifyPostgresUserModsConf(confFileName, propertyName, propertyValue){
        def fileName = confFileName
        def newProperty = propertyName
        def newPropertyValue = propertyValue

        // Load existing properties (optional, if you want to preserve them)
        def props = new Properties()
        try {
            new File(fileName).withInputStream { stream ->
            props.load(stream)
          }
        } catch (FileNotFoundException e) {
          // Handle file not found exception (optional)
          log().error("File not found: $fileName")
        }

        // Add or modify the property
        props.setProperty(newProperty, newPropertyValue)

        // Save the updated properties
        new File(fileName).withWriter { writer ->
          props.store(writer, "Updated properties")
        }
    }


    /**
     * Modify the postgresql.conf file
     * @param confFile | postgresql.conf file to be updated
     * @return
     */
    def modifyPostgresqlConfFile(confFile, flatversion, datasrvdir_data, datasrvdir_twelvebyte, log_directory){
        if (continueProcessing){
            log().info("Attempting to modify the postgresql.conf file.")
            String dataLogDirectory = log_directory
            if(log_directory == ""){
                dataLogDirectory = "'" + datasrvdir_data + File.separator + "Logs'"
            }

            if(determineOS() == "WIN"){
                dataLogDirectory = dataLogDirectory.replaceAll("\\\\", "/")
            }

            //String dataServerCode = getOptionfromConfFile()
            def pgconf = new File(confFile)
            String stringToWrite = pgconf.text
            stringToWrite = stringToWrite.replaceAll("#log_destination", "log_destination")
            stringToWrite = stringToWrite.replaceAll("#log_directory = 'log'", "log_directory = ${dataLogDirectory}")
            stringToWrite = stringToWrite.replaceAll("#log_filename = 'postgresql-%Y-%m-%d_%H%M%S.log'", "log_filename = '$datasrvdir_twelvebyte-%Y-%m-%d_%H%M%S.log'")
            stringToWrite = stringToWrite.replaceAll("#log_rotation_size", "log_rotation_size")
            stringToWrite = stringToWrite.replaceAll("#log_truncate_on_rotation = off", "log_truncate_on_rotation = on")
            stringToWrite = stringToWrite.replaceAll("#logging_collector = off", "logging_collector = on")
            stringToWrite = stringToWrite.replaceAll("#max_prepared_transactions = 0", "max_prepared_transactions = 256")
            stringToWrite = stringToWrite.replaceAll("#wal_buffers = -1", "wal_buffers = 16MB")
            stringToWrite = stringToWrite.replaceAll("#log_min_messages", "log_min_messages")
            stringToWrite = stringToWrite.replaceAll("max_connections = 100", "max_connections = 256")
            stringToWrite = stringToWrite.replaceAll("#listen_addresses = 'localhost'", "listen_addresses = '*'")
            stringToWrite = stringToWrite.replaceAll(/(?i)unix_socket_directory(.*)/, 'unix_socket_directories$1\n' +
                                                       '\t\t\t\t\t#comma-separated list of directories')
            stringToWrite = stringToWrite.replaceAll(/(?im)^[\s]*checkpoint_segments[\s=]*([\d]*)\s*(#(.*))?$/) { allResults -> 
                                                       int convertedSegmentSize = allResults[1].toInteger() * 48
                                                       String maxWalSizeValue = "1GB"
                                                       if (convertedSegmentSize > 1024){
                                                           maxWalSizeValue = "${convertedSegmentSize}MB"
                                                       }
                                                       return "max_wal_size = ${maxWalSizeValue}\nmin_wal_size = 80MB"
                                                    }
            if (!(stringToWrite =~ /(?im)^[\s]*dynamic_shared_memory_type/)){
                stringToWrite = stringToWrite +"\ndynamic_shared_memory_type = posix	# the default is the first option\n" + 
                 "\t\t\t\t\t# supported by the operating system:\n" +
                 "\t\t\t\t\t#   posix\n" +
                 "\t\t\t\t\t#   sysv\n" +
                 "\t\t\t\t\t#   windows\n" +
                 "\t\t\t\t\t#   mmap\n" +
                 "\t\t\t\t\t# (change requires restart)\n"
            }

            pgconf.write(stringToWrite)
         }
    }

    /**
     * grabs specified options from the specified file
     * @param confFile | old postgres.conf file
     * @param option | the specific option that we want to find
     * @return
     */
    def getOptionfromConfFile(confFile, option){
        File file = new File(confFile)
        String line
        String setting = ""
        file.withReader { reader ->
            while ((line = reader.readLine())!=null) {
                if (line.contains(option)){
                   setting = line
                   log().debug("Located the option $option from the file $confFile")
                }
            }
         }
         if(setting == ""){
             log().debug("Option $option was not found in the file $confFile")
         }
         return setting
    }
    /**
     * Dumps the database content to a file
     * @param datasrvbin | pg_dumpall executable location
     * @param dumpfile | the file to be created with the contents of the database
     * @return
     */
    def runPGDump(datasrvbin, dumpfile){
        if (continueProcessing){
            log().debug("Attempting to dump out Data Server contents for ")
            String dumpcmd =  datasrvbin + File.separator + "pg_dumpall  -f  " + dumpfile
            launchCommandwEnv(dumpcmd)

            if(new File(dumpfile).exists()){
                log().info("Database dump $dumpfile was successfully created.")
            }else
            {
                log().error("Unable to create dumpfile $dumpfile. Processing will not continue for this dataserver.")
                continueProcessing = false
            }
        }
    }

    /**
     * Updates the dump file to comment out commands related to the creation of the inituser
     * This user is already created the initialization of the new db. Leaving these commands in there
     * well cause warnings when the restore occurs
     * @param dumpfile | dumpfile to be updated
     * @param inituser | the db user that was used to create the dump
     * @return
     */

    def updateDumpFile(dumpfile, inituser){
        if (continueProcessing){
            log().info("Updating the dumpfile not to initiate the super user")

            File file = new File(dumpfile)
            File newfile = new File(dumpfile + "_temp")
            String line

            def create_bol = false
            def alter_bol = false
            newfile.withWriter('UTF-8') { writer ->
                file.withReader('UTF-8') { reader ->
                   while ((line = reader.readLine())!=null) {
                           if(!create_bol && !alter_bol){
                               if (line.contains("CREATE ROLE " + inituser) || line.contains("CREATE ROLE \"" + inituser + "\"")){
                                   writer << line.replace("CREATE ROLE", "-- CREATE ROLE") + newline
                                   create_bol = true
                               }
                               else if (line.contains("ALTER ROLE " + inituser) || line.contains("ALTER ROLE \"" + inituser + "\"")){
                                  writer << line.replace("ALTER ROLE", "-- ALTER ROLE") + newline
                                  alter_bol = true
                               }
                               else{
                                 writer << line + newline
                               }
                           }
                           else{
                               writer << line + newline
                           }
                       }
               }
            }

            if(newfile.exists()){
                //delete old files
                new File(dumpfile).delete()
                new File(dumpfile + "_temp").renameTo(new File(dumpfile))
                //change encoding to UTF8
                //new File(dumpfile).append("--", 'utf-8')

                //change encoding to UTF8
                //def f = new File(dumpfile + "_temp").getText()
                //new File(dumpfile + "_temp_utf8").write(f, 'utf-8')

                //delete old files
                //new File(dumpfile + "_temp").delete()

                //rename new file
                //new File(dumpfile + "_temp_utf8").renameTo(new File(dumpfile))
            }
            else{
                log().error("Unable to update dump file $dumpfile. Processing will not continue for this Data Server.")
                continueProcessing = false
            }
         }
    }

    /**
     * Updates the dump file to comment out commands related to the creation of the inituser
     * This user is already created the initialization of the new db. Leaving these commands in there
     * well cause warnings when the restore occurs
     * @param dumpfile | dumpfile to be updated
     * @param inituser | the db user that was used to create the dump
     * @return
     */

    def updateDump(dumpfile, inituser){
        if (continueProcessing){
            File file = new File(dumpfile)
            File newfile = new File(dumpfile + "_temp")

            def currentText = file.text
            def newText = currentText.replaceAll("CREATE ROLE " + inituser, "-- CREATE ROLE " + inituser)
            newText = newText.replaceAll("ALTER ROLE " + inituser, "-- ALTER ROLE " + inituser)

            newfile.write(newText, 'utf-8')

            if(newfile.exists()){
                //delete old file
                file.delete()

                //rename new file
                newfile.renameTo(file)
            }
            else{
                log().error("Unable to update dump file $dumpfile. Processing will not continue for this Data Server.")
                continueProcessing = false
            }
        }
    }
    /**
     *
     * @param inituser | user being used to initialize the db
     * @param datasrvbin | initdb bin location
     * @param datasrvdata | data folder for the db
     * @return
     */

    def initializeNewDB(inituser, datasrvbin, datasrvdata){
        if (continueProcessing){
            log().debug("Attempting to initialize the database")
            String initcmd = datasrvbin + File.separator + "initdb" + "  -D  " + datasrvdata + "  -U  " + inituser + "  -A  trust  -E  UTF8  --locale=C"
            //String initcmd = datasrvbin + File.separator + "initdb" + "  -D  " + datasrvdata + "  -U  " + inituser + "  -A  trust  --locale=C"
            launchCommandwEnv(initcmd)

            if(new File(datasrvdata).exists()){
                log().info("Database was successfully initialized.")
            }else{
                println("Unable to to initialize the database. Processing will not continue for this Data Server.")
                continueProcessing = false
            }
        }
    }

    /**
     *
     * @param datasrvbin
     * @param inituser
     * @param dumpFile
     * @return
     */

    def restoreDB(datasrvbin, inituser, dumpFile){
        if (continueProcessing){
            log().info("Attempting to restore Data Server contents for " + dumpFile)
            String restorecmd =  datasrvbin + File.separator + "psql  -d  postgres  -U  " + inituser + "  --set  ON_ERROR_STOP=1  -f  " + dumpFile
            launchCommandwEnv(restorecmd)
        }
    }

    /**
     * Scans a given Postgresql Instance.  Compiles a list of databases and their
     * associated users.  Then, each user is granted access to the PUBLIC schema.
     *
     * @param datasrvbin
     * @return
     */

    def updatePublicSchemaPermissions(datasrvbin){
        log().info("Attempting to update the PUBLIC schema")
        // The PGUSER, PGPASSWORD, and PGPORT area all set as environment properties.
        // So, they are omitted on the command line.
        def pqslBaseCmd = datasrvbin + File.separator + "psql  -dpostgres"

        String queryDBscmd =  pqslBaseCmd + "  --csv  -cSELECT datname, pg_database.datacl from pg_catalog.pg_database;"
        def databaseInfo = sendCommandgetOutput(queryDBscmd)
        //Remove the column headers.
        databaseInfo = databaseInfo.replaceAll('datname,', '')
        databaseInfo = databaseInfo.replaceAll('dataclpostgres,', '')
        //Remove the opening brackets.
        databaseInfo = databaseInfo.replaceAll('\\"\\{', '')
        //Replace the closing brackets with new lines.
        databaseInfo = databaseInfo.replaceAll('\\}\\"', '\n')
        def databaseAndUserArray = databaseInfo.split("\n")
        def dbToUserMap = [:]
        for (line in databaseAndUserArray) {
            def tmpLineBreak = line.split(",")
            dbToUserMap.put(tmpLineBreak[0], tmpLineBreak[1..-1])
        } 
        //create temporary file to run the commands in.
        def updateSchemaFile = System.properties['java.io.tmpdir'] + File.separator + "updateSchemas.sql"
        log().info("Using file:  ${updateSchemaFile}")
        new FileWriter (updateSchemaFile).with {
            dbToUserMap.each{database, userArray -> 
                if (!database.contains("template")) {
                    def usersToUpdate = ""
                    userArray.eachWithIndex{currentUser, index ->
                        // If the first character is a "=", that means it's a special user of that database.  
                        // It will be stripped out, for consistency's sake.
                        if (currentUser.length() > 0){
                            def formattedUser = currentUser.startsWith("=") ? currentUser[1..-1] : currentUser
                            if (formattedUser.contains("=")){
                                def currentUserSplit = formattedUser.split("=")
                                if (usersToUpdate == "") {
                                    usersToUpdate = "\"${currentUserSplit[0]}\""
                                } else {
                                    usersToUpdate = "${usersToUpdate}, \"${currentUserSplit[0]}\""
                                }
                            }
                        }
                    }
                    write("\\connect ${database};\n")
                    write("GRANT ALL ON SCHEMA PUBLIC TO ${usersToUpdate};\n\n")
                }
            }
            flush()
        }
        //run the commands.
        String runUpdatescmd = pqslBaseCmd + "  -f${updateSchemaFile}"
        def updatesOutput = sendCommandgetOutput(runUpdatescmd, true)
        log().info(newline + updatesOutput)
        def lowercaseOutput = updatesOutput.toLowerCase()
        def errorString = "error"
        def failString  = "fail"
        if(lowercaseOutput.contains(errorString) ||
            lowercaseOutput.contains(failString))
            {
                log().error("Error updating PUBLIC schema. Processing will not continue for this dataserver.")
                continueProcessing = false
            }
        // delete the temporary file.
        new File(updateSchemaFile).delete()
    }

    /**
     * Parses the database port from the corresponding script
     * @param datasrvdir_script | postgres server script that contains the port
     * @return
     */

    def String findPortFromScript(datasrvdir_script){
        String port = "Unavailable"
        String line
        def props = new Properties()
        log().debug("Attempting to find the database port by scanning the file $datasrvdir_script")
        File file = new File(datasrvdir_script)
        if(file.exists()){
            String fileText = file.text.replace("-i -p ", "-p ")
            file.write(fileText)
            file.withReader { reader ->
               while ((line = reader.readLine())!=null) {
                   if (line.contains("\"-p ")){
                       def holder = line.split()
                       holder.eachWithIndex {var, index ->
                       if(var.equals("\"-p")){
                            port = holder[index.value + 1].minus('"')
                           try{
                               if(port.startsWith('$')){
                                   props.load(file.newDataInputStream())
                                   port = props.getProperty('PORT')
                                   log().info(port)
                               }
                            }
                           catch (Exception e){
                               log().trace(e);
                           }
                        }
                      }
                   }
               }
            }
        }
        else{
            //we will already know that the file doesn't exist from the precheck and will not do any actual work
        }

        return port
    }

    def String findDataDirFromScript(datasrvdir_script){
        String datadir = "Unavailable"
        if (continueProcessing){
            log().debug("Attempting to find the database data directory by scanning the file $datasrvdir_script")
            File file = new File(datasrvdir_script)
            datadir = file.getAbsoluteFile().getParent() + File.separator + "data"
            File datadir_file = new File (datadir)
            if (!datadir_file.exists()) {
                log().error("Database data directory does not exist.")
                System.exit(2)
            }
            return datadir
        }
    }


    def String findConfigDirFromScript(datasrvdir_script){
        String configdir = "Unavailable"
        if (continueProcessing){
            log().debug("Attempting to find the database configuration directory by scanning the file $datasrvdir_script")
            File file = new File(datasrvdir_script)

            if(determineOS() == "WIN"){
                configdir = file.getAbsoluteFile().getParent()
            }
            else{
                String line
                try {
                    file.withReader { reader ->
                       while ((line = reader.readLine())!=null) {
                           if (line.contains("CONFIGDIR=")){
                               def holder = line.split("=")
                               //println("Found line: " + holder)
                               holder.eachWithIndex {var, index ->
                                   //println(var)
                                   //println(index)
                                   if(var.equals("CONFIGDIR")){
                                       configdir = holder[index.value + 1].minus('"')
                                       //log().debug("Configuration directory $configdir was found.")
                                   }
                              }
                           }
                       }
                    }
                } catch (FileNotFoundException e) {
                    log().error ("File $datasrvdir_script was not found")
                    System.exit(2)
                }
            }
            return configdir
        }
    }


    def String insertNewCfgWizardDirInScript(datasrvdir_script){
        String configdir = "Unavailable"
        boolean cfgwizard_altsku_added = false

        def newline=""
        if (continueProcessing){
            log().debug("Add code to find latest cfgwizard alt sku directory to the file $datasrvdir_script")
            File file = new File(datasrvdir_script)
            File newfile = new File(datasrvdir_script + "_temp")

            String line
            file.withReader { reader ->
                while (((line = reader.readLine())!=null) && !(cfgwizard_altsku_added)) {
                    if (line.contains("set cfgwiz_dir=")) {
                        cfgwizard_altsku_added = true
                    }
                    if (line.contains("level_env"))
                    {
                        newfile.append(line + '\n')
                        newfile.append('\n')
                        newfile.append("REM Code to find the latest cfgwizard altsku directory" + '\n')
                        newfile.append("set SASHome=%SAS_HOME%" + '\n')
                        newline = "for /f %%a in ( 'dir /b /a:d /o:n \"%SASHome%\\SASDeploymentManager\\9.4\\products\\cfgwizard__94*__prt__*\"' ) ^"
                        newfile.append(newline + '\n')
                        newline = "do ("
                        newfile.append(newline + '\n')
                        newline = "   set cfgwiz_dir=%SASHome%\\SASDeploymentManager\\9.4\\products\\%%~na"
                        newfile.append(newline + '\n')
                        newline = ")"
                        newfile.append(newline + '\n')
                    }
                    else if (line.contains("sassvc") || line.contains("SASSvc"))
                    {
                        String[] str;
                        str = line.split("Utilities")
                        newfile.append("   \"%cfgwiz_dir%\\" + "Utilities" + str[1] + '\n')
                    }
                    else
                    {
                        newfile.append(line + '\n')
                    }
                }
            }
            if (!cfgwizard_altsku_added)
            {
                // delete the original datasrvdir_script file
                new File(datasrvdir_script).delete()

                //rename new file
                new File(datasrvdir_script + "_temp").renameTo(new File(datasrvdir_script))
            }
            else
            {
                new File(datasrvdir_script + "_temp").delete()
            }
        }
    }




    /**
     * Assumes that the data server execution script is named after the 12byte and grabs that 12byte/file name
     * @param datasrvdir_script
     * @return
     */
    def String findTwelveByteFromScript(datasrvdir_script){
        def file = new File(datasrvdir_script)
        return file.name.replaceFirst(~/\.[^\.]+$/, '')
    }

    /**
     * Assumes that the data server execution script is named after the 12byte and grabs that 12byte/file name
     * @param datasrvdir_script
     * @return
     */
    def String findTDBServerNameFromScript(datasrvdir_script){
        def parentDirName = new File(datasrvdir_script).getParentFile().getName()
        return parentDirName
    }

    /**
     * Determines if the system is Unix or Windows
     * @return WIN or UNIX
     */

    def String determineOS(){
        if (System.properties['os.name'].toLowerCase().contains('windows')) {
            return "WIN"
        } else {
            return "UNIX"
        }
    }

    /**
     * Will determine to use shell or batch specification based on the current system OS
     * @return "sh" or "bat"
     */

    def String getShellorBat(){
        String os = determineOS()

        if (os.equalsIgnoreCase("WIN")){
            return ".bat"
        }
        else{
            return ".sh"
        }

    }

    /**
     * Set environment variables
     * @param user | postgres user
     * @param pass | postgres password
     * @param port | database port
     * @return
     */

    def setPGEnv(user, pass, port){
        System.properties['PGUSER'] = user
        System.properties['PGPASSWORD'] = pass
        System.properties['PGPORT'] = port
    }
    /**
     * Set environment variables
     * @param library | path to the library required
     * @return
     */

    def setLIBEnv(library){
        System.properties['LD_LIBRARY_PATH'] = library
        System.properties['LIBPATH'] = library
    }

    /**
     * displays all the current system properties available top groovy
     * @return
     */
    def displaySystemProperties(){
        println("== Display system properties ==")
        System.properties.each { k,v->
            println ("$k = $v")
        }
    }

    /**
     * Displays the current version of postgres as well as a list of all the schemas
     * @param datasrvbin
     * @return
     */

    def displayVerification(datasrvbin){
        if (continueProcessing){
            String versioncmd = datasrvbin + File.separator + "psql  -d  postgres  -c  select version()"
            launchCommandwEnv(versioncmd)
            String listcmd = datasrvbin + File.separator + "psql  -d  postgres  -c  \\l"
            launchCommandwEnv(listcmd)
        }
    }

    /**
     * Gets the PG_VERSION that is current on disk in the data directory
     * @param datasrvdir_datadir | directory that contains the PG_VERSION file
     * @return
     */
    def String getDatasrvVersiononDisk(datasrvdir_datadir){
        def ver = 0
        def pg_verfile = new File(datasrvdir_datadir + File.separator + "PG_VERSION")
        if(pg_verfile.exists()){
            ver = new BigDecimal(pg_verfile.text.trim())
        }
        return ver
    }

    /**
     * Disables security if it was configured on the system
     * @param datasrvdir_datadir | data directory
     */
    def disableSecurity(datasrvdir_datadir){
        def pgconf = new File(datasrvdir_datadir + File.separator + "postgresql.conf")
        def pgconf_temp = new File(datasrvdir_datadir + File.separator + "postgresql.conf.tmp")

        def pghba = new File(datasrvdir_datadir + File.separator + "pg_hba.conf")
        def pghba_temp = new File(datasrvdir_datadir + File.separator + "pg_hba.conf.tmp")

        if (continueProcessing){
            String line
            //handle postgresql.conf edits
            log().debug("Updating postgresl.conf to remove security settings.")
            pgconf.withReader { reader ->
                while ((line = reader.readLine())!=null) {
                   if (line.contains("ssl_ciphers") || line.replaceAll("\\s", "").contains("ssl=on")){
                    //ignore line
                     log().debug("Found security setting! Commenting it out...")
                     pgconf_temp.append("#" + line + newline )
                   }
                   else{
                     pgconf_temp.append(line + newline )
                   }
                }
            }
            //delete old file
            pgconf.delete()
            //rename new file
            pgconf_temp.renameTo(pgconf)

            //handle certs
            log().debug("Removing any existing certs/keys")
            if( new File(datasrvdir_datadir + File.separator + "server.crt").exists() ){
                new File(datasrvdir_datadir + File.separator + "server.crt").delete()
            }
            if( new File(datasrvdir_datadir + File.separator + "server.key").exists() ){
                new File(datasrvdir_datadir + File.separator + "server.key").delete()
            }
            if( new File(datasrvdir_datadir + File.separator + "root.crt").exists() ){
                new File(datasrvdir_datadir + File.separator + "root.crt").delete()
            }
            if( new File(datasrvdir_datadir + File.separator + "root.crl").exists() ){
                new File(datasrvdir_datadir + File.separator + "root.crl").delete()
            }

        }

        //handle pg_hba.conf file
        log().debug("Updating pg_hba.conf file")
        def currentText = pghba.text
        def newText = currentText.replaceAll("clientcert=1" , "")
        newText = newText.replaceAll("hostssl", "host")
        newText = newText.replaceAll("cert", "md5")
        pghba_temp.write(newText)
        //delete old file
        pghba.delete()
        //rename new file
        pghba_temp.renameTo(pghba)
    }

    /**
     * Checks to make sure the script has been provided valid system paths
     */
    def checkSystemPaths(datasrvdir_home, datasrvdir_home_old, datasrvdir_config, datasrvdir_data, datasrvdir_bin, datasrvdir_bin_old, datasrvdir_lib_old, datasrvdir_conffile, datasrvdir_script){
        if(!new File(datasrvdir_home).exists()){
            log().error("Alert! Directory $datasrvdir_home does not exist. Processing will not continue for this Data Server.")
            continueProcessing = false
        }
        if(!new File(datasrvdir_home_old).exists()){
            log().error("Alert! Directory $datasrvdir_home_old does not exist. Processing will not continue for this Data Server.")
            continueProcessing = false
        }
        if(!new File(datasrvdir_config).exists()){
            log().error("Alert! Directory $datasrvdir_config does not exist. Processing will not continue for this Data Server.")
            continueProcessing = false
        }
        if(!new File(datasrvdir_data).exists()){
            //data directory does not exist so check for a valid backup directory (IMPORTANT FOR REENTRY: IN CASE SCRIPT EXITED DURING EXECUTION AND THE BACKUP WAS ALREADY CREATED)
            if(!new File(datasrvdir_data + "_backup_" + getVersion(datasrvdir_home_old))){
                log().error("Alert! Directory $datasrvdir_data does not exist. Processing will not continue for this Data Server.")
                log().error("Alert! Script also could not find a valid backup directory.")
                continueProcessing = false
            }
        }
        if(!new File(datasrvdir_bin).exists()){
            log().error("Alert! Directory $datasrvdir_bin does not exist. Processing will not continue for this Data Server.")
            continueProcessing = false
        }
        if(!new File(datasrvdir_bin_old).exists()){
            log().error("Alert! Directory $datasrvdir_bin_old does not exist. Processing will not continue for this Data Server.")
            continueProcessing = false
        }
        //if(!new File(datasrvdir_conffile).exists()){
        //  println("Alert! The following directory/file does not exist and therefore processing will not continue for this dataserver will not continue. -->" + datasrvdir_conffile)
        //  continueProcessing = false
        //  }
        if(!new File(datasrvdir_script).exists()){
            log().error("Alert! Directory $datasrvdir_script does not exist. Processing will not continue for this Data Server.")
            continueProcessing = false
        }
    }

    /**
     * Gets postgres version from installed binaries
     * @param datasrvdir_bin | binary bin location
     * @param datasrvdir_bin | binary library location
     */
    def String getVersion(datasrvdir_home){
        log().debug("Getting the version number from the postgres binary location.")
        def datasrvdir_lib = datasrvdir_home + File.separator +"lib"
        def datasrvdir_bin = datasrvdir_home + File.separator +"bin"

        //Need to make sure the library is set prior to any call to the binary location
        setLIBEnv(datasrvdir_lib)

        String ver = "0"
        String execmd = ""
        if(new File(datasrvdir_bin).exists()){

            if(determineOS() == "WIN"){
                execmd = datasrvdir_bin + File.separator + "postgres.exe  -V"
            }
            else{
                execmd = datasrvdir_bin + File.separator + "postgres  -V"
            }
            //get rid of the end of the minor version
            ver = sendCommandgetOutput(execmd).minus("postgres (PostgreSQL)").trim()
            if (ver.startsWith("9.")) {
                ver = ver.substring(0, 3)
            }
            else{
                ver = ver.substring(0, 2)
            }
        }
        else
        {
            log().error("Postges binary location is not available! -> " + datasrvdir_bin)
            System.exit(1)
        }
        return ver
    }


    /**
     * Determines if the PostgreSQL Database instance is TLS enabled
     * @return true or false
     */
    def boolean isPostgreSQLDBInstanceTlsEnabled(String datasrvdir_data_backdir){
        boolean isEnabled = false
        String postgreSQLDBConf_baseFile  = datasrvdir_data_backdir + File.separator + "postgresql.conf"
        String postgreSQLDBConf_usermodsfile  = datasrvdir_data_backdir + File.separator + "postgresql.usermods.conf"
        String baseFileResult, userModsFileResult = ""

        baseFileResult = searchConfigFile(postgreSQLDBConf_baseFile)

        File usermodsfile = new File(postgreSQLDBConf_usermodsfile)
        if(usermodsfile.exists()){
            userModsFileResult = searchConfigFile(postgreSQLDBConf_usermodsfile)
        }
        else{
            userModsFileResult = "undetermined"
        }

        //Basically, if usermods doesn't have an opinion and the base file is true, return true.
        //Alternatively, if the usermods file is true, return true.
        //Otherwise return false.
        if ((userModsFileResult == "undetermined" && baseFileResult == "true") || 
             userModsFileResult == "true"){
            isEnabled = true
        }

        return isEnabled
    }

    /**
     * Searches postgresql.config or postgresql.usermods.conf, determines if ssl is enabled, 
     * and prints other fun facts in debug mode.  "Undetermined" was needed because there's 
     * a chance that the user mod file exists, but SSL hasn't been defined in there.
     * @return true, false or undetermined
     */
    def String searchConfigFile(String configFile){
        String sslEnabled = "undetermined"

        log().debug("Investigating file:  ${configFile}")

        // Read the configuration file
        def configFileContents = new File(configFile).text

        configFileContents.eachLine { line ->
            def matchLine = ""
            if (matchLine = line =~ /(?i)^ssl[\W=]*(\w*)\W*$/){
                def propValue = matchLine.group(1)
                if (propValue =~ /(?i)\b(true|on|yes|1)\b/){
                    log().debug("RESULTS : ssl property is activated.")
                    sslEnabled = "true"
                }
                else if (propValue =~ /(?i)\b(false|off|no|0)\b/){
                    log().debug("RESULTS : ssl property is not activated.")
                    sslEnabled = "false"
                }
                else{
                    log().debug("RESULTS : ssl property was malformed.")
                    sslEnabled = "undetermined"
                }
            }

            if(matchLine = line =~ /^ssl_cert_file[\W=]*\'(.*)\'[\W]*/){
                log().debug("SSL certificate file: ${matchLine.group(1)}")
                certFileName = matchLine.group(1)
            }

            if(matchLine = line =~ /^ssl_key_file[\W=]*\'(.*)\'[\W]*/){
                log().debug("SSL key file: ${matchLine.group(1)}")
                keyFileName = matchLine.group(1)
            }

            if(matchLine = line =~ /^ssl_ca_file[\W=]*\'(.*)\'[\W]*/){
                log().debug("SSL ca file: ${matchLine.group(1)}")
                caFileName = matchLine.group(1)
            }

            if(matchLine = line =~ /^password_encryption[\W=]*\'(.*)\'[\W]*/){
                log().debug("PW Encryption method: ${matchLine.group(1)}")
            }

        }

        if(sslEnabled == "undetermined"){
            log().debug("RESULTS : No properly formed ssl property was found.")
        }

        return sslEnabled
    }


    /**
     * Determines if the client authentication for DB instance is
     * TSL enabled
     * @return true or false
     */
    def boolean isDBClientAuthenticationTlsEnabled(String datasrvdir_data_backdir){
        String clientAuthenticationConfFile  = datasrvdir_data_backdir + File.separator + "pg_hba.conf"
        // Read the configuration file
        def configFileContents = new File(clientAuthenticationConfFile).text
        String hostssl = ""
        boolean isEnabled = false
        configFileContents.eachLine { line ->
            //if ((line.contains("ssl") && (line.contains("true") || line.contains(""))) {
            if (line.contains("hostssl ")) {
                hostssl = line
            }
        }

        if (hostssl != "") {
            if (!hostssl.startsWith('#')) {
                isEnabled = true
            }
        }
        return isEnabled
    }

    /**
     * Searches the specified file for log_directory.  If the line is found,
     * the value is returned. Otherwise, blank string is returned.
     * @return log_directory value or blank (if not found)
     */
    def String findLogFileInConfigFile(fileToSearch){
    String logFileLocation = ""
    String line
    File configFile = new File(fileToSearch)
    Boolean logLineNotFound = true
    if (configFile.exists())
    {
        configFile.withReader { reader ->
            while((line = reader.readLine())!=null &&
                  logLineNotFound ){
                if (line.startsWith("log_directory")){
                    logLineNotFound = false
                    int indexOfFirstEquals = line.indexOf("=") + 2  // This offset takes out the equals sign and the space.
                    int indexOfLastHashtag = line.lastIndexOf("#")
                    logFileLocation = line.substring(indexOfFirstEquals, indexOfLastHashtag)
                    logFileLocation = logFileLocation.replaceAll("\t", "") // This tidies up the extra tabs.
                }
            }
        }
    } 
    return logFileLocation
}


    /**
     * Upgrades a Postgres instance from 9.1 to 9.5 or 9.4 to 9.5
     * @param twelvebyte | Dataserver 12byte
     * @param home | SASHome location of the 9.5 binaries
     * @param config | configured location
     * @param user | pg user
     * @param pass | pg pass
     * @return
     */
    def int updateInstance(twelvebyte, home, config, data, user, pass){
        continueProcessing = true
        log().info("Starting Postgres Update for: " + twelvebyte)
        String datasrvdir_twelvebyte = twelvebyte.toLowerCase()
        String datasrvdir_home = home
        String datasrvdir_config = config
        String datasrvdir_os = determineOS()
        String minus_string = "SASWebInfrastructurePlatformDataServer" + File.separator + "9.4"
        String plus_string  = "SASWebInfrastructurePlatformDataServer" + File.separator + "9.4_PREVIOUS"
        String datasrvdir_home_old = datasrvdir_home.minus(minus_string).plus(plus_string)
        String datasrvdir_script = datasrvdir_config + File.separator + datasrvdir_twelvebyte + getShellorBat()
        //S1558203 - Reassign the value of data and config based on what's in the datasrvdir_script=fmdbsvrc.bat
        if(determineOS() == "WIN"){
            data = findDataDirFromScript(datasrvdir_script) //S1558203
            if ( data.endsWith('/data') ) {
                config = data.minus('/data')
            }
            if( data.endsWith('\\data') ) {
                config = data.minus('\\data')
            }
            datasrvdir_config = config
            insertNewCfgWizardDirInScript(datasrvdir_script)
        }
        String datasrvdir_data = data
        String datasrvdir_temp = datasrvdir_config + File.separator + "temp"
        String datasrvdir_temp_script_name = datasrvdir_twelvebyte + "_temp" + getShellorBat()
        String datasrvdir_temp_script = datasrvdir_config + File.separator + datasrvdir_temp_script_name
        String datasrvdir_bin = datasrvdir_home + File.separator + "bin"
        String datasrvdir_bin_old = datasrvdir_home_old + File.separator + "bin"
        String datasrvdir_lib = datasrvdir_home + File.separator + "lib"
        String datasrvdir_lib_old = datasrvdir_home_old + File.separator + "lib"
        String datasrvdir_conffile = datasrvdir_data + File.separator + "postgresql.conf"
        String datasrvdir_dumpfile = ""
        String pg_hba_confFile = datasrvdir_data + File.separator + "pg_hba.conf"

        if ( !System.properties['DUMPLOCATION']) {
            datasrvdir_dumpfile = datasrvdir_temp + File.separator + "db_update.dat"
        }
        else{
            datasrvdir_dumpfile = System.properties['DUMPLOCATION']+ File.separator + "db_update.dat"
        }

        //check to make sure all the paths are valid before doing anything
        checkSystemPaths(datasrvdir_home, datasrvdir_home_old, datasrvdir_config, datasrvdir_data, datasrvdir_bin, datasrvdir_bin_old, datasrvdir_lib_old, datasrvdir_conffile, datasrvdir_script)

        //set env variables
        setPGEnv(user, pass, findPortFromScript(datasrvdir_script))//get the port from a file on disk
        setLIBEnv(datasrvdir_home_old + File.separator + "lib")

        //get db user
        String datasrvdir_inituser = System.properties['PGUSER']

        //set SOURCE AND TARGET VERSIONS
        String currentver = getVersion(datasrvdir_home_old)//SCOTT!!get from the binaries in SASHOME!
        String targetver = getVersion(datasrvdir_home) //SCOTT!!get from the binaries in SASHOME!
        String verforfile_current = currentver.toString().minus(".").trim()
        String verforfile_target = targetver.toString().minus(".").trim()
        String datasrvdir_conffile_old = datasrvdir_data + "_backup_" + verforfile_current + File.separator + "postgresql.conf"

        log().trace("currentver: " + currentver)
        log().trace("targetver: " + targetver)
        log().trace("verforfile_current: " + verforfile_current)
        log().trace("verforfile_target: " + verforfile_target)

        //display current system properties
        //displaySystemProperties()

        //print important values for debugging purposes
        log().debug("Data Server Home Directory is " + datasrvdir_home)
        log().debug("Data Server Config Directory is " + datasrvdir_config)
        log().debug("Data Server Data Directory is " + datasrvdir_data)
        log().debug("Data Server Shell Script is " + datasrvdir_script)
        log().debug("Data Server Temp Location is " + datasrvdir_temp)
        log().debug("Data Server Temp Script Location is " + datasrvdir_temp_script)
        log().debug("Data Server Init User is " + datasrvdir_inituser)

        // 1. Check to make sure the database is not already at Postgres version 9.5
        //    stop servers from that may be running from previous attempt
        stopService(datasrvdir_temp_script)
        stopService(datasrvdir_script)

        // 2. Rename to backup the data dir of the postgres instance (needs to be retry friendly in the event of failure at any stage for example due to running out of disk space).
        String datasrvdir_data_back = copyToBackup(datasrvdir_data, verforfile_current, verforfile_target)

        // 3. Copy the webinfdsvr sh/bat from existing config location and change the SASHOME reference to point to the old content; now name 9.4_PREVIOUS (for Windows make sure SERVERTYPE=scripts).
        createDir(datasrvdir_temp)
        updateExecutionScript(datasrvdir_script, datasrvdir_temp_script, datasrvdir_data, datasrvdir_data_back)

        // 4. Run the script to start the service and wait for the port to come up (should use same port number per that instance configuration)
        setLIBEnv(datasrvdir_home_old + File.separator + "lib")
        startServiceForCurrent(datasrvdir_temp_script, datasrvdir_bin_old)
        displayVerification(datasrvdir_bin_old)

        //4a. Update permissions on all Public Schemas, if update is needed.
        if (currentver.split("\\.", 2)[0].toInteger() < 15){
            setLIBEnv(datasrvdir_home + File.separator + "lib")
            updatePublicSchemaPermissions(datasrvdir_bin)
            setLIBEnv(datasrvdir_home_old + File.separator + "lib")
        } else {
            log().info("Current version of Data Server (${currentver}) is 15 or higher and therefore no Public Schema updates are needed.")
        }

        // 5. Use pg_dumpall to dump all databases in data server to sql file
        // Modify the dump to comment out references to the database instance user (example see webinfdsvrcTasks.xml: under <!-- comment out recreating the superuser from the dump -->)
        runPGDump(datasrvdir_bin_old, datasrvdir_dumpfile)
        updateDumpFile(datasrvdir_dumpfile, datasrvdir_inituser)

        // 6. Stop the service and wait for the port to be available.
        stopService(datasrvdir_temp_script)

        // 7. Run the initdb command from the new Postgres version folder
        setLIBEnv(datasrvdir_home + File.separator + "lib")
        initializeNewDB(datasrvdir_inituser, datasrvdir_bin, datasrvdir_data)

        // 8. Rename postgresql.conf file from the newly initialized DB server and
        //    restore the conf file from the previous PG version
        renameToBackupFile(datasrvdir_conffile, verforfile_target)
        copyFile(datasrvdir_conffile_old, datasrvdir_conffile)

        // 10. Copy a current version of pg_hba.conf from backup data directory
        renameToBackupFile(pg_hba_confFile, verforfile_target)
        copyFile(datasrvdir_data + "_backup_" + verforfile_current + File.separator + "pg_hba.conf", pg_hba_confFile)

        // 11.0.  If there is no postgresql.usermods.conf file (UIP source is from pre-M9), copy it in.
        String postgresql_usermods_conf_data = datasrvdir_data + File.separator + "postgresql.usermods.conf"
        File postgresql_usermods_conf_data_file = new File (postgresql_usermods_conf_data)
        String postgresql_usermods_conf_scripts = datasrvdir_home + File.separator + "scripts" + File.separator + "postgresql.usermods.conf"
        File postgresql_usermods_conf_scripts_file = new File (postgresql_usermods_conf_scripts)

        if (! postgresql_usermods_conf_data_file.exists()){
            copyFile(postgresql_usermods_conf_scripts, postgresql_usermods_conf_data)
            log().debug("${postgresql_usermods_conf_scripts} copied to ${postgresql_usermods_conf_data}.")
        }
        else{
            log().debug("${postgresql_usermods_conf_scripts} NOT copied to ${postgresql_usermods_conf_data}.")
        }

        // 11.1. Copy a current version of postgresql.usermods.conf from the backup data directory when it exists
        String postgresql_usermods_conf_backup = datasrvdir_data + "_backup_" + verforfile_current + File.separator + "postgresql.usermods.conf"
        File postgresql_usermods_conf_backup_file = new File (postgresql_usermods_conf_backup)

        if(postgresql_usermods_conf_backup_file.exists()){
            if(postgresql_usermods_conf_data_file.exists()){
                if (! new File(postgresql_usermods_conf_data + "_backup_" + verforfile_target).exists()){
                    log().info("backing up ${postgresql_usermods_conf_data} (to ${renameToBackupFile})")
                    renameToBackupFile(postgresql_usermods_conf_data, verforfile_target)
                }
                else{
                    log().debug("No back up needed for ${postgresql_usermods_conf_data}")
                }
            }
            log().info("${postgresql_usermods_conf_backup} copied to ${postgresql_usermods_conf_data}.")
            copyFile(postgresql_usermods_conf_backup, postgresql_usermods_conf_data)
        }
        else {
            log().debug("${postgresql_usermods_conf_backup_file} doesn't exist - skipping.")
        }

        File pgconf = new File(datasrvdir_data + File.separator + "postgresql.conf")
        if(pgconf.exists()){
            String fileText = pgconf.text.replaceAll("#include = '...'", "include = 'postgresql.usermods.conf'")
            pgconf.write(fileText)
            log().info("PostgreSQL config file postgresql.conf gets updated to activate the include statement.")
        }
        else{
            log().debug("${datasrvdir_data}${File.separator}postgresql.conf doesn't exist.  Skipping")
        }
        
        // 12. Check the postgresql.conf and pg_hba.conf to see if the Data Server is TLS enabled.
        //     Also copy the SSL cert and key files from the backed up datadir to the new datadir if they exist in the backed-up datadir
        def twelbyteProductName = new File(datasrvdir_script).getParentFile().getName()
        if (isPostgreSQLDBInstanceTlsEnabled(datasrvdir_data_back)){
            log().info("$twelbyteProductName is TLS enabled; maintaining settings")
            if( certFileName?.trim() &&
                new File(datasrvdir_data_back + File.separator + certFileName).exists()){
                copyFile(datasrvdir_data_back + File.separator + certFileName, datasrvdir_data + File.separator + certFileName)
            }
            if( keyFileName?.trim() &&
                new File(datasrvdir_data_back + File.separator + keyFileName).exists()){
                copyFile(datasrvdir_data_back + File.separator + keyFileName, datasrvdir_data + File.separator + keyFileName)
            }
            if( caFileName?.trim() &&
                new File(datasrvdir_data_back + File.separator + caFileName).exists()){
                copyFile(datasrvdir_data_back + File.separator + caFileName, datasrvdir_data + File.separator + caFileName)
            }
            isTLSenabled = true
            modifyPostgresUserModsConf(postgresql_usermods_conf_data, "password_encryption", "md5")
        }
        else{
            log().info("$twelbyteProductName is not TLS enabled; skipping update.")
        }

        def hostSSLtrustLine = ""
        if (isDBClientAuthenticationTlsEnabled(datasrvdir_data)){
            log().info("$twelbyteProductName client authentication is TLS enabled; maintaining SSL settings.")
            hostSSLtrustLine = "hostssl    all             all             all                     trust"
        }
        else{
            log().info("$twelbyteProductName client authentication is not TLS enabled; maintaining settings.")
            hostSSLtrustLine = "host    all             all             all                     trust"
        }
        modifyPGHBAconfigfile(pg_hba_confFile, hostSSLtrustLine, true)
        log().info("Added a new line of $hostSSLtrustLine to " + pg_hba_confFile)


        // 13. Modify the postgresql.conf file with the additions/updates needed for updated postgres
        String logFileFromPreviousPGVersion = findLogFileInConfigFile(datasrvdir_conffile_old)
        modifyPostgresqlConfFile(datasrvdir_conffile, verforfile_target, datasrvdir_data, datasrvdir_twelvebyte, logFileFromPreviousPGVersion)

        // 13.1 Ensure that all of the files in the top level of the data directory are permission level 600 on Linux.
        if(determineOS() == "UNIX"){ 
            def fileList = new File(datasrvdir_data).eachFile(FileType.FILES) { currentFile ->
                //Wipe out all file permissions. 
                currentFile.setReadable(false, false); 
                currentFile.setWritable(false, false); 
                currentFile.setExecutable(false, false); 

                //Add 600 file permissions.
                currentFile.setReadable(true, true); 
                currentFile.setWritable(true, true); 
            }
        }

        // 14. Start the service using the script in the data server instance Config Lev dir and wait for the port to be available.
        startServiceForTarget(datasrvdir_script, datasrvdir_bin, isTLSenabled)

        // 15. Use psql to restore the sql file dump
        restoreDB(datasrvdir_bin, datasrvdir_inituser, datasrvdir_dumpfile)

        // 16. Display information to verify the dump restored successfully (Postgres version and list of databases)
        displayVerification(datasrvdir_bin)

        // 17. Stop the service
        stopService(datasrvdir_script)

        // 18. Clean Up
        if (continueProcessing){
            //remove backup if flag is set
            if(System.properties['REMOVEDATABACKUP'] == true){
                log().info("The --REMOVEDATABACKUP option has been set. Deleting the following folder: " + datasrvdir_data + "_backup_" + verforfile_current)
                new File(datasrvdir_data + "_backup_" + verforfile_current).deleteDir()
            }
            //keep dumpfile if flag is set
            if(System.properties['KEEPDUMP'] != true){
                log().info("The --KEEPDUMP option has NOT been set. The dumpfile $datasrvdir_dumpfile will be deleted.")
                new File(datasrvdir_dumpfile).delete()
                new File(datasrvdir_temp).deleteDir()

            }
            else{
                log().info("The --KEEPDUMP option has been set. The dumpfile %datasrvdir_dumpfile will NOT be deleted.")
            }

            // Restore pg_hba.conf file from step 12.
            modifyPGHBAconfigfile(pg_hba_confFile, hostSSLtrustLine, false)
 
            new File(datasrvdir_temp_script).delete()
            //new File(datasrvdir_conffile + "_backup_" + verforfile_target).delete()
            log().info("Postgres Update for $twelvebyte is complete.")

            return 0
        }
        else{
            log().error("ERROR: Postgres Update for $twelvebyte is NOT complete.")

            return -2
        }
    }


    static void main(def args){

    }
}
