#!/bin/ksh -p

# $Id: ose.sh,v 1.1.4.5.2.5.2.7.2.1.2.4.2.2 2019/04/23 18:26:37 paniko Exp $

# terminate on non-zero command exit status
set -e

# monitor mode (background processes run in a separate process group), prevent propagation of Ctrl-C (SIGINT)
set -m

PROGRAM_NAME="$0"

COMMANDS="clean kill help save start redeploy stats status stop displaypython startpython stoppython redeploypython tail test importMEH <directory> exportMEH <directory> purgeMEH <arguments> reloadLookupList <sor_id> <lookupListName> alertcache clear"

function main
{
    typeset BIN=$(cd $(dirname "$PROGRAM_NAME") ; pwd -P)

    . "$BIN/functions.sh"

    clear_env

    read_properties_file "${OSE_PROPERTIES:="$BIN/../etc/ose.properties"}"

    init_server_env

    : ${CONSOLE_LOG:="${LOGS_DIR:?}/console.log"}

    for CMD in $@
    do
        # XXX check for flags, e.g. "-config=.."

        if [ "$CMD" = "kill" ] ; then
            kill_ose
        elif [ "$CMD" = "stop" ] ; then
            stop_ose
        elif [ "$CMD" = "test" ] ; then
            test_ose
        else
            case "$CMD" in
                clean    )  clean    ;;
                help     )  usage    ;;
                save     )  save     ;;
                start    )  start    ;;
                print    )  printConfig "$@" ;;
                redeploy )  redeploy ;;
                stats    )  stats    ;;
                status   )  status   ;;
                startpython    ) start_python ;;
                stoppython     ) stop_python ;;
                redeploypython ) redeploy_python ;;
                displaypython  ) display_python ;;
                importMEH   )  importexport "$@"
                            break    ;;
                exportMEH   )  importexport "$@"
                            break    ;;
                purgeMEH    )  purgemeh "$@"
                            break    ;;
                reloadLookupList    )  reload_lookup_list "$@"
                            break    ;;
                alertcache    )  alert_cache "$@"
                            break    ;;

                tail     )  tail -f "$CONSOLE_LOG" ;;
                *        )  unknown_command "$CMD" ;;
            esac
        fi
    done

    if [ $# -eq 0 ] ; then
        usage
    fi
}

function clean
{
    if [ -n "$OSE_PID" ] ; then
        echo "OSE is running.  You must stop OSE before running clean"
        exit
    fi
   # XXX this can remove in-use shared memory and semaphores, leave
   # the ids and the server pid in a file at startup or use named keys

    ME="$(whoami)"

    # "-c" doesn't truncate the uids to 10 chars ("-X" isn't available on Linux)
    IPCS_S=$(ipcs -sc | grep " ${ME:?} " | sed "${IPCS_SED_XPRN:?}")
    IPCS_M=$(ipcs -mc | grep " ${ME:?} " | sed "${IPCS_SED_XPRN:?}")

    for id in $IPCS_M; do
        ipcrm -m $id || :
    done

    for id in $IPCS_S; do
        ipcrm -s $id || :
    done
}

function kill_ose
{
    if [ -n "$OSE_PID" ] ; then
        status
        kill -KILL -${OSE_PGID:?}
        OSE_PID=
        clean
    fi
}

function prune_by_count
{
    integer CNT=0

    for ARCHIVE in $(ls -1t "${ARCHIVE_DIR:?}" | tr '\n' ' ')
    do
        if [ -d "${ARCHIVE_DIR:?}/${ARCHIVE:?}" -a -z "${ARCHIVE#$DT_PATTERN}" ]
        then
            CNT=$CNT+1
            if [ $CNT -gt ${ARCHIVE_COUNT_LIMIT:?} ] ; then
                rm -Rf "${ARCHIVE_DIR:?}/${ARCHIVE:?}"
            fi
        fi
    done
}

function set_DT
{
    if [ $(uname) = "AIX" ] ; then

        LM=$(istat "${1:?}" | grep "Last modified:")
        LM=${LM##*( )}

        # Last modified:  Thu Aug 14 09:07:42 EDT 2014

        MONTH=$(echo "$LM" | sed 's/  / /'|sed 's/\t/ /'|cut -f 4 -d " " | tr 'a-z' 'A-Z')
        DAY=$(  echo "$LM" | sed 's/  / /'|sed 's/\t/ /'|cut -f 5 -d " ")
        TIME=$( echo "$LM" | sed 's/  / /'|sed 's/\t/ /'|cut -f 6 -d " " | tr -d ':')
        ZONE=$( echo "$LM" | sed 's/  / /'|sed 's/\t/ /'|cut -f 7 -d " ")
        YEAR=$( echo "$LM" | sed 's/  / /'|sed 's/\t/ /'|cut -f 8 -d " " | cut -c 3-4)

        if [ ${#DAY} -eq 1 ] ; then
            DAY="0$DAY"
        fi

        if [ -n "$YEAR" ] ; then

            DT="$YEAR$MONTH${DAY}_$TIME$ZONE"

            DT_PATTERN="[0-9][0-9][A-Z][A-Z][A-Z][0-9][0-9]_[0-9][0-9][0-9][0-9][0-9][0-9][A-Z][A-Z][A-Z]"

        else

            # no ZONE, ZONE holds YEAR
            DT="$ZONE$MONTH${DAY}_$TIME"

            DT_PATTERN="[0-9][0-9][A-Z][A-Z][A-Z][0-9][0-9]_[0-9][0-9][0-9][0-9][0-9][0-9]"

        fi

    else

        LM=$(stat -c "%y" "${1:?}")
        LM=${LM##*( )}

        # 2014-08-14 09:11:55.661492414 -0400

        DATE=$(echo $LM | cut -d " " -f 1 | tr -d '-')
        TIME=$(echo $LM | cut -d ' ' -f 2 | cut -d '.' -f 1 | tr -d ':')
        ZONE=$(echo $LM | cut -d ' ' -f 3)

        DT="${DATE}_$TIME$ZONE"

        DT_PATTERN="[0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9]_[0-9][0-9][0-9][0-9][0-9][0-9][+-][0-9][0-9][0-9][0-9]"

    fi
}

function save
{
    set -e

    LM_FILE=$(ls -1t "${LOGS_DIR:?}" | tr '\n' ' ' | cut -d ' ' -f 1)

    if [ -z "$LM_FILE" ] ; then
        return
    fi

    set_DT "${LOGS_DIR:?}/$LM_FILE"

    prune_by_count

    # TODO add prune_by_time and prune_by_footprint

    SAVE_DIR="${ARCHIVE_DIR:?}/${DT:?}"

    if [ ! -d "${SAVE_DIR}" ] ; then
        mkdir "${SAVE_DIR}"
    else
        chmod -f u+w "${SAVE_DIR}/"*
    fi

    if [ -n "$OSE_PID" ] ; then
        touch "${SAVE_DIR}/STILL_RUNNING"
		$RSYNC "${LOGS_DIR}/"* "${SAVE_DIR}"
    else
        rm -f "${SAVE_DIR}/STILL_RUNNING"
        mv "${LOGS_DIR}/"* "${SAVE_DIR}"
    fi

    chmod -f a-w "${SAVE_DIR}/"*
}

function bundle_env
{
    # XXX i don't think set -e catches pipeline errors

    # *assumes* etc and bin are the config and program dir...

    tar -cf - -C "${SERVER_HOME:?}" \
    $(cd ${SERVER_HOME:?}; ls -1 bin/*.sh etc/*.xml etc/*.properties | tr '\n' ' ') \
    | gzip -c >"${LOGS_DIR:?}/env.tgz"

    # TODO if its the same as the one in the most recent archive, use a hard link
}

function start
{
    set -e

    if [ -n "$OSE_PID" ] ; then
        echo "already running"
        exit
    fi

    save

    rm -Rf "${LOGS_DIR:?}/"*

    cd "${RUN_DIR:="${LOGS_DIR}"}"

    bundle_env

    export LOGS_DIR="${LOGS_DIR}"

    CMD="${JAVA:?}"
    CMD="$CMD --add-opens=java.base/java.nio=ALL-UNNAMED --add-opens=java.base/sun.nio.ch=ALL-UNNAMED --add-opens=java.base/java.util=ALL-UNNAMED --add-opens=java.base/java.lang=ALL-UNNAMED --add-opens=java.base/java.util.concurrent=ALL-UNNAMED"
    CMD="$CMD -DIGNITE_NO_ASCII=true"
    # Uncomment to use the default Ignite logger or point to the custom one
    #CMD="$CMD -Djava.util.logging.config.file=${JAVA_HOME:?}/conf/logging.properties"
    
    # accommodate model scores
    CMD="$CMD -Djava.lang.Integer.IntegerCache.high=1000"

    CMD="$CMD -Dlog4j.configurationFile=${LOG4J_CONFIG:?}"

    # comment this out to get log4j logs in local time
    CMD="$CMD -Duser.timezone=UTC"

    CMD="$CMD -Xmx${JAVA_HEAP:?}"
    CMD="$CMD -Xms${JAVA_HEAP:?}"
    CMD="$CMD -Xmn${JAVA_YOUNG_HEAP:?}"

    if [ "${MODE:?}" != "PROD" -o -n "$DEBUG_PORT" ]; then
        OPTS="$OPTS $(debug_opts)"
    fi

#   OPTS="$OPTS -verbose:class"
#   OPTS="$OPTS -verbose:jni"

    # http://docs.oracle.com/javase/7/docs/technotes/guides/rmi/javarmiproperties.html
    if [ -n "$LOG_JMX_CALLS" ] ; then
        OPTS="$OPTS -Djava.rmi.server.logCalls=$LOG_JMX_CALLS"
    fi

    # XXX what's the overhead, should it be disabled for PROD?
    OPTS="$OPTS -verbose:gc"

    # these are Oracle JVM specific but should be ignored by the IBM JVM
    #OPTS="$OPTS -XX:+PrintGCDateStamps"
    # OPTS="$OPTS -XX:+PrintGCDetails" (small performance penalty)

    # Uncomment the following line when enabling SSL
    # OPTS="$OPTS -Djavax.net.ssl.keyStore=<fully qualified path to keystore>"

    # Enable SSL debugging
    # OPTS="$OPTS -Djavax.net.debug=all"

    CMD="$CMD $OPTS"

    CMD="$CMD ${OSE_CLASS:?} -config ${OSE_XML:?} ${OSE_BEAN:?}"

    echo >"${CONSOLE_LOG:?}"

    date >>"${CONSOLE_LOG:?}"
    echo >>"${CONSOLE_LOG:?}"

    ${JAVA:?} -version >>"${CONSOLE_LOG:?}" 2>&1
    echo >>"${CONSOLE_LOG:?}"

    echo "nohup $CMD </dev/null 2>&1 &" >>"${CONSOLE_LOG:?}"
    echo >>"${CONSOLE_LOG:?}"

    nohup $CMD </dev/null >>"${CONSOLE_LOG:?}" 2>&1 &
}

function redeploy
{
    set -e

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.RestartScoringEngines -config ${OSE_XML:?} ${OSE_BEAN:?}"

    $CMD
}

function printConfig
{
    set -e

    shift
    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.PrintConfig -config ${OSE_XML:?} ${OSE_BEAN:?} $@"

    $CMD
}

function stats
{
    set -e

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.DumpStats -config ${OSE_XML:?} ${OSE_BEAN:?}"

    $CMD
}

function status
{
    if [ -n "$OSE_PID" ]; then
        echo
        echo "$OSE_PID $OSE_ARGS"
        echo
        ${PSTREE:?} $OSE_PID
        echo
    fi
}

function stop_ose
{
    if [ -z "$OSE_PID" ]; then
        return
    fi

    status

    # XXX capture the children pids so they can be checked
    # for and waxed if necessary

    echo "kill -s TERM $OSE_PID"
    kill -s TERM $OSE_PID

    while [ `ps -o pid= -p $OSE_PID | wc -l` -ne 0 ] ; do
       sleep 1
       ${ECHO_N:?} ".\c"
    done

    echo " done"

    OSE_PID=
}

function stop_python
{
    set -e

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.StopPython -config ${OSE_XML:?} ${OSE_BEAN:?}"

    $CMD
}

function start_python
{
    set -e

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.StartPython -config ${OSE_XML:?} ${OSE_BEAN:?}"

    $CMD
}

function redeploy_python
{
    set -e

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.RedeployPython -config ${OSE_XML:?} ${OSE_BEAN:?}"

    $CMD
}

function display_python
{
    set -e

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.DisplayPython -config ${OSE_XML:?} ${OSE_BEAN:?}"

    $CMD
}


function test_ose
{
    set -e

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.PingConnectors -config ${OSE_XML:?} ${OSE_BEAN:?}"

    $CMD
}

function importexport
{
    set -e
    if [[ $1 != "importMEH" && $1 != "exportMEH" ]] ; then
      echo "importMEH/exportMEH cannot be combined with another command, Usage: ${PROGRAM_NAME:?} [importMEH|exportMEH] <directory>"
      exit
    fi

    if [ $# -lt 2 ] ; then
      echo "A directory was not specified, Usage: ${PROGRAM_NAME:?} $1 <directory>"
      exit
    elif [ $# -gt 2 ] ; then
      echo "Too many parameters specified, Usage: ${PROGRAM_NAME:?} $1 <directory>"
      exit
    fi

    if [ ! -d $2 ]; then
      echo "$2 is not a directory, Usage: ${PROGRAM_NAME:?} $1 <directory>"
      exit
    fi

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.MEHImportExport $1 $2 -config ${OSE_XML:?} ${OSE_BEAN:?}"

    $CMD
}

function purgemeh
{
    set -e
    if [[ $1 != "purgeMEH" ]] ; then
      echo "purgeMEH cannot be combined with another command, Usage: ${PROGRAM_NAME:?} purgeMEH 'table=table name multiorg=moid purgeday=days rmax=maxrecs smax=maxsec segment=vv'"
      exit
    fi

    if [ $# -lt 2 ] ; then
      echo "An argument string was not specified, Usage: ${PROGRAM_NAME:?} $1 'table=table name multiorg=moid purgeday=days rmax=maxrecs smax=maxsec segment=vv' "
      exit
    elif [ $# -gt 2 ] ; then
      echo "Too many parameters specified, Usage: ${PROGRAM_NAME:?} $1 'table=table name multiorg=moid purgeday=days rmax=maxrecs smax=maxsec segment=vv' "
      exit
    fi

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.MEHPurge"
    # Need to break up the command like this so that we can send over all of the arguments $2 in quotes for JAVA
    $CMD "$2" -config ${OSE_XML:?} ${OSE_BEAN:?}
}

function reload_lookup_list
{
    set -e
    if [[ $1 != "reloadLookupList" ]] ; then
      echo "reloadLookupList cannot be combined with another command, Usage: ${PROGRAM_NAME:?} reloadLookupList <LookupListName>"
      exit
    fi

    if [ $# -lt 3 ] ; then
      echo "A lookup list name was not specified, Usage: ${PROGRAM_NAME:?} $1 reloadLookupList <SOR_ID> <LookupListName>"
      exit
    elif [ $# -gt 3 ] ; then
      echo "Too many parameters specified, Usage: ${PROGRAM_NAME:?} $1 reloadLookupList <SOR_ID> <LookupListName>"
      exit
    fi

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.ReloadLookupList"
    $CMD LUP_$2_$3 -config ${OSE_XML:?} ${OSE_BEAN:?}
}

function alert_cache
{
    set -e
    if [[ $1 != "alertcache" ]] ; then
      echo "alertcache cannot be combined with another command, Usage: ${PROGRAM_NAME:?} alertcache clear"
      exit
    fi

    if [ $# -lt 2 ] ; then
      echo "A command was not specified, Usage: ${PROGRAM_NAME:?} $1 clear"
      exit
    elif [ $# -gt 3 ] ; then
      echo "Too many parameters specified, Usage: ${PROGRAM_NAME:?} $1 clear"
      exit
    fi

    if [ -z "$OSE_PID" ] ; then
        echo "OSE not running"
        exit
    fi

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.engine.tools.AlertCache  $2 -config ${OSE_XML:?} ${OSE_BEAN:?}"
    
    $CMD
}

function usage
{
    echo "Usage: ${PROGRAM_NAME:?} [ ${COMMANDS:?} ]"
}

main "$@"


