// sas.finserv.creditfraud.webapp/Source/content/rules/scripts/sort.js

var workflow;
function getSortCol() {
	if (typeof(workflow) === undefined) {
		workflow = "none";
	}
	if (!isDefined(parent._rulesSortOrder)) {
		parent._rulesSortOrder = {};
	}
	if(!isDefined(parent._rulesSortOrder[workflow])) {
		setSortColumn(0, "desc");
	}
	return parent._rulesSortOrder[workflow];
}
function getSortOrder()
{
   if (isDefined(parent._sortOrder) && isDefined(parent._sortOrder[currentCol]))
      return parent._sortOrder[currentCol].colId + " " + parent._sortOrder[currentCol].order;
   else
      return "";
}

// index is 0-based
function setSortColumn(index, sortOrder) {
	if (workflow != undefined) {	
	parent._rulesSortOrder[workflow] = {
			index : index,
			sortOrder : sortOrder
		}
	}
}

function sortRules(tableId) {
	var table = getElement(tableId);
	if (!table) {
		return; // handle zero rules condition
	}

	var sortColumn = getSortCol();
	var columnIndex = sortColumn.index;
	var headerRow = table.tHead.rows[0];    
	var headerCell = headerRow.cells[columnIndex];        
	var dataType = headerCell.getAttribute("famSortType");                 
	
	sortTable(tableId, columnIndex, dataType);
	if(sortColumn.sortOrder == "desc") { // need to find a better way to do desc sorting
		sortTable(tableId, columnIndex, dataType);               
	}
}

// Parameters:
//    tableId:  id of the table to be sorted
//    columnIndex: 0-based column index
//    dataType: 'int', 'float', 'date';  Optional, defaults to text
function sortTable(tableId, columnIndex, dataType) {
	var table = document.getElementById(tableId);
	var tBody = table.tBodies[0];
	var newTRs = [];
	var isEmpty = (field) => field === "";
	var columnToBeSorted = [];
	var newTRsIsEmpty;
	var sameData = (field) => field === tBody.rows[0].cells[columnIndex].textContent;
	var columnHasSameData;

	for (var i = 0, n = tBody.rows.length; i < n; i++) {
		newTRs[i] = tBody.rows[i];
		columnToBeSorted.push(newTRs[i].cells[columnIndex].textContent);
	}
	newTRsIsEmpty = columnToBeSorted.every(isEmpty);
	columnHasSameData = columnToBeSorted.every(sameData);
	var sortOrder = "asc";   
	if (table.sortedColumnIndex == columnIndex) {
		if (!newTRsIsEmpty) {
			if (!columnHasSameData) {
				newTRs.reverse();
			}
		}
      
		// special case to handle already selected column in workflow
		var workflowSortColumn = getSortCol();
		if(isDefined(workflowSortColumn) && (workflowSortColumn.index == columnIndex)) {   
			sortOrder = (workflowSortColumn.sortOrder == "desc") ? "asc" : "desc";         
		} else {
			sortOrder = "desc";
		}
	} else {
		if (dataType == "ruleExec") { // S1392138
			newTRs = sortByRuleExecution(newTRs, columnIndex);
		} else {
			newTRs.sort(getComparator(columnIndex, dataType));
		}
	}

	var fragment = document.createDocumentFragment();
	for (var i = 0, n = newTRs.length; i < n; i++) {
		fragment.appendChild(newTRs[i]);
	}
	tBody.appendChild(fragment);
	table.sortedColumnIndex = columnIndex;
	setSortColumn(columnIndex, sortOrder);
	alternateRowClassNames(tableId, "dataRow2_tableRow", "dataRow1_tableRow");
}

function sortByRuleExecution(tableRows, columnIndex) {
	var ruleMapping = {};  // Map<Rule ID, tr>
	var rules = [];
	for (let key in tableRows) {
		let tr = tableRows[key];
		let td = tr.cells[columnIndex];
		// Eg. data-rule="{"id":${rule.id},"typeName":"${rule.ruleTypeName}","execType":${rule.executionOrderType},"execPriority":${rule.executionOrderPriority}"
		let ruleStr = td.dataset.rule;
		let ruleJson = JSON.parse(ruleStr);
		let rule = new Rule(ruleJson);

		ruleMapping[rule.getId()] = tr;
		rules.push(rule);
	}

	var sortedRules = new RuleExectionSorter(rules).sort();
	// IE11 doesn't support arrow functions!
	//var answer = sortedRules.map(each => ruleMapping[each.getId()]);
	var answer = [];
	sortedRules.forEach(function (each) {
		answer.push(ruleMapping[each.getId()]);
	} );
	return answer;
}

function getComparator(col, dataType) {
	return function comparator(tr1, tr2) {
		var value1, value2;
		if (dataType && dataType=='localDate') {
			value1 = (tr1.cells[col]) ? convert(tr1.cells[col], dataType) : '';
			value2 = (tr2.cells[col]) ? convert(tr2.cells[col], dataType) : '';
		} else {
			value1 = (tr1.cells[col].firstChild != null) ? convert(tr1.cells[col].firstChild.nodeValue, dataType) : '';
			value2 = (tr2.cells[col].firstChild != null) ? convert(tr2.cells[col].firstChild.nodeValue, dataType) : '';
		}
		return (value1 < value2) 
			? -1 
			: ((value1 > value2) ? 1 : 0);
	};
}

// expects date and time in the format mm/dd/yyyy hh:mm:ss a
// 10/12/2006 12:05:23 PM 
// cache date times for efficiency
var dateTimeMap = new Array(); 

function convert(value, dataType) {
	switch(dataType) {
		case "num":
			return  parseFloat(value.replace(',',''));
		case "date":
			return new Date(2000+parseInt(value.substr(value.lastIndexOf('/')+1)), value.substring(0,value.indexOf('/'))-1, value.substring(value.indexOf('/')+1,value.lastIndexOf('/')));
		case "time":
			return new Date(2000,1,1,parseFloat(value.substring(0,value.indexOf(':'))),parseFloat(value.substring(value.indexOf(':')+1,value.lastIndexOf(':'))),parseFloat(value.substr(value.lastIndexOf(':')+1)));
		case "dateTime":
			var dateTime = dateTimeMap[value];
			if((dateTime == null) || (dateTime === undefined)) {         
				var dateTime = new Date(2000+parseInt(value.substr(value.lastIndexOf('/')+1)), 
						value.substring(0,value.indexOf('/'))-1, 
						value.substring(value.indexOf('/')+1,value.lastIndexOf('/')),
						parseFloat(value.substring( value.indexOf(' '),value.indexOf(':'))),
						parseFloat(value.substring(value.indexOf(':')+1,value.lastIndexOf(':'))),
						parseFloat(value.substr(value.lastIndexOf(':') +1, value.lastIndexOf(':') + 3 )) );
          
				if((value.indexOf("PM") != -1) || (value.indexOf("pm") != -1)) {            
					dateTime.setHours(dateTime.getHours()%12 + 12);         
				}
            	if(isNaN(dateTime)) {// handle null values
            		dateTime = ""; 
            	}
            	dateTimeMap[value] = dateTime;
			}
			return dateTime;
		case "localDate":
			var sts = value.getElementsByTagName("SPAN");
			if (sts && sts.length > 0) {
				var element = sts[0];
				var valueStr = element.getAttribute("sortvalue");
				value = parseInt(valueStr);
				return value;
			}
			return 0;
		default:
			return (!value) ? "" : value.toString().toUpperCase();
	}
}

function attachEventHandlerToTableCells(tableIds) {
	for(var key in tableIds) {
		enableSortableColumns(tableIds[key]);
	}
}

function enableSortableColumns(tableId) {
	var table = getElement(tableId);
	if (!table || !table.tHead) {
		return;
	}
	// Event handler
	var clickAction = function() {
		var dataType = this.parentNode.getAttribute("famSortType");
		sortTable(tableId, this.parentNode.cellIndex, dataType);
		return false;
	};

	var headerRow = table.tHead.rows[0];
	var columnCount = headerRow.cells.length;

	for(var i = 0; i < columnCount; i++) {
		var headerCell = headerRow.cells[i];
		attachColumnSortHandler(headerCell, clickAction);
	}
}

const sortAnchor = (function () {
	var sortAnchor = document.createElement("A");
	sortAnchor.className="sort";
	sortAnchor.href = "#";
	return sortAnchor;
} )();

function attachColumnSortHandler(headerCell, clickHandler) {
	var sortDataType = getSortDataType(headerCell);
	if(sortDataType && headerCell.hasChildNodes()) {
		headerCell.setAttribute("famSortType", sortDataType);
		var oldNode = headerCell.firstChild;
		var newNode = sortAnchor.cloneNode(true);
		newNode.appendChild(oldNode.cloneNode(false));
		newNode.onclick = clickHandler;
		headerCell.replaceChild(newNode, oldNode);
	}
}

function getSortDataType(headerCell) {
	var answer = null;
	if (headerCell.getAttribute("rules:sortAsText") != null) {
		answer = "text";
	} else if (headerCell.getAttribute("rules:sortAsNum") != null) {
		answer = "num";
	} else if (headerCell.getAttribute("rules:sortAsDate") != null) {
		answer = "date";
	} else if (headerCell.getAttribute("rules:sortAsTime") != null) {
		answer = "time";
	} else if (headerCell.getAttribute("rules:sortAsLocalDate") != null) {
		answer = "localDate";
	} else if (headerCell.getAttribute("rules:sortAsDateTime") != null) {
		answer = "dateTime";
	} else if (headerCell.getAttribute("rules:sortAsRuleExec") != null) {
		answer = "ruleExec";
	}
	return answer;
}

function alternateRowClassNames(tableId, oddClass, evenClass) {
	var table = document.getElementById(tableId);
	var rows = table.tBodies[0].rows;
	for (var i = 0, n = rows.length; i < n; i++) {
		rows[i].className = (i % 2) ? evenClass : oddClass;
	}
}

//Sort rules by execution order and priority
/*
 * Note:
 * From RuleType.java
 * public static final Integer RULE_TYPE_VAR_ID = 201;
 * public static final Integer RULE_TYPE_AUTH_ID = 202;
 * public static final Integer RULE_TYPE_QUEUE_ID = 203;
 * 
 * From RuleItemImpl.java
 * public static final Integer PRE_RULE_EXECUTION_ORDER_TYPE = -1;
 * public static final Integer MAIN_RULE_EXECUTION_ORDER_TYPE = 0;
 * public static final Integer POST_RULE_EXECUTION_ORDER_TYPE = 1;
 * 
 * rules - An array of JSONs
 * {"id":<int>,"typeType":<String>,"execType":<int>,"execPriority":<int>}
 */
const RuleType = {
	VARIABLE : 201,
	AUTHORIZATION : 202,
	QUEUE : 203,
};
const RuleExecutionOrder = {
	PRE : -1,
	MAIN : 0,
	POST : 1,
};

var RuleExectionSorter = (function () {
	function RuleExectionSorter (rules) {
		var rules = rules;
		this.getRules = function () {return rules;};
	}
	
	// Sort order: pre-variable, variable, post-variable, pre-auth, auth, post-auth, pre-queue, queue, and post-queue.
	RuleExectionSorter.prototype.sort = function () {
		var answer = [];
		// Separate the rules by rule type: authorization, queue, variable.
		var rulesMappingByType = this.getRulesMappingByType();
		// Rule type execution order: variable -> authorization -> queue	
		var rulesByTypeExecutionOrder = [
			rulesMappingByType[RuleType.VARIABLE],
			rulesMappingByType[RuleType.AUTHORIZATION], 
			rulesMappingByType[RuleType.QUEUE] ];
		// Sort the rules of the same rule type by execution order and priority.
		for (let i = 0, n = rulesByTypeExecutionOrder.length; i < n; i++) {
			var rulesOfSameType = rulesByTypeExecutionOrder[i];
			var rulesOfSameTypeByExecOrderAndPriority = this.sortWithinRuleType(rulesOfSameType);
			answer.push.apply(answer, rulesOfSameTypeByExecOrderAndPriority);
		};
		return answer;
	};
	
	// All rules are expected to have the same rule type.
	// Rule execution order within rule type: pre -> main -> post.
	RuleExectionSorter.prototype.sortWithinRuleType = function (rules) {
		var answer = [];
		var rulesMappingByExecOrder = this.getRulesMappingByExecOrder(rules);
		
		var preRules = rulesMappingByExecOrder[RuleExecutionOrder.PRE];
		var preRulesSortedByPriority = this.sortRulesByPriority(preRules);
		answer.push.apply(answer, preRulesSortedByPriority);
		
		var mainRules = rulesMappingByExecOrder[RuleExecutionOrder.MAIN];
		answer.push.apply(answer, mainRules);
		
		var postRules = rulesMappingByExecOrder[RuleExecutionOrder.POST];
		var postRulesSortedByPriority = this.sortRulesByPriority(postRules);
		answer.push.apply(answer, postRulesSortedByPriority);
		
		return answer;
	};
	
	// All rules are expected to have the same execution order.
	RuleExectionSorter.prototype.sortRulesByPriority = function (rules) {
		return rules.sort(function (obj1, obj2) {
		  	return obj1.getExecPriority() - obj2.getExecPriority();
		} );
	}
	
	// All rules are expected to have the same rule type.
	RuleExectionSorter.prototype.getRulesMappingByExecOrder = function (rules) {
		var answer = this.getNewRulesMappingByExecOrder();
		rules.forEach(function (rule) {
			switch (rule.getExecOrder()) {
				case RuleExecutionOrder.MAIN:
					answer[RuleExecutionOrder.MAIN].push(rule);
					break;
				case RuleExecutionOrder.POST:
					answer[RuleExecutionOrder.POST].push(rule);
					break;
				case RuleExecutionOrder.PRE:
					answer[RuleExecutionOrder.PRE].push(rule);
					break;
				default:
			}
		} );
		return answer;
	};
	
	RuleExectionSorter.prototype.getRulesMappingByType = function () {
		var answer = this.getNewRulesMappingByType();
		this.getRules().forEach(function (rule) {
			switch (rule.getTypeCode()) {
				case RuleType.AUTHORIZATION:
					answer[RuleType.AUTHORIZATION].push(rule);
					break;
				case RuleType.QUEUE:
					answer[RuleType.QUEUE].push(rule);
					break;
				case RuleType.VARIABLE:
					answer[RuleType.VARIABLE].push(rule);
					break;
				default:
			}
		} );
		return answer;
	};
	
	RuleExectionSorter.prototype.getNewRulesMappingByExecOrder = function () {
		var answer = {};
		answer[RuleExecutionOrder.MAIN] = [];
		answer[RuleExecutionOrder.POST] = [];
		answer[RuleExecutionOrder.PRE] = [];
		return answer;	
	};
	
	RuleExectionSorter.prototype.getNewRulesMappingByType = function () {
		var answer = {};
		answer[RuleType.AUTHORIZATION] = [];
		answer[RuleType.QUEUE] = [];
		answer[RuleType.VARIABLE] = [];
		return answer;
	};
	return RuleExectionSorter;
}() );

var Rule = (function () {
	function Rule (json) {  // eg. {id:123, typeName:"variable", execOrder:-1, execPriority:2}
		var json = json;
		
		this.getJson = function () {return json;};
	}
	
	// Assist debugging
	Rule.prototype.getBaseRuleId = function () {
		return this.getJson().baseRuleId;
	};
	
	Rule.prototype.getExecOrder = function () {
		return this.getJson().execOrder;
	};
	
	Rule.prototype.getExecPriority = function () {
		return this.getJson().execPriority;
	};
	
	Rule.prototype.getId = function () {
		return this.getJson().id;
	};
	
	Rule.prototype.getTypeCode = function () {
		return RuleType[this.getTypeName().toUpperCase()];
	};
	
	Rule.prototype.getTypeName = function () {
		return this.getJson().typeName;
	};
	
	return Rule;
}() );
