/*!
 * (c) 2014-2018, SAS Institute Inc.
 */

// Provides control sas.hc.ui.table.TreeTable.
sap.ui.define([
    "jquery.sap.global",
    "./Table",
    "sap/ui/model/ClientTreeBindingAdapter",
    "sas/hc/ui/model/PageableJSONTreeBindingAdapter",
    "sap/ui/core/Element",
    "./TableUtils",
    "./library",
    "sap/ui/commons/library",
    "./TriStateCheckBox",
    "./TriStateSelectAll",
    "./TreeSelectionModel"
],
function(
    jQuery,
    Table,
    ClientTreeBindingAdapter,
    PageableJSONTreeBindingAdapter,
    Element,
    TableUtils,
    library,
    SapCommonsLibrary,
    TriStateCheckBox,
    TriStateSelectAll,
    TreeSelectionModel) {
    "use strict";

    //var logger = library._getLogger("TreeTable");
    var SelectionMode = library.SelectionMode;
    var SelectionBehavior = library.SelectionBehavior;
    var TriStateCheckBoxState = SapCommonsLibrary.TriStateCheckBoxState;

    /**
     * Constructor for a new TreeTable.
     *
     * @param {string} [sId] id for the new control, generated automatically if no id is given
     * @param {object} [mSettings] initial settings for the new control
     *
     * @class
     * The TreeTable control provides a comprehensive set of features to display hierarchical data.
     * @extends sas.hc.ui.table.Table
     * @version 904001.11.16.20251118090100_f0htmcm94p
     *
     * @constructor
     * @public
     * @alias sas.hc.ui.table.TreeTable
     * @ui5-metamodel This control/element also will be described in the UI5 (legacy) designtime metamodel
     */
    var TreeTable = Table.extend("sas.hc.ui.table.TreeTable", /** @lends sas.hc.ui.table.TreeTable.prototype */ { metadata : {

        library : "sas.hc.ui.table",
        properties : {

            /**
             * Flag to enable or disable expanding of first level.
             */
            expandFirstLevel : {type : "boolean", defaultValue : false},

            /**
             * If group mode is enable nodes with subitems are rendered as if they were group headers.
             * This can be used to do the grouping for an OData service on the backend and visualize this in a table.
             * This mode only makes sense if the tree has a depth of exacly 1 (group headers and entries)
             */
            useGroupMode : {type : "boolean", group : "Appearance", defaultValue : false},

            /**
             * The property name of the rows data which will be displayed as a group header if the group mode is enabled
             */
            groupHeaderProperty : {type : "string", group : "Data", defaultValue : null},

            /**
             * Setting collapseRecursive to true means, that when collapsing a node all subsequent child nodes will also be collapsed.
             * This property is only supported with sap.ui.model.odata.v2.ODataModel
             */
            collapseRecursive : {type: "boolean", defaultValue: true},

            /** Indent for each subnode. Default: <code>17</code>
             * @private
             */
            "_subnodeIndent": {type: "int", group: "Dimension", defaultValue: 17}

        },
        aggregations: {
            treeSelectionModel: {
                type: "sas.hc.ui.table.TreeSelectionModel",
                multiple: false
            }
        },
        events : {

            /**
             * fired when a node has been expanded or collapsed (only available in hierachical mode)
             */
            toggleOpenState : {
                parameters : {

                    /**
                     * index of the expanded/collapsed row
                     */
                    rowIndex : {type : "int"},

                    /**
                     * binding context of the selected row
                     */
                    rowContext : {type : "object"},

                    /**
                     * flag whether the node has been expanded or collapsed
                     */
                    expanded : {type : "boolean"}
                }
            }
        }
    }, renderer: "sas.hc.ui.table.TreeTableRenderer"});

    /**
     * Initialization of the TreeTable control
     * @private
     */
    TreeTable.prototype.init = function() {
        Table.prototype.init.apply(this, arguments);
        TableUtils.Grouping.setTreeMode(this);
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    TreeTable.prototype.bindRows = function(oBindingInfo, vTemplate, aSorters, aFilters) {
        var sPath,
            oTemplate,
            aSorters,
            aFilters;

        // Old API compatibility (sName, sPath, oTemplate, oSorter, aFilters)
        if (typeof oBindingInfo == "string") {
            sPath = arguments[0];
            oTemplate = arguments[1];
            aSorters = arguments[2];
            aFilters = arguments[3];
            oBindingInfo = {path: sPath, sorter: aSorters, filters: aFilters, template: oTemplate};
        }

        if (typeof oBindingInfo === "object") {
            oBindingInfo.parameters = oBindingInfo.parameters || {};
            oBindingInfo.parameters.collapseRecursive = this.getCollapseRecursive();
            // number of expanded levels is taken from the binding parameters first,
            // if not found, we check if they are set on the table
            oBindingInfo.parameters.numberOfExpandedLevels = oBindingInfo.parameters.numberOfExpandedLevels || (this.getExpandFirstLevel() ? 1 : 0);
            oBindingInfo.parameters.rootNodeID = oBindingInfo.parameters.rootNodeID;
        }

        var oResult = this.bindAggregation("rows", oBindingInfo);

        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            this._setupParentChildSelectionMode();
        }

        return oResult;
    };

    /**
     * Sets the selection mode. The current selection is lost.
     * @param {string} sSelectionMode the selection mode, see sas.hc.ui.table.SelectionMode
     * @public
     * @return a reference on the table for chaining
     */
    TreeTable.prototype.setSelectionMode = function (sSelectionMode) {

        var sPreviousSelectionMode = this.getSelectionMode();

        if (sSelectionMode === SelectionMode.ParentChild) {

            if (sPreviousSelectionMode !== SelectionMode.ParentChild) {
                this._resetRowTemplate();
            }

            this.clearSelection();
            this.setProperty("selectionMode", sSelectionMode);
            this._setupParentChildSelectionMode();
            return this;
        }

        if (sPreviousSelectionMode === SelectionMode.ParentChild) {
            this._resetRowTemplate();
        }

        var oBinding = this.getBinding("rows");
        if (oBinding && oBinding.clearSelection) {
            oBinding.clearSelection();
            this.setProperty("selectionMode", sSelectionMode);

            if (sSelectionMode === SelectionMode.Single) {
                this.setSelectionBehavior(SelectionBehavior.RowOnly);
            }
        } else {
            Table.prototype.setSelectionMode.call(this, sSelectionMode);
        }
        this._recreateSelectAllControl();
        return this;
    };

    TreeTable.prototype._setupParentChildSelectionMode = function() {
        var oCurrentModel = this.getTreeSelectionModel();
        oCurrentModel && oCurrentModel.destroy();

        var oRowBinding = this.getBinding("rows");
        if (oRowBinding) {
            var oTreeSelectionModel = new TreeSelectionModel(oRowBinding);
            this.setTreeSelectionModel(oTreeSelectionModel);

            oTreeSelectionModel.attachEvent("selectionChanged", this._onTreeSelectionModelSelectionChanged.bind(this));
        }

        this._recreateSelectAllControl();
    };

    TreeTable.prototype._onTreeSelectionModelSelectionChanged = function(oEvent) {
        var mTooltipTexts = this._getAccExtension().getAriaTextsForSelectionMode(true);
        var bSelectOnCellsAllowed = TableUtils.isRowSelectionAllowed(this);
        this._refreshSelectAllControl();
        this._refreshSelectionControls();

        var self = this;
        this.getRows().forEach(function(oRow) {
            oRow._updateSelection(self, mTooltipTexts, bSelectOnCellsAllowed);
        });

        var iRowIndex = this._iSourceRowIndex !== undefined ? this._iSourceRowIndex : this.getSelectedIndex();
        var oContexts = oEvent.getParameter("contexts");

        this.fireRowSelectionChange({
            rowIndex: iRowIndex,
            rowContext: this.getContextByIndex(iRowIndex),
            rowIndices: this._getRowIndicesFromContexts(oContexts),
            selectAll: undefined,
            userInteraction: undefined
        });
    };

    /**
     * Setter for property <code>fixedRowCount</code>.
     *
     * <b>This property is not supportd for the TreeTable and will be ignored!</b>
     *
     * Default value is <code>0</code>
     *
     * @param {int} iFixedRowCount  new value for property <code>fixedRowCount</code>
     * @return {sas.hc.ui.table.TreeTable} <code>this</code> to allow method chaining
     * @public
     */
    TreeTable.prototype.setFixedRowCount = function(iRowCount) {
        // this property makes no sense for the TreeTable
        jQuery.sap.log.warning("TreeTable: the property \"fixedRowCount\" is not supported and will be ignored!");
        return this;
    };

    /**
     * Setter for property <code>fixedBottomRowCount</code>.
     *
     * <b>This property is not supportd for the TreeTable and will be ignored!</b>
     *
     * Default value is <code>0</code>
     *
     * @param {int} iFixedRowCount  new value for property <code>fixedRowCount</code>
     * @return {sas.hc.ui.table.TreeTable} <code>this</code> to allow method chaining
     * @public
     */
    TreeTable.prototype.setFixedBottomRowCount = function(iRowCount) {
        // this property makes no sense for the TreeTable
        jQuery.sap.log.warning("TreeTable: the property \"fixedBottomRowCount\" is not supported and will be ignored!");
        return this;
    };

    /**
     * @override
     */
    TreeTable.prototype.isTreeBinding = function(sName) {
        sName = sName || "rows";
        if (sName === "rows") {
            return true;
        }
        return Element.prototype.isTreeBinding.apply(this, arguments);
    };

    /**
     * @override
     */
    TreeTable.prototype.getBinding = function(sName) {
        sName = sName || "rows";
        var oBinding = Element.prototype.getBinding.call(this, sName);

        if (oBinding && sName === "rows" && !oBinding.getLength) {

            var PageableJSONTreeBinding = sap.ui.require("sas/hc/ui/model/PageableJSONTreeBinding");
            var ClientTreeBinding = sap.ui.require("sap/ui/model/ClientTreeBinding");

            if (PageableJSONTreeBinding && oBinding instanceof PageableJSONTreeBinding) {
                PageableJSONTreeBindingAdapter.apply(oBinding);
            } else if (ClientTreeBinding && oBinding instanceof ClientTreeBinding) {
                ClientTreeBindingAdapter.apply(oBinding);
            } else {
                jQuery.sap.log.error("Binding not supported by sas.hc.ui.table.TreeTable");
            }
        }

        return oBinding;
    };

    /**
     * @private
     */
    TreeTable.prototype._updateTableContent = function() {
        var oBinding = this.getBinding("rows"),
            aRows = this.getRows(),
            iCount = aRows.length;

        if (oBinding) {
            var oRowBindingInfo = this.getBindingInfo("rows"),
                sModelName = oRowBindingInfo && oRowBindingInfo.model,
                oRow, sGroupTitle;

            for (var iRow = 0; iRow < iCount; iRow++) {
                oRow = aRows[iRow];
                sGroupTitle = TableUtils.Grouping.isGroupMode(this) && this.getGroupHeaderProperty() ? this.getModel(sModelName).getProperty(this.getGroupHeaderProperty(), oRow.getBindingContext(sModelName)) : "";

                TableUtils.Grouping.updateTableRowForGrouping(this, oRow, oRow._bHasChildren, oRow._bIsExpanded,
                    oRow._bHasChildren, false, oRow._iLevel, sGroupTitle);
            }

            this._updateRowHeader(this._collectRowHeights());

        } else {
            for (var iRow = 0; iRow < iCount; iRow++) {
                TableUtils.Grouping.cleanupTableRowForGrouping(this, aRows[iRow]);
            }
        }
    };

    /**
     * @private
     */
    TreeTable.prototype._getContexts = function(iStartIndex, iLength, iThreshold) {
        var oBinding = this.getBinding("rows");
        if (oBinding) {
            // first call getContexts to trigger data load but return nodes instead of contexts
            return oBinding.getNodes(iStartIndex, iLength, iThreshold);
        } else {
            return [];
        }
    };

    /**
     * @private
     */
    TreeTable.prototype._onGroupHeaderChanged = function(iRowIndex, bExpanded) {
        this.fireToggleOpenState({
            rowIndex: iRowIndex,
            rowContext: this.getContextByIndex(iRowIndex),
            expanded: bExpanded
        });
    };

    /**
     * expands the row for the given row index
     *
     * @param {int} iRowIndex
     *         index of the row to expand
     * @return {sas.hc.ui.table.TreeTable} a reference on the TreeTable control, can be used for chaining
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.expand = function(iRowIndex) {
        var oBinding = this.getBinding("rows");
        if (oBinding && iRowIndex >= 0) {
            oBinding.expand(iRowIndex);
        }

        return this;
    };

    /**
     * collapses the row for the given row index
     *
     * @param {int} iRowIndex
     *         index of the row to collapse
     * @return {sas.hc.ui.table.TreeTable} a reference on the TreeTable control, can be used for chaining
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.collapse = function(iRowIndex) {
        var oBinding = this.getBinding("rows");
        if (oBinding && iRowIndex >= 0) {
            oBinding.collapse(iRowIndex);
        }

        return this;
    };

    /**
     * Collapses all nodes (and lower if collapseRecursive is activated)
     *
     * @return {sas.hc.ui.table.TreeTable} a reference on the TreeTable control, can be used for chaining
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.collapseAll = function () {
        var oBinding = this.getBinding("rows");
        if (oBinding) {
            oBinding.collapseToLevel(0);
            this.setFirstVisibleRow(0);
        }

        return this;
    };

    /**
     * Expands all nodes starting from the root level to the given level 'iLevel'.
     *
     * Only supported with ODataModel v2, when running in OperationMode.Client or OperationMode.Auto.
     * Fully supported for <code>sap.ui.model.ClientTreeBinding</code>, e.g. if you are using a <code>sap.ui.model.json.JSONModel</code>.
     *
     * Please also see <code>sap.ui.model.odata.OperationMode</code>.
     *
     * @param {int} iLevel the level to which the trees shall be expanded
     * @return {sas.hc.ui.table.TreeTable} a reference on the TreeTable control, can be used for chaining
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.expandToLevel = function (iLevel) {
        var oBinding = this.getBinding("rows");

        jQuery.sap.assert(oBinding && oBinding.expandToLevel, "TreeTable.expandToLevel is not supported with your current Binding.");

        if (oBinding && oBinding.expandToLevel) {
            oBinding.expandToLevel(iLevel);
        }

        return this;
    };

    /**
     * Returns whether the row is expanded or collapsed.
     *
     * @param {int} iRowIndex index of the row to check
     * @return {boolean} true if the node at "iRowIndex" is expanded, false otherwise (meaning it is collapsed)
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.isExpanded = function(iRowIndex) {
        var oBinding = this.getBinding("rows");
        if (oBinding) {
            return oBinding.isExpanded(iRowIndex);
        }
        return false;
    };

    /**
     * Checks if the row at the given index is selected.
     *
     * @param {int} iRowIndex The row index for which the selection state should be retrieved
     * @return {boolean} true if the index is selected, false otherwise
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.isIndexSelected = function (iRowIndex) {
        if (this.getSelectionMode() === SelectionMode.ParentChild) {

            var oTreeSelectionModel = this.getTreeSelectionModel();
            if (!oTreeSelectionModel) {
                return false;
            }

            var oContext = this.getContextByIndex(iRowIndex);
            return oTreeSelectionModel.isContextSelected(oContext);
        }

        var oBinding = this.getBinding("rows");
        //when using the treebindingadapter, check if the node is selected
        if (oBinding && oBinding.isIndexSelected) {
            return oBinding.isIndexSelected(iRowIndex);
        }
        return Table.prototype.isIndexSelected.call(this, iRowIndex);
    };

    /**
     * Overriden from Table.js base class.
     * In a TreeTable you can only select indices, which correspond to the currently visualized tree.
     * Invisible tree nodes (e.g. collapsed child nodes) can not be selected via Index, because they do not
     * correspond to a TreeTable row.
     *
     * @param {int} iRowIndex The row index which will be selected (if existing)
     * @return {sas.hc.ui.table.TreeTable} a reference on the TreeTable control, can be used for chaining
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.setSelectedIndex = function (iRowIndex) {
        if (iRowIndex === -1) {
            //If Index eq -1 no item is selected, therefore clear selection is called
            //SelectionModel doesn't know that -1 means no selection
            this.clearSelection();
        }

        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            if (this.getTreeSelectionModel()) {
                var oContext = this.getContextByIndex(iRowIndex);
                this.getTreeSelectionModel().selectContext(oContext);
            }
            return this;
        }

        //when using the treebindingadapter, check if the node is selected
        var oBinding = this.getBinding("rows");

        if (oBinding && oBinding.findNode && oBinding.setNodeSelection) {
            // set the found node as selected
            oBinding.setSelectedIndex(iRowIndex);
            //this.fireEvent("selectionChanged");
        } else {
            Table.prototype.setSelectedIndex.call(this, iRowIndex);
        }
        return this;
    };

    /**
     * Returns an array containing the row indices of all selected tree nodes (ordered ascending).
     *
     * Please be aware of the following:
     * Due to performance/network traffic reasons, the getSelectedIndices function returns only all indices
     * of actually selected rows/tree nodes. Unknown rows/nodes (as in "not yet loaded" to the client), will not be
     * returned.
     *
     * @return {int[]} an array containing all selected indices
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.getSelectedIndices = function () {

        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            var oTreeSelectionModel = this.getTreeSelectionModel();
            if (!oTreeSelectionModel) {
                return [];
            }

            var aSelectedContexts = oTreeSelectionModel.getSelectedContexts();
            var aSelectedIndices = this._getRowIndicesFromContexts(aSelectedContexts);
            return aSelectedIndices;
        }

        //when using the treebindingadapter, check if the node is selected
        var oBinding = this.getBinding("rows");

        if (oBinding && oBinding.findNode && oBinding.getSelectedIndices) {
            return oBinding.getSelectedIndices();
        } else {
            return Table.prototype.getSelectedIndices.call(this);
        }
    };

    TreeTable.prototype._isSomeRowSelected = function() {

        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            var oTreeSelectionModel = this.getTreeSelectionModel();
            if (oTreeSelectionModel) {
                return oTreeSelectionModel.isSomeContextSelected();
            }
            return false;
        }

        return this._getSelectedIndicesCount() > 0;
    };

    /**
     * Sets the selection of the TreeTable to the given range (including boundaries).
     * Beware: The previous selection will be lost/overriden. If this is not wanted, please use "addSelectionInterval" and
     * "removeSelectionIntervall".
     *
     * @param {int} iFromIndex the start index of the selection range
     * @param {int} iToIndex the end index of the selection range
     * @return {sas.hc.ui.table.TreeTable} a reference on the TreeTable control, can be used for chaining
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.setSelectionInterval = function (iFromIndex, iToIndex) {

        if (this.getSelectionMode() === SelectionMode.ParentChild) {

            var oTreeSelectionModel = this.getTreeSelectionModel();
            if (oTreeSelectionModel) {
                oTreeSelectionModel.deselectAll();
                for (var i = iFromIndex; i <= iToIndex; i++) {
                    var oContext = this.getContextByIndex(i);
                    oTreeSelectionModel.selectContext(oContext);
                }
            }
            return this;
        }

        //when using the treebindingadapter, check if the node is selected
        var oBinding = this.getBinding("rows");

        if (oBinding && oBinding.findNode && oBinding.setSelectionInterval) {
            oBinding.setSelectionInterval(iFromIndex, iToIndex);
        } else {
            Table.prototype.setSelectionInterval.call(this, iFromIndex, iToIndex);
        }

        return this;
    };

    /**
     * Marks a range of tree nodes as selected, starting with iFromIndex going to iToIndex.
     * The TreeNodes are referenced via their absolute row index.
     * Please be aware, that the absolute row index only applies to the the tree which is visualized by the TreeTable.
     * Invisible nodes (collapsed child nodes) will not be regarded.
     *
     * Please also take notice of the fact, that "addSelectionInterval" does not change any other selection.
     * To override the current selection, please use "setSelctionInterval" or for a single entry use "setSelectedIndex".
     *
     * @param {int} iFromIndex The starting index of the range which will be selected.
     * @param {int} iToIndex The starting index of the range which will be selected.
     * @return {sas.hc.ui.table.TreeTable} a reference on the TreeTable control, can be used for chaining
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.addSelectionInterval = function (iFromIndex, iToIndex) {

        if (this.getSelectionMode() === SelectionMode.ParentChild) {

            var oTreeSelectionModel = this.getTreeSelectionModel();
            if (oTreeSelectionModel) {
                for (var i = iFromIndex; i <= iToIndex; i++) {
                    var oContext = this.getContextByIndex(i);
                    oTreeSelectionModel.selectContext(oContext);
                }
            }
            return this;
        }

        var oBinding = this.getBinding("rows");
        //TBA check
        if (oBinding && oBinding.findNode && oBinding.addSelectionInterval) {
            oBinding.addSelectionInterval(iFromIndex, iToIndex);
        } else {
            Table.prototype.addSelectionInterval.call(this, iFromIndex, iToIndex);
        }
        return this;
    };

    /**
     * All rows/tree nodes inside the range (including boundaries) will be deselected.
     * Tree nodes are referenced with theit absolute row index inside the tree-
     * Please be aware, that the absolute row index only applies to the the tree which is visualized by the TreeTable.
     * Invisible nodes (collapsed child nodes) will not be regarded.
     *
     * @param {int} iFromIndex The starting index of the range which will be deselected.
     * @param {int} iToIndex The starting index of the range which will be deselected.
     * @return {sas.hc.ui.table.TreeTable} a reference on the TreeTable control, can be used for chaining
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.removeSelectionInterval = function (iFromIndex, iToIndex) {

        if (this.getSelectionMode() === SelectionMode.ParentChild) {

            var oTreeSelectionModel = this.getTreeSelectionModel();
            if (oTreeSelectionModel) {
                for (var i = iFromIndex; i <= iToIndex; i++) {
                    var oContext = this.getContextByIndex(i);
                    oTreeSelectionModel.deselectContext(oContext);
                }
            }
            return this;
        }

        var oBinding = this.getBinding("rows");
        //TBA check
        if (oBinding && oBinding.findNode && oBinding.removeSelectionInterval) {
            oBinding.removeSelectionInterval(iFromIndex, iToIndex);
        } else {
            Table.prototype.removeSelectionInterval.call(this, iFromIndex, iToIndex);
        }
        return this;
    };

    /**
     * Selects all available nodes/rows.
     *
     * Explanation of the SelectAll function and what to expect from its behavior:
     * All rows/tree nodes locally stored on the client are selected.
     * In addition all subsequent rows/tree nodes, which will be paged into view are also immediatly selected.
     * However, due to obvious performance/network traffic reasons, the SelectAll function will NOT retrieve any data from the backend.
     *
     * @return {sas.hc.ui.table.TreeTable} a reference on the TreeTable control, can be used for chaining
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.selectAll = function () {
        //select all is only allowed when SelectionMode is "ParentChild" or "MultiToggle"
        var oSelMode = this.getSelectionMode();
        if (!this.getEnableSelectAll() || oSelMode === SelectionMode.Single || !this._getSelectableRowCount()) {
            return this;
        }

        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            if (this.getTreeSelectionModel()) {
                this.getTreeSelectionModel().selectAll();
                this._refreshSelectAllControl();
                this._refreshSelectionControls();
            }
            return this;
        }

        //The OData TBA exposes a selectAll function
        var oBinding = this.getBinding("rows");
        if (oBinding.selectAll) {
            oBinding.selectAll();
            this._updateSelectAllContainer(true);
        } else {
            //otherwise fallback on the tables own function
            Table.prototype.selectAll.call(this);
        }

        return this;
    };

    /**
     * Retrieves the lead selection index. The lead selection index is, among other things, used to determine the
     * start/end of a selection range, when using Shift-Click to select multiple entries at once.
     *
     * @return {int[]} an array containing all selected indices (ascending ordered integers)
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.getSelectedIndex = function() {
        //when using the treebindingadapter, check if the node is selected
        var oBinding = this.getBinding("rows");

        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            var aSelectedIndices = this.getSelectedIndices();

            if (aSelectedIndices.length === 0) {
                return -1;
            }

            var iMinIndex = aSelectedIndices.reduce(function(iCurrentSmallest, iNextIndex) {
                return Math.min(iCurrentSmallest, iNextIndex);
            }, Number.MAX_VALUE);
            return iMinIndex;
        }

        if (oBinding && oBinding.findNode) {
            return oBinding.getSelectedIndex();
        } else {
            return Table.prototype.getSelectedIndex.call(this);
        }
    };

    /**
     * Clears the complete selection (all tree table rows/nodes will lose their selection)
     *
     * @return {sas.hc.ui.table.TreeTable} a reference on the TreeTable control, can be used for chaining
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    TreeTable.prototype.clearSelection = function () {

        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            if (this.getTreeSelectionModel()) {
                this.getTreeSelectionModel().deselectAll();
            }
            return this;
        }

        var oBinding = this.getBinding("rows");

        if (oBinding && oBinding.clearSelection) {
            oBinding.clearSelection();
        } else {
            Table.prototype.clearSelection.call(this);
        }

        return this;
    };

    /**
     * Sets the node hierarchy to collapse recursive. When set to true, all child nodes will get collapsed as well.
     * This setting has only effect when the binding is already initialized.
     * @param {boolean} bCollapseRecursive
     * @public
     */
    TreeTable.prototype.setCollapseRecursive = function(bCollapseRecursive) {
        var oBinding = this.getBinding("rows");
        if (oBinding) {
            jQuery.sap.assert(oBinding.setCollapseRecursive, "Collapse Recursive is not supported by the used binding");
            if (oBinding.setCollapseRecursive) {
                oBinding.setCollapseRecursive(bCollapseRecursive);
            }
        }
        this.setProperty("collapseRecursive", !!bCollapseRecursive, true);
        return this;
    };

    /**
     * Returns the number of selected entries.
     * Depending on the binding it is either retrieved from the binding or the selection model.
     * @private
     */
    TreeTable.prototype._getSelectedIndicesCount = function () {

        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            return this.getSelectedIndices().length;
        }

        //when using the treebindingadapter, check if the node is selected
        var oBinding = this.getBinding("rows");
        if (oBinding && oBinding.findNode && oBinding.getSelectedNodesCount) {
            return oBinding.getSelectedNodesCount();
        }

        return Table.prototype._getSelectedIndicesCount.call(this);
    };

    /**
     * Setter for "useGroupMode" property
     * @param {bGroup} bGroup useGroupMode
     * @public
     */
    TreeTable.prototype.setUseGroupMode = function (bGroup) {
        this.setProperty("useGroupMode", !!bGroup);
        if (!!bGroup) {
            TableUtils.Grouping.setGroupMode(this);
        } else {
            TableUtils.Grouping.setTreeMode(this);
        }
        return this;
    };

    /**
     * Setter for "enableGrouping" property
     * @param {bGroup} bEnableGrouping enableGrouoing value
     * @public
     */
    TreeTable.prototype.setEnableGrouping = function(bEnableGrouping) {
        jQuery.sap.log.warning("The property enableGrouping is not supported by control sas.hc.ui.table.TreeTable");
        return this;
    };

    /**
     * Check whether every selection control is selected or not
     * @override
     * @returns {boolean} Whether every row is selected
     */
    TreeTable.prototype.isEveryIndexSelected = function() {

        var iSelectedIndices = this._getSelectedIndicesCount();
        var iTotalIndices = this._getRowCount();

        return iSelectedIndices === iTotalIndices;
    };

    TreeTable.prototype._allowColumnMove = function(oColumn) {
        // TreeTable does not allow the first column to be dragged for reordering.
        return Table.prototype._allowColumnMove.apply(this, arguments) && oColumn.getIndex() > 0;
    };

    TreeTable.prototype._createSelectionControlTemplate = function(sId) {
        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            var oSelectionControl = new TriStateCheckBox({
                id: sId + "-selectionControl",
                change: function(oEvent) {
                    var oTriStateCheckBox = oEvent.getSource();
                    var oRow = oTriStateCheckBox.getParent();

                    if (oRow) {
                        var oTreeTable = oRow.getParent();
                        if (oTreeTable) {
                            var bShift = oEvent.getParameter("withShift");
                            var bCtrl = oEvent.getParameter("withControl");
                            var iIndex = oRow.getIndex();
                            oTreeTable._onRowSelect(iIndex, bShift, bCtrl);
                        }
                    }
                }
            });
            return oSelectionControl;
        }
        return Table.prototype._createSelectionControlTemplate.apply(this, arguments);
    };

    /**
     * Sync up selection control state with the corresponding row.
     * This gets called after paging occurs.
     * @private
     * @override
     */
    TreeTable.prototype._refreshSelectionControls = function() {

        if (this.getSelectionMode() !== SelectionMode.ParentChild) {
            return Table.prototype._refreshSelectionControls.apply(this, arguments);
        }

        if (!this.getTreeSelectionModel()) {
            // No TreeSelectionModel to work with and check anything.
            return;
        }

        // Update each row's selection control.
        var self = this;
        this.getRows().forEach(function(oRow) {
            self._refreshRowSelectionControl(oRow);
        });
    };

    TreeTable.prototype._refreshRowSelectionControl = function(oRow) {

        if (this.getSelectionMode() !== SelectionMode.ParentChild) {
            return Table.prototype._refreshRowSelectionControl.apply(this, arguments);
        }

        var oContext = this._getAccessedContextByIndex(oRow.getIndex());
        var oContextSelectionState = this.getTreeSelectionModel().getSelectionStateForContext(oContext);
        var oSelectionControl = oRow.getSelectionControl();

        if (oContextSelectionState === TreeSelectionModel.SelectionState.SELECTED) {
            oSelectionControl.setSelectionState(TriStateCheckBoxState.Checked);
        } else if (oContextSelectionState === TreeSelectionModel.SelectionState.PARTIALLY_SELECTED) {
            oSelectionControl.setSelectionState(TriStateCheckBoxState.Mixed);
        } else {
            oSelectionControl.setSelectionState(TriStateCheckBoxState.Unchecked);
        }
    };

    TreeTable.prototype._refreshSelectAllControl = function() {
        this.getSelectAllControl().updateRelativeToTable();

        // TODO This implementation assumes ParentChild selectionMode
        // but right now should only be called when selectionMode is just that.

        // Since the selectAllControl just checked the model for if everything is selected, it is often
        // faster to get the state of the checkbox instead of asking the TreeSelectionModel.
        var bEverythingIsSelected = this.getSelectAllControl().getSelectControl().getSelectionState() === TriStateCheckBoxState.Checked;
        this._updateSelectAllContainer(bEverythingIsSelected);
    };

    TreeTable.prototype._getAccessedContextByIndex = function(iIndex) {
        var oRowBinding = this.getBinding("rows");
        if (oRowBinding._bIsAdaptedForPaging) {
            return oRowBinding.getAccessedContextByIndex(iIndex);
        }
        return this.getContextByIndex(iIndex);
    };

    TreeTable.prototype._recreateSelectAllControl = function() {
        var oOldSelectAllControl = this.getSelectAllControl();
        oOldSelectAllControl && oOldSelectAllControl.destroy();

        var oNewSelectAllControl = this._createSelectAllControl();
        this.setSelectAllControl(oNewSelectAllControl);
    };

    TreeTable.prototype._createSelectAllControl = function() {
        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            return new TriStateSelectAll({
                id: this.getId()+"-selectAll",
                table: this
            });
        }
        return Table.prototype._createSelectAllControl.apply(this, arguments);
    };

    TreeTable.prototype._getItemNavRowHdrDomRefs = function() {

        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            var aRowHdrsWithControls = this.$().find(".sapUiTableRowHdr > span:nth-child(2)").get();
            if (aRowHdrsWithControls.length > 0) {
                return aRowHdrsWithControls;
            }
        }

        return Table.prototype._getItemNavRowHdrDomRefs.apply(this, arguments);
    };

    TreeTable.prototype._getItemNavSelectAllDomRef = function() {
        if (this.getSelectionMode() === SelectionMode.ParentChild) {
            return this.getSelectAllControl().getSelectControl().getDomRef();
        }
        return Table.prototype._getItemNavSelectAllDomRef.apply(this, arguments);
    };

    TreeTable.prototype._getRowIndicesFromContexts = function(aContexts) {
        var aIndices = [];
        var oRowBinding = this.getBinding("rows");

        if (oRowBinding) {
            var iIndex = -1;
            var aSearchResults = [];

            // Create hash map of context paths to efficiently check if
            // the node is one of the contexts we're interested in.
            var mContextPaths = aContexts.reduce(function(mCurrentPaths, oContext) {
                if (oContext) {
                    mCurrentPaths[oContext.getPath()] = true;
                }
                return mCurrentPaths;
            }, {});

            oRowBinding._match(oRowBinding._oRootNode, aSearchResults, aContexts.length, function(oNode) {
                var bIsRealNode = oNode && !oNode.isArtificial && oNode.context;
                var bIsOneOfTheNodes = bIsRealNode && mContextPaths[oNode.context.getPath()];

                if (bIsOneOfTheNodes) {
                    aIndices.push(iIndex);
                }

                iIndex++;

                return bIsOneOfTheNodes;
            });
        }

        return aIndices;
    };

    TreeTable.prototype._updateHSb = function(oTableSizes) {
        // get the width of the container
        var $tableCell = this.$().find(".sapUiTableCell");

        // Make sure the table is wide enough to display all subnodes
        // that are currently expanded
        var iLevel = 0;
        this.getRows().forEach(function(oRow) {
            // find the max level of subnodes
            iLevel = Math.max(iLevel, oRow._iLevel);
        });

        // get the width of the expand/collapse icon
        var iIconWidth = 0;
        var $icon;
        if (this.getRows().length > 0) {
            $icon = $(this.getRows()[0].getDomRef()).find(".sapUiIcon").get(0);
            if ($icon) {
                iIconWidth = $icon.clientWidth;
            }
        }

        // consider left and right padding on the cell
        // (right padding with include space for the vertical scroll bar, if present)
        var iCellPadding = this._CSSSizeToPixel($tableCell.css("padding-right"));
        iCellPadding += this._CSSSizeToPixel(this.$().find(".sapUiTableTr>td.sapUiTableTdFirst").css("padding-left"));
        iCellPadding += this._CSSSizeToPixel($tableCell.css("padding-left"));

        // each subnode adds indentation before the expand/collapse icon
        // then add the min column width
        // TODO - once we implement the story to honor min column width on a per column basis,
        // we may need to update this code to get the min width of the first column
        var iMinWidth = (iLevel * this.getProperty("_subnodeIndent")) + iIconWidth + this.getMinColumnWidth() + iCellPadding;

        // if there is only one column then set the width to the min width needed so
        // we won't have unnecessary scroll bars after collapsing nodes
        // otherwise, there could be other columns we need to scroll to, so only set
        // the width to minWidth if it's bigger than tableCtrlScrollWidth
        if (iMinWidth > oTableSizes.tableCtrlScrollWidth || this.getColumns().length === 1) {
            // Set the min-width of the contents table so that it will scroll
            // to the full width needed to view the subnodes
            // otherwise, you will see a scroll bar, but it does nothing
            this.$().find(".sapUiTableCCnt .sapUiTableCtrlScroll").css({
                "min-width": iMinWidth + "px"
            });

            oTableSizes.tableCtrlScrollWidth = iMinWidth;
        }

        Table.prototype._updateHSb.apply(this, arguments);
    };

    return TreeTable;
});
