/*!
 * (c) 2014-2018, SAS Institute Inc.
 */

// Provides helper sas.hc.ui.table.TablePointerExtension.
sap.ui.define([
    'jquery.sap.global',
    './TableExtension',
    './TableUtils',
    'sap/ui/Device',
    'sas/hc/ui/core/AnimationUtil',
    './Column',
    './library'
], function(jQuery, TableExtension, TableUtils, Device, AnimationUtil, Column, library) {
    "use strict";

    //var logger = library._getLogger("TablePointerExtension");

    /*
     * Provides utility functions used by this extension (and table)
     */
    var ExtensionHelper = {
        /*
         * Returns the pageX and pageY position of the given mouse/touch event.
         */
        getEventPosition: function(oEvent, oTable) {
            var oPosition = oEvent,
                aTouchEventObjectNames = ["touches", "targetTouches", "changedTouches"],
                sTouchEventObjectName,
                i;

            //if the event has touch information, then attempt to extract pageX/Y info from it
            if (oTable._isTouchMode(oEvent)) {
                for (i = 0; i < aTouchEventObjectNames.length; i++) {
                    sTouchEventObjectName = aTouchEventObjectNames[i];

                    if (oEvent[sTouchEventObjectName] && oEvent[sTouchEventObjectName][0]) {
                        oPosition = oEvent[sTouchEventObjectName][0];
                        break;
                    }
                    if (oEvent.originalEvent[sTouchEventObjectName] && oEvent.originalEvent[sTouchEventObjectName][0]) {
                        oPosition = oEvent.originalEvent[sTouchEventObjectName][0];
                        break;
                    }
                }
            }

            return oPosition;
        }
    };

    /*
     * Provides helper functionality (e.g. drag&drop capabilities) for column resizing.
     */
    var ColumnResizeHelper = {

        /*
         * Initializes the drag&drop for resizing
         */
        initColumnResizing : function(oTable, oEvent){
            if (oTable._bIsColumnResizerMoving) {
                return;
            }

            oTable._bIsColumnResizerMoving = true;
            oTable.$().toggleClass("sapUiTableResizing", true);

            var $Document = jQuery(document),
                bTouch = oTable._isTouchMode(oEvent);

            oTable._$colResize = oTable.$("rsz");
            oTable._iColumnResizeStart = ExtensionHelper.getEventPosition(oEvent, oTable).pageX;

            $Document.bind((bTouch ? "touchend" : "mouseup") + ".sapUiTableColumnResize", ColumnResizeHelper.exitColumnResizing.bind(oTable));
            $Document.bind((bTouch ? "touchmove" : "mousemove") + ".sapUiTableColumnResize", ColumnResizeHelper.onMouseMoveWhileColumnResizing.bind(oTable));

            //HTMLCOMMONS-6821
            var oColumn = oTable._getVisibleColumns()[oTable._iLastHoveredColumnIndex];
            if (!oTable.getColumnHeaderVisible()) {
                var oCell = oTable.$().find(".sapUiTableCCnt").find("th[data-sap-ui-headcolindex=" + oTable._iLastHoveredColumnIndex + "]");
                if (oCell) {
                    oTable._iColumnResizeOriginalWidth = oCell.width();
                    oTable._disableTextSelection();
                    return;
                }
            }
            oTable._iColumnResizeOriginalWidth = oColumn.$().width();
            oTable._disableTextSelection();
        },

        /*
         * Drops the previous dragged column resize bar and recalculates the new column width.
         */
        exitColumnResizing: function(oEvent) {
            ColumnResizeHelper._resizeColumn(this, this._iLastHoveredColumnIndex);
            if (AnimationUtil.getVisualEffectsEnabled()) {
                this.invalidate();
            }
        },

        /*
         * Handler for the move events while dragging the column resize bar.
         */
        onMouseMoveWhileColumnResizing: function(oEvent) {
            var iLocationX = ExtensionHelper.getEventPosition(oEvent, this).pageX;

            if (this._iColumnResizeStart && iLocationX < this._iColumnResizeStart + 3 && iLocationX > this._iColumnResizeStart - 3) {
                return;
            }

            if (this._isTouchMode(oEvent)) {
                oEvent.stopPropagation();
                oEvent.preventDefault();
            }

            this._$colResize.toggleClass("sapUiTableColRszActive", true);

            var oColumn = this._getVisibleColumns()[this._iLastHoveredColumnIndex];
            var iDeltaX = iLocationX - this._iColumnResizeStart;

            //HTMLCOMMONS-6821
            var iWidth = Math.max(this._iColumnResizeOriginalWidth + iDeltaX * (this._bRtlMode ? -1 : 1), this._iColMinWidth);

            // calculate and set the position of the resize handle
            var iRszOffsetLeft = this.$().find(".sapUiTableCnt").offset().left;
            var iRszLeft = Math.floor((iLocationX - iRszOffsetLeft) - (this._$colResize.width() / 2));
            this._$colResize.css("left", iRszLeft + "px");

            // store the width of the column to apply later
            oColumn._iNewWidth = iWidth;

            // If animation is on then resize of column should occur immediately
            if (AnimationUtil.getVisualEffectsEnabled()) {

                var oColumn = this._getVisibleColumns()[this._iLastHoveredColumnIndex];
                if (oColumn._iNewWidth) {
                    var sWidth;
                    // if the column is temporarily unfrozen due to column width,
                    // keep using px based size so we don't lose it when it refreezes
                    var bShouldBeFrozen = this._shouldBeFrozenColumn(oColumn);
                    if (!this._hasSomePercentageWidthColumn() || this._iLastHoveredColumnIndex < this.getTotalFrozenColumnCount() || bShouldBeFrozen) {
                        sWidth = oColumn._iNewWidth + "px";
                    } else {
                        var iAvailableSpace = this.$().find(".sapUiTableCtrlScroll").width();
                        var iColumnWidth = Math.round(100 / iAvailableSpace * oColumn._iNewWidth);
                        sWidth = iColumnWidth + "%";
                    }

                    if (this._updateColumnWidth(oColumn, sWidth, false)) {
                        this._resizeDependentColumns(oColumn, sWidth);
                    }

                    var oTableSizes = this._collectTableSizes();
                    this._syncColumnHeaders(oTableSizes);

                    // so that _resizeColumn will not try to resize again once mouse is up
                    oColumn._bPreventResize = true;
                }
            }
        },

        /*
         * Cleans up the state which is created while resize a column via drag&drop.
         */
        _cleanupColumResizing: function(oTable) {
            if (oTable._$colResize) {
                oTable._$colResize.toggleClass("sapUiTableColRszActive", false);
                oTable._$colResize = null;
            }
            oTable._iColumnResizeStart = null;
            oTable._bIsColumnResizerMoving = false;
            oTable.$().toggleClass("sapUiTableResizing", false);
            oTable._enableTextSelection();

            var $Document = jQuery(document);
            $Document.unbind("touchmove.sapUiTableColumnResize");
            $Document.unbind("touchend.sapUiTableColumnResize");
            $Document.unbind("mousemove.sapUiTableColumnResize");
            $Document.unbind("mouseup.sapUiTableColumnResize");

            //HTMLCOMMONS-6821
            this._iColumnResizeOriginalWidth = null;
        },

        /*
         * Cleans up the state which is created while resize a column via drag&drop and recalculates the new column width.
         */
        _resizeColumn: function(oTable, iColIndex) {
            var aVisibleColumns = oTable._getVisibleColumns(),
                oColumn;

            if (iColIndex >= 0 && iColIndex < aVisibleColumns.length) {
                oColumn = aVisibleColumns[iColIndex];

                this._resizeColumnsTogether(oTable, [oColumn], true);

                oColumn.focus();
            }
        },

        _resizeColumnsTogether: function(oTable, aColumnsToResize, bResizeDependents) {
            var bResized = false;

            aColumnsToResize.forEach(function(oColumn) {
                if (oColumn._iNewWidth) {
                    var sWidth,
                        bExecuteDefault,
                        $tableCtrl = oColumn.$().closest(".sapUiTableCtrl");

                    if ((!$tableCtrl ||$tableCtrl.length === 0) && !oTable.getColumnHeaderVisible()) {
                        // if there are no column headers, there is only one other .sapUiTableCtrl
                        // element in the table so get it's width
                        $tableCtrl = oTable.$().find(".sapUiTableCtrl");
                    }

                    var iAvailableSpace = $tableCtrl.width(),
                        fnGetWidth = function (iWidth) {
                            // if the column is temporarily unfrozen due to column width,
                            // keep using px based size so we don't lose it when it refreezes
                            var bShouldBeFrozen = oTable._shouldBeFrozenColumn(oColumn);
                            if (!oTable._hasSomePercentageWidthColumn() || oTable.isFrozenColumn(oColumn) || bShouldBeFrozen) {
                                return iWidth + "px";
                            } else {
                                var iColumnWidth = Math.round(100 / iAvailableSpace * iWidth);
                                return iColumnWidth + "%";
                            }
                        };

                    sWidth = fnGetWidth(oColumn._iNewWidth);

                    bExecuteDefault = oTable.fireColumnResize({
                        column: oColumn,
                        width: sWidth
                    });

                    if (!bExecuteDefault) {
                        // Cancelled? Then revert back to original width.
                        sWidth = fnGetWidth(oTable._iColumnResizeOriginalWidth);
                    }

                    bResized = !oColumn._bPreventResize;

                    if (oTable._updateColumnWidth(oColumn, sWidth, false /* already fired the event */)) {
                        if (bResized && bResizeDependents === true) {
                            oTable._resizeDependentColumns(oColumn, sWidth);
                        }
                    }

                    delete oColumn._iNewWidth;
                    delete oColumn._bPreventResize;
                }
            });

            ColumnResizeHelper._cleanupColumResizing(oTable);

            // rerender if size of the column was changed
            if (bResized) {
                oTable.invalidate();
            }
        },

        /*
         * Computes the optimal width for a column and changes the width if the auto resize feature is activated for the column.
         *
         * Experimental feature.
         */
        doAutoResizeColumn : function(oTable, iColIndex) {
            var aVisibleColumns = oTable._getVisibleColumns(),
                oColumn;

            if (iColIndex >= 0 && iColIndex < aVisibleColumns.length) {
                oColumn = aVisibleColumns[iColIndex];
                if (!oColumn.getAutoResizable || !oColumn.getResizable || !oColumn.getAutoResizable() || !oColumn.getResizable()) {
                    return;
                }
                this.doBestFitColumn(oTable, iColIndex);
            }
        },

        /**
         * Computes the optimal width for a column based on the longest width for the given column.
         */
        doBestFitColumn: function(oTable, iColIndex) {
            var aVisibleColumns = oTable._getVisibleColumns(),
                oColumn;

            if (iColIndex >= 0 && iColIndex < aVisibleColumns.length) {
                oColumn = aVisibleColumns[iColIndex];
                if (!oColumn.getAutoResizable || !oColumn.getAutoResizable()) {
                    return;
                }
                var iNewWidth = ColumnResizeHelper._calculateAutomaticColumnWidth.apply(oTable, [oColumn, iColIndex]);
                if (iNewWidth) {
                    oColumn._iNewWidth = iNewWidth;
                    ColumnResizeHelper._resizeColumn(oTable, iColIndex);
                }
            }
        },

        autoFitAllColumns: function(oTable) {
            var aColumns = oTable._getVisibleColumns().reverse()
                    .filter(function(o) { //array of index values
                        if (o instanceof Column && o.getAutoResizable()) {
                            var iColIndex = o.getIndex(),
                                iNewWidth = ColumnResizeHelper._calculateAutomaticColumnWidth.apply(oTable, [o, iColIndex]);

                            if (iNewWidth) {
                                o._iNewWidth = iNewWidth;
                                return true;
                            }
                        }
                    });

            ColumnResizeHelper._resizeColumnsTogether(oTable, aColumns);
        },

        /**
         * Calculates the widest content width of the column
         * also takes the column header and potential icons into account
         * @param {int} iColIndex index of the column which should be resized
         * @return {int} minWidth minimum width the column needs to have
         *
         * Note: Experimental, only works with a limited control set
         *
         * TBD: Cleanup this function and find a proper mechanismn
         */
        _calculateAutomaticColumnWidth : function(oCol, iColIndex) {
            var $cols = this.$().find('td[headers*=\"' + this.getId() + '_col' + iColIndex + ' \"]').children("div");

            var hiddenSizeDetector = document.createElement("div");
            document.body.appendChild(hiddenSizeDetector);
            jQuery(hiddenSizeDetector).addClass("sapUiTable");
            jQuery(hiddenSizeDetector).addClass("sapUiTableHiddenSizeDetector");

            //get the max width of the currently displayed cells in this column
            var minWidth = Math.max.apply(null, $cols.children().map(
                function() {
                    var $table = oCol.getParent().$();

                    // Create a copy of  all visible cells in the column, including the header cells without colspan
                    var $cells = $table.find("[data-sap-ui-colid = \"" + oCol.getId() + "\"]")
                        .filter(function(index, element) {
                            return element.style.display !== "none";
                        }).children().clone();
                    $cells.find("[id]").removeAttr("id"); // remove all id attributes

                    // Determine the column width
                    var iWidth = $(hiddenSizeDetector).append($cells).width() + 4; // widest cell + 4px for borders, padding and rounding
                    iWidth = Math.min(iWidth, $table.find(".sapUiTableCnt").width()); // no wider as the table
                    iWidth = Math.max(iWidth + 4, oCol.getParent().getMinColumnWidth()); // not to small

                    return iWidth;
                }).get());

            jQuery(hiddenSizeDetector).remove();
            return Math.max(minWidth, this._iColMinWidth);
        },

        /*
         * Initialize the event listener for positioning the column resize bar and computing the currently hovered column.
         */
        initColumnTracking : function(oTable) {
            // attach mousemove listener to update resizer position
            oTable.$().find(".sapUiTableCtrlScr, .sapUiTableCtrlScrFixed, .sapUiTableColHdrScr, .sapUiTableColHdrFixed").mousemove(function(oEvent){

                this._aTableHeaders = this.$().find(".sapUiTableCtrlFirstCol > th:not(.sapUiTableColSel)");

                var oDomRef = this.getDomRef();
                if (!oDomRef || this._bIsColumnResizerMoving) {
                    return;
                }

                var $target = jQuery(oEvent.target);
                var oColumn = oTable._colFromRef($target);

                if (oColumn === null || oColumn === undefined) {
                    // We didn't hover over something in a column, bail out.
                    return;
                }

                // Try to get $domRef as if target is in table body.
                var $domRef = $target.closest('td[role=gridcell]').get(0);
                if (!$domRef) {
                    // If nothing, get it as target is a header cell.
                    $domRef = $target.closest('.sapUiTableCol').get(0);
                }
                if (!$domRef) {
                    // if we didn't find it going up the DOM tree, try going down
                    // this is necessary for headers that span multiple rows
                    $domRef = $target.find('.sapUiTableCol').get(0);
                }

                var oTableRect = oDomRef.getBoundingClientRect();
                var oColumnRect = $domRef.getBoundingClientRect();
                var iResizerPositionX = this._bRtlMode
                    ? (oColumnRect.left - oTableRect.left)
                    : (oColumnRect.right - oTableRect.left);
                var iLastHoveredColumn = oTable._getVisibleColumns().indexOf(oColumn);

                if (iLastHoveredColumn > -1 && oColumn.getResizable && oColumn.getResizable()) {
                    this.$("rsz").css("left", iResizerPositionX + "px");
                    this._iLastHoveredColumnIndex = iLastHoveredColumn;
                }
            }.bind(oTable));
        }

    };

    /*
     * Provides utility functions for drag&drop column reordering.
     * Largely taken from sap.ui.table.TablePointerExtension.
     */
    var ReorderHelper = {

        /*
         * Finds the column which belongs to the current x position and returns information about this column.
         */
        findColumnForPosition: function(oTable, iLocationX) {
            var oHeaderDomRef, $headerDomRef, oRect, iWidth, oPos, bBefore, bAfter, iRight, iCenter, i;

            for (i = 0; i < oTable._aTableHeaders.length; i++) {
                oHeaderDomRef = oTable._aTableHeaders[i];
                $headerDomRef = jQuery(oHeaderDomRef);
                oRect = oHeaderDomRef.getBoundingClientRect();
                iWidth = $headerDomRef.outerWidth();
                iRight = oRect.left + iWidth;
                iCenter = oRect.left + (iWidth / 2);
                bBefore = (iLocationX >= oRect.left && iLocationX <= iCenter);
                bAfter = (iLocationX >= iCenter && iLocationX <= iRight);

                if (bBefore || bAfter) {
                    oPos = {
                        left: oRect.left,
                        center: iCenter,
                        right: iRight,
                        width: iWidth,
                        index: parseInt($headerDomRef.attr("data-sap-ui-headcolindex"), 10),
                        id: $headerDomRef.attr("data-sap-ui-colid"),
                        before: oTable._bRtlMode ? bAfter : bBefore,
                        after: oTable._bRtlMode ? bBefore : bAfter
                    };

                    return oPos;
                }
            }

            return null;
        }
    };

    /*
     * Event handling of touch and mouse events.
     * "this" in the function context is the table instance.
     */
    var ExtensionDelegate = {

        onmousedown : function(oEvent) {
            var oPointerExtension = this._getPointerExtension();
            var $Cell = TableUtils.getCell(this, oEvent.target);
            var oCellInfo = TableUtils.getCellInfo($Cell) || {};

            // check whether item navigation should be reapplied from scratch
            this._getKeyboardExtension().initItemNavigation();

            if (oEvent.button === 0) { // left mouse button
                if (oEvent.target === this.getDomRef("rsz")) { // mousedown on column resize bar
                    ColumnResizeHelper.initColumnResizing(this, oEvent);
                } else if (jQuery(oEvent.target).hasClass("sapUiTableColResizer")) { // mousedown on mobile column resize button
                    var iColIndex = jQuery(oEvent.target).closest(".sapUiTableCol").attr("data-sap-ui-colindex");
                    this._iLastHoveredColumnIndex = parseInt(iColIndex, 10);
                    ColumnResizeHelper.initColumnResizing(this, oEvent);
                } else if (oCellInfo.type === TableUtils.CELLTYPES.COLUMNHEADER) {
                    // A long click starts column reordering, so it should not also perform an action in the onclick event handler.
                    oPointerExtension._bColumnHeaderClickAction = true;
                    this._mTimeouts.delayedMenuTimerId = jQuery.sap.delayedCall(200, this, function () {
                        //if the delay has passed before this value has been checked,
                        // then we're doing a longer action and should not execute the click action
                        oPointerExtension._bColumnHeaderClickAction = false;
                    });
                }
            }
        },

        onmouseup : function(oEvent) {
            // clean up the timer
            jQuery.sap.clearDelayedCall(this._mTimeouts.delayedActionTimer);
        },

        ondblclick : function(oEvent) {
            if (Device.system.desktop && oEvent.target === this.getDomRef("rsz")) {
                oEvent.preventDefault();
                ColumnResizeHelper.doAutoResizeColumn(this, this._iLastHoveredColumnIndex);
            }
        },

        onclick : function(oEvent) {
            // clean up the timer
            jQuery.sap.clearDelayedCall(this._mTimeouts.delayedActionTimer);

            if (oEvent.isMarked()) {
                // the event was already handled by some other handler, do nothing.
                return;
            }

            var $Target = jQuery(oEvent.target);

            if ($Target.hasClass("sapUiAnalyticalTableSum")) {
                // Analytical Table: Sum Row cannot be selected
                oEvent.preventDefault();
                return;
            } else if ($Target.hasClass("sapUiTableGroupMenuButton")) {
                // Analytical Table: Mobile Group Menu Button in Grouping rows
                this._onContextMenu(oEvent);
                oEvent.preventDefault();
                return;

            } else if ($Target.hasClass("sapUiTableGroupIcon") || $Target.hasClass("sapUiTableTreeIcon") || $Target.hasClass("sasUiTreeNodeBullet")) {
                // Grouping Row: Toggle grouping
                if (TableUtils.toggleGroupHeader(this, oEvent.target)) {
                    return;
                }
            }

            var $Cell = TableUtils.getCell(this, oEvent.target);
            var oCellInfo = TableUtils.getCellInfo($Cell) || {};

            if (oCellInfo.type === TableUtils.CELLTYPES.COLUMNHEADER) {
                var $focusable = $Target.closest(":focusable");
                var bFocusableInHeader = oCellInfo.cell.find($focusable).length > 0;

                var oPointerExtension = this._getPointerExtension();
                if (!!oPointerExtension._bColumnHeaderClickAction && !bFocusableInHeader) {
                    oPointerExtension._bColumnHeaderClickAction = false;    //tidy up delayed action flag

                    //apply/toggle sort on target column

                    //get column object for target
                    var iColIndex = oCellInfo.cell.attr("data-sap-ui-colindex");
                    var oColumn = this.retrieveLeafColumns()[iColIndex];

                    // 1) run column select and look at if default was prevented
                    var bExecuteDefault = this.fireColumnSelect({
                        column: oColumn
                    });

                    // 2) if default not prevented, then apply sorting behavior
                    if (bExecuteDefault) {
                        if (this._applySortActionFromEvent(oColumn, oEvent.ctrlKey, oEvent.shiftKey)) {
                            //prevent whatever default action may still be waiting to execute if a sorting operation took place
                            oEvent.preventDefault();
                        }
                    }
                }
            } else {
                // forward the event
                if (!this._findAndfireCellEvent(this.fireCellClick, oEvent)) {
                    this._onSelect(oEvent);
                } else {
                    oEvent.preventDefault();
                }
            }
        }

    };



    /**
     * Extension for sas.hc.ui.table.Table which handles mouse and touch related things.
     *
     * @class Extension for sas.hc.ui.table.Table which handles mouse and touch related things.
     *
     * @extends sas.hc.ui.table.TableExtension
     * @author SAP SE
     * @version 904001.11.16.20251118090100_f0htmcm94p
     * @constructor
     * @private
     * @alias sas.hc.ui.table.TablePointerExtension
     */
    var TablePointerExtension = TableExtension.extend("sas.hc.ui.table.TablePointerExtension", /* @lends sas.hc.ui.table.TablePointerExtension */ {

        /*
         * @see TableExtension._init
         */
        _init : function(oTable, sTableType, mSettings) {
            this._type = sTableType;
            this._delegate = ExtensionDelegate;

            // Register the delegate
            oTable.addEventDelegate(this._delegate, oTable);

            oTable._iLastHoveredColumnIndex = 0;
            oTable._bIsColumnResizerMoving = false;

            return "PointerExtension";
        },

        /*
         * @see sap.ui.base.Object#destroy
         */
        destroy : function() {
            // Deregister the delegates
            var oTable = this.getTable();
            if (oTable) {
                oTable.removeEventDelegate(this._delegate);
            }
            this._delegate = null;

            TableExtension.prototype.destroy.apply(this, arguments);
        },

        /*
         * Resizes the given column to its optimal width if the auto resize feature is available for this column.
         * @public (Part of the API for Table control only!)
         */
        doAutoResizeColumn : function(iColIndex) {
            var oTable = this.getTable();
            if (oTable) {
                ColumnResizeHelper.doAutoResizeColumn(oTable, iColIndex);
            }
        },

        doBestFitColumn : function(iColIndex) {
            var oTable = this.getTable();
            if (oTable) {
                ColumnResizeHelper.doBestFitColumn(oTable, iColIndex);
            }
        },

        autoFitAllColumns : function() {
            var oTable = this.getTable();
            if (oTable) {
                ColumnResizeHelper.autoFitAllColumns(oTable);
            }
        },

        /*
         * Initialize the basic event handling for column resizing.
         * @public (Part of the API for Table control only!)
         */
        initColumnResizeEvents : function() {
            var oTable = this.getTable();
            if (oTable) {
                ColumnResizeHelper.initColumnTracking(oTable);
            }
        },

        /*
         * Cleans up the basic event handling for column resizing.
         * @public (Part of the API for Table control only!)
         */
        cleanupColumnResizeEvents : function() {
            var oTable = this.getTable();
            if (oTable) {
                oTable.$().find(".sapUiTableCtrlScr, .sapUiTableCtrlScrFixed, .sapUiTableColHdrScr, .sapUiTableColHdrFixed").unbind();
            }
        }

    });

    // HTMLCOMMONS-6821
    // Make internal Helper objects availabe for subclasses.
    TablePointerExtension._ColumnResizeHelper = ColumnResizeHelper;
    TablePointerExtension._ExtensionDelegate = ExtensionDelegate;

    TablePointerExtension.prototype.getExtensionHelper = function() {
        return ExtensionHelper;
    };
    TablePointerExtension.prototype.getColumnResizeHelper = function() {
        return ColumnResizeHelper;
    };
    TablePointerExtension.prototype.getReorderHelper = function() {
        return ReorderHelper;
    };
    TablePointerExtension.prototype.getExtensionDelegate = function() {
        return ExtensionDelegate;
    };

    return TablePointerExtension;

}, /* bExport= */ true);
