/*!
 * (c) 2014-2018, SAS Institute Inc.
 */

// Provides helper sas.hc.ui.table.TableKeyboardDelegate.
sap.ui.define(['jquery.sap.global', 'sap/ui/base/Object', './library', './Row', './TableExtension', './TableUtils', './ItemNavigation'],
    function(jQuery, BaseObject, library, Row, TableExtension, TableUtils, ItemNavigation) {
        "use strict";

        //var logger = library._getLogger("TableKeyboardDelegate");

        /**
         * Delegate for keyboard events of sas.hc.ui.table.Table controls.
         *
         * @class Delegate for keyboard events of sas.hc.ui.table.Table controls.
         *
         * @extends sap.ui.base.Object
         * @author SAP SE
         * @version 904001.11.16.20251118090100_f0htmcm94p
         * @constructor
         * @private
         * @alias sas.hc.ui.table.TableKeyboardDelegate
         */
        var TableKeyboardDelegate = BaseObject.extend("sas.hc.ui.table.TableKeyboardDelegate", /* @lends sas.hc.ui.table.TableKeyboardDelegate */ {

        /*
         * @see sap.ui.base.Object#getInterface
         */
            getInterface : function() {
                return this;
            }

        });

        TableKeyboardDelegate.prototype.onmousedown = function(oEvent) {
            var $cellInFocus = jQuery(oEvent.target).closest("td");
            if (oEvent.button === 0) {
                $cellInFocus.addClass("sasLeftMouseDown");
            } else if (oEvent.button === 2) {
                $cellInFocus.addClass("sasRightMouseDown");
            }
        };

        /*
         * NOTE: "this" in the function context is the table instance
         */

        TableKeyboardDelegate.prototype.onmouseup = function(oEvent) {
            if (oEvent.isMarked()) { // the event was already handled by some other handler, do nothing.
                return;
            }

            // When clicking into a focusable control we enter the action mode
            // When clicking anywhere else in the table we leave the action mode
            var $Dom = this.$().find(".sapUiTableCCnt .sapUiTableCtrl td :focus");
            var oKeyboardExtension = this._getKeyboardExtension();
            var bShouldBeInActionMode = $Dom.length > 0;

            var $cellInFocus = jQuery(oEvent.target).closest("td");
            if (oEvent.button === 0) {
                $cellInFocus.removeClass("sasLeftMouseDown");
            } else if (oEvent.button === 2) {
                $cellInFocus.removeClass("sasRightMouseDown");
            }

            oKeyboardExtension.setActionMode(bShouldBeInActionMode, {$Dom: $Dom, event: oEvent});

            // Not every cell control is acceptable for action mode.
            // If we tried to enter action mode but failed, set focus to nearest cell.
            var bIsInActionMode = oKeyboardExtension.isInActionMode();
            if (bShouldBeInActionMode && !bIsInActionMode) {
                var oCellToFocus = jQuery(oEvent.target).closest("td").get(0);
                this._focusItemByDomRef(oCellToFocus, oEvent);
            }
        };


        TableKeyboardDelegate.prototype.onfocusin = function(oEvent) {
            if (oEvent.isMarked("sapUiTableIgnoreFocusIn")) {
                return;
            }

            var $target = jQuery(oEvent.target),
                bTargetOuterBefore = $target.hasClass("sapUiTableOuterBefore"),
                bTargetOuterAfter = $target.hasClass("sapUiTableOuterAfter"),
                bNoData = TableUtils.isNoDataVisible(this),
                oInfo,
                bVisibleColumns;

            // focus on the OuterBefore tab stop from outside of the table should jump to the table.
            if (bTargetOuterBefore && !jQuery.sap.containsOrEquals(this.getDomRef(), oEvent.relatedTarget)) {
                // focus has come to top bumper from outside of table - make sure we are not in action mode
                this._getKeyboardExtension().setActionMode(false);

                // focus will move to an element of the table - 'no data' or 'data cell'
                if (bNoData) {
                    this.$("noDataCnt").focus();
                } else {
                    oInfo = TableUtils.getFocusedItemInfo(this);
                    bVisibleColumns = TableUtils.getVisibleColumnCount(this) > 0;

                    // move to header of column if available, cell in column otherwise
                    if (bVisibleColumns && this.getColumnHeaderVisible()) {
                        TableUtils.focusItem(this, oInfo.cellInRow, oEvent);
                    } else {
                        this._getKeyboardExtension()._restoreFocusOnLastFocusedDataCell(this, oEvent);
                    }
                }

                return;
            }

            if (bTargetOuterBefore || bTargetOuterAfter
                || (oEvent.target !== this.getDomRef("overlay") && this.getShowOverlay())) {
                this._getKeyboardExtension().setActionMode(false);
                this.$("overlay").focus();
            } else if ($target.hasClass("sapUiTableCtrlBefore")) {
                this._getKeyboardExtension().setActionMode(false);
                bVisibleColumns = TableUtils.getVisibleColumnCount(this) > 0;

                if (!bNoData || (bNoData && bVisibleColumns && oEvent.isMarked("sapUiTableInitItemNavigation") && this.getColumnHeaderVisible())) {
                    // Special handling for nodata case when the item navigation is initialized with this focus events (also adds additional
                    // tabindex attributes) -> later with next entry into the table tabindices are already set up properly
                    oInfo = TableUtils.getFocusedItemInfo(this);
                    //refocus on last focused cell
                    TableUtils.focusItem(this, oInfo.cell, oEvent);
                } else if (bNoData) {
                    this._getKeyboardExtension()._setSilentFocus(this.$("noDataCnt"));
                }

                if (!bNoData) {
                    oEvent.preventDefault();
                }
            } else if ($target.hasClass("sapUiTableCtrlAfter")) {
                this._getKeyboardExtension().setActionMode(false);
                this._getKeyboardExtension()._restoreFocusOnLastFocusedDataCell(this, oEvent);
            }
        };

        /*
         * handle the row selection via SPACE or ENTER key if key is pressed on a group header, the open state is toggled
         * @private
         */
        TableKeyboardDelegate.prototype.onkeyup = function(oEvent) {
            if (!this._bEventSapSelect === true) {
                return;
            }

            this._bEventSapSelect = false;

            // if TreeTable, and space was hit, prevent the browser from handling the event
            if (oEvent.keyCode === jQuery.sap.KeyCodes.SPACE && this._getKeyboardExtension()._getTableType() === TableExtension.TABLETYPES.TREE) {
                oEvent.preventDefault(); // To prevent the browser window to scroll down.
                oEvent.stopImmediatePropagation(true);
            } else {
                // this mimics the sapselect event but on keyup
                if (oEvent.keyCode !== jQuery.sap.KeyCodes.ENTER &&
                oEvent.keyCode !== jQuery.sap.KeyCodes.SPACE &&
                oEvent.keyCode !== jQuery.sap.KeyCodes.F4 ||
                oEvent.srcControl !== this &&
                jQuery.inArray(oEvent.srcControl,this.getRows()) === -1 &&
                jQuery.inArray(oEvent.srcControl,this.retrieveLeafColumns()) === -1) {
                    return;
                }
            }

            //get context on what was event target
            var oCellInfo = TableUtils.getCellInfo(oEvent.target) || {};
            if (oEvent.keyCode === jQuery.sap.KeyCodes.SPACE && oCellInfo.type === TableUtils.CELLTYPES.COLUMNHEADER) {
                oEvent.preventDefault(); // To prevent the browser window to scroll down.
                oEvent.stopImmediatePropagation(true);

                var iColIndex = oCellInfo.cell.attr("data-sap-ui-colindex");
                var oColumn = this.retrieveLeafColumns()[iColIndex];

                // Handle column header action
                // 1) run column select and look at if default was prevented
                var bExecuteDefault = this.fireColumnSelect({
                    column: oColumn
                });

                // 2) if default not prevented, then apply sorting behavior
                if (bExecuteDefault) {
                    this._applySortActionFromEvent(oColumn, oEvent.ctrlKey, oEvent.shiftKey);
                }
            }

            if (TableUtils.toggleGroupHeader(this, oEvent.target)) {
                oEvent.preventDefault();
                return;
            }

            //Ctrl+Enter to simulate double-click
            if (oEvent.keyCode === jQuery.sap.KeyCodes.ENTER && oEvent.ctrlKey === true) {
                this._oClickHandlerDelegate.ondblclick(oEvent);
                oEvent.preventDefault();
                return;
            }

            this._bShowMenu = true;
            this._onSelect(oEvent);
            this._bShowMenu = false;
            oEvent.preventDefault();
        };

        /*
         * handle the row selection via SPACE or ENTER key
         */
        TableKeyboardDelegate.prototype.onkeydown = function(oEvent) {
            var $this = this.$(),
                bActionMode = this._getKeyboardExtension().isInActionMode(),
                oInfo,
                bIsTreeTable = this._getKeyboardExtension()._getTableType() === TableExtension.TABLETYPES.TREE,
                $domRef = $this.find(".sapUiTableCCnt .sapUiTableCtrl td:focus").find(":sapFocusable").eq(0);

            if (bIsTreeTable && oEvent.keyCode === jQuery.sap.KeyCodes.ESCAPE) {
                // if we're toggling the expand/collapse state, we need to use ESCAPE to get
                // back to the table cell...otherwise we'll get stuck with focus on the icon
                // we may not be in action mode, but we still need to get focus back to the cell
                // leaveActionMode does the work we need
                this._getKeyboardExtension().leaveActionMode(this._getKeyboardExtension().getTable(), {});
            } else if (bIsTreeTable && oEvent.keyCode === jQuery.sap.KeyCodes.ENTER && ($domRef.hasClass("sapUiTableTreeIconNodeOpen") || $domRef.hasClass("sapUiTableTreeIconNodeClosed"))) {
                $domRef.attr("tabindex", 0).focus();
            } else if (!bActionMode && oEvent.keyCode === jQuery.sap.KeyCodes.F2 ||
                (oEvent.keyCode === jQuery.sap.KeyCodes.ENTER && oEvent.metaKey === false && oEvent.ctrlKey === false)) {
                if ($this.find(".sapUiTableCCnt .sapUiTableCtrl td:focus").length > 0) {
                    this._getKeyboardExtension().setActionMode(true, {$Dom: $this.find(".sapUiTableCCnt .sapUiTableCtrl td:focus").find(":sapFocusable")});
                    oEvent.preventDefault();
                    oEvent.stopPropagation();
                } else if ($this.find(".sapUiTableColHdrCnt .sapUiTableCtrl div:focus").length > 0) {
                    // if we're not in the content section of the table, maybe we're in an editable header
                    var InPlaceEdit = sap.ui.requireSync("sas/hc/ui/commons/InPlaceEdit");
                    var aControl = jQuery(oEvent.target).find(":sapFocusable").control();
                    if (aControl && aControl.length > 0) {
                        if (aControl[0] instanceof InPlaceEdit) {
                            this._getKeyboardExtension().setActionMode(true, {$Dom: jQuery(oEvent.target).find(":sapFocusable")});
                            oEvent.preventDefault();
                            oEvent.stopPropagation();
                        }
                    }
                }
            } else if (bActionMode &&
            oEvent.keyCode === jQuery.sap.KeyCodes.F2) {
                this._getKeyboardExtension().setActionMode(false);
            } else if (oEvent.keyCode === jQuery.sap.KeyCodes.TAB && bActionMode) {
                //Set tabindex to second table if fixed columns are used
                if (this.getTotalFrozenColumnCount() > 0) {
                    var $cell = jQuery(oEvent.target);
                    if ($cell.is("td[role=gridcell]") === false) {
                        $cell = $cell.parents("td[role=gridcell]");
                    }
                    var $row = $cell.parent("tr[data-sap-ui-rowindex]");
                    var $table = $row.closest(".sapUiTableCtrl");
                    var iRowIndex = parseInt($row.attr("data-sap-ui-rowindex"),10);
                    var $cells = $row.find("td[role=gridcell]");
                    var iColIndex = $cells.index($cell);
                    var iTableCols = $cells.length;
                    if (iColIndex === (iTableCols - 1)) {
                        var $otherTable;
                        if ($table.hasClass("sapUiTableCtrlFixed")) {
                            $otherTable = $this.find(".sapUiTableCtrl.sapUiTableCtrlScroll");
                        } else {
                            $otherTable = $this.find(".sapUiTableCtrl.sapUiTableCtrlFixed");
                            iRowIndex++;
                            if (iRowIndex === this.getVisibleRowCount()) {
                                iRowIndex = 0;
                            }
                        }
                        var $otherRow = $otherTable.find("tr[data-sap-ui-rowindex='" + iRowIndex + "']");
                        var $nextFocus = $otherRow.find(".sapUiTableCCnt td :sapFocusable[tabindex='0']").first();
                        if ($nextFocus.length > 0) {
                            $nextFocus.focus();
                            oEvent.preventDefault();
                        }
                    }
                }
            } else if (oEvent.keyCode === jQuery.sap.KeyCodes.A && (oEvent.metaKey || oEvent.ctrlKey)) {

                if (this._customCtrlAHandling === true) {
                    this._customHandleCtrlA(oEvent);
                    return;
                }

                // CTRL + A handling
                oInfo = TableUtils.getFocusedItemInfo(this);

                this._toggleSelectAll();

                TableUtils.focusItem(this, oInfo.cell, oEvent);

                oEvent.preventDefault();
                oEvent.stopImmediatePropagation(true);
            } else if (oEvent.keyCode === jQuery.sap.KeyCodes.CONTEXT_MENU || (oEvent.keyCode === jQuery.sap.KeyCodes.F10 && (oEvent.shiftKey))) {
                // SHIFT + 10 should open the context menu
                this.oncontextmenu(oEvent);
            } else if (oEvent.keyCode === jQuery.sap.KeyCodes.NUMPAD_PLUS) {
                TableUtils.toggleGroupHeader(this, oEvent.target, true);
            } else if (oEvent.keyCode === jQuery.sap.KeyCodes.NUMPAD_MINUS) {
                TableUtils.toggleGroupHeader(this, oEvent.target, false);
            } else if (oEvent.keyCode === jQuery.sap.KeyCodes.SPACE) {
                var $target = jQuery(oEvent.target);
                //determine if event target is:
                // a) somewhere in the table's column header (unless editing an editable header)
                // b) a table cell itself (not the control within, but that should be covered by action mode above)
                if ($target.hasClass("sapUiTableColCell")
                    || $target.hasClass("sapUiTableCol")
                    || ($target.hasClass("sapUiTableTd"))) {
                    oEvent.preventDefault();    //prevent default action to avoid browser scrolling
                }
            }
        };

        /**
         * Hook function for onkeydown
         * Execute custom handling for ctrl-a key press
         * @protected
         * @param {object} oEvent keypress event
         */
        TableKeyboardDelegate.prototype._customHandleCtrlA = function(oEvent) {
            // no-op...extension point for consumers
        };

        return TableKeyboardDelegate;

    }, /* bExport= */ true);
