/*!
 * (c) 2014-2018, SAS Institute Inc.
 */

// Provides control sas.hc.ui.table.Table.
sap.ui.define([
    'jquery.sap.global',
    "./TableOptionsMenuButton",
    "./ColumnDragBar",
    "./SelectAllControl",
    "./TableOptionsColumnsDialog",
    "sas/hc/ui/unified/Menu", // eslint-disable-line sashtmlcommons/no-deprecated-resource
    "sas/hc/m/MenuItem",
    "./CheckBox",
    "sas/hc/ui/core/delegate/TableClickHandlerDelegate",
    "sas/hc/ui/core/AnimationUtil",
    "./TableKeyboardDelegate",
    'sap/ui/Device',
    'sap/ui/core/Control',
    'sap/ui/core/Element',
    'sap/ui/core/IconPool',
    'sap/ui/core/IntervalTrigger',
    'sap/ui/core/library',
    'sap/ui/core/Popup',
    'sap/ui/core/ResizeHandler',
    'sap/ui/core/ScrollBar',
    './ItemNavigation',
    'sap/ui/core/theming/Parameters',
    'sap/ui/model/ChangeReason',
    'sap/ui/model/Context',
    'sap/ui/model/Filter',
    'sap/ui/model/SelectionModel',
    'sap/ui/model/Sorter',
    './ColumnGroup',
    './Column',
    './Row',
    './library',
    './TableUtils',
    './TableExtension',
    './TableAccExtension',
    './TableKeyboardExtension',
    './TablePointerExtension',
    './TableGrouping',
    'jquery.sap.dom',
    'jquery.sap.trace',
    "sap/ui/core/Core",
    "sas/hc/ui/core/Core"
],
function(jQuery,
    TableOptionsMenuButton, ColumnDragBar, SelectAllControl, TableOptionsColumnsDialog, Menu, sasHcMMenuItem,
    CheckBox, TableClickHandlerDelegate, AnimationUtil, TableKeyboardDelegate,
    Device,
    Control, Element, IconPool, IntervalTrigger, coreLibrary, Popup,
    ResizeHandler, ScrollBar, ItemNavigation, Parameters,
    ChangeReason, Context, Filter, SelectionModel, Sorter,
    ColumnGroup, Column, Row, library, TableUtils, TableExtension, TableAccExtension, TableKeyboardExtension, TablePointerExtension, TableGrouping /*, jQuerySapPlugin,jQuerySAPTrace */) {
    "use strict";


    // shortcuts
    var GroupEventType = library.GroupEventType,
        SelectionMode = library.SelectionMode,
        SelectionBehavior = library.SelectionBehavior,
        SharedDomRef = library.SharedDomRef,
        SortOrder = library.SortOrder,
        VisibleRowCountMode = library.VisibleRowCountMode,
        FilteredSelectionBehavior = library.FilteredSelectionBehavior;

    var logger = library._getLogger("Table");

    // lazy dependencies
    var Menu,
        MenuItem;

    /**
     * Constructor for a new Table.
     *
     * @param {string} [sId] id for the new control, generated automatically if no id is given
     * @param {object} [mSettings] initial settings for the new control
     *
     * @class
     * <p>
     *     Provides a comprehensive set of features for displaying and dealing with vast amounts of data. The Table control supports
     *     desktop PCs and tablet devices. On tablets, special consideration should be given to the number of visible columns
     *     and rows due to the limited performance of some devices.
     * </p>
     * <p>
     *     In order to keep the document DOM as lean as possible, the Table control reuses its DOM elements of the rows.
     *     When the user scrolls, only the row contexts are changed but the rendered controls remain the same. This allows
     *     the Table control to handle huge amounts of data. Nevertheless, restrictions apply regarding the number of displayed
     *     columns. Keep the number as low as possible to improve performance. Due to the nature of tables, the used
     *     control for column templates also has a big influence on the performance.
     * </p>
     * <p>
     *     The Table control relies completely on data binding, and its supported feature set is tightly coupled to
     *     the data model and binding being used.
     * </p>
     *
     * @extends sap.ui.core.Control
     *
     * @constructor
     * @public
     * @alias sas.hc.ui.table.Table
     * @ui5-metamodel This control/element also will be described in the UI5 (legacy) designtime metamodel
     */
    var Table = Control.extend("sas.hc.ui.table.Table", /** @lends sas.hc.ui.table.Table.prototype */ { metadata : {

        library : "sas.hc.ui.table",
        publicMethods: [
            "isFrozenColumn",
            "freezeColumn",
            "unfreezeColumn",
            "addTableOptionsCallback",
            "getCellDom",
            "highlightColumn",
            "removeHighlighting",
            "getTableOptionsMenu"
        ],
        properties : {

            /**
             * Width of the Table.
             */
            width : {type : "sap.ui.core.CSSSize", group : "Dimension", defaultValue : 'auto'},

            /**
             * Height of a row of the Table in pixel.
             */
            rowHeight : {type : "int", group : "Appearance", defaultValue : null},

            /**
             * Height of the column header of the Table in pixel.
             */
            columnHeaderHeight : {type : "int", group : "Appearance", defaultValue : null},

            /**
             * Flag whether the column header is visible or not.
             */
            columnHeaderVisible : {type : "boolean", group : "Appearance", defaultValue : true},

            /**
             * Number of visible rows of the table.
             */
            visibleRowCount : {type : "int", group : "Appearance", defaultValue : 10},

            /**
             * First visible row.
             */
            firstVisibleRow : {type : "int", group : "Appearance", defaultValue : 0},

            /**
             * Selection mode of the Table. This property controls whether single or multiple rows can be selected and
             * how the selection can be extended. It may also influence the visual appearance.
             */
            "selectionMode": {
                type: "sas.hc.ui.table.SelectionMode",
                group: "Behavior",
                defaultValue: SelectionMode.None
            },

            /**
             * Selection behavior of the Table. This property defines whether the row selector is displayed and whether the row, the row selector or both
             * can be clicked to select a row.<br />
             * If selectionMode is set to "Single", "RowOnly" is recommended for this property. If a different value is used a warning will be shown on the console.
             */
            selectionBehavior : {type : "sas.hc.ui.table.SelectionBehavior", group : "Behavior", defaultValue : SelectionBehavior.RowSelector},

            /**
             * Zero-based index of selected item. Index value for no selection is -1.
             * When multi-selection is enabled and multiple items are selected, the method returns
             * the lead selected item. Sets the zero-based index of the currently selected item. This method
             * removes any previous selections. When the given index is invalid, the call is ignored.
             */
            selectedIndex : {type : "int", group : "Appearance", defaultValue : -1},

            /**
             * Flag whether the controls of the Table are editable or not (currently this only controls the background color in certain themes!)
             */
            editable : {type : "boolean", group : "Behavior", defaultValue : true},

            /**
             * The <code>threshold</code> defines how many additional (not yet visible records) shall be pre-fetched to enable smooth
             * scrolling. The threshold is always added to the <code>visibleRowCount</code>. If the <code>visibleRowCount</code> is 10 and the
             * <code>threshold</code> is 100, there will be 110 records fetched with the initial load.
             * If the <code>threshold</code> is lower than the <code>visibleRowCount</code>, the <code>visibleRowCount</code> will be used as
             * the <code>threshold</code>. If the value is 0 then the thresholding is disabled.
             */
            threshold : {type : "int", group : "Appearance", defaultValue : 100},

            /**
             * Flag to enable or disable column reordering
             */
            enableColumnReordering : {type : "boolean", group : "Behavior", defaultValue : true},

            /**
             * Flag to enable or disable column grouping. (experimental!)
             */
            enableGrouping : {type : "boolean", group : "Behavior", defaultValue : false},

            /**
             * Flag to show or hide the column visibility menu. This menu will get displayed in each
             * generated column header menu. It allows to show or hide columns
             */
            showColumnVisibilityMenu : {type : "boolean", group : "Appearance", defaultValue : false},

            /**
             * Flag whether to show the no data overlay or not once the table is empty. If set to false
             * the table will just show a grid of empty cells
             */
            showNoData : {type : "boolean", group : "Appearance", defaultValue : true},

            /**
             * This defines how the table handles the visible rows in the table. The default behavior is,
             * that a fixed row count is defined. If you change it to auto the visibleRowCount property is
             * changed by the table automatically. It will then adjust its maximum row count to the space it is
             * allowed to cover (limited by the surrounding container) and its minimum row count to the value of
             * the property minAutoRowCount (default value : 5) In manual mode the user can change
             * the visibleRowCount interactively.
             * @since 1.9.2
             * @see sas.hc.ui.table.VisibleRowCountMode
             */
            visibleRowCountMode : {type : "sas.hc.ui.table.VisibleRowCountMode", group : "Appearance", defaultValue : VisibleRowCountMode.Fixed},

            /**
             * This property is used to set the minimum count of visible rows when the property visibleRowCountMode is set to Auto.
             * For any other visibleRowCountMode, it is ignored.
             */
            minAutoRowCount : {type : "int", group : "Appearance", defaultValue : 5},

            /**
             * Number of columns that are fix on the left. When you use a horizontal scroll bar, only
             * the columns which are not fixed, will scroll. Fixed columns need a defined width for the feature to work.
             * Please note that the aggregated width of all fixed columns must not exceed the table width since there
             * will be no scrollbar for fixed columns.
             */
            fixedColumnCount : {type : "int", group : "Appearance", defaultValue : 0},

            /**
             * Number of rows that are fix on the top. When you use a vertical scroll bar, only the rows which are not fixed, will scroll.
             */
            fixedRowCount : {type : "int", group : "Appearance", defaultValue : 0},

            /**
             * Number of rows that are fix on the bottom. When you use a vertical scroll bar, only the rows which are not fixed, will scroll.
             * @since 1.18.7
             */
            fixedBottomRowCount : {type : "int", group : "Appearance", defaultValue : 0},

            /**
             * Flag whether to show or hide the column menu item to freeze or unfreeze a column.
             * @since 1.21.0
             */
            "enableColumnFreeze": {type: "boolean", group: "Behavior", defaultValue: true},

            /**
             * Flag whether to enable or disable the context menu on cells to trigger a filtering with the cell value.
             * @since 1.21.0
             */
            enableCellFilter : {type : "boolean", group : "Behavior", defaultValue : false},

            /**
             * Setting this property to true will show an overlay on top of the Table content and users cannot click anymore on the Table content.
             * @since 1.21.2
             */
            showOverlay : {type : "boolean", group : "Appearance", defaultValue : false},

            /**
             * Specifies if a select all button should be displayed in the top left corner. This button is only displayed
             * if the row selector is visible and the selection mode is set to any kind of multi selection.
             * @since 1.23.0
             */
            enableSelectAll : {type : "boolean", group : "Behavior", defaultValue : true},

            /**
             * Set this parameter to true to implement your own filter behaviour. Instead of the filter input box a button
             * will be rendered for which' press event (customFilter) you can register an event handler.
             * @since 1.23.0
             */
            enableCustomFilter : {type : "boolean", group : "Behavior", defaultValue : false},

            /**
             * Set this parameter to true to make the table handle the busy indicator by its own.
             * The table will switch to busy as soon as it scrolls into an unpaged area. This feature can only
             * be used when the navigation mode is set to scrolling.
             * @since 1.27.0
             */
            enableBusyIndicator : {type : "boolean", group : "Behavior", defaultValue : false},

            "highlightClass": {type: "string", group: "Appearance", defaultValue: 'tableColumnHighlight'},

            /** Minimum column width allowed. Default: <code>50</code> */
            "minColumnWidth": {type: "int", group: "Dimension", defaultValue: 50},

            "showSelectionControls": {type: "string", group: "Behavior", defaultValue: "true"},

            /** Flag whether to show or hide the table options overflow icon. Default value: <code>true</code> */
            "showTableOptions": {type: "boolean", group: "Behavior", defaultValue: true},

            /** Flag whether to include the manage columns item on the context menu. Default value: <code>false</code> */
            "showManageColumnsMenuItem": {type: "boolean", group: "Behavior", defaultValue: false},

            /**
             * NOTE: this feature is being taken out for (at least) the first part of 4.0
             * and will be re-implemented at a later date.
             * Leaving the property here for compatibility reasons.
             */
            "allowReflow": {type: "boolean", group: "Appearance", defaultValue: false},

            "showSummaryHeader": {type: "boolean", group: "Behavior", defaultValue: false},

            /**
             * Height of the summary header of the Table in pixels.
             */
            summaryHeaderHeight : {type : "int", group : "Appearance", defaultValue : null},

            /**
             * Use pixel-based scrolling, as opposed to row-based scrolling.
             * If setting this property to false, please review: http://sww.sas.com/saspedia/HTML_Commons_Table#Sizing_and_Scrolling
             */
            enablePixelScrolling : {type : "boolean", group : "Appearance", defaultValue : true},

            /**
             * Whether or not to support double-click events on table rows. Setting to true introduces a brief delay
             * between single click and firing the associated event so that we can determine if this is really a double-click.
             * Calling the attachDoubleClick method on this table will automatically set this to true
             */
            "enableDoubleClick": { type: "boolean", defaultValue: false },

            /** Turn on or off whether the table displays alternate row shading. Default: <code>false</code> */
            "showAlternateRowShading": { type: "boolean", group : "Appearance", defaultValue: false },

            /** Turn on or off whether the table displays the outer border. Default: <code>false</code> */
            "showOuterBorder": {type: "boolean", group: "Appearance", defaultValue: false},

            /**
             * Allowed diff in height/width during interval sizing calculation.
             * Use to prevent infinite sizing adjustments
             * See: S1234853
             * @private
             */
            "_allowedSizeDiff": {type: "int", group: "Behavior", defaultValue: 5},

            /**
             * This allows user to use their own extensions
             */
            useExternalExtensions : {type : "boolean", group : "Appearance", defaultValue : false},

            /**
             * If set to true, this control remembers and retains the selection of the items after a binding update has been performed (e.g. sorting, filtering).
             */
            rememberSelections : {type : "boolean", group : "Behavior", defaultValue : false},

            /**
             * The filteredSelectionBehavior property determines how the existing selection state is managed when a filter is applied
             * to the assigned model via the binding.  The default value of 'RetainAll' will retain all selected items even for items
             * that have been filtered out of view.  The alternative value of 'RetainMatching' will only retain selected items that
             * match the newly filtered view.
             */
            filteredSelectionBehavior : {type : "sas.hc.ui.table.FilteredSelectionBehavior", group : "Behavior", defaultValue: FilteredSelectionBehavior.RetainMatching },

            /**
             * The selectionIdProperty allows consumers using the 'rememberSelections' feature to specify a property of the model that
             * contains a unique identifier for each item that can be used when managing the selection state for the table.
             * It is recommended to use this feature when the table is bound to a model that may have items inserted or deleted
             * which would result in context paths referring to different items as the data changes.
             */
            selectionIdProperty : {type : "string", group : "Behavior", defaultValue: null}

        },
        defaultAggregation : "columns",
        aggregations : {

            /**
             * Control or text of title section of the Table (if not set it will be hidden)
             */
            title : {type : "sap.ui.core.Control", altTypes : ["string"], multiple : false},

            /**
             * Control or text of footer section of the Table (if not set it will be hidden)
             */
            footer : {type : "sap.ui.core.Control", altTypes : ["string"], multiple : false},

            /**
             * Toolbar of the Table (if not set it will be hidden)
             * @deprecated Since version 9.0. This aggregation is deprecated, use the <code>extension</code> aggregation instead.
             */
            toolbar : {type : "sap.ui.core.Toolbar", multiple : false, deprecated: true},

            /**
             * Extension section of the Table (if not set it will be hidden)
             */
            extension : {type : "sap.ui.core.Control", multiple : true, singularName : "extension"},

            /**
             * Columns of the Table
             */
            columns : {type : "sas.hc.ui.table.ColumnBase", multiple : true, singularName : "column", bindable : "bindable"},

            /**
             * Rows of the Table
             */
            rows : {type : "sas.hc.ui.table.Row", multiple : true, singularName : "row", bindable : "bindable"},

            /**
             * MenuItems for the Table options
             */
            menuItems : {type : "sas.hc.m.MenuItem", multiple : true, singularName : "menuItem", bindable : "bindable"},

            /**
             * The value for the noData aggregation can be either a string value or a control instance.
             * The control is shown, in case there is no data for the Table available. In case of a string
             * value this will simply replace the no data text.
             */
            noData : {type : "sap.ui.core.Control", altTypes : ["string"], multiple : false},

            "tableOptionsMenuButton": {
                type: "sas.hc.ui.table.TableOptionsMenuButton",
                multiple: false,
                visibility: "hidden"
            },

            "tableOptionsColumnsDialog": {
                type: "sas.hc.ui.table.TableOptionsColumnsDialog",
                multiple: false,
                visibility: "hidden"
            },

            "columnDragBar": {type: "sas.hc.ui.table.ColumnDragBar", multiple: false},

            /**
             * The value for the allColumnsHidden aggregation can be either a string value or a control instance. The control is shown, in case there is all columns hidden for the Table.
             */
            "allColumnsHidden" : {type : "sap.ui.core.Control", altTypes : ["string"], multiple : false},

            /**
             * Select-all checkbox in Table header
             */
            "selectAllControl" : {type : "sap.ui.core.Control",  multiple : false}
        },
        associations : {

            /**
             * Group By Column (experimental!)
             */
            groupBy : {type : "sas.hc.ui.table.Column", multiple : false},

            /**
             * Association to controls / ids which label this control (see WAI-ARIA attribute aria-labelledby).
             */
            ariaLabelledBy : {type : "sap.ui.core.Control", multiple : true, singularName : "ariaLabelledBy"}
        },
        events : {

            /**
             * fired when the row selection of the table has been changed (the event parameters can be used to determine
             * selection changes - to find out the selected rows you should better use the table selection API)
             */
            rowSelectionChange : {
                parameters : {

                    /**
                     * row index which has been clicked so that the selection has been changed (either selected or deselected)
                     */
                    rowIndex : {type : "int"},

                    /**
                     * binding context of the row which has been clicked so that selection has been changed
                     */
                    rowContext : {type : "object"},

                    /**
                     * array of row indices which selection has been changed (either selected or deselected)
                     */
                    rowIndices : {type : "int[]"},

                    /**
                     * indicator if "select all" function is used to select rows
                     */
                    selectAll : {type : "boolean"},

                    /**
                     * indicates that the event was fired due to an explicit user interaction like clicking the row header
                     * or using the keyboard (SPACE or ENTER) to select a row or a range of rows.
                     */
                    userInteraction: {type: "boolean"}
                }
            },

            /**
             * Fired before the columns dialog is opened.
             * The columns dialog is opened via the table option in the upper right corner of the table.
             */
            "beforeColumnsDialogOpen": {},

            /**
             * Fired after the columns dialog is opened.
             * The columns dialog is opened via the table option in the upper right corner of the table.
             */
            "afterColumnsDialogOpen": {},

            /**
             * Fired before the columns dialog is closed.
             * The columns dialog is opened via the table option in the upper right corner of the table.
             */
            "beforeColumnsDialogClose": {},

            /**
             * Fired after the columns dialog is closed.
             * The columns dialog is opened via the table option in the upper right corner of the table.
             */
            "afterColumnsDialogClose": {},

            /**
             * Fired when a single click, single tap, and ENTER key occurs on a row.
             * The <code>defaultAction</code> event will notify the consumer to perform default action for the table.
             * @public
             * @since 3.0
             */
            "defaultAction" : {
                parameters: {
                    /**
                     * Index of this row within the underlying data model
                     */
                    rowIndex: { type: "int" },

                    /**
                     * Actual DOM element that was the target of the event
                     */
                    target: { type: "object" },

                    /**
                     * jQuery object corresponding to the row that was the target of the event
                     */
                    item: { type: "object" }
                }
            },

            /**
             * Fired when a double-click or double-tab occurs on a row or cell.
             * @public
             * @since 6.0
             */
            "doubleClick": {
                parameters: {
                    /**
                     * Index of the double-clicked row within the underlying data model
                     */
                    rowIndex: { type: "int" },

                    /**
                     * The data model context for the double-clicked row
                     */
                    rowBindingContext: { type: "sap.ui.model.Context" },

                    /**
                     * Index of the double-clicked cell within the containing row
                     */
                    columnIndex: { type: "int" },

                    /**
                     * The data model context for the double-clicked cell
                     */
                    cellBindingContext: { type: "sap.ui.model.Context" },

                    /**
                     * Actual DOM element that was double-clicked
                     */
                    target: { type: "object" },

                    /**
                     * DOM element corresponding to the cell that was double-clicked
                     */
                    cellDomRef: { type: "object" },

                    /**
                     * DOM element corresponding to the row that was double-clicked
                     */
                    rowDomRef: { type: "object" }
                }
            },

            /**
             * fired when a column of the table has been selected
             */
            columnSelect : {allowPreventDefault : true,
                parameters : {

                    /**
                     * reference to the selected column
                     */
                    column : {type : "sas.hc.ui.table.Column"}
                }
            },

            /**
             * fired when a table column is resized.
             */
            columnResize : {allowPreventDefault : true,
                parameters : {

                    /**
                     * resized column.
                     */
                    column : {type : "sas.hc.ui.table.Column"},

                    /**
                     * new width of the table column as CSS Size definition.
                     */
                    width : {type : "sap.ui.core.CSSSize"}
                }
            },

            /**
             * fired when a table column is moved.
             */
            columnMove : {allowPreventDefault : true,
                parameters : {

                    /**
                     * moved column.
                     */
                    column : {type : "sas.hc.ui.table.Column"},

                    /**
                     * new position of the column.
                     */
                    newPos : {type : "int"}
                }
            },

            /**
             * fired when the table is sorted.
             */
            sort : {allowPreventDefault : true,
                parameters : {

                    /**
                     * sorted column.
                     */
                    column : {type : "sas.hc.ui.table.Column"},

                    /**
                     * Sort Order
                     */
                    sortOrder : {type : "sas.hc.ui.table.SortOrder"},

                    /**
                     * If column was added to sorter this is true. If new sort is started this is set to false
                     */
                    columnAdded : {type : "boolean"}
                }
            },

            /**
             * fired when the table is filtered.
             */
            filter : {allowPreventDefault : true,
                parameters : {

                    /**
                     * filtered column.
                     */
                    column : {type : "sas.hc.ui.table.Column"},

                    /**
                     * filter value.
                     */
                    value : {type : "string"}
                }
            },

            /**
             * fired when the table is grouped (experimental!).
             */
            group : {allowPreventDefault : true,
                parameters : {
                    /**
                     * grouped column.
                     */
                    column : {type : "sas.hc.ui.table.Column"}
                }
            },

            /**
             * fired when the visibility of a table column is changed.
             */
            columnVisibility : {allowPreventDefault : true,
                parameters : {

                    /**
                     * affected column.
                     */
                    column : {type : "sas.hc.ui.table.Column"},

                    /**
                     * new value of the visible property.
                     */
                    visible : {type : "boolean"}
                }
            },

            /**
             * fired when the user clicks a cell of the table (experimental!).
             * @since 1.21.0
             */
            cellClick : {allowPreventDefault : true,
                parameters : {
                    /**
                     * The control of the cell.
                     */
                    cellControl : {type : "sap.ui.core.Control"},

                    /**
                     * DOM reference of the clicked cell. Can be used to position the context menu.
                     */
                    cellDomRef : {type : "Object"},

                    /**
                     * Row index of the selected cell.
                     */
                    rowIndex : {type : "int"},

                    /**
                     * Column index of the selected cell. This is the index of visible columns and might differ from
                     * the index maintained in the column aggregation.
                     */
                    columnIndex : {type : "int"},

                    /**
                     * Column ID of the selected cell.
                     */
                    columnId : {type : "string"},

                    /**
                     * Row binding context of the selected cell.
                     */
                    rowBindingContext : {type : "sap.ui.model.Context"}
                }
            },

            /**
             * fired when the user clicks a cell of the table.
             * @since 1.21.0
             */
            cellContextmenu : {allowPreventDefault : true,
                parameters : {
                    /**
                     * The control of the cell.
                     */
                    cellControl : {type : "sap.ui.core.Control"},

                    /**
                     * DOM reference of the clicked cell. Can be used to position the context menu.
                     */
                    cellDomRef : {type : "Object"},

                    /**
                     * Row index of the selected cell.
                     */
                    rowIndex : {type : "int"},

                    /**
                     * Column index of the selected cell. This is the index of visible columns and might differ from
                     * the index maintained in the column aggregation.
                     */
                    columnIndex : {type : "int"},

                    /**
                     * Column ID of the selected cell.
                     */
                    columnId : {type : "string"},

                    /**
                     * Row binding context of the selected cell.
                     */
                    rowBindingContext : {type : "sap.ui.model.Context"}
                }
            },

            /**
             * fired when a column of the table should be freezed
             * @since 1.21.0
             */
            columnFreeze : {allowPreventDefault : true,
                parameters : {

                    /**
                     * reference to the column to freeze
                     */
                    column : {type : "sas.hc.ui.table.Column"}
                }
            },

            /**
             * This event is triggered when the custom filter item of the column menu is pressed. The column on which the event was triggered is passed as parameter.
             * @since 1.23.0
             */
            customFilter : {
                /**
                 * The column instance on which the custom filter button was pressed.
                 */
                column : {type : "sas.hc.ui.table.Column"},

                /**
                 * Filter value.
                 */
                value : {type : "string"}
            },

            /**
             * This event gets fired when the first visible row is changed. It should only be used by composite controls.
             * The event even is fired when setFirstVisibleRow is called programmatically.
             * @since 1.37.0
             * @protected
             */
            firstVisibleRowChanged : {
                /**
                 * First visible row
                 */
                firstVisibleRow : {type : "int"}
            },

            /**
             * This event gets fired when the busy state of the table changes. It should only be used by composite controls.
             * @since 1.37.0
             * @protected
             */
            busyStateChanged : {
                /**
                 * busy state
                 */
                busy : {type : "boolean"}
            },

            /**
             * Fired after <code>rows</code> binding is updated and processed by the control.
             */
            updateFinished : {
                parameters : {
                    /**
                     * The reason of the update, e.g. change, filter, sort, unbindAggregation.
                     */
                    reason : {type : "string"}
                }
            },

            /**
             * fired when the remembered selection context paths array values change
             * is useful only when rememberSelection is set to true
             */
            selectionUpdated: {
                parameters: {
                    selectionKeys: {type : "string[]"}
                }
            }
        },
        designTime : true
    },

    constructor: function () {
        Control.apply(this, arguments);

        this._contextMenuItems = [];
        this.getContextMenu();      //force creation of default menu
    }
    });

    Table.M_EVENTS = {
        'beforeColumnsDialogOpen': 'beforeColumnsDialogOpen',
        'afterColumnsDialogOpen': 'afterColumnsDialogOpen',
        'beforeColumnsDialogClose': 'beforeColumnsDialogClose',
        'afterColumnsDialogClose': 'afterColumnsDialogClose',
        'defaultAction': 'defaultAction',
        'doubleClick': 'doubleClick'
    };

    // =============================================================================
    // BASIC CONTROL API
    // =============================================================================

    IconPool.insertFontFaceStyle();

    /**
     * Initialization of the Table control
     * @private
     */
    Table.prototype.init = function() {

        logger.debug("Initializing new instance.");

        // load modules used by collator
        // loaded here so ltjs is still lazilly-loaded
        // (at Table instantiation-time, rather then when library is loaded)

        // The following line was moved to the sas.hc.ui.table.Column module as part of the "requires cleanup"
        // effort.  But, that might result in ltjs being loaded earlier in the app startup process and slow
        // things down.  Keep an eye on this...
        //jQuery.sap.require('sas.ltjs.sasicu.util.SASCollatorUtil');

        this._iBaseFontSize = parseFloat(jQuery("body").css("font-size")) || 16;

        // create an information object which contains always required infos
        this._oResBundle = sap.ui.getCore().getLibraryResourceBundle("sas.hc.ui.table");
        this._bRtlMode = sap.ui.getCore().getConfiguration().getRTL();

        this._bBindingLengthChanged = false;
        this._mTimeouts = {};

        /*
         * Updates the row binding contexts and synchronizes the row heights. This function will be called by updateRows
         */
        this._lastCalledUpdateRows = 0;
        this._iBindingTimerDelay = 50;

        var that = this;

        this._performUpdateRows = function(sReason) {
            // if the row update came from the user expanding a tree node while also
            // vertically scrolled all the way to the bottom, prevent the pixel-scrolling
            // adjustment as we don't need to adjust the table position
            var bFromTreeBottomNodeExpansion = sReason === ChangeReason.Expand && this._isScrolledToBottom() === true;

            // update only if control not marked as destroyed (could happen because updateRows is called during destroying the table)
            if (!that.bIsDestroyed) {
                that._lastCalledUpdateRows = Date.now();
                that._updateBindingContexts(undefined, undefined, sReason);

                if (!that._bInvalid) {
                    // subsequent DOM updates are only required if there is no rendering to be expected

                    that._updateTableContent();

                    that._getAccExtension().updateAccForCurrentCell(false);
                    that._updateSelection();
                    that._accUpdateARIARowIndices();    //update ARIA row indices

                    //we want to let the cells of the rows with updated binding contexts to actually update before we do this next part
                    sap.ui.getCore().applyChanges();    //let UI5 do its thing

                    var oTableSizes = that._collectTableSizes();
                    that._updateRowHeader(oTableSizes.tableRowHeights);
                    that._syncColumnHeaders(oTableSizes);

                    if (TableUtils.isVariableRowHeightEnabled(that) && bFromTreeBottomNodeExpansion === false) {
                        that._adjustTablePosition();
                    }

                    if (that._bBindingLengthChanged || TableUtils.isVariableRowHeightEnabled(that)) {
                        that._updateVSb(oTableSizes);
                    }

                    // If this is a TreeTable or LargeTree, make sure the table is
                    // wide enough to display all expanded subnodes
                    if (TableGrouping.isTreeMode(that)) {
                        that._updateHSb(oTableSizes);
                    }
                }

                that._mTimeouts.bindingTimer = undefined;
                // Helper event for testing
                that.fireEvent("_rowsUpdated");
            }

            that._bBindingLengthChanged = false;
        };

        // basic selection model (by default the table uses multi selection)
        this._initSelectionModel(SelectionModel.MULTI_SELECTION);

        this._aTableHeaders = [];

        // columns to cells map
        this._aIdxCols2Cells = [];

        // visible columns
        this._aVisibleColumns = [];

        // flag whether the editable property should be inherited or not
        this._bInheritEditableToControls = false;

        // text selection for column headers?
        this._bAllowColumnHeaderTextSelection = false;

        // determine whether jQuery version is less than 1.8 (height and width behaves different!!)
        this._bjQueryLess18 = jQuery.sap.Version(jQuery.fn.jquery).compareTo("1.8") < 0;
        this._iDataRequestedCounter = 0;

        this._iBindingLength = 0;
        this._iTableRowContentHeight = 0;
        this._bFirstRendering = true;

        // F6 Handling is done in TableRenderer to make sure the table content gets the focus. The
        // Toolbar has its own F6 stop.
        // this.data("sap-ui-fastnavgroup", "true", true); // Define group for F6 handling

        this._bInvalid = true;

        this._bIsScrollVertical = null;

        this.oResBundle = sap.ui.getCore().getLibraryResourceBundle("sas.hc.ui.table");

        this._initOptionsButton();

        this._initDelegates();

        this.setSelectAllControl(this._createSelectAllControl());

        // store all timeout id's so they can be cleared together
        this._aTimeoutIds = [];

        // used to signal usage of hook function at sdk layer
        this._customCtrlAHandling = true;
        this._bCustomSortIcon = true;
        // S1121064
        this._openMenuOnColumnSelect = true;
        // HTMLCOMMONS-4581: Add summary header style class to main element
        if (this.getShowSummaryHeader()) {
            // show column summary headers
            this.addStyleClass("sasUiTableSHdr");
        }

        // minimum width of a table column in pixel:
        // should at least be larger than the paddings for cols and cells!
        this._iColMinWidth = this.getMinColumnWidth();
        if ('ontouchstart' in document) {
            this._iColMinWidth = Math.max(88, this._iColMinWidth);
        }

        // Used during column resize. Attempts to prevent errors when resizing %-based columns to an extreme.
        this._iMinPercentWidth = 2;

        this.bCustomTableCellAria = true;
        this.bSetMaxHeightForTableCellInVisibleRowCountAutoMode = false;
        this._bCustomAddSort = true;

        // keyboard navigation flags
        this._skipVScroll = false;

        // Cache unused but created rows in case they are needed later.
        // Must make sure to clear the cache when the row template is reset.
        this._mCachedRows = {};

        // the number of pixels the scrollbar will be moved per-nudge
        this._iHorizontalNudgeAmount = 30;
        // the distance from the edge of the table where scroll nudging will initiate
        this._iHorizontalNudgeThreshold = 40;
        // the time delay between each nudge operation when auto-nudging is occuring
        this._iHorizontalNudgeRepeatDelay = 150;

        // Allow consumers to restore original columns widths on Table resize. Turned off by default.
        this._bRestoreColumnWidthsOnResize = false;

        // During certain operations we want to prevent restoring columns widths.
        // This provides the mechanism to stop restoring column widths.
        // Clients that need to temporarily turn off this feature should capture the original
        // value before changing it and then resetting back to the captured value. This will
        // help in case a consumer wishes to explicitly turn it off all the time.
        this._bPreventRestoreColumWidths = false;

        // the amount of column width change for non-mouse column resizing
        this._iColumnWidthAdjustLength = 4;

        // The number of mandatory Columns.
        // It was too expensive to dynamically compute this each time it was needed.
        // A negative value indicates that the count needs to be recalculated.
        this._iMandatoryColumnCount = 0;

        // reset the column group count
        this._iColumnGroupCount = 0;

        //Initialize remembered selection cache
        this._aSelectedKeys = [];

        // add the default manage columns menu item if there are no column groups present
        if (!this.hasColumnGroups()) {
            this.addMenuItem(this._createManageColumnsMenuItem(), true);
        }
    };

    Table.prototype._createManageColumnsMenuItem = function() {
        return new sasHcMMenuItem({
            id: this.getId() + "-menuitem-columns",
            text: this.oResBundle.getText("table.tableOptions.optionsMenu.columnsMenuItem.label"),
            press: this._showColumnsDialog.bind(this)
        });
    };

    Table.prototype.applySettings = function(mSettings, oScope) {
        if (Control.prototype.applySettings) {
            Control.prototype.applySettings.apply(this, arguments);
        }

        //S1416191
        if (!(mSettings && mSettings.useExternalExtensions === true)) {
            this._attachExtensions();
            this._getPointerExtension().cleanupColumnResizeEvents();
        }
    };

    /**
     * Attach table extensions
     * @private
     */
    Table.prototype._attachExtensions = function() {
        if (this._bExtensionsInitialized) {
            return;
        }

        //HTMLCOMMONS-6821
        // TableExtension.enrich(this, TablePointerExtension);
        TableExtension.enrich(this, this._getPointerExtensionType());

        //HTMLCOMMONS-6546
        //TableExtension.enrich(this, TableKeyboardExtension);
        TableExtension.enrich(this, this._getKeyboardExtensionType());

        TableExtension.enrich(this, TableAccExtension); //Must be registered after keyboard to reach correct delegate order
        this._bExtensionsInitialized = true;
    };


    /**
     * Termination of the Table control
     * @private
     */
    Table.prototype.exit = function() {

        logger.debug("Exiting instance", this.getId());

        // destroy the child controls
        this._bExitCalled = true;

        this._resetRowTemplate();

        // destroy helpers
        this._detachExtensions();

        // cleanup
        this._cleanUpTimers();
        this._detachEvents();

        this._cleanupColumnDragBar();

        this._destroyDelegates();

        this._clearTimeouts();
    };


    /**
     * Detach table extensions
     * @private
     */
    Table.prototype._detachExtensions = function(){
        if (!this._bExtensionsInitialized) {
            return;
        }
        this._getPointerExtension().destroy();
        this._getKeyboardExtension().destroy();
        this._getAccExtension().destroy();
        delete this._bExtensionsInitialized;
    };


    /**
     * theme changed
     * @private
     */
    Table.prototype.onThemeChanged = function() {
        if (this.getDomRef()) {
            this.invalidate();
        }
    };

    /**
     * Determines the row heights of the fixed and scroll area.
     * @private
     */
    Table.prototype._collectRowHeights = function() {
        var oDomRef = this.getDomRef();
        if (!oDomRef) {
            return [];
        }

        var aFixedRowItems = oDomRef.querySelectorAll(".sapUiTableCCnt .sapUiTableCtrlFixed > tbody > tr");
        var aScrollRowItems = oDomRef.querySelectorAll(".sapUiTableCCnt .sapUiTableCtrlScroll > tbody > tr");
        var aRowItemHeights = [];
        for (var i = 0; i < aScrollRowItems.length; i++) {
            var iFixedRowHeight = 0;
            if (aFixedRowItems[i]) {
                var oFixedRowClientRect = aFixedRowItems[i].getBoundingClientRect();
                iFixedRowHeight = oFixedRowClientRect.bottom - oFixedRowClientRect.top;
            }

            var oScrollRowClientRect = aScrollRowItems[i].getBoundingClientRect();
            var iRowHeight = oScrollRowClientRect.bottom - oScrollRowClientRect.top;

            aRowItemHeights.push(Math.max(iFixedRowHeight, iRowHeight));
        }

        return aRowItemHeights;
    };

    /**
     * Resets the height style property of all TR elements of the table
     * @private
     */
    Table.prototype._resetRowHeights = function() {
        var iRowHeight = this.getRowHeight();

        var sRowHeight = "";
        if (iRowHeight) {
            sRowHeight = iRowHeight + "px";
        }

        var oDomRef = this.getDomRef();
        if (oDomRef) {
            var aRowItems = oDomRef.querySelectorAll(".sapUiTableCCnt .sapUiTableCtrlFixed > tbody > tr, .sapUiTableCCnt .sapUiTableCtrlScroll > tbody > tr");
            for (var i = 0; i < aRowItems.length; i++) {
                aRowItems[i].style.height = sRowHeight;
            }
        }
    };

    /**
     * Scroll horizontally a little bit
     * @param {boolean} bForward the direction to scroll
     * @private
     */
    Table.prototype._horizontalScrollNudge = function(bForward) {
        var iNudgeAmount = this._iHorizontalNudgeAmount,
            $hsb = this.$(SharedDomRef.HorizontalScrollBar),
            fnScrollPos = (this._bRtlMode ? $hsb.scrollLeftRTL : $hsb.scrollLeft).bind($hsb),
            iScrollAmount;
        // account for nudge direction
        if (bForward === false) {
            iNudgeAmount = iNudgeAmount * -1;
        }
        iScrollAmount = fnScrollPos() + iNudgeAmount;
        fnScrollPos(iScrollAmount);
    };

    /**
     * Determines if the Table should have a horizontal scrollbar.
     *
     * @private
     * @returns {Boolean} Returns true if the table should have a horizontal scrollbar. False otherwise.
     */
    Table.prototype._shouldHaveHSB = function() {

        var oDomRef = this.getDomRef();
        if (!oDomRef) {
            return false;
        }

        var oCCnt = oDomRef.querySelector(".sapUiTableCCnt");
        if (!oCCnt) {
            return false;
        }

        var $tableCtrlScroll = jQuery(this.getDomRef("tableCtrlCnt")).find('table.sapUiTableCtrlScroll'),
            $tableCtrlFixed = this.$("table-fixed"),
            $tableCCnt = jQuery(oCCnt),
            $rowHdrScr = this.$("sapUiTableRowHdrScr"),
            iRowHeaderWidth = (TableUtils.hasRowHeader(this) === true && $rowHdrScr.length > 0) ? $rowHdrScr.outerWidth() : 0;

        return ($tableCtrlScroll.outerWidth() + $tableCtrlFixed.outerWidth()) > ($tableCCnt.outerWidth() - iRowHeaderWidth);
    };

    /**
     * Determines if the Table currently has a horizontal scrollbar.
     *
     * @private
     * @returns {Boolean} Returns true if the Table currently has a horizontal scrollbar. False otherwise.
     */
    Table.prototype._hasHSB = function() {
        return jQuery(this.getDomRef('hsb-content')).is(':visible');
    };

    /**
     * Determines if the Table currently has a vertical scrollbar.
     *
     * @private
     * @returns {Boolean} Returns true if the Table currently has a vertical scrollbar. False otherwise.
     */
    Table.prototype._hasVSB = function() {
        return jQuery(this.getDomRef('vsb-content')).is(':visible');
    };

    /**
     * Determines all needed table size at one dedicated point,
     * for avoiding layout thrashing through read/write UI operations.
     * @private
     */
    Table.prototype._determineAvailableSpace = function() {
        var oDomRef = this.getDomRef();

        if (oDomRef && oDomRef.parentNode) {
            var oCCnt = oDomRef.querySelector(".sapUiTableCCnt");
            if (oCCnt) {
                var iUsedHeight = Math.max(0, oDomRef.scrollHeight - oCCnt.clientHeight);   //S1422883: cannot use negative space

                // return jQuery(oDomRef.parentNode).height() - iUsedHeight;
                var iAvailableSpace = jQuery(oDomRef.parentNode).height() - iUsedHeight;

                // if there will be a hsb, but it hasn't been rendered yet,
                // still account for the extra added height
                if (this._shouldHaveHSB() === true && this._hasHSB() === false) {
                    iAvailableSpace -= TableUtils._determineHorizontalScrollBarHeight();
                }

                return iAvailableSpace;
            }
        }
        return 0;
    };

    /**
     * Determine the width of a column in px during collectTableSizes. In particular,
     * will try to make sure the returned value is the actual rendered width of the
     * column. Browsers may "stretch" column elements to fill otherwise unused space.
     *
     * @protected
     * @param {Element} oHeaderElement The Column's header element
     * @param {sas.hc.ui.table.Column} oColumn The Column object.
     * @param {number} iIndex The index of the Column.
     * @return {number} The width of the column in px.
     */
    Table.prototype._calculateHeaderWidth = function(oHeaderElement, oColumn, iIndex) {
        var oHeaderElementClientBoundingRect = oHeaderElement.getBoundingClientRect();
        var iHeaderWidth = oHeaderElementClientBoundingRect.right - oHeaderElementClientBoundingRect.left;

        return iHeaderWidth;
    };

    /**
     * Determines all needed table size at one dedicated point,
     * for avoiding layout thrashing through read/write UI operations.
     * @private
     */
    Table.prototype._collectTableSizes = function(aTableRowHeights) {
        var oSizes = {
            tableCtrlScrollWidth: 0,
            tableRowHdrScrWidth: 0,
            tableCtrlRowScrollTop: 0,
            tableCtrlRowScrollHeight: 0,
            tableCtrlScrWidth: 0,
            tableHSbScrollLeft: 0,
            tableCtrlFixedWidth: 0,
            tableCntHeight: 0,
            tableCntWidth: 0,
            columnRowHeight: 0,
            columnRowOuterHeight: 0,
            invisibleColWidth: 0
        };

        var oDomRef = this.getDomRef();
        if (!oDomRef) {
            return oSizes;
        }

        var oSapUiTableCnt = oDomRef.querySelector(".sapUiTableCnt");
        if (oSapUiTableCnt) {
            oSizes.tableCntHeight = oSapUiTableCnt.clientHeight;
            oSizes.tableCntWidth = oSapUiTableCnt.clientWidth;
        }


        var oSapUiTableCtrlScroll;
        if (this.getColumnHeaderVisible()) {
            oSapUiTableCtrlScroll = oDomRef.querySelector(".sapUiTableCtrlScroll");
        } else {
            // if the headers table is hidden, get the width from the contents table
            var aScrollTables = $(oDomRef).find(".sapUiTableCtrlScroll");
            if (aScrollTables.length > 1) {
                oSapUiTableCtrlScroll = aScrollTables[1];
            }
        }
        if (oSapUiTableCtrlScroll) {
            oSizes.tableCtrlScrollWidth = oSapUiTableCtrlScroll.clientWidth;
        }

        var oSapUiTableRowHdrScr = oDomRef.querySelector(".sapUiTableRowHdrScr");
        if (oSapUiTableRowHdrScr) {
            oSizes.tableRowHdrScrWidth = oSapUiTableRowHdrScr.clientWidth;
        }

        var oSapUiTableCtrlRowScroll = oDomRef.querySelector(".sapUiTableCtrl.sapUiTableCtrlRowScroll.sapUiTableCtrlScroll");
        if (oSapUiTableCtrlRowScroll) {
            oSizes.tableCtrlRowScrollTop = oSapUiTableCtrlRowScroll.offsetTop;
            oSizes.tableCtrlRowScrollHeight = oSapUiTableCtrlRowScroll.offsetHeight;
        }

        var oCtrlScrDomRef = oDomRef.querySelector(".sapUiTableCtrlScr");
        if (oCtrlScrDomRef) {
            oSizes.tableCtrlScrWidth = oCtrlScrDomRef.clientWidth;
        }

        var oHsb = this.getDomRef(SharedDomRef.HorizontalScrollBar);
        if (oHsb) {
            oSizes.tableHSbScrollLeft = oHsb.scrollLeft;
        }

        var oCtrlFixed = oDomRef.querySelector(".sapUiTableCCnt .sapUiTableCtrlFixed");
        if (oCtrlFixed) {
            oSizes.tableCtrlFixedWidth = oCtrlFixed.clientWidth;
        }

        var iFixedColumnCount = this._getActualFrozenColumnCount();
        var iOriginalFixedCount = iFixedColumnCount;
        var aHeaderWidths = [];
        var iFixedHeaderWidthSum = 0;
        var aHeaderElements = oDomRef.querySelectorAll(".sapUiTableCtrlFirstCol > th:not(.sapUiTableColSel)");

        // if (aHeaderElements) {
        if (aHeaderElements && this.retrieveLeafColumns().length > 0) {

            var aColumns = this.retrieveLeafColumns();

            for (var i = 0; i < aHeaderElements.length; i++) {
                var iHeaderWidth = this._calculateHeaderWidth(aHeaderElements[i], aColumns[i], i);
                aHeaderWidths.push(iHeaderWidth);

                if (i < iOriginalFixedCount) {
                    if (aColumns[i] && !aColumns[i].getVisible()) {
                    // if (!this.getColumns()[i].getVisible()) {
                        // the fixedColumnCount does not consider the visibility of the column, whereas the DOM only represents
                        // the visible columns. In order to match both, the fixedColumnCount (aggregation) and fixedColumnCount
                        // of the DOM, for each invisible column, 1 must be deducted from the fixedColumnCount (aggregation).
                        iFixedColumnCount--;
                    } else {
                        iFixedHeaderWidthSum += iHeaderWidth;
                    }
                }
            }
        }

        if (iFixedColumnCount > 0) {
            var iUsedHorizontalTableSpace = 0;
            var oRowHdrScr = this.getDomRef("sapUiTableRowHdrScr");
            if (oRowHdrScr) {
                iUsedHorizontalTableSpace += oRowHdrScr.clientWidth;
            }

            var oVsb = this.getDomRef("vsb");
            if (oVsb) {
                iUsedHorizontalTableSpace += oVsb.offsetWidth;
            }

            // There must be enough space for at least one scrollable column to have frozen columns.
            var iMinColumnWidth = this.getMinColumnWidth();
            var bIgnoreFixedColumnCountCandidate = (oDomRef.clientWidth - iUsedHorizontalTableSpace - iFixedHeaderWidthSum) < iMinColumnWidth;
            if (this._bIgnoreFixedColumnCount !== bIgnoreFixedColumnCountCandidate) {
                this._bIgnoreFixedColumnCount = bIgnoreFixedColumnCountCandidate;
                // S1417085 - if we got here while keyboard resizing, no need to invalidate
                // if you do, you cannot resize an unfrozen column smaller than the
                // freeze/unfreeze threshold using the keyboard
                if (this._bDuringResizeWithKeyboard !== true) {
                    this.invalidate();
                }
            }
        }

        oSizes.headerWidths = aHeaderWidths;

        if (TableUtils.hasRowHeader(this)) {
            var oFirstInvisibleColumn = oDomRef.querySelector(".sapUiTableCtrlFirstCol > th:first-child");
            if (oFirstInvisibleColumn) {
                oSizes.invisibleColWidth = oFirstInvisibleColumn.clientWidth;
            }
        }

        function getColumnSize(oColumn) {
            oSizes.columnRowHeight = Math.max(oColumn.clientHeight || 0, oSizes.columnRowHeight);
            oSizes.columnRowOuterHeight = Math.max(oColumn.offsetHeight || 0, oSizes.columnRowOuterHeight);
        }
        Array.prototype.forEach.call(oDomRef.querySelectorAll(".sapUiTableCol"), getColumnSize);

        if (!aTableRowHeights) {
            oSizes.tableRowHeights = this._collectRowHeights();
        } else {
            oSizes.tableRowHeights = aTableRowHeights;
        }


        return oSizes;
    };

    /**
     * Synchronizes the row heights with the row header heights.
     * @private
     */
    Table.prototype._updateRowHeader = function(aRowItemHeights) {
        var oDomRef = this.getDomRef();
        if (!oDomRef) {
            return;
        }
        var aRowHeaderItems = oDomRef.querySelectorAll(".sapUiTableRowHdr");

        var aFixedRowItems = oDomRef.querySelectorAll(".sapUiTableCCnt .sapUiTableCtrlFixed > tbody > tr");
        var aScrollRowItems = oDomRef.querySelectorAll(".sapUiTableCCnt .sapUiTableCtrlScroll > tbody > tr");

        var iLength = Math.max(aRowHeaderItems.length, aScrollRowItems.length, 0);
        for (var i = 0; i < iLength; i++) {
            var iRowItemHeight = aRowItemHeights[i];
            if (iRowItemHeight) {
                if (aRowHeaderItems[i]) {
                    aRowHeaderItems[i].style.height = iRowItemHeight + "px";
                }

                if (aFixedRowItems[i]) {
                    aFixedRowItems[i].style.height = iRowItemHeight + "px";
                }

                if (aScrollRowItems[i]) {
                    aScrollRowItems[i].style.height = iRowItemHeight + "px";
                }
            }
        }
    };

    /**
     * Synchronizes the heights of all elements in each visible row of the table.  These elements include the row
     * selection header (if present), any frozen columns, and the scrollable/main columns.
     */
    Table.prototype.syncRowHeights = function() {
        var oDomRef = this.getDomRef();

        if (oDomRef) {
            //reset row heights
            this._resetRowHeights();

            //collect the table's row heights and sync them
            this._updateRowHeader(this._collectRowHeights());
        }
    };

    /**
     * Rerendering handling
     * @private
     */
    Table.prototype.onBeforeRendering = function(oEvent) {

        logger.debug("onBeforeRendering");

        if (oEvent && (oEvent.isMarked("insertTableRows") || oEvent.isMarked("refreshScrollableContent"))) {
            return;
        }

        if (this._mTimeouts.bindingTimer) {
            this._updateBindingContexts();
        }

        this._cleanUpTimers();
        this._detachEvents();

        // if there are column groups, populate the array of headers for rendering
        if (this.hasColumnGroups()) {
            if (!this._aRenderArray) {
                this._aRenderArray = this._populateRenderArray();
            }
        } else {
            this._aRenderArray = new Array(1);
            this._aRenderArray[0] = this.getColumns();
        }

        var sVisibleRowCountMode = this.getVisibleRowCountMode();

        if (TableUtils.isVariableRowHeightEnabled(this)) {
            var oVSb = this.getDomRef(SharedDomRef.VerticalScrollBar);
            if (oVSb) {
                this._iOldScrollTop = oVSb.scrollTop;
            }
        }

        var aRows = this.getRows();
        if (sVisibleRowCountMode === VisibleRowCountMode.Fixed ||
            (sVisibleRowCountMode === VisibleRowCountMode.Auto && this._iTableRowContentHeight && aRows.length === 0)) {
            if (this.getBinding("rows")) {
                this._adjustRows(this._calculateRowsToDisplay());
            } else {
                var that = this;
                this._mTimeouts.onBeforeRenderingAdjustRows = this._mTimeouts.onBeforeRenderingAdjustRows || window.setTimeout(function() {
                    that._adjustRows(that._calculateRowsToDisplay());
                    that._mTimeouts.onBeforeRenderingAdjustRows = undefined;
                }, 0);
            }
        } else if (!this._oRowTemplate && aRows.length > 0) {
            // Rows got invalidated, recreate rows with new template
            this._adjustRows(aRows.length);
        }

        this._cleanupColumnDragBar(); // this also removes highlighting...
        if (this._isAutoResizeEnabled())
        {
            this.addResizeToFit();
        }
    };

    /**
     * Rerendering handling
     * @private
     */
    Table.prototype.onAfterRendering = function(oEvent) {
        var $this = this.$(),
            i,
            self = this,
            aColumns,
            oColumn;

        logger.debug("onAfterRendering");

        // manage selection controls for selection (checkboxes)
        if (this._useSelectionControls() === true) {
            this._refreshSelectionControls();
        }

        // highlight columns if needed
        aColumns = this.retrieveLeafColumns();
        for (i = 0; i < aColumns.length; i++) {
            oColumn = aColumns[i];
            if (!!oColumn && jQuery.isFunction(oColumn.getHighlighted) === true && oColumn.getHighlighted() === true) {
                self.highlightColumn(i);
            }
        }

        if (oEvent && (oEvent.isMarked("insertTableRows") || oEvent.isMarked("refreshScrollableContent"))) {
            return;
        }

        this._iDefaultRowHeight = undefined;
        this._bInvalid = false;
        this._bOnAfterRendering = true;

        this._attachEvents();

        // since the row is an element it has no own renderer. Anyway, logically it has a domref. Let the rows
        // update their domrefs after the rendering is done. This is required to allow performant access to row domrefs
        this._initRowDomRefs();

        // restore the column icons
        var aCols = this.retrieveLeafColumns();
        for (var i = 0, l = aCols.length; i < l; i++) {
            if (aCols[i].getVisible() && aCols[i]._restoreIcons) {
                aCols[i]._restoreIcons();
            }
        }

        // enable/disable text selection for column headers
        if (!this._bAllowColumnHeaderTextSelection) {
            this._disableTextSelection($this.find(".sapUiTableColHdrCnt"));
        }

        this._bOnAfterRendering = false;

        // invalidate item navigation
        this._getKeyboardExtension().invalidateItemNavigation();

        if (this._bFirstRendering && this.getVisibleRowCountMode() === VisibleRowCountMode.Auto) {
            this._bFirstRendering = false;
            // Wait until everything is rendered (parent height!) before reading/updating sizes. Use a promise to make sure
            // to be executed before timeouts may be executed.
            Promise.resolve().then(this._updateTableSizes.bind(this, true));
        } else if (!this._mTimeouts.onAfterRenderingUpdateTableSizes) {
            this._updateTableSizes();
        }

        if (TableUtils.isVariableRowHeightEnabled(this)) {
            var oVSb = this.getDomRef(SharedDomRef.VerticalScrollBar);
            if (oVSb) {
                oVSb.scrollTop = Math.max(oVSb.scrollTop, this._iOldScrollTop);
            }
        }

        this._updateTableContent();

        if (this.getBinding("rows")) {
            this.fireEvent("_rowsUpdated");
        }

        this.getSelectAllControl().updateRelativeToTable();

        this._normalizeColumnWidths();

        this._stretchColumns();

        this._restoreHScroll();
    };

    /**
     * Scroll the vertical scrollbar all the way to the bottom of the Table
     * @returns {object} a promise that is resolved when the operation is complete
     * @public
     */
    Table.prototype.scrollToBottom = function() {
        var self = this,
            oVsb = this.getDomRef("vsb");
        return new Promise(function(resolve) {
            self._showLastPage(function() {
                oVsb.scrollTop = oVsb.scrollHeight;
                resolve();
            });
        });
    };

    /**
     * Prevent the browser from stretching absolutely sized columns by configuring
     * them to match their rendered width.
     *
     * @private
     */
    Table.prototype._stretchColumns = function() {
        var self = this;

        // S1415714 - We need to prevent a recursive loop where the column
        // keeps growing and growing when some columns have percent based width.
        // If there is a percent based column, the px based columns will not
        // be the ones that need to stretch anyway.
        if (this._hasSomePercentageWidthColumn()) {
            return;
        }
        var oContentsConainer;
        if (!this.getColumnHeaderVisible()) {
            oContentsConainer = self.$().find(".sapUiTableCCnt");
        }
        this.retrieveLeafColumns().forEach(function(oCol) {
            if (oCol.hasAbsoluteWidth) {
                var iConfiguredWidth = self._CSSSizeToPixel(oCol.getWidth());
                var iRenderedWidth;
                if (self.getColumnHeaderVisible()) {
                    iRenderedWidth = oCol.$().width();
                } else if (oContentsConainer) {
                    // if the column headers are hidden, we'll need to get
                    // an accurate width from the contents table cells
                    var sSelector = "[data-sap-ui-colid=" + oCol.getId() + "]";
                    iRenderedWidth = oContentsConainer.find(sSelector).width();
                }
                //allow 1 pixel wiggle room for rounding errors
                if (oCol.hasAbsoluteWidth() && ((iRenderedWidth-iConfiguredWidth) > 1)) {
                    oCol.setWidth(iRenderedWidth + "px");
                }
            }
        });
    };

    /**
     * Adjust columns with percent based widths to ensure 100% of available
     * width is covered.
     *
     * @private
     */
    Table.prototype._normalizeColumnWidths = function() {
        var self = this,
            iTotalPixelWidth = 0,
            iTotalPercentWidth = 0,
            iAvailablePercent,
            iPixelWidth,
            iPixelPercent,
            iPercentWidth,
            iNewPercentWidth,
            sColumnWidth,
            iScrollableSectionWidth,
            aColumnsToAdjust = [],
            aColumnsWithNoWidth = [],
            iMinPercentWidth = this._iMinPercentWidth;

        // if there are only columns with pixel based width, do nothing
        if (!this._hasSomePercentageWidthColumn() || this._bNormalizingColumnWidths) {
            return;
        }
        // S1423815 - When ignoring fixed column count, there will be only one section;
        // use the width of the entire table in case we haven't re-rendered yet
        if (this._bIgnoreFixedColumnCount) {
            iScrollableSectionWidth = this.$().find(".sapUiTableCtrl").width();
        } else {
            iScrollableSectionWidth = this.$().find(".sapUiTableCtrlScr").width();
        }
        // if the table does not have a width specified, do nothing
        if (!iScrollableSectionWidth) {
            return;
        }

        this.retrieveLeafColumns().forEach(function(oCol) {
            if (!self.isFrozenColumn(oCol) && oCol.getWidth) {
                sColumnWidth = oCol.getWidth();
                if (!sColumnWidth ) {
                    // add this column to an array of columns with no width
                    aColumnsWithNoWidth[aColumnsWithNoWidth.length] = oCol;
                } else if (sColumnWidth.substr(-1) !== "%") {
                    iPixelWidth = self._CSSSizeToPixel(sColumnWidth);
                    if (!isNaN(iPixelWidth)) {
                        // we have a pixel width; add it to the pixel total
                        iTotalPixelWidth += iPixelWidth;
                    }
                } else {
                    iPercentWidth = Number(sColumnWidth.substr(0, sColumnWidth.length-1));
                    if (!isNaN(iPercentWidth)) {
                        // we have a percent width; add it to the percent total
                        iTotalPercentWidth += iPercentWidth;
                        // add this column to an array of columns that need to be adjusted
                        aColumnsToAdjust[aColumnsToAdjust.length] = oCol;
                    }
                }
            }
        });

        iPixelPercent = iTotalPixelWidth / iScrollableSectionWidth * 100;
        iAvailablePercent = 100 - iPixelPercent;
        if (iMinPercentWidth === undefined || iMinPercentWidth === null) {
            iMinPercentWidth = 2;
        }

        // if there are columns with no width, handle those first
        if (aColumnsWithNoWidth.length > 0) {
            var iPercentDiff = iAvailablePercent - iTotalPercentWidth;
            // make sure there is enough room for every column with no width
            // to have the minimum percent width
            if (Math.round(iPercentDiff / aColumnsWithNoWidth.length) >= iMinPercentWidth) {
                // there is enough room...
                // we're done because the columns with no width will stretch
                // to fill the area not covered by the % based columns, therefore
                // the remaining columns don't need to be adjusted
                return;
            } else {
                // if there is not enough room, we will set the width of
                // each of these columns to the minimum, so add it to the
                // total percent width; then we'll adjust all columns accordingly
                aColumnsWithNoWidth.forEach(function(oCol) {
                    iTotalPercentWidth += iMinPercentWidth;
                });
            }
        }

        // go through each column that needs to be adjusted and set the new width
        this._bNormalizingColumnWidths = true;
        aColumnsToAdjust.forEach(function(oCol) {
            if (oCol.getWidth) {
                sColumnWidth = oCol.getWidth();
                if (sColumnWidth) {
                    iPercentWidth = Number(sColumnWidth.substr(0, sColumnWidth.length-1));
                    // we have a percent width; adjust it to add up to 100
                    if (!isNaN(iPercentWidth)) {
                        iNewPercentWidth = Math.min((iPercentWidth * iAvailablePercent / iTotalPercentWidth),100);
                        iNewPercentWidth = Math.round(Math.max(iNewPercentWidth, iMinPercentWidth));
                        if (!isNaN(iNewPercentWidth)) {
                            self._updateColumnWidth(oCol, iNewPercentWidth+"%");
                        }
                    }
                }
            }
        });
        this._bNormalizingColumnWidths = false;
    };

    /**
     * Triggers rerendering of this element and its children, however invalidation from the 'Select All' control will be ignored.
     *
     * As <code>sap.ui.core.Element</code> "bubbles up" the invalidate, changes to children
     * potentially result in rerendering of the whole sub tree.
     *
     * @param {object} oOrigin
     * @protected
     */
    Table.prototype.invalidate = function(oOrigin) {
        if (oOrigin !== this.getSelectAllControl()) {   //S1241581: the select-all control can invalidate all it wants, but it won't bother me
            if (!this._ignoreInvalidateOfChildControls) {
                this._bInvalid = true;
                var vReturn = Control.prototype.invalidate.call(this, oOrigin);
            }

            return vReturn;
        }
    };

    /**
     * @private
     */
    Table.prototype._initRowDomRefs = function() {
        var aRows = this.getRows();
        for (var i = 0; i < aRows.length; i++) {
            aRows[i].initDomRefs();
        }
    };

    /**
     * First collects all table sizes, then synchronizes row/column heights, updates scrollbars and selection.
     * @private
     */
    Table.prototype._updateTableSizes = function(bForceUpdateTableSizes, bSkipHandleRowCountMode) {
        this._mTimeouts.onAfterRenderingUpdateTableSizes = undefined;
        var oDomRef = this.getDomRef();

        if (this._bInvalid || !oDomRef) {
            return;
        }

        this._resetRowHeights();
        var aRowHeights = this._collectRowHeights();
        this._getDefaultRowHeight(aRowHeights);

        var iRowContentSpace = 0;
        if (!bSkipHandleRowCountMode && this.getVisibleRowCountMode() === VisibleRowCountMode.Auto) {
            iRowContentSpace = this._determineAvailableSpace();
            // if no height is granted we do not need to do any further row adjustment or layout sync.
            // Saves time on initial start up and reduces flickering on rendering.
            if (this._handleRowCountModeAuto(iRowContentSpace) && !bForceUpdateTableSizes) {
                // updateTableSizes was already called by insertTableRows, therefore skip the rest of this function execution
                return;
            }
        }

        TableUtils.deregisterResizeHandler(this, "");

        // update Vertical Scrollbar before collection because it changes sizes
        this._toggleVSb();

        var oTableSizes = this._collectTableSizes(aRowHeights);

        if (this._mTimeouts.afterUpdateTableSizes) {
            window.clearTimeout(this._mTimeouts.afterUpdateTableSizes);
        }

        if (oTableSizes.tableCntHeight === 0 && oTableSizes.tableCntWidth === 0) {
            // the table has no size at all. This may be due to one of the parents has display:none. In order to
            // recognize when the parent size changes, the resize handler must be registered synchronously, otherwise
            // the browser may finish painting before the resize handler is registered
            TableUtils.registerResizeHandler(this, "", this._onTableResize.bind(this), true);

            return;
        }

        // Manipulation of UI Sizes
        this._updateRowHeader(oTableSizes.tableRowHeights);
        this._syncColumnHeaders(oTableSizes);
        this._determineVisibleCols(oTableSizes);
        if (!bSkipHandleRowCountMode) {
            this._setRowContentHeight(iRowContentSpace);
        }
        this._updateHSb(oTableSizes);
        this._updateVSb(oTableSizes);

        this.$().find(".sapUiTableNoOpacity").addBack().removeClass("sapUiTableNoOpacity");

        if (this._mTimeouts.afterUpdateTableSizes) {
            window.clearTimeout(this._mTimeouts.afterUpdateTableSizes);
        }

        var that = this;

        // S1220894 - Remove the delay to handle scenarios when layouts
        // decide to change width in between Table's onAfterRendering
        // and when this timeout actually runs (mostly Form Layouts).
        // Running updateTableSizes twice isn't ideal but I don't see
        // any reason that it's such a bad thing. - niroth
        // this._mTimeouts.afterUpdateTableSizes = window.setTimeout(function () {

        // size changes of the parent happen due to adaptations of the table sizes. In order to first let the
        // browser finish painting, the resize handler is registered in a timeout. If this would be done synchronously,
        // updateTableSizes would always run twice.
        TableUtils.registerResizeHandler(that, "", that._onTableResize.bind(that), true);
    };

    var _originalAttachDoubleClick = Table.prototype.attachDoubleClick;

    /**
     * @override
     */
    Table.prototype.attachDoubleClick = function(oData, fnFunction, oListener) {
        // As a convenience for callers, if you attach a listener we go ahead and enable
        // the double-click behavior. Callers can always set the doubleClickEnabled property
        // on the table to FALSE if they don't want it for some reason
        this.setEnableDoubleClick(true);

        // Call the real method to attach the handler now that we've turned on the flag in the delegate
        _originalAttachDoubleClick.call(this, oData, fnFunction, oListener);
    };

    /**
     * @override
     */
    Table.prototype.setEnableDoubleClick = function(bEnabled) {
        this.setProperty("enableDoubleClick", bEnabled);

        // Inform the delegate of this setting so that it can behave appropriately
        if (this._oClickHandlerDelegate) {
            this._oClickHandlerDelegate.setEnableDoubleClick(bEnabled);
        }
    };

    /**
     * @override
     */
    Table.prototype.setShowOverlay = function(bShow) {
        bShow = !!bShow;
        this.setProperty("showOverlay", bShow, true);

        if (this.getDomRef()) {
            var oFocusRef = document.activeElement;
            this.$().toggleClass("sapUiTableOverlay", bShow);
            this._getAccExtension().updateAriaStateForOverlayAndNoData();
            this._getKeyboardExtension().updateNoDataAndOverlayFocus(oFocusRef);
        }

        return this;
    };

    /**
     * @private
     */
    Table.prototype._updateFixedBottomRows = function() {
        var iFixedBottomRows = this.getFixedBottomRowCount();

        var oDomRef = this.getDomRef();
        if (oDomRef && iFixedBottomRows > 0) {
            var $sapUiTableFixedPreBottomRow = jQuery(oDomRef).find(".sapUiTableFixedPreBottomRow");
            $sapUiTableFixedPreBottomRow.removeClass("sapUiTableFixedPreBottomRow");

            var oBinding = this.getBinding("rows");

            if (oBinding) {
                var iVisibleRowCount = this.getVisibleRowCount();
                var bIsPreBottomRow = false;
                var aRows = this.getRows();
                var iFirstVisibleRow = this._getSanitizedFirstVisibleRow();
                for (var i = 0; i < aRows.length; i++) {
                    var $rowDomRefs = aRows[i].getDomRefs(true);
                    if (this._iBindingLength >= iVisibleRowCount) {
                        bIsPreBottomRow = (i === iVisibleRowCount - iFixedBottomRows - 1);
                    } else {
                        bIsPreBottomRow = (iFirstVisibleRow + i) === (this._iBindingLength - iFixedBottomRows - 1) && (iFirstVisibleRow + i) < this._iBindingLength;
                    }

                    $rowDomRefs.row.toggleClass("sapUiTableFixedPreBottomRow", bIsPreBottomRow);
                }
            }
        }
    };


    // =============================================================================
    // FOCUS
    // =============================================================================

    /*
     * @see JSDoc generated by SAPUI5 control
     */
    Table.prototype.getFocusInfo = function() {
        var sId = this.$().find(":focus").attr("id");
        if (sId) {
            return {customId: sId};
        } else {
            return Element.prototype.getFocusInfo.apply(this, arguments);
        }
    };

    /*
     * @see JSDoc generated by SAPUI5 control
     */
    Table.prototype.applyFocusInfo = function(mFocusInfo) {
        if (mFocusInfo && mFocusInfo.customId) {
            this.$().find("#" + mFocusInfo.customId).focus();
        } else {
            Element.prototype.applyFocusInfo.apply(this, arguments);
        }
        return this;
    };


    // =============================================================================
    // PUBLIC TABLE API
    // =============================================================================


    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setTitle = function(vTitle) {
        var oTitle = vTitle;
        if (typeof (vTitle) === "string" || vTitle instanceof String) {
            oTitle = library.TableHelper.createTextView({
                text: vTitle,
                width: "100%"
            });
            oTitle.addStyleClass("sapUiTableHdrTitle");
        }
        this.setAggregation("title", oTitle);
        return this;
    };


    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setFooter = function(vFooter) {
        var oFooter = vFooter;
        if (typeof (vFooter) === "string" || vFooter instanceof String) {
            oFooter = library.TableHelper.createTextView({
                text: vFooter,
                width: "100%"
            });
        }
        this.setAggregation("footer", oFooter);
        return this;
    };


    /**
     * Sets the selection mode. The current selection is lost.
     * @param {string} sSelectionMode the selection mode, see sas.hc.ui.table.SelectionMode
     * @public
     * @return a reference on the table for chaining
     */
    Table.prototype.setSelectionMode = function(sSelectionMode) {
        this.clearSelection();

        if (sSelectionMode === SelectionMode.ParentChild) {
            logger.warning("Cannot set ParentChild SelectionMode on a Table. Defaulting to MultiToggle.");
            sSelectionMode = SelectionMode.MultiToggle;
        }

        if (sSelectionMode === SelectionMode.Single) {
            this._oSelection.setSelectionMode(SelectionModel.SINGLE_SELECTION);
        } else {
            this._oSelection.setSelectionMode(SelectionModel.MULTI_SELECTION);
        }

        // UX recommends using RowOnly selectionBehavior with Single selection
        if (sSelectionMode === SelectionMode.Single) {
            this.setSelectionBehavior(SelectionBehavior.RowOnly);
        }

        this.setProperty("selectionMode", sSelectionMode);
        return this;
    };

    /**
     * @override
     */
    Table.prototype.setSelectionBehavior = function(sSelectionBehavior) {
        if (this.getSelectionMode() === SelectionMode.Single && sSelectionBehavior !== SelectionBehavior.RowOnly) {
            jQuery.sap.log.warning("When using Single selection mode, it is recommended to use RowOnly for the selection behavior");
        }
        this.setProperty("selectionBehavior", sSelectionBehavior);
        return this;
    };

    /**
     * Shifts the vertical table position according to the delta of the estimated row heights to actual row heights.
     * The table simulates the pixel-based scrolling by adjusting the vertical position of the scrolling areas.
     * Additionally when there are rows inside which have a larger height than estimated, this will also be corrected
     * and leads to a bigger vertical shift.
     * @private
     */
    Table.prototype._adjustTablePosition = function() {

        if (!this.getDomRef(SharedDomRef.VerticalScrollBar)) {
            var iScrollTop = 0;
        } else {
            var iScrollTop = this.getDomRef(SharedDomRef.VerticalScrollBar).scrollTop;
        }

        var iDefaultRowHeight = this._getDefaultRowHeight();
        var iRowHeightOffset = iScrollTop % iDefaultRowHeight;

        var oInnerScrollContainer = this.$().find(".sapUiTableCtrlRowScroll, .sapUiTableRowHdrScr");
        if (this._iBindingLength <= this.getVisibleRowCount()) {
            oInnerScrollContainer.css({"transform": "translate3d(0px, " + (-iScrollTop) + "px, 0px)"});
        } else {
            var iRowCorrection = this._calculateRowCorrection(iDefaultRowHeight, iRowHeightOffset, iScrollTop);
            oInnerScrollContainer.css({"transform": "translate3d(0px, " + (-iRowCorrection ) + "px, 0px)"});
        }
    };

    /**
     * Calculates the amount of pixels the scrolling areas have to be shifted vertically, when the actual row heights are
     * bigger than the estimated row heights.
     * @param {int} iDefaultRowHeight the estimated row height
     * @param {int} iRowHeightOffset the current offset for simulated pixel-based scrolling
     * @returns {number} the amount of pixels the scrolling areas have to be shifted vertically
     * @private
     */
    Table.prototype._calculateRowCorrection = function(iDefaultRowHeight, iRowHeightOffset, iScrollTop) {
        var iRowIndex = this.getFirstVisibleRow();
        var iMaxRowIndex = this._iBindingLength - this.getVisibleRowCount();

        if (iRowIndex < iMaxRowIndex) {
            var iFirstRowHeightDelta = TableUtils.getRowHeightByIndex(this, 0) - iDefaultRowHeight;
            return Math.floor(iRowHeightOffset * (iFirstRowHeightDelta / iDefaultRowHeight)) + iRowHeightOffset;
        } else if (iRowIndex === iMaxRowIndex) {
            return iScrollTop - (this._iBindingLength * iDefaultRowHeight) + (this.getDomRef("tableCCnt") ? this.getDomRef("tableCCnt").clientHeight : 0);
        }
    };

    /**
     * Calculate the scrollTop value for the vertical scroll bar
     *
     * Usually, an estimation for scrollTop can be used where the
     * number of rows is simply multiplied by the default row height.
     *
     * But, at the very end of the Table this estimation breaks down when
     * row heights don't match the default, resulting in cut-off rows
     * or even not seeing a final row.
     *
     * So, when the last page of rows is loaded, the amount of scrolling
     * required when using "setFirstVisibleRow" must be calculated exactly.
     *
     * iRowIndex is the _desired_ first visible row, and won't always be the
     * actual firstVisibleRow property value.
     * This is because at the end of the Table if the desired row is past the
     * first page of rows, the firstVisibleRow value will be set to the start of
     * the page, with additional scrolling to the desired row happening just by
     * increasing the vertical scroll bar's scrollTop value.
     *
     * @param {int} iRowIndex the desired first visible row
     * @returns {int} the number of pixels to scroll (scrollTop)
     * @private
     */
    Table.prototype._determineScrollTop = function(iRowIndex) {
        var i = 0,
            //get the array of visible Row elements
            aRows = this.getRows(),

            // used to accumulate the total scrollTop desired
            // start out with the estimated height of the rows prior to the current page
            iScrollTop = aRows[0].getIndex() * this._getDefaultRowHeight(),

            // this is the index of where the last page of rows start
            // i.e., if a table binding had a length of 200,
            // and a page-size of 10, then this value would be 189
            // (subtract 1 since the index is 0-based)
            iLastPageStartingIndex = Math.max(0, this._getRowCount() - this.getVisibleRowCount() - 1),

            // number of rows requested past the start of the final page of rows
            iExtraRows = Math.min(aRows.length, iRowIndex - iLastPageStartingIndex);

        // add up any additional row heights
        for (; i < iExtraRows; i++) {
            iScrollTop += aRows[i].$().outerHeight();
        }

        return iScrollTop;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setFirstVisibleRow = function(iRowIndex, bOnScroll, bSupressEvent) {
        //S1326003: Negative row numbers should translate to row 0
        iRowIndex = Math.max(iRowIndex, 0);
        var bFirstVisibleRowChanged = this.getFirstVisibleRow() !== iRowIndex;

        this.setProperty("firstVisibleRow", iRowIndex, true);

        if (TableUtils.isVariableRowHeightEnabled(this)
            && this.getBinding("rows") && !this._bRefreshing && !bFirstVisibleRowChanged) {
            this._adjustTablePosition();
        }

        // update the bindings:
        //  - prevent the rerendering
        //  - use the databinding fwk to update the content of the rows
        if (bFirstVisibleRowChanged && this.getBinding("rows") && !this._bRefreshing) {
            this.updateRows();
            if (!bOnScroll) {
                var oVSb = this.getDomRef(SharedDomRef.VerticalScrollBar);
                if (oVSb) {
                    oVSb.scrollTop = this._determineScrollTop(iRowIndex);
                }
            }
        }

        if (bFirstVisibleRowChanged && !bSupressEvent) {
            this.fireFirstVisibleRowChanged({firstVisibleRow: iRowIndex});
        }
        return this;
    };


    // enable calling 'bindAggregation("rows")' without a factory
    Table.getMetadata().getAggregation("rows")._doesNotRequireFactory = true;

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.bindRows = function(oBindingInfo, vTemplate, oSorter, aFilters) {
        // ensure old Table API compatibility (sPath, [oSorter], [aFilters])
        if (typeof oBindingInfo === "string" &&
              (vTemplate instanceof Sorter || jQuery.isArray(oSorter) && oSorter[0] instanceof Filter) ) {
            aFilters = oSorter;
            oSorter = vTemplate;
            vTemplate = undefined;
        }

        return this.bindAggregation("rows", oBindingInfo, vTemplate, oSorter, aFilters);
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype._bindAggregation = function(sName, sPath, oTemplate, oSorter, aFilters) {
        Element.prototype._bindAggregation.apply(this, arguments);
        var oBinding = this.getBinding("rows");
        if (sName === "rows" && oBinding) {
            oBinding.attachChange(this._onBindingChange, this);
            this._updateBindingLength();    //S1330453: need to update binding length if we can
        }

        // re-initialize the selection model, might be necessary in case the table gets "rebound"
        this._initSelectionModel(SelectionModel.MULTI_SELECTION);

        // currently only required for TreeBindings, will be relevant for ListBindings later
        if (oBinding && this.isTreeBinding("rows") && !oBinding.hasListeners("selectionChanged")) {
            oBinding.attachSelectionChanged(this._onSelectionChanged, this);
        }

        if (sName === "rows" && oBinding) {
            this.refreshSort(); //look at any existing sorters on the binding
        }

        return this;
    };

    /**
     * Initialises a new selection model for the Table instance.
     * @param {sap.ui.model.SelectionModel.MULTI_SELECTION|sap.ui.model.SelectionModel.SINGLE_SELECTION} sSelectionMode the selection mode of the selection model
     * @return {sas.hc.ui.table.Table} the table instance for chaining
     * @private
     */
    Table.prototype._initSelectionModel = function (sSelectionMode) {
        // detach old selection model event handler
        if (this._oSelection) {
            this._oSelection.detachSelectionChanged(this._onSelectionChanged, this);
        }
        //new selection model with the currently set selection mode
        this._oSelection = new SelectionModel(sSelectionMode);
        this._oSelection.attachSelectionChanged(this._onSelectionChanged, this);

        return this;
    };

    /**
     * handler for change events of the binding
     * @param {sap.ui.base.Event} oEvent change event
     * @private
     */
    Table.prototype._onBindingChange = function(oEvent) {
        var sReason = typeof (oEvent) === "object" ? oEvent.getParameter("reason") : oEvent;
        if (sReason === ChangeReason.Sort || sReason === ChangeReason.Filter) {
            this.clearSelection(sReason);
            this.setFirstVisibleRow(0);
            this.refreshSort(); //look at any existing sorters on the binding
            this._updateA11yVisibleRowsLabel();

            if (sReason === ChangeReason.Filter && this.getRememberSelections()
                && this.getFilteredSelectionBehavior() === FilteredSelectionBehavior.RetainMatching) {
                //if remembering and filteredSelectionBehavior is RetainMatching and reason is filter, then prune selected keys
                //PERF - looping through all contexts in binding to figure out which items match will be non-performant
                this._pruneSelectedKeysAsync();
            }
        } else if (sReason === ChangeReason.Change && this.getRememberSelections() && !!this.getSelectionIdProperty()) {
            // if we get a change event on the binding and we're using primary keys, let's go through and re-assess selection
            //  1) clear the index-based selection directly (passing false because we are not using 'select all' to do it)
            this._oSelection.clearSelection(false);
            //  2) reapply remembered selections
            this.applyRememberedSelections(sReason);
        }
    };

    Table.prototype._pruneSelectedKeysAsync = function() {
        var oBinding = this.getBinding("rows"),
            oTable = this,
            iIndexFrom = 0,
            iLength,
            aSelectedKeys;

        if (!oBinding) {
            return Promise.reject("no binding");
        }

        // Binding.getContexts requires a starting index and the number of contexts
        iLength = oBinding.getLength();
        aSelectedKeys = this.getSelectedKeys();

        return new Promise(function(resolve, reject) {
            var aContexts = oBinding.getContexts(iIndexFrom, iLength),
                sSelectionIdProperty = oTable.getSelectionIdProperty(),
                aMatchingKeys = [], oContext, i, sKey, iIdx;

            //PERF - looping through many/all contexts in binding will be non-performant
            for(i = 0; i < aContexts.length; i++){
                oContext = aContexts[i];
                if (!sSelectionIdProperty) {
                    sKey = oContext.getPath();
                } else {
                    sKey = oContext.getProperty(sSelectionIdProperty);
                }

                iIdx = aSelectedKeys.indexOf(sKey);
                if(iIdx >= 0) {
                    // save the current context's key as a match
                    aMatchingKeys.push(sKey);
                    // make our list of selected keys faster to look through
                    aSelectedKeys.slice(iIdx, 1);
                }

                // stop looking once we run out of selected keys
                if (aSelectedKeys.length === 0) {
                    break;
                }
            }

            // update our selected paths
            oTable.setSelectedKeys(aMatchingKeys, "behavior");
            resolve(aMatchingKeys);
        });
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.unbindAggregation = function(sName, bSuppressReset) {
        var oBinding = this.getBinding("rows");
        if (sName === "rows" && this.isBound("rows")) {
            bSuppressReset = true;
        }

        var vReturn = Element.prototype.unbindAggregation.apply(this, [sName, bSuppressReset]);

        if (sName === "rows" && oBinding) {
            //Reset needs to be resetted, else destroyRows is called, which is not allowed to be called
            this._restoreAppDefaultsColumnHeaderSortFilter();
            // metadata might have changed
            this._invalidateColumnMenus();
            this._updateBindingLength();
            this.updateRows("unbindAggregation");
        }

        return vReturn;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setVisibleRowCount = function(iVisibleRowCount) {
        if (iVisibleRowCount !== null && !isFinite(iVisibleRowCount)) {
            return this;
        }

        var sVisibleRowCountMode = this.getVisibleRowCountMode();
        if (sVisibleRowCountMode === VisibleRowCountMode.Auto) {
            jQuery.sap.log.error("VisibleRowCount will be ignored since VisibleRowCountMode is set to Auto", this);
            return this;
        }

        var iFixedRowsCount = this.getFixedRowCount() + this.getFixedBottomRowCount();
        if (iVisibleRowCount <= iFixedRowsCount && iFixedRowsCount > 0) {
            jQuery.sap.log.error("Table: " + this.getId() + " visibleRowCount('" + iVisibleRowCount + "') must be bigger than number of fixed rows('" + (this.getFixedRowCount() + this.getFixedBottomRowCount()) + "')", this);
            return this;
        }

        iVisibleRowCount = this.validateProperty("visibleRowCount", iVisibleRowCount);
        if (this.getBinding("rows") && this.getBinding("rows").getLength() <= iVisibleRowCount) {
            this.setProperty("firstVisibleRow", 0);
        }
        this.setProperty("visibleRowCount", iVisibleRowCount);
        this._setRowContentHeight(iVisibleRowCount * this._getDefaultRowHeight());
        return this;
    };

    /**
     * Sets a new tooltip for this object. The tooltip can either be a simple string
     * (which in most cases will be rendered as the <code>title</code> attribute of this
     * Element) or an instance of {@link sap.ui.core.TooltipBase}.
     *
     * If a new tooltip is set, any previously set tooltip is deactivated.
     *
     * Please note that tooltips are not rendered for the table. The tooltip property will be set
     * but it won't effect the DOM.
     *
     * @param {string|sap.ui.core.TooltipBase} vTooltip
     * @returns {sas.hc.ui.table.Table} This-reference for chaining
     * @public
     * @override
     */
    Table.prototype.setTooltip = function(vTooltip) {
        jQuery.sap.log.warning("The aggregation tooltip is not supported for sas.hc.ui.table.Table");
        return this.setAggregation("tooltip", vTooltip, true);
    };

    /**
     * Requests fixed bottom row contexts from the binding.
     * @returns {sap.ui.model.Context[]} Array of fixed bottom row context
     * @private
     */
    Table.prototype._getFixedBottomRowContexts = function (iFixedBottomRowCount, iBindingLength) {
        var oBinding = this.getBinding("rows");
        var aContexts = [];
        if (!oBinding) {
            return aContexts;
        }

        iFixedBottomRowCount = iFixedBottomRowCount || this.getFixedBottomRowCount();
        iBindingLength = iBindingLength || oBinding.getLength();

        var iVisibleRowCount = this.getVisibleRowCount();
        if (iFixedBottomRowCount > 0 && (iVisibleRowCount - iFixedBottomRowCount) < iBindingLength) {
            aContexts = this._getContexts(iBindingLength - iFixedBottomRowCount, iFixedBottomRowCount, 1);
        }

        return aContexts;
    };

    /**
     * Requests fixed top row contexts from the binding.
     * @returns {sap.ui.model.Context[]} Array of fixed top row context
     * @private
     */
    Table.prototype._getFixedRowContexts = function(iFixedRowCount) {
        iFixedRowCount = iFixedRowCount || this.getFixedRowCount();
        if (iFixedRowCount > 0) {
            return this._getContexts(0, iFixedRowCount);
        } else {
            return [];
        }
    };

    /**
     * @private
     */
    Table.prototype._getContexts = function(iStartIndex, iLength, iThreshold) {
        var oBinding = this.getBinding("rows");
        if (oBinding) {
            return oBinding.getContexts(iStartIndex, iLength, iThreshold);
        } else {
            return [];
        }
    };

    /**
     * Requests all required contexts for visibleRowCount from the binding
     * @returns {sap.ui.model.Context[]} Array of row contexts
     * @private
     */
    Table.prototype._getRowContexts = function (iVisibleRows, bSkipSecondCall, sReason) {
        var bRecievedLessThanRequested = false;
        var aContexts = [];
        var oBinding = this.getBinding("rows");
        var iVisibleRowCount = iVisibleRows || this.getRows().length;
        if (!oBinding || iVisibleRowCount <= 0) {
            // without binding there are no contexts to be retrieved
            return [];
        }

        var iFirstVisibleRow = this.getFirstVisibleRow();

        var iFixedRowCount = this.getFixedRowCount();
        var iFixedBottomRowCount = this.getFixedBottomRowCount();
        var iReceivedLength = 0;
        var aTmpContexts;

        // because of the analytical table the fixed bottom row must always be requested separately as it is the grand
        // total row for the table.
        var iLength = iVisibleRowCount - iFixedBottomRowCount;
        var iMergeOffsetScrollRows = 0;
        var iMergeOffsetBottomRow = iLength;

        // if the threshold is not explicitly disabled by setting it to 0,
        // the default threshold should be at the the visibleRowCount.
        var iThreshold = this.getThreshold();
        iThreshold = iThreshold ? Math.max(iVisibleRowCount, iThreshold) : 0;

        // data can be requested with a single getContexts call if the fixed rows and the scrollable rows overlap.
        var iStartIndex = iFirstVisibleRow;

        var fnMergeArrays = function (aTarget, aSource, iStartIndex) {
            for (var i = 0; i < aSource.length; i++) {
                aTarget[iStartIndex + i] = aSource[i];
            }
        };

        if (iFixedRowCount > 0 && iFirstVisibleRow > 0) {
            // since there is a gap between first visible row and fixed rows it must be requested separately
            // the first visible row always starts counting with 0 in the scroll part of the table no matter
            // how many fixed rows there are.
            iStartIndex = iFirstVisibleRow + iFixedRowCount;
            // length must be reduced by number of fixed rows since they were just requested separately
            iLength -= iFixedRowCount;
            iMergeOffsetScrollRows = iFixedRowCount;
            // retrieve fixed rows separately
            aTmpContexts = this._getFixedRowContexts(iFixedRowCount);
            iReceivedLength += aTmpContexts.length;
            aContexts = aContexts.concat(aTmpContexts);
        }

        // request scroll part contexts but may include fixed rows depending on scroll and length settings
        // if this is done before requesting fixed bottom rows, it saves some performance for the analytical table
        // since the tree gets only build once (as result of getContexts call). If first the fixed bottom row would
        // be requested the analytical binding would build the tree twice.
        aTmpContexts = this._getContexts(iStartIndex, iLength, iThreshold);
        var iBindingLength = this._updateBindingLength(sReason);
        // iLength is the number of rows which shall get filled. It might be more than the binding actually has data.
        // Therefore Math.min is required to make sure to not request data again from the binding.
        bRecievedLessThanRequested = aTmpContexts.length < Math.min(iLength, iBindingLength - iFixedBottomRowCount);

        // get the binding length after getContext call to make sure that for TreeBindings the client tree was correctly rebuilt
        // this step can be moved to an earlier point when the TreeBindingAdapters all implement tree invalidation in case of getLength calls
        iReceivedLength += aTmpContexts.length;
        fnMergeArrays(aContexts, aTmpContexts, iMergeOffsetScrollRows);

        // request binding length after getContexts call to make sure that in case of tree binding and analytical binding
        // the tree gets only built once (by getContexts call).
        iMergeOffsetBottomRow = Math.min(iMergeOffsetBottomRow, Math.max(iBindingLength - iFixedBottomRowCount, 0));
        if (iFixedBottomRowCount > 0) {
            // retrieve fixed bottom rows separately
            // instead of just concatenating them to the existing contexts it must be made sure that they are put
            // to the correct row index otherwise they would flip into the scroll area in case data gets requested for
            // the scroll part.
            aTmpContexts = this._getFixedBottomRowContexts(iFixedBottomRowCount, iBindingLength);
            iReceivedLength += aTmpContexts.length;
            fnMergeArrays(aContexts, aTmpContexts, iMergeOffsetBottomRow);
        }

        if (bRecievedLessThanRequested && !bSkipSecondCall) {
            // check of binding length required
            var iFirstVisibleRowSanitized = this._getSanitizedFirstVisibleRow(true);
            if (iFirstVisibleRow !== iFirstVisibleRowSanitized) {
                // get contexts again, this time with adjusted scroll position
                aContexts = this._getRowContexts(iVisibleRowCount, true);
                iReceivedLength = aContexts.length;
            }
        }

        if (!bSkipSecondCall) {
            var that = this;
            if (this._mTimeouts.getContextsSetBusy) {
                window.clearTimeout(this._mTimeouts.getContextsSetBusy);
            }
            this._mTimeouts.getContextsSetBusy = window.setTimeout(function() {
                that._setBusy({
                    requestedLength: iFixedRowCount + iLength + iFixedBottomRowCount,
                    receivedLength: iReceivedLength,
                    contexts: aContexts,
                    reason: sReason});
            }, 0);
        }

        return aContexts;
    };

    /**
     * @private
     */
    Table.prototype._getSanitizedFirstVisibleRow = function(bUpdate) {
        var iVisibleRowCount = this.getVisibleRowCount();
        var iFirstVisibleRow = this.getFirstVisibleRow();
        // calculate the boundaries (at least 0 - max the row count - visible row count)
        iFirstVisibleRow = Math.max(iFirstVisibleRow, 0);

        if (this._iBindingLength > 0) {
            iFirstVisibleRow = Math.min(iFirstVisibleRow, Math.max(this._iBindingLength - iVisibleRowCount, 0));
        }

        if (bUpdate) {
            this.setProperty("firstVisibleRow", iFirstVisibleRow, true);
        }

        return iFirstVisibleRow;
    };

    /**
     * @private
     */
    Table.prototype._updateBindingLength = function(sReason) {
        // get current binding length. If the binding length changes it must call updateAggregation (updateRows)
        // therefore it should be save to buffer the binding length here. This gives some performance advantage
        // especially for tree bindings using the TreeBindingAdapter where a tree structure must be created to
        // calculate the correct length.
        var oBinding = this.getBinding("rows");
        var iBindingLength = 0;
        if (oBinding) {
            iBindingLength = oBinding.getLength();
        }

        if (iBindingLength !== this._iBindingLength) {
            this._iBindingLength = iBindingLength;
            this._onBindingLengthChange(sReason);
        }

        return iBindingLength;
    };

    /**
     * @private
     */
    Table.prototype._onBindingLengthChange = function(sReason) {
        // update visualization of fixed bottom row
        this._updateFixedBottomRows();
        this._toggleVSb();
        this._bBindingLengthChanged = true;
        // show or hide the no data container
        if (sReason !== "skipNoDataUpdate") {
            // in order to have less UI updates, the NoData text should not be updated when the reason is refresh. When
            // refreshRows was called, the table will request data and later get another change event. In that turn, the
            // noData text gets updated.
            this._updateNoData();
        }
        this._updateA11yVisibleRowsLabel();
    };

    /**
     * updates the rows - called internally by the updateAggregation function when
     * anything in the model has been changed.
     * @private
     */
    Table.prototype.updateRows = function(sReason) {
        if (this._bExitCalled) {
            return;
        }

        // update busy indicator state
        this._setBusy(sReason ? {changeReason: sReason} : false);

        // if the binding length has changed due to filter or sorter, it may happened that the noData text was not updated in order
        // to avoid flickering of the table.
        // therefore we need to update the noData text here
        if (this._bBindingLengthChanged) {
            this._updateNoData();
        }

        // Rows should only be created/cloned when the number of rows can be determined. For the VisibleRowCountMode: Auto
        // this can only happen after the table control was rendered one. At this point in time we know how much space is
        // consumed by the table header, toolbar, footer... and we can calculate how much space is left for the table rows.
        var sVisibleRowCountMode = this.getVisibleRowCountMode();
        if ((this.getRows().length <= 0 || !this._oRowTemplate) && ((sVisibleRowCountMode === VisibleRowCountMode.Auto && this.bOutput) || sVisibleRowCountMode !== VisibleRowCountMode.Auto)) {
            if (this._iTableRowContentHeight) {
                this._adjustRows(this._calculateRowsToDisplay());
            }
        }

        // when not scrolling we update also the scroll position of the scrollbar
        //if (this._oVSb.getScrollPosition() !== iStartIndex) {
        // TODO
        //this._updateAriaRowOfRowsText(true);
        //}

        // update the bindings only once the table is rendered
        if (!this.bIsDestroyed) {
            // update the bindings by using a delayed mechanism to avoid to many update
            // requests: by using the mechanism below it will trigger an update each 50ms
            // except if the reason is coming from the binding with reason "change" then
            // we do an immediate update instead of a delayed one

            if(sReason === ChangeReason.Change && !this.getSelectionIdProperty()){
                // selected paths are unreliable if the length changes w/o using primary key
//                this.clearSelectedKeys(sReason);
            }

            var iBindingTimerDelay = (sReason === ChangeReason.Change || (!this._mTimeouts.bindingTimer && Date.now() - this._lastCalledUpdateRows > this._iBindingTimerDelay) || sReason === "unbindAggregation" ? 0 : this._iBindingTimerDelay);
            var that = this;
            if (iBindingTimerDelay === 0 && sReason) {
                Promise.resolve().then(function() {
                    that._performUpdateRows(sReason);
                });
            } else {
                this._mTimeouts.bindingTimer = this._mTimeouts.bindingTimer || window.setTimeout(function() {
                    that._performUpdateRows(sReason);
                }, iBindingTimerDelay);
            }
        }
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.insertRow = function() {
        jQuery.sap.log.error("The control manages the rows aggregation. The method \"insertRow\" cannot be used programmatically!", this);
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.addRow = function() {
        jQuery.sap.log.error("The control manages the rows aggregation. The method \"addRow\" cannot be used programmatically!", this);
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.removeRow = function() {
        jQuery.sap.log.error("The control manages the rows aggregation. The method \"removeRow\" cannot be used programmatically!", this);
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.removeAllRows = function() {
        jQuery.sap.log.error("The control manages the rows aggregation. The method \"removeAllRows\" cannot be used programmatically!", this);
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.destroyRows = function() {
        jQuery.sap.log.error("The control manages the rows aggregation. The method \"destroyRows\" cannot be used programmatically!", this);
    };

    /**
     * Triggers automatic resizing of a column to the widest content.
     *
     * @experimental Experimental! Presently implemented to only work with a very limited set of controls (e.g. sap.m.Text).
     * @param {int} iColIndex The index of the column in the list of visible columns.
     * @function
     * @public
     */
    Table.prototype.autoResizeColumn = function(iColIndex) {
        this._getPointerExtension().doAutoResizeColumn(iColIndex);
    };

    /**
     * Sets the column width to the widest content.
     * @param {int} iColIndex The index of the column in the list of visbile columns
     * @public
     */
    Table.prototype.autoFitColumn = function(iColIndex) {
        this._getPointerExtension().doBestFitColumn(iColIndex);
    };

    /**
     * Sets all visible columns to the widest content of each column.
     * @public
     */
    Table.prototype.autoFitColumns = function() {
        this._getPointerExtension().autoFitAllColumns();
    };

    /**
     * Returns true if any of the columns are auto resizable.
     * @private
     */
    Table.prototype._isAutoResizeEnabled = function() {
        var aCols = this.retrieveLeafColumns();
        for (var i = 0, l = aCols.length; i < l; i++) {
            if (aCols[i].getAutoResizable && aCols[i].getAutoResizable()) {
                return true;
            }
        }
        return false;
    };

    /**
     * Scrolls horizontally to the beginning of the scrollable portion of the table
     * @public
     */
    Table.prototype.scrollToFirstScrollableColumn = function() {
        var $hsb = this._getHsbRef();
        if ($hsb) {
            $hsb.scrollLeft(0);
        }
    };


    // =============================================================================
    // EVENT HANDLING & CLEANUP
    // =============================================================================

    /**
     * attaches the required native event handlers
     * @private
     */
    Table.prototype._attachEvents = function() {
        var $this = this.$();

        // listen to the scroll events of the containers (for keyboard navigation)
        $this.find(".sapUiTableColHdrScr").scroll(jQuery.proxy(this._oncolscroll, this));
        $this.find(".sapUiTableCtrlScr").scroll(jQuery.proxy(this._oncntscroll, this));
        $this.find(".sapUiTableCtrlScrFixed").scroll(jQuery.proxy(this._oncntscroll, this));

        $this.find(".sapUiTableCtrlScrFixed, .sapUiTableColHdrFixed").on("scroll.sapUiTablePreventFixedAreaScroll", function(oEvent) {oEvent.target.scrollLeft = 0;});
        if (TableUtils.isVariableRowHeightEnabled(this)) {
            $this.find(".sapUiTableCCnt").on("scroll.sapUiTablePreventCCntScroll", function(oEvent) {oEvent.target.scrollTop = 0;});
        }

        // sync row header > content (hover effect)
        $this.find(".sapUiTableCCnt .sapUiTableRowHdr").hover(function() {
            jQuery(this).addClass("sapUiTableRowHvr");
            var iIndex = $this.find(".sapUiTableCCnt .sapUiTableRowHdr").index(this);
            $this.find(".sapUiTableCCnt .sapUiTableCtrlFixed > tbody > tr").filter(":eq(" + iIndex + ")").addClass("sapUiTableRowHvr");
            $this.find(".sapUiTableCCnt .sapUiTableCtrlScroll > tbody > tr").filter(":eq(" + iIndex + ")").addClass("sapUiTableRowHvr");
        }, function() {
            jQuery(this).removeClass("sapUiTableRowHvr");
            $this.find(".sapUiTableCCnt .sapUiTableCtrlFixed > tbody > tr").removeClass("sapUiTableRowHvr");
            $this.find(".sapUiTableCCnt .sapUiTableCtrlScroll > tbody > tr").removeClass("sapUiTableRowHvr");
        });

        // sync content fixed > row header (hover effect)
        $this.find(".sapUiTableCCnt .sapUiTableCtrlFixed > tbody > tr").hover(function() {
            jQuery(this).addClass("sapUiTableRowHvr");
            var iIndex = $this.find(".sapUiTableCCnt .sapUiTableCtrlFixed > tbody > tr").index(this);
            $this.find(".sapUiTableCCnt .sapUiTableRowHdr").filter(":eq(" + (iIndex) + ")").addClass("sapUiTableRowHvr");
            $this.find(".sapUiTableCCnt .sapUiTableCtrlScroll > tbody > tr").filter(":eq(" + iIndex + ")").addClass("sapUiTableRowHvr");
        }, function() {
            jQuery(this).removeClass("sapUiTableRowHvr");
            $this.find(".sapUiTableCCnt .sapUiTableRowHdr").removeClass("sapUiTableRowHvr");
            $this.find(".sapUiTableCCnt .sapUiTableCtrlScroll > tbody > tr").removeClass("sapUiTableRowHvr");
        });

        // sync content scroll > row header (hover effect)
        $this.find(".sapUiTableCCnt .sapUiTableCtrlScroll > tbody > tr").hover(function() {
            jQuery(this).addClass("sapUiTableRowHvr");
            var iIndex = $this.find(".sapUiTableCCnt .sapUiTableCtrlScroll > tbody > tr").index(this);
            $this.find(".sapUiTableCCnt .sapUiTableRowHdr").filter(":eq(" + iIndex + ")").addClass("sapUiTableRowHvr");
            $this.find(".sapUiTableCCnt .sapUiTableCtrlFixed > tbody > tr").filter(":eq(" + iIndex + ")").addClass("sapUiTableRowHvr");
        }, function() {
            jQuery(this).removeClass("sapUiTableRowHvr");
            $this.find(".sapUiTableCCnt .sapUiTableRowHdr").removeClass("sapUiTableRowHvr");
            $this.find(".sapUiTableCCnt .sapUiTableCtrlFixed > tbody > tr").removeClass("sapUiTableRowHvr");
        });

        this._getPointerExtension().initColumnResizeEvents();

        var $vsb = jQuery(this.getDomRef(SharedDomRef.VerticalScrollBar));
        var $hsb = jQuery(this.getDomRef(SharedDomRef.HorizontalScrollBar));
        $vsb.bind("scroll.sapUiTableVScroll", this.onvscroll.bind(this));
        $hsb.bind("scroll.sapUiTableHScroll", this.onhscroll.bind(this));

        // For preventing the ItemNavigation to re-apply focus to old position (table cell)
        // when clicking on ScrollBar
        $hsb.bind("mousedown.sapUiTableHScroll", function(oEvent) {
            oEvent.preventDefault();
        });
        $vsb.bind("mousedown.sapUiTableVScroll", function(oEvent) {
            oEvent.preventDefault();
        });

        if (Device.browser.firefox) {
            this._getScrollTargets().bind("MozMousePixelScroll.sapUiTableMouseWheel", this._onMouseWheel.bind(this));
        } else {
            this._getScrollTargets().bind("wheel.sapUiTableMouseWheel", this._onMouseWheel.bind(this));
        }
    };

    /**
     * detaches the required native event handlers
     * @private
     */
    Table.prototype._detachEvents = function() {
        var $this = this.$();

        $this.find(".sapUiTableRowHdrScr").unbind();
        $this.find(".sapUiTableCtrl > tbody > tr").unbind();
        $this.find(".sapUiTableRowHdr").unbind();
        //this._getPointerExtension().cleanupColumnResizeEvents();
        $this.find(".sapUiTableCtrlScrFixed, .sapUiTableColHdrFixed").unbind("scroll.sapUiTablePreventFixedAreaScroll");

        if (TableUtils.isVariableRowHeightEnabled(this)) {
            $this.find(".sapUiTableCCnt").unbind("scroll.sapUiTablePreventCCntScroll");
        }

        var $vsb = jQuery(this.getDomRef(SharedDomRef.VerticalScrollBar));
        $vsb.unbind("scroll.sapUiTableVScroll");
        $vsb.unbind("mousedown.sapUiTableVScroll");

        var $hsb = jQuery(this.getDomRef(SharedDomRef.HorizontalScrollBar));
        $hsb.unbind("scroll.sapUiTableHScroll");
        $hsb.unbind("mousedown.sapUiTableHScroll");

        var $scrollTargets = this._getScrollTargets();
        $scrollTargets.unbind("MozMousePixelScroll.sapUiTableMouseWheel");
        $scrollTargets.unbind("wheel.sapUiTableMouseWheel");

        var $body = jQuery(document.body);
        $body.unbind('webkitTransitionEnd transitionend');

        TableUtils.deregisterResizeHandler(this);
    };

    /**
     * Collect the scroll wheel/touch targets needed for scrolling the table.
     * @returns {*}
     * @private
     */
    Table.prototype._getScrollTargets = function() {
        var $ctrlScr = jQuery(this.getDomRef("sapUiTableCtrlScr"));
        var $rsz = jQuery(this.getDomRef("rsz"));
        var $ctrlScrFixed = jQuery(this.getDomRef("sapUiTableCtrlScrFixed"));
        var $rowHdrScr = jQuery(this.getDomRef("sapUiTableRowHdrScr"));
        return $ctrlScr.add($ctrlScrFixed).add($rowHdrScr).add($rsz);
    };

    /**
     * cleanup the timers when not required anymore
     * @private
     */
    Table.prototype._cleanUpTimers = function() {

        for (var sKey in this._mTimeouts) {
            if (this._mTimeouts.hasOwnProperty(sKey)) {
                if (this._mTimeouts[sKey]) {
                    clearTimeout(this._mTimeouts[sKey]);
                    this._mTimeouts[sKey] = undefined;
                }
            }
        }
    };

    // =============================================================================
    // PRIVATE TABLE STUFF :)
    // =============================================================================
    /**
     * updates the horizontal scrollbar
     * @private
     */
    Table.prototype._updateHSb = function(oTableSizes) {
        // get the width of the container
        var $this = this.$();
        var iColsWidth = oTableSizes.tableCtrlScrollWidth;
        if (!!Device.browser.safari) {
            iColsWidth = Math.max(iColsWidth, this._getColumnsWidth(this.getTotalFrozenColumnCount()));
        }

        // add the horizontal scrollbar
        if (iColsWidth > oTableSizes.tableCtrlScrWidth) {
            // show the scrollbar
            if (!$this.hasClass("sapUiTableHScr")) {
                $this.addClass("sapUiTableHScr");
            }

            var iScrollPadding = oTableSizes.tableCtrlFixedWidth;
            if ($this.find(".sapUiTableRowHdrScr").length > 0) {
                iScrollPadding += oTableSizes.tableRowHdrScrWidth;
            }

            // For S1403330. The high contrast theme needs just a little less margin
            // to get the HSB border to align with the fixed column border.
            if (sap.ui.getCore().getConfiguration().getTheme() === "sas_hcb") {
                iScrollPadding -= 1;
            }

            if (this.getRows().length > 0) {
                var $sapUiTableHSb = $this.find(".sapUiTableHSb");
                if (this._bRtlMode) {
                    $sapUiTableHSb.css('margin-right', iScrollPadding + 'px');
                } else {
                    $sapUiTableHSb.css('margin-left', iScrollPadding + 'px');
                }
            }

            var oHSbContent = this.getDomRef("hsb-content");
            if (oHSbContent) {
                oHSbContent.style.width = iColsWidth + "px";
            }
        } else {
            // hide the scrollbar
            if ($this.hasClass("sapUiTableHScr")) {
                $this.removeClass("sapUiTableHScr");
                if (!!Device.browser.safari) {
                    // min-width on table elements does not work for safari
                    $this.find(".sapUiTableCCnt .sapUiTableCtrlScroll, .sapUiTableColHdr").css("width", "");
                }
            }
        }
    };

    /**
     * Update the vertical scrollbar position
     * @private
     */
    Table.prototype._updateVSb = function(oTableSizes) {
        // move the vertical scrollbar to the scrolling table only
        var oVSb = this.getDomRef(SharedDomRef.VerticalScrollBar);
        if (!oVSb) {
            return;
        }

        var iFixedRowCount = this.getFixedRowCount();
        var iFixedBottomRowCount = this.getFixedBottomRowCount();
        if (iFixedRowCount > 0 || iFixedBottomRowCount > 0) {
            var $sapUiTableVSb = jQuery(oVSb);
            if (iFixedRowCount > 0) {
                $sapUiTableVSb.css('top', (oTableSizes.tableCtrlRowScrollTop - 1) + 'px');
            }
            if (iFixedBottomRowCount > 0) {
                $sapUiTableVSb.css('height', oTableSizes.tableCtrlRowScrollHeight + 'px');
            }
        }

        var iDefaultRowHeight = this._getDefaultRowHeight();
        var iVSbHeight = (this._iBindingLength - iFixedRowCount - iFixedBottomRowCount) * iDefaultRowHeight;
        if (TableUtils.isVariableRowHeightEnabled(this)) {
            var iCCntHeight = 0;
            var oCCntDomRef = this.getDomRef("tableCCnt");
            if (oCCntDomRef) {
                iCCntHeight = oCCntDomRef.clientHeight;
            }

            var iTableHeight = this._getScrollableTableHeight();

            this._iRowHeightsDelta = iTableHeight - iCCntHeight - TableUtils.getRowHeightByIndex(this, this.getRows().length - 1);
            if (this._iBindingLength <= this.getVisibleRowCount()) {

                // if bound model has less items than visible rows, we only need to scroll within the rows with data
                iVSbHeight = this._iRowHeightsDelta + (this._iBindingLength * iDefaultRowHeight);

            } else {
                iVSbHeight = this._iRowHeightsDelta + iVSbHeight;
            }
            this._updateA11yVisibleRowsLabel();
            this._toggleVSb();
        }
        this.getDomRef("vsb").style.maxHeight = (this.getVisibleRowCount() * iDefaultRowHeight) + "px";
        this.getDomRef("vsb-content").style.height = iVSbHeight + "px";

        if (!TableUtils.isVariableRowHeightEnabled(this)) {
            oVSb.scrollTop = this._getSanitizedFirstVisibleRow() * iDefaultRowHeight;
        }
    };

    /**
     * Attempts to determine the height of scrollable section of the Table.
     * If it cannot determine the height, it returns 0.
     * @private
     * @returns {int} The height of the scrollable section of the Table. If the height cannot be determined, 0.
     */
    Table.prototype._getScrollableTableHeight = function() {
        var oTableDomRef = this.getDomRef("table");
        if (oTableDomRef && oTableDomRef.clientHeight > 0) {
            return oTableDomRef.clientHeight;
        }

        // Some browsers (FF) report an empty non-frozen section as 0 height.
        // Also check the frozen section for a possible height.
        var oFixedTableDomRef = this.getDomRef("table-fixed");
        if (oFixedTableDomRef && oFixedTableDomRef.clientHeight > 0) {
            return oFixedTableDomRef.clientHeight;
        }

        return 0;
    };

    /**
     * Toggles the visibility of the Vertical Scroll Bar
     * @private
     */
    Table.prototype._toggleVSb = function() {
        var $this = this.$();
        if (this.getDomRef()) {
            // in case of Scrollbar Mode show/hide the scrollbar depending whether it is needed.
            $this.toggleClass("sapUiTableVScr", this._isVSbRequired());
        }
    };

    /**
     * updates the binding contexts of the currently visible controls
     * @param {boolean} bSuppressUpdate if true, only context will be requested but no binding context set
     * @param {int} iRowCount number of rows to be updated and number of contexts to be requested from binding
     * @param {String} sReason reason for the update; used to control further lifecycle
     * @private
     */
    Table.prototype._updateBindingContexts = function(bSuppressUpdate, iRowCount, sReason) {
        var aRows = this.getRows(),
            oBinding = this.getBinding("rows"),
            oBindingInfo = this.mBindingInfos["rows"],
            aContexts;

        // fetch the contexts from the binding
        if (oBinding) {
            aContexts = this._getRowContexts(iRowCount, false, sReason);
        }

        if (!bSuppressUpdate) {
            var iFirstVisibleRow = this.getFirstVisibleRow();
            // row heights must be reset to make sure that rows can shrink if they may have smaller content. The content
            // shall control the row height.
            this._resetRowHeights();
            for (var iIndex = aRows.length - 1; iIndex >= 0; iIndex--) {
                var oContext = aContexts ? aContexts[iIndex] : undefined;
                var oRow = aRows[iIndex];
                if (oRow) {
                    //calculate the absolute row index, used by the Tree/AnalyticalTable to find the rendering infos for this row
                    var iAbsoluteRowIndex = iFirstVisibleRow + iIndex;
                    this._updateRowBindingContext(oRow, oContext, oBindingInfo && oBindingInfo.model, iAbsoluteRowIndex, oBinding);
                }
            }
        }

        //If rememberSelections is on and we are here due to sorting or filtering then reapply the remembered selections
        if(this.getRememberSelections() && (sReason === ChangeReason.Sort || sReason === ChangeReason.Filter)) {
            this.applyRememberedSelections(sReason);
        }

        this.fireUpdateFinished({
            reason : sReason
        });
    };

    /**
     * updates the binding context a row
     * @param {sas.hc.ui.table.Row} oRow row to update
     * @param {sap.ui.model.Context} oContext binding context of the row
     * @param {String} sModelName name of the model
     * @param {int} iAbsoluteRowIndex index of row considering the scroll position
     * @private
     */
    Table.prototype._updateRowBindingContext = function(oRow, oContext, sModelName, iAbsoluteRowIndex, oBinding) {
        // check for a context object (in case of grouping there could be custom context objects)
        oRow.setRowBindingContext(oContext, sModelName, oBinding);
    };

    /**
     * show or hide the no data container
     * @private
     */
    Table.prototype._updateNoData = function() {
        if (!this.getDomRef()) {
            return;
        }

        var oFocusRef = document.activeElement;
        this.$().toggleClass("sapUiTableEmpty", TableUtils.isNoDataVisible(this));
        this._getAccExtension().updateAriaStateForOverlayAndNoData();
        this._getKeyboardExtension().updateNoDataAndOverlayFocus(oFocusRef);
    };

    /**
     * determines the currently visible columns (used for simply updating only the
     * controls of the visible columns instead of the complete row!)
     * @private
     */
    Table.prototype._determineVisibleCols = function(oTableSizes) {
        // TODO: to be implemented; currently, all columns are counted
        var aColumns = [];
        this.retrieveLeafColumns().forEach(function(column, i){
            if (column.shouldRender()) {
                aColumns.push(i);
            }
        });
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.removeColumn = function (oColumn, bSuppressInvalidate) {
        var oResult = this.removeAggregation('columns', oColumn, bSuppressInvalidate);

        if (typeof oColumn === "number" && oColumn > -1) {
            oColumn = this.getColumns()[oColumn];
        }

        var iIndex = jQuery.inArray(oColumn, this._aSortedColumns);
        if (this._iNewColPos === undefined && iIndex >= 0) {
            this._aSortedColumns.splice(iIndex, 1);
        }

        if (!this.isBound("columns")) {
            this._resetRowTemplate();
        }

        this._aLeafCols = null;
        if (oColumn instanceof ColumnGroup) {
            this._iColumnGroupCount--;
            this._restrictTableFeatures(this.hasColumnGroups());
        }

        return oResult;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.removeAllColumns = function() {
        var oResult = this.removeAllAggregation('columns');
        this._aSortedColumns = [];

        if (!this.isBound("columns")) {
            this._resetRowTemplate();
        }

        this._iColumnGroupCount = 0;
        this._restrictTableFeatures(false);
        this._aLeafCols = null;

        return oResult;
    };

    /*
     * @see JSDoc generated by SAPUI5 contdrol API generator
     */
    Table.prototype.destroyColumns = function() {
        var oResult = this.destroyAggregation('columns');
        this._aSortedColumns = [];

        if (!this.isBound("columns")) {
            this._resetRowTemplate();
        }

        this._iColumnGroupCount = 0;
        this._restrictTableFeatures(false);
        this._aLeafCols = null;

        return oResult;
    };


    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.addColumn = function (oColumn, bSuppressInvalidate) {
        // perform validation on the incoming control
        oColumn = this.validateAggregation('columns', oColumn, /* multiple */ true, /* omit forwarding */ true);

        // Mandatory Columns must be before non-mandatory Columns.
        if (oColumn && oColumn.getMandatory()) {
            return this.insertColumn(oColumn, this._getMandatoryColumnCount(), bSuppressInvalidate);
        }

        this.addAggregation('columns', oColumn, bSuppressInvalidate);

        this._aLeafCols = null;
        if (oColumn instanceof ColumnGroup) {
            this._iColumnGroupCount++;
            this._restrictTableFeatures(true);
        }

        if (!this.isBound("columns")) {
            this._resetRowTemplate();
        }

        return this;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.insertColumn = function (oColumn, iIndex, bSuppressInvalidate) {
        // perform validation on the incoming control
        oColumn = this.validateAggregation('columns', oColumn, /* multiple */ true, /* omit forwarding */ true);

        // Make sure mandatory Column is not in the middle of non-mandatory Columns
        var bColumnIsMandatory = oColumn && oColumn.getMandatory();
        if (bColumnIsMandatory) {
            iIndex = Math.min(iIndex, this._getMandatoryColumnCount());
        } else {
            iIndex = Math.max(iIndex, this._getMandatoryColumnCount());
        }

        this.insertAggregation('columns', oColumn, iIndex, bSuppressInvalidate);

        this._aLeafCols = null;
        if (oColumn instanceof ColumnGroup) {
            this._iColumnGroupCount++;
            this._restrictTableFeatures(true);
        }

        if (!this.isBound("columns")) {
            this._resetRowTemplate();
        }

        return this;
    };


    Table.prototype.updateAggregation = function(sName) {
        var vReturnValue = Control.prototype.updateAggregation.apply(this, arguments);

        if (sName === "columns") {
            this._resetRowTemplate();
        }

        return vReturnValue;
    };

    /**
     * Returns true if there are any column groups set
     *
     * @return True if this Table has at least one ColumnGroup, False otherwise.
     */
    Table.prototype.hasColumnGroups = function() {
        return this._iColumnGroupCount > 0;
    };

    /**
     * @private
     */
    Table.prototype._enforceMandatoryColumnOrder = function() {
        var aMandatoryColumns = [];
        var aNonMandatoryColumns = [];
        this.retrieveLeafColumns().forEach(function(oCol) {
            if (oCol.getMandatory()) {
                aMandatoryColumns.push(oCol);
            } else {
                aNonMandatoryColumns.push(oCol);
            }
        });

        this.removeAllColumns();

        var aCombinedCols = aMandatoryColumns.concat(aNonMandatoryColumns);
        var self = this;
        aCombinedCols.forEach(function(oCol) {
            self.addColumn(oCol);
        });
    };

    /**
     * returns the count of rows when bound or 0
     * @private
     */
    Table.prototype._getRowCount = function() {
        return this._iBindingLength;
    };

    /**
     * returns the count of rows which can ca selected when bound or 0
     * @private
     */
    Table.prototype._getSelectableRowCount = function() {
        return this._iBindingLength;
    };


    /**
     * Returns the current top scroll position of the scrollbar (row number)
     * @private
     */
    Table.prototype._getFirstVisibleRowByScrollTop = function(iScrollTop) {
        var oVsb = this.getDomRef(SharedDomRef.VerticalScrollBar);
        if (oVsb) {
            iScrollTop = (typeof iScrollTop === "undefined") ? oVsb.scrollTop : iScrollTop;
            if (TableUtils.isVariableRowHeightEnabled(this)) {
                if (this.getVisibleRowCount() >= this._iBindingLength) {
                    return 0;
                } else {
                    return Math.min(this._iBindingLength - this.getVisibleRowCount(), Math.floor(iScrollTop / this._getDefaultRowHeight()));
                }
            } else {
                return Math.ceil(iScrollTop / this._getDefaultRowHeight());
            }
        }

        return 0;
    };

    /**
     * returns the count of visible columns
     * @private
     */
    Table.prototype._getVisibleColumns = function() {
        var aColumns = [];
        var aCols = this.retrieveLeafColumns();
        for (var i = 0, l = aCols.length; i < l; i++) {
            if (aCols[i].shouldRender()) {
                aColumns.push(aCols[i]);
            }
        }
        return aColumns;
    };


    /**
     * Returns the summed width of all rendered columns
     * @private
     * @param {Number} iStartColumn starting column for calculating the width
     * @param {Number} iEndColumn ending column for calculating the width
     * @returns {Number} the summed column width
     */
    Table.prototype._getColumnsWidth = function(iStartColumn, iEndColumn) {
        // first calculate the min width of the table for all columns
        var aCols = this.retrieveLeafColumns();
        var iColsWidth = 0;

        if (iStartColumn !== 0 && !iStartColumn) {
            iStartColumn = 0;
        }
        if (iEndColumn !== 0 && !iEndColumn) {
            iEndColumn = aCols.length;
        }

        for (var i = iStartColumn, l = iEndColumn; i < l; i++) {
            if (aCols[i] && aCols[i].shouldRender() && aCols[i].getWidth) {
                iColsWidth += this._CSSSizeToPixel(aCols[i].getWidth());
            }
        }

        return iColsWidth;

    };

    /**
     * Calculates the pixel value from a given CSS size and returns it with or without unit.
     * @param sCSSSize
     * @param bReturnWithUnit
     * @returns {string|number} Converted CSS value in pixel
     * @private
     */
    Table.prototype._CSSSizeToPixel = function(sCSSSize, bReturnWithUnit) {
        var sPixelValue = this._iColMinWidth;

        if (typeof(sCSSSize) === "number" ) {
            sPixelValue = sCSSSize.toString();
        } else if (sCSSSize) {
            if (jQuery.sap.endsWith(sCSSSize, "px")) {
                sPixelValue = parseInt(sCSSSize, 10);
            } else if (jQuery.sap.endsWith(sCSSSize, "em") || jQuery.sap.endsWith(sCSSSize, "rem")) {
                sPixelValue = Math.ceil(parseFloat(sCSSSize) * this._getBaseFontSize());
            }
        }

        if (bReturnWithUnit) {
            return sPixelValue + "px";
        } else {
            return parseInt(sPixelValue, 10);
        }
    };

    /**
     * @private
     */
    Table.prototype._getBaseFontSize = function() {
        return this._iBaseFontSize;
    };

    /**
     * Triggered by the ResizeHandler if width/height changed.
     * @private
     */
    Table.prototype._onTableResize = function() {
        // AppSwitching will trigger the ResizeHandler which can cause problems
        // when Table invalidates within an iframe that is being hidden
        var bTriggeredFromAppSwitch = this.$().outerWidth() === 0;

        if (this._bInvalid || !this.getDomRef() || bTriggeredFromAppSwitch) {
            return;
        }

        if (this._bRestoreColumnWidthsOnResize) {
            this._restoreOriginalColumnWidths();
        }

        this._normalizeColumnWidths();

        this._updateTableSizes();
        this._syncHeaderAndContent(this._collectTableSizes());
    };

    /**
     * @private
     */
    Table.prototype._handleRowCountModeAuto = function(iTableAvailableSpace) {
        var oBinding = this.getBinding("rows");
        if (oBinding && this.getRows().length > 0) {
            return this._executeAdjustRows(iTableAvailableSpace);
        } else {
            var that = this;
            var bReturn = !this._mTimeouts.handleRowCountModeAutoAdjustRows;
            var iBusyIndicatorDelay = that.getBusyIndicatorDelay();
            var bEnableBusyIndicator = this.getEnableBusyIndicator();
            if (oBinding && bEnableBusyIndicator) {
                that.setBusyIndicatorDelay(0);
                that.setBusy(true);
            }

            if (iTableAvailableSpace) {
                this._setRowContentHeight(iTableAvailableSpace);
            }

            this._mTimeouts.handleRowCountModeAutoAdjustRows = this._mTimeouts.handleRowCountModeAutoAdjustRows || window.setTimeout(function() {
                if (!that._executeAdjustRows()) {
                    // table sizes were not updated by AdjustRows
                    that._updateTableSizes(false, true);
                }
                that._mTimeouts.handleRowCountModeAutoAdjustRows = undefined;
                if (bEnableBusyIndicator) {
                    that.setBusy(false);
                    that.setBusyIndicatorDelay(iBusyIndicatorDelay);
                }
            }, 0);
            return bReturn;
        }
    };

    /**
     * @private
     */
    Table.prototype._executeAdjustRows = function(iTableAvailableSpace) {
        iTableAvailableSpace = iTableAvailableSpace || this._determineAvailableSpace();

        //if visibleRowCountMode is auto change the visibleRowCount according to the parents container height
        var iRows = this._calculateRowsToDisplay(iTableAvailableSpace);
        // if minAutoRowCount has reached, table should use block this height.
        // In case row > minAutoRowCount, the table height is 0, because ResizeTrigger must detect any changes of the table parent.
        if (iRows === this._determineMinAutoRowCount()) {
            this.$().height("auto");
        } else {
            this.$().height("0px");
        }

        return this._adjustRows(iRows);
    };

    /**
     * Synchronizes the <th> width of the table, with the rendered header divs.
     * @private
     */
    Table.prototype._syncColumnHeaders = function(oTableSizes) {
        var oDomRef = this.getDomRef();
        if (!oDomRef) {
            // _syncColumnHeaders gets called async, there might be no DomRef anymore
            return;
        }
        var $this = this.$();

        var oTableOptionsDiv = this.getDomRef('tableoptions');

        var aHeaderWidths = oTableSizes.headerWidths;
        var iFrozenColumns = this.getTotalFrozenColumnCount();
        var aVisibleColumns = this._getVisibleColumns();
        if (aVisibleColumns.length === 0) {
            return;
        }

        // Select only table headers (identified by data-sap-ui-headcolindex attribute). Not the row header.
        var $colHeaderContainer = $this.find(".sapUiTableColHdr");
        var $cols = $colHeaderContainer.find(".sapUiTableCol:not(.sapUiTableColGroup)");
        var $tableHeaders = $this.find(".sapUiTableCtrlFirstCol > th:not(.sapUiTableColSel)");
        this._aTableHeaders = $tableHeaders;

        // Create map with source table headers and their corresponding resizers.
        var mHeaders = {};

        // Traverse the source table headers, which are needed to determine the column head width
        $tableHeaders.each(function(iIndex, oElement) {
            var iHeadColIndex = oElement.getAttribute("data-sap-ui-headcolindex");
            var iHeaderWidth = aHeaderWidths[iIndex];

            // set width of target column div
            var iTargetWidth;
            var oVisibleColumn = aVisibleColumns[iIndex];
            if (oVisibleColumn) {
                iTargetWidth = iHeaderWidth;
            }

            // for the first column also calculate the width of the hidden column
            if (iIndex === 0 || iIndex === iFrozenColumns) {
                iTargetWidth += Math.max(0, oTableSizes.invisibleColWidth);
            }

            // apply the width of the column
            var vHeaderSpan = aVisibleColumns[iIndex] && aVisibleColumns[iIndex].getHeaderSpan ? aVisibleColumns[iIndex].getHeaderSpan() : 1,
                aHeaderData = [],
                aSpans;

            if (vHeaderSpan) {
                // vHeaderSpan can be an array for multi column header rows purpose.
                if (!jQuery.isArray(vHeaderSpan)) {
                    vHeaderSpan = [vHeaderSpan];
                }
                jQuery.each(vHeaderSpan, function(iSpanIndex, iSpan) {
                    vHeaderSpan[iSpanIndex] = Math.max(iSpan, 1);
                });
                aSpans = vHeaderSpan;
            } else {
                aSpans = [1];
            }

            for (var i = 0; i < aSpans.length; i++) {
                aHeaderData[i] = {
                    width: iTargetWidth,
                    span: 1
                };

                for (var j = 1; j < aSpans[i]; j++) {
                    var oHeader = $tableHeaders[iIndex + j];
                    if (oHeader) {
                        aHeaderData[i].width += aHeaderWidths[iIndex + j];
                        aHeaderData[i].span = aSpans[i];
                    }
                }
            }

            if (oVisibleColumn) {
                mHeaders[iHeadColIndex] = {
                    domRefColumnTh: oElement,
                    domRefColumnDivs: [],
                    aHeaderData: aHeaderData
                };
            }
        });

        var that = this;
        // Map target column header divs to corresponding source table header.
        $cols.each(function(iIndex, oElement) {
            var iColIndex = parseInt(oElement.getAttribute("data-sap-ui-colindex"), 10);
            if (!isNaN(iColIndex)) {
                var mHeader = mHeaders[iColIndex];
                if (mHeader) {
                    mHeader.domRefColumnDivs.push(oElement);
                } else {
                    jQuery.sap.log.error("Inconsistent DOM / Control Tree combination", that);
                }
            }
        });

        jQuery.each(mHeaders, function(iIndex, mHeader) {
            for (var i = 0; i < mHeader.domRefColumnDivs.length; i++) {
                var oHeaderData = mHeader.aHeaderData[0];
                if (mHeader.aHeaderData[i]) {
                    oHeaderData = mHeader.aHeaderData[i];
                }
                if (mHeader.domRefColumnDivs[i]) {
                    mHeader.domRefColumnDivs[i].setAttribute("data-sap-ui-colspan", oHeaderData.span);
                } else {
                    jQuery.sap.log.error("Inconsistent DOM / Control Tree combination", that);
                }
            }
        });

        // Table Column Height Calculation
        // we change the height of the cols, col header and the row header to auto to
        // find out whether to use the height of a cell or the min height of the col header.
        var bHasColHdrHeight = this.getColumnHeaderHeight() > 0;
        if (!bHasColHdrHeight) {
            // Height of one row within the header
            // avoid half pixels
            $cols.each(function(index, item) {
                item.style.height = oTableSizes.columnRowOuterHeight + "px";
            });


            if (oTableOptionsDiv !== null) {
                oTableOptionsDiv.style.height = oTableSizes.columnRowOuterHeight + "px";
            }

            // set height for only regular column header - not summary column header
            this._syncColumnHeaderHeight(oDomRef, oTableSizes.columnRowHeight);
        } else {
            if (oTableOptionsDiv !== null) {
                oTableOptionsDiv.style.height = this.getColumnHeaderHeight()  + "px";
            }
        }
    };

    /**
     * disables text selection on the document (disabled fro Dnd)
     * @private
     */
    Table.prototype._disableTextSelection = function (oElement) {
        // prevent text selection
        jQuery(oElement || document.body).
            attr("unselectable", "on").
            css({
                "-moz-user-select": "none",
                "-webkit-user-select": "none",
                "user-select": "none"
            }).
            bind("selectstart", function(oEvent) {
                oEvent.preventDefault();
                return false;
            });
    };

    /**
     * enables text selection on the document (disabled fro Dnd)
     * @private
     */
    Table.prototype._enableTextSelection = function (oElement) {
        jQuery(oElement || document.body).
            attr("unselectable", "off").
            css({
                "-moz-user-select": "",
                "-webkit-user-select": "",
                "user-select": ""
            }).
            unbind("selectstart");
    };

    /**
     * clears the text selection on the document (disabled fro Dnd)
     * @private
     */
    Table.prototype._clearTextSelection = function () {
        if (window.getSelection) {
            if (window.getSelection().empty) {  // Chrome
                window.getSelection().empty();
            } else if (window.getSelection().removeAllRanges) {  // Firefox
                window.getSelection().removeAllRanges();
            }
        } else if (document.selection && document.selection.empty) {  // IE?
            try {
                document.selection.empty();
            } catch (ex) {
                // ignore error to as a workaround for bug in IE8
            }
        }
    };

    /**
     * Create and return an instance of the columns dialog.
     * @private
     */
    Table.prototype._createColumnsDialog = function() {
        var oDialog,
            self = this,
            aFrozenColumns = [];

        logger.info("Table creating TableOptionsColumnsDialog.");

        oDialog = new TableOptionsColumnsDialog(this.getId() + '-dialog-columns', {
            beforeOpen: function (oEvent) {
                if (self.fireBeforeColumnsDialogOpen) {
                    self.fireBeforeColumnsDialogOpen();
                    //save off the list of previously visible frozen columns
                    aFrozenColumns = self._getVisibleColumns().slice(0, self.getFixedColumnCount());
                }
            },
            afterOpen: function (oEvent) {
                if (self.fireAfterColumnsDialogOpen) {
                    self.fireAfterColumnsDialogOpen();
                }
            },
            beforeClose: function (oEvent) {
                if (self.fireBeforeColumnsDialogClose) {
                    var oOrigin = oEvent.getParameter("origin"),
                        bOk = oOrigin ? oOrigin.getId().indexOf("-okButton") > 1 : false;

                    self.fireBeforeColumnsDialogClose({
                        origin: oOrigin,
                        acceptedChange: bOk
                    });

                    //if the column manager changes were accepted, then check for changes to the fixed column count
                    if (bOk) {
                        var l = aFrozenColumns.length,  //we don't need to look beyond this number of columns
                            aCols = oDialog.colData.columns,    //the list of columns coming from the manager dialog
                            oCol;   //a column we're interested in

                        // Loop over the first few columns and make sure the one from the manager was also in our previously
                        // saved list of frozen columns. If not, then the frozen section stops there.
                        for (var i=0; i<l && i<aCols.length; i++) {
                            oCol = sap.ui.getCore().byId(aCols[i].id);
                            if (aFrozenColumns.indexOf(oCol) < 0) {
                                self.setFixedColumnCount(self._getMandatoryColumnCount() + i);
                                break;
                            }
                        }
                    }
                }
            },
            afterClose: function (oEvent) {
                if (self.fireAfterColumnsDialogClose) {
                    var oOrigin = oEvent.getParameter("origin"),
                        bOk = oOrigin ? oOrigin.getId().indexOf("-okButton") > 1 : false;

                    self.fireAfterColumnsDialogClose({
                        origin: oOrigin,
                        acceptedChange: bOk
                    });
                }
            }
        });

        return oDialog;
    };

    /**
     * On the desktop, show the columns dialog.
     * @private
     */
    Table.prototype._showColumnsDialog = function () {
        var oDialog = this.getAggregation("tableOptionsColumnsDialog");

        if (!oDialog) {
            // delay creating until the first and only time
            oDialog = this._createColumnsDialog();
            this.setAggregation("tableOptionsColumnsDialog", oDialog);
        }

        logger.info("Table opening TableOptionsColumnsDialog.");
        oDialog.open();
    };

    Table.prototype._calculateHeaderColumnCount = function() {
        return this.retrieveLeafColumns().length;
    };

    /**
     * Find the lowest level columns displayed.
     * Different from getColumns when there are ColumnGroups.
     *
     * @returns {Array} An array of Columns
     */
    Table.prototype.retrieveLeafColumns = function() {
        if (this._aLeafCols) {
            return this._aLeafCols;
        }

        var aLeafCols = [],
            aColGroups = this.getColumns();

        aColGroups.forEach(function(oColumn) {
            oColumn.retrieveLeafColumns(aLeafCols);
        });

        this._aLeafCols = aLeafCols;
        return aLeafCols;
    };

    /**
     * Populate the nested array of headers to be rendered from column groups and columns.
     * @private
     */
    Table.prototype._populateRenderArray = function() {
        var iLeafCount = this._calculateHeaderColumnCount(),
            iHeaderRowCount = TableUtils.getHeaderRowCount(this),
            aRenderArray = new Array(iHeaderRowCount),
            iColIndex=0,
            self=this;

        // ColumnGroups are set in a hierarchical fashion, which has by design
        // more of a vertical nature.
        // Take the "vertical" column groups and create an array to be used for
        // rendering that will be done horizontally, row by row.

        this._aItemNavArray = new Array(iHeaderRowCount);

        // initialize arrays with all undefined cells
        // for rendering and for item navigation
        for (var h = 0; h < iHeaderRowCount; h++) {
            aRenderArray[h] = new Array(iLeafCount);
            this._aItemNavArray[h] = new Array(iLeafCount);
        }

        // populate the appropriate cells for each columnin the column group
        self.getColumns().forEach(function(oColumn) {
            aRenderArray = self._populateColumnArray(aRenderArray, iColIndex, 0, oColumn, self, iHeaderRowCount, iLeafCount);
            if (oColumn.calculateColumnSpan) {
                // this is a column group with columns under it, so
                // set the column index based on the number of columns underneath
                iColIndex = iColIndex + oColumn.calculateColumnSpan();
            } else {
                // this is a leaf node, so increment the column index by 1
                iColIndex++;
            }
        });

        // _populateColumnArray (that was just called in the loop above)
        // populates a separate array for navigation
        // which will contain indices that map
        // to indices in the simple navigation items array
        //
        // after the array is populated, we need to fill in any
        // remaining holes so we don't have issues with keyboard navigation
        var iTempIndex;
        for (var j=0; j<aRenderArray[0].length; j++) {
            for (var k=0; k<aRenderArray.length; k++) {
                if (this._aItemNavArray[k][j] !== undefined && this._aItemNavArray[k][j] !== null) {
                    // this is not an empty spot, so save this index for the next empty spot
                    iTempIndex = this._aItemNavArray[k][j];
                } else {
                    // this is an empty spot, so put the saved index here
                    this._aItemNavArray[k][j] = iTempIndex;
                }
            }
        }

        return aRenderArray;
    };

    /**
     * Populate the array of columns to be rendered from column groups and columns.
     * @private
     */
    Table.prototype._populateColumnArray = function(aRenderArray, iColIndex, iRowIndex, oColumnGroup, oParent, iHeaderRowCount, iLeafCount) {
        var self = this,
            iColSpanIndex;

        if (oColumnGroup.getColumns) {
            // this is a column group with columns
            if (aRenderArray.length > iRowIndex && aRenderArray[iRowIndex].length > iColIndex) {
                // add this column group to the render array
                aRenderArray[iRowIndex][iColIndex] = oColumnGroup;
                if (oColumnGroup.calculateColumnSpan) {
                    // find out how many columns are in this group
                    iColSpanIndex = iColIndex + oColumnGroup.calculateColumnSpan();
                    // for each column in this group, calculate the corresponding
                    // index to map to the item navigation simple array
                    for (var i=iColIndex; i<iColSpanIndex; i++) {
                        this._aItemNavArray[iRowIndex][i] = (iRowIndex*aRenderArray[iRowIndex].length)+iColIndex;
                    }
                }
            }
            // after the current row is handled, increment row index
            // and move to the lower rows
            iRowIndex++;
            oColumnGroup.getColumns().forEach(function(oColumn) {
                // recurse through each column group inside this column group
                self._populateColumnArray(aRenderArray, iColIndex, iRowIndex, oColumn, oColumnGroup, iHeaderRowCount, iLeafCount);
                // update the column index appropriately
                // based on whether this is a leaf column or column group
                if (oColumn.calculateColumnSpan) {
                    iColIndex = iColIndex + oColumn.calculateColumnSpan();
                } else {
                    iColIndex++;
                }
            });
        } else {
            // this is a leaf column
            if (aRenderArray.length > iRowIndex && aRenderArray[iRowIndex].length > iColIndex) {
                // add this column to the render array
                aRenderArray[iRowIndex][iColIndex] = oColumnGroup;
                // calculate and set the index to map to the item navigation array
                this._aItemNavArray[iRowIndex][iColIndex] = (iRowIndex*aRenderArray[iRowIndex].length)+iColIndex;
                iColIndex++;
            }
        }
        return aRenderArray;
    };

    /**
     * used to restrict disallowed features when column groups are present
     * @private
     */
    Table.prototype._restrictTableFeatures = function(bHasColumnGroups) {
        // TODO - for now, we only disable if column groups are present
        // in the future, we need to re-enable when all column groups are removed
        // but we will have to restore the previous state of each feature,
        // not just blindly set it to true

        if (bHasColumnGroups) {
            this.setShowColumnVisibilityMenu(false);
            this.setShowManageColumnsMenuItem(false);
            this.setEnableColumnReordering(false);
            this.setEnableColumnFreeze(false);

            var aMenuItems = this.getMenuItems(),
                iMenuItemsLength = aMenuItems.length,
                that = this;

            // if the column manager menu item is included in menu, remove it
            aMenuItems.forEach(function (oMenuItem) {
                if (oMenuItem && oMenuItem.getId() === (that.getId() + "-menuitem-columns")) {
                    that.removeMenuItem(oMenuItem, true);
                    oMenuItem.destroy();
                    if (iMenuItemsLength <= 1) {
                        that.setShowTableOptions(false);
                    }
                }
            });
        }
    };

    // =============================================================================
    // CONTROL EVENT HANDLING
    // =============================================================================

    /**
     * will be called by the vertical scrollbar. updates the visualized data by
     * applying the first visible (scrollpos) row from the vertical scrollbar
     * @private
     */
    Table.prototype.onvscroll = function(oEvent) {
        var that = this;

        // skip this scroll event callback if triggered from an action such as a keyboard pageup/pagedown
        if (this._skipVScroll === true) {
            this._skipVScroll = false;
            return;
        }

        // for interaction detection
        jQuery.sap.interaction.notifyScrollEvent && jQuery.sap.interaction.notifyScrollEvent(oEvent);
        // do not scroll in action mode!
        this._getKeyboardExtension().setActionMode(false);
        if (this._bLargeDataScrolling && !this._bIsScrolledByWheel) {
            window.clearTimeout(this._mTimeouts.scrollUpdateTimerId);
            this._mTimeouts.scrollUpdateTimerId = window.setTimeout(function() {
                that.setFirstVisibleRow(that._getFirstVisibleRowByScrollTop(), true);
                that._mTimeouts._sScrollUpdateTimerId = null;
            }, 300);
        } else {
            this.setFirstVisibleRow(this._getFirstVisibleRowByScrollTop(), true);
        }
        this._bIsScrolledByWheel = false;
    };

    /**
     * Handler for mousewheel event on scroll areas.
     * @private
     */
    Table.prototype._onMouseWheel = function(oEvent) {
        var oOriginalEvent = oEvent.originalEvent;
        //if ctrlKey is pressed, we should honor the browsers default behavior instead of scrolling the table. S1354826
        var bIsCtrlKey = oOriginalEvent.ctrlKey;
        if (bIsCtrlKey) {
            this._bIsScrolledByWheel = false;
            return;
        }

        var bIsHorizontal = oOriginalEvent.shiftKey;
        var iScrollDelta = 0;
        if (Device.browser.firefox) {
            iScrollDelta = oOriginalEvent.detail;
        } else {
            if (bIsHorizontal) {
                iScrollDelta = oOriginalEvent.deltaX;
            } else {
                iScrollDelta = oOriginalEvent.deltaY;
            }
        }

        if (bIsHorizontal) {
            var oHsb = this.getDomRef(SharedDomRef.HorizontalScrollBar);
            if (oHsb) {
                // match touch behavior on when to prevent browser page scrolling
                var iScrollLeft = oHsb.scrollLeft + iScrollDelta;
                if (iScrollLeft > 0 && iScrollLeft < (this.getDomRef("hsb-content").clientWidth - oHsb.clientWidth) - 1) {
                    oEvent.preventDefault();
                    oEvent.stopPropagation();
                }

                oHsb.scrollLeft = oHsb.scrollLeft + iScrollDelta;
            }
        } else {
            var oVsb = this.getDomRef(SharedDomRef.VerticalScrollBar);
            if (oVsb) {
                var iScrollTop = oVsb.scrollTop + iScrollDelta,
                    iVsbHeight = this.getDomRef("vsb-content").clientHeight,
                    iVHScrollMax = (this.getDomRef("vsb-content").clientHeight - oVsb.clientHeight) - 1;

                // IE has to be special
                if (Device.browser.msie) {
                    // subtract a small portion of the scroll wheel delta from max/min scroll
                    iVHScrollMax -= Math.floor(iScrollDelta * 0.05);
                }

                if (iVsbHeight > 0 && iScrollTop > 0 && iScrollTop < iVHScrollMax) {
                    oEvent.preventDefault();
                    oEvent.stopPropagation();
                }

                this._bIsScrolledByWheel = true;
                oVsb.scrollTop = oVsb.scrollTop + iScrollDelta;
            }
        }
    };

    /**
     * sync the column header and content
     * @private
     */
    Table.prototype._syncHeaderAndContent = function(oTableSizes) {
        if (!this._bSyncScrollLeft) {
            this._doSyncHeaderAndContent(oTableSizes.tableHSbScrollLeft);
        }
    };

    /**
     * @private
     */
    Table.prototype._doSyncHeaderAndContent = function(iHScrollPos) {
        // TODO Refactor this to scroll all horizontally scrollable table section to be inline
        // Could also consider making the function param a DOM element to have all the other
        // scrollable sections match that one's scroll position.
        this._bSyncScrollLeft = true;
        // synchronize the scroll areas
        var $this = this.$();
        $this.find(".sapUiTableColHdrScr").scrollLeft(iHScrollPos);
        $this.find(".sapUiTableCtrlScr").scrollLeft(iHScrollPos);
        this._bSyncScrollLeft = false;
    };

    /**
     * Will be called when the horizontal scrollbar is used. Since the table does
     * not render/update the data of all columns (only the visible ones) in case
     * of scrolling horizontally we need to update the content of the columns which
     * became visible.
     * @private
     */
    Table.prototype.onhscroll = function(oEvent) {
        jQuery.sap.interaction.notifyScrollEvent && jQuery.sap.interaction.notifyScrollEvent(oEvent);

        if (!this._bOnAfterRendering) {
            // The scroll event is fired multiple times from start to end of one horizontal scrolling action. The event is fired
            // the first time right after scrolling was started.
            // Scrolling is interrupted when syncing header and content and is therefore performed only over a short distance.
            // The timeout is used to overcome this issue, so syncing is performed only after the last occurrence of a scroll event.
            if (this._mTimeouts.hScrollUpdateTimer) {
                window.clearTimeout(this._mTimeouts.hScrollUpdateTimer);
            }
            this._mTimeouts.hScrollUpdateTimer = window.setTimeout(function() {
                var oTableSizes = this._collectTableSizes();
                this._syncHeaderAndContent(oTableSizes);
                this._determineVisibleCols(oTableSizes);
                this._captureHScroll();
            }.bind(this), 50);
        }
    };

    /**
     * @private
     */
    Table.prototype._captureHScroll = function() {
        var $hsb = this._getHsbRef();
        if ($hsb) {
            this._iHorizontalScrollPosition = $hsb.scrollLeft();
        }
    };

    /**
     * @private
     */
    Table.prototype._restoreHScroll = function() {
        var $hsb = this._getHsbRef();
        if ($hsb) {
            $hsb.scrollLeft(this._iHorizontalScrollPosition);
        }
    };

    /**
     * @private
     */
    Table.prototype._getHsbRef = function() {
        var $hsb = jQuery(this.getDomRef(SharedDomRef.HorizontalScrollBar));
        return $hsb;
    };

    /**
     * when navigating within the column header we need to synchronize the content
     * area with the position (scrollLeft) of the column header.
     * @private
     */
    Table.prototype._oncolscroll = function() {
        if (!this._bSyncScrollLeft) {
            var oHsb = this.getDomRef(SharedDomRef.HorizontalScrollBar);
            if (oHsb) {
                var oColHdrScr = this.getDomRef().querySelector(".sapUiTableColHdrScr");
                var iScrollLeft = 0;
                if (oColHdrScr) {
                    iScrollLeft = oColHdrScr.scrollLeft;
                }
                oHsb.scrollLeft = iScrollLeft;
            }
        }
    };

    /**
     * when navigating within the content area we need to synchronize the column
     * header with the position (scrollLeft) of the content area.
     * @private
     */
    Table.prototype._oncntscroll = function() {
        if (this._bRtlMode && jQuery.sap.touchEventMode === "ON") {
            // need comment to explain why to do this...
            return;
        }

        if (!this._bSyncScrollLeft) {
            var oHsb = this.getDomRef(SharedDomRef.HorizontalScrollBar);
            if (oHsb) {
                var oColHdrScr = this.getDomRef().querySelector(".sapUiTableCtrlScr");
                oHsb.scrollLeft = oColHdrScr.scrollLeft;
            }
        }
    };


    /**
     * listens to the mousedown events for starting column drag & drop. therefore
     * we wait 200ms to make sure it is no click on the column to open the menu.
     * Currently, this prevents the default drag-n-drop column resizing
     * from occurring when touch is present.
     * @private
     */
    Table.prototype.onmousedown = function(oEvent) {
        if (!this._bActionMode && this.getVisibleRowCountMode() === VisibleRowCountMode.Auto) {
            var $target = jQuery(oEvent.target),
                $currentRow = $target.closest(".sapUiTableTr"),
                iRowIndex = parseInt($currentRow.attr("data-sap-ui-rowindex"),10),
                iRowHeight,
                iTop,
                oIN = this._oItemNavigation,
                iAdjustedRowIndexToFocus = 0,
                aRows = this.getRows(),
                iTotalHeight = 0,
                $row,
                iIndexOfTarget,
                iIndexToFocus,
                i;

            // only on mousedown on rows, not the rest of the table
            if ($currentRow.length > 0) {
                iRowHeight = $currentRow.outerHeight(true);
                iTop = $currentRow.position().top;
                if ((iRowHeight + iTop) >= this._iAvailableSpace) {
                    // partial hidden row! move the focus...
                    for (i = 0; i < iRowIndex; i++) {
                        $row = aRows[i].$();
                        iTotalHeight += $row.outerHeight(true);

                        if (iRowHeight < iTotalHeight) {
                            break;
                        }
                        iAdjustedRowIndexToFocus++;
                    }

                    iAdjustedRowIndexToFocus++;

                    this.setFirstVisibleRow(this.getFirstVisibleRow() + iAdjustedRowIndexToFocus);
                    for (i = 0; i < oIN.aItemDomRefs.length;i++) {
                        var oItem = oIN.aItemDomRefs[i];
                        if (jQuery.sap.containsOrEquals(oItem,oEvent.target)) {
                            iIndexOfTarget = i;
                            iIndexToFocus = iIndexOfTarget - (oIN.iColumns * iAdjustedRowIndexToFocus);
                            oIN.focusItem(iIndexToFocus, oEvent);
                            oEvent.stopImmediatePropagation(true);
                            oEvent.preventDefault();
                            return;
                        }
                    }
                }
            }
        }


        // only move on left click!
        var bLeftButton = oEvent.button === 0;
        var bIsTouchMode = this._isTouchMode(oEvent);

        if (bLeftButton) {
            var $target = jQuery(oEvent.target);

            var $col = $target.closest(".sapUiTableCol");
            if ($col.length === 0) {
                // if we didn't find it going up the DOM tree, try going down
                // this is necessary for headers that span multiple rows
                $col = $target.find(".sapUiTableCol");
            }
            if ($col.length === 1 && oEvent.target !== this.getDomRef("sb")) {

                this._bShowMenu = true;
                this._mTimeouts.delayedMenuTimer = jQuery.sap.delayedCall(200, this, function() {
                    this._bShowMenu = false;
                });

                var bIsColumnMenuTarget = this._isTouchMode(oEvent) && ($target.hasClass("sapUiTableColDropDown") || $target.hasClass("sapUiTableColResizer"));
                if (this.getEnableColumnReordering() && !bIsColumnMenuTarget) {
                    var iIndex = parseInt($col.attr("data-sap-ui-colindex"), 10);
                    var oColumn = this.retrieveLeafColumns()[iIndex];

                    //S1228512: if column is frozen, don't allow moving
                    if (this._allowColumnMove(oColumn) === true) {
                        this._mTimeouts.delayedActionTimer = jQuery.sap.delayedCall(200, this, function() {
                            this._onColumnMoveStart(oColumn, bIsTouchMode);
                        });
                    }
                }
            }

            // in case of FireFox and CTRL+CLICK it selects the target TD
            //   => prevent the default behavior only in this case (to still allow text selection)
            var bCtrl = !!(oEvent.metaKey || oEvent.ctrlKey);
            if (!!Device.browser.firefox && bCtrl) {
                oEvent.preventDefault();
            }
        }

    };

    /**
     * S1246474: Conflicting context menu handling results in odd column menu positioning
     * override super.oncontextmenu to allow Table.customizeContextMenu/ContextMenuHandler to handle the event
     * @private
     */
    Table.prototype.oncontextmenu = function(oEvent) {
        jQuery(oEvent.target).closest("td.sasRightMouseDown").removeClass("sasRightMouseDown");
        return; //customizecontextmenu will handle this
    };

    /**
     * handles the default cell contextmenu
     * @private
     */
    Table.prototype._oncellcontextmenu = function(mParams) {
        if (this.getEnableCellFilter()) {
            // create the contextmenu instance the first time it is needed
            if (!this._oContextMenu) {
                if ( !Menu || !MenuItem ) {
                    // retrieve lazy dependencies
                    // TODO consider to load them async (should be possible as this method ends with an "open" call which is async by nature
                    Menu = sap.ui.requireSync("sap/ui/unified/Menu");
                    MenuItem = sap.ui.requireSync("sap/ui/unified/MenuItem");
                }
                this._oContextMenu = new Menu(this.getId() + "-contextmenu");
                this.addDependent(this._oContextMenu);
            }

            // does the column support filtering?
            var oColumn = sap.ui.getCore().byId(mParams.columnId);
            var sProperty;
            if (oColumn.getFilterProperty) {
                sProperty= oColumn.getFilterProperty();
            }
            // currently only filter is possible by default cell context menu, if filtering is not allowed by
            // menu, don't open the context menu at all.
            if (oColumn && oColumn.isFilterableByMenu && oColumn.isFilterableByMenu() && mParams.rowBindingContext) {
                // destroy all items of the menu and recreate
                this._oContextMenu.destroyItems();
                this._oContextMenu.addItem(new MenuItem({
                    text: this._oResBundle.getText("TBL_FILTER"),
                    select: [function() {
                        var oContext = this.getContextByIndex(mParams.rowIndex);
                        var sValue = oContext.getProperty(sProperty);
                        if (this.getEnableCustomFilter()) {
                            // only fire custom filter event
                            this.fireCustomFilter({
                                column: oColumn,
                                value: sValue
                            });
                        } else {
                            this.filter(oColumn, sValue);
                        }

                    }, this]
                }));

                // open the popup below the cell
                var eDock = Popup.Dock;
                this._oContextMenu.open(false, mParams.cellDomRef, eDock.BeginTop, eDock.BeginBottom, mParams.cellDomRef, "none none");
                return true;
            }
        }
    };

    /**
     * Given an event representing an event on a table cell, compute useful
     * parameters for firing various table event handlers
     * @param {object} oEvent The browser event
     * @private
     */
    Table.prototype._extractCellEventParams = function(oEvent) {

        var mParams, iRow, iCol, iRealRowIndex, sColId, oRow, oCell;
        var oRowBindingContext, oCellBindingContext, oModel, oRowBindingInfo;

        var $target = jQuery(oEvent.target);
        var $cell = $target.closest("td[role='gridcell']");
        var $row = $cell.closest("tr");
        var sId = $cell.attr("id");
        var aMatches = /.*-row(\d+)-col(\d+)/i.exec(sId);

        if (aMatches) {
            iRow = aMatches[1];
            iCol = aMatches[2];
            oRow = this.getRows()[iRow];
            oCell = oRow && oRow.getCells()[iCol];
            iRealRowIndex = oRow && oRow.getIndex();
            sColId = oCell && oCell.data("sap-ui-colid");
            oRowBindingInfo = this.getBindingInfo("rows");
            oModel = oRowBindingInfo && oRowBindingInfo.model;

            if (oRowBindingInfo) {
                oRowBindingContext = oRow.getBindingContext(oModel);
                oCellBindingContext = oCell.getBindingContext(oModel);
            }

            mParams = {
                rowIndex: iRealRowIndex,
                columnIndex: iCol,
                columnId: sColId,
                cellControl: oCell,
                rowBindingContext: oRowBindingContext,
                cellBindingContext: oCellBindingContext,
                cellDomRef: $cell[0],
                rowDomRef: $row[0]
            };
        }

        return mParams;
    };

    /**
     * finds the cell on which the click or contextmenu event is executed and
     * notifies the listener which control has been clicked or the contextmenu
     * should be openend.
     * @param {function} fnFire function to fire the event
     * @param {DOMEvent} oEvent event object
     * @return {boolean} cancelled or not
     * @private
     */
    Table.prototype._findAndfireCellEvent = function(fnFire, oEvent, fnContextMenu) {
        // find out which cell has been clicked
        var bCancel,
            mParams = this._extractCellEventParams(oEvent);

        if (mParams) {
            bCancel = !fnFire.call(this, mParams);
            if (!bCancel && typeof fnContextMenu === "function") {
                bCancel = fnContextMenu.call(this, mParams);
            }
        }
        return bCancel;
    };

    /**
     * @override
     */
    Table.prototype.getFocusDomRef = function() {
        if (this._bIsBeingDestroyed || this.bIsDestroyed) {
            return null;
        }

        this._getKeyboardExtension().initItemNavigation();
        // focus is handled by item navigation. It's  not the root element of the table which may get the focus but
        // the last focused column header or cell.
        return TableUtils.getFocusedItemInfo(this).domRef || Control.prototype.getFocusDomRef.apply(this, arguments);
    };

    Table.prototype.focusPreviousFocusableHeader = function(oEvent, iNewFocusedIndex, bShouldFocus) {
        var oIN = this._getItemNavigation(),
            oFocusedItem = TableUtils.getFocusedItemInfo(this),
            iNewFocusedIndex,
            iIndexToFocus,
            bColumnGroups = this.hasColumnGroups();

        if (!bColumnGroups) {
            if (bShouldFocus) {
                TableUtils.focusItem(this, iNewFocusedIndex, oEvent);
            }
            return;
        }

        // default to current focused index
        if (iNewFocusedIndex === undefined) {
            iNewFocusedIndex = oIN.getFocusedIndex();
        }
        // default to false
        if (bShouldFocus === undefined) {
            bShouldFocus = false;
        }

        iIndexToFocus = iNewFocusedIndex - oFocusedItem.columnCount;
        while (oIN.aItemDomRefs[iNewFocusedIndex] === oIN.aItemDomRefs[iIndexToFocus] && iIndexToFocus >= 0) {
            iIndexToFocus = iIndexToFocus - oFocusedItem.columnCount;
            bShouldFocus = true;
        }
        if (bShouldFocus) {
            iNewFocusedIndex = iIndexToFocus + oFocusedItem.columnCount;
        }

        iIndexToFocus = iNewFocusedIndex - 1;
        while (oIN.aItemDomRefs[iNewFocusedIndex] === oIN.aItemDomRefs[iIndexToFocus] && iIndexToFocus > 0) {
            iIndexToFocus = iIndexToFocus - 1;
            bShouldFocus = true;
        }
        if (bShouldFocus) {
            TableUtils.focusItem(this, iIndexToFocus + 1, oEvent);
        }
    };

    Table.prototype.focusNextFocusableHeader = function(oEvent, iNewFocusedIndex, bShouldFocus) {
        var oIN = this._getItemNavigation(),
            oFocusedItem = TableUtils.getFocusedItemInfo(this),
            iNewFocusedIndex,
            iIndexToFocus,
            bColumnGroups = this.hasColumnGroups();

        if (!bColumnGroups) {
            if (bShouldFocus) {
                TableUtils.focusItem(this, iNewFocusedIndex, oEvent);
            }
            return;
        }

        // default to current focused index
        if (iNewFocusedIndex === undefined) {
            iNewFocusedIndex = oIN.getFocusedIndex();
        }
        // default to false
        if (bShouldFocus === undefined) {
            bShouldFocus = false;
        }

        iNewFocusedIndex = oIN.getFocusedIndex();
        iIndexToFocus = iNewFocusedIndex;

        // go down
        if (oEvent.type === "sapdown") {
            while (oEvent.target === oIN.aItemDomRefs[iIndexToFocus] && iIndexToFocus < oIN.aItemDomRefs.length) {
                iIndexToFocus = iIndexToFocus + oFocusedItem.columnCount;
                bShouldFocus = true;
            }
        }

        // go over
        if (oEvent.type === "sapnext") {
            while (oEvent.target === oIN.aItemDomRefs[iIndexToFocus] && iIndexToFocus < (oFocusedItem.columnCount*(oFocusedItem.row+1)-1)) {
                iIndexToFocus = iIndexToFocus + 1;
                bShouldFocus = true;
            }
        }

        // now make sure we're on the focusable cell
        // which is the first one with this id
        this.focusPreviousFocusableHeader(oEvent, iIndexToFocus, bShouldFocus);
    };

    /**
     * handles the focus in to reposition the focus or prevent default handling for
     * column resizing
     * @private
     */
    Table.prototype.onfocusin = function(oEvent) {
        var $ctrlScr;
        var $FocusedDomRef = jQuery(oEvent.target);
        if ($FocusedDomRef.parent('.sapUiTableTr').length > 0) {
            $ctrlScr = jQuery(this.getDomRef("sapUiTableCtrlScr"));
        } else if ($FocusedDomRef.parent('.sapUiTableColHeader').length > 0) {
            $ctrlScr = jQuery(this.getDomRef("sapUiTableColHdrScr"));
        }

        var oTargetedColumn = this._colFromRef(oEvent.target);
        if ((Device.browser.firefox || Device.browser.chrome || Device.browser.edge || Device.browser.safari) && $ctrlScr && $ctrlScr.length > 0 && this.isFrozenColumn(oTargetedColumn) === false) {

            var iCtrlScrScrollLeft = $ctrlScr.scrollLeft();
            var iCtrlScrWidth = $ctrlScr.width();
            //https://github.com/w3c/csswg-drafts/issues/1354 browsers implement RTL scrolling differently from each other
            if ((Device.browser.firefox || Device.browser.safari) && this._bRtlMode) {
                //S1346723: MS Edge, Firefox, and Safari report the scrollLeft position differently from everyone else in RTL
                // $ctrlScr.scrollLeft() is negative distance between the most right and the right of scrollbar instead of
                // the positive distance between the most left and the left of scrollbar
                iCtrlScrScrollLeft = $ctrlScr[0].scrollWidth - iCtrlScrWidth + iCtrlScrScrollLeft;
            } else if (Device.browser.edge && this._bRtlMode) {
                // $ctrlScr.scrollLeft() is positive distance between the most right and the right of scrollbar
                iCtrlScrScrollLeft = $ctrlScr[0].scrollWidth - iCtrlScrWidth - iCtrlScrScrollLeft;
            }

            var iCellLeft = $FocusedDomRef.position().left;
            var iCellRight = iCellLeft + $FocusedDomRef.width();
            var iOffsetLeft = iCellLeft - iCtrlScrScrollLeft;
            var iOffsetRight = iCellRight - iCtrlScrWidth - iCtrlScrScrollLeft;

            var oHsb = this.getDomRef(SharedDomRef.HorizontalScrollBar);
            if ((iOffsetRight > 0 && iOffsetLeft > 0) || (iOffsetLeft < 0 && iOffsetRight < 0)) {
                var iNewScrollLeft = iOffsetRight > 0 ? iCtrlScrScrollLeft + iOffsetRight + 2 : iCtrlScrScrollLeft + iOffsetLeft - 1;
                if ((Device.browser.firefox || Device.browser.safari) && this._bRtlMode) {
                    //In firefox and safari, oHsb.scrollLeft must be negative distance between the most right and the right of scrollbar in RTL mode
                    oHsb.scrollLeft = iNewScrollLeft + iCtrlScrWidth - $ctrlScr[0].scrollWidth;
                } else if (Device.browser.edge && this._bRtlMode) {
                    //In edge, oHsb.scrollLeft must be positive distance between the most right and the right of scrollbar in RTL mode
                    oHsb.scrollLeft = $ctrlScr[0].scrollWidth - iNewScrollLeft - iCtrlScrWidth;
                } else {
                    oHsb.scrollLeft = iNewScrollLeft;
                }
            }
        }

        this.$().addClass("sasContainsFocusedItem");
    };

    /**
     * @private
     */
    Table.prototype.onfocusout = function(oEvent) {
        this.$().removeClass("sasContainsFocusedItem");
    };

    /**
     * The row index only considers the position of the row in the aggregation. It must be adapted
     * to consider the firstVisibleRow offset or if a fixed bottom row was pressed
     * @param {int} iRow row index of the control in the rows aggregation
     * @returns {int} the adapted (absolute) row index
     * @private
     */
    Table.prototype._getAbsoluteRowIndex = function(iRow) {
        var iIndex = 0;
        var iFirstVisibleRow = this.getFirstVisibleRow();
        var iFixedBottomRowCount = this.getFixedBottomRowCount();
        var iVisibleRowCount = this.getVisibleRowCount();
        var iFirstFixedBottomRowIndex = iVisibleRowCount - iFixedBottomRowCount;

        if (iFixedBottomRowCount > 0 && iRow >= iFirstFixedBottomRowIndex) {
            iIndex = this.getBinding().getLength() - iVisibleRowCount + iRow;
        } else {
            iIndex = iFirstVisibleRow + iRow;
        }

        return iIndex;
    };

    // =============================================================================
    // SELECTION HANDLING
    // =============================================================================

    /**
     * handles the row selection and the column header menu
     * @private
     */
    Table.prototype._onSelect = function(oEvent) {
        function isRowHeaderSelection() {
            var $row = jQuery(oEvent.target).closest(".sapUiTableRowHdr");
            return $row.length === 1;
        }

        if (this._useSelectionControls() === true && isRowHeaderSelection() === true) {
            // NO-OP since we want the selection to come from the checkbox in this case
            return;
        }

        // trigger column menu
        var $target = jQuery(oEvent.target);

        // determine modifier keys
        var bShift = oEvent.shiftKey;
        var bCtrl = !!(oEvent.metaKey || oEvent.ctrlKey);

        // column header?
        var $col = $target.closest(".sapUiTableCol");
        if ($col.length === 0) {
            // if we didn't find it going up the DOM tree, try going down
            // this is necessary for headers that span multiple rows
            $col = $target.find(".sapUiTableCol");
        }
        if (this._bShowMenu && $col.length === 1) {
            var iIndex = parseInt($col.attr("data-sap-ui-colindex"), 10);
            var oColumn = this.retrieveLeafColumns()[iIndex];

            if ($target.hasClass("sapUiTableColDropDown")) {
                var bExecuteDefault = this.fireColumnSelect({
                    column: oColumn
                });

                if (bExecuteDefault && oColumn._openMenu) {
                    oColumn._openMenu($col[0], oEvent.type === "keyup");
                }
            } else {
                this._onColumnSelect(oColumn, $col[0], this._isTouchMode(oEvent), oEvent.type === "keyup");
            }

            return;
        }

        // row header?
        var $row = $target.closest(".sapUiTableRowHdr");
        if ($row.length === 1) {
            var iIndex = parseInt($row.attr("data-sap-ui-rowindex"), 10);
            this._onRowSelect(this._getAbsoluteRowIndex(iIndex), bShift, bCtrl);
            return;
        }

        // table control? (only if the selection behavior is set to row)
        var oClosestTd;
        if (oEvent.target) {
            var $ClosestTd = jQuery(oEvent.target).closest("td");
            if ($ClosestTd.length > 0) {
                oClosestTd = $ClosestTd[0];
            }
        }

        if (oClosestTd && (oClosestTd.getAttribute("role") === "gridcell" || jQuery(oClosestTd).hasClass("sapUiTableTDDummy"))
            && TableUtils.isRowSelectionAllowed(this)) {
            var $row = $target.closest(".sapUiTableCtrl > tbody > tr");
            if ($row.length === 1) {
                var iIndex = parseInt($row.attr("data-sap-ui-rowindex"), 10);
                this._onRowSelect(this._getAbsoluteRowIndex(iIndex), bShift, bCtrl);
                return;
            }
        }

        // select all?
        if (jQuery.sap.containsOrEquals(this.getDomRef("selall"), oEvent.target)) {
            this._toggleSelectAll();
            return;
        }

    };


    // =============================================================================
    // ROW EVENT HANDLING
    // =============================================================================

    /**
     * handles the row selection (depending on the mode)
     * @private
     */
    Table.prototype._onRowSelect = function(iRowIndex, bShift, bCtrl) {

        // in case of SHIFT, we clear the text selection
        if (bShift) {
            this._clearTextSelection();
        }

        // is the table bound?
        var oBinding = this.getBinding("rows");
        if (!oBinding) {
            return;
        }

        //var iRowIndex = Math.min(Math.max(0, iRowIndex), this.getBinding("rows").getLength() - 1);
        if (iRowIndex < 0 || iRowIndex >= (oBinding.getLength() || 0)) {
            return;
        }

        this._iSourceRowIndex = iRowIndex;

        var oSelMode = this.getSelectionMode();
        if (oSelMode !== SelectionMode.None) {
            if (oSelMode === SelectionMode.Single) {
                if (!this.isIndexSelected(iRowIndex)) {
                    this.setSelectedIndex(iRowIndex);
                } else {
                    this.clearSelection("clearAll");
                }
            } else {
                // in case of multi toggle behavior a click on the row selection
                // header adds or removes the selected row and the previous seleciton
                // will not be removed
                if (oSelMode === SelectionMode.MultiToggle || oSelMode === SelectionMode.ParentChild) {
                    bCtrl = true;
                }
                if (bShift) {
                    // If no row is selected getSelectedIndex returns -1 - then we simply
                    // select the clicked row:
                    var iSelectedIndex = this.getSelectedIndex();
                    if (iSelectedIndex >= 0) {
                        this.addSelectionInterval(iSelectedIndex, iRowIndex);
                    } else {
                        this.setSelectedIndex(iRowIndex);
                    }
                } else {
                    if (!this.isIndexSelected(iRowIndex)) {
                        if (bCtrl) {
                            this.addSelectionInterval(iRowIndex, iRowIndex);
                        } else {
                            this.setSelectedIndex(iRowIndex);
                        }
                    } else {
                        if (bCtrl) {
                            this.removeSelectionInterval(iRowIndex, iRowIndex);
                        } else {
                            if (this._getSelectedIndicesCount() === 1) {
                                this.clearSelection();
                            } else {
                                this.setSelectedIndex(iRowIndex);
                            }
                        }
                    }
                }
            }
        }

        //announce the possibly updated row state
        this._getAccExtension().announceCellNavigation();
        this._iSourceRowIndex = undefined;

    };


    // =============================================================================
    // COLUMN EVENT HANDLING
    // =============================================================================

    /**
     * column select event => opens the column menu
     * @private
     */
    Table.prototype._onColumnSelect = function(oColumn, oDomRef, bIsTouchMode, bWithKeyboard) {
        // On tablet open special column header menu
        if (bIsTouchMode && ((oColumn.getResizable && oColumn.getResizable()) || (oColumn.getResizable && oColumn._menuHasItems()))) {
            var $ColumnHeader = jQuery(oDomRef);
            var $ColumnCell = $ColumnHeader.find(".sapUiTableColCell");

            if ($ColumnHeader.find(".sapUiTableColCellMenu").length < 1) {
                $ColumnCell.hide();

                var sColumnDropDownButton = "";
                if (oColumn._menuHasItems()) {
                    sColumnDropDownButton = "<div class='sapUiTableColDropDown'></div>"; // Used to contain hex code E170 as the sole <div> contents.
                }

                var sColumnResizerButton = "";
                if (oColumn.getResizable()) {
                    sColumnResizerButton = "<div class='sapUiTableColResizer''></div>"; // Used to contain hex code E209 as the sole <div> contents.
                }

                var $ColumnHeaderMenu = jQuery("<div class='sapUiTableColCellMenu'>" + sColumnDropDownButton + sColumnResizerButton + "</div>");
                $ColumnHeader.append($ColumnHeaderMenu);
                $ColumnHeader.bind("focusout", function() {
                    this.cell.show();
                    this.menu.remove();
                    this.self.unbind("focusout");
                }.bind({
                    cell: $ColumnCell,
                    menu: $ColumnHeaderMenu,
                    self: $ColumnHeader
                }));
            }

            return;
        }

        // forward the event
        var bExecuteDefault = this.fireColumnSelect({
            column: oColumn
        });

        // if the default behavior should be prevented we suppress to open
        // the column menu!

        if (bExecuteDefault && this._openMenuOnColumnSelect !== true && oColumn._openMenu) {
            oColumn._openMenu(oDomRef, bWithKeyboard);
        }
    };

    /**
     * Handler for touchstart on the table, needed for scrolling.
     * @param oEvent
     */
    Table.prototype.ontouchstart = function(oEvent) {
        if ('ontouchstart' in document) {
            this._aTouchStartPosition = null;
            this._bIsScrollVertical = null;
            var $scrollTargets = this._getScrollTargets();
            var bDoScroll = jQuery(oEvent.target).closest($scrollTargets).length > 0;
            if (bDoScroll) {
                var oTouch = oEvent.targetTouches[0];
                this._aTouchStartPosition = [oTouch.pageX, oTouch.pageY];
                var oVsb = this.getDomRef(SharedDomRef.VerticalScrollBar);
                if (oVsb) {
                    this._iTouchScrollTop = oVsb.scrollTop;
                }

                var oHsb = this.getDomRef(SharedDomRef.HorizontalScrollBar);
                if (oHsb) {
                    this._iTouchScrollLeft = oHsb.scrollLeft;
                }
            }
        }
    };

    /**
     * Handler for touchmove on the table, needed for scrolling.
     * @param oEvent
     */
    Table.prototype.ontouchmove = function(oEvent) {
        if (this._bIsColumnResizerMoving) {
            //if the column resizer is moving, then we should not handle this touchmove event as a scroll of the table content area
            return;
        }

        if ('ontouchstart' in document && this._aTouchStartPosition) {
            var oTouch = oEvent.targetTouches[0];
            var iDeltaX = (oTouch.pageX - this._aTouchStartPosition[0]);
            var iDeltaY = (oTouch.pageY - this._aTouchStartPosition[1]);
            if (this._bIsScrollVertical === null) {
                this._bIsScrollVertical = Math.abs(iDeltaY) > Math.abs(iDeltaX);
            }

            if (this._bIsScrollVertical) {
                var oVsb = this.getDomRef(SharedDomRef.VerticalScrollBar);
                if (oVsb) {
                    var iScrollTop = this._iTouchScrollTop - iDeltaY;

                    if (iScrollTop > 0 && iScrollTop < (this.getDomRef("vsb-content").clientHeight - oVsb.clientHeight) - 1) {
                        oEvent.preventDefault();
                        oEvent.stopPropagation();
                    }
                    oVsb.scrollTop = iScrollTop;
                }
            } else {
                var oHsb = this.getDomRef(SharedDomRef.HorizontalScrollBar);
                if (oHsb) {
                    var iScrollLeft = this._iTouchScrollLeft - iDeltaX;

                    if (iScrollLeft > 0 && iScrollLeft < (this.getDomRef("hsb-content").clientWidth - oHsb.clientWidth) - 1) {
                        oEvent.preventDefault();
                        oEvent.stopPropagation();
                    }
                    oHsb.scrollLeft = iScrollLeft;
                }
            }
        }
    };


    /**
     * start column moving
     * Turn off highlighting when a column move starts.
     * @private
     */
    Table.prototype._onColumnMoveStart = function(oColumn, bIsTouchMode) {
        this.removeHighlighting();
        this._disableTextSelection();

        var $col = this._getColumnDomRefForMove(oColumn);

        var iColIndex = parseInt($col.attr("data-sap-ui-colindex"), 10);

        if (iColIndex < this.getTotalFrozenColumnCount()) {
            return;
        }

        this.$().addClass("sapUiTableDragDrop");
        this._$colGhost = $col.clone().removeAttr("id");

        $col.css({
            "opacity": ".25"
        });

        this._$colGhost.addClass("sapUiTableColGhost").css({
            "left": -10000,
            "top": -10000,
            //Position is set to relative for columns later, if the moving is started a second time the position: relative overwrites
            //the absolut position set by the sapUiTableColGhost class, so we overrite the style attribute for position here to make
            //sure that the position is absolute
            "position": "absolute",
            "z-index": this.$().zIndex() + 10
        });

        // TODO: only for the visible columns!?
        this.$().find(".sapUiTableCol").each(function(iIndex, oElement) {

            var $col = jQuery(this);
            $col.css({position: "relative"});

            $col.data("pos", {
                left: $col.offset().left,
                center: $col.offset().left + $col.outerWidth() / 2,
                right:  $col.offset().left + $col.outerWidth()
            });

        });

        this._$colGhost.appendTo(document.body);

        var $doc = jQuery(document);
        if (bIsTouchMode) {
            $doc.bind("touchmove.sapUiColumnMove", jQuery.proxy(this._onColumnMove, this));
            $doc.bind("touchend.sapUiColumnMove", jQuery.proxy(this._onColumnMoved, this));
        } else {
            $doc.bind("mousemove.sapUiColumnMove", jQuery.proxy(this._onColumnMove, this));
            $doc.bind("mouseup.sapUiColumnMove", jQuery.proxy(this._onColumnMoved, this));
        }
    };

    /**
     * move the column position the ghost
     * Will nudge horizontal scrollbar when mouse cursor crosses nudge boundary
     * Initiates and helps manage the "nudge until" action
     * @private
     */
    Table.prototype._onColumnMove = function(oEvent) {
        this._clearHorizontalNudgeTimeout();

        var oPointerExtension = this._getPointerExtension(),
            oEventPosition = oPointerExtension.getExtensionHelper().getEventPosition(oEvent, this),
            iLocationX = oEventPosition.pageX,
            iLocationY = oEventPosition.pageY,
            iOldColPos = this._iNewColPos;

        oEvent.preventDefault(); // Avoid default actions e.g. scrolling on mobile devices

        // find out the new col position
        var oPos = oPointerExtension.getReorderHelper().findColumnForPosition(this, iLocationX);

        if (!oPos || !oPos.id) {
            //Special handling for dummy column (in case the other columns does not occupy the whole space),
            //row selectors and row actions
            this._iNewColPos = iOldColPos;
            return;
        }

        var iDnDColIndex = parseInt(this._$colGhost.attr("data-sap-ui-colindex"), 10);

        if (oPos.before || (oPos.after && oPos.index === iDnDColIndex)) {
            this._iNewColPos = oPos.index;
        } else if (oPos.after && oPos.index !== iDnDColIndex) {
            this._iNewColPos = oPos.index + 1;
        }

        var oTargetColumn = this.retrieveLeafColumns()[this._iNewColPos];
        if (!this._allowColumnMove(oTargetColumn)) {
            this._iNewColPos = iOldColPos;
        }

        //clamp the new column position to no farther than max column index
        if (oPos.after) {
            this._iNewColPos = Math.min(this._iNewColPos, this.retrieveLeafColumns().length-1);
        }

        // animate the column move
        this._animateColumnMove(iDnDColIndex, iOldColPos, this._iNewColPos);

        // update the ghost position
        this._$colGhost.css({
            "left": iLocationX + 5,
            "top": iLocationY + 5
        });


        var $scrollArea = this.$("sapUiTableCtrlScr"),
            // x-coords of the thresholds where scroll-nudging could occur
            iEndNudgeThresholdXCoord = $scrollArea.offset().left + $scrollArea.outerWidth() - this._iHorizontalNudgeThreshold,
            iBeginningNudgeThresholdXCoord = $scrollArea.offset().left + this._iHorizontalNudgeThreshold;

        // is the cursor within the _end_ nudge threshold with room still to scroll?
        function shouldNudgeForward() {
            return iLocationX > iEndNudgeThresholdXCoord;
        }
        // is the cursor within the _beginning_ nudge threshold with room still to scroll?
        function shouldNudgeBackward() {
            return iLocationX < iBeginningNudgeThresholdXCoord;
        }

        // Determine if a scroll-nudge is needed.
        // The timeout can be cancelled by:
        // - the "should nudge" logic in this function
        // - mousemove event (see top of _onColumnMove)
        // - mouseup event (see top of _onColumnMoved)
        if (shouldNudgeForward()) {
            this._horizontalScrollNudgeForwardsUntil(shouldNudgeForward);
        } else if (shouldNudgeBackward()) {
            this._horizontalScrollNudgeBackwardsUntil(shouldNudgeBackward);
        }
    };

    /**
     * Keep nudging the horizontal scrollbar until a condition is met
     * @private
     * @param {boolean} bForward the direction to scroll
     * @param {function} fnCondition determines if nudging should continue
     * @returns {object} A promise that is resolved when the operation is complete
     */
    Table.prototype._horizontalScrollNudgeUntil = function(bForward, fnCondition) {
        var self = this;
        return new Promise(function(resolve) {
            function resolveAndClear() {
                self._clearHorizontalNudgeTimeout();
                resolve();
            }

            if (self._iScrollNudgeTimeoutId !== undefined) {
                logger.debug("nudge operation already running");
                resolve();
                return;
            }

            (function nudge(bFirstNudge) {
                self._horizontalScrollNudge(bForward);

                // the timeout was cancelled from somewhere else
                // can't check the first time because the nudge process hasn't
                // event started yet for it to have already been cancelled from elsewhere
                if (bFirstNudge === false && self._iScrollNudgeTimeoutId === undefined) {
                    resolveAndClear();
                    return;
                }

                self._iScrollNudgeTimeoutId = self._setManagedTimeout(function() {
                    // if the scroll bar reaches either the full maximum or minimum of the Table stop nudging
                    if (self._isHorizontallyScrolledMaximally() === true || self._isHorizontallyScrolledMinimally() === true) {
                        resolveAndClear();
                        return;
                    }

                    // check if the user-provided condition function
                    // calls for continuing or stopping nudging
                    if (fnCondition() === true) {
                        nudge(false);
                    } else {
                        resolveAndClear();
                    }
                }, self._iHorizontalNudgeRepeatDelay);
            }(true));
        });
    };

    /**
     * Clear the setTimeout id for the "horizontal scroll nudge" operation
     * @private
     */
    Table.prototype._clearHorizontalNudgeTimeout = function() {
        this._clearTimeout(this._iScrollNudgeTimeoutId);
        this._iScrollNudgeTimeoutId = undefined;
    };

    /**
     * Keep nudging the horizontal scrollbar forwards until a condition is met
     * @private
     * @param {function} fnCondition determines if nudging should continue
     * @returns {object} A promise that is resolved when the operation is complete
     */
    Table.prototype._horizontalScrollNudgeForwardsUntil = function(fnCondition) {
        return this._horizontalScrollNudgeUntil(true, fnCondition);
    };

    /**
     * Keep nudging the horizontal scrollbar backwards until a condition is met
     * @private
     * @param {function} fnCondition determines if nudging should continue
     * @returns {object} A promise that is resolved when the operation is complete
     */
    Table.prototype._horizontalScrollNudgeBackwardsUntil = function(fnCondition) {
        return this._horizontalScrollNudgeUntil(false, fnCondition);
    };

    /**
     * animates the column movement
     * @private
     */
    Table.prototype._animateColumnMove = function(iColIndex, iOldPos, iNewPos) {

        var bRtl = this._bRtlMode;

        var $DnDCol = this._getColumnDomRefForMove(this.retrieveLeafColumns()[iColIndex]);

        // position has been changed => reorder
        if (iOldPos !== iNewPos) {

            for (var i = Math.min(iOldPos, iNewPos), l = Math.max(iOldPos, iNewPos); i <= l; i++) {
                var oCol = this.retrieveLeafColumns()[i];
                if (i !== iColIndex && oCol.getVisible()) {
                    this._getColumnDomRefForMove(oCol).stop(true, true).animate({left: "0px"});
                }
            }

            var iOffsetLeft = 0;
            if (iNewPos < iColIndex) {
                for (var i = iNewPos; i < iColIndex; i++) {
                    var oCol = this.retrieveLeafColumns()[i];
                    if (oCol && oCol.getVisible()) {

                        var $col = this._getColumnDomRefForMove(oCol);

                        iOffsetLeft -= $col.outerWidth();
                        $col.stop(true, true).animate({left: $DnDCol.outerWidth() * (bRtl ? -1 : 1) + "px"});
                    }
                }
            } else {
                for (var i = iColIndex + 1, l = iNewPos + 1; i < l; i++) {
                    var oCol = this.retrieveLeafColumns()[i];
                    if (oCol && oCol.getVisible()) {

                        var $col = this._getColumnDomRefForMove(oCol);

                        iOffsetLeft += $col.outerWidth();
                        $col.stop(true, true).animate({left: $DnDCol.outerWidth() * (bRtl ? 1 : -1) + "px"});
                    }
                }
            }
            $DnDCol.stop(true, true).animate({left: iOffsetLeft * (bRtl ? -1 : 1) + "px"});
        }

    };

    /**
     * columns is moved => update!
     * Cancels any in-progress horizontal scroll nudging
     * @private
     */
    Table.prototype._onColumnMoved = function(oEvent) {
        var that = this;
        this._clearHorizontalNudgeTimeout();
        this.$().removeClass("sapUiTableDragDrop");

        var iDnDColIndex = parseInt(this._$colGhost.attr("data-sap-ui-colindex"), 10);
        var oDnDCol = this.retrieveLeafColumns()[iDnDColIndex];

        // Save horizontal scroll position
        var oHsb = this.getDomRef(SharedDomRef.HorizontalScrollBar);
        var $hsb = jQuery(oHsb);
        var iHScrollPos = oHsb ? $hsb.scrollLeft() : 0;

        var $doc = jQuery(document);
        $doc.unbind("touchmove.sapUiColumnMove");
        $doc.unbind("touchend.sapUiColumnMove");
        $doc.unbind("mousemove.sapUiColumnMove");
        $doc.unbind("mouseup.sapUiColumnMove");

        this._$colGhost.remove();
        this._$colGhost = undefined;

        this._enableTextSelection();

        // forward the event
        var bExecuteDefault = this.fireColumnMove({
            column: oDnDCol,
            newPos: this._iNewColPos
        });

        var bMoveRight = iDnDColIndex < this._iNewColPos;

        if (bExecuteDefault
                && this._iNewColPos !== undefined
                && !isNaN(this._iNewColPos)
                && this._iNewColPos !== iDnDColIndex) {
            this.removeColumn(oDnDCol);
            this.insertColumn(oDnDCol, this._iNewColPos);
            // if columns are bound, we need to reset the row template
            // because we did not do that in column insert or delete
            if (this.isBound("columns")) {
                this._resetRowTemplate();
            }
            var vHeaderSpan = 1,
                iSpan;

            if (oDnDCol.getHeaderSpan) {
                vHeaderSpan = oDnDCol.getHeaderSpan();
            }

            if (vHeaderSpan) {
                if (jQuery.isArray(vHeaderSpan)) {
                    iSpan = vHeaderSpan[0];
                } else {
                    iSpan = vHeaderSpan;
                }
            } else {
                iSpan = 1;
            }

            if (iSpan > 1) {
                if (!bMoveRight) {
                    this._iNewColPos++;
                }
                for (var i = 1; i < iSpan; i++) {
                    var oDependentCol = this.retrieveLeafColumns()[bMoveRight ? iDnDColIndex : iDnDColIndex + i];
                    this.removeColumn(oDependentCol);
                    this.insertColumn(oDependentCol, this._iNewColPos);
                    this.fireColumnMove({
                        column: oDependentCol,
                        newPos: this._iNewColPos
                    });
                    if (!bMoveRight) {
                        this._iNewColPos++;
                    }
                }
            }
        } else {
            this._animateColumnMove(iDnDColIndex, this._iNewColPos, iDnDColIndex);

            this._getColumnDomRefForMove(oDnDCol).css({
                "backgroundColor": "",
                "backgroundImage": "",
                "opacity": ""
            });
        }

        // Re-apply focus
        if (this._mTimeouts.reApplyFocusTimer) {
            window.clearTimeout(this._mTimeouts.reApplyFocusTimer);
        }
        this._mTimeouts.reApplyFocusTimer = window.setTimeout(function() {
            var iOldFocusedIndex = TableUtils.getFocusedItemInfo(that).cell;
            TableUtils.focusItem(that, 0, oEvent);
            TableUtils.focusItem(that, iOldFocusedIndex, oEvent);
        }, 0);

        delete this._iNewColPos;

        // Restore horizontal scroll position
        if (oHsb && iHScrollPos) {
            var fnRestoreHScrollPos = function() {
                var oHsb = that.getDomRef(SharedDomRef.HorizontalScrollBar);
                // Since this timeout doesn't get cleaned up like others,
                // make sure HSB still exists before doing anything.
                if (oHsb) {
                    var $hsb = jQuery(oHsb);
                    $hsb.scrollLeft(iHScrollPos);
                    that._doSyncHeaderAndContent(iHScrollPos);
                }
            };

            // Cannot use this._mTimeouts since those get cleaned up before rendering.
            window.setTimeout(fnRestoreHScrollPos, 100);
        }
    };

    /**
     *
     * @param oColumn
     * @param sWidth
     * @private
     */
    Table.prototype._resizeDependentColumns = function(oColumn, sWidth) {
        if (!oColumn || oColumn instanceof ColumnGroup) {
            return;
        }

        logger.debug("Resizing dependent columns", oColumn.getIndex() + " -> " + sWidth );

        var that = this;
        var aColumns = this.retrieveLeafColumns();
        var aVisibleColumns = this._getVisibleColumns();
        var iColumnIndex = aColumns.indexOf(oColumn);
        var iFrozenColumnCount = this.getTotalFrozenColumnCount();
        var bIsFrozenColumn = this.isFrozenColumn(oColumn);
        var oDomRef = this.getDomRef();

        // if we got here while normalizing column widths, return
        // to avoid an infinite rendering loop
        if (this._bNormalizingColumnWidths) {
            return;
        }

        if (bIsFrozenColumn) {
            // Fixed columns should only have px widths and must also change the size of fixed column container.
            var iNewFrozenSectionWidth = this._getColumnsWidth(0, iFrozenColumnCount);
            var $frozenSections = jQuery(oDomRef).find(".sapUiTableCtrlFixed");
            $frozenSections.css({
                "min-width": iNewFrozenSectionWidth+"px",
                "width": iNewFrozenSectionWidth+"px"
            });

            return;
        }

        // Adjust columns only if the columns have percentage values
        if (this._hasSomePercentageWidthColumn()) {
            // Not sure why this section uses visible columns instead of all.
            iColumnIndex = aVisibleColumns.indexOf(oColumn);

            var iLastIndex = aVisibleColumns.length - 1;
            var bResizedColumnIsLastCol = iColumnIndex === iLastIndex;
            var bResizedColumnIsNotVisible = iColumnIndex === -1;
            var iPercentageSumOfColumnsToNotResize;
            if (iColumnIndex === undefined) {
                iPercentageSumOfColumnsToNotResize = 0;
            } else {
                iPercentageSumOfColumnsToNotResize = this._getColumnPercentageWidth(oColumn);
            }
            var iPercentageSumOfColumnsToResize = 0;
            var aColumnsToResize = [];

            jQuery.each(aVisibleColumns, function(iIndex, oCurrentColumn) {
                var iColumnPercentage = that._getColumnPercentageWidth(oCurrentColumn);
                var bResizedColumnIsCurrentCol = iIndex === iColumnIndex;
                var bThisColIsBeforeResizedCol = iIndex < iColumnIndex;
                var bThisColIsAfterResizedCol = iIndex > iColumnIndex;
                if (!that.isFrozenColumn(oCurrentColumn)) {
                    if (
                        (((bResizedColumnIsLastCol && bThisColIsBeforeResizedCol) || (!bResizedColumnIsLastCol && bThisColIsAfterResizedCol)
                        ) && oCurrentColumn.getFlexible && oCurrentColumn.getFlexible()
                        ) || bResizedColumnIsNotVisible) {
                        iPercentageSumOfColumnsToResize += iColumnPercentage;
                        aColumnsToResize.push(oCurrentColumn);
                    } else if (!bResizedColumnIsCurrentCol) {
                        iPercentageSumOfColumnsToNotResize += iColumnPercentage;
                    }
                }
            });

            logger.debug("Cols to resize sum", iPercentageSumOfColumnsToResize);
            logger.debug("Cols to not resize sum", iPercentageSumOfColumnsToNotResize);

            var iCalcPercentage = iPercentageSumOfColumnsToNotResize;
            jQuery.each(aColumnsToResize, function(iIndex, oCurrentColumn){
                var iColumnPercentage = that._getColumnPercentageWidth(oCurrentColumn);
                var iNewWidth = Math.round((100 - iCalcPercentage) / iPercentageSumOfColumnsToResize * iColumnPercentage);
                if (iIndex === aColumnsToResize.length - 1) {
                    iNewWidth = 100 - iPercentageSumOfColumnsToNotResize;
                    if (iNewWidth < that._iMinPercentWidth || Number.NaN === iNewWidth || !iNewWidth) {
                        iNewWidth = that._iMinPercentWidth;
                    }
                } else {
                    if (iNewWidth < that._iMinPercentWidth || Number.NaN === iNewWidth || !iNewWidth) {
                        iNewWidth = that._iMinPercentWidth;
                    }
                    iPercentageSumOfColumnsToNotResize += iNewWidth;
                }
                // S1417085 - only set a percent width if we had one to start with
                if (oCurrentColumn.getWidth && jQuery.sap.endsWith(oCurrentColumn.getWidth(), "%")) {
                    that._updateColumnWidth(oCurrentColumn, iNewWidth + "%");
                }
            });
        } else if (!this._hasOnlyFixColumnWidths()) {

            // ignore frozen columns
            if (oColumn.getIndex() < iFrozenColumnCount) {
                return;
            }

            var aVisibleColumns = this._getVisibleColumns(),
                iAvailableSpace = this.$().find(".sapUiTableCtrlScr").width(),
                iColumnIndex = -1,
                iLeftWidth = 0,
                iRightWidth = 0;

            var bSomeRelativeSizedColumnExists = aVisibleColumns.slice(iFrozenColumnCount)
                .some(function(oCol) {
                    if (oCol.hasAbsoluteWidth) {
                        return !oCol.hasAbsoluteWidth();
                    } else {
                        return false;
                    }
                });

            //If there are non fixed columns we don't do this
            if (bSomeRelativeSizedColumnExists) { return; }

            var iScrollableWidthSum = this._getWidthSumForScrollableSection();
            logger.debug(iScrollableWidthSum);
            if (iScrollableWidthSum > iAvailableSpace || TableUtils.isWideTable(this) === true) {
                this.$().find(".sapUiTableCtrlScroll").css({
                    "min-width": iScrollableWidthSum + "px",
                    "width": iScrollableWidthSum + "px"
                });
                this.$().find(".sapUiTableColHdrScr .sapUiTableColHdr").css({
                    "min-width": iScrollableWidthSum + "px"
                });
                return;
            }

            // Sum the width left of and right of our target column.
            jQuery.each(aVisibleColumns, function(iIndex, oCurrentColumn) {

                // ignore frozen columns
                if (iIndex < iFrozenColumnCount) {
                    return;
                }

                //if iColumnIndex is defined we already found our column and all other columns are right of that one
                if (iColumnIndex !== -1 && oCurrentColumn.getWidth) {
                    iRightWidth += that._CSSSizeToPixel(oCurrentColumn.getWidth());
                } else if (oColumn !== oCurrentColumn && oCurrentColumn.getWidth) {
                    iLeftWidth += that._CSSSizeToPixel(oCurrentColumn.getWidth());
                } else if (oColumn === oCurrentColumn) {
                    iColumnIndex = iIndex;
                    //Use new width of column
                    iLeftWidth += that._CSSSizeToPixel(sWidth);
                }
            });

            //Available space is all space right of the modified columns
            iAvailableSpace -= iLeftWidth;
            for (var i = iColumnIndex + 1; i < aVisibleColumns.length; i++) {
                //Calculate new column width based on previous percentage width
                var oColumn = aVisibleColumns[i];
                if (oColumn.getWidth) {
                    var iColWidth = this._CSSSizeToPixel(oColumn.getWidth());
                    var iPercent = iColWidth / iRightWidth * 100;
                    var iNewWidth = iAvailableSpace / 100 * iPercent;
                    var sNewWidth = Math.round(iNewWidth) + "px";

                    this._updateColumnWidth(oColumn, sNewWidth);
                }
            }
        }
    };

    /**
     * @private
     */
    Table.prototype._getWidthSumForScrollableSection = function() {
        return this._getColumnsWidth(this.getTotalFrozenColumnCount());
    };

    /**
     *
     * @param oColumn
     * @returns {*}
     * @private
     */
    Table.prototype._getColumnPercentageWidth = function(oColumn) {
        if (!oColumn || oColumn instanceof ColumnGroup) {
            return 0;
        }

        var sColumnWidth = oColumn.getWidth();
        var iColumnPercentage = parseInt(sColumnWidth,10);
        var sTableSelector = (oColumn.getIndex() < this.getTotalFrozenColumnCount()) ? ".sapUiTableCCnt .sapUiTableCtrlFixed" : ".sapUiTableCCnt .sapUiTableCtrlScroll";

        // S1417085 - if columns are unfrozen, use the entire table width
        // because there are no frozen columns (this helps in case the table
        // hasn't rendered since columns were unfrozen)
        if (this._bIgnoreFixedColumnCount) {
            sTableSelector = ".sapUiTableCtrl";
        }
        var iTotalWidth = this.$().find(sTableSelector).width();
        if (oColumn.hasAbsoluteWidth && oColumn.hasAbsoluteWidth()) {
            iColumnPercentage = Math.round((iColumnPercentage / iTotalWidth) * 100);
        } else if (!jQuery.sap.endsWith(sColumnWidth, "%")) {
            iColumnPercentage = Math.round((oColumn.$().width() / iTotalWidth) * 100);
        }
        return iColumnPercentage;
    };

    /**
     * Apply min and max bounds to a proposed column width
     * @param iWidth {int} the proposed column width
     * @return {int} the updated column width
     * @private
     */
    Table.prototype._constrainColumnWidth = function(iWidth) {
        var $sapUiTableCnt = this.$("sapUiTableCnt");

        // don't allow the column width to go under the minimum width
        iWidth = Math.max(iWidth, this.getMinColumnWidth());

        // don't allow the column width to go past the overall width of the table
        iWidth = Math.min(iWidth, $sapUiTableCnt.width());

        return iWidth;
    };

    /**
     * Determine if a width is within the max/min column width bounds
     * @param iWidth {int} a potential column width
     * @return {boolean} if the width is allowed
     * @private
     */
    Table.prototype._isWithinColumnWidthBounds = function(iWidth) {
        return this._constrainColumnWidth(iWidth) === iWidth;
    };

    /**
     * Update a column's width.
     * If needed, convert a unit-less width to use units (px and % supported).
     * @param oColumn {object} the column whose width to act upon
     * @param vWidth {int|string} can be int or string
     * @param bFireEvent {boolean} whether to fire the column resize event or not
     * @private
     */
    Table.prototype._updateColumnWidth = function(oColumn, vWidth, bFireEvent) {
        var sWidth = vWidth,
            $sapUiTableCtrlScroll = this.$().find(".sapUiTableCCnt .sapUiTableCtrlScroll"),
            bExecuteDefault = true;

        if (oColumn instanceof ColumnGroup) {
            return bExecuteDefault;
        }

        // if vWidth is a unit-less number, convert to it's CSSSize string type value
        if (typeof vWidth === "number") {
            // S1417085 - if this column was fixed to begin with, set a fixed width
            if (oColumn.getWidth && oColumn.getWidth() && !jQuery.sap.endsWith(oColumn.getWidth(), "%")) {
                sWidth = vWidth + "px";
            } else {
                sWidth = Math.round((100 / $sapUiTableCtrlScroll.width()) * vWidth) + "%";
            }
        }

        // optionally forward the event
        if (bFireEvent) {
            bExecuteDefault = this.fireColumnResize({
                column: oColumn,
                width: sWidth
            });
        }

        // set the width of the column (when not cancelled)
        if (bExecuteDefault) {
            oColumn.setProperty("width", sWidth, true);
            this.$().find('th[data-sap-ui-colid="' + oColumn.getId() + '"]').css('width', sWidth);
        }

        return bExecuteDefault;
    };

    /**
     * Increase the width of a column by a small bit (controlled by this._iColumnWidthAdjustLength).
     * This is useful for situations like keyboard column resizing.
     *
     * Since calculations are rounded in _updateColumnWidth, width changes will sometimes result in a no-op.
     * If this occurs, keep attempting the bump with increasingly larger width values until the actual width changes.
     *
     * @param {sas.hc.ui.table.Column} oColumn the column to be adjusted
     * @returns {object} A promise that is resolved when the operation is complete (there may be horizontal scroll nudging)
     * @private
     */
    Table.prototype._bumpUpColumnWidth = function(oColumn) {
        var self = this;
        return new Promise(function(resolve) {
            // context menu that drives this call is not available for ColumnGroups
            if (oColumn instanceof ColumnGroup) {
                resolve();
            }

            var iNewWidth,
                // either the column header itself or the summary header label
                $colRef = oColumn._getKeyboardResizeReference();

            iNewWidth = $colRef.width() + self._iColumnWidthAdjustLength;
            iNewWidth = self._constrainColumnWidth(iNewWidth);

            // invoke _updateColumnWidth with new width
            // don't allow width to exceed max col width bounds
            // since the overall-width of a Table can prevent col width increases,
            // only allow a few attempts to avoid stack overflow
            // if actual width doesn't change, increase value and keep trying
            (function increaseWidth(iWidth, iAttempt) {
                var iOriginalWidth = $colRef.width();
                if (self._isWithinColumnWidthBounds(iWidth) === false || iAttempt > 5) {
                    return;
                }
                if (self._updateColumnWidth(oColumn, iWidth, false) && oColumn.getWidth) {
                    self._resizeDependentColumns(oColumn, oColumn.getWidth());
                }
                if (iOriginalWidth >= $colRef.width()) {
                    // Increase the width by 1% of the total table width and try again.
                    // Using 1% instead of 1px insures the widths of the other columns
                    // actually change in _normalizeColumnWidths; otherwise resize larger
                    // after unfreezing due to width won't work.
                    var iDelta = 1;
                    var $tableContainer = self.$("sapUiTableCnt");
                    var iTableWidth;
                    if ($tableContainer && $tableContainer.width) {
                        iTableWidth = $tableContainer.width();
                    }
                    if (iTableWidth) {
                        iDelta = Math.round(iTableWidth*.01);
                    }
                    increaseWidth(iWidth + iDelta, iAttempt + 1);
                }
            }(iNewWidth, 0));
            self._syncColumnHeaders(self._collectTableSizes());

            // if column is not frozen and it's far edge is no longer visible,
            // horizontally nudge the scrollbar to compensate
            if (self.isFrozenColumn(oColumn) === false && oColumn._isFarEdgeWithinScrollRange() === false) {
                // in RTL, nudge backwards, otherwise nudge forwards
                if (self._bRtlMode === true) {
                    self._horizontalScrollNudgeBackwardsUntil(function() {
                        return oColumn._isFarEdgeWithinScrollRange() === false || oColumn._areBothEdgesOutsideOfScrollRange() === true;
                    }).then(function() {
                        resolve();
                    });
                } else {
                    self._horizontalScrollNudgeForwardsUntil(function() {
                        if (oColumn._isFarEdgeWithinScrollRange() === false) {
                            return false;
                        }
                        return oColumn._areBothEdgesOutsideOfScrollRange();
                    }).then(function() {
                        resolve();
                    });
                }
            } else {
                resolve();
            }
        });
    };

    /**
     * Decrease the width of a column by a small bit (controlled by this._iColumnWidthAdjustLength).
     * This is useful for situations like keyboard column resizing.
     *
     * Since calculations are rounded in _updateColumnWidth, width changes will sometimes result in a no-op.
     * If this occurs, keep attempting the bump with decreasingly smaller width values until the actual width changes.
     *
     * @param {sas.hc.ui.table.Column} oColumn the column to be adjusted
     * @returns {object} A promise that is resolved when the operation is complete
     * @private
     */
    Table.prototype._bumpDownColumnWidth = function(oColumn) {
        var self = this;
        return new Promise(function(resolve) {
            // context menu that drives this call is not available for ColumnGroups
            if (oColumn instanceof ColumnGroup) {
                resolve();
            }

            var iNewWidth,
                // either the column header itself or the summary header label
                $colRef = oColumn._getKeyboardResizeReference();

            // determine the new width
            iNewWidth = $colRef.width() - self._iColumnWidthAdjustLength;
            iNewWidth = self._constrainColumnWidth(iNewWidth);

            // invoke _updateColumnWidth with new width
            // don't allow width to exceed min col width bounds
            // if actual width doesn't change, decrease value and keep trying
            // only allow a few attempts to avoid stack overflow, the same as increaseWidth
            (function decreaseWidth(iWidth, iAttempt) {
                var iOriginalWidth = $colRef.width();
                if (self._isWithinColumnWidthBounds(iWidth) === false || iAttempt > 5) {
                    return;
                }
                if (self._updateColumnWidth(oColumn, iWidth, false) && oColumn.getWidth) {
                    self._resizeDependentColumns(oColumn, oColumn.getWidth());
                }

                if (self._shouldBeFrozenColumn(oColumn)) {
                    // S1423815 - When resizing with the keyboard we don't invalidate
                    // so we need to adjust other columns based on the new width
                    if (self._hasSomePercentageWidthColumn()) {
                        // if there is a column with a relative width
                        // call _normalizeColumnWidths directly to adjust
                        // other columns based on the new width
                        self._normalizeColumnWidths();
                    } else if (self.isFrozenColumn(oColumn)) {
                        // if there are no % based columns, you can't adjust the
                        // other columns, so the table size needs to adjust;
                        // if this was an unfrozen column that just refroze
                        // but has not re-rendered, change the min-width and
                        // width of the scroll section based on the new width;
                        // if this is not a frozen column, no action is necessary
                        // for the other columns
                        var $fixedSection = self.$().find(".sapUiTableCCnt .sapUiTableCtrlFixed");
                        if ($fixedSection.length === 0) {
                            var $scrollSection = self.$().find(".sapUiTableCCnt .sapUiTableCtrlScroll");
                            var iSectionMinWidth = parseInt($scrollSection.css("min-width"));
                            var iSectionNewMinWidth;
                            if (iSectionMinWidth) {
                                iSectionNewMinWidth = iSectionMinWidth - (iOriginalWidth - iWidth);
                            }
                            if (iSectionNewMinWidth && !isNaN(iSectionNewMinWidth)) {
                                $scrollSection.css({
                                    "min-width": iSectionNewMinWidth+"px",
                                    "width": iSectionNewMinWidth+"px"
                                });
                            }
                        }
                    }
                }
                if (iOriginalWidth <= $colRef.width()) {
                    // Decrease the width by 1% of the total table width and try again.
                    // Using 1% instead of 1px insures the widths of the other columns
                    // actually change in _normalizeColumnWidths; otherwise resize smaller
                    // after unfreezing due to width won't work.
                    var iDelta = 1;
                    var $tableContainer = self.$("sapUiTableCnt");
                    var iTableWidth;
                    if ($tableContainer && $tableContainer.width) {
                        iTableWidth = $tableContainer.width();
                    }
                    if (iTableWidth) {
                        iDelta = Math.round(iTableWidth*.01);
                    }
                    decreaseWidth(iWidth - iDelta, iAttempt + 1);
                }
            }(iNewWidth, 0));
            self._syncColumnHeaders(self._collectTableSizes());
            resolve();
        });
    };

    /**
     * Tests whether there is at least one column that has
     * a percentage-based width. Returns true if there is
     * at least one column with a percent-based width and
     * returns false otherwise.
     *
     * @private
    */
    Table.prototype._hasSomePercentageWidthColumn = function() {
        return this.retrieveLeafColumns().some(function(oCol) {
            // IE compatible .endsWith()
            if (oCol.getWidth) {
                return oCol.getWidth().substr(-1) === "%";
            } else {
                return false;
            }
        });
    };

    /**
     * Check if at least one column has a percentage value
     *
     * @private
     */
    Table.prototype._checkPercentageColumnWidth = function() {
        // TODO: Remove in favor of _hasSomePercentageWidthColumn
        // That function is preferred for a more compact
        // implementation and uses a more descriptive name.
        this._hasSomePercentageWidthColumn();
    };

    /**
     * Check if table has only non flexible columns with fixed widths and only then
     * the table adds a dummy column to fill the rest of the width instead of resizing
     * the columns to fit the complete table width
     * @private
     */
    Table.prototype._hasOnlyFixColumnWidths = function() {
        var bOnlyFixColumnWidths = true;
        jQuery.each(this.retrieveLeafColumns(), function(iIndex, oColumn) {
            if (!oColumn.getWidth || !oColumn.getFlexible) {
                return false;
            }
            var sWidth = oColumn.getWidth();
            if (oColumn.getFlexible() || !sWidth || sWidth.substr(-2) !== "px") {
                bOnlyFixColumnWidths = false;
                return false;
            }
        });
        return bOnlyFixColumnWidths;
    };


    // =============================================================================
    // SORTING & FILTERING
    // =============================================================================

    /**
     * pushes the sorted column to array
     *
     * @param {sas.hc.ui.table.Column} oColumn
     *         column to be sorted
     * @param {Boolean} bAdd Set to true to add the new sort criterion to the existing sort criteria
     * @type sas.hc.ui.table.Table
     * @private
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.pushSortedColumn = function(oColumn, bAdd) {

        if (!bAdd) {
            this._aSortedColumns = [];
        }

        this._aSortedColumns.push(oColumn);

    };

    /**
     * gets sorted columns
     *
     * @return Array of sorted columns
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.getSortedColumns = function() {

        return this._aSortedColumns;

    };

    /**
     * sorts the given column ascending or descending
     *
     * @param {sas.hc.ui.table.SapColumn} oColumn
     *         column to be sorted
     * @param {sas.hc.ui.table.SortOrder} oSortOrder
     *         sort order of the column (if undefined the default will be ascending)
     * @param {Boolean} bAdd Set to true to add the new sort criterion to the existing sort criteria
     * @type sas.hc.ui.table.Table
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.sort = function(oColumn, oSortOrder, bAdd) {
        if (oColumn.sort && (oColumn.getTable() === this)) {
            oColumn.sort(oSortOrder === SortOrder.Descending, bAdd);
        }
    };


    /**
     * filter the given column by the given value
     *
     * @param {sas.hc.ui.table.Column} oColumn
     *         column to be filtered
     * @param {string} sValue
     *         filter value as string (will be converted)
     * @type sas.hc.ui.table.Table
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.filter = function(oColumn, sValue) {
        if (oColumn.filter && (oColumn.getTable() === this)) {
            oColumn.filter(sValue);
            this._updateA11yVisibleRowsLabel();
        }
    };

    /**
     * Applies a sort operation on the given column based on the supplied modifier flags.
     * @param {sas.hc.ui.table.Column} oColumn the column to base the sort operation on
     * @param {boolean} bCtrl true if removing a sort or adding/toggling a multicolumn sort
     * @param {boolean} bShift true if the intended operation is removal of sort parameter
     * @returns {boolean} true if some sort operation took place (toggle, add, remove)
     * @private
     */
    Table.prototype._applySortActionFromEvent = function(oColumn, bCtrl, bShift) {
        if (!oColumn || !oColumn.getSorted || !oColumn.getSortOrder) {
            return false;
        }

        var bSorted = oColumn.getSorted(),
            oSortOrder = oColumn.getSortOrder(),
            bHandled = false;

        //columns can be not sortable, so look for that first
        if (oColumn.isSortableByMenu && oColumn.isSortableByMenu()) {
            if (!bCtrl && !bShift) {
                //left click w/o modifer:
                //  1) get column's current sorted state
                //  2a) if column was previously sorted, apply single column sort in opposite direction
                //  2b) if column was not sorted, apply single column sort in default direction
                var oNewOrder = (bSorted && (oSortOrder === SortOrder.Ascending)) ? SortOrder.Descending : SortOrder.Ascending;
                this.sort(oColumn, oNewOrder, false);
                bHandled = true;
            } else if (bCtrl && !bShift) {
                //left click while holding Ctrl
                //  1) get column's current sorted state
                //  2a) if column was previously sorted, then toggle sort direction
                //  2b) if column was not previously sorted, then add additional sort parameter for column in default direction
                var oNewOrder = (bSorted && (oSortOrder === SortOrder.Ascending)) ? SortOrder.Descending : SortOrder.Ascending;
                this.sort(oColumn, oNewOrder, true);
                bHandled = true;
            } else if (bCtrl && bShift) {
                //left click while holding Ctrl+Shift
                //  1a) if column not sorted, then exit
                //  1b) remove sort info from column
                if (bSorted) {
                    this._removeColumnSort(oColumn);
                }
                bHandled = true;
            }
        }

        return bHandled;
    };

    // =============================================================================
    // SELECTION HANDLING
    // =============================================================================

    /**
     * updates the visual selection in the HTML markup
     * @private
     */
    Table.prototype._updateSelection = function() {
        if (this.getSelectionMode() === SelectionMode.None) {
            // there is no selection which needs to be updated. With the switch of the
            // selection mode the selection was cleared (and updated within that step)
            return;
        }

        // retrieve tooltip and aria texts only once and pass them to the rows _updateSelection function
        var mTooltipTexts = this._getAccExtension().getAriaTextsForSelectionMode(true);

        // check whether the row can be clicked to change the selection
        var bSelectOnCellsAllowed = TableUtils.isRowSelectionAllowed(this);

        // trigger the rows to update their selection
        var aRows = this.getRows();
        for (var i = 0; i < aRows.length; i++) {
            var oRow = aRows[i];
            oRow._updateSelection(this, mTooltipTexts, bSelectOnCellsAllowed);
        }

        // update state of selection controls with their corresponding row
        this._refreshSelectionControls();
    };


    /**
     * notifies the selection listeners about the changed rows
     * @private
     */
    Table.prototype._onSelectionChanged = function(oEvent) {
        var aRowIndices = oEvent.getParameter("rowIndices");
        var bSelectAll = oEvent.getParameter("selectAll");
        var iRowIndex = this._iSourceRowIndex !== undefined ? this._iSourceRowIndex : this.getSelectedIndex();
        this._updateSelection();
        var oSelMode = this.getSelectionMode();
        if (oSelMode === "MultiToggle" || oSelMode === "ParentChild") {
            this._updateSelectAllContainer(false);
        }

        this.fireRowSelectionChange({
            rowIndex: iRowIndex,
            rowContext: this.getContextByIndex(iRowIndex),
            rowIndices: aRowIndices,
            selectAll: bSelectAll,
            userInteraction: this._iSourceRowIndex !== undefined
        });

        // will update state and tooltip
        this.getSelectAllControl().updateRelativeToTable();
    };

    /**
     * Returns the context of a row by its index. Please note that for server-based models like OData,
     * the supplied index might not have been loaded yet. If the context is not available at the client,
     * the binding will trigger a backend request and request this single context. Although this API
     * looks synchronous it may not return a context but load it and fire a change event on the binding.
     *
     * For server-based models you should consider to only make this API call when the index is within
     * the currently visible scroll area.
     *
     * @param {int} iIndex
     *         Index of the row to return the context from.
     * @type object
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.getContextByIndex = function(iIndex) {
        // TODO: ODataListBinding needs to make sure to prevent loading multiple times
        // index must not be smaller than 0! otherwise the ODataModel fails
        var oBinding = this.getBinding("rows");
        return iIndex >= 0 && oBinding ? oBinding.getContexts(iIndex, 1)[0] : null;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.getSelectedIndex = function() {
        return this._oSelection.getLeadSelectedIndex();
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setSelectedIndex = function(iIndex) {
        if (iIndex === -1) {
            //If Index eq -1 no item is selected, therefore clear selection is called
            //SelectionModel doesn't know that -1 means no selection
            this._oSelection.clearSelection();
        } else {
            //If remember selection is on then add the selected index's context path to the remembered list
            //This must be done before calling _oSelection.setSelectionInterval() to have the remembered context paths set right ahead of time.
            //Since _oSelection's change event fires the _onSelectionChanged event which fires the rowSelectionChange event,
            //      we need to get the context paths right which consumer will ask for during rowSelectionChange event handling.
            if(this.getRememberSelections()){
                this.handleSelectionChange([iIndex], [], "setSelectedIndex");
            }

            this._oSelection.setSelectionInterval(iIndex, iIndex);
        }
        return this;
    };

    /**
     * Removes complete selection.
     *
     * @type sas.hc.ui.table.Table
     * @public
     * @param {string} sReason reason for calling clearSelection. Possible values:
     *                      "selectAllUnchecked" : when select all checkbox is unchecked,
     *                      "clearAll": when we really need to clear all the table selections, usually cause of an user initiated action, not internal action.
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.clearSelection = function(sReason) {
        var bFromSelectAll = (sReason === true) || (sReason === "selectAllUnchecked"),
            oTable = this;

        //If remember selection is on then handle clearSelection call based on the reasons
        //This must be done before calling _oSelection.clearSelection() to have the remembered context paths set right ahead of time.
        //Since _oSelection's change event fires the _onSelectionChanged event which fires the rowSelectionChange event,
        //      we need to get the context paths right which consumer will ask for during rowSelectionChange event handling.
        if (this.getRememberSelections()) {
            if (bFromSelectAll && this.isFiltered()) {
                //If it is a filtered list, then only clear the selection paths for the current filtered list of rows.
                //Otherwise clear all the selection paths
                this._gatherSelectionKeysAsync(0).then(function(aKeys) {
                    oTable.removeSelectedKeys(aKeys, true);
                    oTable._oSelection.clearSelection(!!bFromSelectAll);
                    oTable.fireEvent("selectionUpdated",{"selectionKeys": aKeys, "reason": (sReason || "clearSelection")});
                });

                return this;
            } else if (sReason !== ChangeReason.Sort && sReason !== ChangeReason.Filter) {
                var aKeys = this.getSelectedKeys();
                this.clearSelectedKeys(sReason, true);
                this._oSelection.clearSelection(!!bFromSelectAll);
                this.fireEvent("selectionUpdated",{"selectionKeys": aKeys, "reason": (sReason || "clearSelection")});

                return this;
            }
        }

        this._oSelection.clearSelection(!!bFromSelectAll);
        return this;
    };

    /**
     * Add all rows to the selection.
     * Please note that for server based models like OData the indices which are considered to be selected might not
     * be available at the client yet. Calling getContextByIndex might not return a result but trigger a roundtrip
     * to request this single entity.
     *
     * @return sas.hc.ui.table.Table
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.selectAll = function() {
        var oSelMode = this.getSelectionMode();
        if (!this.getEnableSelectAll() || (oSelMode !== "MultiToggle" && oSelMode !== "ParentChild")) {
            return this;
        }
        var oBinding = this.getBinding("rows");
        if (oBinding) {
            //If rememberSelections is set to true then also update the remembered selected context paths
            //This must be done before calling _oSelection.selectAll() to have the remembered context paths set right ahead of time.
            //Since _oSelection's change event fires the _onSelectionChanged event which fires the rowSelectionChange event,
            //      we need to get the context paths right which consumer will ask for during rowSelectionChange event handling.
            if(this.getRememberSelections()){
                this._selectAllKeys("selectAll");
            }

            this._oSelection.selectAll((oBinding.getLength() || 0) - 1);
            this._updateSelectAllContainer(true);
        }
        return this;
    };

    Table.prototype._updateSelectAllContainer = function(bEverythingIsSelected) {
        var $selall = this.$("selall");
        if (bEverythingIsSelected) {
            $selall.attr('title',this._oResBundle.getText("TBL_DESELECT_ALL.txt"));
            $selall.removeClass("sapUiTableSelAll");
        } else {
            $selall.attr('title',this._oResBundle.getText("TBL_SELECT_ALL.txt"));
            $selall.addClass("sapUiTableSelAll");
        }
    };

    /**
     * Zero-based indices of selected items, wrapped in an array. An empty array means "no selection".
     *
     * @return int[]
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.getSelectedIndices = function() {
        return this._oSelection.getSelectedIndices();
    };

    /**
     * Adds the given selection interval to the selection. In case of single selection the "indexTo" value will be used for as selected index.
     *
     * @param {int} iIndexFrom
     *         Index from which .
     * @param {int} iIndexTo
     *         Indices of the items that shall additionally be selected.
     * @type sas.hc.ui.table.Table
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.addSelectionInterval = function(iIndexFrom, iIndexTo) {
        //If rememberSelections is set to true then also update the remembered selected context paths
        //This must be done before calling _oSelection.addSelectionInterval() to have the remembered context paths set right ahead of time.
        //Since _oSelection's change event fires the _onSelectionChanged event which fires the rowSelectionChange event,
        //      we need to get the context paths right which consumer will ask for during rowSelectionChange event handling.
        if(this.getRememberSelections()){
            var aKeysToAdd = [],
                sSelectionIdProperty = this.getSelectionIdProperty(),
                oContext, sPath;

            for(var i = iIndexFrom; i <= iIndexTo; i++){
                oContext = this.getContextByIndex(i);
                //if 'selectionIdProperty' is not null, then need to use the value of that property instead of path
                if (!sSelectionIdProperty) {
                    sPath = oContext.getPath();
                } else {
                    sPath = oContext.getProperty(sSelectionIdProperty);
                }

                if(sPath){
                    aKeysToAdd.push(sPath);
                }
            }
            if(aKeysToAdd.length>0){
                this.updateSelectedKeys(aKeysToAdd, [], "addSelectionInterval");
            }
        }

        this._oSelection.addSelectionInterval(iIndexFrom, iIndexTo);
        return this;
    };

    /**
     * Sets the given selection interval as selection. In case of single selection the "indexTo" value will be used for as selected index.
     *
     * @param {int} iIndexFrom
     *         Index from which .
     * @param {int} iIndexTo
     *         Indices of the items that shall additionally be selected.
     * @type sas.hc.ui.table.Table
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.setSelectionInterval = function(iIndexFrom, iIndexTo) {
        var oTable = this;

        //If rememberSelections is set to true then also update the remembered selected context paths
        //This must be done before calling _oSelection.setSelectionInterval() to have the remembered context paths set right ahead of time.
        //Since _oSelection's change event fires the _onSelectionChanged event which fires the rowSelectionChange event,
        //      we need to get the context paths right which consumer will ask for during rowSelectionChange event handling.
        if(this.getRememberSelections()){
            this._gatherSelectionKeysAsync(iIndexFrom, iIndexTo).then(function(aKeys) {
                oTable.setSelectedKeys(aKeys, "setSelectionInterval", true);
                oTable._oSelection.setSelectionInterval(iIndexFrom, iIndexTo);
                oTable.fireEvent("selectionUpdated",{"selectionKeys": aKeys, "reason": "setSelectionInterval"});
            });
        } else {
            this._oSelection.setSelectionInterval(iIndexFrom, iIndexTo);
        }

        return this;
    };

    /**
     * Removes the given selection interval from the selection. In case of single selection this call removeSelectedIndex with the "indexTo" value.
     *
     * @param {int} iIndexFrom
     *         Index from which .
     * @param {int} iIndexTo
     *         Indices of the items that shall additionally be selected.
     * @type sas.hc.ui.table.Table
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.removeSelectionInterval = function(iIndexFrom, iIndexTo) {
        var oTable = this;

        //If rememberSelections is set to true then also update the remembered selected context paths
        //This must be done before calling _oSelection.removeSelectionInterval() to have the remembered context paths set right ahead of time.
        //Since _oSelection's change event fires the _onSelectionChanged event which fires the rowSelectionChange event,
        //      we need to get the context paths right which consumer will ask for during rowSelectionChange event handling.
        if(this.getRememberSelections()){
            this._gatherSelectionKeysAsync(iIndexFrom, iIndexTo).then(function(aKeys) {
                oTable.removeSelectedKeys(aKeys, true);
                oTable._oSelection.removeSelectionInterval(iIndexFrom, iIndexTo);
                oTable.fireEvent("selectionUpdated",{"selectionKeys": aKeys, "reason": "setSelectionInterval"});
            });
        } else {
            this._oSelection.removeSelectionInterval(iIndexFrom, iIndexTo);
        }

        return this;
    };

    /**
     * Returns whether the given index is selected.
     *
     * @param {int} iIndex
     *         Index which is checked for selection state.
     * @type boolean
     * @public
     * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
     */
    Table.prototype.isIndexSelected = function(iIndex) {
        return this._oSelection.isSelectedIndex(iIndex);
    };

    Table.prototype._gatherSelectionKeysAsync = function(iIndexFrom, iIndexTo) {
        var oBinding = this.getBinding("rows"),
            oTable = this,
            iLength;

        if (!oBinding) {
            return Promise.reject("no binding");
        }

        // Binding.getContexts requires a starting index and the number of contexts
        if (iIndexTo >= 0) {
            iLength = iIndexTo - iIndexFrom + 1;
        } else {
            iLength = oBinding.getLength() - iIndexFrom;
        }

        return new Promise(function(resolve, reject) {
            var aContexts = oBinding.getContexts(iIndexFrom, iLength),
                sSelectionIdProperty = oTable.getSelectionIdProperty(),
                aKeys = [], oContext, i, sPath;

            //PERF - looping through many/all contexts in binding will be non-performant
            for(i = 0; i < aContexts.length; i++){
                oContext = aContexts[i];
                if (!sSelectionIdProperty) {
                    sPath = oContext.getPath();
                } else {
                    sPath = oContext.getProperty(sSelectionIdProperty);
                }

                if(sPath) {
                    aKeys.push(sPath);
                }
            }

            resolve(aKeys);
        });
    };

    /**
     * Overrides ManageObject's version to clear selection after changing keys.
     * @param sValue
     * @param bSuppressInvalidate
     */
    Table.prototype.setSelectionIdProperty = function(sValue, bSuppressInvalidate) {
        if (sValue !== this.getSelectionIdProperty()) {
            Control.prototype.setProperty.call(this, "selectionIdProperty", sValue, bSuppressInvalidate);
            this.clearSelection("selectionIdProperty");
        }
    };

    // =============================================================================
    // GROUPING
    // =============================================================================

    /**
     * @override
     */
    Table.prototype.setGroupBy = function(vValue) {
        // determine the group by column
        var oGroupBy = vValue;
        if (typeof oGroupBy === "string") {
            oGroupBy = sap.ui.getCore().byId(oGroupBy);
        }

        // only for columns we do the full handling here - otherwise the method
        // setAssociation will fail below with a specific fwk error message
        var bReset = false;
        if (oGroupBy && oGroupBy instanceof Column) {

            // check for column being part of the columns aggregation
            if (jQuery.inArray(oGroupBy, this.retrieveLeafColumns()) === -1) {
                throw new Error("Column has to be part of the columns aggregation!");
            }

            // fire the event (to allow to cancel the event)
            var bExecuteDefault = this.fireGroup({column: oGroupBy, groupedColumns: [oGroupBy.getId()], type: GroupEventType.group});

            // first we reset the grouping indicator of the old column (will show the column)
            var oOldGroupBy = sap.ui.getCore().byId(this.getGroupBy());
            if (oOldGroupBy) {
                oOldGroupBy.setGrouped(false);
                bReset = true;
            }

            // then we set the grouping indicator of the new column (will hide the column)
            // ==> only if the default behavior is not prevented
            if (bExecuteDefault && oGroupBy instanceof Column) {
                oGroupBy.setGrouped(true);
            }

        }

        // reset the binding when no value is given or the binding needs to be reseted
        // TODO: think about a better handling to recreate the group binding
        if (!oGroupBy || bReset) {
            var oBindingInfo = this.getBindingInfo("rows");
            delete oBindingInfo.binding;
            this._bindAggregation("rows", oBindingInfo);
        }

        // set the new group by column (TODO: undefined doesn't work!)
        return this.setAssociation("groupBy", oGroupBy);
    };

    /**
     * @override
     */
    Table.prototype.getBinding = function(sName) {
        TableUtils.Grouping.setupExperimentalGrouping(this);
        return Element.prototype.getBinding.call(this, [sName || "rows"]);
    };

    /**
     * @private
     */
    Table.prototype.setEnableGrouping = function(bEnableGrouping) {
        // set the property
        this.setProperty("enableGrouping", bEnableGrouping);
        // reset the grouping
        if (!bEnableGrouping) {
            TableUtils.Grouping.resetExperimentalGrouping(this);
        }
        // update the column headers
        this._invalidateColumnMenus();
        return this;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setEnableCustomFilter = function(bEnableCustomFilter) {
        this.setProperty("enableCustomFilter", bEnableCustomFilter);
        // update the column headers
        this._invalidateColumnMenus();
        return this;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setEnableColumnFreeze = function(bEnableColumnFreeze) {
        if (this.hasColumnGroups()) {
            // do not allow column freeze menu when column groups are present
            bEnableColumnFreeze = false;
        }
        this.setProperty("enableColumnFreeze", bEnableColumnFreeze);
        this._invalidateColumnMenus();
        return this;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setEnableColumnReordering = function(bEnableColumnReordering) {
        if (this.hasColumnGroups()) {
            // do not allow column reordering menu when column groups are present
            bEnableColumnReordering = false;
        }
        this.setProperty("enableColumnReordering", bEnableColumnReordering);
        this._invalidateColumnMenus();
        return this;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setShowColumnVisibilityMenu = function(bShowColumnVisibilityMenu) {
        if (this.hasColumnGroups()) {
            // do not allow column visibility menu when column groups are present
            bShowColumnVisibilityMenu = false;
        }
        this.setProperty("showColumnVisibilityMenu", bShowColumnVisibilityMenu);
        this._invalidateColumnMenus();
        return this;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setShowManageColumnsMenuItem = function(bShowManageColumnsMenuItem) {
        if (this.hasColumnGroups()) {
            // do not allow column manager menu when column groups are present
            bShowManageColumnsMenuItem = false;
        }
        this.setProperty("showManageColumnsMenuItem", bShowManageColumnsMenuItem);
        this._invalidateColumnMenus();
        return this;
    };

    /*
    * @see JSDoc generated by SAPUI5 control API generator
    */
    Table.prototype.getFixedColumnCount = function() {
        if (this._bIgnoreFixedColumnCount) {
            return 0;
        }
        return this.getProperty("fixedColumnCount");
    };

    /**
     * The fixedColumnCount property is not a complete picture of how many columns need to be rendered as frozen.
     * This function is intended to account for other such columns (such as mandatory columns).
     * @returns {number}
     */
    Table.prototype.getTotalFrozenColumnCount = function() {
        if (this._bIgnoreFixedColumnCount) {
            return 0;
        }
        return this._getActualFrozenColumnCount();
    };

    /**
     * Computes the frozen column count regardless of space available for frozen columns.
     * @private
     * @returns {number}
     */
    Table.prototype._getActualFrozenColumnCount = function() {
        return this.getProperty("fixedColumnCount") + this._getMandatoryColumnCount();
    };

    /**
     * Get the number of Columns that are mandatory.
     *
     * @protected
     * @returns {number}
     */
    Table.prototype._getMandatoryColumnCount = function() {

        if (this._iMandatoryColumnCount < 0) {
            this._iMandatoryColumnCount = this._getMandatoryColumns().length;
        }

        return this._iMandatoryColumnCount;
    };

    /**
     * @private
     */
    Table.prototype._resetMandatoryColumnCount = function() {
        this._iMandatoryColumnCount = -1;
    };

    /**
     * Get the number of Columns that are not mandatory.
     *
     * @protected
     * @returns {number}
     */
    Table.prototype._getNonMandatoryColumnCount = function() {
        return this._getNonMandatoryColumns().length;
    };

    /**
     * Get the Columns that are mandatory.
     * Note: Calling this function frequently can bottleneck Table performance.
     *
     * @protected
     * @returns {Column[]}
     */
    Table.prototype._getMandatoryColumns = function() {
        return this.retrieveLeafColumns().filter(function(oCol) {
            return oCol.getMandatory();
        });
    };

    /**
     * Get the Columns that are not mandatory.
     * Note: Calling this function frequently can cause high CPU usage.
     *
     * @protected
     * @returns {Column[]}
     */
    Table.prototype._getNonMandatoryColumns = function() {
        return this.retrieveLeafColumns().filter(function(oCol) {
            return !oCol.getMandatory();
        });
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setFixedColumnCount = function(iFixedColumnCount, bSuppressInvalidate) {
        var aCols = this._getVisibleColumns();
        var iFixedColumnIndex = this._getMandatoryColumnCount() + iFixedColumnCount - 1;
        var vHeaderSpan = aCols[iFixedColumnIndex] && aCols[iFixedColumnIndex].getHeaderSpan && aCols[iFixedColumnIndex].getHeaderSpan();
        if (vHeaderSpan) {
            var iHeaderSpan;
            if (jQuery.isArray(vHeaderSpan)) {
                iHeaderSpan = parseInt(vHeaderSpan[0], 10);
            } else {
                iHeaderSpan = parseInt(vHeaderSpan, 10);
            }
            iFixedColumnCount += iHeaderSpan - 1;
        }
        //Set current width as fixed width for cols
        var $ths = this.$().find(".sapUiTableCtrlFirstCol > th");
        var iFrozenColumnCount = iFixedColumnCount + this._getMandatoryColumnCount();
        for (var i = 0; i < (iFrozenColumnCount); i++) {
            var oColumn = aCols[i];
            if (oColumn) {
                var iColumnIndex = jQuery.inArray(oColumn, this.retrieveLeafColumns());
                if (oColumn.getWidth && !oColumn.getWidth()) {
                    var iRenderedWidth = $ths.filter("[data-sap-ui-headcolindex='" + iColumnIndex + "']").width();
                    if (iRenderedWidth > 0 && oColumn.setWidth) {
                        oColumn.setWidth(iRenderedWidth + "px");
                    }
                }
            }
        }

        this.setProperty("fixedColumnCount", iFixedColumnCount, bSuppressInvalidate);

        // call collectTableSizes to determine whether the number of fixed columns can be displayed at all
        // this is required to avoid flickering of the table in IE if the fixedColumnCount must be adjusted
        this._collectTableSizes();
        this._invalidateColumnMenus();
        return this;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setFixedRowCount = function(iFixedRowCount) {
        if (!(parseInt(iFixedRowCount, 10) >= 0)) {
            jQuery.sap.log.error("Number of fixed rows must be greater or equal 0", this);
            return this;
        }

        if ((iFixedRowCount + this.getFixedBottomRowCount()) < this.getVisibleRowCount()) {
            this.setProperty("fixedRowCount", iFixedRowCount);
            this._updateBindingContexts();
        } else {
            jQuery.sap.log.error("Table '" + this.getId() + "' fixed rows('" + (iFixedRowCount + this.getFixedBottomRowCount()) + "') must be smaller than numberOfVisibleRows('" + this.getVisibleRowCount() + "')", this);
        }
        return this;
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setFixedBottomRowCount = function(iFixedRowCount) {
        if (!(parseInt(iFixedRowCount, 10) >= 0)) {
            jQuery.sap.log.error("Number of fixed bottom rows must be greater or equal 0", this);
            return this;
        }

        if ((iFixedRowCount + this.getFixedRowCount()) < this.getVisibleRowCount()) {
            this.setProperty("fixedBottomRowCount", iFixedRowCount);
            this._updateBindingContexts();
        } else {
            jQuery.sap.log.error("Table '" + this.getId() + "' fixed rows('" + (iFixedRowCount + this.getFixedRowCount()) + "') must be smaller than numberOfVisibleRows('" + this.getVisibleRowCount() + "')", this);
        }
        return this;
    };

    /**
     * Sets the threshold value, which will be added to all data requests in
     * case the Table is bound against an OData service.
     * @public
     */
    Table.prototype.setThreshold = function (iThreshold) {
        this.setProperty("threshold", iThreshold, true);
    };

    /**
     *
     * @private
     */
    Table.prototype._invalidateColumnMenus = function() {
        var aCols = this.retrieveLeafColumns();
        if (!aCols) {
            return;
        }
        for (var i = 0, l = aCols.length; i < l; i++) {
            if (aCols[i].invalidateMenu) {
                aCols[i].invalidateMenu();
            }
        }
    };

    /**
     * Checks whether the passed oEvent is a touch event.
     * @private
     * @param {jQuery.Event} oEvent The event to check
     * @returns {boolean} Returns <code>true</code>, if <code>oEvent</code> is a touch event
     */
    Table.prototype._isTouchMode = function(oEvent) {
        return oEvent && oEvent.originalEvent && oEvent.originalEvent.touches;
    };

    /**
     * @private
     */
    Table.prototype._determineParent = function() {
        var oParent = this.getParent();

        if (oParent) {
            var oParentDomRef;
            if (oParent.getDomRef) {
                // for Controls
                oParentDomRef = oParent.getDomRef();
            } else if (oParent.getRootNode) {
                // for UIArea
                oParentDomRef = oParent.getRootNode();
            }

            if (oParentDomRef) {
                return jQuery(oParentDomRef);
            }
        }
        return jQuery();
    };

    /**
     * @private
     */
    Table.prototype._includeColumnInRowTemplate = function(oColumn) {
        return oColumn.getVisible();
    };

    /**
     * @private
     */
    Table.prototype._getRowTemplate = function() {
        if (!this._oRowTemplate) {
            // create the new template

            this._oRowTemplate = this._createRowTemplate(this.getId() + "-rows");

            var aColumns = this.retrieveLeafColumns();
            for (var i = 0, l = aColumns.length; i < l; i++) {
                if (this._includeColumnInRowTemplate(aColumns[i])) {
                    var oColumnTemplate;
                    if (aColumns[i].getTemplate) {
                        oColumnTemplate = aColumns[i].getTemplate();
                    }
                    if (oColumnTemplate) {
                        var oColumnTemplateClone = oColumnTemplate.clone("col" + i);
                        oColumnTemplateClone.data("sap-ui-colindex", i);
                        oColumnTemplateClone.data("sap-ui-colid", aColumns[i].getId());
                        this._oRowTemplate.addCell(oColumnTemplateClone);
                    }
                }
            }
        }

        return this._oRowTemplate;
    };

    /**
     * @private
     */
    Table.prototype._getDummyRow = function() {
        if (!this._oDummyRow) {
            this._oDummyRow = this._getRowTemplate().clone("dummy");
            this._oDummyRow._bDummyRow = true;
            this._oDummyRow._bHidden = true;
        }

        return this._oDummyRow;
    };

    /**
     * @private
     */
    Table.prototype._resetRowTemplate = function() {
        if (this._oRowTemplate) {
            this._oRowTemplate.destroy();
            this._oRowTemplate = undefined;
        }
        if (this._oDummyRow) {
            this._oDummyRow.destroy();
            this._oDummyRow = undefined;
        }

        var self = this;
        Object.keys(this._mCachedRows).forEach(function(sCacheKey) {
            var oCachedRow = self._mCachedRows[sCacheKey];
            // If cached row still has a parent, let the parent's destroyAggregation destroy it.
            if (oCachedRow && !oCachedRow.getParent()) {
                self._mCachedRows[sCacheKey].destroy();
            }
        });
        this._mCachedRows = {};

        //reset column widths from anti-stretched assignments (may be reassigned in onAfterRendering)
        this._restoreOriginalColumnWidths();

        this._resetMandatoryColumnCount();
    };

    /**
     * @private
     */
    Table.prototype._restoreOriginalColumnWidths = function() {

        if (this._bPreventRestoreColumWidths) {
            return;
        }

        this.retrieveLeafColumns()
            .slice(this.getTotalFrozenColumnCount())
            .forEach(function(oCol) {
                if (oCol._restoreOriginalWidth) {
                    oCol._restoreOriginalWidth();
                }
            });
    };

    /**
     * @private
     */
    Table.prototype._cacheRow = function(oRow) {
        var sCacheKey = oRow.getId().split("-").pop();
        if (this._mCachedRows[sCacheKey] !== oRow) {
            this._mCachedRows[sCacheKey].destroy();
        }
        this._mCachedRows[sCacheKey] = oRow;
    };

    /**
     * @private
     */
    Table.prototype._getRowTemplateClone = function(iRowNumber) {
        var sCacheKey = "row" + iRowNumber;
        if (!this._mCachedRows[sCacheKey]) {
            var oRowTemplate = this._getRowTemplate();
            var oClone = oRowTemplate.clone("row" + iRowNumber);
            this._mCachedRows[sCacheKey] = oClone;
        }

        return this._mCachedRows[sCacheKey];
    };

    /**
     * creates the rows for the rows aggregation
     * @private
     */
    Table.prototype._adjustRows = function(iNumberOfRows, bNoUpdate) {
        var bChanged = false;

        if (!isNaN(iNumberOfRows)) {
            // Create one additional row, for half-scrolled rows at the bottom.
            if (TableUtils.isVariableRowHeightEnabled(this)) {
                iNumberOfRows = iNumberOfRows + 1;
            }

            var i;
            var aRows = this.getRows();
            // destroy the rows aggregation if any exist but there is no current row template (it has changed)
            if (!this._oRowTemplate && aRows.length > 0) {
                this.destroyAggregation("rows", true);
                aRows = [];
            }

            // check if we have the number of rows that we want
            if (iNumberOfRows !== aRows.length) {

                // set up a counter to allow us to break out of adjustRows infinite loops
                if (this.hasOwnProperty("_iAdjustRowsCount") === false) {
                    this._iAdjustRowsCount = 0;
                }

                // increment our counter
                this._iAdjustRowsCount++;

                // give the routine a few tries to get it right
                if (this._iAdjustRowsCount < 4) {

                    // remove rows from aggregation if they are not needed anymore required
                    for (i = aRows.length - 1; i >= iNumberOfRows; i--) {
                        this._cacheRow(this.removeAggregation("rows", i, true));
                    }

                    if (TableUtils.isVariableRowHeightEnabled(this)) {
                        // One additional row was created for half-scrolled rows at the bottom.,
                        // this should not lead to a increase of the visibleRowCount defined by the user.
                        this.setProperty("visibleRowCount", iNumberOfRows - 1, true);
                    } else {
                        this.setProperty("visibleRowCount", iNumberOfRows, true);
                    }

                    // this call might cause the cell (controls) to invalidate theirself and therefore also the table. It should be
                    // avoided to rerender the complete table since rendering of the rows is handled here. All child controls get
                    // rendered.
                    this._ignoreInvalidateOfChildControls = true;
                    var aContexts;
                    var iFirstVisibleRow = this.getFirstVisibleRow();
                    var iAbsoluteRowIndex = 0;
                    var oBindingInfo;
                    var oBinding = this.getBinding("rows");

                    if (!bNoUpdate) {
                        // set binding contexts for known rows
                        oBindingInfo = this.getBindingInfo("rows");
                        aContexts = this._getRowContexts(iNumberOfRows);

                        for (i = 0; i < aRows.length; i++) {
                            iAbsoluteRowIndex = iFirstVisibleRow + i;
                            this._updateRowBindingContext(aRows[i], aContexts[i], oBindingInfo && oBindingInfo.model, iAbsoluteRowIndex, oBinding);
                        }
                    }

                    if (aRows.length < iNumberOfRows) {

                        for (i = aRows.length; i < iNumberOfRows; i++) {
                            // add new rows and set their binding contexts in the same run in order to avoid unnecessary context
                            // propagations.
                            var oClone = this._getRowTemplateClone(i);
                            if (!bNoUpdate) {
                                iAbsoluteRowIndex = iFirstVisibleRow + i;
                                this._updateRowBindingContext(oClone, aContexts[i], oBindingInfo && oBindingInfo.model, iAbsoluteRowIndex, oBinding);
                            }
                            this.addAggregation("rows", oClone, true);
                        }
                    }
                    this._ignoreInvalidateOfChildControls = false;

                    aRows = this.getRows();
                    bNoUpdate = bNoUpdate || aContexts.length === 0;
                    bChanged = this._insertTableRows(aRows, bNoUpdate);
                } else {
                    logger.debug("AdjustRows loop detected");
                }
            }
        }

        // if there was no change to the rows, then we've successfully cleared the adjustRows process and can reset our counter
        if (!bChanged) {
            delete this._iAdjustRowsCount;
        }

        return bChanged;
    };

    /**
     * Insert table rows into DOM.
     *
     * @param {sas.hc.ui.table.Row[]} [aRows] Rows aggregation to be rendered.
     * @param {Number} [iMaxRowCount] Maximum amount of row to be rendered.
     * @private
     */
    Table.prototype._insertTableRows = function(aRows, bNoUpdate) {
        var bReturn = false;
        if (!this._bInvalid) {
            this._detachEvents();

            var oTBody = this.getDomRef("tableCCnt");
            aRows = aRows || this.getRows();
            if (!aRows.length || !oTBody) {
                return;
            }

            if (this.getVisibleRowCountMode() === VisibleRowCountMode.Auto) {
                var oDomRef = this.getDomRef();
                if (oDomRef) {
                    oDomRef.style.height = "0px";
                }
            }

            // make sure to call rendering event delegates even in case of DOM patching
            var oEvent = jQuery.Event("BeforeRendering");
            oEvent.setMarked("insertTableRows");
            oEvent.srcControl = this;
            this._handleEvent(oEvent);

            var oRM = new sap.ui.getCore().createRenderManager(),
                oRenderer = this.getRenderer();

            this._iDefaultRowHeight = undefined;
            oRenderer.renderTableCCnt(oRM, this);
            oRM.flush(oTBody, false, false);
            oRM.destroy();

            // make sure to call rendering event delegates even in case of DOM patching
            oEvent = jQuery.Event("AfterRendering");
            oEvent.setMarked("insertTableRows");
            oEvent.srcControl = this;
            this._handleEvent(oEvent);

            // since the row is an element it has no own renderer. Anyway, logically it has a domref. Let the rows
            // update their domrefs after the rendering is done. This is required to allow performant access to row domrefs
            this._initRowDomRefs();
            this._getKeyboardExtension().invalidateItemNavigation();

            // restore the column icons
            var aCols = this.retrieveLeafColumns();
            for (var i = 0, l = aCols.length; i < l; i++) {
                if (aCols[i].getVisible() && aCols[i]._restoreIcons) {
                    aCols[i]._restoreIcons();
                }
            }

            this._updateTableSizes();

            bReturn = true;

            this._updateTableContent();
            this._attachEvents();
        }

        if (!bNoUpdate && !this._bInvalid && this.getBinding("rows")) {
            var that = this;
            if (this._mTimeouts._rowsUpdated) {
                window.clearTimeout(this._mTimeouts._rowsUpdated);
            }
            this._mTimeouts._rowsUpdated = window.setTimeout(function() {
                that.fireEvent("_rowsUpdated");
            }, 0);
        }

        return bReturn;
    };

    /**
     * Determines the default row height, based upon the height of the row template.
     * @private
     */
    Table.prototype._getDefaultRowHeight = function(aRowHeights) {
        if (TableUtils.isVariableRowHeightEnabled(this)) {
            this._iDefaultRowHeight = this.getRowHeight() || 28;
        } else {
            if (!this._iDefaultRowHeight && this.getDomRef()) {
                aRowHeights = aRowHeights || this._collectRowHeights();
                if (aRowHeights && aRowHeights.length > 0) {
                    this._iDefaultRowHeight = aRowHeights[0];
                }
            }

            if (!this._iDefaultRowHeight) {
                this._iDefaultRowHeight = 28;
            }
        }

        return this._iDefaultRowHeight;
    };

    /**
     * Determines and sets the height of tableCtrlCnt based upon the VisibleRowCountMode and other conditions.
     * @param iHeight
     * @private
     */
    Table.prototype._setRowContentHeight = function(iHeight) {
        iHeight = iHeight || 0;
        var sVisibleRowCountMode = this.getVisibleRowCountMode();
        var iVisibleRowCount = this.getVisibleRowCount();
        var iMinVisibleRowCount = this.getMinAutoRowCount();
        var iMinHeight;

        var iDefaultRowHeight = this._getDefaultRowHeight();
        if (sVisibleRowCountMode === VisibleRowCountMode.Fixed) {
            iMinHeight = iVisibleRowCount * iDefaultRowHeight;
            iHeight = iMinHeight;
        } else if (sVisibleRowCountMode === VisibleRowCountMode.Auto) {
            iMinHeight = iMinVisibleRowCount * iDefaultRowHeight;
        }

        var iRowContentHeight = Math.max(iHeight, iMinHeight);
        if ((sVisibleRowCountMode === VisibleRowCountMode.Fixed && this.getRows().length === 0) || sVisibleRowCountMode !== VisibleRowCountMode.Fixed) {
            // when visibleRowCountMode is fixed, the content height is only required to be set if there are no rows. If rows are already created, the height
            // is implicitly controlled by the total of row heights
            this._iTableRowContentHeight = Math.floor(iRowContentHeight / iDefaultRowHeight) * iDefaultRowHeight;
        } else {
            this._iTableRowContentHeight = undefined;
        }

        if (TableUtils.isVariableRowHeightEnabled(this)) {
            var $tableCCnt = jQuery(this.getDomRef("tableCCnt"));
            if (sVisibleRowCountMode === VisibleRowCountMode.Fixed) {
                $tableCCnt.css("height", this._getDefaultRowHeight() * this.getVisibleRowCount() + "px");
            } else if (sVisibleRowCountMode === VisibleRowCountMode.Auto) {
                $tableCCnt.css("height", this._iTableRowContentHeight + "px");
            }
        } else {
            if (sVisibleRowCountMode === VisibleRowCountMode.Fixed && this.getRows().length > 0) {
                jQuery(this.getDomRef("tableCtrlCnt")).css("height", "auto");
            } else {
                jQuery(this.getDomRef("tableCtrlCnt")).css("height", this._iTableRowContentHeight + "px");
            }
        }
    };

    /**
     * Determines the minimal row count for rowCountMode "auto".
     * @private
     */
    Table.prototype._determineMinAutoRowCount = function() {
        var iVisibleRowCount = this.getVisibleRowCount();
        var iMinAutoRowCount = this.getMinAutoRowCount();
        var iMinRowCount = iMinAutoRowCount || iVisibleRowCount || 5;
        return iMinRowCount;
    };

    /**
     * Calculates the maximum rows to display within the table.
     * @private
     */
    Table.prototype._calculateRowsToDisplay = function(iTableRowContentHeight) {
        iTableRowContentHeight = iTableRowContentHeight || this._iTableRowContentHeight;
        var sVisibleRowCountMode = this.getVisibleRowCountMode();
        var iCalculatedRowsToDisplay = 0;
        if (sVisibleRowCountMode === VisibleRowCountMode.Fixed) {
            // at least one row must be rendered in a table
            iCalculatedRowsToDisplay = this.getVisibleRowCount() || 0;
        } else if (sVisibleRowCountMode === VisibleRowCountMode.Auto) {
            var iMinAutoRowCount = this._determineMinAutoRowCount();
            var iDefaultRowHeight = this._getDefaultRowHeight();
            if (!iDefaultRowHeight || !iTableRowContentHeight) {
                iCalculatedRowsToDisplay = iMinAutoRowCount;
            } else {
                // Make sure that table does not grow to infinity
                // Maximum height of the table is the height of the window minus two row height, reserved for header and footer.
                var iAvailableSpace = Math.min(iTableRowContentHeight, 50000);
                // the last content row height is iRowHeight - 1, therefore + 1 in the formula below:
                // to avoid issues with having more fixed rows than visible row count, the number of visible rows must be
                // adjusted.
                var iRowCount = Math.floor(iAvailableSpace / iDefaultRowHeight);
                iCalculatedRowsToDisplay = Math.max((this.getFixedRowCount() + this.getFixedBottomRowCount() + 1), Math.max(iMinAutoRowCount, iRowCount));
            }
        }

        return Math.max(iCalculatedRowsToDisplay, 0);
    };

    /*
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setShowNoData = function(bShowNoData) {
        this.setProperty('showNoData', bShowNoData, true);
        this._updateNoData();
        return this;
    };

    /**
     * Creates a new {@link sap.ui.core.util.Export} object and fills row/column information from the table if not provided. For the cell content, the column's "sortProperty" will be used (experimental!)
     *
     * <p><b>Please note: The return value was changed from jQuery Promises to standard ES6 Promises.
     * jQuery specific Promise methods ('done', 'fail', 'always', 'pipe' and 'state') are still available but should not be used.
     * Please use only the standard methods 'then' and 'catch'!</b></p>
     *
     * @param {object} [mSettings] settings for the new Export, see {@link sap.ui.core.util.Export} <code>constructor</code>
     * @return {Promise} Promise object
     *
     * @experimental Experimental because the property for the column/cell definitions (sortProperty) could change in future.
     * @public
     */
    Table.prototype.exportData = function(mSettings) {
        var Export = sap.ui.requireSync("sap/ui/core/util/Export");

        mSettings = mSettings || {};

        if (!mSettings.rows) {
            var oBinding = this.getBinding("rows"),
                oBindingInfo = this.getBindingInfo("rows");

            var aFilters = oBinding.aFilters.concat(oBinding.aApplicationFilters);

            mSettings.rows = {
                path: oBindingInfo.path,
                model: oBindingInfo.model,
                sorter: oBinding.aSorters,
                filters: aFilters,
                parameters: oBindingInfo.parameters
            };
        }

        // by default we choose the export type CSV
        if (!mSettings.exportType) {
            var ExportTypeCSV = sap.ui.requireSync("sap/ui/core/util/ExportTypeCSV");
            mSettings.exportType = new ExportTypeCSV();
        }

        var sModelName = mSettings.rows.model;
        if (!sModelName) {
            // if a model separator is found in the path, extract model name from there
            var sPath = mSettings.rows.path;
            var iSeparatorPos = sPath.indexOf(">");
            if (iSeparatorPos > 0) {
                sModelName = sPath.substr(0, iSeparatorPos);
            }
        }

        if (!mSettings.columns) {
            mSettings.columns = [];

            var aColumns = this.retrieveLeafColumns();
            for (var i = 0, l = aColumns.length; i < l; i++) {
                var oColumn = aColumns[i];
                if (oColumn.getSortProperty && oColumn.getSortProperty()) {
                    mSettings.columns.push({
                        name: oColumn.getLabel().getText(),
                        template: {
                            content: {
                                path: oColumn.getSortProperty(),
                                model: sModelName
                            }
                        }
                    });
                }
            }
        }

        var oExport = new Export(mSettings);
        this.addDependent(oExport);

        return oExport;
    };

    /**
     * no-op. legacy of old sap code
     * @private
     */
    Table.prototype._toggleSelectAll = function() {
        // NO-OP
        // This is handled by sas.hc.ui.table.SelectAllControl
    };

    /**
     *
     * @private
     */
    Table.prototype._restoreAppDefaultsColumnHeaderSortFilter = function () {
        var aColumns = this.retrieveLeafColumns();
        jQuery.each(aColumns, function(iIndex, oColumn){
            if (oColumn._restoreAppDefaults) {
                oColumn._restoreAppDefaults();
            }
        });
    };

    /**
     *
     * @param mParameters
     * @private
     */
    Table.prototype._setBusy = function (mParameters) {
        var oBinding,
            i,
            bSetBusy;

        if (!this.getEnableBusyIndicator() || !this._bBusyIndicatorAllowed) {
            return;
        }

        oBinding = this.getBinding("rows");
        if (!oBinding) {
            return;
        }

        this.setBusy(false);
        if (mParameters && this._iDataRequestedCounter > 0) {
            var sReason = mParameters.reason;
            if (mParameters.contexts && mParameters.contexts.length !== undefined) {
                // TreeBinding and AnalyticalBinding always return a contexts array with the
                // requested length. Both put undefined in it for contexts which need to be loaded
                // Check for undefined in the contexts array.
                bSetBusy = false;
                for (i = 0; i < mParameters.contexts.length; i++) {
                    if (mParameters.contexts[i] === undefined) {
                        bSetBusy = true;
                        break;
                    }
                }
            } else if (mParameters.changeReason === ChangeReason.Expand) {
                this.setBusy(true);
            }

            var iLength = oBinding.getLength();
            if ((sReason === ChangeReason.Expand && this._iDataRequestedCounter !== 0) || bSetBusy || (oBinding.isInitial()) || (mParameters.receivedLength === 0 && this._iDataRequestedCounter !== 0) ||
                (mParameters.receivedLength < mParameters.requestedLength && mParameters.receivedLength !== iLength &&
                 mParameters.receivedLength !== iLength - this.getFirstVisibleRow())) {
                this.setBusy(true);
            }
        }
    };

    /**
     * @override
     */
    Table.prototype.setBusy = function (bBusy, sBusySection) {
        var bBusyChanged = this.getBusy() !== bBusy;

        sBusySection = "sapUiTableCnt";
        var vReturn = Control.prototype.setBusy.call(this, bBusy, sBusySection);
        if (bBusyChanged) {
            this.fireBusyStateChanged({busy: bBusy});
        }
        return vReturn;
    };

    /*
     * Prevents re-rendering, when enabling/disabling busy indicator.
     * Avoids the request delays.
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.setEnableBusyIndicator = function (bValue) {
        this.setProperty("enableBusyIndicator", bValue, true);
    };

    /**
     * @private
     */
    Table.prototype._attachDataRequestedListeners = function () {
        var oBinding = this.getBinding("rows");
        if (oBinding) {
            oBinding.detachDataRequested(this._onBindingDataRequestedListener, this);
            oBinding.detachDataReceived(this._onBindingDataReceivedListener, this);
            this._iDataRequestedCounter = 0;
            oBinding.attachDataRequested(this._onBindingDataRequestedListener, this);
            oBinding.attachDataReceived(this._onBindingDataReceivedListener, this);
        }
    };

    /**
     * @private
     */
    Table.prototype._onBindingDataRequestedListener = function (oEvent) {
        if (oEvent.getSource() === this.getBinding("rows") && !oEvent.getParameter("__simulateAsyncAnalyticalBinding")) {
            this._iDataRequestedCounter++;
        }
    };

    /**
     * @private
     */
    Table.prototype._onBindingDataReceivedListener = function (oEvent) {
        if (oEvent.getSource() === this.getBinding("rows") && !oEvent.getParameter("__simulateAsyncAnalyticalBinding")) {
            this._iDataRequestedCounter--;
        }
    };

    /**
     * @private
     */
    Table.prototype._attachBindingListener = function() {
        this._attachDataRequestedListeners();
    };

    /**
     * Lets you control in which situation the <code>ScrollBar</code> fires scroll events.
     *
     * @param {boolean} bLargeDataScrolling Set to true to let the <code>ScrollBar</code> only fires scroll events when
     * the scroll handle is released. No matter what the setting is, the <code>ScrollBar</code> keeps on fireing scroll events
     * when the user scroll with the mousewheel or using touch
     * @private
     */
    Table.prototype._setLargeDataScrolling = function(bLargeDataScrolling) {
        this._bLargeDataScrolling = !!bLargeDataScrolling;
    };

    /**
     * Retrieves the number of selected entries.
     * @private
     */
    Table.prototype._getSelectedIndicesCount = function () {
        return this.getSelectedIndices().length;
    };

    Table.prototype._isSomeRowSelected = function() {
        return this._getSelectedIndicesCount() > 0;
    };

    /**
     * @private
     */
    Table.prototype._updateTableContent = function() {
        // TODO: can we remove this?
    };

    /**
     * Return the PointerExtension type
     * For consumers to override
     * @prototype
     * @protected
     */
    Table.prototype._getPointerExtensionType = function() {
        return TablePointerExtension;
    };

    /**
     * Return the KeyboardExtension type
     * For consumers to override
     * @prototype
     * @protected
     */
    Table.prototype._getKeyboardExtensionType = function() {
        return TableKeyboardExtension;
    };

    /**
     * Return the appropriate DOM Ref for the 'Select All' feature
     * @protected
     */
    Table.prototype._getItemNavSelectAllDomRef = function() {
        // return the select-all checkbox control's root div
        return this.getDomRef("selectAll") ||
            // ...or the blank div (if for example, showSelectionControls === false)
            this.$().find('.sapUiTableColRowHdr').get(0);
    };

    /**
     * Return the appropriate DOM Refs for the row headers
     * @protected
     */
    Table.prototype._getItemNavRowHdrDomRefs = function() {
        // return the row header control's root divs
        var aRowHdrsWithControls = this.$().find(".sapUiTableRowHdr > div").get();
        if (aRowHdrsWithControls.length > 0) {
            return aRowHdrsWithControls;
        }

        // ...or the blank row header divs (if for example, showSelectionControls === false)
        return this.$().find(".sapUiTableRowHdr").get();
    };

    /**
     * Hook function for _syncColumnHeaders
     * Sets the height of the column header container, but only the regular header - excludes the summary column header
     * @private
     */
    Table.prototype._syncColumnHeaderHeight = function(oDomRef, iColumnRowHeight) {
        var oColHdrCnt = oDomRef.querySelector(".sapUiTableColHdrCnt:not(.sasUiTableSumColHdrCnt)");
        // this calculation was changed as part of HTMLCOMMONS-8429
        // previously, it was calculated by multiplying the column row height by the header row count
        // now, due to css changes, we simply get the height of the table
        // this seems to work well, but if we find problems in the future, we may want
        // to revisit this calculation
        if (oColHdrCnt) {
            oColHdrCnt.style.height = $(oColHdrCnt).find(".sapUiTableCtrl").height() + "px";
        }
    };

    /**
     * Hook function for onmousedown.
     * Determine if oColumn can be moved
     * @override
     * @protected
     * @param {object} oColumn The column to potentially move
     * @return {boolean} whether the column is allowed to move
     */
    Table.prototype._allowColumnMove = function(oColumn) {
        return oColumn && !this.isFrozenColumn(oColumn) && !oColumn.getMandatory();
    };

    /**
     * Create a new Row object to use as template for all rows rendered to the table.
     * @returns {sas.hc.ui.table.Row}
     * @protected
     */
    Table.prototype._createRowTemplate = function(sId) {
        sId = sId || (this.getId() + "-rows");  //default value is based on Table's ID
        var oTemplate = new Row(sId);
        //assign a template selection control (if desired)
        oTemplate.setSelectionControl(this._createSelectionControlTemplate(sId));

        return oTemplate;
    };

    /**
     * Create a new selection control to use as template for all row headers rendered to the table.
     * @param sId optional ID
     * @returns {sas.hc.ui.table.Row}
     * @private
     */
    Table.prototype._createSelectionControlTemplate = function(sId) {
        var oSelectionControl;

        sId = sId || this.getId();  //default to table's ID

        //selection event handler for selection controls
        function _onSelect(oEvent) {
            var
            // the original jQuery event
                $jQueryEvent = oEvent.getParameter("jQueryEvent"),
                bShift = false, bCtrl = false,

                // the source of the event (the checkbox)
                oSource = oEvent.getSource(),
                // the row the selection control belongs to
                oRow = oSource.getParent(),
                oTable,
                iIndex;

            // were modifier keys pressed
            if ($jQueryEvent) {
                bShift = $jQueryEvent.shiftKey;
                bCtrl = $jQueryEvent.ctrlKey;
            }

            if (!!oRow) {
                oTable = oRow.getParent();
                if (!!oTable) {
                    iIndex = oRow.getIndex();
                    oTable._onRowSelect(iIndex, bShift, bCtrl);
                }
            }
        }

        oSelectionControl = new CheckBox({
            id: sId + "-selectionControl",
            select: _onSelect
        });

        return oSelectionControl;
    };

    /**
     * Factory method. Creates the ItemNavigationDelegate instance.
     * @protected
     * @return {object} ItemNavigationDelegate instance
     */
    Table.prototype._createItemNavigation = function () {
        return new ItemNavigation();
    };

    /**
     * Factory method. Creates the TableKeyboardDelegate instance.
     * @param {string} sTableType the type of table
     * @protected
     * @return {object} TableKeyboardDelegate instance
     */
    TableKeyboardExtension.prototype._createTableKeyboardDelegate = function(sTableType) {
        return new TableKeyboardDelegate(sTableType);
    };

    /**
     * Used by onColumnMoveStart, onColumnMove, etc
     * Returns the DOM of the column to be used when creating the ghost for drag reordering columns
     * @param oColumn
     * @returns {*}
     * @private
     */
    Table.prototype._getColumnDomRefForMove = function(oColumn) {
        return (this.getColumnHeaderVisible() === false && this.getShowSummaryHeader() === true) ? oColumn.$("shead") : oColumn.$();
    };

    /**
     * Override the base behavior to reset first visible row and selection
     * @param {object} oNewModel the new model to assign
     * @param {string} sName the name for the new model
     * @public
     */
    Table.prototype.setModel = function (oNewModel, sName) {
        var oOldModel = this.getModel(sName);

        //let super do it's job
        if (Control.prototype.setModel) {
            Control.prototype.setModel.apply(this, arguments);
        }

        if (!!oOldModel && oOldModel !== oNewModel) {  //if changing models, reset
            //S1173785: we need to reset first visible row
            this.setFirstVisibleRow(0); //back to the top
            this.clearSelection();      //can't guarentee the old selection will apply
            this.clearSelectedKeys("setModel"); //clear remembered paths
        }

        return this;
    };

    /**
     * Add a callback to provide custom table options.
     *
     * @param {function} fnCallback callback function to set up the table options.
     * @name sas.hc.ui.table.Table.prototype.addTableOptionsCallback
     * @function
     * @public
     * @return {object} sas.hc.ui.table.Table <code>this</code> to allow method chaining.
     */
    Table.prototype.addTableOptionsCallback = function (fnCallback) {
        var oTableOptionsMenuItemCallback;
        if (typeof fnCallback === "function") {
            if (!this.fnTableOptionsCallback) {
                oTableOptionsMenuItemCallback = new sasHcMMenuItem({
                    id: this.getId() + "-menuitem-tableoptions",
                    text: this.oResBundle.getText("table.tableOptions.optionsMenu.tableOptionsMenuItem.label")                });
                this.addMenuItem(oTableOptionsMenuItemCallback, true);
            }

            this.fnTableOptionsCallback = fnCallback;
            if (oTableOptionsMenuItemCallback) {
                oTableOptionsMenuItemCallback.attachPress(this.fnTableOptionsCallback);
            }

            logger.info("Table options menu item has been added.");
        } else {
            if (this.fnTableOptionsCallback) {
                this.fnTableOptionsCallback = null;
            }
        }
        return this;
    };

    /**
     * Adds the Resize all columns to fit menu
     * @private
     */
    Table.prototype.addResizeToFit = function () {
        if (!this._tableResizeToFitMenuItem) {
            var sId = this.getId();
            this._tableResizeToFitMenuItem = new sasHcMMenuItem({
                id: sId + "-menuitem-columns-resize",
                text: this.oResBundle.getText("table.tableOptions.optionsMenu.columnsResizeMenuItem.label"),
                press: jQuery.proxy(this.autoFitColumns, this)
            });
            this.addMenuItem(this._tableResizeToFitMenuItem, true);
        }
        return this;
    };

    /**
     * @see JSDoc generated by SAPUI5 control API generator
     */
    Table.prototype.addMenuItem = function (oMenuItem, bSuppressInvalidate) {
        var aMenuItems,
            oTableOptionsMenuButton = this.getAggregation("tableOptionsMenuButton");

        this.addAggregation("menuItems", oMenuItem, bSuppressInvalidate);
        aMenuItems = this.getMenuItems();
        if (oTableOptionsMenuButton) {
            if (aMenuItems && aMenuItems.length === 1 && aMenuItems[0].getId() === (this.getId() + "-menuitem-columns")) {
                oTableOptionsMenuButton.setTooltip(this.oResBundle.getText("table.tableOptions.optionsMenu.columnsMenuItem.label"));
            } else {
                oTableOptionsMenuButton.setTooltip(this.oResBundle.getText("table.tableOptions.optionsMenu.tip"));
            }
        }
        return this;
    };

    /**
     * Return the table options' menu.
     *
     * @name sas.hc.ui.table.Table.prototype.getTableOptionsMenu
     * @public
     * @deprecated since 9.0
     * @return {object} as.hc.ui.unified.Menu
     */
    Table.prototype.getTableOptionsMenu = function () {
        sas.log.warning("sas.hc.ui.table.Table.getTableOptionsMenu is deprecated.  Use sas.hc.ui.table.Table menuItems aggregation instead.");
        var oTableOptionsMenuButton = this.getAggregation("tableOptionsMenuButton");

        if (oTableOptionsMenuButton) {
            return oTableOptionsMenuButton.getMenu();
        }
        return null;
    };

    /**
     * minColumnWidth setter override.
     * Set's the internal _iColMinWidth variable (used at the sap level)
     * Avoids invalidation
     * @param {integer} iMinColWidth The minColWidth property value
     * @public
     */
    Table.prototype.setMinColumnWidth = function (iMinColWidth) {
        this.setProperty('minColumnWidth', iMinColWidth);
        this._iColMinWidth = iMinColWidth;
        return this;
    };

    /**
     * Return the the cell domRef for (iCol, iRow)
     * @param {integer} iCol The visible column index
     * @param {integer} iRow The visible row index
     * @public
     * @return {object} The requested dom node
     */
    Table.prototype.getCellDom = function (iCol, iRow) {
        var oCellDom,
            $cellDom,
            $cellInner = jQuery("[data-sap-ui$='-col" + iCol + "-row" + iRow + "']", this.getDomRef());

        // first grab the dom ref that is the actual content of the cell ($cellInner)
        // then, get it's enclosing cellref ($cellDom)
        // return a dom node, not a jQuery object

        if ($cellInner) {
            $cellDom = $cellInner.closest('.sapUiTableCell');
        }

        if ($cellDom && $cellDom.length > 0) {
            oCellDom = $cellDom.get(0);
        }

        return oCellDom;
    };

    /**
     * For the specified column, render the ColumnDragBar control
     * This allows mobile-friendly column resizing and moving.
     *
     * @param {integer} iColIndex the index (0-based) of the column to show the drag bar for
     * @public
     */
    Table.prototype.showColumnDragBar = function (iColIndex) {
        var self = this,
            bRTL = this._bRtlMode,
            oColumnDragBar,
            oRenderManager = sap.ui.getCore().createRenderManager(),
            iMinColumnWidth = this.getMinColumnWidth(),
            $column = this.retrieveLeafColumns()[iColIndex].$();

        // remove the old drag bar if it exists
        this._cleanupColumnDragBar();

        // create the drag bar
        // when the drag handle receives touch events
        // re-use the sap resize functions that table currently uses for mouse resizing
        oColumnDragBar = new ColumnDragBar({
            resizeHandleDragStart: function (e) {
                self._iColumnResizeStart = e.getParameter("clientX");
                self._disableTextSelection();
                self._$colResize = e.getParameter("$closestRsz");
            },
            resizeHandleDragMove: function (e) {
                var iColumnBoundaryDifference = e.getParameter("columnBoundaryDifference"),
                    iClientX = e.getParameter("clientX"),
                    iColResizeLeft = self._$colResize.offset().left;

                function isTouchPointPastColumn() {
                    if (bRTL === true) {
                        return iClientX < iColResizeLeft;
                    } else {
                        return iClientX > iColResizeLeft;
                    }
                }

                // if resize width is not too narrow OR TODO:
                if ((iColumnBoundaryDifference > iMinColumnWidth) || (isTouchPointPastColumn() === true)) {
                    self._onColumnResize({
                        pageX: iClientX
                    });
                }

                jQuery(".sapUiTableColRszActive").addClass("sapUiTableColRszPointer");
            },
            resizeHandleDragEnd: function (e) {
                self._onColumnResized();
                jQuery(".sapUiTableColRszActive").removeClass("sapUiTableColRszPointer");
            },
            focusLost: function () {
                self._cleanupColumnDragBar();

                self._setManagedTimeout(function () {
                    self._syncHeaderAndContent(self._collectTableSizes());
                }, 0);
            }
        });

        this.highlightColumn(iColIndex);

        // render the drag bar
        oRenderManager.render(oColumnDragBar, $column);

        // prevent invalidation
        this.setAggregation("columnDragBar", oColumnDragBar, true);

        this._setManagedTimeout(function () {
            self._syncHeaderAndContent(self._collectTableSizes());
        }, 0);
    };

    /**
     * showSelectionControls setter override
     * Castes to string if necessary
     * Avoids invalidation
     * @param {string} sShowSelectionControls the property value
     * @public
     */
    Table.prototype.setShowSelectionControls = function (sShowSelectionControls) {
        if (sShowSelectionControls !== undefined && sShowSelectionControls !== null) {
            this.setProperty("showSelectionControls", sShowSelectionControls.toString());
        }
        return this;
    };

    /**
     * Render the next visible page of rows.
     * @public
     * @param {function} fnCallback invoke after operation
     */
    Table.prototype.displayNextPage = function(fnCallback) {
        var iDeterminedFirstVisibleRow, iMaxFirstVisibleRow,
            oLastVisibleRow = this._determineLastVisibleRow();

        if (!oLastVisibleRow) {
            fnCallback();
            return;
        }

        if (typeof fnCallback === 'function') {
            this.attachEventOnce("_rowsUpdated", fnCallback);
        }

        // to avoid errors related to the px scrolling feature,
        // only allow setting to the 2nd to last row
        iMaxFirstVisibleRow = this._getRowCount() - 1;

        // make sure we actually change the firstVisibleRow property
        iDeterminedFirstVisibleRow = Math.min(oLastVisibleRow.getIndex(), iMaxFirstVisibleRow);
        // if firstVisibleRow property is already set to what we determine, just increment by 1 to avoid the no-op
        if (iDeterminedFirstVisibleRow === this.getFirstVisibleRow()) {
            iDeterminedFirstVisibleRow = Math.min(iDeterminedFirstVisibleRow + 1, iMaxFirstVisibleRow);
        }

        this._skipVScroll = true;
        this.setFirstVisibleRow(iDeterminedFirstVisibleRow);
    };

    /**
     * Render the previous visible page of rows.
     * @public
     * @param {function} fnCallback invoke after operation
     */
    Table.prototype.displayPreviousPage = function(fnCallback) {
        var self = this,
            bStartedAtBottom = this._isLastPageLoaded(),
            iOrigFirstVisibleRowIndex,
            oOrigFirstVisibleRow = this._determineFirstVisibleRow();

        if (!oOrigFirstVisibleRow) {
            if (typeof fnCallback === 'function') {
                fnCallback();
            }
            return;
        }

        // store original first visible row index
        iOrigFirstVisibleRowIndex = oOrigFirstVisibleRow.getIndex();

        function scrollUpOneRow() {
            var iDeterminedFirstVisibleRow,
                oLastFullyVisibleRow = self._determineLastFullyVisibleRow(),
                oVSb;

            if (self._isScrolledToTop() === true || iOrigFirstVisibleRowIndex > oLastFullyVisibleRow.getIndex()) {

                // if scrolling upwards went too far
                // (the original top visible row is no longer visible)
                // then, just scroll up by a single row
                if (bStartedAtBottom === false && self._isRowIndexVisible(iOrigFirstVisibleRowIndex) === false) {
                    self._skipVScroll = true;
                    self.setFirstVisibleRow(Math.max(iOrigFirstVisibleRowIndex - 1, 0));
                }

                if (typeof fnCallback === 'function') {
                    fnCallback();
                }
                return;
            } else if (!self._isScrolledToTop() && self.getFirstVisibleRow() === 0) {
                oVSb = self.getDomRef(SharedDomRef.VerticalScrollBar);
                oVSb.scrollTop = 0;
                self._adjustTablePosition();
                if (typeof fnCallback === 'function') {
                    fnCallback();
                }
                return;
            }

            self.attachEventOnce("_rowsUpdated", scrollUpOneRow);

            // decrement first visible row
            // make sure we actually change the firstVisibleRow property (decrement by 1 if not)
            iDeterminedFirstVisibleRow = Math.max(self.getFirstVisibleRow() - 1, 0);
            if (iDeterminedFirstVisibleRow === self.getFirstVisibleRow()) {
                iDeterminedFirstVisibleRow = Math.max(iDeterminedFirstVisibleRow - 1, 0);
            }

            self._skipVScroll = true;
            self.setFirstVisibleRow(iDeterminedFirstVisibleRow);
        }
        scrollUpOneRow();
    };


    // =============================================================================
    // HIGHLIGHTING
    // =============================================================================

    /**
     * Highlight the column with index iColIndex.
     * @param {integer} iColIndex The index of the column to highlight
     * Will remove highlighting from any other column
     * @public
     */
    Table.prototype.highlightColumn = function (iColIndex) {
        var oTargetColHeader,
            aRows = this.getRows(),
            sHighlightClass = this.getHighlightClass();

        // error check the input param
        if (iColIndex === undefined || iColIndex === null || isNaN(iColIndex) || iColIndex % 1 !== 0 || iColIndex < 0 || iColIndex >= this.retrieveLeafColumns().length) {
            sas.log.error(iColIndex + ' is not a valid column to highlight.');
            return;
        }

        if (!aRows) {
            sas.log.error('No rows found');
            return;
        }

        this.removeHighlighting();

        var aColumns = this.retrieveLeafColumns();

        if (aColumns[iColIndex].setHighlighted) {
            aColumns[iColIndex].setHighlighted(true);
        }

        // iterate through each cell, highlighting the cells that are in column iColIndex
        aRows.forEach(function (oRow) {
            oRow.getCells().forEach(function (oCell) {
                var iCol = oCell.getCustomData()[0].getValue();
                if (iCol === iColIndex) {
                    if (oCell.$()) {
                        oCell.$().closest('td').addClass(sHighlightClass);
                    }
                }
            });
        });

        // highlight the column header
        oTargetColHeader = this.retrieveLeafColumns()[iColIndex];
        jQuery(oTargetColHeader.getDomRef()).addClass(sHighlightClass);

        this._highlightedColIndex = iColIndex;
    };

    /**
     * Removes all column highlighting
     * @public
     */
    Table.prototype.removeHighlighting = function () {
        var $this = this.$(), sHighlightClass = this.getHighlightClass();
        $this.find('.sapUiTableCell').removeClass(sHighlightClass);
        $this.find('.sapUiTableCol').removeClass(sHighlightClass);
        $this.find('td').removeClass(sHighlightClass);

        var aColumns = this.retrieveLeafColumns();
        aColumns.forEach(function (oColumn) {
            if (typeof oColumn.setHighlighted === "function") {
                oColumn.setHighlighted(false);
            }
        });

        this._highlightedColIndex = -1;
    };

    /**
     * Wrapper around setTimeout.
     * Used so all timeout id's can be collected and managed
     * @param {function} fn function passed to setTimeout
     * @param {int} fn iDuration delay length
     * @returns {int} the setTimeout id
     * @private
     */
    Table.prototype._setManagedTimeout = function(fn, iDuration) {
        var self = this;
        var iTimeoutId = window.setTimeout(function() {
            fn.call(self);

            // if timeout completes naturally, remove from array
            self._aTimeoutIds = self._aTimeoutIds.filter(function(i) {
                return i !== iTimeoutId;
            });
        }, iDuration);

        this._aTimeoutIds.push(iTimeoutId);
        return iTimeoutId;
    };

    /**
     * Clear an individual setTimeout call.
     * Will keep this._aTimeoutIds in sync
     * @param {int} iTimeoutId the timeout to clear
     * @private
     */
    Table.prototype._clearTimeout = function(iTimeoutId) {
        window.clearTimeout(iTimeoutId);
        this._aTimeoutIds = this._aTimeoutIds.filter(function(i) {
            return i !== iTimeoutId;
        });
    };

    /**
     * Call this._clearTimeout on all stored timeout ids which will
     * * clearTimeout
     * * remove from managed timeout id array
     * @private
     */
    Table.prototype._clearTimeouts = function() {
        // this._clearTimeout will modify this._aTimeoutIds, so clone first
        this._aTimeoutIds.slice(0).forEach(this._clearTimeout.bind(this));
    };

    /**
     * Determine if the dimensions of the table have changed
     * Compare to previous calculations (saved as instance variables)
     * hook function for _checkTableSize
     * @protected
     * @override
     * @param {object} oParentDomRef The new dom ref
     * @param {int} iHeight The new height value
     * @param {int} iWidth The new width value
     * @return {boolean} whether the table's size has changed
     */
    Table.prototype._didTableSizeChange = function(oParentDomRef, iHeight, iWidth) {
        var iAllowedSizeDiff = this.getProperty("_allowedSizeDiff"); // in px
        function isWithin(num1, num2) {
            var absNum1 = Math.abs(num1),
                absNum2 = Math.abs(num2);
            return Math.abs(absNum1 - absNum2) <= iAllowedSizeDiff;
        }
        return isWithin(iHeight, this._lastParentHeight) === false || isWithin(iWidth, this._lastParentWidth) === false;
    };

    /**
     * Initialization of the Table Options Menu Button
     * @private
     */
    Table.prototype._initOptionsButton = function () {
        var sTableOptionsId = this.getId() + "-tableoptions",
            oTableOptionsMenuButton = new TableOptionsMenuButton(sTableOptionsId);

        this.setAggregation("tableOptionsMenuButton", oTableOptionsMenuButton);
    };

    /**
     * Destroy the drag bar
     * @private
     */
    Table.prototype._cleanupColumnDragBar = function () {
        this.removeHighlighting();
        var oColumnDragBar = this.getColumnDragBar();
        if (oColumnDragBar) {
            oColumnDragBar.destroy();
        }
    };

    /**
     * Put focus (activeElement) on cell
     * @private
     * @param {object} oDomRef dom reference of cell to put focus
     * @param {object} oEvent originating event
     */
    Table.prototype._focusItemByDomRef = function(oDomRef, oEvent) {
        var iItemNavIndex = this._getItemNavigation().getIndexFromDomRef(oDomRef);
        TableUtils.focusItem(this, iItemNavIndex, oEvent);
    };

    /**
     * If the table is vertically scrolled to the top row
     * @private
     * @returns {boolean} if top row is the very first row
     */
    Table.prototype._isScrolledToTop = function() {
        var oVSb = this.getDomRef(SharedDomRef.VerticalScrollBar);
        return !oVSb || oVSb.scrollTop === 0;
    };

    /**
     * If the table is vertically scrolled to the very bottom
     * @private
     * @returns {boolean} if vertically scrolled to the bottom
     */
    Table.prototype._isScrolledToBottom = function() {
        var oVSb = this.getDomRef(SharedDomRef.VerticalScrollBar);
        if (!oVSb) {
            return false;
        }
        var iOuterHeight = jQuery(oVSb).outerHeight();

        return (oVSb.scrollTop + iOuterHeight) >= oVSb.scrollHeight;
    };

    /**
     * Determine if the horizontal scrollbar is scrolled all the way to it's max limit
     * @private
     * @returns {boolean} if scrolled all the way
     */
    Table.prototype._isHorizontallyScrolledMaximally = function() {
        var $hsb = this.$(SharedDomRef.HorizontalScrollBar),
            $vsb = this.$(SharedDomRef.VerticalScrollBar),
            iDistanceScrolled = this._bRtlMode ? $hsb.scrollLeftRTL() : $hsb.scrollLeft();
        return (iDistanceScrolled + $hsb.outerWidth() + $vsb.outerWidth()) >= $hsb[0].scrollWidth;
    };

    /**
     * Determine if the horizontal scrollbar has not been scrolled at all
     * @private
     * @returns {boolean} if scrolled none of the way
     */
    Table.prototype._isHorizontallyScrolledMinimally = function() {
        var $hsb = jQuery(this.getDomRef(SharedDomRef.HorizontalScrollBar));
        var iDistanceScrolled = this._bRtlMode ? $hsb.scrollLeftRTL() : $hsb.scrollLeft();
        return iDistanceScrolled === 0;
    };

    /**
     * If last page of rows is currently active
     * @private
     * @returns {boolean} if last page of rows is on-screen
     */
    Table.prototype._isLastPageLoaded = function() {
        var oLastVisibleRow = this._determineLastVisibleRow();
        return oLastVisibleRow && oLastVisibleRow.getIndex() >= (this._getRowCount() - 1);
    };

    /**
     * Return column object that contains dom reference
     * @private
     * @param {object} oDomRef child dom reference of desired column
     * @returns {object} sas.hc.ui.table.Column instance whose rendering contains dom ref
     */
    Table.prototype._colFromRef = function(oDomRef) {
        var $domRef, oColumn, oTd, $colHeader, aColMatch;

        if (!oDomRef) {
            return;
        }

        $domRef = jQuery(oDomRef);

        oTd = $domRef.closest('td[role=gridcell]').get(0);
        if (oTd) {
            aColMatch = oTd.id.match(/-col(\d+)$/);
            if (aColMatch && aColMatch.length === 2) {
                oColumn = this.retrieveLeafColumns()[parseInt(aColMatch[1])];
            }
        } else {
            $colHeader = $domRef.closest('.sapUiTableCol');
            if ($colHeader.length === 1) {
                oColumn = this.retrieveLeafColumns()[parseInt($colHeader.attr('data-sap-ui-colindex'))];
            } else {
                // if we didn't find it going up the DOM tree, try going down
                // this is necessary for headers that span multiple rows
                $colHeader = $domRef.find('.sapUiTableCol');
                if ($colHeader.length === 1) {
                    oColumn = this.retrieveLeafColumns()[parseInt($colHeader.attr('data-sap-ui-colindex'))];
                }
            }
        }

        return oColumn;
    };

    /**
     * Return row object that contains dom reference
     * @private
     * @param {object} oDomRef child dom reference of desired row
     * @returns {object} sas.hc.ui.table.Row instance whose rendering contains dom ref
     */
    Table.prototype._rowFromRef = function(oDomRef) {
        var $ref = jQuery(oDomRef),
            oRow;

        // row header
        var $rowHdr = $ref.closest('.sapUiTableRowHdr');
        if ($rowHdr.attr('data-sap-ui-rowindex') !== undefined) {
            oRow = this.getRows()[parseInt($rowHdr.attr('data-sap-ui-rowindex'))];
        } else {
            // regular cell
            oRow = this.getRows()[parseInt($ref.closest("tr[data-sap-ui-rowindex]").attr('data-sap-ui-rowindex'))];
        }

        return oRow;
    };

    /**
     * Return the top-most rendered (perhaps partially) visible row.
     * @private
     * @returns {object} top on-screen row
     */
    Table.prototype._determineFirstVisibleRow = function() {
        var i = 0, oRow, aRows = this.getRows();
        for (; i < aRows.length; i++) {
            oRow = aRows[i];
            if (oRow.isFullyVisible() === true && (oRow.getIndex() < this._getRowCount())) {
                return oRow;
            }
        }
        return this.getRows()[0];
    };

    /**
     * Return the top-most rendered _fully_ visible row.<br />
     * Meaning that it's entire bounds exists within the Table's content container
     * @private
     * @returns {object} top on-screen row
     */
    Table.prototype._determineFirstFullyVisibleRow = function() {
        var i = 0, oRow, aRows = this.getRows();
        for (; i < aRows.length; i++) {
            oRow = aRows[i];
            if (oRow.isFullyVisible() === true && (oRow.getIndex() < this._getRowCount())) {
                return oRow;
            }
        }
        return this.getRows()[0];
    };

    /**
     * Return the bottom-most rendered (perhaps partially) visible row.
     * @private
     * @returns {object} bottom on-screen row
     */
    Table.prototype._determineLastVisibleRow = function() {
        var i = 0, oRow, aRows = this.getRows().reverse();
        for (; i < aRows.length; i++) {
            oRow = aRows[i];
            if (oRow.isVisible() === true && (oRow.getIndex() < this._getRowCount())) {
                return oRow;
            }
        }
        return this.getRows()[0];
    };

    /**
     * Return the bottom-most rendered _fully_ visible row.<br />
     * Meaning that it's entire bounds exists within the Table's content container
     * @private
     * @returns {object} bottom on-screen row
     */
    Table.prototype._determineLastFullyVisibleRow = function() {
        var i = 0, oRow, aRows = this.getRows().reverse();
        for (; i < aRows.length; i++) {
            oRow = aRows[i];
            if (oRow.isFullyVisible() === true && (oRow.getIndex() < this._getRowCount())) {
                return oRow;
            }
        }
        return this.getRows()[0];
    };

    /**
     * Determine if a particular row (which may or may not be currently loaded) is visible
     * @private
     * @param {int} iRowIndex The row to check
     * @returns {boolean} whether a row is currently visible
     */
    Table.prototype._isRowIndexVisible = function(iRowIndex) {
        var aRows = this.getRows(), i = 0;
        for (; i < aRows.length; i++) {
            if (aRows[i].getIndex() === iRowIndex && aRows[i].isVisible() === true) {
                return true;
            }
        }
        return false;
    };

    /**
     * Using the currently loaded page of rows, return the bottom-most _fully_ rendered cell within the given column
     * @private
     * @param {object} oStartingCol Active column when operation started. If not passed in then row header is assumed.
     * @returns {object} dom ref at bottom of page
     */
    Table.prototype._determineCellDownTillPage = function(oStartingCol) {
        var oDomRefToFocusOn,
            oLastFullyVisibleRow = this._determineLastFullyVisibleRow(),
            oTargetCell;

        if (oStartingCol && !(oStartingCol instanceof ColumnGroup)) {
            // table cell
            oTargetCell = oLastFullyVisibleRow.getCells()[oStartingCol.getIndex()];
            if (!oTargetCell) {
                return undefined;
            }
            oDomRefToFocusOn = oTargetCell.$().closest('td.sapUiTableTd').get(0);
        } else {
            // row header
            oDomRefToFocusOn = this._rowHeaderFromIndex(this.indexOfRow(oLastFullyVisibleRow));
        }

        return oDomRefToFocusOn;
    };

    /**
     * Using the currently loaded page of rows, return the bottom-most, right-most _fully_ rendered cell within the given column
     * @private
     * @returns {object} dom ref at bottom of page
     */
    Table.prototype._determineCellDownAndRightTillPage = function() {
        var oLastFullyVisibleRow = this._determineLastFullyVisibleRow(),
            aCells = oLastFullyVisibleRow.getCells(),
            oTargetCell = aCells[aCells.length - 1];
        if (oTargetCell) {
            return oTargetCell.$().closest('td.sapUiTableTd').get(0);
        }
        return undefined;
    };

    /**
     * Using the currently loaded page of rows, return the top-most _fully_ rendered cell within the given column
     * @private
     * @param {object} oStartingCol Active column when operation started. If not passed in then row header is assumed.
     * @returns {object} dom ref at top of page
     */
    Table.prototype._determineCellUpTillPage = function(oStartingCol) {
        var oDomRefToFocusOn,
            oFirstFullyVisibleRow = this._determineFirstFullyVisibleRow(),
            oTargetCell;

        if (oStartingCol && !(oStartingCol instanceof ColumnGroup)) {
            // table cell
            oTargetCell = oFirstFullyVisibleRow.getCells()[oStartingCol.getIndex()];
            if (!oTargetCell) {
                return undefined;
            }
            oDomRefToFocusOn = oTargetCell.$().closest('td.sapUiTableTd').get(0);
        } else {
            // row header
            oDomRefToFocusOn = this._rowHeaderFromIndex(this.indexOfRow(oFirstFullyVisibleRow));
        }

        return oDomRefToFocusOn;
    };

    /**
     * Using the currently loaded page of rows, return the top-most, left-most _fully_ rendered cell within the given column
     * @private
     * @returns {object} dom ref at top of page
     */
    Table.prototype._determineCellUpAndLeftTillPage = function() {
        var oFirstFullyVisibleRow = this._determineFirstFullyVisibleRow(),
            oTargetCell = oFirstFullyVisibleRow.getCells()[0];
        if (oTargetCell) {
            return oTargetCell.$().closest('td.sapUiTableTd').get(0);
        }
        return undefined;
    };

    /**
     * Is at least one of the "visible" rows (from getRows()) fully visible?
     * @private
     * @returns {boolean} if at least one row is fully visible
     */
    Table.prototype._isAtLeastOneRowFullyVisible = function() {
        return this.getRows().filter(function(oRow) {
            return oRow.isFullyVisible();
        }).length > 0;
    };

    /**
     * Get a particular row header dom reference (registered in the ItemNavigation array)
     * @private
     * @param {int} iIndex _visible_ row index, not logical
     * @returns {object} dom reference to specified row header
     */
    Table.prototype._rowHeaderFromIndex = function(iIndex) {
        return this.getRows()[iIndex].getDomRefs().rowSelector;
    };

    /**
     * Whether pending keyboard navigation operation is regarding a column header.
     * @private
     * @returns {boolean} if column header navigation
     */
    Table.prototype._isColumnHeaderNavigation = function() {
        var oInfo = TableUtils.getFocusedItemInfo(this),
            iHeaderRows = this.$().find(".sapUiTableColHdrScr>.sapUiTableColHdr").length;
        return this.getColumnHeaderVisible() && oInfo.cell < (oInfo.columnCount * iHeaderRows);
    };

    /**
     * Freezes the given column. If the column is not currently in a fixed position, it will be moved adjacent to the
     * scrollable columns and the fixed column count will be increased to include the column.
     * @param {sas.hc.ui.table.Column} oColumn the column to freeze
     * @returns {sas.hc.ui.table.Table}
     * @public
     */
    Table.prototype.freezeColumn = function (oColumn) {
        if (!oColumn || oColumn instanceof ColumnGroup) {
            return this;
        }

        var iColIndex = oColumn.getIndex(),
            oWidth = oColumn.getWidth(), // CSSSize
            iFrozenColumnCount = this._bIgnoreFixedColumnCount
                ? this._getMandatoryColumnCount()
                : this.getTotalFrozenColumnCount();

        // the column is already frozen do nothing
        if (iColIndex < iFrozenColumnCount) {
            return this;
        }

        // no width declared or using %-based widths will cause issues when freezing
        if (!oWidth || jQuery.sap.endsWith(oWidth, "%")) {
            var iWidth = oColumn.$().outerWidth();

            // S1233585: if the column headers are not visible
            // we need to get the width from a cell, and not the col header
            if (iWidth === null || iWidth === undefined) {
                iWidth = this.getRows()[0].getCells()[iColIndex].$().closest("td").outerWidth();
            }

            if (oColumn.setWidth) {
                oColumn.setWidth(iWidth + "px");  //so specify the exact current width as column width
            }
        }

        // increase the fixedColumnCount by 1
        this.setFixedColumnCount(this.getFixedColumnCount() + 1);

        // if necessary, move the column to the first non-frozen index
        if (iColIndex > iFrozenColumnCount) {
            var bPreventRestoreColumWidths = this._bPreventRestoreColumWidths;
            this._bPreventRestoreColumWidths = true;
            this.removeColumn(oColumn);
            this.insertColumn(oColumn, iFrozenColumnCount);
            this._bPreventRestoreColumWidths = bPreventRestoreColumWidths;
            // if columns are bound, we need to reset the row template
            // because we did not do that in column insert or delete
            if (this.isBound("columns")) {
                this._resetRowTemplate();
            }
        }

        // update the focus using ItemNavigation
        if (jQuery.isFunction(this._getItemNavigation)) {
            var oIN = this._getItemNavigation();
            if (oIN) {
                oIN.setFocusedIndex(iFrozenColumnCount);
            }
        }

        return this;
    };

    /**
     * Unfreezes the given column.
     * @param {sas.hc.ui.table.Column} oColumn the column to unfreeze
     * @returns {sas.hc.ui.table.Table}
     * @public
     */
    Table.prototype.unfreezeColumn = function (oColumn) {
        if (oColumn && !oColumn.getMandatory() && !(oColumn instanceof ColumnGroup)) {
            var iColIndex = oColumn.getIndex(),
                iFrozenCount = this.getTotalFrozenColumnCount();

            //1st, check if the column is frozen (if not, do nothing)
            if (iColIndex < iFrozenCount && oColumn.getWidth) {
                var oWidth = oColumn.getWidth();    //CSSSize

                if ((!oWidth || jQuery.sap.endsWith(oWidth, "%")) && oColumn.setWidth) {  //no width declared or using %-based widths will cause issues when thawing
                    var iWidth = oColumn.$().outerWidth();
                    oColumn.setWidth(iWidth + "px");  //so specify the exact current width as column width
                }

                var iNewIndex = iFrozenCount - 1;
                //2nd, if necessary, move the column to the first non-frozen index
                if (iColIndex < iNewIndex) {
                    var bPreventRestoreColumWidths = this._bPreventRestoreColumWidths;
                    this._bPreventRestoreColumWidths = true;
                    this.removeColumn(oColumn);
                    this.insertColumn(oColumn, iNewIndex);
                    this._bPreventRestoreColumWidths = bPreventRestoreColumWidths;
                    // if columns are bound, we need to reset the row template
                    // because we did not do that in column insert or delete
                    if (this.isBound("columns")) {
                        this._resetRowTemplate();
                    }
                }
                //3rd, decrease the fixedColumnCount by 1
                if (this.getFixedColumnCount() > 0) {
                    this.setFixedColumnCount(this.getFixedColumnCount() - 1);
                } else {
                    this.setFixedColumnCount(0);
                }

                // update the focus using ItemNavigation
                if (jQuery.isFunction(this._getItemNavigation)) {
                    var oIN = this._getItemNavigation();
                    if (oIN) {
                        oIN.setFocusedIndex(iNewIndex);
                    }
                }
            }
        }

        return this;
    };

    /**
     * Returns true if the provided Column is in the Table and is frozen.
     *
     * @public
     * @returns {boolean}
     */
    Table.prototype.isFrozenColumn = function (oColumn) {
        if (!oColumn || oColumn instanceof ColumnGroup) {
            return false;
        }
        var colIndex = oColumn.getIndex();
        var fixedNow = this.getTotalFrozenColumnCount();
        return (colIndex < fixedNow);
    };

    /**
     * Returns true if the provided Column is in the Table and is frozen
     * or was unfrozen due to column widths.
     *
     * @private
     * @returns {boolean} Whether the column should be frozen
     */
    Table.prototype._shouldBeFrozenColumn = function (oColumn) {
        if (!oColumn || oColumn instanceof ColumnGroup) {
            return false;
        }
        var colIndex = oColumn.getIndex();
        var fixedNow = this._getActualFrozenColumnCount();
        return (colIndex < fixedNow);
    };

    /**
     * Check whether every selection control is selected or not
     * @returns {boolean} Whether every row is selected
     */
    Table.prototype.isEveryIndexSelected = function() {
        var iSelectedIndices = this._oSelection.getSelectedIndices().length,
            iTotalIndices = this._getRowCount();
        return iSelectedIndices === iTotalIndices;
    };


    /**
     * hook function for updateRows
     * When updateRows is called, determine the delay before the bindings are updated
     * sas needs a bigger time delay to update the row headers when the table is displayed inside a dialog
     * @override
     * @param {string} sReason reason for the update
     * @protected
     */
    Table.prototype.determineUpdateRowBindingsDelay = function(sReason) {
        return AnimationUtil.getVisualEffectsEnabled() ? AnimationUtil.DURATION_NORMAL/* 270 */ : 50;
    };


    // =============================================================================
    // ANIMATIONS
    // =============================================================================

    /**
     * Hook function for _updateColumnHeader.
     * When syncing col headers with fnSyncColumnHeaders
     * provide a hook to further filter the col header selector
     * @protected
     * @override
     * @param {object} $cols jQuery-wrapped column headers
     * @return {object} the filtered jquery-wrapped column headers array
     */
    Table.prototype._filterColHdrCntFromSelectorWhenSyncColHeaders = function($jqColHdrCnt) {
        return $jqColHdrCnt.not(".sasUiTableSumColHdrCnt");
    };

    /**
     * Hook function for _updateColumnHeader.
     * Execute additional logic when determining when to call fnSyncColumnHeaders
     * @protected
     * @override
     * @return {boolean} whether col header syncing should be delayed
     */
    Table.prototype._shouldDelayColHeaderSync = function() {
        return this._bOnAfterRendering || AnimationUtil.getVisualEffectsEnabled();
    };

    /**
     * column is resized => update!
     * @override
     * @protected
     */
    Table.prototype._onColumnResized = function(oEvent, iIndex) {
        // rerender if animations are on, need to invalidate table
        //at end of resizing or when mouse is released at end of resize
        //for HTMLCOMMONS-4427 which is subtask of HTMLCOMMONS-4016
        if (AnimationUtil.getVisualEffectsEnabled()) {
            this.invalidate();
        }
    };


    // =============================================================================
    // SELECTION CONTROLS
    // =============================================================================

    /**
     * Determines whether to show selection controls for selection.
     * @private
     * @returns {boolean} whether to show selection controls or just the blank selection spaces
     */
    Table.prototype._useSelectionControls = function () {
        var bResult = false, sShowSelectionControls = this.getShowSelectionControls();

        if (sShowSelectionControls === "true" || (sShowSelectionControls === "auto" && sas.hasTouch() === true)) {
            bResult = true;
        }

        return bResult;
    };

    /**
     * Sync up selection control state with the corresponding row.
     * This gets called after paging occurs.
     * @private
     */
    Table.prototype._refreshSelectionControls = function () {
        var i, aRows;

        if (this._useSelectionControls() === true) {    //are we using selection controls?
            aRows = this.getRows();     //get the rows for the table
            for (i=0; i<aRows.length; i++) {    //iterate over the rows
                this._refreshRowSelectionControl(aRows[i]);
            }
        }
    };

    /**
     * Syncs up the Row's selection control state.
     * @param oRow The Row to refresh.
     * @private
     */
    Table.prototype._refreshRowSelectionControl = function(oRow) {
        var iIndex = oRow.getIndex();   //get the index of the row
        var bIsSelected = this.isIndexSelected(iIndex); //check if it was selected
        var oSelectionControl = oRow.getSelectionControl();
        oSelectionControl.setSelected(bIsSelected); //assign selection state to control
    };


    // =============================================================================
    // CONTEXT MENU
    // =============================================================================

    /**
     * Factory method. Creates the context menu.
     *
     * @private
     * @return {sap.ui.commons.Menu} The created column menu.
     */
    Table.prototype._createContextMenu = function () {
        return new Menu(this.getId() + "-cmenu");
    };

    /**
     * @return {sap.ui.commons.Menu} The context menu
     */
    Table.prototype.getContextMenu = function () {
        var oMenu = this.getAggregation("contextMenu");
        if (!oMenu) {
            oMenu = this._createContextMenu();
            this.setAggregation("contextMenu", oMenu);
        }
        return oMenu;
    };

    /**
     * @protected
     */
    Table.prototype.customizeContextMenu = function (oMenu, oEvent) {
        var $menu = oEvent.menu = oMenu;
        var $context = oEvent.context = this.getContextMenuContext(oEvent) || {}; //should return *something*, but let's be safe

        var fnCheckKeyboard = function () {

            // This is called also for mouse events.  Those are handled elsewhere (ItemNavigation).  Only do keyboard here.
            if (oEvent.type !== "keydown") {
                return;
            }

            // Check for context menu invocation via keyboard.  There are two ways: shift+f10 or the menu key on ff, chrome,
            // edge (there apparently isn't a menu key on mac).  The menu key code varies from browser to browser.
            // There aren't any constants to use in KeyCodes, so we hardcode them here.
            var tempWhich = 0;  // chrome
            if ( Device.browser.firefox) {
                tempWhich = 1;
            } else
            if ( Device.browser.edge) {
                tempWhich = 3;
            }

            if (oEvent.which === jQuery.sap.KeyCodes.F10 && oEvent.shiftKey){
                jQuery(oEvent.target).data("sas_lastActiveElement", oEvent.target);
            } else
            if ((oEvent.type === "contextmenu" && oEvent.which === tempWhich)) {

                // For the "menu" key on a cell we need to find the appropriate parent.
                var tableHeaderCell = $(oEvent.target).closest(".sapUiTableCol"),
                    tableCell = $(oEvent.target).closest(".sapUiTableTd");

                if (tableHeaderCell.length > 0) {
                    jQuery(tableHeaderCell).data("sas_lastActiveElement", tableHeaderCell);
                } else
                if (tableCell.length > 0) {
                    jQuery(tableCell).data("sas_lastActiveElement", tableCell);
                } else {
                    jQuery(oEvent.target).data("sas_lastActiveElement", oEvent.target);
                }
            }
        };

        if ($context.region === "column" && $context.column) {
            if ($context.column.getMenu) {
                $menu = $context.column.getMenu();
            }
            if (!$menu) {
                return null;    //no menu? no way!!
            }
            if (jQuery.isFunction($menu.validateMenu)) {        //S1233306: validateMenu function is SAS-specific
                $menu.validateMenu();   //force menu items into being created (if necessary)
            }

            if (oEvent.originalEvent instanceof KeyboardEvent) {
                fnCheckKeyboard();
            }

            if (oEvent.target && oEvent.target.id && jQuery.isFunction($menu.focusAfterRendering)) {        //S1233306: focusAfterRendering is SAS-specific
                var sTargetId = oEvent.target.id;
                //determine if event target is sort icon
                if (sTargetId === ($context.column.getId() + "-sortIcon")) {
                    $menu.focusAfterRendering("sort");
                } else if (sTargetId === ($context.column.getId() + "-filterIcon")) {            //OR determine if event target is filter icon
                    $menu.focusAfterRendering("filter");
                } else {
                    $menu.focusAfterRendering(null);    //focus naturally
                }
            }
        } else if ($context.region === "cell") {

            if (oEvent.originalEvent instanceof KeyboardEvent) {
                fnCheckKeyboard();
            }

            if (this._findAndfireCellEvent(this.fireCellContextmenu, oEvent, this._oncellcontextmenu)) {
                oEvent.preventDefault();
            }
        } else {
            var items = this.__getContextMenuItemsForRegion($menu, oEvent, $context);
            if (items && items.length > 0) {
                oMenu.removeAllItems();
                jQuery.each(items, function (i, item) {
                    oMenu.addItem(item);
                });
            }
        }

        return $menu;
    };

    /**
     * Computes various properies related to the context menu event.
     *
     * @public
     * @param {Event} oEvent The context menu event
     * @return {Object} An object with various computed properties
     */
    Table.prototype.getContextMenuContext = function (oEvent) {
        var mParams = this._extractCellEventParams(oEvent) || {},
            context = {
                trigger: oEvent,
                region: null,
                rowIndex: mParams.rowIndex,
                columnIndex: mParams.columnIndex,
                getColumnLabel: function () {
                    if (this.column) {
                        return this.column.getName() || (this.column.getLabel() && this.column.getLabel().getText ? this.column.getLabel().getText() : null);
                    }
                    return null;
                }
            };

        // trigger column menu
        var $target = jQuery(oEvent.target);
        var iIndex, oColumn, $row;

        // cell?
        var $cell = $target.closest('tr.sapUiTableTr > td');  //S1261096 - find only the closest TD that is part of our table
        if ($cell.length === 1) {
            context.region = "cell";
            context.cellDomRef = $cell;
            context.cellId = $cell.attr("id");
            context.cellHeaders = $cell.attr("headers");
            context.cellBindingContext = mParams.cellBindingContext;
            context.cellControl = mParams.cellControl;
            //search for column by id
            $col = jQuery.sap.domById(mParams.columnId); //returns the DIV inside the TH
            if ($col) {
                iIndex = parseInt($col.getAttribute("data-sap-ui-colindex"), 10);   //slightly different attr on TH
                oColumn = this.retrieveLeafColumns()[iIndex];
                //we definitely *should* have a column at this point, but we'll check just to be sure
                if (oColumn) {
                    context.column = oColumn;
                    context.columnId = oColumn.getId();
                    context.columnDomRef = $col;    //slightly different DOM node via TH
                }
            }
            //search for row as if selectionmode was RowOnly or Row
            $row = $target.closest(".sapUiTableCtrl > tbody > tr");
            if ($row.length === 1) {
                iIndex = parseInt($row.attr("data-sap-ui-rowindex"), 10);
                context.visibleRowIndex = iIndex;
                context.rowId = $row.attr("data-sap-ui");
                context.rowDomRef = $row;
                context.rowBindingContext = mParams.rowBindingContext;
            }
            return context;
        }

        // column header?
        var $col = $target.closest(".sapUiTableCol");
        if ($col.length === 0) {
            // if we didn't find it going up the DOM tree, try going down
            // this is necessary for headers that span multiple rows
            $col = $target.find('.sapUiTableCol');
        }
        if ($col.length === 1) {
            context.region = "column";
            iIndex = parseInt($col.attr("data-sap-ui-colindex"), 10);
            context.columnIndex = iIndex;
            oColumn = this.retrieveLeafColumns()[iIndex];
            context.column = oColumn;
            context.columnId = $col.attr("data-sap-ui");
            context.columnDomRef = $col;
            return context;
        }

        // row header?
        $row = $target.closest(".sapUiTableRowHdr");
        if ($row.length === 1) {
            context.region = "row";     //TODO - worth distinguishing between row content and row header?
            iIndex = parseInt($row.attr("data-sap-ui-rowindex"), 10);
            context.rowIndex = this.getFirstVisibleRow() + iIndex;
            context.visibleRowIndex = iIndex;
            context.rowId = $row.attr("data-sap-ui");
            context.rowDomRef = $row;
            return context;
        }

        // table control? (only if the selection behavior is set to row)
        if (/*!this._bActionMode && */ (
            this.getSelectionBehavior() === SelectionBehavior.Row ||
        this.getSelectionBehavior() === SelectionBehavior.RowOnly)) {
            $row = $target.closest(".sapUiTableCtrl > tbody > tr");
            if ($row.length === 1) {
                context.region = "row";
                iIndex = parseInt($row.attr("data-sap-ui-rowindex"), 10);
                context.rowIndex = this.getFirstVisibleRow() + iIndex;
                context.visibleRowIndex = iIndex;
                context.rowId = $row.attr("data-sap-ui");
                context.rowDomRef = $row;
                return context;
            }
        }

        // select all?
        if (jQuery.sap.containsOrEquals(jQuery.sap.domById(this.getId() + "-selall"), oEvent.target)) {
            context.region = "select-all";
        }

        /*
         Known additional targets w/in table that are not currently well handled:
         - Column Resizer handles
         - Upper-right corner adjacent to column headers and vertical scrollbar
         - Vertical Scrollbar
         - Very bottom pixel (horizontal scrollbar container, perhaps?)
         */

        return context;
    };

    /**
     * @private
     */
    Table.prototype.__getContextMenuItemsForRegion = function ($menu, oEvent, $context) {
        var items = this._contextMenuItems[$context.region];

        //lookup region in internal map
        // if not found, create and store items
        //return items
        if (items === null) {
            switch ($context.region) {
                //case 'column':
                default:
                    sas.log.trace("default menu for region=" + $context.region);
            }
        }

        return items;
    };

    /**
     * Removes any applied sorting parameters from the given column and row binding.
     * @param {sas.hc.ui.table.Column} oColumn The column to remove from the sort
     * @private
     */
    Table.prototype._removeColumnSort = function (oColumn) {
        var oSorter = oColumn._oSorter,
            binding = this.getBinding("rows"),
            self = this,
            aSorters, newSorters;

        if (oColumn instanceof ColumnGroup) {
            return;
        }

        if (binding !== null && oSorter !== null) {
            // notify the event listeners
            var bExecuteDefault = this.fireSort({
                column: oColumn,
                sortOrder: undefined,
                columnAdded: false
            });

            // potentially a listener can prevent this action, so check to see if they did
            if (bExecuteDefault) {
                oColumn.setProperty("sorted", false, true); //this column should no longer be sorted
                oColumn._oSorter = null;                // remove the sorting status of all columns

                aSorters = binding.aSorters;
                newSorters = [];
                jQuery.each(aSorters, function (i, sorter) {
                    if (sorter !== oSorter) {
                        newSorters.push(sorter);
                    }
                });

                if (newSorters.length === 0) {
                    // if it's really empty, then it should be null.
                    newSorters = null;
                }
                // sort the binding
                if (binding.sort) {
                    binding.sort(newSorters);
                }
                this._setManagedTimeout(function() {
                    //announce the possibly updated state
                    self._getAccExtension().announceCellNavigation();
                }, 10);
            }
        }

        this.refreshSort();
        oColumn._renderSortIcon();   //we skipped this earlier, but let's call it now
    };

    /**
     * Returns true if provided Column is sorted. False otherwise
     * @public
     * @param {sas.hc.ui.table.Column} oColumn The Column object to check.
     * @returns {boolean}
     */
    Table.prototype.isColumnSorted = function (oColumn) {
        if (oColumn !== null && oColumn.getSorted) {
            if (oColumn.getSorted() === true) {
                return true;
            }
        }
        return false;
    };

    /**
     * Examines the sort information from the binding to determine which columns should indicate sorting and how
     * @public
     * @returns {sas.hc.ui.table.Table} TODO-jsdoc
     */
    Table.prototype.refreshSort = function () {
        var oBinding = this.getBinding("rows"),
            aSorters = (oBinding !== null) ? oBinding.aSorters : null,    //get the array of sorters
            aCols = this.retrieveLeafColumns().slice(),  //our array of columns to explore
            iNumCols = aCols.length,    //how many columns do we have?
            oSorter,    //an individual Sorter
            oCol,       //an individual Column
            i, l,       //loop vars
            oDirtyCols = {};   //dirty flags to optimize redraw

        //reset the sorting status of all columns - we're redoing them all
        for (i = 0, l = iNumCols; i < l; i++) {
            oCol = aCols[i];

            if (!(oCol instanceof ColumnGroup) && oCol.getProperty("sorted") === true) {
                oCol.setProperty("sorted", false, true);    //we'll rerender the sort icons for all columns later in this method

                //remember what changed in a map
                oDirtyCols[oCol.getId()] = oCol;
            }
        }

        this._aSortedColumns = [];   //reset this list

        //match sorters with columns and update columns appropriately
        if (aSorters !== null) {
            for (i = 0, l = aSorters.length; i < l; i++) {    //loop through sorters (likely <= columns.length)
                oSorter = aSorters[i];  //current sorter
                oCol = this._findColumnForSorter(oSorter, aCols);
                if (oCol !== null && !(oCol instanceof ColumnGroup)) {    //null means we couldn't figure out which column should be marked (this would be unfortunate)
                    //update oCol to reflect sorting state from oSorter
                    oCol.setProperty("sorted", true, true);
                    oCol.setProperty("sortOrder",
                        oSorter.bDescending ? SortOrder.Descending : SortOrder.Ascending,
                        true);
                    oCol._oSorter = oSorter;
                    //add to pushed sorted columns array
                    this._aSortedColumns.push(oCol);

                    //remember what changed
                    oDirtyCols[oCol.getId()] = oCol;
                }
            }
        }

        //rerender sort icon for each column of table that was marked as dirty
        for (var sId in oDirtyCols) {
            if (oDirtyCols.hasOwnProperty(sId) && oDirtyCols[sId]._renderSortIcon) {
                //create/recreate/or hide the sort icon for each column as needed
                oDirtyCols[sId]._renderSortIcon();
            }
        }

        return this;
    };

    /**
     * Attempts to match the given sorter with a column in the array and will return the matched column
     * if one is found.
     * @param {sap.ui.model.Sorter} oSorter the Sorter to match
     * @param {Array} aColumns an array of sas.hc.ui.table.Column columns
     * @returns {sas.hc.ui.table.Column} the column that best matches the given sorter
     * @private
     */
    Table.prototype._findColumnForSorter = function (oSorter, aColumns) {
        var i,
            oCol,
            oColumnMatch = null,
            iNumCols = aColumns.length,
            fnColComparator;

        for (i = 0; i < iNumCols; i++) {  //loop through each column
            oCol = aColumns[i];    //current column

            /* sap.ui.model.Sorter features to consider:
             - sPath : the path from the model used to get the value to use while sorting
             - bDescending: boolean flag to show sort order (ascending vs descending)
             - fnCompare: custom sort comparator

             sas.hc.ui.table.Column features to consider:
             - _oSorter: stored reference to Table created Sorter object already in use for this column
             - "sorted": boolean to show whether column is marked as being sorted
                * not guaranteed to be set, unless the Table's API/UI were used to apply the sort
             - "sortProperty": the binding property on which the column will sort
             - "sortOrder": sas.hc.ui.table.SortOrder (Ascending||Descending)
             - "sortFunction": custom sort comparator function
             */

            // a column group cannot be sorted
            if (oCol instanceof ColumnGroup) {
                break;
            }

            //check if the column's _oSorter property is this exact Sorter instance
            if (oCol._oSorter === oSorter) {
                //consider this a match
                oColumnMatch = oCol;
                break;
            }

            //match on paths
            if (oSorter.sPath === oCol.getSortProperty()) {
                //next check comparators
                fnColComparator = oCol.getSortFunction();
                if (!jQuery.isFunction(fnColComparator) || (oSorter.fnCompare === fnColComparator)) {
                    //consider this a match
                    oColumnMatch = oCol;
                    break;
                }
            }
        }

        return oColumnMatch;
    };


    // =============================================================================
    // ACCESSIBILITY
    // =============================================================================

    /**
     * Constructs from a localized resource, a description of the table based on the visible row count and the length of the
     * currently bound model.
     * @returns {String} the formatted descriptive text for this table
     * @private
     */
    Table.prototype._getARIADescription = function () {
        var oBinding = this.getBinding("rows"),
            iModelLength = ((oBinding !== null && oBinding !== undefined) ? oBinding.getLength() : 0),
            oRB = this.oResBundle;
        if (iModelLength === 0)
            return this._oResBundle.getText("TBL_NO_DATA.txt");
        else {
            //visibleRowCount is actually availableRowCount, need to make sure it's not greater than the number of rows on the model
            var rowCount = Math.min(this.getVisibleRowCount(), iModelLength);
            return oRB.getText("table.description.aria.fmt", [rowCount, iModelLength]);
        }
    };


    // =============================================================================
    // ITEM DEFAULT ACTION
    // =============================================================================

    /**
     * @private
     */
    Table.prototype._getRowIndex = function($cell) {
        var aRows = this.getRows(),
            $row = $cell.parent("tr[data-sap-ui-rowindex]"),
            iRowIndex = parseInt($row.attr("data-sap-ui-rowindex"), 10),
            oRow = aRows[iRowIndex],
            iRealRowIndex = oRow && oRow.getIndex();

        return iRealRowIndex;
    };

    /**
     * @private
     */
    Table.prototype._initDelegates = function() {
        var mParams;

        if (!this._oClickHandlerDelegate) {
            mParams = {
                fireSingle: this.fireDefaultAction,
                fireDouble: this.fireDoubleClick,
                fnGetRowIndex: this._getRowIndex,
                fnGetCellEventParams: this._extractCellEventParams
            };
            this._oClickHandlerDelegate = new TableClickHandlerDelegate(this, mParams);
        }
    };

    /**
     * @private
     */
    Table.prototype._destroyDelegates = function() {
        if (this._oClickHandlerDelegate) {
            this.removeEventDelegate(this._oClickHandlerDelegate);
            this._oClickHandlerDelegate.destroy();
            this._oClickHandlerDelegate = null;
        }
    };

    /**
     * Custom ctrl-a keypress handling
     * Includes changes from JIRA-4582 and S1217237
     * hook function for onkeydown
     * @param {object} oEvent keypress event
     * @override
     * @protected
     */
    Table.prototype._customHandleCtrlA = function(oEvent) {
        //if the table is in action mode OR focus is in an element outside of the main table region, then abort handling Ctrl+A
        if (!jQuery.sap.containsOrEquals(this.getDomRef("sapUiTableCnt"), oEvent.target)
            || this._getKeyboardExtension().isInActionMode()) {
            return;
        }

        // column header is InPlaceEdit?
        var $InPlaceEdit = jQuery(oEvent.target).closest(".sasUiTableColHdrIpe");
        if ($InPlaceEdit.length === 1) {
            $InPlaceEdit.find("input").select();
            return;
        }

        var oInfo = TableUtils.getFocusedItemInfo(this);

        // select-all toggling handled by selectAllControl
        this.getSelectAllControl().toggleSelected();

        TableUtils.focusItem(this, oInfo.cell, oEvent);

        oEvent.preventDefault();
        oEvent.stopImmediatePropagation(true);
    };

    /**
     * Return the text value for the label of the given Column
     * @public
     * @param oColumn the Column to examine
     * @returns {*} the text of the column label as a String or undefined if the value cannot be determined
     */
    Table.prototype.getColumnHeaderText = function(oColumn) {
        var sValue, oControl = oColumn.getLabel();
        if(!!oControl) {
            if(oControl.getText) {
                sValue = oControl.getText();
            }else if(oControl === oColumn._oInPlaceEdit) {
                sValue = oControl.getContent().getValue();
            }else if(oControl.getTooltip) {
                sValue = oControl.getTooltip();
            }
        }

        return sValue;
    };


    // =============================================================================
    // SUMMARY ROW HEADER
    // =============================================================================

    /**
     * Simply toggles the value of the 'showSummaryHeader' property between true and false.
     * @public
     */
    Table.prototype.toggleSummaryHeader = function() {
        var bShown = this.getProperty("showSummaryHeader");
        this.setShowSummaryHeader(!bShown);
    };

    /**
     * check if data is available in the table
     * @private
     */
    Table.prototype._hasData = function() {
        var oBinding = this.getBinding("rows");
        if (!oBinding || (oBinding.getLength() || 0) === 0) {
            return false;
        }
        return true;
    };

    /**
     * Invoke setFirstVisibleRow so that a targeted page of rows is rendered.
     * Handles flags to ensure no extra invalidations occur.
     *
     * @param {function} fnCallback invoked when operation complete
     * @private
     */
    Table.prototype._showPageStartingAt = function(iRow, fnCallback) {
        var self = this;
        // stay within bounds of bound rows
        var iNewFirstVisibleRow = Math.max(iRow, 0);
        iNewFirstVisibleRow = Math.min(iNewFirstVisibleRow, this._getRowCount());

        // prevent vscroll event from firing and causing invalidation
        this._skipVScroll = true;

        // account for the possibility that setFirstVisibleRow is a no-op
        if (this.getFirstVisibleRow() === iNewFirstVisibleRow) {
            if (typeof fnCallback === "function") {
                fnCallback();
            }
            return;
        }
        this.setFirstVisibleRow(iNewFirstVisibleRow);
        this.attachEventOnce("_rowsUpdated", function() {
            if (typeof fnCallback === "function") {
                // Sometimes waiting for a single _rowsUpdated event isn't enough.
                // Adding the extra wait adds a tiny bit of extra security that things have
                // "settled down" once the firstVisibleRow operation has completed
                self._setManagedTimeout(function() {
                    fnCallback();
                }, 0);
            }
        });
    };

    /**
     * Invoke setFirstVisibleRow so that the first page of rows is shown.
     *
     * @param {function} fnCallback invoked when operation complete
     * @private
     */
    Table.prototype._showFirstPage = function(fnCallback) {
        var self = this;
        this._showPageStartingAt(0, function() {

            // edge-case: if firstVisibleRow was already 0 then
            // scrolling to the very top may need to be forced
            if (self.getRows()[0] && self.getRows()[0].isFullyVisible() === false) {
                self._skipVScroll = true;
                self.getDomRef(SharedDomRef.VerticalScrollBar).scrollTop = 0;
                self._adjustTablePosition();
            }

            if (typeof fnCallback === "function") {
                fnCallback();
            }
        });
    };

    /**
     * Invoke setFirstVisibleRow so that the end of the table is shown.
     *
     * Ensures that the final page of rows will have at least one fully-visible row.
     * Will keep invoking decreasing values of setFirstVisibleRow until
     * at least one fully visible row is showing.
     *
     * @param {function} fnCallback invoked when operation complete
     * @private
     */
    Table.prototype._showLastPage = function(fnCallback) {
        var self = this,
            iRowCount = self._getRowCount();

        function isLastAbsoluteRowFullyVisible() {
            var oAbsoluteLastRow = self.getRows().slice(0).reverse().filter(function(oRow) {
                // exclude dummy rows
                return oRow._bHidden === false && oRow.getIndex() === (iRowCount - 1);
            })[0];
            return oAbsoluteLastRow !== undefined && oAbsoluteLastRow.isFullyVisible() === true;
        }

        function goToEnd(iFirstVisibleRow) {
            self._showPageStartingAt(iFirstVisibleRow, function() {
                // if the very last row is not fully visible,
                // and we have not reached the end of the Table firstVisibleRow-wise
                // then try again with an incremented firstVisibleRow
                if (isLastAbsoluteRowFullyVisible() === false && (iFirstVisibleRow < iRowCount)) {
                    goToEnd(iFirstVisibleRow + 1);
                    return;
                }

                if (typeof fnCallback === "function") {
                    fnCallback();
                }
            });
        }

        // usually (rowcount - 2) is a good approximation
        goToEnd(iRowCount - 2);
    };

    /**
     * Indicates whether a Vertical Scroll Bar is needed.
     * @private
     * @returns {Boolean} true/false when Vertical Scroll Bar is required
     */
    Table.prototype._isVSbRequired = function() {
        var aVisibleNonDummyRows, oLastVisibleRow, oFirstVisibleRow;

        // scrollbar can only exist if Table rendering is available
        if (!this.getDomRef("sapUiTableCnt")) {
            return false;
        }

        // scrollbar can only exist if rows have been rendered
        if (this.getRows() && this.getRows()[0] && this.getRows()[0].getDomRef() === null) {
            return false;
        }

        // get visible data rows, not dummy rows that fill blank space
        aVisibleNonDummyRows = this.getRows().filter(function(oRow) {
            return oRow._bHidden === false && oRow.$().length === 1;
        });

        // gather first and last visible rows
        oLastVisibleRow = aVisibleNonDummyRows.slice(0).reverse()[0];
        oFirstVisibleRow = aVisibleNonDummyRows[0];

        // with px-scrolling, it seems that when only one row is shown
        // the container is slightly smaller than the single row,
        // so make this a special case of not showing the scrollbar
        if (this._iBindingLength === 1) {
            return (oLastVisibleRow && !oLastVisibleRow.isFullyVisible());
        }

        // if there are more rows to render, then show scrollbar
        if (this._iBindingLength > (this.getFirstVisibleRow() + aVisibleNonDummyRows.length)) {
            return true;
        }

        // this case could only happen during Table startup when the rows aggregation hasn't been assigned yet
        if (!oLastVisibleRow || !oFirstVisibleRow) {
            return false;
        }

        // if the last visible non-dummy row's bottom point is below
        // the row's container div, then show scrollbar
        if (oLastVisibleRow.getDomRef().getBoundingClientRect().bottom > this.getDomRef("sapUiTableCnt").getBoundingClientRect().bottom) {
            return true;
        }

        // if the first visible non-dummy row's is not the absolute first index then show scrollbar
        if (oFirstVisibleRow.getIndex() > 0) {
            return true;
        }

        // if either the first or last rows are not fully visible, then show scrollbar
        if ((oFirstVisibleRow.isFullyVisible() === false) || (oLastVisibleRow.isFullyVisible() === false)) {
            return true;
        }

        return false;
    };

    /**
     * Updates the aria summary content as the value changes when scrolling or data size changes.
     * @private
    */
    Table.prototype._updateA11yVisibleRowsLabel = function () {
        var $ariaSummary = this.$("ariasummary"),
            $gridContainer = this.$("sapUiTableCnt");
        if ($ariaSummary) {
            $ariaSummary.text(this._getARIADescription());
        }
        if ($gridContainer) {
            $gridContainer.attr("aria-rowcount", this._getRowCount());
        }

    };

    /**
     * @private
     */
    Table.prototype._accUpdateARIARowIndices = function() {
        // loop over the table rows
        var aRows = this.getRows();
        for (var i = 0; i < aRows.length; i++) {
            var oRow = aRows[i],
                $Row = oRow.getDomRefs(true);   //will always return an object with 'row' attribute
            //but the rowSelector property may be undefined
            if ($Row.rowSelector) {
                //give the row selector an aria-label with the row index
                $Row.rowSelector.attr("aria-label", this.oResBundle.getText("table.aria.row_label.fmt", (oRow.getIndex()+1)));
            }
        }
    };

    /**
     * Return the row control that matches an absolute index
     * @param {int} iRowAbsoluteIndex absolute index of desired row
     * @returns {sas.hc.ui.table.Row} row instance
     * @private
     */
    Table.prototype._getRowByAbsoluteIndex = function(iRowAbsoluteIndex) {
        return this.getRows().filter(function(oRow) {
            return oRow.getIndex() === iRowAbsoluteIndex;
        })[0];
    };

    /**
     * Determine if an x-coord is withing the visible scroll range
     * @param {int} iXCoord the x-coord to check
     * @returns {boolean} whether the x-coord is within range
     * @private
     */
    Table.prototype._isXCoordWithinScrollRange = function(iXCoord) {
        var $scrollArea = this.$("sapUiTableCtrlScr"),
            iScrollAreaLeft, iScrollAreaRight;
        if (this.$().length === 0) {
            return false;
        }
        iScrollAreaLeft = $scrollArea.offset().left;
        iScrollAreaRight = ($scrollArea.offset().left + $scrollArea.outerWidth());
        return (iScrollAreaLeft <= iXCoord) && (iScrollAreaRight >= iXCoord);
    };

    Table.prototype._createSelectAllControl = function() {
        return new SelectAllControl({
            id: this.getId() + "-selectAll",
            table: this
        });
    };

    // =============================================================================
    // REMEMBER SELECTIONS
    // =============================================================================
    Table.prototype.setRememberSelections = function(bRemember) {
        this.setProperty("rememberSelections", bRemember, true);
        if (!this.getRememberSelections()) {
            //clear any remembered paths if feature is disabled
            this._aSelectedKeys = [];
        }
        return this;
    };

    /*
     * Sets internal remembered selected context paths.
     * This method can be called to reset remembered selection
     * and does not change selection of the items until binding update.
     * @param {String[]} aSelectedKeys valid binding context path array
     * @param {String} sReason: reason for this call, the root event causing this call
     * @param {boolean} bSuppressEvent: true if no 'selectionUpdated' event should be fired
     */
    Table.prototype.setSelectedKeys = function(aSelectedKeys, sReason, bSuppressEvent) {
        this._aSelectedKeys = aSelectedKeys || [];
        if (!bSuppressEvent) {
            this.fireEvent("selectionUpdated",{"selectionKeys": aSelectedKeys, "reason": sReason});
        }
        //this.applyRememberedSelections(sReason);
    };

    /*
     * Returns internal remembered selected context paths as a copy
     * @return {String[]} selected items binding context path
     */
    Table.prototype.getSelectedKeys = function() {
        return this._aSelectedKeys.slice(0);
    };

    Table.prototype._cloneRowBinding = function(bUseSorters, bUseFilters) {
        var oBindingInfo = this.getBindingInfo("rows") || {},
            oBinding = oBindingInfo.binding,
            oModel = oBinding.getModel() || this.getModel(oBindingInfo.model),
            oNewBinding = null,
            aSorters, aFilters;

        // only deal with binding case
        if (!!oModel && !!oBinding) {
            // attempt to create an exact duplicate of the current list binding - fragile?
            //TODO should we include the current sorting? application filters?
            aSorters = bUseSorters === true ? oBinding.aSorters : [];
            aFilters = bUseFilters === true ? oBinding.aFilters : [];
            oNewBinding = oModel.bindList(oBinding.getPath(), oBinding.getContext(), aSorters, aFilters, oBinding.mParameters);
        }

        return oNewBinding;
    };

    /**
     * Returns the binding contexts of the remembered selected paths.
     * Note: This method returns an empty array if no data binding is used.
     * @return {Context[]}
     */
    Table.prototype.getSelectedContexts = function() {
        var oBindingInfo = this.getBindingInfo("rows") || {},
            oBinding = oBindingInfo.binding,
            sModelName = oBindingInfo.model,
            oModel = this.getModel(sModelName),
            bUseRawBinding = (this.getFilteredSelectionBehavior() === FilteredSelectionBehavior.RetainAll),
            self = this;

        // only deal with binding case
        if (!oModel || !oBinding) {
            return [];
        }

        // if we're remembering selections and using primary key, then we'll need to find contexts based on key.
        //  !!this can be resource intensive!!
        if (this.getRememberSelections()) {
            return this._gatherContextsForKeysAsync(this.getSelectedKeys(), bUseRawBinding);
        }

        return new Promise(function(resolve, reject) {
            // return binding contexts from all selected indices
            var aContexts = self.getSelectedIndices().map(function(iIndex) {
                return oBinding.getContexts(iIndex, 1)[0];
            });

            resolve(aContexts);
        });
    };

    Table.prototype._gatherContextsForKeysAsync = function(aKeys, bRawBinding) {
        var oBindingInfo = this.getBindingInfo("rows"),
            oRowBinding,
            oTable = this,
            iIndexFrom = 0,
            oBinding,
            iLength;

        if (!oBindingInfo) {
            return Promise.reject("no binding");
        }

        oRowBinding = oBindingInfo.binding;

        if (bRawBinding === true) {
            // create a temporary unfiltered binding to the model
            oBinding = this._cloneRowBinding(false, false);
        } else {
            oBinding = oRowBinding;
        }

        // Binding.getContexts requires a starting index and the number of contexts
        iLength = oBinding.getLength();

        return new Promise(function(resolve, reject) {
            var aContexts = oBinding.getContexts(iIndexFrom, iLength),
                sSelectionIdProperty = oTable.getSelectionIdProperty(),
                aResults = [], oContext, i, sPath, iIdx;

            //PERF - looping through all contexts in binding to figure out which items match will be non-performant
            for(i = 0; i < aContexts.length; i++){
                oContext = aContexts[i];
                if (!sSelectionIdProperty) {
                    sPath = oContext.getPath();
                } else {
                    sPath = oContext.getProperty(sSelectionIdProperty);
                }

                if(sPath) {
                    iIdx =  aKeys.indexOf(sPath);
                    if (iIdx >= 0) {
                        aResults.push(oContext);
                        // shrink the array of keys we have to search through
                        aKeys.splice(iIdx, 1);
                    }
                }

                if (aKeys.length === 0) {
                    //no more keys to match, so stop looping through contexts
                    break;
                }
            }

            resolve(aResults);
        });
    };

    /**
     * Returns true if the path is one of the selected paths
     * @param {String} path: context path
     * @returns {boolean}
     */
    Table.prototype.isSelectedPath = function(path) {
        return $.inArray(path, this.getSelectedKeys()) !== -1;
    };

    /**
     * Remove all remembered paths
     * @param {string} sReason: reason for calling clearSelection. Possible values:
     *                      "clearAll": when we really need to clear all the table selections, usually result of an user initiated action, not internal action.
     * @param {boolean} bSuppressEvent: true if no 'selectionUpdated' event should be fired
     */
    Table.prototype.clearSelectedKeys = function(sReason, bSuppressEvent ) {
        //Clear the preserved selection cache
        this.setSelectedKeys([], sReason, bSuppressEvent);
    };

    /**
     * Remove the given paths from the selected list
     * @param {String[]} aKeys - paths to remove from remembered selections cache
     * @param {String} reason: reason for this call, the root event causing this call
     * @param {boolean} bSuppressEvent: true if no 'selectionUpdated' event should be fired
     */
    Table.prototype.removeSelectedKeys = function(aKeys, bSuppressEvent) {
        var selectedKeys = this.getSelectedKeys();
        aKeys.forEach(function(path) {
            var iPathIndex = selectedKeys.indexOf(path);
            if (iPathIndex > -1) {
                //remove just the matched path
                selectedKeys.splice(iPathIndex, 1);
            }
        });
        if(selectedKeys.length > 0){
            this.setSelectedKeys(selectedKeys, "removeSelectedKeys", bSuppressEvent);
        }else{
            this.clearSelectedKeys("clearAll", bSuppressEvent);
        }
    };

    /**
     * Insert or Remove given paths from the remembered cache
     * @param {String[]} addSelectionKeys: array of paths to remember
     * @param {String[]} removeSelectionKeys: array of paths to remove from remembered list
     * @param {String} reason: reason for updating the selected paths, the root event causing this call
     */
    Table.prototype.updateSelectedKeys = function(addSelectionKeys, removeSelectionKeys, reason){
        addSelectionKeys = addSelectionKeys || [];
        removeSelectionKeys = removeSelectionKeys || [];

        var selectionContextKeys = this.getSelectedKeys();
        var pathToAdd, pathToRemove;
        for(var i=0; i<addSelectionKeys.length; i++){
            pathToAdd = addSelectionKeys[i];
            if(selectionContextKeys.indexOf(pathToAdd) < 0){
                selectionContextKeys.push(pathToAdd);
            }
        }
        for(var j = 0; j<removeSelectionKeys.length; j++){
            pathToRemove = removeSelectionKeys[j];
            var iPathIndex = selectionContextKeys.indexOf(pathToRemove);
            if (iPathIndex > -1) {
                // just remove the matched path
                selectionContextKeys.splice(iPathIndex, 1);
            }
        }
        this.setSelectedKeys(selectionContextKeys, reason);
    };


    /**
     * Insert or Remove given indices as paths from selection array
     * @param {int[]} addSelectionIndices: array of indices whose paths need to be remembered
     * @param {int[]} removeSelectionIndices: array of indices whose paths need to be removed from the remembered list
     * @param {String} reason: the root event causing this call
     */
    Table.prototype.handleSelectionChange = function(addSelectionIndices, removeSelectionIndices, reason){
        addSelectionIndices = addSelectionIndices || [];
        removeSelectionIndices = removeSelectionIndices || [];

        //For single selection table, clear any previous selection so that it can preserve only the last selection below.
        if(addSelectionIndices.length > 0 && this.getSelectionMode() === SelectionMode.Single){
            this.clearSelectedKeys("clearAll");
        }

        var contextKeysToAdd=[],
            contextKeysToRemove=[],
            sSelectionIdProperty = this.getSelectionIdProperty();

        var addSelectionIndex, removeSelectionIndex, context, sPath;
        for(var i =addSelectionIndices.length-1; i >= 0; i--){
            addSelectionIndex = addSelectionIndices[i];
            context = this.getContextByIndex(addSelectionIndex);
            //checking for context not undefined is useful when table has more selected rows, but the number of rows shown is less due to filter
            if(!context){
                continue;
            }
            //if 'selectionIdProperty' is not null, then need to use the value of that property instead of path
            if (!sSelectionIdProperty) {
                sPath = context.getPath();
            } else {
                sPath = context.getProperty(sSelectionIdProperty);
            }
            contextKeysToAdd.push(sPath);
        }
        for(var j = removeSelectionIndices.length-1; j >= 0; j--){
            removeSelectionIndex = removeSelectionIndices[j];
            context = this.getContextByIndex(removeSelectionIndex);
            //if 'selectionIdProperty' is not null, then need to use the value of that property instead of path
            if (!sSelectionIdProperty) {
                sPath = context.getPath();
            } else {
                sPath = context.getProperty(sSelectionIdProperty);
            }
            contextKeysToRemove.push(sPath);
        }
        this.updateSelectedKeys(contextKeysToAdd, contextKeysToRemove, reason);
    };

    /**
     * Table data is updated, now apply the remembered selections
     */
    Table.prototype.applyRememberedSelections = function(sReason) {
        var aSelectedKeys = this._aSelectedKeys;

        if(aSelectedKeys.length > 0){ //Exit early if nothing to select anyway
            var oBinding = this.getBinding("rows"),
                oModel = oBinding && oBinding.getModel(),
                oTable = this,
                sSelectionIdProperty,
                fnMatchKeys,
                aSelectedIndices, iChunkSize, iBindingLength;

            if (!oBinding || !oModel) {
                return;
            }

            // this process is going to iterate over *every* context in the model to determine if the path matches one
            //  that was previously selected. this process could potentially take a while and we're going to set the busy
            //  state on the table and yield thread control as we go
            this.setBusy(true);

            iChunkSize = (oModel.hasOwnProperty("iSizeLimit") && oModel.iSizeLimit) || 100;
            // limit valid chunk sizes to integers between 1 and 100 (inclusive)
            if (isNaN(iChunkSize) || iChunkSize < 1 || iChunkSize > 100) {
                iChunkSize = 100;
            }

            aSelectedIndices = this.getSelectedIndices();
            iBindingLength = oBinding.getLength();
            sSelectionIdProperty = this.getSelectionIdProperty();

            fnMatchKeys = function(iPointer) {
                if (iPointer < iBindingLength) {
                    oTable._setManagedTimeout(function() {
                        var aContexts = oBinding.getContexts(iPointer, iChunkSize),
                            iSelectionIndex, sPath, bContainsPath, bContainsIndex, i;

                        for(i=0; i<aContexts.length && iPointer<iBindingLength; i++, iPointer++){
                            iSelectionIndex = iPointer;
                            bContainsIndex = aSelectedIndices.indexOf(iSelectionIndex) > -1;
                            //if 'selectionIdProperty' is not null, then need to use the value of that property instead of path
                            if (!sSelectionIdProperty) {
                                sPath = aContexts[i].getPath();
                            } else {
                                sPath = aContexts[i].getProperty(sSelectionIdProperty);
                            }
                            bContainsPath = aSelectedKeys.indexOf(sPath) > -1;

                            if(bContainsPath && (!bContainsIndex)){
                                oTable.addSelectionInterval(iSelectionIndex, iSelectionIndex);
                            }
                        }

                        fnMatchKeys(iPointer);
                    }, 10);
                } else {
                    // we've reached the end, so unbusy the table
                    oTable.setBusy(false);
                    oTable.fireEvent("selectionUpdated",{"selectionKeys": aSelectedKeys, "reason": sReason});
                }
            };

            // start the process with the 0th context
            fnMatchKeys(0);
        }
    };

    Table.prototype._selectAllKeys = function(sReason) {
        var oBinding = this.getBinding("rows"),
            oModel = oBinding && oBinding.getModel(),
            oTable = this,
            aKeysToAdd = [],
            sSelectionIdProperty,
            fnSelectKeys, iChunkSize, iBindingLength;

        if (!oBinding || !oModel) {
            return Promise.reject("No model/binding");
        }

        // this process is going to iterate over *every* context in the model to mark it as selected
        this.setBusy(true);

        iChunkSize = (oModel.hasOwnProperty("iSizeLimit") && oModel.iSizeLimit) || 100;
        // limit valid chunk sizes to integers between 1 and 100 (inclusive)
        if (isNaN(iChunkSize) || iChunkSize < 1 || iChunkSize > 100) {
            iChunkSize = 100;
        }

        iBindingLength = oBinding.getLength();
        sSelectionIdProperty = this.getSelectionIdProperty();

        return new Promise(function(resolve, reject) {
            fnSelectKeys = function(iPointer) {
                if (iPointer < iBindingLength) {
                    oTable._setManagedTimeout(function() {
                        var aContexts = oBinding.getContexts(iPointer, iChunkSize),
                            sPath, i;

                        for(i=0; i<aContexts.length && iPointer<iBindingLength; i++, iPointer++){
                            sPath = aContexts[i].getPath();
                            //if 'selectionIdProperty' is not null, then need to use the value of that property instead of path
                            if (!sSelectionIdProperty) {
                                sPath = aContexts[i].getPath();
                            } else {
                                sPath = aContexts[i].getProperty(sSelectionIdProperty);
                            }
                            aKeysToAdd.push(sPath);
                        }

                        fnSelectKeys(iPointer);
                    }, 10);
                } else {
                    // we've reached the end, so unbusy the table
                    oTable.setBusy(false);

                    if(aKeysToAdd.length > 0){
                        oTable.updateSelectedKeys(aKeysToAdd, [], sReason || "selectAll");
                    }
                    resolve(aKeysToAdd);
                }
            };

            // start the process with the 0th context
            fnSelectKeys(0);
        });
    };

    /**
     * Returns true or false based on whether table is filtered or not
     * @returns {boolean}
     */
    Table.prototype.isFiltered = function(){
        var oBindingInfo = this.getBindingInfo("rows"),
            oBinding = this.getBinding("rows"),
            sModelName = (oBindingInfo || {}).model,
            oModel = this.getModel(sModelName);

        if (!oBindingInfo || !oModel) {
            return false;
        }

        var tableDataBindPath = oBinding.getPath();
        return oBinding.getLength() !== oModel.getProperty(tableDataBindPath).length;
    };

    /**
     * Returns true if the table should be showing a summary column header section
     * @param {sas.hc.ui.table.Table} oTable Instance of the table
     * @return {boolean}
     */
    Table.prototype.shouldShowSummaryHeader = function() {
        //check if table allows it at all
        if (!this.getShowSummaryHeader()) {
            return false;
        }

        var aCols = this._getVisibleColumns().filter(function(oCol) {
            return (!!(oCol.getAggregation("summaryLabel") || oCol.getAggregation("summaryControl")));
        });

        return (aCols.length > 0);
    };

    return Table;
});
