/*!
 * (c) 2014-2018, SAS Institute Inc.
 */

// Provides control sas.hc.ui.table.Row.
sap.ui.define(['jquery.sap.global', 'sap/ui/core/Element', 'sap/ui/core/Icon', 'sap/ui/model/Context', './library', "sas/hc/ui/core/Core"
],
    function(jQuery, Element, Icon, Context, library, sasCore) {
        "use strict";



        /**
         * Constructor for a new Row.
         *
         * @param {string} [sId] id for the new control, generated automatically if no id is given
         * @param {object} [mSettings] initial settings for the new control
         *
         * @class
         * The row.
         * @extends sap.ui.core.Element
         * @version 904001.11.16.20251118090100_f0htmcm94p
         *
         * @constructor
         * @public
         * @alias sas.hc.ui.table.Row
         * @ui5-metamodel This control/element also will be described in the UI5 (legacy) designtime metamodel
         */
        var Row = Element.extend("sas.hc.ui.table.Row", /** @lends sas.hc.ui.table.Row.prototype */ { metadata : {

            library : "sas.hc.ui.table",
            defaultAggregation : "cells",
            properties : {
                "cacheDomRefs" : {type : "boolean", defaultValue: true},
                "expandedStateIconId": {type: "string", defaultValue: sas.icons.HC.TREEEXPANDOPENED },
                "collapsedStateIconId": {type: "string", defaultValue: sas.icons.HC.TREEEXPANDCLOSED }
            },
            aggregations : {

                /**
                 * The controls for the cells.
                 */
                cells : {type : "sap.ui.core.Control", multiple : true, singularName : "cell"},

                /**
                 * The selection control for the row header.
                 */
                selectionControl : {type : "sap.ui.core.Control", multiple : false, default: null },

                /**
                     * The expand/collapse icon to render when the Row is in a TreeTable
                     */
                treeIcon: { type: "sap.ui.core.Icon", multiple: false, default: null }
            }
        }});

        Row.prototype.init = function() {
            this.initDomRefs();
        };

        Row.prototype.exit = function() {
            this.initDomRefs();
        };

        /**
         * @private
         */
        Row.prototype.initDomRefs = function() {
            this._mDomRefs = {};
        };

        /**
         * Returns the index of the row in the table or -1 if not added to a table. This
         * function considers the scroll position of the table and also takes fixed rows and
         * fixed bottom rows into account.
         *
         * @return {int} index of the row (considers scroll position and fixed rows)
         * @public
         * @ui5-metamodel This method also will be described in the UI5 (legacy) designtime metamodel
         */
        Row.prototype.getIndex = function() {
            var oTable = this.getParent();
            if (oTable) {
                // get the index of the row in the aggregation
                var iRowIndex = oTable.indexOfRow(this);

                // check for fixed rows. In this case the index of the context is the same like the index of the row in the aggregation
                var iNumberOfFixedRows = oTable.getFixedRowCount();
                if (iNumberOfFixedRows > 0 && iRowIndex < iNumberOfFixedRows) {
                    return iRowIndex;
                }

                // check for fixed bottom rows
                var iNumberOfFixedBottomRows = oTable.getFixedBottomRowCount();
                var iVisibleRowCount = oTable.getVisibleRowCount();
                if (iNumberOfFixedBottomRows > 0 && iRowIndex >= iVisibleRowCount - iNumberOfFixedBottomRows) {
                    var oBinding = oTable.getBinding("rows");
                    if (oBinding && oBinding.getLength() >= iVisibleRowCount) {
                        return oBinding.getLength() - (iVisibleRowCount - iRowIndex);
                    } else {
                        return iRowIndex;
                    }
                }

                var iFirstRow = oTable.getFirstVisibleRow();
                return iFirstRow + iRowIndex;
            }
            return -1;
        };

        /**
         *
         * @param bJQuery Set to true to get jQuery object instead of DomRef
         * @param bRefreshRefs Set to true to force this Row to recompute the DOM References.
         * @returns {object} contains DomRefs or jQuery objects of the row
         * @public
         */
        Row.prototype.getDomRefs = function (bJQuery, bRefreshRefs) {
            if (bRefreshRefs || !this.getCacheDomRefs()) {
                this._mDomRefs = {};
            }

            var fnAccess;
            var sKey;
            if (bJQuery === true) {
                fnAccess = jQuery.sap.byId;
                sKey = "jQuery";
            } else {
                fnAccess = jQuery.sap.domById;
                sKey = "dom";
            }

            if (!this._mDomRefs[sKey]) {
                this._mDomRefs[sKey] = {};
                var oTable = this.getParent();
                if (oTable) {
                    var iRowIndex = oTable.indexOfRow(this);
                    // row selector domRef
                    this._mDomRefs[sKey].rowSelector = fnAccess(oTable.getId() + "-rowsel" + iRowIndex);
                }

                // row domRef
                this._mDomRefs[sKey].rowScrollPart = fnAccess(this.getId());
                // row domRef (the fixed part)
                this._mDomRefs[sKey].rowFixedPart = fnAccess(this.getId() + "-fixed");
                // row selector domRef
                this._mDomRefs[sKey].rowSelectorText = fnAccess(this.getId() + "-rowselecttext");

                if (bJQuery === true) {
                    this._mDomRefs[sKey].row = this._mDomRefs[sKey].rowScrollPart;

                    if (this._mDomRefs[sKey].rowFixedPart.length > 0) {
                        this._mDomRefs[sKey].row = this._mDomRefs[sKey].row.add(this._mDomRefs[sKey].rowFixedPart);
                    } else {
                        // since this won't be undefined in jQuery case
                        this._mDomRefs[sKey].rowFixedPart = undefined;
                    }

                    if (this._mDomRefs[sKey].rowSelector && this._mDomRefs[sKey].rowSelector.length > 0) {
                        this._mDomRefs[sKey].row = this._mDomRefs[sKey].row.add(this._mDomRefs[sKey].rowSelector);
                    } else {
                        // since this won't be undefined in jQuery case
                        this._mDomRefs[sKey].rowSelector = undefined;
                    }
                }
            }

            return this._mDomRefs[sKey];
        };

        /**
         * Synchronizes the heights of all elements in this row of the table.  These elements include the row
         * selection header (if present), any frozen columns, and the scrollable/main columns.
         * @param {boolean} [bRefreshRefs] if true, then the DOM will be freshly examined ignoring any cached elements
         */
        Row.prototype.syncRowHeights = function (bRefreshRefs) {
            var oDomRefs = this.getDomRefs(false, bRefreshRefs),
                oTable = this.getParent(),
                sRowHeight = "",
                iRowHeight;

            //determine base row height
            if (oTable) {
                iRowHeight = oTable.getRowHeight();
                if (iRowHeight) {
                    sRowHeight = iRowHeight + "px";
                }
            }

            //reset heights of row sections and measure to find the tallest height
            iRowHeight = this._resetAndMeasureRowHeight(oDomRefs, sRowHeight);

            //and set all sections to that height
            this.setRowHeight(iRowHeight, oDomRefs);
        };

        /**
         * Resets the heights of each section of columns for the row and then measures to find the tallest section.
         * @param {object} oDomRefs the object holding references to various row section DOM elements returned by the
         *  getDomRefs method
         * @param {string} [sRowHeight] an optional value that the row height will be set to before measuring
         * @returns {number} the height of the tallest section of the row
         * @private
         */
        Row.prototype._resetAndMeasureRowHeight = function(oDomRefs, sRowHeight) {
            var iFixedRowHeight = 0,
                iScrollRowHeight,
                oClientRect;

            //reset heights of row sections and measure - fixed section if present
            if (oDomRefs.rowFixedPart) {
                oDomRefs.rowFixedPart.style.height = sRowHeight;
                oClientRect = oDomRefs.rowFixedPart.getBoundingClientRect();
                iFixedRowHeight = oClientRect.bottom - oClientRect.top;
            }

            //reset height of scrollable part and measure
            oDomRefs.rowScrollPart.style.height = sRowHeight;
            oClientRect = oDomRefs.rowScrollPart.getBoundingClientRect();
            iScrollRowHeight = (oClientRect.bottom - oClientRect.top);

            //use the row's tallest section height
            return Math.max(iFixedRowHeight, iScrollRowHeight);
        };

        /**
         * Sets the height of the sections of this row, including the selection header, frozen columns, and scrollable columns.
         * @param {number} iRowHeight the height in pixels to set all sections to
         * @param {object} [oDomRefs] the object holding references to various row section DOM elements returned by the
         *  getDomRefs method
         */
        Row.prototype.setRowHeight = function(iRowHeight, oDomRefs) {
            //implementation is similar to Table's _updateRowHeader but applies to just this single row
            var sRowHeight = iRowHeight + "px";
            oDomRefs = oDomRefs || this.getDomRefs(false);

            //now sync the sections - starting w/ row selection header
            if (oDomRefs.rowSelector) {
                oDomRefs.rowSelector.style.height = sRowHeight;
            }
            //if present, include fixed section
            if (oDomRefs.rowFixedPart) {
                oDomRefs.rowFixedPart.style.height = sRowHeight;
            }
            //finally, set height on scrollable section
            oDomRefs.rowScrollPart.style.height = sRowHeight;
        };

        /**
         *
         * @param {sas.hc.ui.table.Table} oTable Instance of the table
         * @param {Object} mTooltipTexts texts for aria descriptions and tooltips
         * @param {Object} mTooltipTexts.mouse texts for tooltips
         * @param {String} mTooltipTexts.mouse.rowSelect text for row select tooltip (if row is unselected)
         * @param {String} mTooltipTexts.mouse.rowDeselect text for row de-select tooltip (if row is selected)
         * @param {Object} mTooltipTexts.keyboard texts for aria descriptions
         * @param {String} mTooltipTexts.keyboard.rowSelect text for row select aria description (if row is unselected)
         * @param {String} mTooltipTexts.keyboard.rowDeselect text for row de-select aria description (if row is selected)
         * @param {Boolean} bSelectOnCellsAllowed set to true when the entire row may be clicked for selecting it
         * @private
         */
        Row.prototype._updateSelection = function(oTable, mTooltipTexts, bSelectOnCellsAllowed) {
            var bIsSelected = oTable.isIndexSelected(this.getIndex());
            var $DomRefs = this.getDomRefs(true);

            var sSelectReference = "rowSelect";
            if (bIsSelected) {
                // when the row is selected it must show texts how to deselect
                sSelectReference = "rowDeselect";
            }

            if ($DomRefs.rowSelectorText) {
                var sText = "";
                if (!(this._oNodeState && this._oNodeState.sum) && !this._bHasChildren) {
                    sText = mTooltipTexts.keyboard[sSelectReference];
                }
                $DomRefs.rowSelectorText.text(sText);
            }

            var $Row = $DomRefs.rowScrollPart;
            if ($DomRefs.rowFixedPart) {
                $Row = $Row.add($DomRefs.rowFixedPart);
            }

            if (bSelectOnCellsAllowed) {
                // the row requires a tooltip for selection if the cell selection is allowed
            } else {
                $Row.removeAttr("title");
            }

            if ($DomRefs.row) {
                // update visual selection state
                $DomRefs.row.toggleClass("sapUiTableRowSel", bIsSelected);
                oTable._getAccExtension().updateAriaStateOfRow(this, $DomRefs, bIsSelected);
            }
        };

        /**
         * Sets the binding context for the Row.
         * @protected
         */
        Row.prototype.setRowBindingContext = function(oContext, sModelName, oBinding) {
            var oNode;
            if (oContext && !(oContext instanceof Context)) {
                oNode = oContext;
                oContext = oContext.context;
            }

            var $rowTargets = this.getDomRefs(true).row;
            this._bHidden = !oContext;
            $rowTargets.toggleClass("sapUiTableRowHidden", this._bHidden);

            // collect rendering information for new binding context
            this._collectRenderingInformation(oContext, oNode, oBinding);

            this.setBindingContext(oContext, sModelName);
        };

        /**
         * @override
         */
        Row.prototype.setBindingContext = function(oContext, sModelName) {
            var bReturn = Element.prototype.setBindingContext.call(this, oContext || null, sModelName);
            return bReturn;
        };

        /**
         * @private
         */
        Row.prototype._collectRenderingInformation = function(oContext, oNode, oBinding) {
            // init node states
            this._oNodeState = undefined;
            this._iLevel = 0;
            this._bIsExpanded = false;
            this._bHasChildren = false;
            this._sTreeIconClass = "";

            if (oNode) {
                this._oNodeState = oNode.nodeState;
                this._iLevel = oNode.level;
                this._bIsExpanded = false;
                this._bHasChildren = false;
                this._sTreeIconClass = "sapUiTableTreeIconLeaf";
                this._sGroupIconClass = "";

                if (oBinding) {
                    if (oBinding.getLevel) {
                        //used by the "mini-adapter" in the TreeTable ClientTreeBindings
                        this._bIsExpanded = oBinding.isExpanded(this.getIndex());
                    } else if (oBinding.findNode) { // the ODataTreeBinding(Adapter) provides the hasChildren method for Tree
                        this._bIsExpanded = this && this._oNodeState ? this._oNodeState.expanded : false;
                    }

                    if (oBinding.nodeHasChildren) {
                        if (this._oNodeState) {
                            this._bHasChildren = oBinding.nodeHasChildren(oNode);
                        }
                    } else if (oBinding.hasChildren) {
                        this._bHasChildren = oBinding.hasChildren(oContext);
                    }

                    if (this._bHasChildren) {
                        this._sTreeIconClass = this._bIsExpanded ? "sapUiTableTreeIconNodeOpen" : "sapUiTableTreeIconNodeClosed";
                        this._sGroupIconClass = this._bIsExpanded ? "sapUiTableGroupIconOpen" : "sapUiTableGroupIconClosed";
                    }
                }
            }

            // Use treeIconClass to determine which icon to use.
            if (this._sTreeIconClass === "sapUiTableTreeIconLeaf") {
                this.getTreeIcon()
                    .removeStyleClass("sapUiTreeNodeExpanded")
                    .removeStyleClass("sapUiTreeNodeCollapsed")
                    .setSrc(sas.icons.HC.BLANK);
            } else if (this._sTreeIconClass === "sapUiTableTreeIconNodeOpen") {
                this.getTreeIcon()
                    .addStyleClass("sapUiTreeNodeExpanded")
                    .removeStyleClass("sapUiTreeNodeCollapsed")
                    .setSrc(this.getExpandedStateIconId());
            } else if (this._sTreeIconClass === "sapUiTableTreeIconNodeClosed") {
                this.getTreeIcon()
                    .removeStyleClass("sapUiTreeNodeExpanded")
                    .addStyleClass("sapUiTreeNodeCollapsed")
                    .setSrc(this.getCollapsedStateIconId());
            }
        };

        /**
         * Returns the treeIcon aggregation.
         * @public
         * @returns {sap.ui.core.Icon}
         */
        Row.prototype.getTreeIcon = function() {
            var oTreeIcon = this.getAggregation("treeIcon");
            // Lazy instaniate the TreeIcon.
            if (oTreeIcon === null) {
                oTreeIcon = new Icon();
                oTreeIcon.addStyleClass("sasUiTreeNodeBullet");
                this.setTreeIcon(oTreeIcon);
            }
            return oTreeIcon;
        };

        /**
         * Whether the rendering of this row is (perhaps partially) visible
         * @public
         * @returns {boolean} if row is visible
         */
        Row.prototype.isVisible = function() {
            var $row = this.$(),
                $tableCCCnt = $row.closest(".sapUiTableCCnt"),

                // gather row and table content bounds
                iRowTop = $row.offset().top,
                iRowBottom = iRowTop + $row.outerHeight(),
                iContainerTop = $tableCCCnt.offset().top,
                iContainerBottom = iContainerTop + $tableCCCnt.outerHeight();

            return $row.hasClass('sapUiTableRowHidden') === false &&
            // check row bounds relative to Table's content container
            (iRowBottom >= iContainerTop) && (iRowTop <= iContainerBottom);
        };

        /**
         * Whether the rendering of this row is _fully_ visible
         * @public
         * @returns {boolean} if row is visible
         */
        Row.prototype.isFullyVisible = function() {
            return this.isBottomVisible() && this.isTopVisible();
        };

        /**
         * Whether the bottom of the rendering of this row is visible (within the Table's content container)
         * @public
         * @returns {boolean} if bottom boundary of row is visible
         */
        Row.prototype.isBottomVisible = function() {
            var $row = this.$(),
                $tableCCCnt = $row.closest(".sapUiTableCCnt"),

                // gather row and table content bounds
                iRowTop = Math.round($row.offset().top),
                iRowBottom = Math.round(iRowTop + $row.outerHeight()),
                iContainerTop = Math.round($tableCCCnt.offset().top),
                iContainerBottom = Math.round(iContainerTop + $tableCCCnt.outerHeight());

            return $row.hasClass('sapUiTableRowHidden') === false &&
            // check row bounds relative to Table's content container
            (iRowBottom >= iContainerTop) && (iRowBottom <= iContainerBottom);
        };

        /**
         * Whether the top of the rendering of this row is visible (within the Table's content container)
         * @public
         * @returns {boolean} if top boundary of row is visible
         */
        Row.prototype.isTopVisible = function() {
            var $row = this.$(),
                $tableCCCnt = $row.closest(".sapUiTableCCnt"),

                // gather row and table content bounds
                iRowTop = Math.round($row.offset().top),
                iContainerTop = Math.round($tableCCCnt.offset().top),
                iContainerBottom = Math.round(iContainerTop + $tableCCCnt.outerHeight());

            return $row.hasClass('sapUiTableRowHidden') === false &&
            // check row bounds relative to Table's content container
            (iRowTop >= iContainerTop) && (iRowTop <= iContainerBottom);
        };

        /**
         * Whether this row is the current top (perhaps partially) visible row
         * @public
         * @returns {boolean} if first visible row
         */
        Row.prototype.isFirstVisible = function() {
            return this.getIndex() === this.getParent().getFirstVisibleRow();
        };

        /**
         * Whether this row is the current top _fully_ visible row.<br />
         * Meaning that it's entire bounds exists within the Table's content container
         * @public
         * @returns {boolean} if first _fully_ visible row
         */
        Row.prototype.isFirstFullyVisible = function() {
            return this.getIndex() === this.getParent()._determineFirstFullyVisibleRow().getIndex();
        };

        /**
         * Whether this row is the current bottom-most _fully_ visible row
         * @public
         * @returns {boolean} if last _fully_ visible row
         */
        Row.prototype.isLastFullyVisible = function() {
            var oLastFullyVisibleRow = this.getParent()._determineLastFullyVisibleRow();
            return this.getIndex() === oLastFullyVisibleRow.getIndex();
        };

        return Row;
    });
