/*!
 * (c) 2014-2018, SAS Institute Inc.
 */

// Provides control sas.hc.ui.table.ColumnMenu.
sap.ui.define([
    'jquery.sap.global',
    'sap/ui/core/RenderManager',
    './library',
    'sas/hc/ui/unified/Menu', // eslint-disable-line sashtmlcommons/no-deprecated-resource
    'sas/hc/ui/unified/MenuItem',
    'sap/ui/Device',
    './TableUtils',
    "sap/ui/core/Core",
    "sas/hc/ui/core/Core"
    
],
function(jQuery, RenderManager, library, Menu, MenuItem, Device, TableUtils, sapCr,sasCr) {
    "use strict";

    /**
     * Constructor for a new ColumnMenu.
     *
     * <b>Note:</b> Applications must not use or change the default <code>sas.hc.ui.table.ColumnMenu</code> of
     * a column in any way or create own instances of <code>sas.hc.ui.table.ColumnMenu</code>.
     * To add a custom menu to a column, use the aggregation <code>menu</code> with a new instance of
     * <code>sap.ui.unified.Menu</code>.
     *
     * @param {string} [sId] id for the new control, generated automatically if no id is given
     * @param {object} [mSettings] initial settings for the new control
     *
     * @class
     * The column menu provides all common actions that can be performed on a column.
     * @extends sas.hc.ui.unified.Menu
     * @version 904001.11.16.20251118090100_f0htmcm94p
     *
     * @constructor
     * @public
     * @alias sas.hc.ui.table.ColumnMenu
     * @ui5-metamodel This control/element also will be described in the UI5 (legacy) design time metamodel
     */
    var ColumnMenu = Menu.extend("sas.hc.ui.table.ColumnMenu", /** @lends sas.hc.ui.table.ColumnMenu.prototype */ { metadata : {

        library : "sas.hc.ui.table"
    }});


    /**
     * Initialization of the ColumnMenu control
     * @private
     */
    ColumnMenu.prototype.init = function() {
        if (Menu.prototype.init) {
            Menu.prototype.init.apply(this, arguments);
        }
        this.addStyleClass("sapUiTableColumnMenu");
        this.oResBundle = sap.ui.getCore().getLibraryResourceBundle("sas.hc.ui.table");
        this._bInvalidated = true;
        this._iPopupClosedTimeoutId = null;
        this._oColumn = null;
        this._oTable = null;
        this._attachPopupClosed();

        this.oResBundle = sap.ui.getCore().getLibraryResourceBundle("sas.hc.ui.table");
    };

    ColumnMenu.prototype.onAfterRendering = function() {
        if (this._sFocusAfterRendering === "sort") {
            this.setHoveredItem(this._oSortItem);  //open the sorting submenu
            //        this.openSubmenu(this._oSortItem);  //open the sorting submenu
            //        this.setHoveredItem(this._oSortItem.getSubmenu().getItems()[0]);    //highlight the 1st item in there
        } else if (this._sFocusAfterRendering === "filter") {
            this.setHoveredItem(this._oFilterItem); //set this now for immediate hover appearance
            //but call it again in a moment to get proper focus in the text field (stupid, right?)
            jQuery.sap.delayedCall(100, this, 'setHoveredItem', [this._oFilterItem, true]);
        } else {
            // Comment out the next line so that when you arrow down the onsapnext() handler
            // selects the first item in the menu. The line would have been necessary if we had highlighted
            // the first item when the menu is opened, but with GUIDESINF-1643 we do not do that anymore.
            //        this.setHoveredItem(this.getNextSelectableItem(-1));
        }

        Menu.prototype.onAfterRendering.apply(this, arguments);
    };

    /**
     * Termination of the ColumnMenu control
     * @private
     */
    ColumnMenu.prototype.exit = function() {
        if (Menu.prototype.exit) {
            Menu.prototype.exit.apply(this, arguments);
        }
        window.clearTimeout(this._iPopupClosedTimeoutId);
        this._detachEvents();
        this._oColumn = this._oTable = null;
    };


    /**
     * Event handler. Called when the theme is changed.
     * @private
     */
    ColumnMenu.prototype.onThemeChanged = function() {
        if (this.getDomRef()) {
            this._invalidate();
        }
    };


    /**
     * Defines this object's new parent. If no new parent is given, the parent is
     * just unset and we assume that the old parent has removed this child from its
     * aggregation. But if a new parent is given, this child is first removed from
     * its old parent.
     *
     * @param {sap.ui.base.ManagedObject} oParent the object that becomes this object's new parent
     * @see {sap.ui.base.ManagedObject}
     *
     * @return {sap.ui.base.ManagedObject}
     *            Returns <code>this</code> to allow method chaining
     * @private
     */
    ColumnMenu.prototype.setParent = function(oParent) {
        this._detachEvents();
        this._invalidate();
        this._updateReferences(oParent);
        this._attachEvents();
        return Menu.prototype.setParent.apply(this, arguments);
    };

    /**
     * @private
     */
    ColumnMenu.prototype._updateReferences = function(oParent) {
        this._oColumn = oParent;
        if (oParent) {
            jQuery.sap.assert(lazyInstanceof(oParent, "sas/hc/ui/table/Column"), "ColumnMenu.setParent: parent must be a subclass of sas.hc.ui.table.Column");

            this._oTable = this._oColumn.getTable();
            if (this._oTable) {
                jQuery.sap.assert(lazyInstanceof(this._oTable, "sas/hc/ui/table/Table"), "ColumnMenu.setParent: parent of parent must be subclass of sas.hc.ui.table.Table");
            }
        }
    };

    /**
     * Attaches the required event handlers.
     * @private
     */
    ColumnMenu.prototype._attachEvents = function() {
        if (this._oTable) {
            this._oTable.attachColumnVisibility(this._invalidate, this);
            this._oTable.attachColumnMove(this._invalidate, this);
            this._oTable.attachSort(this._invalidate, this);
        }
    };

    /**
     * Detaches the required event handlers.
     * @private
     */
    ColumnMenu.prototype._detachEvents = function() {
        if (this._oTable) {
            this._oTable.detachColumnVisibility(this._invalidate, this);
            this._oTable.detachColumnMove(this._invalidate, this);
            this._oTable.detachSort(this._invalidate, this);
        }
    };

    /**
     * Invalidates the column menu control items. Forces recreation of the menu items when the menu is opened.
     * @private
     */
    ColumnMenu.prototype._invalidate = function() {
        this._bInvalidated = true;
    };

    /**
     * Special handling for IE < 9 when the popup is closed.
     * The associated column of the menu is focused when the menu is closed.
     * @private
     */
    ColumnMenu.prototype._attachPopupClosed = function() {
        // put the focus back into the column header after the
        // popup is being closed.
        var that = this;

        if (!Device.support.touch) {
            this.getPopup().attachClosed(function() {
                that._iPopupClosedTimeoutId = window.setTimeout(function() {
                    if (that._oColumn) {
                        if (that._lastFocusedDomRef) {
                            that._lastFocusedDomRef.focus();
                        } else {
                            that._oColumn.focus();
                        }
                    }
                }, 0);
            });
        }
    };

    /**
     * Override sdk "open" function
     * @param {boolean} bWithKeyboard todo
     * @param {object} oOpenerRef todo
     * @param {object} my todo
     * @param {object} at todo
     * @param {object} of todo
     * @param {object} offset todo
     * @param {object} collision todo
     * @override
     * @protected
     */
    ColumnMenu.prototype.open = function(bWithKeyboard, oOpenerRef, my, at, of, offset, collision) {
        var bExecuteDefault = true;

        if (this.bOpen) {
            return;
        }

        if (this._oColumn) {
            bExecuteDefault = this._oColumn.fireColumnMenuOpen({
                menu: this
            });
        }

        if (bExecuteDefault) {
            //cause all appropriate items to come into being
            this.validateMenu();

            //update freeze menu item? pattern so far is to listen for events on table and destroy full menu then recreate...
            if (this._oFreezeItem !== null && this._oFreezeItem !== undefined && this._oTable !== null) {
                var isFrozen = this._oTable.isFrozenColumn(this._oColumn);
                var rbKey = (isFrozen) ? "TBL_UNFREEZE" : "TBL_FREEZE";
                this._oFreezeItem.setText(this.oResBundle.getText(rbKey+".txt"));
            }

            if (this.getItems().length > 0) {
                Menu.prototype.open.apply(this, arguments);
            }
        }
    };

    /**
     * Adds the menu items to the menu.
     * @private
     */
    ColumnMenu.prototype._addMenuItems = function() {
        if (this._oColumn) {
            if (this._oTable.retrieveLeafColumns().length > 1) {
                this._addResizeOrMoveItem();
            }
            if (this._oColumn.getAutoResizable && this._oColumn.getAutoResizable()) {
                this._addBestFitItem();
            }
            this._addFreezeItem();
            this._addSortMenu();
            this._addFilterMenuItem();
            this._addGroupMenuItem();
            this._addColumnVisibilityMenuItem();
            this._addManageColumnsMenuItem();
        }
    };

    /**
     * Adds the sort menu item to the menu.
     * @param {boolean} bDesc the sort direction. <code>true</code> for descending.
     * @private
     */
    ColumnMenu.prototype._addSortMenuItem = function(bDesc, oMenu) {
        var oColumn = this._oColumn;
        oMenu = oMenu || this;

        // sort is not supported on column groups
        var ColumnGroup = sap.ui.requireSync("sas/hc/ui/table/ColumnGroup");
        if (oColumn instanceof ColumnGroup) {
            return;
        }

        var sDir = bDesc ? "desc" : "asc";
        if (oColumn.getSortProperty() && oColumn.getShowSortMenuEntry()) {
            oMenu.addItem(this._createMenuItem(
                sDir,
                "TBL_SORT_" + sDir.toUpperCase()+"_SINGLE",
                null,
                function(oEvent) {
                    oColumn.sort(bDesc, false); //always single
                }
            ));
        }
    };

    /**
     * Adds the filter menu item to the menu.
     * @private
     */
    ColumnMenu.prototype._addFilterMenuItem = function() {
        var oColumn = this._oColumn,
            oTable = oColumn.getTable(),
            bEnableCustomFilter = false,
            oItem;

        // currently we do not support filter on column groups
        var ColumnGroup = sap.ui.requireSync("sas/hc/ui/table/ColumnGroup");
        if (oColumn instanceof ColumnGroup) {
            return;
        }

        if (oTable) {
            bEnableCustomFilter = oTable.getEnableCustomFilter();
        }

        if (oColumn.getFilterProperty() && oColumn.getShowFilterMenuEntry()) {
            if (bEnableCustomFilter) {
                oItem = this._createMenuItem(
                    "filter",
                    "TBL_FILTER_ITEM",
                    "filter",
                    function(oEvent) {
                        oTable.fireCustomFilter({
                            column: oColumn
                        });
                    }
                );
            } else {
                oItem = this._createMenuTextFieldItem(
                    "filter",
                    "TBL_FILTER",
                    null,
                    oColumn.getFilterValue(),
                    function (oEvent) {
                        oColumn.filter(this.getValue());
                    }
                );
            }
            this.addItem(oItem);
        }

        this._oFilterItem = oItem; //save for later
    };


    /**
     * Adds the group menu item to the menu.
     * @private
     */
    ColumnMenu.prototype._addGroupMenuItem = function() {
        var oColumn = this._oColumn;
        var oTable = this._oTable;
        if (oTable && oTable.getEnableGrouping() && oColumn.getSortProperty && oColumn.getSortProperty()) {
            this.addItem(this._createMenuItem(
                "group",
                "TBL_GROUP",
                null,
                jQuery.proxy(function(oEvent) {
                    oTable.setGroupBy(oColumn);
                }, this)
            ));
        }
    };

    /**
     * Adds the freeze menu item to the menu.
     * @private
     */
    ColumnMenu.prototype._addFreezeMenuItem = function() {
        var oColumn = this._oColumn;
        var oTable = this._oTable;
        if (oTable && oTable.getEnableColumnFreeze()) {
            var iColumnIndex = oColumn.getIndex();
            var bIsFixedColumn = iColumnIndex + 1 === oTable.getFixedColumnCount();
            this.addItem(this._createMenuItem(
                "freeze",
                bIsFixedColumn ? "TBL_UNFREEZE" : "TBL_FREEZE",
                null,
                function() {
                    // forward the event
                    var bExecuteDefault = oTable.fireColumnFreeze({
                        column: oColumn
                    });

                    // execute the column freezing
                    if (bExecuteDefault) {
                        if (bIsFixedColumn) {
                            oTable.setFixedColumnCount(0);
                        } else {
                            oTable.setFixedColumnCount(iColumnIndex + 1);
                        }
                    }
                }
            ));
        }
    };

    /**
     * Adds the column visibility menu item to the menu.
     * @private
     */
    ColumnMenu.prototype._addColumnVisibilityMenuItem = function() {
        var oTable = this._oTable;

        if (oTable && oTable.getShowColumnVisibilityMenu()) {
            var oColumnVisibiltyMenuItem = this._createMenuItem("column-visibilty", "TBL_COLUMNS");
            this.addItem(oColumnVisibiltyMenuItem);

            var oColumnVisibiltyMenu = new Menu(oColumnVisibiltyMenuItem.getId() + "-menu");
            oColumnVisibiltyMenu.addStyleClass("sapUiTableColumnVisibilityMenu");
            oColumnVisibiltyMenuItem.setSubmenu(oColumnVisibiltyMenu);

            var aColumns = oTable.retrieveLeafColumns();

            if (oTable.getColumnVisibilityMenuSorter && typeof oTable.getColumnVisibilityMenuSorter === "function") {
                var oSorter = oTable.getColumnVisibilityMenuSorter();
                if (typeof oSorter === "function") {
                    aColumns = aColumns.sort(oSorter);
                }
            }

            for (var i = 0, l = aColumns.length; i < l; i++) {
                var oMenuItem = this._createColumnVisibilityMenuItem(oColumnVisibiltyMenu.getId() + "-item-" + i, aColumns[i]);
                oColumnVisibiltyMenu.addItem(oMenuItem);
            }
        }
    };

    /**
     * Factory method for the column visibility menu item.
     * @param {string} sId the id of the menu item.
     * @param {sas.hc.ui.table.Column} oColumn the associated column to the menu item.
     * @return {sap.ui.unified.MenuItem} the created menu item.
     * @private
     */
    ColumnMenu.prototype._createColumnVisibilityMenuItem = function(sId, oColumn) {
        var sText = oColumn.getName() || (oColumn.getLabel() && oColumn.getLabel().getText ? oColumn.getLabel().getText() : null);
        return new MenuItem(sId, {
            text: sText,
            icon: oColumn.getVisible() ? this._getThemedIcon("ico_tick.png") : null,
            select: jQuery.proxy(function(oEvent) {
                var oMenuItem = oEvent.getSource();
                var bVisible = !oColumn.getVisible();
                if (bVisible || TableUtils.getVisibleColumnCount(this._oTable) > 1) {
                    var oTable = oColumn.getTable();
                    var bExecuteDefault = true;
                    if (oTable && (lazyInstanceof(oTable, "sas/hc/ui/table/Table"))) {
                        bExecuteDefault = oTable.fireColumnVisibility({
                            column: oColumn,
                            newVisible: bVisible
                        });
                    }
                    if (bExecuteDefault) {
                        oColumn.setVisible(bVisible);
                    }
                    //                oMenuItem.setIcon(bVisible ? "sap-icon://accept" : null);
                    oMenuItem.setIcon(bVisible ? this._getThemedIcon("ico_tick.png") : null);
                }
            }, this)
        });
    };

    /**
     * Adds the group menu item to the menu.
     * @private
     */
    ColumnMenu.prototype._addManageColumnsMenuItem = function() {
        var oTable = this._oTable;
        if (oTable && oTable.getShowManageColumnsMenuItem()) {
            this.addItem(this._createMenuItem(
                "manageColumns",
                "TBL_MANAGE_COLUMNS",
                null,
                jQuery.proxy(function(oEvent) {
                    oTable._showColumnsDialog();
                }, this)
            ));
        }
    };

    /**
     * Factory method for a menu item.
     * @param {string} sId the id of the menu item.
     * @param {string} sTextI18nKey the i18n key that should be used for the menu item text.
     * @param {string} sIcon the icon name
     * @param {function} fHandler the handler function to call when the item gets selected.
     * @return {sap.ui.unified.MenuItem} the created menu item.
     * @private
     */
    ColumnMenu.prototype._createMenuItem = function(sId, sTextI18nKey, sIcon, fHandler) {
        return new MenuItem(this.getId() + "-" + sId, {
            text: this.oResBundle.getText(sTextI18nKey+".txt"),
            icon: sIcon ? this._getThemedIcon(sIcon) : null,
            select: fHandler || function() {}
        });
    };

    /**
     * Factory method for a menu text field item.
     * @param {string} sId the id of the menu item.
     * @param {string} sTextI18nKey the i18n key that should be used for the menu item text.
     * @param {string} sIcon the icon name
     * @param {string} sValue the default value of the text field
     * @param {function} fHandler the handler function to call when the item gets selected.
     * @return {sap.ui.unified.MenuTextFieldItem} the created menu text field item.
     * @private
     */
    ColumnMenu.prototype._createMenuTextFieldItem = function(sId, sTextI18nKey, sIcon, sValue, fHandler) {
        jQuery.sap.require("sas.hc.ui.unified.MenuTextFieldItem");
        fHandler = fHandler || function() {};
        return new sas.hc.ui.unified.MenuTextFieldItem(this.getId() + "-" + sId, {
            label: this.oResBundle.getText(sTextI18nKey+".txt"),
            icon: sIcon ? this._getThemedIcon(sIcon) : null,
            value: sValue,
            select: fHandler || function() {}
        });
    };

    /**
     * sets a new filter value into the filter field
     * @param {String} sValue value of the filter input field to be set
     * @return {sas.hc.ui.table.ColumnMenu} this reference for chaining
     * @private
     */
    ColumnMenu.prototype._setFilterValue = function(sValue) {
        var oFilterField = sap.ui.getCore().byId(this.getId() + "-filter");
        if (oFilterField) {
            oFilterField.setValue(sValue);
        }
        return this;
    };

    /**
     * sets a new filter value into the filter field
     * @param {sap.ui.core.ValueState} sFilterState value state for filter text field item
     * @return {sas.hc.ui.table.ColumnMenu} this reference for chaining
     * @private
     */
    ColumnMenu.prototype._setFilterState = function(sFilterState) {
        var oColumn = this.getParent();
        var oTable = (oColumn ? oColumn.getTable() : undefined);

        var oFilterField = sap.ui.getCore().byId(this.getId() + "-filter");
        if (oFilterField && (oTable && !oTable.getEnableCustomFilter())) {
            oFilterField.setValueState(sFilterState);
        }
        return this;
    };


    function lazyInstanceof(o, sModule) {
        var FNClass = sap.ui.require(sModule);
        return typeof FNClass === 'function' && (o instanceof FNClass);
    }

    /**
     * Factory method. Creates the column menu.
     *
     * @return {sas.hc.ui.unified.Menu} The created column menu.
     * @private
     */
    ColumnMenu.prototype._createSubMenu = function(sId) {
        jQuery.sap.require("sas.hc.ui.unified.Menu"); // eslint-disable-line sashtmlcommons/no-deprecated-resource
        return new sas.hc.ui.unified.Menu(this.getId() + "-" + sId);
    };

    /**
     * Makes sure that all the menu items are ready for use.
     * @public
     */
    ColumnMenu.prototype.validateMenu = function() {
        if (this._bInvalidated) {
            this._bInvalidated = false;
            this.destroyItems();
            this._addMenuItems();
        }
    };

    /**
     * @private
     */
    ColumnMenu.prototype.focusAfterRendering = function(sWhere) {
        this._sFocusAfterRendering = sWhere;
    };

    /**
     * Returns a menu item by semantic id.
     * By semantic id, I mean what is used for the first arg to _createMenuItem,
     * which in turn get's appended to the id.
     * This provides a i18n-safe way of retrieving a menu item.
     *
     * @param {string} sSemanticId The semantic id to search for. Will be the suffix to the id.
     * @returns {object} oTargetMenuItem The desired menu item, or undefined if not found.
     * @private
     */
    ColumnMenu.prototype._getMenuItemBySemanticId = function( sSemanticId ) {
        var i, oItem, oTargetMenuItem, aItems = this.getItems();
        if( aItems ) {
            for( i = 0; i < aItems.length; i++ ) {
                oItem = aItems[i];
                if( oItem.getId().match(new RegExp("-" + sSemanticId + "$")) ) {
                    oTargetMenuItem = oItem;
                    break;
                }
            }
        }

        return oTargetMenuItem;
    };

    /**
     * Adds the 'Hide' menu item to the menu.
     * @private
     */
    ColumnMenu.prototype._addHideItem = function() {
        var oColumn = this._oColumn;
        var oTable = this._oTable;

        // hidden columns are not currently supported with column groups
        var ColumnGroup = sap.ui.requireSync("sas/hc/ui/table/ColumnGroup");
        if (oColumn instanceof ColumnGroup) {
            return;
        }

        if (oColumn.getResizable() || (oTable && oTable.getAllowColumnReordering())) {
            this.addItem(this._createMenuItem(
                "hide",
                "TBL_HIDE",
                null,
                function(oEvent) {
                    oColumn.setVisible(false);
                }
            ));
        }
    };

    /**
     * Adds the 'Resize or Move' menu item to the menu.
     * @private
     */
    ColumnMenu.prototype._addResizeOrMoveItem = function() {
        var self = this,
            oTable = this._oTable,
            oColumn = this._oColumn,
            iColIndex;

        if (oTable && oColumn && (lazyInstanceof(oTable, "sas/hc/ui/table/Table"))) {
            // if not the last column
            var item = this._createMenuItem(
                "resize",
                "TBL_RESIZE",
                null,
                function(oEvent) {
                    // if a summary header is present it needs to be given focus after Menu is closed
                    var $sumHeader = oTable.$().find(".sasUiTableColSumHead").eq(oColumn.getIndex());
                    if (Device.system.desktop === true) {
                        // remember if there is a summary header so it can recive focus when the menu is closed
                        if ($sumHeader.length === 1) {
                            self._lastFocusedDomRef = $sumHeader;
                        }
                        if (oColumn._resizeWithKeyboard) {
                            oColumn._resizeWithKeyboard();
                        }
                    } else {
                        iColIndex = Number(oColumn.$().attr("data-sap-ui-colindex"));
                        setTimeout(function() {
                            oTable.showColumnDragBar(iColIndex);
                        }, 0);
                    }
                }
            );
            this.addItem(item);
        }
    };

    /**
     * Adds the 'Freeze/Unfreeze' menu item to the menu.
     * @private
     */
    ColumnMenu.prototype._addFreezeItem = function() {
        var oColumn = this._oColumn;
        var oTable = this._oTable;

        if (this._shouldAddFreezeItem()) {
            var isFrozen = oTable.isFrozenColumn(oColumn);
            var rbKey = (isFrozen) ? "TBL_UNFREEZE" : "TBL_FREEZE";
            var item = this._createMenuItem(
                "freeze",
                rbKey,
                null,
                function(oEvent) {
                    // forward the event
                    var bExecuteDefault = oTable.fireColumnFreeze({
                        column: oColumn
                    });

                    //allow consumer to prevent action
                    if (bExecuteDefault) {
                        // execute the column freezing
                        if (oTable.isFrozenColumn(oColumn)) {
                            oTable.unfreezeColumn(oColumn);
                        } else {
                            oTable.freezeColumn(oColumn);
                        }
                    }
                }
            );
            this.addItem(item);
            this._oFreezeItem = item;   //save for later
        }
    };

    /**
     * @private
     */
    ColumnMenu.prototype._shouldAddFreezeItem = function() {
        return lazyInstanceof(this._oTable, "sas/hc/ui/table/Table")
            && this._oTable.getEnableColumnFreeze()
            && !this._oColumn.getMandatory();
    };

    /**
     * Adds the 'Sort' menu item to the menu along with submenu items.
     * @private
     */
    ColumnMenu.prototype._addSortMenu = function() {
        var oColumn = this._oColumn;
        var oTable = this._oTable;

        // sort is not supported on column groups
        var ColumnGroup = sap.ui.requireSync("sas/hc/ui/table/ColumnGroup");
        if (oColumn instanceof ColumnGroup) {
            return;
        }

        if (oColumn.getShowSortMenuEntry() === true){
            var item = this._createMenuItem(
                "sort",
                "TBL_SORT",
                null,
                null
            );

            var bSortable = (oColumn.getSortProperty() !== null &&
                oColumn.getSortProperty() !== undefined &&
                oColumn.getSortProperty() !== "");

            item.setEnabled(bSortable);

            if (bSortable) {
                var submenu = this._createSubMenu("sortMenu"),
                    bMultiSortEnabled = false;

                item.setSubmenu(submenu);

                // set the sorted flag and sort the model
                if (oTable && oTable.isBound("rows")) {
                    // sort the binding
                    var aSorters = oTable.getBinding("rows").aSorters;
                    if (aSorters !== null){
                        if (aSorters.length !== 0){
                            // the current column is not a primary, add the multi sort
                            if (aSorters[0].sPath !== oColumn.getSortProperty()){
                                bMultiSortEnabled = true;
                            }
                        }
                    }
                }

                this._addMultiSortMenuItem(false, submenu, bMultiSortEnabled);
                this._addMultiSortMenuItem(true, submenu, bMultiSortEnabled);
                this._addSortMenuItem(false, submenu);
                this._addSortMenuItem(true, submenu);
                this._addRemoveSortMenuItem(submenu);
            }
            this.addItem(item);
            this._oSortItem = item; //save for later
        }
    };

    /**
     * Adds the multiple sort menu item to the menu.
     * @param {boolean} bDesc the sort direction. <code>true</code> for descending.
     * @param {sap.ui.commons.Menu} oMenu the menu to add the item to (defaults to this).
     * @param {boolean} bEnabled whether the item should be disabled or not
     * @private
     */
    ColumnMenu.prototype._addMultiSortMenuItem = function(bDesc, oMenu, bEnabled) {
        var oColumn = this._oColumn;
        oMenu = oMenu || this;

        // sort is not supported on column groups
        var ColumnGroup = sap.ui.requireSync("sas/hc/ui/table/ColumnGroup");
        if (oColumn instanceof ColumnGroup) {
            return;
        }

        var sDir = bDesc ? "desc" : "asc";
        if (oColumn.getSortProperty() && oColumn.getShowSortMenuEntry()) {
            var item = this._createMenuItem(
                sDir+"-multi",
                "TBL_SORT_" + sDir.toUpperCase()+"_MULTI",
                null,
                function(oEvent) {
                    oColumn.sort(bDesc, true);  //always multi
                }
            );
            item.setEnabled(bEnabled);
            oMenu.addItem(item);
        }
    };

    /**
     * Adds the remove sort menu item to the menu.
     * @param {sap.ui.commons.Menu} oMenu the menu to add the item to (defaults to this).
     * @private
     */
    ColumnMenu.prototype._addRemoveSortMenuItem = function(oMenu) {
        var oColumn = this._oColumn,
            oMenu = oMenu || this,
            oTable = this._oTable,
            that = this;

        // sort is not supported on column groups
        var ColumnGroup = sap.ui.requireSync("sas/hc/ui/table/ColumnGroup");
        if (oColumn instanceof ColumnGroup) {
            return;
        }

        if (oColumn.getSortProperty() && oColumn.getShowSortMenuEntry()) {
            if (oTable && (lazyInstanceof(this._oTable, "sas/hc/ui/table/Table"))) {
                var isSorted = oTable.isColumnSorted(oColumn);
                var item = this._createMenuItem(
                    "remove",
                    "TBL_SORT_REMOVE",
                    null,
                    function(oEvent) {
                        oTable._removeColumnSort(oColumn);
                        // remove sort and add sort items' enabled state have change
                        that._bInvalidated = true;
                    }
                );
                item.setEnabled(isSorted);
                oMenu.addItem(item);
                this._oRemoveSortItem = item;   //save for later
            }
        }
    };

    /**
     * Adds the 'Resize to fit' menu item to the menu.
     * @private
     */
    ColumnMenu.prototype._addBestFitItem = function() {
        var oColumn = this._oColumn;
        var oTable = this._oTable;
        if (lazyInstanceof(this._oTable, "sas/hc/ui/table/Table")) {
            var item = this._createMenuItem(
                "bestFit",
                "TBL_COLUMN_FIT",
                null,
                function (oEvent) {
                    var iColumnIndex = oColumn.getIndex();
                    oTable.autoFitColumn(iColumnIndex);
                }
            );
            this.addItem(item);
        }
    };

    /**
     * Returns the path to the themed icon for a given icon.
     * @param {string} sIcon the icon name
     * @return {string} the path to the themed icon.
     * @private
     */
    ColumnMenu.prototype._getThemedIcon = function(sIcon) {
        // TODO Theme Icon has not been forked yet. Leaving this as sap.ui.table for now.
        return sas.themes.getThemeResourceUrl("sap.ui.table", "img/" + sIcon);
    };

    return ColumnMenu;
});
