// (c) 2015, SAS Institute Inc.
sap.ui.define([
    "jquery.sap.global",
    "sas/hc/ui/layout/VerticalLayout",
    "sas/hc/ui/layout/VerticalLayoutRenderer",
    "./_BaseSliderBody",
    "sas/hc/m/Label",
    "./operatorMenu/_OperatorsButton",
    "./operatorMenu/OperatorEnum",
    "./a11y/A11ySlider",
    "./_calculationUtil",
    "sas/hc/ui/core/util/StringUtil",
    "sap/ui/base/ManagedObjectMetadata",
    "sap/ui/core/Core",
    "sap/ui/core/CustomData",
    "sap/ui/core/format/NumberFormat",
    "sap/ui/core/ResizeHandler"
], function(jQuery, VerticalLayout, VerticalLayoutRenderer, BaseSliderBody, Label, OperatorsButton, OperatorEnum, A11ySlider, calculationUtil, StringUtil, ManagedObjectMetadata, sapCore, CustomData, NumberFormat, ResizeHandler) {
    "use strict";

    /**
     * Constructor for a new Slider.
     *
     * @class
     * Base class for either a vertical or horizontal slider.  This class is not intended to be instantiated directly.  Please see VerticalSlider or HorizontalSlider.
     *
     * ## Features: ##
     *
     * * supports variable number of thumbs and ranges of thumbs
     * * operator menu to change how the slider values are interpreted
     * * callout above thumbs for direct entry
     * * native HTML5 slider's behind the visual slider for a11y
     *
     * This is a re-write of the SAP-provided {@link sap.ui.commons.Slider} and {@link sap.ui.commons.RangeSlider}.
     * Rather than hard-code the number of allowed Thumbs as in SAP's approach, this slider
     * take any number of thumbs and range-thumbs by setting the _thumbs_ or _thumbRanges_ properties.
     *
     * @name sas.hc.m.slider.BaseSlider
     * @extends sas.hc.ui.layout.VerticalLayout
     * @version 904001.11.16.20251118090100_f0htmcm94p
     * @author Jennifer Cole, Jonathan Brink
     *
     * @constructor
     * @public
     */
    var BaseSlider = VerticalLayout.extend("sas.hc.m.slider.BaseSlider", /** @lends sas.hc.m.slider.BaseSlider.prototype */ {
        metadata: {
            library: "sas.hc.m",
            properties: {
                /** Width of the slider. */
                width: {type: "sap.ui.core.CSSSize", defaultValue: "", group: "Dimension"},

                /** Width of the slider. */
                minWidth: {type: "sap.ui.core.CSSSize", defaultValue: "", group: "Dimension"},

                /** height of the slider. */
                height: {type: "sap.ui.core.CSSSize", group: "Dimension"},

                /** Minimum value. */
                min: {type: "float", group: "Appearance", defaultValue: 0},

                /** Maximum value. */
                max: {type: "float", group: "Appearance", defaultValue: 100},

                /** Minimum value for date input. */
                minDate: {type: "object", group: "Appearance"},

                /** Maximum value for date input.*/
                maxDate: {type: "object", group: "Appearance"},

                /** Minimum SAS date value for date input. */
                minSASDate: {type: "float", group: "Appearance"},

                /** Maximum SAS date value for date input.*/
                maxSASDate: {type: "float", group: "Appearance"},

                /** Number of thumbs to display. */
                thumbs: {type: "int", defaultValue: 0, group: "Data"},

                /** Number of thumb ranges to display.*/
                thumbRanges: {type: "int", defaultValue: 0, group: "Data"},

                /** Whether to show the step (or tick) labels.*/
                showStepLabels: {type: "boolean", defaultValue: false, group: "Appearance"},

                /** The step (tick) labels to display. If not set, default labels will be displayed. */
                labels: {type: "string[]", defaultValue: null, group: "Misc"},

                /** Whether to show the slider label above the slider. */
                showLabel: {type: "boolean", defaultValue: true, group: "Appearance"},

                /** Whether to show the value label above the thumb or thumbRange. */
                showValueLabel: {type: "boolean", defaultValue: true, group: "Appearance"},

                /** A callback function used to format the values. */
                formatCallback: {type: "object", defaultValue: null, group: "Appearance"},

                /** Whether to display arrow thumbs.  (Default is round thumbs) */
                arrowThumbs: {type: "boolean", defaultValue: false, group: "Appearance"},

                /** Whether the slider is editable. */
                editable: {type: "boolean", defaultValue: true, group: "Behavior"},

                /** Whether the slider is enabled. */
                enabled: {type: "boolean", defaultValue: true, group: "Behavior"},

                /** Whether to show the input field for each thumb. */
                showInput: {type: "boolean", defaultValue: true, group: "Appearance"},

                /** The thumb can only be moved in steps of this width*/
                smallStepWidth: {type: "float", defaultValue: 1, group: "Appearance"},

                /** The thumb will be moved in steps of this width on shift+arrow */
                mediumStepWidth: {type: "float", defaultValue: 0, group: "Appearance"},

                /** The thumb be moved in steps of this width on page up/down */
                largeStepWidth: {type: "float", defaultValue: 0, group: "Appearance"},

                /** The number of units to display in the tick layout below the slider. */
                totalUnits: {type: "int", defaultValue: 0, group: "Appearance"},

                /** The slider label. */
                sliderLabel: {type: "string", defaultValue: "", group: "Appearance"},

                /** The metadata for the operators menu. */
                operatorMetadata: {type: "object", group: "Behavior"},

                /** Whether to show the operators menu button. */
                showOperators: {type: "boolean", defaultValue: false, group: "Appearance"},

                /** The format to be applied to the values. */
                format: {type:"string", defaultValue:null, group: "Appearance"},

                /** The operator for the slider. */
                operator: {type: "string", defaultValue: null, group: "Behavior"},

                /** The locale of the slider data, as opposed to the locale of the browser. */
                dataLocale: {type:"string", defaultValue:"en"},

                /** Array of thumb values, left-to-right */
                values: {type:"float[]", defaultValue:null},

                /** Array of thumb date values, left-to-right */
                dateValues: {type:"object[]", defaultValue:null},

                /** Array of thumb SAS date values, left-to-right */
                SASDateValues: {type:"float[]", defaultValue:null},

                /** Internally generated tooltip for the slider.  To set a different tooltip, use tooltipCallback */
                tooltip: {type:"string", defaultValue:null},

                /** A callback function used to generate the tooltip text. */
                tooltipCallback: {type: "object", defaultValue: null, group: "Appearance"},

                /** Type of input field to edit the slider values. Valid values are "Input", "DateInput", "TimeInput", and "DateTimeInput" */
                inputType: {type:"string", defaultValue:"Input"},

                /** Tells whether the width set on the horizontal slider should be applied to the track or the slider as a whole */
                applyWidthToTrack: {type:"boolean", defaultValue:false},

                /** Tells whether the height set on the vertical slider should be applied to the track or the slider as a whole */
                applyHeightToTrack: {type:"boolean", defaultValue:false},

                /** Tells whether the range label for values with decimal precision should be
                * rounded to fit within the width of the slider
                */
                roundPreciseValuesToFit: {type:"boolean", defaultValue:false},

                /** Tells whether the width of the input should be restricted to the width of the track */
                restrictInputWidth: {type:"boolean", defaultValue:false},

                /** Do not use this property. */
                /** Used internally to parse numeric values when no format is present, mostly for default locale formatting */
                /** @private */
                numberFormat: {type:"object", defaultValue:null},

                /** Do not use this property. */
                /** Tells whether the date values specified are SAS date numbers. The alternative is Java date numbers.  Default is false. */
                /** @private */
                useSASDates: {type:"boolean", defaultValue:false},

                /** Do not use this property. */
                /** Previous values set on the slider.  Used internally for the value change events */
                /** @private */
                previousValues: {type:"float[]", defaultValue:null},

                /** Do not use this property. */
                /** Tells whether the slider was disabled internally rather than by the consumer if enabled is false */
                /** @private */
                internallyDisabled: {type:"boolean", defaultValue:false},

                /** Do not use this property. */
                /** Tells whether the slider input was hidden internally rather than by the consumer if showInput is false */
                /** @private */
                internallyHideInput: {type:"boolean", defaultValue:false},

                /** Do not use this property. */
                /** Tells whether the slider should validate whether max > min when enabling */
                /** @private */
                validateMinMax: {type:"boolean", defaultValue:true},

                /** Do not use this property. */
                /** The internally stored small step width used to differentiate between consumer specified and internally calculated */
                /** @private */
                internalSmallStepWidth: {type:"float", defaultValue:0},

                /** Do not use this property. */
                /** The internally stored medium step width used to differentiate between consumer specified and internally calculated */
                /** @private */
                internalMediumStepWidth: {type:"float", defaultValue:0},

                /** Do not use this property. */
                /** The internally stored large step width used to differentiate between consumer specified and internally calculated */
                /** @private */
                internalLargeStepWidth: {type:"float", defaultValue:0}
            },
            aggregations: {
                /** The button that launches the operators menu. */
                operatorsButton: {type: "sas.hc.m.slider.operatorMenu.OperatorsButton", multiple: false},

                /** The hidden slider used for A11y. */
                a11ySlider: {type: "sas.hc.m.slider.a11y.A11ySlider", multiple: false},

                /** A control to be displayed above the track as a background */
                backgroundControl: {type:"sap.ui.core.Control", multiple: false}
            },
            associations: {
                /** The body of the slider.  Contains all parts except the slider label. */
                sliderBody: {type: "sas.hc.m.slider.BaseSliderBody"},

                /** The slider label object. */
                sliderLabelObject: {type: "sas.hc.m.Label"}
            },
            events: {
                /** Fired any time the values of the slider change (only fired on mouseup when dragging). */
                valueChange: {
                    // no parameters
                },
                /** Fired when the values of the slider are changing due to the thumb being dragged. */
                valueChanging: {
                    // no parameters
                },
                /** Fired when the values of the slider change due to user interaction. (This is not fired when the values change via API.) */
                liveChange: {
                    // no parameters
                }
            }
        },
        renderer: {
            getCustomCSSRules: function(oSlider) {
                var a = {},
                    sId = "#" + oSlider.getId(),
                    cssSelector = sId + ".sasSlider, " +
                                  sId + ".sasSlider .sapUiSliText," +
                                  sId + ".sasSlider .sasMLabel>.sasMLabelText:not(.sasSliderCalloutLabelSpan)," +
                                  sId + ".sasSlider .sapMInputBaseInner," +
                                  sId + ".sasSlider .sapMInputBaseState .sapMInputAttentionStatus," +
                                  sId;

                a[cssSelector] = oSlider.getCustomStyles().getStyles();

                return a;
            },

            render: function(oRm, oSlider) {
                if (oSlider.getHeight()) {
                    oRm.addStyle("height", oSlider.getHeight());
                }
                VerticalLayoutRenderer.render(oRm,oSlider);
            }
        }
    });

    jQuery.sap.require("sap.ui.core.EnabledPropagator");
    sap.ui.core.EnabledPropagator.call(BaseSlider.prototype);

    /**
     * Clone the slider.
     * @override
     */
    BaseSlider.prototype.clone = function(sIdSuffix) {
        var oClone,
            oBackgroundControl = this.getBackgroundControl(),
            aCustomData = this.getAggregation("customData"),
            sId,
            mSettings = {},
            mProps = this.mProperties,
            sKey,
            oMetadata = this.getMetadata(),
            oClass = oMetadata._oClass;

        if (StringUtil.isEmpty(sIdSuffix)) {
            sIdSuffix = ManagedObjectMetadata.uid("clone") || jQuery.sap.uid();
        }
        sId = this.getId() + "-" + sIdSuffix;

        // Clone properties (only those with non-default value)
        for(sKey in mProps) {
            //do not clone properties if property is bound; Property is set on update
            if ( mProps.hasOwnProperty(sKey) && !(this.isBound(sKey))){
                mSettings[sKey] = mProps[sKey];
            }
        }

        // Clone models
        mSettings["models"] = this.oModels;

        // Clone BindingContext
        mSettings["bindingContexts"] = this.oBindingContexts;

        oClone = new oClass(sId, mSettings);

        /* Clone element bindings: Clone the objects not the parameters
         * Context will only be updated when adding the control to the control tree;
         * Maybe we have to call updateBindingcontext() here?
         */
        jQuery.each(this.mBoundObjects, function(sName, oBoundObject) {
            oClone.mBoundObjects[sName] = jQuery.extend({}, oBoundObject);
        });

        // Clone events
        jQuery.each(this.mEventRegistry, function(sName, aListeners) {
            oClone.mEventRegistry[sName] = aListeners.slice();
        });

        // Clone bindings
        jQuery.each(this.mBindingInfos, function(sName, oBindingInfo) {
            var oCloneBindingInfo = jQuery.extend({}, oBindingInfo);
            delete oCloneBindingInfo.binding; // remove the runtime binding info (otherwise the property will not be connected again!)
            if (oBindingInfo.factory) {
                oClone.bindAggregation(sName, oCloneBindingInfo);
            } else {
                oClone.bindProperty(sName, oCloneBindingInfo);
            }
        });

        if (oBackgroundControl !== null && oBackgroundControl !== undefined) {
            oClone.setBackgroundControl(oBackgroundControl.clone(sIdSuffix));
        }

        if (aCustomData !== null && aCustomData !== undefined && aCustomData.length > 0) {
            for (var i = 0; i < aCustomData.length; i++) {
                var dataObject = new CustomData({key:aCustomData[i].getKey()});
                if (aCustomData[i].isBound("value")) {
                    jQuery.each(aCustomData[i].mBindingInfos, function(sName, oBindingInfo) {
                        var oCloneBindingInfo = jQuery.extend({}, oBindingInfo);
                        delete oCloneBindingInfo.binding; // remove the runtime binding info (otherwise the property will not be connected again!)
                        if (oBindingInfo.factory) {
                            dataObject.bindAggregation(sName, oCloneBindingInfo);
                        } else {
                            dataObject.bindProperty(sName, oCloneBindingInfo);
                        }
                    });
                } else {
                    dataObject.setValue(aCustomData[i].getValue());
                }
                oClone.addAggregation("customData", dataObject, true);
            }
        }
        return oClone;
    };


    // =============================================================================
    // LIFECYCLE FUNCTIONS
    // =============================================================================

    /**
     * init lifecycle callback
     * @private
     */
    BaseSlider.prototype.init = function() {
        var self = this,
            sId = this.getId(),
            oOperatorsButton,
            oA11ySlider;

        if (VerticalLayout.prototype.init) {
            VerticalLayout.prototype.init.apply(this, arguments);
        }

        this.addStyleClass("sasSlider");

        // create operators button
        oOperatorsButton = new OperatorsButton({
            id: sId + "-operatorsButton",
            slider: this,
            select: function(e) {
                var oSelectedItem = e.getParameter("selectedKey");
                if (oSelectedItem) {
                    self.setOperator(oSelectedItem);
                }
            }
        });

        // create a11y slider
        oA11ySlider = new A11ySlider({
            id: sId + "-a11ySlider",
            visualSlider: this
        });

        // populate aggregations
        this.setOperatorsButton(oOperatorsButton);
        this.setA11ySlider(oA11ySlider);

        //we have to store this resizeHandler registration number
        //so that we can deregister the handler before re-rendering
        this._resizeHandlerRegistrationID = null;

        this.setNumberFormat(NumberFormat.getFloatInstance({
            groupingEnabled: true
        }));
    };

    BaseSlider.prototype.destroy = function() {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody()),
            oSliderLabelObject = sap.ui.getCore().byId(this.getSliderLabelObject());

        if (oSliderBody) {
            oSliderBody.destroy();
        }
        if (oSliderLabelObject) {
            oSliderLabelObject.destroy();
        }
        if (VerticalLayout.prototype.destroy) {
            VerticalLayout.prototype.destroy.apply(this, arguments);
        }
    };

    /**
     * onBeforeRendering lifecycle callback
     * @private
     */
    BaseSlider.prototype.onBeforeRendering = function() {
        if (VerticalLayout.prototype.onBeforeRendering) {
            VerticalLayout.prototype.onBeforeRendering.apply(this, arguments);
        }

        //deregister the resize handler until after rendering is complete
        if (this._resizeHandlerRegistrationID) {
            ResizeHandler.deregister(this._resizeHandlerRegistrationID);
            this._resizeHandlerRegistrationID = null;
        }

        if (this.getBackgroundControl()) {
            this.addStyleClass("withBackgroundControl");
        }

        this._cleanupEvents();

        this._manageEnabled();
    };

    /**
     * onAfterRendering lifecycle callback
     * @private
     */
    BaseSlider.prototype.onAfterRendering = function() {
        var self = this,
            $this = this.$(),
            minWidth;

        if (VerticalLayout.prototype.onAfterRendering) {
            VerticalLayout.prototype.onAfterRendering.apply(this, arguments);
        }

        this.$().on("selectstart.baseslider", function() {
            return false;
        });

        $this.on("focusin.baseslider", function() {
            $this.off("focusin.baseslider");
        });

        if (this.getProperty("tooltip") !== null && this.getProperty("tooltip") !== undefined) {
            document.getElementById(this.getId()).title = this.getProperty("tooltip");
        }

        //register a handler for resize
        if (!this._resizeHandlerRegistrationID) {
            this._resizeHandlerRegistrationID = ResizeHandler.register(this, self.handleResize);
        }

        //if there is a minWidth, set it on the DOM
        minWidth = this.getMinWidth();
        if (this.$().length > 0 && !StringUtil.isEmpty(minWidth)) {
            this.$().css("min-width", minWidth);
        }

        this._calculateStepWidths();

        self._reveal();
    };

    BaseSlider.prototype.renderA11yControl = function() {
        var oA11ySlider = this.getA11ySlider();
        sap.ui.getCore().createRenderManager().render(oA11ySlider, this.getDomRef());
    };

    BaseSlider.prototype.handleResize = function(oEvent) {
        var oSlider,
            oSliderBody,
            oTickLayout,
            aAllThumbs,
            oCallout,
            oTrack,
            oThumbGroups,
            oA11ySlider;

        if (oEvent.oldSize.width === oEvent.size.width && oEvent.oldSize.height === oEvent.size.height) {
            return;
        }

        if (oEvent.currentTarget) {
            oSlider = sap.ui.getCore().byId(oEvent.currentTarget.id);
            if (oSlider) {
                oA11ySlider = oSlider.getA11ySlider();
                oSliderBody = sap.ui.getCore().byId(oSlider.getSliderBody());
                oTrack = oSliderBody.getTrack();

                aAllThumbs = oTrack.retrieveAllThumbs();

                //invalidate the tick layout so it will resize itself
                oTickLayout = oSliderBody.getTickLayout();
                if (oTickLayout) {
                    oTickLayout.invalidate();
                }

                oSliderBody._positionSliderBody();
                oA11ySlider._positionA11ySlider();

                //reposition the range slider labels
                oThumbGroups = oTrack.getThumbGroups();
                oThumbGroups.forEach(function(oThumbGroup) {
                    oSliderBody._positionRangeSliderLabel(oThumbGroup.getHighlightBar());
                });
                aAllThumbs.forEach(function(oThumb) {
                    oCallout = oThumb.getCallout();
                    if (oCallout) {
                        oCallout.invalidate();
                    }
                });
            }
        }
    };

    /**
     * exit lifecycle callback
     * @private
     */
    BaseSlider.prototype.exit = function() {
        if (VerticalLayout.prototype.exit) {
            VerticalLayout.prototype.exit.apply(this, arguments);
        }
        this._cleanupEvents();
    };

    // =============================================================================
    // SETTERS/GETTERS
    // most of these pass-through to SliderBody
    // =============================================================================

    /**
     * sliderLabel setter override
     * Avoids invalidating Slider, but invalidates Body
     * @param {string} sSliderLabel label for the slider
     * @public
     */
    BaseSlider.prototype.setSliderLabel = function(sSliderLabel) {
        var sId = this.getId(),
            oSliderLabelObject = sap.ui.getCore().byId(this.getSliderLabelObject()),
            oSliderBody = sap.ui.getCore().byId(this.getSliderBody());

        this.setProperty("sliderLabel", sSliderLabel, true);
        oSliderBody.setSliderLabel(sSliderLabel);

        if (!oSliderLabelObject) {
            oSliderLabelObject = new Label({
                id: sId + "-sliderLabel",
                text: sSliderLabel
            });
            this.insertContent(oSliderLabelObject,0);
            this.setSliderLabelObject(oSliderLabelObject);
        } else {
            oSliderLabelObject.setText(sSliderLabel);
        }
        oSliderLabelObject.addStyleClass("sasSliderLabel");

        if (this.getShowLabel()) {
            oSliderLabelObject.removeStyleClass("sasMHide");
        } else {
            oSliderLabelObject.addStyleClass("sasMHide");
        }
    };

    /**
     * showLabel setter override
     * Avoids invalidating Slider, but invalidates the A11Y slider so value labels will be repositioned
     * @param {boolean} bShowLabel whether to display the slider label
     * @public
     */
    BaseSlider.prototype.setShowLabel = function(bShowLabel) {
        var oSliderLabelObject = sap.ui.getCore().byId(this.getSliderLabelObject()),
            oSliderBody = sap.ui.getCore().byId(this.getSliderBody()),
            oA11ySlider = this.getA11ySlider();

        this.setProperty("showLabel", bShowLabel, true);
        oSliderBody.setShowLabel(bShowLabel);

        if (oSliderLabelObject) {
            if (bShowLabel) {
                oSliderLabelObject.removeStyleClass("sasMHide");
            } else {
                oSliderLabelObject.addStyleClass("sasMHide");
            }
        }

        //invalidate the a11ySlider because it needs to be repositioned
        //so that the callouts appear in the correct location
        if (oA11ySlider) {
            oA11ySlider.invalidate();
        }
    };

    /**
     * min setter override
     * Avoids invalidating Slider, but invalidates the tick layout and track
     * @param {boolean} fMin the min value of the slider
     * @public
     */
    BaseSlider.prototype.setMin = function(fMin) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("min", fMin, true);
        oSliderBody.setMin(fMin);
        this._calculateStepWidths();
    };

    /**
     * max setter override
     * Avoids invalidating Slider, but invalidates the tick layout and track
     * @param {boolean} fMax the max value of the slider
     * @public
     */
    BaseSlider.prototype.setMax = function(fMax) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("max", fMax, true);
        oSliderBody.setMax(fMax);
        this._calculateStepWidths();
    };

    BaseSlider.prototype.setMinDate = function(dMin) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("minDate", dMin, true);
        oSliderBody.setMinDate(dMin);
    };

    BaseSlider.prototype.getMinDate = function() {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());

        return oSliderBody.getMinDate();
    };

    BaseSlider.prototype.setMaxDate = function(dMax) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("maxDate", dMax, true);
        oSliderBody.setMaxDate(dMax);
    };

    BaseSlider.prototype.getMaxDate = function() {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        return oSliderBody.getMaxDate();
    };

    BaseSlider.prototype.setMinSASDate = function(fMinSASDate) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("minSASDate", fMinSASDate, true);
        oSliderBody.setMinSASDate(fMinSASDate);
    };

    BaseSlider.prototype.getMinSASDate = function() {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        return oSliderBody.getMinSASDate();
    };

    BaseSlider.prototype.setMaxSASDate = function(fMaxSASDate) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("maxSASDate", fMaxSASDate, true);
        oSliderBody.setMaxSASDate(fMaxSASDate);
    };

    BaseSlider.prototype.getMaxSASDate = function() {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        return oSliderBody.getMaxSASDate();
    };

    BaseSlider.prototype.setThumbs = function(iThumbs) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("thumbs", iThumbs);
        oSliderBody.setThumbs(iThumbs);
    };

    BaseSlider.prototype.setThumbRanges = function(iThumbRanges) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("thumbRanges", iThumbRanges);
        oSliderBody.setThumbRanges(iThumbRanges);
    };

    BaseSlider.prototype.setShowStepLabels = function(bShowStepLabels) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("showStepLabels", bShowStepLabels);
        oSliderBody.setShowStepLabels(bShowStepLabels);
    };

    /**
     * setLabels setter override
     * Avoids invalidating Slider, but invalidates TickLayout
     * @param {array} saLabels the label text next to each tick mark
     * @public
     */
    BaseSlider.prototype.setLabels = function(saLabels) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("labels", saLabels, true);
        oSliderBody.setLabels(saLabels);
    };

    BaseSlider.prototype.setShowValueLabel = function(bShowValueLabel) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("showValueLabel", bShowValueLabel);
        oSliderBody.setShowValueLabel(bShowValueLabel);
    };

    BaseSlider.prototype.setFormatCallback = function(fnFormatCallback) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("formatCallback", fnFormatCallback, true);
        oSliderBody.setFormatCallback(fnFormatCallback);
    };

    BaseSlider.prototype.setArrowThumbs = function(bArrowThumbs) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("arrowThumbs", bArrowThumbs, true);
        oSliderBody.setArrowThumbs(bArrowThumbs);
    };

    /**
     * setEditable setter override
     * Avoids invalidating Slider
     * @param {boolean} bEditable whether the slider is editable
     * @public
     */
    BaseSlider.prototype.setEditable = function(bEditable) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("editable", bEditable, true);
        oSliderBody.setEditable(bEditable);
    };

    /**
     * setInternallyDisabled setter override
     * Avoids invalidating Slider
     * @param {boolean} bDisabled whether the slider has been disabled internally
     * @private
     */
    BaseSlider.prototype.setInternallyDisabled = function(bDisabled) {
        this.setProperty("internallyDisabled", bDisabled, true);
    };

    /**
     * setInternallyHideInput setter override
     * Avoids invalidating Slider
     * @param {boolean} bDisabled whether the slider has been disabled internally
     * @private
     */
    BaseSlider.prototype.setInternallyHideInput = function(bHide) {
        this.setProperty("internallyHideInput", bHide, true);
    };

    /**
     * setEnabled setter override
     * Avoids invalidating Slider
     * @param {boolean} bEnabled whether the slider is enabled
     * @public
     */
    BaseSlider.prototype.setEnabled = function(bEnabled) {
        this.setProperty("enabled", bEnabled, true);
        this._manageEnabled();
    };

    /**
     * Manage the behavior that needs to occur relating to the "enabled" property
     * This function is meant to be invoked from within onBeforeRendering
     * @public
     */
    BaseSlider.prototype._manageEnabled = function() {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody()),
            oSliderLabelObject = sap.ui.getCore().byId(this.getSliderLabelObject()),
            bInvalidMinMax = false,
            bEnabled = this.getEnabled();

        // err on side of true if a non-boolean value is given
        if (bEnabled !== false) {
            bEnabled = true;
        }

        if (this.getMin() >= this.getMax()) {
            bInvalidMinMax = true;
        }

        //don't enable the slider if the min is greater than or equal to the max
        if (this.getValidateMinMax() === true && bInvalidMinMax === true && bEnabled === true) {
            this.setProperty("enabled", false, true);
            this.setInternallyDisabled(true);
            return;
        }

        if (bEnabled === true) {
            this.removeStyleClass("sasSliderDisabled");
        } else {
            if (!this.hasStyleClass("sasSliderDisabled")) {
                this.addStyleClass("sasSliderDisabled");
            }
        }
        oSliderBody.setEnabled(bEnabled);

        if (oSliderLabelObject !== undefined) {
            oSliderLabelObject.setEnabled(bEnabled);
        }
    };

    /**
     * setShowInput setter override
     * Avoids invalidating Slider
     * @param {boolean} bShowInput whether whether to show the input for editing
     * @public
     */
    BaseSlider.prototype.setShowInput = function(bShowInput) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("showInput", bShowInput, true);
        oSliderBody.setShowInput(bShowInput);
    };

    /**
     * setSmallStepWidth setter override
     * Avoids invalidating Slider
     * @param {float} fSmallStepWidth width of steps that thumb can be moved
     * @public
     */
    BaseSlider.prototype.setSmallStepWidth = function(fSmallStepWidth) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        //setting smallStepwidth needs to invalidate the slider
        //because it can affect the min and max
        this.setProperty("smallStepWidth", fSmallStepWidth, true);
        oSliderBody.setSmallStepWidth(fSmallStepWidth);
        this.setInternalSmallStepWidth(fSmallStepWidth);
    };

    /**
     * setMediumStepWidth setter override
     * Avoids invalidating Slider
     * @param {float} fMediumStepWidth width of steps that thumb will be moved with shift+arrow
     * @public
     */
    BaseSlider.prototype.setMediumStepWidth = function(fMediumStepWidth) {
        // when step widths are set publicly, set the pulic step width as well as the internal step width
        this.setProperty("mediumStepWidth", fMediumStepWidth, true);
        this.setInternalMediumStepWidth(fMediumStepWidth);
    };

    /**
     * setLargeStepWidth setter override
     * Avoids invalidating Slider
     * @param {float} fLargeStepWidth width of steps that thumb will be moved with page up/down
     * @public
     */
    BaseSlider.prototype.setLargeStepWidth = function(fLargeStepWidth) {
        // when step widths are set publicly, set the pulic step width as well as the internal step width
        this.setProperty("largeStepWidth", fLargeStepWidth, true);
        this.setInternalLargeStepWidth(fLargeStepWidth);
    };

    /**
     * setInternalSmallStepWidth setter override
     * Avoids invalidating Slider
     * @param {float} fSmallStepWidth width of steps that thumb will be moved with arrow
     * @private
     */
    BaseSlider.prototype.setInternalSmallStepWidth = function(fSmallStepWidth) {
        // when step widths are set internally, do not set the public step width
        this.setProperty("internalSmallStepWidth", fSmallStepWidth, true);
    };

    /**
     * setInternalMediumStepWidth setter override
     * Avoids invalidating Slider
     * @param {float} fMediumStepWidth width of steps that thumb will be moved with shift+arrow
     * @private
     */
    BaseSlider.prototype.setInternalMediumStepWidth = function(fMediumStepWidth) {
        // when step widths are set internally, do not set the public step width
        this.setProperty("internalMediumStepWidth", fMediumStepWidth, true);
    };

    /**
     * setInternalLargeStepWidth setter override
     * Avoids invalidating Slider
     * @param {float} fLargeStepWidth width of steps that thumb will be moved with page up/down
     * @private
     */
    BaseSlider.prototype.setInternalLargeStepWidth = function(fLargeStepWidth) {
        // when step widths are set internally, do not set the public step width
        this.setProperty("internalLargeStepWidth", fLargeStepWidth, true);
    };

    /**
     * totalUnits setter override
     * Avoids invalidating Slider, but invalidates TickLayout
     * @param {int} iTotalUnits number of tick units to display
     * @public
     */
    BaseSlider.prototype.setTotalUnits = function(iTotalUnits) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("totalUnits", iTotalUnits, true);
        oSliderBody.setTotalUnits(iTotalUnits);
    };

    BaseSlider.prototype.getOperator = function() {
        var oOperatorsButton = this.getOperatorsButton();
        return oOperatorsButton.getOperator();
    };

    BaseSlider.prototype.setOperator = function(sOperator) {
        var oOperatorsButton = this.getOperatorsButton(),
            sPrevOperator = this.getOperator(),
            oTrack = sap.ui.getCore().byId(this.getSliderBody()).getTrack();

        if (sOperator === sPrevOperator) {
            return;
        }
        this.setProperty("operator", sOperator);
        //adjust the values when switching between operators with thumbs vs thumb ranges
        oTrack.adjustThumbValuesFromOperatorChange(sPrevOperator, sOperator);
        oOperatorsButton.setOperator(sOperator);
        this._generateTooltipForOperator();
    };

    BaseSlider.prototype.setOperatorMetadata = function(aValues) {
        var oOperatorsButton = this.getOperatorsButton();
        oOperatorsButton.setOperatorMetadata(aValues);
    };

    BaseSlider.prototype.getOperatorMetadata = function() {
        var oOperatorsButton = this.getOperatorsButton();
        return oOperatorsButton.getOperatorMetadata();
    };

    BaseSlider.prototype.setFormat = function(sFormat) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody()),
            aSASDateValues;
        this.setProperty("format", sFormat, true);
        if (this._applySASDates) {
            aSASDateValues = this.getProperty("SASDateValues");
            if (aSASDateValues && aSASDateValues.length > 0) {
                this.setSASDateValues(aSASDateValues);
            }
        }
        oSliderBody.setFormat(sFormat);
    };

    BaseSlider.prototype.setDataLocale = function(sLocale) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("dataLocale", sLocale);
        oSliderBody.setDataLocale(sLocale);
    };

    BaseSlider.prototype.setTooltip = function(sTooltipText) {
        // the tooltip cannot be set by the consumer
        // use tooltipCallback instead
        return;
    };

    /**
     * values setter override
     * Avoids invalidating Slider
     * @param {array} faValues array of float values
     * @public
     */
    BaseSlider.prototype.setValues = function(faValues) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());

        if (faValues && faValues.length > 0 && (faValues[0] === null || faValues[0] === undefined)) {
            return;
        }

        this.setProperty("values", faValues, true);
        oSliderBody.setValues(faValues);
    };

    /**
     * dateValues setter override
     * Avoids invalidating Slider
     * @param {array} aDateValues array of date values
     * @public
     */
    BaseSlider.prototype.setDateValues = function(aDateValues) {
        var faValues = [];

        aDateValues.forEach(function (oDate) {
            if (oDate !== null && oDate !== undefined) {
                faValues[faValues.length] = oDate.valueOf();
            }
        });
        this.setValues(faValues);
    };

    /**
     * SASDateValues setter override
     * Avoids invalidating Slider
     * @param {array} aSASDateValues array of SAS date values
     * @public
     */
    BaseSlider.prototype.setSASDateValues = function(aSASDateValues) {
        var faValues = [],
            oJsDate,
            sFormat = this.getFormat();

        this.setProperty("SASDateValues", aSASDateValues, true);
        if (sFormat) {
            jQuery.sap.require("sas.hc.m.slider._formatUtil");
            aSASDateValues.forEach(function (fSASDate) {
                if (fSASDate !== null) {
                    oJsDate = sas.hc.m.slider._formatUtil.SASDateToJsDate(fSASDate, sFormat);
                    if (oJsDate) {
                        faValues[faValues.length] = oJsDate.valueOf();
                    }
                }
            });
            this.setValues(faValues);
            this._applySASDates = false;
        } else {
            this._applySASDates = true;
        }
    };

    BaseSlider.prototype.getValues = function() {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        return oSliderBody.getValues();
    };

    BaseSlider.prototype.getDateValues = function() {
        var faValues = this.getValues(),
            aDateValues = [];

        faValues.forEach(function (fValue) {
            if (fValue !== null && fValue !== undefined) {
                aDateValues[aDateValues.length] = new Date(fValue);
            }
        });
        return aDateValues;
    };

    BaseSlider.prototype.getSASDateValues = function() {
        var faValues = this.getValues(),
            aSASDateValues = [],
            fSASDate,
            sFormat = this.getFormat(),
            oJsDate;

        if (sFormat) {
            jQuery.sap.require("sas.hc.m.slider._formatUtil");
            faValues.forEach(function (fValue) {
                if (fValue !== null && fValue !== undefined) {
                    oJsDate = new Date(fValue);
                    fSASDate = sas.hc.m.slider._formatUtil.jsDateToSASDate(oJsDate, sFormat);
                    aSASDateValues[aSASDateValues.length] = fSASDate;
                }
            });
            return aSASDateValues;
        } else {
            return this.getProperty("SASDateValues");
        }
    };

    BaseSlider.prototype.setInputType = function(sInputType) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        this.setProperty("inputType", sInputType);
        oSliderBody.setInputType(sInputType);
    };

    BaseSlider.prototype.setUseSASDates = function(bUseSASDates) {
        // do nothing
        // this private property is not used
    };

    BaseSlider.prototype.setSliderLabelObject = function(oSliderLabelObject) {
        oSliderLabelObject.setEnabled(this.getEnabled());
        this.setAssociation("sliderLabelObject", oSliderLabelObject);
    };

    /**
     * setRoundPreciseValuesToFit setter override
     * Avoids invalidating Slider, but invalidates value labels
     * @param {boolean} bRound whether to round the values for the label
     * @public
     */
    BaseSlider.prototype.setRoundPreciseValuesToFit = function(bRound) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody()),
            oTrack = oSliderBody.getTrack(),
            oThumbGroups = oTrack.getThumbGroups(),
            oHighlightBar,
            aThumbs,
            oCallout;

        this.setProperty("roundPreciseValuesToFit", bRound, true);
        oThumbGroups.forEach(function(oThumbGroup) {
            oHighlightBar = oThumbGroup.getHighlightBar();
            oSliderBody.renderRangeSliderLabel(oHighlightBar);
            aThumbs = oThumbGroup.getThumbs();
            aThumbs.forEach(function(oThumb) {
                oCallout = oThumb.getCallout();
                if (oCallout) {
                    oCallout.invalidate();
                }
            });
        });
    };

    BaseSlider.prototype.setRestrictInputWidth = function(bRestrictWidth) {
        this.setProperty("restrictInputWidth", bRestrictWidth, true);
    };

    // =============================================================================
    // PUBLIC API
    // =============================================================================

    /**
     * Return array of thumb values from left-to-right
     * @returns {float[]} Array of thumb values, left-to-right
     * @public
     */
    // BaseSlider.prototype.retrieveValues = function() {
    //     var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
    //     return oSliderBody.retrieveValues();
    // };

    /**
     * Set thumb values all at once, left-to-right
     * @param {array} faValues (array of floats) The values of all thumbs from left-to-right
     * @public
     */
    // BaseSlider.prototype.applyValues = function(faValues) {
    //     var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
    //     oSliderBody.applyValues(faValues);
    // };

    /**
     * Retrieve a particular thumb value by index (position)
     * For example, the 2nd thumb from the left.
     * @param {int} iIndex The index of the thumb
     * @returns {float} The thumb value
     * @public
     */
    BaseSlider.prototype.retrieveValueAt = function(iIndex) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        return oSliderBody.retrieveValueAt(iIndex);
    };

    /**
     * Set a thumb value at a particular index (position)
     * For example, the 2nd thumb from the left.
     * @param {float} fValue The value to set the thumb at
     * @param {int} iIndex The index of the thumb
     * @public
     */
    BaseSlider.prototype.applyValueAt = function(fValue, iIndex) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        oSliderBody.applyValueAt(fValue, iIndex);
    };

    /**
     * find out whether this is a vertical slider
     * @returns {boolean} whether or not this belongs to a vertical slider
     */
    BaseSlider.prototype.isVertical = function() {
        return /\.VerticalSlider$/.test(this.getMetadata().getName());
    };

    /**
     * Find out whether the slider has been revealed yet (_reveal invoked)
     * Determine using SliderBody's opacity
     * @returns {boolean} if previously revealed
     */
    BaseSlider.prototype.isRevealed = function() {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        if (oSliderBody && oSliderBody.$()) {
            return Number(oSliderBody.$().css("opacity") || 0) === 1;
        }
        return false;
    };

    /**
     * Get the formatted value for the given raw value
     * @param {float} fValue the raw value
     * @returns {string} the formatted value
     */
    BaseSlider.prototype.retrieveFormattedValue = function(fValue, sFormat) {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody());
        if (oSliderBody) {
            return oSliderBody.retrieveFormattedValue(fValue, sFormat);
        } else {
            return fValue;
        }
    };


    // =============================================================================
    // PRIVATE FUNCTIONS
    // =============================================================================

    /**
     * dom event cleanup
     * @private
     */
    BaseSlider.prototype._cleanupEvents = function() {
        var oOperatorsButton = this.getOperatorsButton();
        if (this.$()) {
            this.$().off("selectstart.baseslider");
            this.$().off("focusin.baseslider");
        }
        if (oOperatorsButton && oOperatorsButton.$()) {
            oOperatorsButton.$().off("focusin.basesliderformatopbutton");
        }
    };

    /**
     * Because Slider performs sizing measurments after rendering
     * elements move around while the Slider is getting settled.
     *
     * To avoid the consumer seeing the Slider before it's ready
     * it initially comes up with an opacity of 0 (set via css)
     *
     * The sub-controls are expected to fire an event signaling that
     * they are ready, and this event bubbles up and eventually invokes
     * this function which set's the opacity to 1
     *
     * @private
     */
    BaseSlider.prototype._reveal = function() {
        var $this = this.$(),
            oSliderBody = sap.ui.getCore().byId(this.getSliderBody()),
            $sliderBody = oSliderBody.$(),
            oA11ySlider = this.getA11ySlider(),
            oOperatorButton = this.getOperatorsButton(),
            $operatorButton,
            $rangeSliderLabels = jQuery(".sliderRangeHighlightBar", $this);

        if (oOperatorButton) {
            $operatorButton = oOperatorButton.$();
        }

        // invalidate all the callouts so they can appear
        // this is because after they are rendered they
        // match the opacity that slider body has
        function invalidateCallouts() {
            if (oA11ySlider) {
                //the callouts are part of the A11ySlider
                //so invalidating that will invalidate all callouts
                oA11ySlider.invalidate();
            }
        }

        $sliderBody
            .add($operatorButton)
            .add($rangeSliderLabels)
            .animate({
                opacity: 1
            }, 100, invalidateCallouts);
    };

    /**
     * Return a sample string used for default positioning of callout input elements
     * @returns {string} The sample string
     * @private
     */
    BaseSlider.prototype._generateSampleInputString = function() {
        var sMax = Math.round(this.getMax()) + "",
            sMin = Math.round(this.getMin()) + "",
            fStepWidth = this.getSmallStepWidth(),
            iMaxLength = String(sMax).length,
            iMinLength = String(sMin).length,
            iGreatestLength = Math.max(iMinLength, iMaxLength),
            iDecIndex = String(fStepWidth).indexOf('.'),
            iPrecision,
            sSampleInputString = "",
            i,
            sFormat = this.getFormat(),
            sInputType = this.getInputType();

        if (sInputType === "DateTimeInput") {
            if (!StringUtil.isEmpty(sFormat)) {
                jQuery.sap.require("sas.hc.m.slider._formatUtil");
                return sas.hc.m.slider._formatUtil.retrieveMaxFormattedString(sFormat);
            }
            return "September 00, 0000 00:00:00 PM";
        }
        if (sInputType === "DateInput") {
            if (!StringUtil.isEmpty(sFormat)) {
                jQuery.sap.require("sas.hc.m.slider._formatUtil");
                return sas.hc.m.slider._formatUtil.retrieveMaxFormattedString(sFormat);
            }
            return "September 00, 0000";
        }
        if (sInputType === "TimeInput") {
            if (!StringUtil.isEmpty(sFormat)) {
                jQuery.sap.require("sas.hc.m.slider._formatUtil");
                return sas.hc.m.slider._formatUtil.retrieveMaxFormattedString(sFormat);
            }
            return "00:00:00 PM";
        }

        //if min is a negative number and the min value is more chars than the max value,
        //add a "-" to the sample input string and decrease the number of characters needed by 1
        if (this.getMin() < 0 && iMinLength > iMaxLength) {
            sSampleInputString = sSampleInputString + "-";
            iGreatestLength--;
        }
        for (i = 0; i < iGreatestLength; i++) {
            //add a comma if we need it
            if (i > 0 && ((iGreatestLength-i)%3) === 0) {
                sSampleInputString = sSampleInputString + ",";
            }
            sSampleInputString = sSampleInputString + "0";
        }

        if (iDecIndex >= 0) {
            iPrecision = String(fStepWidth).length - iDecIndex - 1;
            if (iPrecision > 0) {
                sSampleInputString = sSampleInputString + ".";
            }
        }

        for (i = 0; i < iPrecision; i++) {
            sSampleInputString = sSampleInputString + "0";
        }

        return sSampleInputString;
    };

    BaseSlider.prototype._determineWidthOfSlider = function() {
        var iWidth = 0;
        if (this.$()) {
            iWidth = this.$().outerWidth();
        }
        return iWidth;
    };

    BaseSlider.prototype._determineHeightOfSlider = function() {
        var iHeight = 0;
        if (this.$()) {
            iHeight = this.$().outerHeight();
        }
        return iHeight;
    };

    BaseSlider.prototype._determineWidthOfSliderLabel = function() {
        var iWidth = 0,
            oSliderLabel = sap.ui.getCore().byId(this.getSliderLabelObject());
        if (oSliderLabel && oSliderLabel.$()) {
            iWidth = oSliderLabel.$().outerWidth();
        }
        return iWidth;
    };

    BaseSlider.prototype._determineHeightOfSliderLabel = function() {
        var iHeight = 0,
            oSliderLabel = sap.ui.getCore().byId(this.getSliderLabelObject());
        if (oSliderLabel && oSliderLabel.$()) {
            iHeight = oSliderLabel.$().outerHeight();
        }
        return iHeight;
    };

    BaseSlider.prototype._calculateStepWidths = function() {
        var sFormat = this.getFormat(),
            bIsDateTime = this.getInputType() !== "Input",
            oDateTimeStepWidths = {},
            fSmallStepWidth = this.getSmallStepWidth(),
            fMediumStepWidth = this.getMediumStepWidth(),
            fLargeStepWidth = this.getLargeStepWidth(),
            fMax,
            fMin,
            fRange;

        // for date/time sliders, default step widths are predifined based on format
        if (bIsDateTime) {
            oDateTimeStepWidths = calculationUtil.calculateDateTimeStepWidths(sFormat);
            if (oDateTimeStepWidths) {
                if (fSmallStepWidth === 1 && oDateTimeStepWidths.small > 0) {
                    fSmallStepWidth = oDateTimeStepWidths.small;
                }
                if (fMediumStepWidth === 0 && oDateTimeStepWidths.medium > 0) {
                    fMediumStepWidth = oDateTimeStepWidths.medium;
                }
                if (fLargeStepWidth === 0 && oDateTimeStepWidths.large > 0) {
                    fLargeStepWidth = oDateTimeStepWidths.large;
                }
            }
        } else {
        // for numeric sliders, default mediumStepWidth is 2% and default largeStepWidth is 10% of the range
            fMax = this.getMax();
            fMin = this.getMin();
            fRange = fMax - fMin;
            if (!fMediumStepWidth) {
                fMediumStepWidth = Math.max(fSmallStepWidth, calculationUtil.getSmallStepWidthValue(fRange/50, fSmallStepWidth));
            }
            if (!fLargeStepWidth) {
                fLargeStepWidth = Math.max(fSmallStepWidth, calculationUtil.getSmallStepWidthValue(fRange/10, fSmallStepWidth));
            }
        }
        //internal smallStepWidth is only used for date/time/duration
        this.setInternalSmallStepWidth(fSmallStepWidth);
        this.setInternalMediumStepWidth(fMediumStepWidth);
        this.setInternalLargeStepWidth(fLargeStepWidth);
    };


    // =============================================================================
    // INPUT ADJUSTMENTS
    // =============================================================================

    /**
     * Set inputType based on the format.
     *
     * For example, "DDMMYY" requires a DatePicker.
     * @private
     */
    BaseSlider.prototype._adjustInputTypeForFormat = function() {
        var sFormat = this.getFormat();

        //if no format was set then do not change the input type
        if (StringUtil.isEmpty(sFormat)) {
            return;
        }

        jQuery.sap.require("sas.hc.m.slider._formatUtil");
        if (sas.hc.m.slider._formatUtil.determineFormatIsDate(sFormat) === true) {
            this.setInputType("DateInput");
        } else if (sas.hc.m.slider._formatUtil.determineFormatIsTime(sFormat) === true) {
            this.setInputType("TimeInput");
        } else if (sas.hc.m.slider._formatUtil.determineFormatIsDateTime(sFormat) === true) {
            this.setInputType("DateTimeInput");
        } else {
            this.setInputType("Input");
        }
    };

    // =============================================================================
    // OPERATOR ADJUSTMENTS
    // =============================================================================

    /**
     * Set thumbs or rangeThumbs based on the operator.
     *
     * For example, "less than" requires thumbs and
     * "between" requires rangeThumbs.
     * @private
     */
    BaseSlider.prototype._adjustThumbsFromOperator = function() {
        var sOperator = this.getOperator(),
            iThumbs,
            iThumbRanges;

        // if no operator was set then do not change the thumbs
        if (!sOperator) {
            return;
        }
        switch(sOperator) {
            case OperatorEnum.BETWEEN_INCLUSIVE:
            case OperatorEnum.BETWEEN_EXCLUSIVE:
            case OperatorEnum.NOT_BETWEEN_INCLUSIVE:
            case OperatorEnum.NOT_BETWEEN_EXCLUSIVE:
                iThumbs = 0;
                iThumbRanges = Math.max(this.getThumbRanges(),1);
                break;
            default:
                iThumbRanges = 0;
                iThumbs = Math.max(this.getThumbs(),1);
        }

        //remove thumbs if we need ranges and vice versa
        if (iThumbs === 0) {
            this.setThumbs(0);
        } else if (iThumbRanges === 0) {
            this.setThumbRanges(0);
        }

        //if the number of thumbs is different from what we have, set it
        if (iThumbs !== 0 && iThumbs !== this.getThumbs()) {
            this.setThumbs(iThumbs);
        }
        //if the number of thumbRanges is different from what we have, set it
        if (iThumbRanges !== 0 && iThumbRanges !== this.getThumbRanges()) {
            this.setThumbRanges(iThumbRanges);
        }
    };

    /**
     * Adjust the classes for the highlight bar and track
     * based on the operator.  The highlight bar is to the
     * right of the thumb for thumbs.  It is between the
     * thumbs for rangeThumbs.
     *
     * Some operators require that the highlight bar be
     * reversed to that it appears after the thumb or
     * outside the range thumbs. In these cases, the
     * styles are switched on the track and highlight bar
     * to give the appearance of the required placement.
     *
     * For example, "greater than" requires that the
     * highlight bar appear to the left of the thumb and
     * "not between" requires that the highlight bar
     * appears outside the thumbs.
     *
     * @private
     */
    BaseSlider.prototype._adjustHighlightForOperator = function() {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody()),
            oTrack = oSliderBody.getTrack(),
            oThumbGroups = oTrack.getThumbGroups(),
            oHighlightBar;

        switch(this.getOperator()) {
            case OperatorEnum.GREATER_THAN:
            case OperatorEnum.GREATER_THAN_OR_EQUAL:
            case OperatorEnum.NOT_BETWEEN_INCLUSIVE:
            case OperatorEnum.NOT_BETWEEN_EXCLUSIVE:
                // highlight the track instead of the highlight bars
                oTrack.removeStyleClass("sasTrackColor");
                oTrack.addStyleClass("sasHighlightColor");
                oThumbGroups.forEach(function(oThumbGroup) {
                    oHighlightBar = oThumbGroup.getHighlightBar();
                    oHighlightBar.removeStyleClass("sasHighlightColor");
                    oHighlightBar.addStyleClass("sasTrackColor");
                });
                break;
            case OperatorEnum.EQUAL_TO:
            case OperatorEnum.NOT_EQUAL_TO:
                //do not highlight the track or the highlight bars
                oTrack.removeStyleClass("sasHighlightColor");
                oTrack.addStyleClass("sasTrackColor");
                oThumbGroups.forEach(function(oThumbGroup) {
                    oHighlightBar = oThumbGroup.getHighlightBar();
                    oHighlightBar.removeStyleClass("sasHighlightColor");
                    oHighlightBar.addStyleClass("sasTrackColor");
                });
                break;
            default:
                //highlight each highlight bar
                oThumbGroups.forEach(function(oThumbGroup) {
                    oHighlightBar = oThumbGroup.getHighlightBar();
                    oHighlightBar.removeStyleClass("sasTrackColor");
                    oHighlightBar.addStyleClass("sasHighlightColor");
                });
                oTrack.removeStyleClass("sasHighlightColor");
                oTrack.addStyleClass("sasTrackColor");
        }
    };

    /**
     * Adjust the classes for the thumbs based on the
     * operator.  Operators can be inclusive or exclusive.
     * Inclusive operators require the thumb to visually
     * indicate that the value for that thumb is included
     * Exclusive operators require the thumb to visually
     * indicate that the value for the thumb is excluded.
     *
     * For example, "greater than" is an exclusive operator
     * because it does not include the value.
     * "Greater than or equal to" is inclusive because it
     * does include the value.
     *
     * @private
     */
    BaseSlider.prototype._adjustThumbColorsForOperator = function() {
        var oSliderBody = sap.ui.getCore().byId(this.getSliderBody()),
            oTrack = oSliderBody.getTrack(),
            oThumbGroups = oTrack.getThumbGroups(),
            oThumbs;

        switch(this.getOperator()) {
            case OperatorEnum.EQUAL_TO:
            case OperatorEnum.LESS_THAN_OR_EQUAL:
            case OperatorEnum.GREATER_THAN_OR_EQUAL:
            case OperatorEnum.BETWEEN_INCLUSIVE:
            case OperatorEnum.NOT_BETWEEN_INCLUSIVE:
                //set the style of the thumbs to be inclusive
                oThumbGroups.forEach(function(oThumbGroup) {
                    oThumbs = oThumbGroup.getThumbs();
                    oThumbs.forEach(function(oThumb) {
                        oThumb.addStyleClass("sasThumbInclusiveColor");
                    });
                });
                break;
            default:
                oThumbGroups.forEach(function(oThumbGroup) {
                    oThumbs = oThumbGroup.getThumbs();
                    oThumbs.forEach(function(oThumb) {
                        oThumb.removeStyleClass("sasThumbInclusiveColor");
                    });
                });
        }
    };

    /**
     * Generate the tooltip text based on the operator.
     * The tooltip should include the operator and the
     * current values.
     *
     * For example: Is between 25 and 45 (inclusive)
     *
     * @private
     */
    BaseSlider.prototype._generateTooltipForOperator = function() {
        var sTooltipText="",
            sResource,
            rb = sap.ui.getCore().getLibraryResourceBundle("sas.hc.m"),
            aValues = this.getValues(),
            aFormattedValues = [],
            i=0,
            fnTooltipCallback = this.getTooltipCallback(),
            self = this;

        if (this.getShowOperators() !== true) {
            return;
        }

        if (fnTooltipCallback !== null && fnTooltipCallback !== undefined) {
            //if the consumer set a tooltip callback function, use it
            sTooltipText = fnTooltipCallback(this.getOperator(), aValues);
        } else if (aValues === null || aValues === undefined || aValues.length === 0) {
            //if there are no values, then we won't display a tooltip
            return;
        } else {
            aValues.forEach(function(value) {
                aFormattedValues[i] = self.retrieveFormattedValue(value);
                i++;
            });
            //get the tooltip from the resource file based on the operator and the values
            sResource = "Slider.operatorTooltip." + this.getOperator() + ".fmt";
            sTooltipText = rb.getText(sResource, aFormattedValues);
        }

        this.setProperty("tooltip", sTooltipText, true);
        if (document.getElementById(this.getId()) !== null && document.getElementById(this.getId()) !== undefined) {
            document.getElementById(this.getId()).title = sTooltipText;
        }
    };

    return BaseSlider;
}, /* bExport= */ true);
