// (c) 2015, SAS Institute Inc.
sap.ui.define([
    "jquery.sap.global",
    "sas/hc/m/Button",
    "./_ObjectInspectorNamedButton",
    "./_FavoritesButton",
    "sas/hc/m/Label",
    "sas/hc/m/ResponsivePopover",
    "./_IconTabBar",
    "sas/hc/m/IconTabFilter",
    "./_ObjectInspectorPopoverContainer",
    "sas/hc/m/SimplePairTable",
    "sas/hc/m/ResponsiveToolbar",
    "sas/hc/ui/core/Icon",
    "sas/hc/m/OverflowToolbarLayoutData",
    "sas/hc/m/ToolbarSpacer",
    "sas/hc/m/MenuButton",
    "sas/hc/m/Heading",
    "sap/m/Popover",
    "sap/m/library",
    "sap/ui/core/Core",
    "sas/hc/ui/core/Core",
    "sas/hc/ui/core/settings/GlobalSettingsHelper"
], function(jQuery, Button, ObjectInspectorNamedButton, FavoritesButton, Label, ResponsivePopover, IconTabBar, IconTabFilter, ObjectInspectorPopoverContainer, SimplePairTable, ResponsiveToolbar, Icon, OverflowToolbarLayoutData, ToolbarSpacer, MenuButton, Heading, sapMPopover, sapMLibrary, sapCore, Core, GlobalSettingsHelper) {
    "use strict";

    var MenuButtonMode = sapMLibrary.MenuButtonMode,
        OverflowToolbarPriority = sapMLibrary.OverflowToolbarPriority,
        PlacementType = sapMLibrary.PlacementType;

    // keep track of ids for sub-controls
    var HEADER_TEXT_LABEL_ID = "HeaderTextLabel",
        SECONDARY_TEXT_LABEL_ID = "SecondaryTextLabel",
        DESCRIPTION_TEXT_LABEL_ID = "DescriptionLabel",
        HEADER_ICON_ID = "HeaderIconIcon",
        OPEN_BUTTON_ID = "NameButton_open",
        EDIT_BUTTON_ID = "NameButton_edit",
        FAVORITES_BUTTON_ID = "NameButton_favorites",
        ICON_TAB_BAR_ID = "DetailsIconTabBar";

    /**
     * Constructor for a new ObjectInspectorBase.
     *
     * Base-class for ObjectInspector
     *
     * The aggregations defined in this control are rendered by the
     * ObjectInspectorPopoverContainer and ResponsiveToolbar controls,
     * thus they are re-parented prior to rendering.
     *
     * For example, the toolbarContent aggregation will be empty because the controls
     * it is given are made aggregates of the ResponsiveToolbar control
     *
     * It is convenient for the consumer to have these aggregation defined at the
     * ObjectInspectorBase level for their binding and xml views, so there needs
     * needs to be a pass-through-like API defined at this level for the other
     * ManagedObject-level functions to work (ex: removeAllAggregations)
     *
     * @class
     * @name sas.hc.m.objectInspector.ObjectInspectorBase
     * @extends sas.hc.m.ResponsivePopover
     * @version 904001.11.16.20251118090100_f0htmcm94p
     * @author Jonathan Brink, Ralph Marshall
     * @constructor
     * @public
     */
    var ObjectInspectorBase = ResponsivePopover.extend("sas.hc.m.objectInspector.ObjectInspectorBase", /** @lends sas.hc.m.objectInspector.ObjectInspectorBase.prototype */ {
        metadata: {
            library: "sas.hc.m",
            publicMethods: [
                "getOpenButton",
                "getEditButton",
                "getFavoritesButton"
            ],
            properties: {
                headerText: {type: "string", defaultValue: "", group: "Data"},
                secondaryText: {type: "string", defaultValue: "", group: "Data"},
                description: {type: "string", defaultValue: "", group: "Data"},
                iconKey: {type: "string", defaultValue: "", group: "Data"},
                contentWidth: {type: "sap.ui.core.CSSSize", group: "Dimension"},
                contentHeight: {type: "sap.ui.core.CSSSize", group: "Dimension"},

                /**
                 * Controls the visibility of the "open" button, which resides in the toolbar
                 */
                showOpenButton: {type: "boolean", defaultValue: true, group: "Behavior"},

                /**
                 * Controls the visibility of the "edit" button, which resides in the toolbar
                 */
                showEditButton: {type: "boolean", defaultValue: true, group: "Behavior"},

                /**
                 * Controls the visibility of the "favorites" button, which resides in the toolbar
                 */
                showFavoritesButton: {type: "boolean", defaultValue: true, group: "Behavior"},
                objectData: {type: "object", group: "Data"},
                favoriteButtonSelectedDefault: {type: "boolean", defaultValue: false, group: "Behavior"},
                autoHeight: {type: "boolean", defaultValue: true, group: "Behavior"},
                objectDataLineHeight: {type: "integer", defaultValue: 5, group: "Appearance"},

                /*
                 * The sap.m.ResponsivePopover default is "Right"
                 * @override
                 */
                placement: {type: "sap.m.PlacementType", group: "Misc", defaultValue: PlacementType.Vertical},

                /* @deprecated Since version 3.0 */
                maxAutoHeightObjectDataLines: {type: "integer", defaultValue: 10, group: "Appearance"},

                /**
                 * Width of the first column of the object metadata SimplePairTable (`firstColWidth`)
                 */
                objectMetadataTableFirstColWidth: {type: "sap.ui.core.CSSSize", defaultValue: undefined, group: "Dimension"}
            },
            aggregations: {
                details: {type: "sas.hc.m.IconTabFilter", multiple: true, singularName: "detail"},

                /**
                 * Passed to the internal Toolbar (sas.hc.m.ResponsiveToolbar)
                 * Supported types:
                 * * sas.hc.m.Button
                 * * sas.hc.m.MenuButton
                 * @public
                 */
                toolbarContent: {type: "sap.ui.core.Control", multiple: true, singularName: "toolbarContent"}
            },
            events: {
                openButtonPress: {},
                editButtonPress: {},
                favoritesButtonPress: {},

                /**
                 * Fired when after the ObjectInspector has launched and has become visible.
                 * This will be fired slightly after the "afterOpen" event as OI hides it's rendering briefly while auto-sizing occurs
                 */
                afterReveal: {}
            }
        },
        renderer: "sas.hc.m.PopoverRenderer",

        // resource bundle for ObjectInspector
        rb: sap.ui.getCore().getLibraryResourceBundle("sas.hc.m")
    });


    // =============================================================================
    // LIFECYCLE FUNCTIONS
    // =============================================================================

    /**
     * init lifecycle callback.
     * Creates high-level controls that make up ObjectInspectorBase
     * @private
     */
    ObjectInspectorBase.prototype.init = function() {
        // invoke super
        if (ResponsivePopover.prototype.init) {
            ResponsivePopover.prototype.init.apply(this, arguments);
        }

        // force placement property to be respected
        this.setPlacement(this.getPlacement());

        // We have to add all of our locally implemented properties to this array so
        // that the code in the SAP version of ResponsivePopover does not attempt to
        // pass the function calls on to the underlying Popover/Dialog
        Array.prototype.push.apply(this._aNotSupportedProperties, [
            "headerText", "secondaryText", "description", "iconKey",
            "showOpenButton", "showEditButton",
            "showFavoritesButton", "objectData", "favoriteButtonSelectedDefault",
            "autoHeight", "objectDataLineHeight", "maxAutoHeightObjectDataLines"
        ]);

        // set defaults
        this.setProperty("showHeader", false, true);
        this.setModal(false);
        this.setAutoClose(true);
        this.setProperty("objectData", [], true);

        // add style-classes
        this.addStyleClass("sasObjectInspectorPopover");
        this.addStyleClass("oi_invisible"); // hide while sizing calculations/adjustments occur

        // fake the control lifecycle b/c of ResponsivePopover extension
        this.attachBeforeOpen(this.preRenderingSteps);
        this.attachAfterOpen(this.postRenderingSteps);
    };

    /**
     * This would be in the onBeforeRendering method, but there isn't really such a method
     * with responsive popovers.
     * Popup sizing, and simple table creation/updating
     * @private
     */
    ObjectInspectorBase.prototype.preRenderingSteps = function() {
        var oPopoverContainer, oSimplePairTable, oIconTabBar, oHeaderIcon;

        // create (if not already created) container controls
        this._initPopoverContainer();
        this._initToolbar();

        // populate (if not already populated) the popover container
        oPopoverContainer = this._lookupPopoverContainer();
        if (oPopoverContainer.getContent().length === 0) {
            oPopoverContainer._populate();
            oPopoverContainer._lookupHeaderTextContainer().setMainContent(this._createHeaderTextLabel());
            oPopoverContainer._lookupSecondaryTextContainer().setMainContent(this._createSecondaryTextLabel());
            oPopoverContainer._lookupDescriptionContainer().setMainContent(this._createDescriptionLabel());
            oPopoverContainer._lookupHeaderIconContainer().setMainContent(this._createHeaderIcon());
            oPopoverContainer._lookupDetailsContainer().setMainContent(this._createIconTabBar());
        }

        // create (if not already created) the simple pair table and it's parent IconTabFilter
        oSimplePairTable = this._lookupSimplePairTable();
        if (!oSimplePairTable) {
            oSimplePairTable = this._createSimplePairTable();
            this.addDetail(new IconTabFilter({
                id: this.getId() + "-IconTabFilter_simplePairTable",
                key: "defaultDataTable",
                text: this.rb.getText("objectInspector.details.txt"),
                content: [
                    oSimplePairTable
                ]
            }));
        }

        // populate icon tab bar with "details" aggregation
        // this will cause re-parenting
        oIconTabBar = this._lookupIconTabBar();
        (this.getAggregation("details") || []).forEach(function(oDetail) {
            if (oDetail.getKey() === "defaultDataTable") {
                oIconTabBar.insertItem(oDetail, 0);
            } else {
                oIconTabBar.addItem(oDetail);
            }
        });

        // pass property values into their corresponding controls
        this._lookupOpenButton().setVisible(this.getShowOpenButton());
        this._lookupEditButton().setVisible(this.getShowEditButton());
        this._lookupFavoritesButton().setVisible(this.getShowFavoritesButton());
        this._lookupHeaderTextLabel().setText(this.getHeaderText());
        this._lookupSecondaryTextLabel().setText(this.getSecondaryText());
        this._lookupDescriptionLabel().setText(this.getDescription());
        oHeaderIcon = this._lookupHeaderIcon();
        oHeaderIcon.setSrc(this.getIconKey());
        oHeaderIcon.setVisible(!!this.getIconKey());

        // objectData can be added at any order by the consumer, so need to sort
        this._sortObjectData();
        this._sortToolbarContent();
        oSimplePairTable.setData(this.getObjectData());

        // update the width of the first simple pair table column
        if (this.getObjectMetadataTableFirstColWidth()) {
            oSimplePairTable.setFirstColWidth(this.getObjectMetadataTableFirstColWidth());
        }
    };

    /**
     * This would be in the onAfterRendering method, but there isn't really such a method
     * with responsive popovers.
     * @private
     */
    ObjectInspectorBase.prototype.postRenderingSteps = function() {
        this.setInitialFocus(this._determineInitialFocus());

        // this delay has been hard-coded at 100 for some time to
        // hide the ObjectInspector wiggling around as it settles
        setTimeout(this._reveal.bind(this), 100);
    };

    ObjectInspectorBase.prototype.destroy = function() {
        // functions stored so they can be properly removed during cloning
        this._openOnPress = undefined;
        this._editOnPress = undefined;
        this._favoritesOnPress = undefined;

        // If we do not pass in the suppress invalidate flag we get internal
        // errors later on when we try to actually call invalidate. Unclear why
        // this happens at all.
        ResponsivePopover.prototype.destroy.call(this, true);
    };

    ObjectInspectorBase.prototype.clone = function() {
        var oClone = ResponsivePopover.prototype.clone.apply(this, arguments),
            oCloneOpenButton, oCloneEditButton, oCloneFavoritesButton;

        // need to manually clone the inner control's footer (toolbar)
        if (this._oControl.getFooter()) {
            oClone._oControl.setFooter(this._oControl.getFooter().clone());
        }

        // fix "openButtonPress" event
        oCloneOpenButton = oClone._lookupOpenButton();
        if (oCloneOpenButton) {
            oCloneOpenButton.detachPress(this._openOnPress);
            oCloneOpenButton.attachPress(oClone.fireOpenButtonPress.bind(oClone));
        }

        // fix "editButtonPress" event
        oCloneEditButton = oClone._lookupEditButton();
        if (oCloneEditButton) {
            oCloneEditButton.detachPress(this._editOnPress);
            oCloneEditButton.attachPress(oClone.fireEditButtonPress.bind(oClone));
        }

        // fix "favoritesButtonPress" event
        oCloneFavoritesButton = oClone._lookupFavoritesButton();
        if (oCloneFavoritesButton) {
            oCloneFavoritesButton.detachPress(this._favoritesOnPress);
            oCloneFavoritesButton.attachPress(oClone.fireFavoritesButtonPress.bind(oClone));
        }

        return oClone;
    };


    // =============================================================================
    // METADATA API
    // =============================================================================

    // Setter for the `showOpenButton` property (JSDoc: see metadata block)
    ObjectInspectorBase.prototype.setShowOpenButton = function(bShowOpenButton) {
        var oOpenButton = this._lookupOpenButton();
        this.setProperty("showOpenButton", bShowOpenButton, true);
        if (oOpenButton) {
            oOpenButton.setVisible(bShowOpenButton);
        }
    };
    // Setter for the `showEditButton` property (JSDoc: see metadata block)
    ObjectInspectorBase.prototype.setShowEditButton = function(bShowEditButton) {
        var oEditButton = this._lookupEditButton();
        this.setProperty("showEditButton", bShowEditButton, true);
        if (oEditButton) {
            oEditButton.setVisible(bShowEditButton);
        }
    };
    // Setter for the `showFavoritesButton` property (JSDoc: see metadata block)
    ObjectInspectorBase.prototype.setShowFavoritesButton = function(bShowFavoritesButton) {
        var oFavoritesButton = this._lookupFavoritesButton();
        this.setProperty("showFavoritesButton", bShowFavoritesButton, true);
        if (oFavoritesButton) {
            oFavoritesButton.setVisible(bShowFavoritesButton);
        }
    };

    /**
     * Add content to the internal ResponsiveToolbar
     * * Enforce that oContent must either be of type sas.hc.m.Button or sas.hc.m.MenuButton
     * * Enforce that only one MenuButton is allowed
     * @param {sap.ui.core.Control} oContent The content to be passed to the toolbar
     * @returns {object} The ObjectInspector instance
     * @public
     */
    ObjectInspectorBase.prototype.addToolbarContent = function(oContent) {
        var oToolbar;

        // create toolbar if not already created
        this._initToolbar();
        oToolbar = this._lookupToolbar();

        var aMenuButtons = oToolbar.getContent()
            .filter(function(o) {
                return o instanceof MenuButton;
            });

        // verify that oContent is of the correct type
        if ((oContent instanceof Button === false) && (oContent instanceof MenuButton === false)) {
            sas.log.warning("toolbarContent must be of type sas.hc.m.Button or sas.hc.m.MenuButton");
            return;
        }
        if ((oContent instanceof MenuButton) && (aMenuButtons.length > 0)) {
            sas.log.warning("Only one MenuButton is allowed in toolbarContent");
            return;
        }

        // Split button mode must be used for the MenuButton
        if (oContent instanceof MenuButton) {
            oContent.setButtonMode(MenuButtonMode.Split);
            oContent.setUseDefaultActionOnly(true);
        }

        // for MenuButtons, if no overflow property is present, set to NeverOverflow
        if (oContent instanceof MenuButton && oContent.getLayoutData() === null) {
            oContent.setLayoutData(new OverflowToolbarLayoutData({
                priority: OverflowToolbarPriority.NeverOverflow
            }));
        }
        // for Buttons, if no overflow property is present, set to AlwaysOverflow
        if (oContent instanceof Button && oContent.getLayoutData() === null) {
            oContent.setLayoutData(new OverflowToolbarLayoutData({
                priority: OverflowToolbarPriority.AlwaysOverflow
            }));
        }
        oToolbar.addContent(oContent);
        return this;
    };


    // =============================================================================
    // PUBLIC API
    // These functions mimic aggregation getters for toolbar buttons that previously were aggregations
    // =============================================================================

    /**
     * Returns and creates (if not already created) the "open" button located in the toolbar.
     * Will also create the toolbar is not already created.
     * @returns {object} The Toolbar's "open" button
     * @public
     */
    ObjectInspectorBase.prototype.getOpenButton = function() {
        this._initToolbar();
        return this._lookupOpenButton();
    };

    /**
     * Returns and creates (if not already created) the "edit" button located in the toolbar.
     * Will also create the toolbar is not already created.
     * @returns {object} The Toolbar's "edit" button
     * @public
     */
    ObjectInspectorBase.prototype.getEditButton = function() {
        this._initToolbar();
        return this._lookupEditButton();
    };

    /**
     * Returns and creates (if not already created) the "favorites" button located in the toolbar.
     * Will also create the toolbar is not already created.
     * @returns {object} The Toolbar's "favorites" button
     * @public
     */
    ObjectInspectorBase.prototype.getFavoritesButton = function() {
        this._initToolbar();
        return this._lookupFavoritesButton();
    };


    // =============================================================================
    // INIT SUB-CONTROLS FUNCTIONS
    // =============================================================================

    /**
     * Checks for and creates (if not already created) the "toolbar" that is placed in the footer
     * @private
     */
    ObjectInspectorBase.prototype._initToolbar = function() {
        if (this._lookupToolbar()) {
            return;
        }
        this._oControl.setFooter(this._createToolbar());
    };

    /**
     * Checks for and creates (if not already created) the "popover container" that contains the middle
     * content of the ObjectInspector, such as the header text and simple pair table
     * @private
     */
    ObjectInspectorBase.prototype._initPopoverContainer = function() {
        if (this._lookupPopoverContainer()) {
            return;
        }
        this.addContent(this._createPopoverContainer());
    };


    // =============================================================================
    // CREATE SUB-CONTROLS FUNCTIONS
    // =============================================================================

    /**
     * Creates and populates a new toolbar instance.
     * The toolbar buttons (open/edit/favorites) will be created (if not already created) and added as content
     * @returns {object} The toolbar instance
     * @private
     */
    ObjectInspectorBase.prototype._createToolbar = function() {
        return new ResponsiveToolbar({
            id: this.getId() + '-responsiveToolbar',
            content: [
                new ToolbarSpacer(this.getId() + '-toolbarSpacer'),
                this._lookupOpenButton() || this._createOpenButton(),
                this._lookupEditButton() || this._createEditButton(),
                this._lookupFavoritesButton() || this._createFavoritesButton()
            ]
        });
    };

    /**
     * Create and return the "open" button.
     * The press event (which feeds the `openButtonPress` event) is saved as an instance variable
     * so it can be properly transferred during cloning.
     * @returns {object} The "open" button
     * @private
     */
    ObjectInspectorBase.prototype._createOpenButton = function() {
        this._openOnPress = this.fireOpenButtonPress.bind(this);
        return new ObjectInspectorNamedButton({
            id: this.getId() + '-' + OPEN_BUTTON_ID,
            type: ObjectInspectorNamedButton.OPEN,
            text: this.rb.getText("objectInspector.open.txt"),
            press: this._openOnPress,
        }).addStyleClass('ButtonOverflowLayout_open');
    };

    /**
     * Create and return the "edit" button.
     * The press event (which feeds the `editButtonPress` event) is saved as an instance variable
     * so it can be properly transferred during cloning.
     * @returns {object} The "edit" button
     * @private
     */
    ObjectInspectorBase.prototype._createEditButton = function() {
        this._editOnPress = this.fireEditButtonPress.bind(this);
        return new ObjectInspectorNamedButton({
            id: this.getId() + '-' + EDIT_BUTTON_ID,
            type: ObjectInspectorNamedButton.EDIT,
            text: this.rb.getText("objectInspector.edit.txt"),
            press: this._editOnPress
        }).addStyleClass('ButtonOverflowLayout_edit');
    };

    /**
     * Create and return the "favorites" button.
     * The press event (which feeds the `favoritesButtonPress` event) is saved as an instance variable
     * so it can be properly transferred during cloning.
     * @returns {object} The "favorites" button
     * @private
     */
    ObjectInspectorBase.prototype._createFavoritesButton = function() {
        this._favoritesOnPress = this.fireFavoritesButtonPress.bind(this);
        return new FavoritesButton({
            id: this.getId() + '-' + FAVORITES_BUTTON_ID,
            press: this._favoritesOnPress,
            selected: !!this.getFavoriteButtonSelectedDefault()
        });
    };

    /**
     * Create the popover container which contains the middle content of the ObjectInspector,
     * such as the header text and simple pair table
     * @returns {object} The popover container instance
     * @private
     */
    ObjectInspectorBase.prototype._createPopoverContainer = function() {
        return new ObjectInspectorPopoverContainer(this.getId() + "-ObjectInspectorPopoverContainer");
    };

    /**
     * Create the icon tab bar instance
     * @returns {object} The icon tab bar instance
     * @private
     */
    ObjectInspectorBase.prototype._createIconTabBar = function() {
        var self = this;
        return new IconTabBar({
            id: this.getId() + "-" + ICON_TAB_BAR_ID,
            items: [],
            select: function(e) {
                setTimeout(self._manageAutoSize.bind(self), 100);
            }
        });
    };

    /**
     * Create the simple pair table instance which holds object data (the `objectData` property)
     * @returns {object} The simple pair table instance
     * @private
     */
    ObjectInspectorBase.prototype._createSimplePairTable = function() {
        return new SimplePairTable(this.getId() + "-SimplePairTable");
    };

    /**
     * Create the header text label instance which holds the `headerText` property value
     * @returns {object} The header text label instance
     * @private
     */
    ObjectInspectorBase.prototype._createHeaderTextLabel = function() {
        return new Heading({
            id: this.getId() + "-" + HEADER_TEXT_LABEL_ID,
            level: 3
        });
    };

    /**
     * Create the secondary text label instance which holds the `secondaryText` property value
     * @returns {object} The secondary text label instance
     * @private
     */
    ObjectInspectorBase.prototype._createSecondaryTextLabel = function() {
        return new Label(this.getId() + "-" + SECONDARY_TEXT_LABEL_ID);
    };

    /**
     * Create the description label instance which holds the `description` property value
     * @returns {object} The description text label instance
     * @private
     */
    ObjectInspectorBase.prototype._createDescriptionLabel = function() {
        return new Label(this.getId() + "-" + DESCRIPTION_TEXT_LABEL_ID);
    };

    /**
     * Create the header icon instance which holds the `iconKey` property value
     * @returns {object} The header icon instance
     * @private
     */
    ObjectInspectorBase.prototype._createHeaderIcon = function() {
        return new Icon(this.getId() + "-" + HEADER_ICON_ID);
    };


    // =============================================================================
    // LOOKUP SUB-CONTROLS FUNCTIONS
    // =============================================================================

    /**
     * Return the toolbar instance, which is stored in the footer.
     * If not already created will return undefined
     * @returns {object} The toolbar instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupToolbar = function() {
        return this._oControl.getFooter();
    };

    /**
     * Return the open button instance, which is stored in the toolbar.
     * If not already created will return undefined
     * @returns {object} The open button instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupOpenButton = function() {
        return this._lookupToolbarContent(OPEN_BUTTON_ID);
    };

    /**
     * Return the edit button instance, which is stored in the toolbar.
     * If not already created will return undefined
     * @returns {object} The edit button instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupEditButton = function() {
        return this._lookupToolbarContent(EDIT_BUTTON_ID);
    };

    /**
     * Return the favorites button instance, which is stored in the toolbar.
     * If not already created will return undefined
     * @returns {object} The favorites button instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupFavoritesButton = function() {
        return this._lookupToolbarContent(FAVORITES_BUTTON_ID);
    };

    /**
     * Return the popover container instance, which is stored in the content aggregation.
     * If not already created will return undefined
     * @returns {object} The popover container instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupPopoverContainer = function() {
        return this.getContent()
            .filter(function(oContent) {
                return oContent instanceof ObjectInspectorPopoverContainer;
            })[0];
    };

    /**
     * Return the icon tab bar instance, which is stored in the popover container.
     * If not already created will return undefined
     * @returns {object} The icon tab bar instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupIconTabBar = function() {
        return this._lookupPopoverContainerContent(ICON_TAB_BAR_ID);
    };

    /**
     * Return the simple pair table instance, which is stored in the icon tab bar.
     * If not already created will return undefined
     * @returns {object} The simple pair table instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupSimplePairTable = function() {
        var oIconTabBar = this._lookupIconTabBar();
        if (!oIconTabBar) {
            return undefined;
        }
        return oIconTabBar.getItems()
            .filter(function(oItem) {
                var oContent = oItem.getContent()[0];
                return oContent && /-SimplePairTable/.test(oContent.getId());
            })
            .map(function(oItem) {
                return oItem.getContent()[0];
            })[0];
    };

    /**
     * Return the header text label instance, which is stored in the popover container.
     * If not already created will return undefined
     * @returns {object} The header text label instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupHeaderTextLabel = function() {
        return this._lookupPopoverContainerContent(HEADER_TEXT_LABEL_ID);
    };

    /**
     * Return the secondary text label instance, which is stored in the popover container.
     * If not already created will return undefined
     * @returns {object} The secondary text label instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupSecondaryTextLabel = function() {
        return this._lookupPopoverContainerContent(SECONDARY_TEXT_LABEL_ID);
    };

    /**
     * Return the description label instance, which is stored in the popover container.
     * If not already created will return undefined
     * @returns {object} The description label instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupDescriptionLabel = function() {
        return this._lookupPopoverContainerContent(DESCRIPTION_TEXT_LABEL_ID);
    };

    /**
     * Return the header icon instance, which is stored in the popover container.
     * If not already created will return undefined
     * @returns {object} The header icon instance
     * @private
     */
    ObjectInspectorBase.prototype._lookupHeaderIcon = function() {
        return this._lookupPopoverContainerContent(HEADER_ICON_ID);
    };

    /**
     * Return the instance that matches an id from the toolbar
     * If not found, will return undefined
     * @param {string} sIdFragment An identifying part of the desired content's id
     * @returns {object} The matching content
     * @private
     */
    ObjectInspectorBase.prototype._lookupToolbarContent = function(sIdFragment) {
        if (!this._oControl || !this._oControl.getFooter()) {
            return undefined;
        }
        return this._oControl.getFooter().getContent()
            .filter(function(oContent) {
                return (new RegExp("-" + sIdFragment)).test(oContent.getId());
            })[0];
    };

    /**
     * Return the instance that matches an id from the popover container.
     * If not found, will return undefined
     * @param {string} sIdFragment An identifying part of the desired content's id
     * @returns {object} The matching content
     * @private
     */
    ObjectInspectorBase.prototype._lookupPopoverContainerContent = function(sIdFragment) {
        var oPopoverContainer = this._lookupPopoverContainer();
        if (!oPopoverContainer) {
            return undefined;
        }
        return oPopoverContainer._lookup(sIdFragment);
    };


    // =============================================================================
    // PRIVATE FUNCTIONS
    // =============================================================================

    /**
     * Add oData to objectData.
     * Depending on if oData's "name" property has already been added to objectData,
     * oData will be either appended, or will overwrite an existing same-name member.
     *
     * Since oData is a primitive array, it's listed as a property, not aggregation,
     * which means I need to roll-my-own "add" function.
     *
     * @param {object} oData  The data object to add
     * @private
     */
    ObjectInspectorBase.prototype._addObjectData = function(oData) {
        var aObjectData = this.getObjectData();

        if (!aObjectData) {
            aObjectData = [];
        }

        // might need to overwrite
        var bFound = false;
        for (var i = 0; i < aObjectData.length; i++) {
            var oObjectData = aObjectData[i];
            if (oObjectData.name === oData.name) {
                aObjectData[i].value = oData.value;
                bFound = true;
                break;
            }
        }

        if (bFound === false) {
            aObjectData.push(oData);
            this.setObjectData(aObjectData);
        }

        if (this.isOpen()) {
            this._updateSimplePairTable();
            this._refreshContent();
        }
    };

    /**
     * Remove opacity from ObjectInspector.
     * This adds a css class which uses animations to remove the opacity
     * This should occur after sizing calculations have taken place.
     * If animations have been turned off (for a11y) then simply remove
     * the css class that is applying the opacity
     * @private
     */
    ObjectInspectorBase.prototype._reveal = function() {
        var $this = this.$(),
            bUseAnimation = sas.hc.ui.core.settings !== undefined &&  // eslint-disable-line sashtmlcommons/missing-require
                GlobalSettingsHelper.getPlayVisualEffects() === true;
        if (bUseAnimation === true) {
            $this.addClass("oi_visible");
        } else {
            $this.removeClass("oi_invisible");
        }
        this.fireAfterReveal();
    };

    /**
     * Determine the value that should be passed to the ResponsivePopover's "initialFocus" association.
     * This function is meant to be invoked after rendering
     * @returns {object} Sub-control to be used for initial focus
     * @private
     */
    ObjectInspectorBase.prototype._determineInitialFocus = function() {
        var aTabbableToolbarContent = this._lookupToolbar().getContent()
                // only return visible MenuButtons or Buttons
                .filter(function(oContent) {
                    return oContent.getVisible() === true &&
                        (oContent instanceof MenuButton || oContent instanceof Button);
                })
                // if MenuButton, use it's internal split button
                .map(function(oContent) {
                    if (oContent instanceof MenuButton) {
                        return oContent._getButtonControl();
                    }
                    return oContent;
                });
        return aTabbableToolbarContent[0] || this._lookupPopoverContainer();
    };

    /**
     * Determine the max-allowed ObjectInspector width
     * @returns {int} Pixel value (without the "px", just the number)
     * @private
     */
    ObjectInspectorBase.prototype._determineMaxWidth = function() {
        return this.$().closest("body").outerWidth() / 2;
    };

    /**
     * Set the contentHeight and contentWidth properties to match the actual content's height and width.
     * Try multiple times until the DOM has settled.
     * @param {int} iCount Number to determine how many times this function has been called (avoid infinite loops)
     * @private
     */
    ObjectInspectorBase.prototype._manageAutoSize = function(iCount) {
        var self = this,
            oObjectInspectorPopoverContainer = this._lookupPopoverContainer(),
            bDidChangeOccur = false,
            iMaxWidth = this._determineMaxWidth(),
            $popoverContainer,
            iDeterminedHeight, iDeterminedWidth;

        if (!oObjectInspectorPopoverContainer) {
            return;
        }

        $popoverContainer = oObjectInspectorPopoverContainer.$();

        // skip auto-sizing
        if (this.getAutoHeight() === false || $popoverContainer.length <= 0) {
            this._reveal();
            return;
        }

        // initialize helper variables
        iCount = iCount || 0;
        this._previousHeight = this._previousHeight || 0;
        this._previousWidth = this._previousWidth || 0;

        iDeterminedHeight = $popoverContainer.outerHeight();
        iDeterminedWidth = Math.min($popoverContainer.outerWidth(), iMaxWidth);

        // if height has changed
        if (iDeterminedHeight > this._previousHeight) {
            this.setContentHeight(iDeterminedHeight + "px");
            this._previousHeight = iDeterminedHeight;
            bDidChangeOccur = true;
        }

        // if width has changed
        if (iDeterminedWidth > this._previousWidth) {
            this.setContentWidth(iDeterminedWidth + "px");
            this._previousWidth = iDeterminedWidth;
            bDidChangeOccur = true;
        }

        // if change occurred, do another run to make sure DOM has settled
        // otherwise, reveal the properly-sized ObjectInspector
        if (bDidChangeOccur === true && iCount < 5) {
            setTimeout(function() {
                self._manageAutoSize(iCount+1);
            }, 100);
        } else {
            this._reveal();
        }
    };

    /**
     * Sort the objectData array so it is rendered in the correct order
     * Correct order (internal key in parens):
     *     - Type (objectType)
     *     - Location (objectLocation)
     *     - Date created (creationDate)
     *     - Date modified (modificationDate)
     *     - Keywords (keywords)
     *
     * @private
     */
    ObjectInspectorBase.prototype._sortObjectData = function() {
        var aSorted = [],
            aObjectData = this.getObjectData().slice(0);

        // find sKey in aObjectData,
        // push it to aSorted,
        // remove the found object from aObjectData
        ["objectType","objectLocation","creationDate","modificationDate","keywords"].forEach(function(sKey) {
            var i = 0, oTargetObject;
            for (; i < aObjectData.length; i++) {
                oTargetObject = aObjectData[i];
                if (oTargetObject.key === sKey) {
                    aSorted.push(oTargetObject);
                    aObjectData.splice(i,1);
                    break;
                }
            }
        });

        // append the rest of the keys
        aSorted = aSorted.concat(aObjectData);

        this.setObjectData(aSorted);
    };

    /**
     * Sort the content in the toolbarContent aggregation
     * Correct order:
     *     - MenuButton
     *     - Text buttons
     *     - Named buttons
     *     - Text/Icon buttons
     * @private
     */
    ObjectInspectorBase.prototype._sortToolbarContent = function() {
        function isMenuButton(o) {
            return o instanceof MenuButton;
        }
        function isNamedButton(o) {
            return o instanceof ObjectInspectorNamedButton;
        }
        function isIconButton(o) {
            return (o instanceof Button) &&
                   (isNamedButton(o) === false) &&
                   (o.getIcon() !== "");
        }
        function isTextButton(o) {
            return (o instanceof Button) &&
                   (isNamedButton(o) === false) &&
                   (o.getIcon() === "");
        }
        var oToolbar = this._lookupToolbar(),
            aNamedButtonOrder = [
                ObjectInspectorNamedButton.OPEN,
                ObjectInspectorNamedButton.EDIT,
                ObjectInspectorNamedButton.FAVORITES
            ],
            aContent = oToolbar.getContent();

        // Remove the content array before sorting it.
        // Otherwise, the toolbar will react to each change and try to relayout the controls
        oToolbar.removeAllContent();
        if (!aContent || aContent.length === 0) {
            return;
        }

        oToolbar.setContent(aContent.sort(function(o0, o1) {
            // MenuButton go before named, text, or icon buttons
            if (isMenuButton(o0) && (isNamedButton(o1) || isTextButton(o1) || isIconButton(o1))) {
                return -1;
            }
            if (isMenuButton(o1) && (isNamedButton(o0) || isTextButton(o0) || isIconButton(o0))) {
                return 1;
            }

            // text buttons go before icon buttons
            if (isTextButton(o0) && isIconButton(o1)) {
                return -1;
            }
            if (isIconButton(o0) && isTextButton(o1)) {
                return 1;
            }

            // named buttons go before text or icon buttons
            if (isNamedButton(o0) && (isTextButton(o1) || isIconButton(o1))) {
                return -1;
            }
            if (isNamedButton(o1) && (isTextButton(o0) || isIconButton(o0))) {
                return 1;
            }

            // comparison between two named buttons
            if (isNamedButton(o0) && isNamedButton(o1)) {
                return aNamedButtonOrder.indexOf(o0.getType()) - aNamedButtonOrder.indexOf(o1.getType());
            }

            return 0;
        }));
    };

    ObjectInspectorBase.prototype._updateSimplePairTable = function() {
        var oSimplePairTable = this._lookupSimplePairTable();
        this._sortObjectData();
        if (oSimplePairTable) {
            oSimplePairTable.setData(this.getObjectData());
        }
    };

    ObjectInspectorBase.prototype._refreshContent = function() {
        var oPopoverContainer = this._lookupPopoverContainer();
        if (oPopoverContainer) {
            oPopoverContainer.invalidate();
        }
    };

    return ObjectInspectorBase;
}, /* bExport= */ true);
