// (c) 2015, SAS Institute Inc.
sap.ui.define([
    "jquery.sap.global",
    "./ObjectInspectorBase",
    "./FormatUtil",
    "sap/ui/core/library", // DataTypes:[TextDirection]
    "sap/ui/core/Core",
    "sas/hc/ui/core/Core"
], function(jQuery, ObjectInspectorBase, FormatUtil, coreLibrary, sapCore, Core) {
    "use strict";

    /**
     * Constructor for a new ObjectInspector.
     *
     * Popup control for showing metadata Objects.
     *
     * ## Features: ##
     * Has properties geared towards SAS Metadata information
     * such as keywords, objectType, etc.
     *
     * These properties assign the correct values to the more
     * abstracted ObjectInspectorBase which it extends.
     *
     * Support for:
     * - header controls
     * - tab-container content
     * - footer action named and custom buttons
     *
     * @class
     * @name sas.hc.m.objectInspector.ObjectInspector
     * @extends sas.hc.m.objectInspector.ObjectInspectorBase
     * @version 904001.11.16.20251118090100_f0htmcm94p
     * @author Jonathan Brink, Ralph Marshall
     * @constructor
     * @public
     */
    var ObjectInspector = ObjectInspectorBase.extend("sas.hc.m.objectInspector.ObjectInspector", /** @lends sas.hc.m.objectInspector.ObjectInspector.prototype */ {
        metadata: {
            library: "sas.hc.m",
            properties: {
                objectName: {type: "string", group: "Data"},
                creationDate: {type: "object", group: "Data"},
                modificationDate: {type: "object", group: "Data"},
                objectType: {type: "string", group: "Data"},
                objectTypeDisplayName: {type: "string", group: "Data"},
                path: {type: "string", group: "Data"},
                keywords: {type: "string[]", group: "Data"},

                // Note that only one of these will be displayed in the header
                owner: {type: "string", group: "Data"},
                creator: {type: "string", group: "Data"}
            }
        },
        renderer: "sas.hc.m.PopoverRenderer",

        // resource bundle for ObjectInspector
        rb: sap.ui.getCore().getLibraryResourceBundle("sas.hc.m")
    });

    /**
     * init lifecycle callback.
     * @private
     */
    ObjectInspector.prototype.init = function() {
        // invoke super
        ObjectInspectorBase.prototype.init.apply(this, arguments);

        Array.prototype.push.apply(this._aNotSupportedProperties, [
            "objectName", "creationDate", "modificationDate",
            "objectType", "objectTypeDisplayName", "path",
            "keywords", "owner", "creator"
        ]);
    };

    /**
     * objectType setter override
     * Avoids invalidation
     * Appends to objectData
     * @param {string} sObjectType The objectType property value
     * @public
     */
    ObjectInspector.prototype.setObjectType = function(sObjectType) {
        this.setProperty("objectType", sObjectType, true);
        this.setIconKey(sas.icons.getObjectTypeIcon(sObjectType));
        if (sObjectType && !this.getObjectTypeDisplayName()) {
            this._addObjectData({
                key: "objectType",
                name: this.rb.getText("objectInspector.objectType.txt"),
                value: sObjectType
            });
        }
    };

    /**
     * objectTypeDisplayName setter override
     * Avoids invalidation
     * Appends to objectData
     * @param {string} sObjectTypeDisplayName The objectTypeDisplayName property value
     * @public
     */
    ObjectInspector.prototype.setObjectTypeDisplayName = function(sObjectTypeDisplayName) {
        this.setProperty("objectTypeDisplayName", sObjectTypeDisplayName, true);
        if (sObjectTypeDisplayName) {
            this._addObjectData({
                key: "objectType",
                name: this.rb.getText("objectInspector.objectType.txt"),
                value: sObjectTypeDisplayName
            });
        }
    };

    /**
     * objectName setter override
     * Avoids invalidation
     * @param {string} sObjectName The objectName property value
     * @public
     */
    ObjectInspector.prototype.setObjectName = function(sObjectName) {
        this.setProperty("objectName", sObjectName, true);
        this.setHeaderText(sObjectName);
    };

    /**
     * owner setter override
     * Avoids invalidation
     * @param {string} sOwner The owner property value
     * @public
     */
    ObjectInspector.prototype.setOwner = function(sOwner) {
        this.setProperty("owner", sOwner, true);
        this.setSecondaryText(this.rb.getText("objectInspector.owner.fmt", [sOwner]));
    };

    /**
     * creator setter override
     * Avoids invalidation
     * @param {string} sCreator The creator property value
     * @public
     */
    ObjectInspector.prototype.setCreator = function(sCreator) {
        this.setProperty("creator", sCreator, true);
        if (sCreator === undefined) {
            this.setSecondaryText(undefined);
        } else {
            this.setSecondaryText(this.rb.getText("objectInspector.creator.fmt", [sCreator]));
        }
    };

    /**
     * keywords setter override
     * Avoids invalidation
     * Appends to objectData
     * Takes in string[]
     * @param {string[]} asKeywords The keywords property value
     * @public
     */
    ObjectInspector.prototype.setKeywords = function(asKeywords) {
        this.setProperty("keywords", asKeywords, true);
        if (asKeywords && asKeywords.length > 0) {
            this._addObjectData({
                key: "keywords",
                name: this.rb.getText("objectInspector.keywords.txt"),
                value: asKeywords.join(", ")
            });
        }
    };

    /**
     * creationDate setter override
     * Avoids invalidation
     * Appends to objectData
     * @param {date} dateCreationDate The creationData property value
     * @public
     */
    ObjectInspector.prototype.setCreationDate = function(dateCreationDate) {
        this.setProperty("creationDate", dateCreationDate, true);

        if (dateCreationDate) {
            this._addObjectData({
                key: "creationDate",
                name: this.rb.getText("objectInspector.creationDate.txt"),
                value: FormatUtil.formatDate(dateCreationDate)
            });
        }
    };

    /**
     * modificationDate setter override
     * Avoids invalidation
     * Appends to objectData
     * @param {date} dateModificationDate The modificationDate property value
     * @public
     */
    ObjectInspector.prototype.setModificationDate = function(dateModificationDate) {
        this.setProperty("modificationDate", dateModificationDate, true);
        if (dateModificationDate) {
            this._addObjectData({
                key: "modificationDate",
                name: this.rb.getText("objectInspector.modificationDate.txt"),
                value: FormatUtil.formatDate(dateModificationDate)
            });
        }
    };

    /**
     * path setter override
     * Avoids invalidation
     * Appends to objectData
     * @param {string} sPath  The path property value
     * @public
     */
    ObjectInspector.prototype.setPath = function(sPath) {
        this.setProperty("path", sPath, true);
        if (sPath) {
            this._addObjectData({
                key: "objectLocation",
                name: this.rb.getText("objectInspector.objectLocation.txt"),
                value: sPath,
                valueTextDirection: coreLibrary.TextDirection.LTR
            });
        }
    };

    return ObjectInspector;
}, /* bExport= */ true);
