// (c) 2015, SAS Institute Inc.
sap.ui.define([
    "jquery.sap.global",
    "sas/hc/m/Dialog",
    "sas/hc/m/Button",
    "sas/hc/m/TextArea",
    "sas/hc/m/Text",
    "sas/hc/m/Link",
    "sap/ui/core/Core",
    "sas/hc/ui/core/Core"
], function(jQuery, Dialog, Button, TextArea, Text, Link, sapCore, Core) {
    "use strict";

    /**
     * Constructor for a new _ErrorDialog.
     *
     * An _ErrorDialog is a dialog containing error information and a close button.
     *
     * The error information is displayed in a textarea and is mostly the stack-trace of the error.
     * The textarea can be hidden and shown.
     *
     * This control is used by the global error handler when window.onerror is invoked by an unchecked exception.
     *
     * NOTE: This class is _only_ to be used by the Global Error Handler.
     * Any other use-cases should be covered by sas.hc.m.MessageDialog.
     *
     * @author Jonathan Brink (Jonathan.Brink@sas.com)
     * @version 904001.11.16.20251118090100_f0htmcm94p
     *
     * @class
     * @constructor
     * @public
     */
    var _ErrorDialog = Dialog.extend("sas.hc.m._ErrorDialog", /** @lends sas.hc.m._ErrorDialog.prototype */ {
        metadata: {
            library: "sas.hc.m",
            properties: {
                /**
                 * The error text that appears in the textarea
                 */
                message: {type: "string", group: "Appearance", defaultValue: ""},

                /**
                 * The message above the details section
                 */
                genericMessage: {type: "string", group: "Appearance", defaultValue: ""}
            },
            associations: {
                /**
                 * generic message label ("An error occurred...")
                 */
                label: {type: "sas.hc.m.Text"},

                /**
                 * holds error details (stack-trace, etc)
                 */
                textArea: {type: "sas.hc.m.TextArea"},

                /**
                 * used as a button to toggle the details textarea
                 */
                link: {type: "sas.hc.m.Link"}
            }
        },
        renderer: {
        }
    });

    /**
     * Initializes the control
     * @private
     */
    _ErrorDialog.prototype.init = function () {
        var self = this,
            sId = this.getId(),
            rb = sap.ui.getCore().getLibraryResourceBundle("sas.hc.m"),
            rbSapM = sap.ui.getCore().getLibraryResourceBundle("sap.m"),

            oTextArea = new TextArea({
                id: sId + "-textarea",
                editable: false,
                visible: false,
                width: "100%",
                rows: 10
            }),
            oBeginButton = new Button({
                id: sId + "-beginButton",
                text: rb.getText("messageBox.action.CLOSE.label")
            }),
            oLink = new Link({
                id: sId + "-link",
                text: rb.getText("MessageBox.message.showDetails.txt")
            }),
            oGenericMessageLabel = new Text({
                id: sId + "-genericMessageLabel",
                text: rb.getText("ErrorDialog.message.anErrorOccurred.txt")
            });

        // invoke super function
        if (Dialog.prototype.init) {
            Dialog.prototype.init.apply(this, arguments);
        }

        // add sub-controls to _ErrorDialog
        this.addContent(oGenericMessageLabel);
        this.addContent(oTextArea);
        this.setBeginButton(oBeginButton);
        this.setLink(oLink);

        // keep references to sub-controls in associations
        this.setLabel(oGenericMessageLabel);
        this.setTextArea(oTextArea);

        // make sure dialog closes properly
        this.attachAfterClose(function() {
            self.destroy();
        });

        // use special event handlers for the buttons
        // because often _ErrorDialog is shown when system is in an error state
        // and typical framework features are not available
        oBeginButton.addEventDelegate({
            "onBeforeRendering": function() {
                if (oBeginButton.$() !== undefined) {
                    oBeginButton.$().off("tap");
                    oBeginButton.$().off("keyup");
                }
            },
            "onAfterRendering": function () {
                if (oBeginButton.$() !== undefined) {
                    oBeginButton.$().on("tap", function() {
                        self.close();
                    });
                    oBeginButton.$().on("keyup", function(e) {
                        if (e.which === jQuery.sap.KeyCodes.SPACE || e.which === jQuery.sap.KeyCodes.ENTER) {
                            self.close();
                        }
                    });
                }
            },
            "exit": function() {
                if (oBeginButton.$() !== undefined) {
                    oBeginButton.$().off("tap");
                    oBeginButton.$().off("keyup");
                }
            }
        });

        oLink.addEventDelegate({
            "onBeforeRendering": function() {
                if (oLink.$() !== undefined) {
                    oLink.$().off("tap");
                    oLink.$().off("keyup");
                }
            },
            "onAfterRendering": function () {
                if (oLink.$() !== undefined) {
                    oLink.$().on("tap", function() {
                        self._toggleTextarea();
                    });
                    oLink.$().on("keyup", function(e) {
                        if (e.which === jQuery.sap.KeyCodes.SPACE || e.which === jQuery.sap.KeyCodes.ENTER) {
                            self._toggleTextarea();
                        }
                    });
                }
            },
            "exit": function() {
                if (oLink.$() !== undefined) {
                    oLink.$().off("tap");
                    oLink.$().off("keyup");
                }
            }
        });

        // set properties
        this.setTitle(window.__sap.title || rbSapM.getText("ERRORDIALOG_TITLE"));

        this.addStyleClass("sasMErrorDialog");
    };

    /**
     * onBeforeRendering lifecycle callback
     * @private
     */
    _ErrorDialog.prototype.onBeforeRendering = function() {
        this.setShowClose(false);
        this.setMessageIcon(sas.icons.HC.ERRORSTATUS);

        if (Dialog.prototype.onBeforeRendering) {
            Dialog.prototype.onBeforeRendering.apply(this, arguments);
        }

        // add link to footer
        // this needs to happen in onBeforeRendering because
        // Dialog invokes "removeAllContent" in it's onBeforeRendering
        var oFooter = this._getToolbar(),
            oLink = sap.ui.getCore().byId(this.getLink());

        oFooter.addContent(oLink);

        this._handleVisibility();
    };

    /**
     * exit lifecycle callback
     * @private
     */
    _ErrorDialog.prototype.exit = function() {
        if (Dialog.prototype.exit) {
            Dialog.prototype.exit.apply(this, arguments);
        }

        // need to delete these manually because Dialog does not manage it's footer toolbar in a standard way
        var oFooter = this._oToolbar;
        if (oFooter && typeof oFooter.getAllContent === "function") {
            oFooter.getAllContent()
                .filter(function(oContent) {
                    return oContent !== undefined && typeof oContent.destroy === "function";
                })
                .forEach(function(oContent) {
                    oContent.destroy();
                });
        }
    };

    /**
     * Overridden setter for "message" property.
     * Pass message to textarea
     * @param {string} sMessage The message that will be displayed in the textarea
     * @public
     */
    _ErrorDialog.prototype.setMessage = function(sMessage) {
        var oLink = sap.ui.getCore().byId(this.getLink()),
            oTextArea = sap.ui.getCore().byId(this.getTextArea());
        this.setProperty("message", sMessage, true);
        oTextArea.setValue(sMessage);
        this.setInitialFocus(oLink);
    };

    /**
     * "genericMessage" setter override
     * @public
     */
    _ErrorDialog.prototype.setGenericMessage = function(sGenericMessage) {
        var oGenericMessageLabel = sap.ui.getCore().byId(this.getLabel());
        this.setProperty("genericMessage", sGenericMessage);
        oGenericMessageLabel.setText(sGenericMessage);
    };

    /**
     * Hide or show sub-controls based on their content
     * @private
     */
    _ErrorDialog.prototype._handleVisibility = function() {
        var oGenericMessageLabel = sap.ui.getCore().byId(this.getLabel()),
            sGenericMessage = oGenericMessageLabel.getText(),
            sMessage = this.getMessage(),
            oLink = sap.ui.getCore().byId(this.getLink());

        // if no generic message, then hide generic message label
        if (sGenericMessage === undefined || sGenericMessage === "") {
            oGenericMessageLabel.setVisible(false);
        }

        // if no message, then hide details section and button
        if (sMessage === undefined || sMessage === "") {
            oLink.setVisible(false);
        }
    };

    /**
     * Toggle visibility of textarea
     * @override
     * @private
     */
    _ErrorDialog.prototype._toggleTextarea = function() {
        var rb = sap.ui.getCore().getLibraryResourceBundle("sas.hc.m"),
            oTextArea = sap.ui.getCore().byId(this.getTextArea()),
            oLink = sap.ui.getCore().byId(this.getLink());

        // toggle visibility and button labels
        if (oTextArea.getVisible() === true) {
            oTextArea.setVisible(false);
            oLink.setText(rb.getText("MessageBox.message.showDetails.txt"));
        } else {
            oTextArea.setVisible(true);
            oLink.setText(rb.getText("MessageBox.message.hideDetails.txt"));
        }
        oTextArea.rerender();
    };

    return _ErrorDialog;
}, /* bExport= */ true);
