// (c) 2016, SAS Institute Inc.

/**
 * SAS extension of sap.m.MessageStrip for HTML Commons.
 * @name sas.hc.m.MessageStrip
 * @extends sap.m.MessageStrip
*/

sap.ui.define([
    "sap/m/MessageStrip",
    "sap/ui/core/Control",
    "sap/ui/core/Core",
    "sap/ui/core/IconPool",
    "sap/ui/core/library",
    "./Button",
    "./Text",
    "sas/hc/ui/core/Core"
], function(SapMessageStrip, Control, sapCore, IconPool, sapCoreLibrary, Button, Text, Core) {
    "use strict";

    var rb = sap.ui.getCore().getLibraryResourceBundle("sas.hc.m");

    var MessageType = sapCoreLibrary.MessageType;

    var mIconsConfig = {};
    mIconsConfig[MessageType.Information] = {
        src: sas.icons.HC.INFORMATIONSTATUS,
        alt: rb.getText("messageStrip.message.information.txt")
    };
    mIconsConfig[MessageType.None] = mIconsConfig[MessageType.Information];
    mIconsConfig[MessageType.Warning] = {
        src: sas.icons.HC.WARNINGSTATUS,
        alt: rb.getText("messageStrip.message.warning.txt")
    };
    mIconsConfig[MessageType.Error] = {
        src: sas.icons.HC.ERRORSTATUS,
        alt: rb.getText("messageStrip.message.error.txt")
    };
    mIconsConfig[MessageType.Success] = {
        src: sas.icons.HC.COMPLETESTATUS,
        alt: rb.getText("messageStrip.message.success.txt")
    };

    /**
     * Constructor for a new MessageStrip.
     *
     * @param {string} [sId] id for the new control, generated automatically if no id is given
     * @param {object} [mSettings] initial settings for the new control
     *
     * @class
     * MessageStrip is a control that enables the embedding of application-related messages in the application. There are 4 types of messages: Information, Success, Warning and Error. Each message can have a close button, so that it can be removed from the UI if needed.
     *
     * @extends sap.m.MessageStrip
     * @version 904001.11.16.20251118090100_f0htmcm94p
     *
     * @constructor
     * @public
     * @alias sas.hc.m.MessageStrip
     */
    var MessageStrip = SapMessageStrip.extend("sas.hc.m.MessageStrip", {
        metadata : {
            library : "sas.hc.m",

            properties: {
                "messageIcon": {type: "object", defaultValue: undefined},

                /**
                 * Indicates that the strip has a transparent background when true. When false the background
                 * is based on the type property.
                 */
                "transparentBackground": { type: "boolean", defaultValue: false },

                /**
                 * Based on UX requirements the close button is required in the SAS version of the control.
                 */
                showCloseButton: { type: "boolean", group: "Appearance", defaultValue: true }
            },

            aggregations: {
                "closeButton": {
                    type: "sas.hc.m.Button",
                    multiple: false
                },
                "helpPopup": {
                    type: "sas.hc.m.HelpPopup",
                    multiple: false
                },
                "actionButton": {
                    type: "sas.hc.m.Button",
                    multiple: false
                }
            }
        },

        renderer : "sas.hc.m.MessageStripRenderer"
    });

    /**
     * Based on UX requirements the close button is required in the SAS version of the control.
     * Attempting to set the value to anything other than true will be ignored, and fetching the
     * value will always return true, regardless of any internal state of the property.
     * @param bShowButton Whether or not to show the close button. IGNORED.
     * @returns {sas.hc.m.MessageStrip} this pointer to allow method chaining
     */
    MessageStrip.prototype.setShowCloseButton = function(bShowButton) {
        this.setProperty("showCloseButton", true);
        return this;
    };

    /**
     * Based on UX requirements the close button is required in the SAS version of the control.
     * This function always returns TRUE, regardless of attempts to disable the close button.
     * @returns {boolean} Flag indicating whether to show the close button. ALWAYS TRUE.
     */
    MessageStrip.prototype.getShowCloseButton = function() {
        return true;
    };

    MessageStrip.prototype.init = function() {
        SapMessageStrip.prototype.init.apply(this, arguments);

        var self = this;
        var rb = sap.ui.getCore().getLibraryResourceBundle("sas.hc.m");
        var oCloseButton = new Button({
            icon: sas.icons.HC.CLOSE,
            press: function() {
                self.close();
            },
            design: Button.Design.SmallNeutral
        });

        // Add both SAP and SAS style classes.
        oCloseButton.addStyleClass("sapMMsgStripCloseButton");
        oCloseButton.addStyleClass("sasMMsgStripCloseButton");
        oCloseButton.setAriaLabel(rb.getText("messageStrip.closeButton.label.txt"));

        this.setCloseButton(oCloseButton);
    };

    MessageStrip.prototype._createText = function() {
        var oText = new Text();
        return oText;
    };

    MessageStrip.prototype.setActionButton = function(oActionButton) {

        if (oActionButton) {
            oActionButton.setDesign(Button.Design.SmallNeutral);
            oActionButton.addStyleClass("sasMMsgStripActionButton");
        }

        this.setAggregation("actionButton", oActionButton);
    };

    MessageStrip.prototype.setHelpPopup = function(oHelpPopup) {

        if (oHelpPopup) {
            oHelpPopup.addStyleClass("sasMMsgStripHelp");
        }

        this.setAggregation("helpPopup", oHelpPopup);
    };

    MessageStrip.prototype.setAggregation = function (sName, oControl, bSupressInvalidate) {
        // SAP's MessageStrip forces the _text aggregation to include the MessageStrip as
        // aria-labelledby of the Text. For our own improved screen reader UX, we don't want that.
        return Control.prototype.setAggregation.call(this, sName, oControl, bSupressInvalidate);
    };

    /**
     * Create image tag object for icon.  This method is intended for
     * internal use only.
     *
     * @returns {sas.hc.m.Image} image object using icon information passed to method
     * @private
     */
    MessageStrip.prototype._createIcon = function(iconSrc, iconAltText) {
        var oType = this.getType();
        var oIconData = mIconsConfig[oType];
        return IconPool.createControlByURI({
            id: this.getId() + "-icon",
            src: oIconData.src,
            decorative: false,
            alt: oIconData.alt
        });
    };

    /**
     * Return icon associated with the current control type
     *
     * @returns {sas.hc.m.Image} Image instance that represents the current control Message type.  The types are complete, error,
     * warning and information
     */
    MessageStrip.prototype.getIcon = function() {
        var oMessageIcon = this.getMessageIcon();

        // check if image source has changed - if yes destroy and reset image control
        // since there is a setType, we need to make sure current icon reflects the correct type
        if (oMessageIcon && !this._iconMatchesMessageType()) {
            oMessageIcon.destroy();
            oMessageIcon = null;
        }

        if (!oMessageIcon) {
            oMessageIcon = this._createIcon();
            this.setMessageIcon(oMessageIcon);
        }
        return oMessageIcon;
    };

    MessageStrip.prototype._iconMatchesMessageType = function() {
        var oType = this.getType();
        var oMessageIcon = this.getMessageIcon();
        var sExpectedSrc = mIconsConfig[oType].src;
        return oMessageIcon && oMessageIcon.getSrc() === sExpectedSrc;
    };

    MessageStrip.prototype.exit = function() {
        var oMessageIcon = this.getMessageIcon();
        if (oMessageIcon) {
            oMessageIcon.destroy();
        }

        if (SapMessageStrip.prototype.exit) {
            SapMessageStrip.prototype.exit.apply(this, arguments);
        }
    };
    return MessageStrip;
});

