jQuery.sap.require("sas.fscf.views.estimation.HcHelper");
jQuery.sap.require("sas.fscf.views.estimation.EstimationHelper");
jQuery.sap.require("sas.fscf.views.estimation.EstimationResultsHelper");

sap.ui.jsfragment("sas.fscf.views.estimation.EstimationPropertiesDialog", {

	createContent: function (controller) {/* jshint -W034 */
		"use strict";                     /* jshint +W034 */

        const HcHelper = sas.fscf.views.estimation.HcHelper;
        const ONE_HUNDRED_PERCENT = HcHelper.ONE_HUNDRED_PERCENT;
        const reduceSpaceAboveDialogButtons = HcHelper.reduceSpaceAboveDialogButtons;
        const EstimationHelper = sas.fscf.views.estimation.EstimationHelper;
        const RuleHelper = EstimationHelper.RuleHelper;
		const SpacerFactory = EstimationHelper.SpacerFactory;
        const formatDateTime = EstimationHelper.formatDateTime;
        const formatGMTOffsetString = EstimationHelper.formatGMTOffsetString;
        const getLocalizedRuleState = EstimationHelper.getLocalizedRuleState;
        const getText = EstimationHelper.getText;
		const newText = EstimationHelper.newText;
        const parseInteger = EstimationHelper.parseInteger;
		const EstimationResultsHelper = sas.fscf.views.estimation.EstimationResultsHelper;
		const Icon = EstimationResultsHelper.Icon;

        const NOT_APPLICABLE = getText("rules.estimation.not.applicable.label");
        const NOT_APPLICABLE_ABBREVIATION = getText("rules.estimation.not.applicable.abbreviated.label");

        const formatter = EstimationHelper.getFormatter();

		const newLabel = function (props, span) {
            const isPropsString = (typeof props === "string");
            const text = isPropsString ? props : props.text;
            const tooltip = isPropsString ? "" : (props.tooltip) ? props.tooltip : "";
            const layoutSpan = (span) ? span : "L5 M6 S9";

			return new sas.hc.m.Label({
                text: text,
                tooltip: tooltip,
				layoutData: new sas.hc.ui.layout.GridData({
					span: layoutSpan,
					linebreakL: true,
					linebreakM: true,
					linebreakS: true,
				} ),
			});
		};

		const getModel = function () {
			return controller.getPropertiesModel();
		};

		const getEstimation = function () {
			return getModel().getProperty("/estimation");
		};

		const generalContent = [
			newLabel(getText("rules.rulesPage.name.txt"), "L3 M6 S9"),
			newText("{/estimation/name}"),
			newLabel(getText("rules.estimation.id.label"), "L3 M6 S9"),
			newText("{/estimation/id}"),
			newLabel(getText("rules.estimation.author.label"), "L3 M6 S9"),
			newText("{/estimation/author}"),
			newLabel(
                {
                    text: getText("rules.estimation.created.label"),
                    tooltip: {
                        path: "/estimation/created",
                        formatter: formatGMTOffsetString,
                    },
                },
                "L3 M6 S9"
            ),
            newText({
                text: {
                    path: "/estimation/created",
                    formatter: formatDateTime,
                } ,
                tooltip: {
                    path: "/estimation/created",
                    formatter: formatGMTOffsetString,
                },
            } ),
		];
		const generalContentGrid = new sap.ui.layout.Grid({
			hSpacing: 1,
			vSpacing: 0,
			content: generalContent,
			defaultSpan: "L9 M10 S12",
			position: sap.ui.layout.GridPosition.Center,
		} );

		const renderRule = function (rule) {
            const state = rule.state.toLowerCase();
            const icon = (state.toLowerCase() === "testing") ? Icon.TEST : Icon.DEPLOY;
            const stateLocalized = getLocalizedRuleState(state);
			const iconToltip = stateLocalized;
            const tooltip = getText("rules.rulesPage.name.txt") + ": " + rule.name + "\n"
                    + getText("rules.estimation.id.label") + ": " + rule.id + "\n"
                    + getText("rules.rulesPage.rules.state.txt") + ": " + stateLocalized;

			return new sas.hc.m.HBox({
				width: ONE_HUNDRED_PERCENT,
				alignContent: sap.m.FlexAlignItems.Center,
				justifyContent: sap.m.FlexJustifyContent.Start,
				items: [
					new sas.hc.ui.core.Icon({
						src: icon,
						tooltip: iconToltip,
					} ),
					SpacerFactory.horizontalSpacer("0.5rem"),
					newText(rule.displayIdString),
					SpacerFactory.horizontalSpacer("1rem"),
					// Long text isn't truncated with ellipses appended.
					newText({
						text: rule.name,
						tooltip: tooltip,
					} ),
				],
			} );
		};

        const setUpRules = function (rules) {
            rules.forEach(function (each) {
                each.displayIdString = each.displayId;
                each.displayId = null;  // Let it be regenerated as a number by RuleHelper.
                RuleHelper.setDisplayId(each);
            } );
        };

		const renderRules = function (rules) {
            setUpRules(rules);
            rules.sort(RuleHelper.displayIdComparator).reverse();

			const rulesRendered = rules.map(function (each) {
				return renderRule(each);
			} );

			return new sas.hc.m.VBox({
				justifyContent: sap.m.FlexJustifyContent.Start,
				items: [rulesRendered],
				// width: "auto",
			} );
		};

		const ruleSection = new sas.hc.ui.commons.windowshade.WindowShadeSection((this.sId+"RuleWindowShadeSection"), {
			title: getText("rules.items.rules.txt"),
            content: [renderRules(getEstimation().criteria.rules)],
			collapsed: false,
			maxHeight: "auto",
         } );

		const renderCriteria = function () {
            const NOT_APPLICABLE_TEXT = {text: NOT_APPLICABLE_ABBREVIATION, tooltip: NOT_APPLICABLE};

            const estimation = getEstimation();
            // User variable calculation
            const isUserVarCalculationApplicable = (estimation.criteria.forceUserVariableRecalculation !== undefined);
            const isUserVarCalculationEnabled = estimation.criteria.forceUserVariableRecalculation;
            const isUserVarCalculationApplicableAndEnabled = isUserVarCalculationApplicable && isUserVarCalculationEnabled;

            // User variable priming period
            let userVarPrimingDurationLabel = getText("rules.estimation.user.variable.priming.label");
            const unit = estimation.criteria.userVariablePrimingUnit;
			if (unit) {
                const resourceKey = "rules.estimation." + unit.trim().toLowerCase() + ".radiobutton.label";
                const unitLocalized = getText(resourceKey);
				userVarPrimingDurationLabel += " (" + unitLocalized.toLowerCase() + ")";
            }
            const userVarPrimingDurationLocalizedText = estimation.criteria.userVariablePrimingDuration
                    ? formatter.formatNumber(estimation.criteria.userVariablePrimingDuration)
                    : NOT_APPLICABLE_TEXT;

            // User variable calculation - parallel processing
            const parallelizeBy = (estimation.criteria.parallelizeBy)
                    ? controller.estimationsController.getSplitTxnsOption(estimation.criteria.parallelizeBy)
                    : null;
            const parallelizeByLabel = parallelizeBy ? parallelizeBy.label : "";
            const parallelizeByText = isUserVarCalculationApplicableAndEnabled
                    ? (parallelizeBy
                            ? parallelizeByLabel
                            : {text: NOT_APPLICABLE_ABBREVIATION, tooltip: NOT_APPLICABLE} )
                    : {text: NOT_APPLICABLE_ABBREVIATION, tooltip: NOT_APPLICABLE};
            const useParallelProcessingText = isUserVarCalculationApplicableAndEnabled
                    ? (parallelizeBy
                            ? getText("fcm.application.yes.txt")
                            : getText("fcm.application.no.txt") )
                    : {text: NOT_APPLICABLE_ABBREVIATION, tooltip: NOT_APPLICABLE};

			const firedTransactionLimit = estimation.criteria.firedTransactionLimit;
            const firedTransactionLimitString = formatter.formatNumber(firedTransactionLimit);

			return [
				newLabel({
                    text: getText("rules.estimation.start.date.time.tip.label"),
                    tooltip: {
                        path: "/estimation/criteria/startDate",
                        formatter: formatGMTOffsetString,
                    },
                } ),
                newText({
                    text: {
                        path: "/estimation/criteria/startDate",
                        formatter: formatDateTime,
                    },
                    tooltip: {
                        path: "/estimation/criteria/startDate",
                        formatter: formatGMTOffsetString,
                    },
                } ),

    			newLabel({
                    text: getText("rules.estimation.end.date.time.tip.label"),
                    tooltip: {
                        path: "/estimation/criteria/endDate",
                        formatter: formatGMTOffsetString,
                    },
                } ),
                newText({
                    text: {
                        path: "/estimation/criteria/endDate",
                        formatter: formatDateTime,
                    },
                    tooltip: {
                        path: "/estimation/criteria/endDate",
                        formatter: formatGMTOffsetString,
                    },
                } ),

				newLabel(getText("rules.estimation.duration.label")),
                newText(formatter.formatNumber(parseInteger(estimation.criteria.duration))),

				newLabel(userVarPrimingDurationLabel),
                newText (userVarPrimingDurationLocalizedText),

                newLabel(getText("rules.estimation.use.parallel.processing.label")),
                newText(useParallelProcessingText),

                newLabel(getText("rules.estimation.split.transactions.by.label")),
                newText(parallelizeByText),

				newLabel(getText("rules.estimation.fired.transactions.limit.input.label")),
                newText(firedTransactionLimitString),

				newLabel(getText("rules.estimation.log.with.additional.details.label")),
				newText(estimation.criteria.logWithAdditionalDetails ? getText("fcm.application.yes.txt") : getText("fcm.application.no.txt")),
			];
		};

		const criteriaContentGrid = new sap.ui.layout.Grid({
			hSpacing: 1,
			vSpacing: 0,
			content: [],
			defaultSpan: "L7 M10 S12",
			position: sap.ui.layout.GridPosition.Center,
		} );
		criteriaContentGrid.addEventDelegate({
			onBeforeRendering: function (event) {
				renderCriteria().map(function (each) {
					event.srcControl.addContent(each);
				} );
			},
		} );

		const criteraSection = new sas.hc.ui.commons.windowshade.WindowShadeSection((this.sId+"CriteriaWindowShadeSection"), {
			title: getText("rules.estimation.criteria.label"),
            content: [criteriaContentGrid],
			collapsed: false,
			maxHeight: "auto",
         } );

        const windowShade = new sas.hc.ui.commons.windowshade.WindowShade((this.sId+"WindowShade"), {
            width: ONE_HUNDRED_PERCENT,
            sections: [ruleSection, criteraSection],
        } ).addEventDelegate({
			onAfterRendering: function (event) {
				// Not sure why even though the last section, criteriaSection, has its property collapsed set to false, it's shown
				// as collapsed.  So we call openAllSections() here to force it to be expanded!
				event.srcControl.openAllSections();
			},
        } );

		const emphasizeSectionHeaders = function ($propertiesDialog) {
			$propertiesDialog.find(".sasUiWndSectionHrdTitle").css("font-weight", "bold");
		};

        const propertiesDialog = new sas.hc.m.Dialog({
        	id: this.sId + "EstimationPropertiesDialog",
			title: {
				parts: ["/estimation/id", "/estimation/name"],
				formatter: function (id, name) {
					return getText("rules.estimation.fmt.title", [name , id]);
				},
			},
    	    content: [
				generalContentGrid,
				SpacerFactory.verticalSpacer("1rem"),
				windowShade,
			],
			contentHeight: "90%",
            contentWidth: "55%",
            cssMaxWidth: ONE_HUNDRED_PERCENT,
            cssMaxHeight: ONE_HUNDRED_PERCENT,
        	resizable: true,
        	draggable: true,
    	    rightButton: EstimationHelper.newCloseButton(),
			beforeOpen: [controller.handlePropertiesDialogBeforeOpen, controller],
			afterClose: [controller.handlePropertiesDialogAfterClose, controller],
		} )
			.addStyleClass("cfEstPropsDialog")
			.addEventDelegate({
				// Not sure why the last WindowShadeSection's height is set to 100% by default, which forces the vertical scroll bar
				// to appear due to extra blank space from the section.  Here we programmatically set it to "auto" to get around
				// the problem.
				onAfterRendering: function (event) {
                    const dialog = event.srcControl;
                    reduceSpaceAboveDialogButtons(dialog);

					const $dialog = dialog.$();
					$dialog.find(".sasUiWnd").css("height", "auto");
					emphasizeSectionHeaders($dialog);

					$dialog.find(".sapMText").each(function () {
						this.style.setProperty("line-height", "1.3rem", "important");
					} );
				},
			} );

        return propertiesDialog;
    },
} );
