sap.ui.define("sas.fscf.views.estimation.EstimationHelper",
	[],
	(function () {    /* jshint -W034 */
        "use strict"; /* jshint +W034 */

        const MessageType = (function () {
            const MessageTypeConstant = function (aName, debug, debugEstimation, error, info, warning, colorString) {
                const answer = {
                    color: (colorString) ? colorString : "#000",
                    name: aName,
                    isDebug: debug,
                    isDebugEstimation: debugEstimation,
                    isError: error,
                    isInfo: info,
                    isText: (!debug && !error && !info && !warning),
                    isWarning: warning,
                };
                Object.freeze && Object.freeze(answer);
                return answer;
            };

            const Name = {
                DEBUG: "DEBUG",
                DEBUG_ESTIMATION: "DEBUG_ESTIMATION",
                ERROR: "ERROR",
                INFO: "INFO",
                TEXT: "TEXT",
                WARNING: "WARNING",
            };
            Object.freeze && Object.freeze(Name);
            const DEBUG_ESTIMATION_INDICATOR = "-Priming/Maturation";

            const map =  new Map();
            map.set(Name.DEBUG, new MessageTypeConstant(Name.DEBUG, true, false, false, false, false));
            map.set(Name.DEBUG_ESTIMATION, new MessageTypeConstant(Name.DEBUG_ESTIMATION, false, true, false, false, false));
            map.set(Name.ERROR, new MessageTypeConstant(Name.ERROR, false, false, true,  false, false, "#F00"));
            map.set(Name.INFO, new MessageTypeConstant(Name.INFO, false, false, false, true,  false, "#00F"));
            map.set(Name.TEXT, new MessageTypeConstant(Name.TEXT, false, false, false, false, false));
            map.set(Name.WARNING, new MessageTypeConstant(Name.WARNING, false, false, false, false, true, "#006000"));

            const valueOf = function (name) {  // default to TEXT
                const key = name.toUpperCase();
                const answer = map.get(key);
                return (answer) ? answer : map.get("TEXT");
            };

            return {
                DEBUG: map.get(Name.DEBUG),
                DEBUG_ESTIMATION_INDICATOR: DEBUG_ESTIMATION_INDICATOR,
                ERROR: map.get(Name.ERROR),
                NOTE: map.get(Name.INFO),
                INFO: map.get(Name.TEXT),
                WARNING: map.get(Name.WARNING),
                valueOf: valueOf,
            };
        } )();

		const IdFactory = (function () {
			let prefix = "";
			let index = 1;

			const create = function (id) {
				return prefix + id + index++;
			};

			const init = function (idPrefix) { /* jshint -W116 */
				if (idPrefix != null) {
					prefix = idPrefix;
				}                              /* jshint +W116 */
			};

			return {
				create: create,
				init: init,
			};
		} )();

		const NullCheck = (function () {
			const LogError = {};
			const NullPointerException = {};

			const verify = function verify(anObject, logError) {
				if ((anObject === undefined) || (anObject === null)) {
					if (logError === LogError) {
							console.error("NullPointerException..." + verify.caller);
					}
					throw NullPointerException;
				}
			};

			return {
				LogError: LogError,
				NullPointerException: NullPointerException,
				verify: verify,
			};
		} )();

		const newText = function (param) {
			NullCheck.verify(param);

			const isParamString = (typeof param === "string");
			const text = (isParamString) ? param : param.text;
			const tooltip = (!isParamString && param.tooltip) ? param.tooltip : "";
			const wrapping = (!isParamString && param.wrapping) ? param.wrapping : false;

			return new sas.hc.m.Text({
				text: text,
				tooltip: tooltip,
				maxLines: 2,
				wrapping: wrapping,
				width: "100%",
			} );
		};

        const formatDateTime = function (val) {
            if (val === undefined || val === null) {
                return val;
            }
            const date = new Date(val);
            if (Number.isNaN(date.getTime())) {
                return val;
            }
            return sas.fscf.dateTimeFormatter.format(date, true);	// true means use UTC
        };

        // Date.toString():   https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Date/toString#Description
        // Standard ECMA-262: https://www.ecma-international.org/publications/standards/Ecma-262.htm
        const formatDateTimeEcma262 = function (val) {
            if (val === undefined || val === null) {
                return val;
            }
            const date = new Date(val);
            if (Number.isNaN(date.getTime())) {
                return val;
            }
            return date.toString();
        };

		const SpacerBuilder = function () {
			const Default = {
				HEIGHT: "0rem",
				WIDTH: "1rem",
			};

			let height = null;
			let width = null;

			this.setHeight = function (aHeight) {height = aHeight; return this;};

			this.setWidth = function (aWidth) {width = aWidth; return this;};

			this.build = function () {
				return new sas.hc.ui.commons.layout.AbsoluteLayout({
					height: (height) ? height : Default.HEIGHT,
					width: (width) ? width : Default.WIDTH,
				} );
			};
        };

		const SpacerFactory = (function (Builder) {
			const horizontalSpacer = function (width) {
				return new Builder().setWidth(width).build();
			};

			const verticalSpacer = function (height) {
				return new Builder().setHeight(height).build();
			};

			return {
				horizontalSpacer: horizontalSpacer,
				verticalSpacer: verticalSpacer,
			};
		} )(SpacerBuilder);

		const Wizard = (function () {
									   /* jshint -W123 */
			function Wizard (aDialog) {/* jshint +W123 */
				let currentIndex = 0;
				const dialog = aDialog;
				const tabs = [];

				this.decrementIndex = function () {
					if (this.isCanGoBack()) {
						--currentIndex;
					}
				};

				this.getCurrentIndex = function () {return currentIndex;};

				this.getDialog = function () {return dialog;};

				this.incrementIndex = function () {
					if (this.isCanGoNext()) {
						++currentIndex;
					}
				};

				this.getTabs = function () {return tabs;};
			}

			const isValidIndex = function (wizard, index) {
				return (index >= 0) && (index <= (wizard.getTabCount()-1));
			};

			const getTab = function (wizard, index) {
				return isValidIndex(wizard, index)
					? wizard.getTabs()[index]
					: null;
			};

			const getCurrentTab = function (wizard) {
				return getTab(wizard, wizard.getCurrentIndex());
			};

			Wizard.prototype.getTabCount = function () {
				return this.getTabs().length;
			};

			Wizard.prototype.isCanGoNext = function () {
				return this.getCurrentIndex() < (this.getTabCount() - 1);
			};

			Wizard.prototype.isCanGoBack = function () {
				return this.getCurrentIndex() > 0;
			};

			Wizard.prototype.addTab = function (controls) {
				const tab = {
					id: this.getTabCount(),
					controls: controls,
				};
				this.getTabs().push(tab);
			};

			Wizard.prototype.next = function () {
				if (this.isCanGoNext()) {
					this.getDialog().hideTab(getCurrentTab(this));
					this.incrementIndex();
					this.getDialog().showTab(getCurrentTab(this));
				}
			};

			Wizard.prototype.back = function () {
				if (this.isCanGoBack()) {
					this.getDialog().hideTab(getCurrentTab(this));
					this.decrementIndex();
					this.getDialog().showTab(getCurrentTab(this));
				}
			};

			return Wizard;
		}() );

		const UrlBuilder = (function () {
			const bizUnitStub = "{businessUnit}";
			const urlPrefix = "rest/businessunits/" + bizUnitStub;
			const estimationBaseUrl = urlPrefix + "/estimations/";
			const estimationIdStub = "{estimationId}";

			const alertTypesUrl = "rest/alerttypes?fields=id,name,description";
			const cancelEstimationUrl     = estimationBaseUrl + "{estimationId}/cancel";
			const createEstimationUrl     = estimationBaseUrl;
            const decisionMetricsUrl      = estimationBaseUrl + "{estimationId}/decisionMetrics";
            const deleteEstimationsUrl    = estimationBaseUrl;
			const estimationSummaryUrl    = estimationBaseUrl + "{estimationId}";
            const estimationsUrl          = estimationBaseUrl + "?showAll=";
            const exportToExcel           = estimationBaseUrl + "{estimationId}/exportTransactionsToExcel";
            const logUrl                  = estimationBaseUrl + "{estimationId}/log";
            const logMessagesUrl          = estimationBaseUrl + "{estimationId}/logMessages";
            const maxFiredTxnsSettingsUrl = estimationBaseUrl + "maxFiredTransactionsSettings";
            const queueMetricsUrl         = estimationBaseUrl + "{estimationId}/queueMetrics";
            const ruleMetricsUrl          = estimationBaseUrl + "{estimationId}/ruleMetrics";
            const rulesUrlTemplate        = urlPrefix + "/rules?filter=state=='Production' or state=='Testing'&fields=id,baseRuleId,version,revision,lastUpdateNumber,name,ruleTypeName,state,alertTypeId";
            const ruleTransactionInfoUrl  = estimationBaseUrl + "rulesTransactionInfo?ruleIds=";
            const ruleUrl                 = estimationBaseUrl + "{estimationId}/rule/{ruleId}";
			const splitTxnsOptionsUrl     = estimationBaseUrl + "splitTransactionsOptions";
			const timingMetricsUrl        = estimationBaseUrl + "{estimationId}" + "/timingMetrics";

			const specifyBizUnitInUrl = function (urlTemplate, bizUnit) {
				return urlTemplate.replace(bizUnitStub, bizUnit);
			};
			return {
				getAlertTypesUrl: function () {
					return alertTypesUrl;
				},
				getCancelEstimationUrl: function (bizUnit, estimationId) {
					return specifyBizUnitInUrl(cancelEstimationUrl, bizUnit).replace(estimationIdStub, estimationId);
				},
				getCreateEstimationUrl: function (bizUnit) {
					return specifyBizUnitInUrl(createEstimationUrl, bizUnit);
				},
				getDecisionMetricsUrl: function (bizUnit, estimationId) {
					return specifyBizUnitInUrl(decisionMetricsUrl, bizUnit).replace(estimationIdStub, estimationId);
				},
				getDeleteEstimationsUrl: function (bizUnit) {
					return specifyBizUnitInUrl(deleteEstimationsUrl, bizUnit);
				},
				getEstimationSummaryUrl: function (bizUnit, estimationId) {
					return specifyBizUnitInUrl(estimationSummaryUrl, bizUnit).replace(estimationIdStub, estimationId);
				},
				getEstimationsUrl: function (bizUnit, toShowAll, nameFilter) {
                    let answer = specifyBizUnitInUrl(estimationsUrl, bizUnit) + toShowAll;
                    if (nameFilter && (nameFilter.trim().length > 0)) {
                        answer += "&filter=name=='" + nameFilter + "'";
                    }
                    return answer;
                },
                getExportToExcelUrl: function (bizUnit, estimationId, ruleId, type) {
                    const myRuleId = ruleId ? ruleId : "";
                    const fireType = type ? type : "";
                    const answer = specifyBizUnitInUrl(exportToExcel, bizUnit).replace(estimationIdStub, estimationId);
                    return (answer) ? (answer + "?ruleId=" + myRuleId + "&type=" + fireType) : answer;
				},
                getLogUrl: function (bizUnit, estimationId, logNumber) {
                    const answer = specifyBizUnitInUrl(logUrl, bizUnit).replace(estimationIdStub, estimationId);
                    return (logNumber) ? (answer + "?filter=logNumber==" + logNumber) : answer;
				},
                getLogMessagesUrl: function (bizUnit, estimationId, logNumber) {
                    const answer = specifyBizUnitInUrl(logMessagesUrl, bizUnit).replace(estimationIdStub, estimationId);
                    return (logNumber) ? (answer + "?filter=logNumber==" + logNumber) : answer;
				},
				getMaxFiredTxnsSettingsUrl: function (bizUnit) {
					return specifyBizUnitInUrl(maxFiredTxnsSettingsUrl, bizUnit);
                },
                getQueueMetricsUrl: function (bizUnit, estimationId) {
                	return specifyBizUnitInUrl(queueMetricsUrl, bizUnit).replace(estimationIdStub, estimationId);
				},
                getRuleMetricsUrl: function (bizUnit, estimationId) {
                	return specifyBizUnitInUrl(ruleMetricsUrl, bizUnit).replace(estimationIdStub, estimationId);
				},
				getRulesUrl: function (bizUnit) {
					return specifyBizUnitInUrl(rulesUrlTemplate, bizUnit);
				},
				getRuleTransactionInfoUrl: function (bizUnit, ruleIds) {
					return specifyBizUnitInUrl(ruleTransactionInfoUrl, bizUnit) + ruleIds;
                },
                getRuleUrl: function (bizUnit, estimationId, ruleId) {
                	return specifyBizUnitInUrl(ruleUrl, bizUnit)
                			.replace(estimationIdStub, estimationId)
                			.replace("{ruleId}", ruleId);
				},
				getSplitTxnsOptionsUrl: function (bizUnit) {
					return specifyBizUnitInUrl(splitTxnsOptionsUrl, bizUnit);
                },
                getTimingMetricsUrl: function (bizUnit, estimationId) {
					return specifyBizUnitInUrl(timingMetricsUrl, bizUnit).replace(estimationIdStub, estimationId);
				},
			};
		} )();
		Object.freeze(UrlBuilder);

		const RequestType = {
			DELETE: "DELETE",
			GET: "GET",
			POST: "POST",
			PUT: "PUT",
		};
		Object.freeze(RequestType);

		const SelectionMode = {
			MULTIPLE: "multiple",
			SINGLE: "single",
		};
		Object.freeze(SelectionMode);

		const ButtonEnabler = (function (Selection) {
		                                           /* jshint -W123 */
			function ButtonEnabler (aMode, aFunc) {/* jshint +W123 */
				const selectionMode = aMode;
				const selectionFilter = aFunc;

				this.isMatchSelectionCount = function (selectedItemsCount) {
					return ((selectionMode === Selection.MULTIPLE) && (selectedItemsCount >= 1))
						|| ((selectionMode === Selection.SINGLE) && (selectedItemsCount === 1));
				};

				this.isToEnable = function (selectedItems) {
					const isMatchSelectionCount = this.isMatchSelectionCount(selectedItems.length);
					return isMatchSelectionCount
						? ((selectionFilter !== undefined) ? selectedItems.every(selectionFilter) : true)
						: false;
				};
			}
			return ButtonEnabler;
		} )(SelectionMode);

        // The sas.hc.m.MessageStrip constructor ignores the properties: showCloseButton, and width.
		// This wrapper supports those 2 properties.
        const MessageStripBuilder = function (properties) {
            const build = function () {
                const answer = new sas.hc.m.MessageStrip(properties);
                const isToHideCloseButton = !properties.showCloseButton;
                const isWidthSpecified = (properties.width !== undefined);

                if (isToHideCloseButton || isWidthSpecified) {
                    answer.addEventDelegate({
                        onAfterRendering: function (event) {
                            // fscf-overrideO: .sapMDialog .sapMText in fscf.css defined font-size: 0.88rem !important; line-height: 2rem !important;
                            const $messageStrip = event.srcControl.$();

                            if (isWidthSpecified) {
                            	$messageStrip.css({width: properties.width});
                            }
                            if (isToHideCloseButton) {
	                            $messageStrip.find("span.sapMText")[0].style.setProperty("line-height", "1rem", "important");
	                            // Hide the close button.  HC "hardcodes" it to make it impossible to remove it via the showCloseButton property or the setShowCloseButton() method.
	                            $messageStrip.find("button").css({display: "none"});
                            }
                        }
                    } );
                }
                return answer;
            };

            return {
                build: build,
            };
        };

        const TransactionsOptionsMap = function (options) {
            /*
             * @type {Map<string : Object.<string, {id: number, name: string, label: string}>>}
             * E.g. {"id": 200, "name": "aqo_sas_acct_rand_dig", "label": "Account"}
             */
            const map = new Map();

            const init = function () {
                options.forEach(function (option) {
                    map.set(option.name, option);
                } );
            };

            const getMap = function () {
                if (map.size === 0) {
                    init();
                }
                return map;
            };

            const getOptions = function () {
                return options;
            };

            const getOption = function (key) {
                return getMap().get(key);
            };

            return {
                getOption: getOption,
                getOptions: getOptions,
            };
        };

        const parseDate = function (aDateString, withZ) {
            // If aDateString has Z at the end, the resulting date/time has the GMT offset (from the local time zone) subtracted from it.
            // If the Z is removed, the date/time is the same as what is specified in aDateString.
            if (!aDateString) {
                return null;
            }
            const indexOfZ = aDateString.indexOf("Z");
            const includeZ = (withZ) ? !withZ : true;
            const dateString = (!includeZ && (indexOfZ === (aDateString.length -1)))
                ? aDateString.substring(0, aDateString.length-2)
                : aDateString;

            return new Date(dateString);
        };

        const OneDayInMilliseconds = 24 * 60 * 60 * 1000; // hours*minutes*seconds*milliseconds.replace(estimationIdStub, estimationId);

		const EstimationDateRanger = (function () {
            const DEFAULT_NUMBER_OF_DAYS_TO_SELECT = 7;
									                      /* jshint -W123 */
            function EstimationDateRanger (ruleTrxnInfo) {/* jshint +W123 */
                let defaultStartDate = null;
				const earliestTrxnDate = parseDate(ruleTrxnInfo.earliestPossibleTrxDate);
                const recommendedTrxnDate = parseDate(ruleTrxnInfo.earliestRecommendedTrxDate);

                const determineStartDate = function () {
                    const now = new Date();
                    // The end date/time is the day after today at hours 0.
                    const endDate = new Date(now.getTime());
                    endDate.setDate(now.getDate() + 1);
                    endDate.setHours(0, 0, 0, 0);
                    // Substract the timezone offset to make the date/time local.
                    endDate.setHours(endDate.getHours() - (endDate.getTimezoneOffset() / 60)); // S1494648

                    const startDateCandidate = (recommendedTrxnDate !== null) ? recommendedTrxnDate : earliestTrxnDate;
                    const diff  = (endDate - startDateCandidate);
                    const diffDays = diff / OneDayInMilliseconds;
                    const diffDaysCeiling = Math.ceil(diffDays);
                    const daysToSelect = Math.min(diffDaysCeiling, DEFAULT_NUMBER_OF_DAYS_TO_SELECT);

                    const startDate = new Date();
                    startDate.setDate(now.getDate() - daysToSelect + 1); // + 1 because the days to select include today.
                    // It should not happen...but do make sure that the recommended start date is NOT before earliestTrxDate! // S1494648
                    const answer = (startDate < earliestTrxnDate) ? earliestTrxnDate : startDate;

                    return (recommendedTrxnDate && (answer < recommendedTrxnDate)) ? recommendedTrxnDate : answer;
                };

                this.getDefaultStartDate = function () {
                    if (defaultStartDate === null) {
                        defaultStartDate = determineStartDate();
                    }
                    return defaultStartDate;
                };

				this.getEarliestTrxnDate = function () {return earliestTrxnDate;};

				this.getRecommendedTrxnDate = function () {return recommendedTrxnDate;};
			}

			return EstimationDateRanger;
		}() );

        const getText = function () {
        	return sas.fscf.rb.getText.apply(sas.fscf.rb, Array.prototype.slice.call(arguments));
        };

		const newCloseButton = function () {
			return new sas.hc.m.Button({
				text: getText("fcm.application.close.txt"),
				press: function (event) {
					event.getSource().getParent().close();
				}
			} );
		};

		const toLogError = true;
        const logError = function (message, detail, estimation) {
        	if (!toLogError) {
        		return;
        	}

        	const estimationInfo = (estimation)
        		? "Estimation \"" + estimation.name + "\" (ID " + estimation.id + ")"  // Log info...no localization needed.
        		: null;
        	const errorInfo = [message, estimationInfo, detail].reduce(function (answer, additionalInfo) {
        		if (additionalInfo) {
        			answer += "\n" + additionalInfo;
        		}
        		return answer;
        	} );
        	jQuery.sap.log.error(errorInfo);
        };

        const RuleHelper = (function () {
            const getRuleDisplayIdString = function (rule) {
                return (rule.baseRuleId && rule.revision)
                    ? rule.baseRuleId + "." + rule.revision
                    : "";
            };

            const getRuleDisplayId = function (rule) {
                const displayIdString = !rule.displayIdString ? getRuleDisplayIdString(rule) : rule.displayIdString;
                const displayId = parseFloat(displayIdString);

                return Number.isNaN(displayId) ? 0 : displayId;
            };

            const setDisplayId = function (rule) {
                if (!rule.displayIdString) {
                    rule.displayIdString = getRuleDisplayIdString(rule);
                }
                rule.displayId = getRuleDisplayId(rule);
            };

            // Sort rules by display ID in ascending order.
            const displayIdComparator = function(rule1, rule2) {
                const displayId1 = rule1.displayId;
                const displayId2 = rule2.displayId;
                return (displayId1 < displayId2)
                    ? -1
                    : (displayId1 > displayId2) ? 1 : 0;
            };

            return {
                setDisplayId: setDisplayId,
                displayIdComparator: displayIdComparator,
            };
        } )();

        const waitThenDo = function (condition, callback, timesToTry) {
            const limit = timesToTry ? timesToTry : 30;

            function doWaitThenDo (aCondition, aCallback, counter) {
                if (aCondition()) {
                    aCallback();
                } else {
                    if (counter <= limit) {                                                         /* jshint -W117 */ // W117 complains about setTimeout() not defined
                        setTimeout(doWaitThenDo.bind(null, aCondition, aCallback, counter+1), 100); /* jshint +W117 */
                    }
                }
            }
            doWaitThenDo(condition, callback, 1);
        };

        // https://stackoverflow.com/a/40724354
        // https://en.wikipedia.org/wiki/International_System_of_Units
        const NumberAbbreviator = (function () {
            const SI_SYMBOLS = ["", "k", "M", "G", "T", "P", "E"];

            const format = function (number, decimalPlaces) {
                const tier = Math.trunc(Math.log10(number) / 3);
                // If zero, i.e. for any number < 1000, we don't need a suffix.
                if ((tier === 0) || (tier === Number.NEGATIVE_INFINITY)) {
                    return number;
                }

                // Get suffix and determine scale.
                const suffix = SI_SYMBOLS[tier];
                const scale = Math.pow(10, tier * 3);
                // scale the number
                const scaled = number / scale;
                // format number and add suffix
                const decimals = (decimalPlaces) ? decimalPlaces : 0;
                return scaled.toFixed(decimals) + suffix;
            };

            return {
                format: format,
            };
        } )();

        const getLocalizedRuleState = function (state) {
            return (state)
                ? getText("rules.rulesPage.rules.states." + state.trim().toLowerCase() + ".txt")
                : "";
        };

        const injectHtmlIntoIframe = function (iframeElt, somHtml) { // someHtml is expected to include both the <head> and <body> tags
            const html = "<html>\n" + somHtml + "\n</html>";

            // Obtain the document associated with the iframe tag.
            // Most of the browser supports .document. Some supports (such as the NetScape series) .contentDocumet,
            // while some (e.g. IE5/6) supports .contentWindow.document we try to read whatever that exists.
            let iframeDoc = iframeElt.document;
            if (iframeElt.contentDocument) {
                iframeDoc = iframeElt.contentDocument;
            } else if (iframeElt.contentWindow) {
                iframeDoc = iframeElt.contentWindow.document;
            }
            if (iframeDoc) {
                // Put the content in the iframe
                iframeDoc.open();
                iframeDoc.writeln(html);
                iframeDoc.close();
            }
        };

        const getGmtOffsetString = function (aDate) {
            const date = aDate ? aDate : new Date();

            // TODO: figure out a way so that we don't have to prepend the minus sign below!
            return "GMT " + -date.getTimezoneOffset() / 60;
        };

        const formatGMTOffsetString = function (val) {
            if (val === undefined || val === null) {
                return val;
            }
            const date = new Date(val);
            if (Number.isNaN(date.getTime())) {
                return val;
            }
            return getGmtOffsetString(date);
        };

        const getAcceptLanguage = function () {  /* jshint -W117 */
            if (navigator.languages && navigator.languages.length) {
                return navigator.languages[0];
            } else {
                return navigator.userLanguage || navigator.language || navigator.browserLanguage || "en";
            }
        };                                       /* jshint +W117 */

        const LONG_DATE_OPTIONS = {year: "numeric", month: "long", day: "numeric"};

        const getLocale = function () {
            return (sap)
                ? sap.ui.getCore().getConfiguration().getLanguage()
                : getAcceptLanguage();
        };

        const Formatter = (function () {   /* jshint -W123 */

            function Formatter (aLocale) { /* jshint +W123 */
                const locale = (aLocale) ? aLocale : getLocale();

                this.getLocale = function () {
                    return locale;
                };
            }

            Formatter.prototype.formatLongDate = function (date) {
                return date.toLocaleString(this.getLocale(), LONG_DATE_OPTIONS);
            };

            Formatter.prototype.formatWebappDate = function (date, UTC) {
                return sas.fscf.dateFormatter.format(date, UTC);             // UTC true or false
            };

            Formatter.prototype.formatNumber = function (number) {
                return number.toLocaleString(this.getLocale());
            };

			return Formatter;
        }() );

        let formatter =  null;

        const getFormatter = function () {
            if (formatter ===  null) {
                formatter = new Formatter();
            }
            return formatter;
        };

        const getContextPath = function () {                                                       /* jshint -W117 */
            return window.location.pathname.substring(0, window.location.pathname.indexOf("/",2)); /* jshint +W117 */
        };

        const downloadFile = function (uri) {                                                                            /* jshint -W117 */
            window.open(window.location.protocol + "//" + window.location.host + getContextPath() + "/" + uri, "_self"); /* jshint +W117 */
        };

        /* BEGIN: polyfills for IE 11 */
        const parseInteger = Number.parseInt ? Number.parseInt : parseInt; // IE11 doesn't support Number.parseInt

        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Number/isInteger#Polyfill
        if (!Number.isInteger) {
        	Number.isInteger = function (value) {
        		return (typeof value === "number") && isFinite(value) && (Math.floor(value) === value);
        	};
        }

        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Number/isNaN#Polyfill
        if (!Number.isNaN) {
            Number.isNaN = Number.isNaN || function(value) {
                return value !== value;
            };
        }

        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object/entries#Polyfill
		if (!Object.entries) {
			Object.entries = function(obj) {
                const ownProps = Object.keys(obj);
                let i = ownProps.length;
				const answer = new Array(i); // preallocate the Array
				while (i--) {
					answer[i] = [ ownProps[i], obj[ownProps[i]] ];
				}
				return answer;
			};
		}

        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Math/log10#Polyfill
		if (!Math.log10) {
			Math.log10 = function(x) {
                return Math.log(x) * Math.LOG10E;
            };
		}

        // MDN https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Math/trunc#Polyfill
		if (!Math.trunc) {
			Math.trunc = function(v) {
				v = +v;
				return (v - v % 1)   ||   (!isFinite(v) || v === 0 ? v : v < 0 ? -0 : 0);
			};
        }

        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/includes#Polyfill
        if (!String.prototype.includes) {                         /* jshint -W121 */
            String.prototype.includes = function(search, start) { /* jshint -W121 */
                if (typeof start !== 'number') {
                    start = 0;
                }

                if (start + search.length > this.length) {
                    return false;
                } else {
                    return this.indexOf(search, start) !== -1;
                }
            };
        }
        /* END: polyfills for IE 11 */

	    return {
            ButtonEnabler : ButtonEnabler,
            EstimationDateRanger: EstimationDateRanger,
            IdFactory: IdFactory,
            MessageStripBuilder: MessageStripBuilder,
            MessageType: MessageType,
            NullCheck: NullCheck,
            NumberAbbreviator: NumberAbbreviator,
            RequestType: RequestType,
            RuleHelper: RuleHelper,
			SelectionMode: SelectionMode,
            SpacerFactory: SpacerFactory,
            TransactionsOptionsMap: TransactionsOptionsMap,
			UrlBuilder: UrlBuilder,
			Wizard: Wizard,

            downloadFile: downloadFile,
            formatDateTime: formatDateTime,
            formatDateTimeEcma262: formatDateTimeEcma262,
            formatGMTOffsetString: formatGMTOffsetString,
            getAcceptLanguage: getAcceptLanguage,
            getFormatter: getFormatter,
            getGmtOffsetString: getGmtOffsetString,
            getLocale: getLocale,
            getLocalizedRuleState: getLocalizedRuleState,
            getText: getText,
            injectHtmlIntoIframe: injectHtmlIntoIframe,
			logError: logError,
            newCloseButton: newCloseButton,
            newText: newText,
            parseDate: parseDate,
            parseInteger: parseInteger,
            waitThenDo: waitThenDo,
	    };
	} )(),
	true
);
