jQuery.sap.require("sas.fscf.views.estimation.EstimationHelper");
jQuery.sap.require("sas.fscf.views.estimation.EstimationController");

"use strict";
const EstimationHelper = sas.fscf.views.estimation.EstimationHelper;
const EstimationDateRanger = EstimationHelper.EstimationDateRanger;
const RequestType = EstimationHelper.RequestType;
const RuleHelper = EstimationHelper.RuleHelper;
const TransactionsOptionsMap = EstimationHelper.TransactionsOptionsMap;
const UrlBuilder = EstimationHelper.UrlBuilder;
const getText = EstimationHelper.getText;
const logError = EstimationHelper.logError;

const INVALID_TEXT_CHARACTERS = ["#", "%", "&", "<", ">", ","];
const INVALID_TEXT_INPUT_KEY_CODES = [35, 37, 38, 44, 60, 62]; // # % & < > ,
const DefaultMaxFiredTxnsSettings = {
		defaultCount: 1000,
		maxCount: 50000,
};

const OpCode = { // texts in this object aren't user-facing.
	CANCEL: "cancel",
	COPY: "copy",
	DELETE: "delete",
	VIEW_PROPERTIES: "viewProperties",
	VIEW_RESULTS: "viewResults",
};
Object.freeze && Object.freeze(OpCode);

const RuleType = { // texts in this object aren't user-facing.
	AUTH: "Authorization",
	QUEUE: "Queue",
	VARIABLE: "Variable",
};
Object.freeze && Object.freeze(RuleType);

// BEGIN: RAPTOR-5329
const getForceUserVarsCal = function () {
	return sas.fscf.estimationForceUserVarsCal;
};

const setForceUserVarsCal = function (value) {
	sas.fscf.estimationForceUserVarsCal = value;
};

const getSystemProperty = function (propertyId, callback) {
	const url = "rest/systemProperties/systemProperty?systemPropertyId=" + propertyId; 
	const systemPropertyCall = sas.ajax({
			async: true,
			cache: false, 			
			url: url,
			dataType: "json",
			contentType:"application/json; charset=utf-8",
			headers:sas.fscf.getAjaxHeaders()
	});   
	
	$.when(systemPropertyCall).done(function (systemPropertyData) {    		
		let value = systemPropertyData.value.toLowerCase();
		
		if ((value === "true") || (value === "false")) {
			value = (value === "true");
		}
		
		if (callback) {
			callback(value);
		}
	});
	$.when(systemPropertyCall).fail(function (oObj, textStatus, errorThrown) {
		if (callback) {
			callback(null);
		}
	});
};

const FORCE_USER_VARS_CAL = "estimation_enable_force_calculation";
const getForceUserVarsCalProperty = function (callback) {
	getSystemProperty(FORCE_USER_VARS_CAL, callback);
};

if (getForceUserVarsCal() === undefined) {
	getForceUserVarsCalProperty(setForceUserVarsCal);
};
// END: RAPTOR-5329

sap.ui.controller("sas.fscf.views.RulesPageEstimations", {
	ESTIMATION_ALL: "all",
	ESTIMATION_USER: "user",
	LAST_ESTIMATION_FILTER_COOKIE_NAME: "cf_lastEstimationsFilter",
	REQUEST_TYPE_DELETE: "DELETE",

	rulesModel: null,

	/*** BEGIN: Estimation Dialog ***/
	estimationDialog: null,
	maxFiredTxnsSettings: null,
	selectedRulesCache: [],
	slowPathDateRange: null, // sap.ui.unified.DateRange

	createEstimationDialog: function () {
		return sap.ui.jsfragment((this.getView().getId() + "EstimationDialog"),  // 3 params: id, view name, controller
				"sas.fscf.views.estimation.NewEstimationDialog", this );
	},

	getDialogNameInput: function () {
		return this.estimationDialog.getNameInput();
	},

	getDialogRuleTable: function () {
		return this.estimationDialog.getRuleTable();
	},

	// Return a sap.ui.unified.DateRange, or null, if no date is selected.
	getDialogSelectedDateRange: function () {
		const selectedDates = this.estimationDialog.getCalendar().getSelectedDates();
        return (selectedDates && (selectedDates.length > 0)) ? selectedDates[0] : null;
	},

	getDialogSelectedRules: function () {
		const ruleTable = this.getDialogRuleTable();
		return ruleTable.getSelectedIndices().map(function (each) {
			const context = ruleTable.getContextByIndex(each);
			const path = context.sPath;

			return this.rulesModel.getProperty(path);
		}, this );
	},

	getDialogSelectedRuleIds: function () {
		return this.getDialogSelectedRules().map(function (each) {
			return each.id;
		} );
	},

	getNameKeyInputValidator: function (event) {
		if ($.inArray(event.which, INVALID_TEXT_INPUT_KEY_CODES) >= 0) {
			this.handleEstimationNameInputError(event);
		}
    },

    getNameValidator: function (event) {
		const text = this.getDialogNameInput().getValue();
    	const isInvalidCharFound = INVALID_TEXT_CHARACTERS.filter(function (each) {
    		return text.indexOf(each) >= 0;
    	} );

    	if (isInvalidCharFound) {
			this.handleEstimationNameInputError(event);
    	}
    },

	getRulesForNewEstimation: function () {
		return this.rulesModel;
	},

	handleEstimationDialogChanged: function () {
		// Need to check if estimationDialog is null, because during the instantiation of the Calendar
		// it calls handleEstimationDialogCalendarSelect, which in turn calls this, before the Estimation
		// Dialog is instantiated.
		if (this.estimationDialog) {
			this.estimationDialog.setUpButtons();
		}
	},

	handleEstimateionDialogBackButtonClick: function () {
		this.getEstimationDialogWizard().back();
		this.estimationDialog.setUpButtons();
	},

	clearCache: function () {
		this.selectedRulesCache = [];
		this.slowPathDateRange = null;
	},

	handleEstimationDialogBeforeOpen: function () {
		this.clearCache();
		this.estimationDialog.setUpButtons();
	},

	handleEstimationDialogClose: function () {
		this.estimationDialogWizard = null;
	},

	ruleTransactionInfo: null,
	toGetRuleTransactionInfo: true,

	isToGetRulesTransactionInfo: function () {
		return this.toGetRuleTransactionInfo;
	},

	handleEstimateionDialogNextButtonClick: function () {
		// See if we are going from the General Tab to the Date Range Tab.  If so, we need to make an
		// ajax call to get transaction info based on the rules selected.
		if (this.getEstimationDialogWizard().getCurrentIndex() === 0) {
			// Sort the selected rules by their ID in ascending order.
			const selectedRules = this.getDialogSelectedRules();
			selectedRules.sort(function (a, b) {
				return a.id - b.id;
			} );
			// Check with the rules the user has selected are the same as those cached (i.e. the ones he selected previously).
			// Or if the rules are the same, but the toGetRuleTransactionInfo flag is set to true.
			if (!this.isSelectedRulesSame(selectedRules) || this.isToGetRulesTransactionInfo()) {
				this.selectedRulesCache = selectedRules;
				this.doGetRulesTransactionInfo();
				return;
			}
		} else {
			this.setUpMaxFiredTxnsInput();
		}
		this.doMoveToNextTab();
	},

	// Return true if the selected rules cached are the same has those in selectedRules.
	// Rules in selectedRules are sorted by their id in ascending order.
	isSelectedRulesSame: function (selectedRules) {
		const currentSelectedRules = this.selectedRulesCache;
		if (currentSelectedRules.length === selectedRules.length) {
			return currentSelectedRules.every(function (each, index) {
				return each.id === selectedRules[index].id;
			} );
		} else {
			return false;
		}
	},

	setUpMaxFiredTxnsInput: function () {
		if (this.maxFiredTxnsSettings !== null) {
			this.doSetUpMaxFiredTxnsInput(this.maxFiredTxnsSettings);
		} else {
			const url = UrlBuilder.getMaxFiredTxnsSettingsUrl(this.getBusinessUnitId());
			const ajaxParams = {
				url: url,
				type: RequestType.GET,
				headers:sas.fscf.getAjaxHeaders()
			};
			const self = this;
			sas.ajax(ajaxParams)
			  	.done(function (response) {
			  		self.maxFiredTxnsSettings = response;
					self.doSetUpMaxFiredTxnsInput(self.maxFiredTxnsSettings);
				} )
				.fail(function () {
					self.maxFiredTxnsSettings = DefaultMaxFiredTxnsSettings;
					self.doSetUpMaxFiredTxnsInput(self.maxFiredTxnsSettings);
				} );
		}
	},

	doSetUpMaxFiredTxnsInput: function (maxFiredTxnsSettings) {
		this.estimationDialog.setUpMaxFiredTxnsInput(maxFiredTxnsSettings);
	},

	determineFastPathDateRange: function (recommendedTrxDate) {
		let answer = null;
		if (recommendedTrxDate !== null) {
			answer = new sap.ui.unified.DateRange({
				startDate: recommendedTrxDate,
				endDate: new Date(),
			} );
		}
		return answer;
	},

	determineSlowPathDateRange: function (recommendedTrxDate, earliestTrxDate) {
		let answer = null;

		if (earliestTrxDate !== null) {
			let slowPathEndDate = null;

			if (recommendedTrxDate === null) {
				slowPathEndDate = new Date();
			} else {
				if ((recommendedTrxDate - earliestTrxDate) > 0) {
					const recommendedTrxDateMinus1Day = new Date(recommendedTrxDate);
					recommendedTrxDateMinus1Day.setDate(recommendedTrxDate.getDate() - 1);
					slowPathEndDate = recommendedTrxDateMinus1Day;
				}
			}
			if (slowPathEndDate !== null) {
				answer = new sap.ui.unified.DateRange({
					startDate: earliestTrxDate,
					endDate: slowPathEndDate,
				} );
			}
		}
		return answer;
	},

	// Answer the date range to be selected by default.
    determineDefaultDateRange: function (ruleTrxnInfo) {
        const dateRanger = new EstimationDateRanger(ruleTrxnInfo);
        const defaultStartDate = dateRanger.getDefaultStartDate();

        return new sap.ui.unified.DateRange({
				startDate: defaultStartDate,
				endDate: new Date(),
		} );
	},

	// eg. "2018-02-01T00:00:00Z"
	parseDate: function (dateString) {
		return (dateString !== undefined)
				? new Date(dateString.substring(0, dateString.indexOf("T")).replace(/-/g, '\/'))
				: null;
	},

	handleNoTransactions: function () {
        this.estimationDialog.showWarningMessage(getText("rules.esimtation.no.transactions.info.msg"));
		this.estimationDialog.getNextButton().setEnabled(false);
		this.toGetRuleTransactionInfo = true;
	},

	doHandleEstimateionDialogNextButtonClick: function (rulesTrxInfo) {
		if (rulesTrxInfo) {
			if (!rulesTrxInfo.transactionsPresent) {
				this.handleNoTransactions();
				return;
			}

			const estimationDialog = this.estimationDialog;
			estimationDialog.purgeDateRangeInfo();

			const earliestTrxDate = this.parseDate(rulesTrxInfo.earliestPossibleTrxDate);
            const recommendedTrxDateSuggested = this.parseDate(rulesTrxInfo.earliestRecommendedTrxDate);
            const recommendedTrxDate = recommendedTrxDateSuggested
                ?  (recommendedTrxDateSuggested < earliestTrxDate) ? earliestTrxDate : recommendedTrxDateSuggested // S1494648
                : null;
            const defaultDateRange = this.determineDefaultDateRange(rulesTrxInfo);
			const fastPathDateRange = this.determineFastPathDateRange(recommendedTrxDate);
			const slowPathDateRange = this.determineSlowPathDateRange(recommendedTrxDate, earliestTrxDate);

			this.slowPathDateRange = slowPathDateRange;
			estimationDialog.setUpCalendar(earliestTrxDate, defaultDateRange, fastPathDateRange, slowPathDateRange);
			estimationDialog.focusToday();

			this.doMoveToNextTab();
		}
	},

	doMoveToNextTab: function () {
		this.getEstimationDialogWizard().next();
		this.estimationDialog.setUpButtons();
	},

    setUpRuleTransactionInfo: function (info) {
        info.isCanForceUserVarsRecalculation = info.rulesReferenceNonRoiRules || info.rulesReferenceUserVariables;
        return this;
    },

	doGetRulesTransactionInfo: function () {
		// RAPTOR-5313 next 3 lines
        const estimationDialog = this.estimationDialog;
        estimationDialog.setBusyIndicatorDelay(0).setBusy(true);
        estimationDialog.getNextButton().setEnabled(false);

		sas.fscf.removeAllMessageStrips();

		const ruleIds = this.getDialogSelectedRuleIds();
		const url = UrlBuilder.getRuleTransactionInfoUrl(this.getBusinessUnitId(), ruleIds);
		const ajaxParams = {
			url: url,
			type: RequestType.GET,
			headers:sas.fscf.getAjaxHeaders()
		};
		const self = this;

		sas.ajax(ajaxParams)
		  	.done(function (response) {
                self.ruleTransactionInfo = response;
                self.setUpRuleTransactionInfo(response);
				self.toGetRuleTransactionInfo = false;
				self.doHandleEstimateionDialogNextButtonClick(response);
		  	} )
			.fail(function (response) {
                const message = getText("rules.estimation.retrieve.rules.transaction.info.for.create.error.msg");
                self.estimationDialog.showErrorMessage(message)
                    .setBusy(false);
                const errorDetail = (response.responseJSON) ? response.responseJSON.message : null;
                logError(message, errorDetail);
            })   
            // RAPTOR-5313 next 3 lines
            .always(function () {
        		estimationDialog.setBusy(false);
            });
	},

	estimationDialogWizard: null,
	getEstimationDialogWizard: function () {
		if (this.estimationDialogWizard === null) {
			this.estimationDialogWizard = new sas.fscf.views.estimation.EstimationHelper.Wizard(this.estimationDialog);
		}
		return this.estimationDialogWizard;
	},

	handleEstimationNameInputError: function (event) {
		event.preventDefault();
		//let nameInput = this.getDialogNameInput();
		this.getDialogNameInput()
			.setValueState(sap.ui.core.ValueState.Error)
			.setValueStateText(getText("rules.rulesPage.specialCharacters.error.txt"));
	},

	handleEstimationRuleSelectionChange: function () {
		this.toGetRuleTransactionInfo = true;

		this.handleEstimationDialogChanged();
	},

	handleEstimationDialogCalendarSelect: function () {
		const estimationDialog = this.estimationDialog;
		let toCheckUserVarsCalCheckBox = false;
		let toEnableUserVarsElements = false;
		let toNameCheckBoxEnableCalculation = true;
		let toMakeUserVarsCalCheckBoxEditable = false;
		let toEnableUserVarsCalculationCheckBox = false;
		let toEnableUserVarsLabel = false;
		let toExcludeTestingRulesCheckBox = false;
		const isCanForceUserVarsRecalculation = // RAPTOR-5329
			this.ruleTransactionInfo.isCanForceUserVarsRecalculation || getForceUserVarsCal();

		if (this.isDateRangeOverlapsSlowPathDateRange()) {
			estimationDialog.showSlowPathWarning();
			toCheckUserVarsCalCheckBox = true;
			toEnableUserVarsCalculationCheckBox = true;
			toEnableUserVarsElements = true;
		} else {
			if (isCanForceUserVarsRecalculation) {
				toMakeUserVarsCalCheckBoxEditable = true;
				toEnableUserVarsCalculationCheckBox = true;
				toNameCheckBoxEnableCalculation = false;
				toEnableUserVarsLabel = true;
            }
            estimationDialog.purgeDateRangeInfo();
		}
		const userExcludeTestingrulesCheckBox=estimationDialog.getUserUseTestRulesCheckBox();
		userExcludeTestingrulesCheckBox.setSelected(toExcludeTestingRulesCheckBox);
		const userExcludeTestingCheckBoxProperties = {
				readOnly: false,
				enable: true,
			};
		estimationDialog.setCheckboxInteractivity(userExcludeTestingrulesCheckBox, userExcludeTestingCheckBoxProperties);

		const userVarsCalculationCheckBox = estimationDialog.getUserVarsCalculationCheckBox();
		userVarsCalculationCheckBox.setSelected(toCheckUserVarsCalCheckBox);
		const userVarsCalculationCheckBoxProperties = {
				readOnly: !toMakeUserVarsCalCheckBoxEditable,
				enable: toEnableUserVarsCalculationCheckBox,
			};
		estimationDialog.setCheckboxInteractivity(userVarsCalculationCheckBox, userVarsCalculationCheckBoxProperties);
		
		// RAPTOR-5329 When applicable, enable the Force recalculation checkbox and check it by default.
		// Also, enable user variables calculation input elements accordingly.
		if (getForceUserVarsCal()) {
			userVarsCalculationCheckBox.setSelected(true);
			toEnableUserVarsElements = true;
		}
		estimationDialog.enableUserVarsElements(toEnableUserVarsElements);
		if (isCanForceUserVarsRecalculation) {
			estimationDialog.getUserVarsLabel().setEnabled(toEnableUserVarsLabel);
		}
		toNameCheckBoxEnableCalculation
			? estimationDialog.nameUserVarsCalculationCheckBoxEnableCalculation()
			: estimationDialog.nameUserVarsCalculationCheckBoxForceRecalculation();
		this.handleEstimationDialogChanged();
	},

	isDateRangeOverlapsSlowPathDateRange: function () {
		const selectedDateRange = this.getDialogSelectedDateRange();
		if (selectedDateRange !== null) {
			const selectedStartDate = selectedDateRange.getStartDate();
			if ((selectedStartDate !== null) && (this.slowPathDateRange !== null)) {
				return this.isDateWithinDateRange(selectedStartDate, this.slowPathDateRange);
			}
		}
		return false;
	},

	// @param date - Date
	// @param targetDateRange - sap.ui.unified.DateRange, both startDate and endDate are non-null
	isDateWithinDateRange: function (date, targetDateRange) {
		if (date !== null) {
			const dateMilli = date.getTime();
			return (dateMilli >= targetDateRange.getStartDate().getTime())
				&& (dateMilli <= targetDateRange.getEndDate().getTime());
		} else {
			return false;
		}
	},

	isControlInErrorState: function (control) {
		return (control.getValueState() === sap.ui.core.ValueState.Error);
	},

	isEstimationNameSpecified: function () {
		const nameInput = this.getDialogNameInput();
		return !this.isControlInErrorState(nameInput)
			&& (nameInput.getValue().length > 0);
	},

	// Answer true if at least one rule has been selected.
	isRuleSelected: function () {
		return this.getDialogRuleTable().getSelectedIndices().length > 0;
	},

	isDateRangeSpecified: function () {
		const selectedDateRange = this.getDialogSelectedDateRange();
        return (selectedDateRange !== null)
        	&& (selectedDateRange.getStartDate() !== null)
        	&& (selectedDateRange.getEndDate() !== null);
	},

	// Answer true if the dialog has all the required data in order to create/run a new estimation.
	isCanCreateEstimation: function () {
		return this.isEstimationNameSpecified()
			&& this.isRuleSelected()
			&& this.isDateRangeSpecified()
			&& this.estimationDialog.isMaxFiredTxnsInputValid()
			&& this.estimationDialog.isUserVarsPrimingDaysDurationValid();
	},

	isCanGoToNextTab: function () {
		switch (this.getEstimationDialogWizard().getCurrentIndex()) {
			case 0:		// General tab
				return this.isEstimationNameSpecified() && this.isRuleSelected();
			case 1:		// Date Range tab
				return this.isDateRangeSpecified();
			default:	// Additional Settings tab
				return false;
		}
	},

	isCanGoToPreviousTab: function () {
		return this.getEstimationDialogWizard().isCanGoBack();
	},
	/*** END: New Estimation Dialog ***/

	/*** BEGIN: Estimation List ***/
	cancelEstimation: function (estimation) {
		if (this.getView().confirmEstimationCancel(estimation, this.doCancelEstimation)) {
			this.doCancelEstimation(estimation);
		}
	},

	doCancelEstimation: function (estimation) {
        sas.fscf.removeAllMessageStrips();

		const myView = this.getView();
		sas.fscf.getApplicationView().setBusyIndicatorDelay(0).setBusy(true);
		const url = UrlBuilder.getCancelEstimationUrl(this.getBusinessUnitId(), estimation.id);
		const ajaxParams = {
			url: url,
			type: RequestType.PUT,
			headers:sas.fscf.getAjaxHeaders()
		};
		const self = this;
   		sas.ajax(ajaxParams)
   			.done(function (cancelEsimtationResult) {
   				if (cancelEsimtationResult.cancelled) {
                    self.updateTable();
                    myView.showSuccessMessage(getText("rules.esimtation.canceled.success.info.fmt.msg", [estimation.name, estimation.id]));
   				} else {
					self.updateTable();
                    myView.showWarningMessage(cancelEsimtationResult.message);
   				}
   			} )
   			.fail(function (response) {
                const message = getText("rules.estimation.cancel.error.fmt.msg", [estimation.name, estimation.id]);
                myView.showErrorMessage(message);
                const errorDetail = (response.responseJSON) ? response.responseJSON.message : null;
                logError(message, errorDetail, estimation);
   			} )
            .always(function () {
            	sas.fscf.getApplicationView().setBusy(false);
            } );
        },

	copyEstimation: function (estimation) {
		this.getView().info("Duplicate Estmiation - \"" + estimation.name, "\"\n... to be implemented.");
	},

	deleteEstimations: function (callback) {
        const deleteEstimationsCallback = this.doDeleteEstimations.bind(this, this.getSelectedEstimations(), callback);
		this.getView().confirmEstimationsDelete(deleteEstimationsCallback);
	},

	doDeleteEstimations: function (estimations, callback) {
        sas.fscf.removeAllMessageStrips();
        sas.fscf.getApplicationView().setBusyIndicatorDelay(0).setBusy(true);

        const estimation = (estimations.length > 1) ? null : estimations[0];
        const estimationIds = estimations.reduce(function (accumulator, each) {
            if (each) {
                accumulator.push({id: each.id});
            }
            return accumulator;
        }, [] );

		const url = UrlBuilder.getDeleteEstimationsUrl(this.getBusinessUnitId());
		const ajaxParams = {
			url: url,
            type: RequestType.DELETE,
            data: estimationIds,
            headers:sas.fscf.getAjaxHeaders()
		};
		const self = this;
   		sas.ajax(ajaxParams)
   			.done(function () {
                if (callback) {
                    callback();
                }
				self.updateTable();
                const message = (estimation)
                    ? getText("rules.esimtation.single.delete.success.info.fmt.msg", [estimation.name, estimation.id])
                    : getText("rules.esimtation.multiple.deletes.success.info.msg");
                self.getView().showSuccessMessage(message);
   			} )
   			.fail(function (response) {
                const message = (estimation)
                    ? getText("rules.estimation.single.delete.error.fmt.msg", [estimation.name, estimation.id])
                    : getText("rules.estimation.multiple.deletes.error.msg");
                self.getView().showErrorMessage(message);
                const errorDetail = (response.responseJSON) ? response.responseJSON.message : null;
                logError(message, errorDetail);
            } )
            .always(function () {
            	sas.fscf.getApplicationView().setBusy(false);
            } );
	},

	getBusinessUnitId: function () {
		// This following doesn't work when Estimations are to be displayed before any Rules have been displayed.
		// sas.fscf.getCurrentPageView() is undefined in such situation.
		//sas.fscf.getCurrentPageView().getBUId()

        // For now, use the code below from getBUId() in RulesPageRules.view.js.
	    return sas.fscf.getCookie("cf_lastRulesBU");
	},

	getLastEstimationsFilterCookie: function() {
		return sas.fscf.getCookie(this.LAST_ESTIMATION_FILTER_COOKIE_NAME);
	},

	getOpCode: function () {
		return OpCode;
	},

	getTableUrl: function(searchTerm) {
		return UrlBuilder.getEstimationsUrl(this.getBusinessUnitId(), this.isShowAllEstimations(), searchTerm);
	},

	showError: function (title, text, detail) {
		this.getView().error(title, text, detail);
    },

	// @date - Date, e.g. Thu Mar 28 2019 08:59:03 GMT-0400 (Eastern Daylight Time), take the date
	// @time - Date, which contains the time that the user specified.
    //               e.g. Thu Jan 01 1970 23:59:00 GMT-0500 (Eastern Standard Time), take the time, and merge the date with the time
	// @return date, e.g. Thu Mar 28 2019 23:59:37 GMT-0400 (Eastern Daylight Time)
	mergeDateAndTime: function (date, time) {
		date.setHours(time.getHours());
		date.setMinutes(time.getMinutes());
	},

	getEstimationParams: function (estimationDialog) {
		const dateRange = this.getDialogSelectedDateRange();
		const endDate = dateRange.getEndDate();                                                // local date...e.g. Thu Mar 28 2019 08:59:03 GMT-0400 (Eastern Daylight Time)
		this.mergeDateAndTime(endDate, estimationDialog.getEndTimePicker().getDateValue());    // local time...e.g. Thu Jan 01 1970 23:59:00 GMT-0500 (Eastern Standard Time)
		const startDate = dateRange.getStartDate();
        this.mergeDateAndTime(startDate, estimationDialog.getStartTimePicker().getDateValue());// local date...e.g. Fri Mar 22 2019 09:14:37 GMT-0400 (Eastern Daylight Time)

		const parallelizeBy = estimationDialog.getParallelProcessingCheckBox().getSelected()   // local time...e.g. Thu Jan 01 1970 00:00:00 GMT-0500 (Eastern Standard Time)
			? estimationDialog.getSplitTxnsOptionsSelect().getSelectedKey()
            : null;
		
		const isExcludeTestingRules=estimationDialog.getUserUseTestRulesCheckBox().getSelected();
		
		const rules = [];
		this.getDialogSelectedRules().forEach(function (each) {
			rules.push({
				id: each.id,
				lastUpdateNumber: each.lastUpdateNumber,
			} );
        } );

        const isUserVarsCalculationChecked = estimationDialog.getUserVarsCalculationCheckBox().getSelected();
        const enableUserVariablesCalculation = this.ruleTransactionInfo.isCanForceUserVarsRecalculation
                ? isUserVarsCalculationChecked
                : null;
        const primingDuration = isUserVarsCalculationChecked ? estimationDialog.getUserVarsPrimingDurationInput().getValue() : 0;

		return {
			toExcludeTestingRules: isExcludeTestingRules,
			enableUserVariablesCalculation: enableUserVariablesCalculation,
			endGmtDateString: endDate,       // e.g. Thu Mar 28 2019 23:59:37 GMT-0400 (Eastern Daylight Time)...will be converted to GMT before being sent to server
			estimationName: estimationDialog.getNameInput().getValue(),
			firedTransactionsLimit: estimationDialog.getMaxFiredTxnsInput().getValue(),
			rules: rules,
            parallelizeBy: parallelizeBy,
			startGmtDateString: startDate,   // e.g. Fri Mar 22 2019 00:00:37 GMT-0400 (Eastern Daylight Time)...will be converted to GMT before being sent to server
			userVariablesPrimingDuration: primingDuration,
			userVariablesPrimingUnit: (estimationDialog.isDaysSelected() ? "DAYS" : "HOURS"), // Text not user-facing.
			verboseLog: estimationDialog.getLogDetailsSwitch().getState(),
		};
	},

	getCreateEstimationErrorMsg: function (estimationParams, response) {
        const message = getText("rules.estimation.create.error.fmt.msg", estimationParams.estimationName);
        const responseJson = response.responseJSON;
        const errorDetail = (responseJson) ? responseJson.message : null;
        const isNoTransactionsError = (responseJson) ? (responseJson.errorCode === 3) : false;
        
        return message + (isNoTransactionsError ? ("  " + errorDetail) : "");
	},
	
	// RAPTOR-5329 For CBA...
	handleCreateEstimationBespoke: function (estimationParams, ajaxParams) {
		// RAPTOR-5347
		this.estimationDialog.forceClose();

		this.getView().showInfoMessage(getText("rules.esimtation.create.request.submitted.info.msg"));
		 
		const self = this;
		sas.ajax(ajaxParams)
			.done(function (estimation) {
				self.updateTable();
                self.getView().showSuccessMessage(getText("rules.esimtation.created.success.info.fmt.msg", [estimation.name, estimation.id]));
			} )
			.fail(function (response) {
                self.getView().showErrorMessage(self.getCreateEstimationErrorMsg(estimationParams, response));
			} );
	},
	
	// Assume that the user has provided all required info.
	handleCreateEstimation: function () {
        sas.fscf.removeAllMessageStrips();

        const estimationDialog = this.estimationDialog;
		estimationDialog
			.setBusy(true)
			.disableButtons();

		const url = UrlBuilder.getCreateEstimationUrl(this.getBusinessUnitId());
		const estimationParams = this.getEstimationParams(estimationDialog);
		const ajaxParams = {        // The dates in startGmtDateString and endGmtDateString are converted to GMT before being sent to the server.
			url: url,               // Note: for the examples above 4 hours are added, to covert from GMT-4:00 to GMT.
			type: RequestType.POST, // e.g. Thu Mar 28 2019 23:59:37 GMT-0400 -> 2019-03-29T03:59:37.671Z
			data: estimationParams, // e.g. Fri Mar 22 2019 00:00:37 GMT-0400 -> 2019-03-22T04:00:37.671Z
			headers:sas.fscf.getAjaxHeaders()
		};
		
		// RAPTOR-5329 For CBA...
		if (getForceUserVarsCal()) {
			this.handleCreateEstimationBespoke(estimationParams, ajaxParams);
			return;
		}
				
		const self = this;
		sas.ajax(ajaxParams)
			.done(function (estimation) {
				self.updateTable();
				estimationDialog.close();
                self.getView().showSuccessMessage(getText("rules.esimtation.created.success.info.fmt.msg", [estimation.name, estimation.id]));
			} )
			.fail(function (response) {
                const message = self.getCreateEstimationErrorMsg(estimationParams, response);
                estimationDialog.showErrorMessage(message)
                    .setUpButtons();  // Re-enable the buttons so that the user can navigate, make changes and resubmit the request to run estimation.
                logError(message);
			} )
            .always(function () {
                estimationDialog.setBusy(false);
            } );
	},

	// Handler for context menu items.
	handleMenuItemPress: function (menuItemCode, rowIndex) {
  		sas.fscf.removeAllMessageStrips();
		const estimation = this.getEstimation(rowIndex);
		if (estimation !== null) {
			switch (menuItemCode) {
				case OpCode.CANCEL:
					this.cancelEstimation(estimation);
					break;
				case OpCode.COPY:
					this.copyEstimation(estimation);
					break;
				case OpCode.DELETE:
					this.deleteEstimations();
					break;
				case OpCode.VIEW_LOG:
					this.showLog();
					break;
				case OpCode.VIEW_PROPERTIES:
					this.showProperties(estimation);
					break;
				case OpCode.VIEW_RESULTS:
					this.showEstimationResults(estimation);
					break;
				default:
			}
		}
	},

    DEFAULT_SELECTED_SPLIT_TXNS_OPTION_KEY: "xqo_sas_cust_rand_dig",
	getDefaultSelectedSplitTxnsOptionKey: function () {
		return this.DEFAULT_SELECTED_SPLIT_TXNS_OPTION_KEY;
	},

	splitTxnsOptionsMap: null,
	getSplitTxnsOptions: function () {
		return this.splitTxnsOptionsMap.getOptions();
    },

    getSplitTxnsOption: function (key) {
        return this.splitTxnsOptionsMap.getOption(key);
    },

	setUpSplitTxnsOptionsMap: function () {
		if (this.splitTxnsOptionsMap !== null) {
			return;
		}
		const self = this;
		const ajaxParams = {
			url: UrlBuilder.getSplitTxnsOptionsUrl(this.getBusinessUnitId()),
			type: RequestType.GET,
			headers:sas.fscf.getAjaxHeaders()
		};
   		sas.ajax(ajaxParams)
   			.done(function (response) {
                self.splitTxnsOptionsMap = new TransactionsOptionsMap(response.items);
				// self.splitTxnsOptions = response.items;
   			} )
   			.fail(function (response) {
                const message = getText("rules.estimation.retrieve.split.transactions.options.error.msg");
                self.getView().showErrorMessage(message);
                const errorDetail = (response.responseJSON) ? response.responseJSON.message : null;
                logError(message, errorDetail);
   			} );
	},

	// TODO: refactor to encapsulate the alert types in a class.
	alertTypes: new Map(),
	getAlertTypes: function () {
		return this.alertTypes;
    },

    // eg. Account Number (A)
    getAlertTypeFullDescription: function (alertType) {
        return alertType.description + " (" + alertType.id + ")";
    },

    getAlertType: function (alertId) {
        const answer = this.getAlertTypes().get(alertId);
        return (answer) ? answer : {id: alertId, name: "", description: ""};
    },

    doSetUpAlertTypes: function (alertTypes) {
        alertTypes.forEach(function (each) {
        	const alertId = each.id;

            this.alertTypes.set(alertId, each);
        	each.fullDescription = this.getAlertTypeFullDescription(each);
        }, this );
    },

	setUpAlertTypes: function () {
		if (this.alertTypes.size !== 0) {
			return;
		}
		const self = this;
		const ajaxParams = {
			url: UrlBuilder.getAlertTypesUrl(this.getBusinessUnitId()),
			type: RequestType.GET,
			headers:sas.fscf.getAjaxHeaders()
		};
   		sas.ajax(ajaxParams)
   			.done(function (response) {
				self.doSetUpAlertTypes(response.items);
   			} )
   			.fail(function () {
				logError(getText("rules.estimation.retrieve.alert.types.error.msg")); // No need to inform the user because without the info if just means that there's no tooltip...not a critical error.
   			} );
	},

	handleNewPress: function () {
  		sas.fscf.removeAllMessageStrips();
        const myView = this.getView();
        sas.fscf.getApplicationView().setBusyIndicatorDelay(0).setBusy(true);

		const self = this;
		const ajaxParams = {
			url: UrlBuilder.getRulesUrl(this.getBusinessUnitId()),
			type: RequestType.GET,
			headers:sas.fscf.getAjaxHeaders()
		};
   		sas.ajax(ajaxParams)
   			.done(function (response) {
				self.doHandleNewPress(response.items);
   			} )
   			.fail(function (response) {
                const message = getText("rules.estimation.retrieve.rules.for.create.error.msg");
                myView.showErrorMessage(message);
                const errorDetail = (response.responseJSON) ? response.responseJSON.message : null;
                logError(message, errorDetail);
            } )
            .always(function () {
            	sas.fscf.getApplicationView().setBusy(false);
            } );
	},

	setUpRulesModel: function (rules) {
		// Exclude Variable rules.
		const authRules = rules.filter(function (each) {
			return each.ruleTypeName !== RuleType.VARIABLE;
		} );
		authRules.forEach(function (each) {		
            // Add 2 properties: displayId and displayIdString, for displaying and sorting purposes.
            RuleHelper.setDisplayId(each);
			// Convert the id to a string.  The tooltip for the id column expects the value to be a string.
			each.id = (each.id).toString();
			// Add a property named alertType, whose value is an object with the following info of the alert: id, name, and description
			each.alertType = this.getAlertType(each.alertTypeId);
		}, this );
		this.rulesModel = new sap.ui.model.json.JSONModel({modelData: authRules});
	},

	doHandleNewPress: function (rules) {
		this.setUpRulesModel(rules);

		this.estimationDialog = this.createEstimationDialog();
		this.estimationDialog.initWizard(this.getEstimationDialogWizard());
		this.estimationDialog.setUpButtons();
        this.estimationDialog.open();
	},

	handleUserFilterAllPress: function() {
  		sas.fscf.removeAllMessageStrips();
		this.setLastEstimationsFilterCookie(this.ESTIMATION_ALL);
		this.updateTable();
	},

	handleUserFilterMinePress: function() {
  		sas.fscf.removeAllMessageStrips();
		this.setLastEstimationsFilterCookie(this.ESTIMATION_USER);
		this.updateTable();
	},

	handleRefreshPress: function() {
  		sas.fscf.removeAllMessageStrips();
		this.updateTable();
	},

	handleViewResultsPress: function(estimation) {
		this.showEstimationResults(estimation);
	},

	isEditMode: function () {
		return sas.fscf.getCurrentPageView().jsonEstimation !== null;
	},

	isShowAllEstimations: function() {
		return this.getLastEstimationsFilterCookie() === this.ESTIMATION_ALL;
	},

	setLastEstimationsFilterCookie: function(value) {
		sas.fscf.setCookie(this.LAST_ESTIMATION_FILTER_COOKIE_NAME, value,
				sas.fscf.getNYearsFromNow(1) );
	},

	estimationResultsController: null,

	doShowEstimationResults: function(decisionMetrics, queueMetrics, ruleMetrics) {
		if (this.estimationResultsController === null) {
			this.estimationResultsController = sap.ui.controller("sas.fscf.views.estimation.EstimationResults");
            this.estimationResultsController.setEstimationsController(this);
		}
		this.estimationResultsController.showResults(decisionMetrics, queueMetrics, ruleMetrics, sas.fscf.getCurrentPageView());
	},

	showEstimationResults: function(anEstimation) {
        sas.fscf.removeAllMessageStrips();
        const estimation = (anEstimation) ? anEstimation : this.getSelectedEstimation();
        if (!estimation || ( !estimation.isResultAvaiable && estimation.status !== "Failed" )) {
        //if (!estimation || !estimation.isResultAvaiable) {
            return;
        }

        const myView = this.getView();
        sas.fscf.getApplicationView().setBusyIndicatorDelay(0).setBusy(true);

        const bizUnitId = this.getBusinessUnitId();
        const url = UrlBuilder.getDecisionMetricsUrl(bizUnitId, estimation.id);
		const ajaxParams = {
			url: url,
			type: RequestType.GET,
			headers:sas.fscf.getAjaxHeaders()
        };
        const decisionMetricsCall = sas.ajax(ajaxParams);
        const getQueueMetricsCall = sas.ajax({
            url: UrlBuilder.getQueueMetricsUrl(bizUnitId, estimation.id),
            type: RequestType.GET,
            headers:sas.fscf.getAjaxHeaders()
        } );
        const getRuleMetricsCall = sas.ajax({
            url: UrlBuilder.getRuleMetricsUrl(bizUnitId, estimation.id),
            type: RequestType.GET,
            headers:sas.fscf.getAjaxHeaders()
        } );

        const self = this;
        const handleError = function (response) {
            const message = getText("rules.estimation.retrieve.decision.metrics.error.fmt.msg", [estimation.name, estimation.id]);
            myView.showErrorMessage(message);
            const errorDetail = (response.responseJSON) ? response.responseJSON.message : null;
            logError(message, errorDetail);
        };

        $.when(decisionMetricsCall, getQueueMetricsCall, getRuleMetricsCall)
            .done(function (decisionMetricsResponse, queueMetricsResponse, ruleMetricsResponse) {
                const decisionMetrics = decisionMetricsResponse[0];
                const queueMetrics = queueMetricsResponse[0].queueMetrics;
                const ruleMetrics = ruleMetricsResponse[0].ruleMetrics;
                self.doShowEstimationResults(decisionMetrics, queueMetrics, ruleMetrics);
            } )
            .fail(function (response) {
                handleError(response);
            } )
            .always(function () {
            	sas.fscf.getApplicationView().setBusy(false);
            } );
	},

	updateTable: function() {
		this.getView().updateTable();
	},

    performDefaultAction: function () {
        const estimation = this.getSelectedEstimation();
        if (estimation) {
        	//if (estimation.status.toLowerCase() === "failed") {
            //    this.showLog();
            //} else if (estimation.isResultAvaiable) {
            //    this.showEstimationResults();
            //}
            if (estimation.status.toLowerCase() === "failed" || estimation.isResultAvaiable) {
                this.showEstimationResults();
            }
            
        }
    },
    // END: Estimation List

    getEstimationCache: function () {
        if (sas.fscf.getCurrentPageView) {
        	if (sas.fscf.getCurrentPageView().jsonEstimation === undefined) {
        		sas.fscf.getCurrentPageView().jsonEstimation = {};
        	}
        	return sas.fscf.getCurrentPageView().jsonEstimation;
        }
        return null;
    },

    cacheEstimationId: function (id) {
        this.getEstimationCache().id = id;
    },

	onInit: function () {
		this.getView().setUpEstimationListContextMenu();
        this.setUpAlertTypes();
        this.setUpSplitTxnsOptionsMap();
	},

	onBeforeRendering: function () {
	},

	onAfterRendering: function () {
        const self = this;

        this.getView().getTable().attachRowSelectionChange(function () {
            const selectedEstimation = self.getSelectedEstimation();
            if (selectedEstimation !== null) {
                self.cacheEstimationId(selectedEstimation.id);
            }
        } );
	},
	/*** END: Estimation List ***/

	/*** BEGIN: Estimation Properties ***/
	estimationController: null,

	getEstimationController: function () {
		if (this.estimationController === null) {
            this.estimationController = new sas.fscf.views.estimation.EstimationController();
            this.estimationController.setEstimationsController(this);
		}
		return this.estimationController;
	},

	showProperties: function (param) {
  		sas.fscf.removeAllMessageStrips();
		// param should be either an estimation or a button (with id "press")
		let estimation = param;
		if (param && param.sId && (param.sId === "press")) {
			estimation = this.getSelectedEstimation();
		}
		if (estimation) {
            const refreshEstimationsCallback = this.refreshEstimationsModel.bind(this);
			this.getEstimationController().showProperties(this.getBusinessUnitId(), estimation, refreshEstimationsCallback);
		}
	},
	/*** END: Estimation Properties ***/

	showLog: function () {
        this.getEstimationController().showLog(this.getView());
    },

    getEstimation: function (idx) {
        const view = this.getView();
        const index = (idx) ? idx : view.getSelectedEstimationIndex();

        if (idx === -1) {
            return null;
        }
        const estimationTable = view.getTable();
        const context = estimationTable.getContextByIndex(index);

        return estimationTable.getModel().getProperty(context.sPath);
    },

    getSelectedEstimation: function () {
        return this.getEstimation(this.getView().getSelectedEstimationIndex());
    },

    getSelectedEstimations: function () {
		return this.getView().getTable().getSelectedIndices().map(function (each) {
			return this.getEstimation(each);
		}, this );
    },

    refreshEstimationsModel: function () {
        this.getView().getTable().getModel().refresh();
    },
} );
