/*!
 * UI development toolkit for HTML5 (OpenUI5)
 * (c) Copyright 2009-2025 SAP SE or an SAP affiliate company.
 * Licensed under the Apache License, Version 2.0 - see LICENSE.txt.
 */
sap.ui.define(["sap/m/library", "sap/ui/Device", "sap/ui/core/library"],
	function(library, Device, coreLibrary) {
		"use strict";

		// shortcut for sap.m.DialogType
		var DialogType = library.DialogType;

		// shortcut for sap.ui.core.ValueState
		var ValueState = coreLibrary.ValueState;

		/**
		 * Dialog renderer.
		 *
		 * @namespace
		 */
		var DialogRenderer = {};

		// Mapping of ValueState to style class
		DialogRenderer._mStateClasses = {};
		DialogRenderer._mStateClasses[ValueState.None] = "";
		DialogRenderer._mStateClasses[ValueState.Success] = "sapMDialogSuccess";
		DialogRenderer._mStateClasses[ValueState.Warning] = "sapMDialogWarning";
		DialogRenderer._mStateClasses[ValueState.Error] = "sapMDialogError";

		/**
		 * Renders the HTML for the given control, using the provided {@link sap.ui.core.RenderManager}.
		 *
		 * @param {sap.ui.core.RenderManager} oRm The RenderManager that can be used for writing to the render output buffer.
		 * @param {sap.ui.core.Control} oControl An object representation of the control that should be rendered.
		 */
		DialogRenderer.render = function(oRm, oControl) {
			var id = oControl.getId(),
				sType = oControl.getType(),
				oHeader = oControl._getAnyHeader(),
				oSubHeader = oControl.getSubHeader(),
				bMessage = (sType === DialogType.Message),
				oLeftButton = oControl.getBeginButton(),
				oRightButton = oControl.getEndButton(),
				bHorizontalScrolling = oControl.getHorizontalScrolling(),
				bVerticalScrolling = oControl.getVerticalScrolling(),
				sState = oControl.getState(),
				bStretch = oControl.getStretch(),
				bStretchOnPhone = oControl.getStretchOnPhone() && Device.system.phone,
				bResizable = oControl.getResizable(),
				bDraggable = oControl.getDraggable(),
				oValueStateText = oControl.getAggregation("_valueState");

			// write the HTML into the render manager
			// the initial size of the dialog have to be 0, because if there is a large dialog content the initial size can be larger than the html's height (scroller)
			// The scroller will make the initial window width smaller and in the next recalculation the maxWidth will be larger.
			var initialWidth = oControl.getContentWidth() && oControl.getContentWidth() != 'auto' ? ' width: ' + oControl.getContentWidth() + ';' : '';
			var initialHeight = oControl.getContentHeight() && oControl.getContentHeight() != 'auto' ? ' height: ' + oControl.getContentHeight() + ';' : '';
			var initialStyles = "style='" + initialWidth + initialHeight + "'";

			oRm.write('<div ' + initialStyles);
			oRm.writeControlData(oControl);
			oRm.addClass("sapMDialog");

			// SAS Edit Start
			this._addSASClassesToDialogDiv(oRm);
			// SAS Edit End

			oRm.addClass("sapMDialog-CTX");
			oRm.addClass("sapMPopup-CTX");

			if (oControl.isOpen()) {
				oRm.addClass("sapMDialogOpen");
			}

			if (window.devicePixelRatio > 1) {
				oRm.addClass("sapMDialogHighPixelDensity");
			}

			if (oControl._bDisableRepositioning) {
				oRm.addClass("sapMDialogTouched");
			}

			if (bStretch || (bStretchOnPhone)) {
				oRm.addClass("sapMDialogStretched");

				// SAS Edit Start
				this._addSASStretchClass(oRm);
				// SAS Edit End
			}

			oRm.addClass(DialogRenderer._mStateClasses[sState]);

			// No Footer
			var noToolbarAndNobuttons = !oControl._oToolbar && !oLeftButton && !oRightButton;
			var emptyToolbarAndNoButtons = oControl._oToolbar && oControl._isToolbarEmpty() && !oLeftButton && !oRightButton;
			if (noToolbarAndNobuttons || emptyToolbarAndNoButtons) {
				oRm.addClass("sapMDialog-NoFooter");
			}

			if (!oHeader) {
				oRm.addClass("sapMDialog-NoHeader");
			}

			// SAS Change Start
			// ARIA role.
			// If the state is set to either Error or Warning we keep the SAP behavior of always using the AlertDialog role.
			// Otherwise we honor whatever role was set in the accessibleRole property, which defaults to Dialog
			var effectiveRole = oControl.getAccessibleRole();
			if (sState === "Error" || sState === "Warning") {
				effectiveRole = sap.ui.core.AccessibleRole.AlertDialog;
			}
			oRm.writeAccessibilityState(oControl, {role: effectiveRole.toLowerCase() });
			// SAS Change End

			if (oControl._forceDisableScrolling) {
				oRm.addClass("sapMDialogWithScrollCont");

				// SAS Edit Start
				this._addSASDialogWithScrollContClass(oRm);
				// SAS Edit End
			}

			if (oSubHeader && oSubHeader.getVisible()) {
				oRm.addClass("sapMDialogWithSubHeader");
			}

			if (bMessage) {
				oRm.addClass("sapMMessageDialog");
			}

			if (!bVerticalScrolling) {
				oRm.addClass("sapMDialogVerScrollDisabled");
			}

			if (!bHorizontalScrolling) {
				oRm.addClass("sapMDialogHorScrollDisabled");
			}

			if (Device.system.phone) {
				oRm.addClass("sapMDialogPhone");
			}

			if (bDraggable && !bStretch) {
				oRm.addClass("sapMDialogDraggable");
			}

			// test dialog with sap-ui-xx-formfactor=compact
			if (library._bSizeCompact) {
				oRm.addClass("sapUiSizeCompact");
			}

			oRm.writeClasses();

			var sTooltip = oControl.getTooltip_AsString();

			if (sTooltip) {
				oRm.writeAttributeEscaped("title", sTooltip);
			}

			oRm.writeAttribute("tabindex", "-1");

			oRm.write(">");

			if (Device.system.desktop) {

				// SAS Edit Start

				// Add the messageIconContainer for MessageDialog (HTMLCOMMONS-8565):
				this._addMessageIconContainer(oRm, oControl);

				// Invisible element which is used to determine when desktop keyboard navigation
				// has reached the first focusable element of a dialog and went beyond. In that case, the controller
				// will focus the last focusable element.
				oRm.write('<span id="' + oControl.getId() + '-firstfe" tabindex="0"></span>');

				if (bResizable && !bStretch) {

					this._writeResizeIcon(oRm, oControl);
				}

				this._addCloseXButton(oRm, oControl);
				// SAS Edit End
			}

			if (oHeader) {
				oHeader._applyContextClassFor("header");
				oRm.write("<header");
				oRm.addClass("sapMDialogTitle");
				oRm.writeClasses();
				oRm.write(">");
				oRm.renderControl(oHeader);
				oRm.write("</header>");
			}

			if (oSubHeader) {
				oSubHeader._applyContextClassFor("subheader");
				oRm.write("<header");
				oRm.addClass("sapMDialogSubHeader");
				oRm.writeClasses();
				oRm.write(">");
				oRm.renderControl(oSubHeader);
				oRm.write("</header>");
			}

			if (oValueStateText) {
				oRm.renderControl(oValueStateText);
			}

			oRm.write('<section id="' + id + '-cont" class="sapMDialogSection">');
			oRm.write('<div id="' + id + '-scroll" class="sapMDialogScroll">');
			oRm.write('<div id="' + id + '-scrollCont" class="sapMDialogScrollCont');

			if (oControl.getStretch() || initialHeight) {
				oRm.write(' sapMDialogStretchContent');
			}
			// SAS Edit Start
			this._addSASScrollContClass(oRm);
			// SAS Edit End

			oRm.write('">');

			var aContent = oControl.getContent();

			for (var i = 0; i < aContent.length; i++) {
				oRm.renderControl(aContent[i]);
			}

			oRm.write("</div>");
			oRm.write("</div>");
			oRm.write("</section>");

			// Add subcontent footer before the footer element:
			this._addSubContentFooter(oRm, oControl);

			if (!(noToolbarAndNobuttons || emptyToolbarAndNoButtons)) {
				oRm.write("<footer");
				oRm.addClass("sapMDialogFooter");

				// SAS Edit Start
				// Add the SAS style classes for the Footer and Buttons
				this._addSASDialogFooterClasses(oRm);
				// SAS Edit End

				oRm.writeClasses();
				oRm.write(">");
				oControl._oToolbar._applyContextClassFor("footer");
				oRm.renderControl(oControl._oToolbar);
				oRm.write("</footer>");
			}

			if (Device.system.desktop) {
				// Invisible element which is used to determine when desktop keyboard navigation
				// has reached the last focusable element of a dialog and went beyond. In that case, the controller
				// will focus the first focusable element.
				oRm.write('<span id="' + oControl.getId() + '-lastfe" tabindex="0"></span>');
			}

			oRm.write("</div>");
		};

		//SAS Edit Start
		/**
		 * Function to add MessageIconContainer. This is overridden in SAS DialogRenderer.
		 * @param oRm
		 * @private
		 */
		DialogRenderer._addMessageIconContainer = function(oRm, oControl) {
			// SAP does not have MessageIconContainer
		};

		/**
		 * Function to write the resize icon. This is overridden in SAS DialogRenderer to use SAS's resize icon.
		 * @param oRm
		 * @private
		 */
		DialogRenderer._writeResizeIcon = function(oRm, oControl) {
			oRm.writeIcon("sap-icon://resize-corner", ["sapMDialogResizeHandler"], {
				"title": ""
			});
		};

		/**
		 * Function to write the Close X buton on top-right corner.
		 * This is overridden in SAS DialogRenderer to add the Close X button as SAP does not have it.
		 * @param oRm
		 * @private
		 */
		DialogRenderer._addCloseXButton = function(oRm, oControl) {
			// SAP does not have Close X button
		};

		/**
		 * Function to add sub-content-footer.
		 * This is overridden in SAS to add a sub-content-footer as SAP does not have it.
		 * @param oRm
		 * @private
		 */
		DialogRenderer._addSubContentFooter = function(oRm, oControl) {
			// SAP does not have sub-content-footer
		};

		/**
		 * Function to add SAS classes to Dialog's main <div>.
		 * This is overridden in SAS as SAP does not the SAS CSS classes.
		 * @param oRm
		 * @private
		 */
		DialogRenderer._addSASClassesToDialogDiv =  function(oRm) {
			// SAP does not have SAS's CSS classes
		};

		/**
		 * Function to add SAS stretch class to Dialog's main <div>.
		 * This is overridden in SAS as SAP does not the SAS CSS classes.
		 * @param oRm
		 * @private
		 */
		DialogRenderer._addSASStretchClass = function(oRm) {
			// SAP does not have SAS's CSS classes
		};

		/**
		 * Function to add SAS scroll container class to Dialog's main <div>.
		 * This is overridden in SAS as SAP does not the SAS CSS classes.
		 */
		DialogRenderer._addSASDialogWithScrollContClass = function(oRm) {
			// SAP does not have SAS's CSS classes
		};

		/**
		 * Function to add SAS scroll container class to Dialog's main <div>.
		 * This is overridden in SAS as SAP does not the SAS CSS classes.
		 */
		DialogRenderer._addSASScrollContClass = function(oRm) {
			// SAP does not have SAS's CSS classes
		};

		/**
		 * Function to add SAS scroll container class to Dialog's main <div>.
		 * This is overridden in SAS as SAP does not the SAS CSS classes.
		 */
		DialogRenderer._addSASDialogFooterClasses = function(oRm) {
			// SAP does not have SAS's CSS classes
		};
		// SAS Edit End

		return DialogRenderer;
	}, /* bExport= */ true);