#!/bin/ksh -p

# $Id: tas.sh,v 1.1.2.1.2.5.2.8 2018/03/19 12:48:35 prsurt Exp $

# terminate on non-zero command exit status
set -e

PROGRAM_NAME="$0"

COMMANDS="help save start status stop tail testomr erase erase-fgdb erase-grdb"

function main
{
    typeset BIN=$(cd $(dirname "$PROGRAM_NAME") ; pwd -P)
    
    . "$BIN/functions.sh"
    
    clear_env
    
    read_properties_file "${TAS_PROPERTIES:="$BIN/../etc/tas.properties"}"

    init_server_env
    
    : ${CONSOLE_LOG:="${LOGS_DIR:?}/console.log"}
    
    for CMD in $@
    do
        # XXX check for flags, e.g. "-config=.."
        
        if [ "$CMD" = "stop" ] ; then
            stop_tas
        else
            case "$CMD" in
                debug    )  MODE="TEST"   ;;
                help     )  usage    ;;
                save     )  save     ;;
                start    )  start    ;;
                status   )  status   ;;
                testomr  )  testomr "$@"
                            break    ;;
                erase      )  erase "$@"
                            break    ;;                            
                erase-grdb )  erase "erase grdb Y"
                            break    ;;                            
#                erase-esdb )  erase "erase esdb Y"
#                            break    ;;                            
                erase-fgdb )  erase "erase fgdb Y"
                            break    ;;                            
                tail     )  tail -f "$CONSOLE_LOG" ;;
                *        )  unknown_command "$CMD" ;;
            esac
        fi
    done

    if [ $# -eq 0 ] ; then
        usage
    fi
}

function kill_tas
{
    if [ -n "$TAS_PID" ] ; then
        status
        kill -KILL -${TAS_PGID:?}
        TAS_PID=
        clean
    fi
}

function save
{
    set -e

    LM_FILE=$(ls -1t "${LOGS_DIR:?}" | tr '\n' ' ' | cut -d ' ' -f 1)

    if [ -z "$LM_FILE" ] ; then
        return
    fi

    set_DT "${LOGS_DIR:?}/$LM_FILE"
    
    prune_by_count
    
    # TODO add prune_by_time and prune_by_footprint

    SAVE_DIR="${ARCHIVE_DIR:?}/${DT:?}"

    if [ ! -d "${SAVE_DIR:?}" ] ; then
        mkdir "${SAVE_DIR:?}"
    else 
        chmod -f u+w "${SAVE_DIR:?}/"*
    fi

    $RSYNC "${LOGS_DIR:?}/"* "${SAVE_DIR:?}"

    if [ -n "$TAS_PID" ] ; then
        touch "${SAVE_DIR:?}/STILL_RUNNING"
    else
        rm -f "${SAVE_DIR:?}/STILL_RUNNING"
    fi

    chmod -f a-w "${SAVE_DIR:?}/"*
}

function bundle_env
{
    # XXX i don't think set -e catches pipeline errors
    
    # *assumes* etc and bin are the config and program dir...
    
    tar -cf - -C "${SERVER_HOME:?}" \
    $(cd ${SERVER_HOME:?}; ls -1 bin/*.sh etc/*.properties | tr '\n' ' ') \
    | gzip -c >"${LOGS_DIR:?}/env.tgz"
    
    # TODO if its the same as the one in the most recent archive, use a hard link
}

function start
{
    set -e

    if [ -n "$TAS_PID" ] ; then
        echo "already running"
        exit
    fi

    save

    rm -Rf "${LOGS_DIR:?}/"*
    
    bundle_env
    
    export LOGS_DIR="${LOGS_DIR}"
    
    CMD="${JAVA:?}"
    
    # comment this out to get log4j logs in local time
    CMD="$CMD -Duser.timezone=UTC"

    CMD="$CMD -Xmx${JAVA_HEAP:?}"
    CMD="$CMD -Xms${JAVA_HEAP:?}"
    CMD="$CMD -Xmn${JAVA_YOUNG_HEAP:?}"

    if [ "${MODE:?}" != "PROD" -o -n "$DEBUG_PORT" ]; then
        OPTS="$OPTS $(debug_opts)"
    fi
    
#   OPTS="$OPTS -verbose:class"
#   OPTS="$OPTS -verbose:jni"

    CMD="$CMD -Dlog4j.configurationFile=${LOG4J_CONFIG:?}"
    
    # XXX what's the overhead, should it be disabled for PROD?
    OPTS="$OPTS -verbose:gc"
    
    # these are Oracle JVM specific but should be ignored by the IBM JVM
    #OPTS="$OPTS -XX:+PrintGCDateStamps"
    # OPTS="$OPTS -XX:+PrintGCDetails" (small performance penalty)

    # Uncomment the following line when enabling SSL for JMX
    # OPTS="$OPTS -Djavax.net.ssl.keyStore=<fully qualified path to keystore>"
    
    # Enable SSL debugging
    # OPTS="$OPTS -Djavax.net.debug=all"  
    
    CMD="$CMD $OPTS"

    CMD="$CMD ${TAS_CLASS:?}"

    echo >"${CONSOLE_LOG:?}"

    date >>"${CONSOLE_LOG:?}"
    echo >>"${CONSOLE_LOG:?}"

    ${JAVA:?} -version >>"${CONSOLE_LOG:?}" 2>&1
    echo >>"${CONSOLE_LOG:?}"

    echo "nohup $CMD </dev/null 2>&1 &" >>"${CONSOLE_LOG:?}"
    echo >>"${CONSOLE_LOG:?}"

    nohup $CMD -start </dev/null >>"${CONSOLE_LOG:?}" 2>&1 &
}

function status
{
    if [ -n "$TAS_PID" ]; then
        echo
        echo "$TAS_PID"
        echo
        ${PSTREE:?} $TAS_PID
        echo
    fi
}

function stop_tas
{
    if [ -z "$TAS_PID" ]; then
        return
    fi
     
    status

    # XXX capture the children pids so they can be checked
    # for and waxed if necessary

    echo "kill -s TERM $TAS_PID"
    kill -s TERM $TAS_PID
    
    while [ `ps -o pid= -p $TAS_PID | wc -l` -ne 0 ] ; do
       sleep 1 
       ${ECHO_N:?} ".\c"
    done

    echo " done"

    TAS_PID=
}

function erase
{
    set -e

    CMD="${JAVA:?}"
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finance.fraud.txserver.Clean $@"

    $CMD

}

function testomr
{
    set -e

    CMD="${JAVA:?}"
    LOG4J_CONFIG="log4j-testomr.xml"  
    echo "LOG4J_CONFIG = $LOG4J_CONFIG" 
    CMD="$CMD -Dlog4j.configurationFile=${LOG4J_CONFIG:?}"
    
    CMD="$CMD $OPTS"
    CMD="$CMD com.sas.finserv.creditfraud.common.util.OmrHelper $@"

    $CMD

}


function usage
{
    echo "Usage: ${PROGRAM_NAME:?} [ ${COMMANDS:?} ]"
}

main "$@"


