# ------------------------------------------------------------------------------
#         Copyright (c) 2022 by SAS Institute Inc., Cary, NC USA 27513
# ------------------------------------------------------------------------------
#
# PUROPSE:  Create, install, uninstall, start, or stop a Windows service that
#           executes a specified Tomcat web application server.  A list of
#           services whose names match a specified pattern can also displayed.
#
# NOTES:
#
# ------------------------------------------------------------------------------


<#
.SYNOPSIS
Manage SAS Web Application servers as a Windows service.

.DESCRIPTION
The SAS Web Application servers can be executed as a Windows service, allowing
them to run without anyone having to login to the machine.  This utility is
used to install and uninstall a server as a service, and can also be used to
start and stop the execution of the service and the server it is hosting.

.PARAMETER Action
REQUIRED - The operation to perform on a Windows Service hosting a server.
Possible operations are: list, install, uninstall, start, and stop.

    CREATE - Create a new tomcate instance.
    INSTALL - Install a new Windows service.
    UNINSTALL - Uninstall a Windows service (WARNING: cannot be undone!).
    START - Start execution of an installed service.
    STOP - Stop execution of an installed service.

    NOTE: All of the actions require that environment variables CATALINA_HOME
    and CATALINA_BASE both be defined, and that the paths they specify exist.
    The default service name is derived by using CATALINA_BASE.

.PARAMETER NoExec
OPTIONAL - Gather all inputs and make all preparations for normal execution, but
do not actually execute the operation.  The final command that would have been
executed is displayed as output.

.PARAMETER Debug
OPTIONAL - Output additional messages showing run-time parameters and values.

.PARAMETER Verbose
OPTIONAL - Output additional messages showing more in-depth information that would
normally be shown.

.EXAMPLE
#
#
# Bare minimum service installation.  The service display name and its description
# are set to defaults, and the JAVA_HOME environment variable must be defined.
#
#
tcshim.ps1 -action install

.EXAMPLE
REM
REM
REM Invoking a PowerShell environment from the Windows command line and passing
REM it the name of the SAS service control script to execute, and the script
REM arguments, without actually executing the specified action.
REM
C:\> powershell -file .\tcshim.ps1 -action install -debug -verbose -noexec

.EXAMPLE
REM
REM
REM Uninstalling an existing service from the Windows command line by
REM invoking PowerShell to execute the SAS service control script.
REM
C:\> powershell -file tcshim.ps1 -action uninstall

.EXAMPLE
#
#
# Start an existing service that is not running.
#
tcshim.ps1 -action start

.EXAMPLE
#
#
# Stop a service that is running.
#
tcshim.ps1 -action stop
#>


[ CmdletBinding ( PositionalBinding = $false ) ]

Param ( [Parameter ( Mandatory = $true,
                     HelpMessage = "(Required) Operation to perform: create | install | uninstall | start | stop"
                   )
        ]
        [validateset ( "Create", "Install", "Uninstall", "Start", "Stop" ) ]
        [ValidateNotNullOrEmpty()]
        [String] $Action,
        # ----------------------------------------------------------------------
        [Parameter ( Mandatory = $false,
                     HelpMessage = "(Optional) Properties file if creating a new instance"
                   )
        ]
        [String] $PropertiesFile,
        # ----------------------------------------------------------------------
        [Parameter ( Mandatory = $false,
                     HelpMessage = "(Optional) Prepare to execute, but do not do so"
                   )
        ]
        [Switch] $NoExec
      )

Set-Variable TOMCAT_EXECUTABLE_NAME    -Option Constant -Value "Tomcat10.exe"


function ValidateEnvPath ( [String] $EnvVar )
{
    [String] $PathValue = [System.Environment]::GetEnvironmentVariable($EnvVar,'Process')
    if ( $PathValue.Length -eq 0 )
    {
        Write-Error "ERROR: The $EnvVar environment variable is not defined." `
        -Category ObjectNotFound
    }

    if ( (Test-Path -Path $PathValue -PathType Container) -eq $false )
    {
        Write-Error -Message "ERROR: The $EnvVar path was not found: $PathValue" `
                    -Category ObjectNotFound
    }

    $PathValue = $PathValue -replace '/', '\'
    return $PathValue
}


function GetEnvPath ( [String] $EnvVar )
{
    Write-Debug 'Creating env path.'
    [String] $PathValue = [System.Environment]::GetEnvironmentVariable($EnvVar,'Process')
    if ( $PathValue.Length -eq 0 )
    {
        Write-Error "ERROR: The $EnvVar environment variable is not defined." `
        -Category ObjectNotFound
    }

    $PathValue = $PathValue -replace '/', '\'
    return $PathValue
}

function GetJVMOptions ( [String] $CatalinaBase )
{
    # TODO: There could be heap settings in here as well.
    [String] $OptionList = ""

    [String] $WrapperFileName = "${CatalinaBase}\conf\wrapper.conf" 
    if ( (Test-Path -Path $WrapperFileName -PathType Leaf) -eq $false )
    {
        Write-Error -Message "ERROR: File not found: $WrapperFileName" `
                    -Category ObjectNotFound
    }

    else
    {
        $Count = 0
        $Parsed = ""

        foreach ( $Line in ( Get-Content $WrapperFileName ) )
        {
            if ( $Line -match "^\s*wrapper.java.additional.\d+\s*=\s*.*" )
            {
                [String] $Option = ( $Line -split "^\s*wrapper.java.additional.\d+\s*=\s*" )
                $Option = $Option.Trim()
                if ( ($Option.Length -gt 0) -and ($Option.StartsWith('"-D')) )
                {
                    # Replace double-quotes around the entire option with
                    # single-quotes around just the option value.
                    if ( $Option -match "^`".*`"$" )
                    {
                        $Option = $Option -replace '"', ''
                        [String] $OptName, [String] $OptValue = $Option -split "="
                        $Option = "$OptName=`'$OptValue`'"
                    }

                    # Replace all double-quotes used in a single option with single-quotes.
                    $Option = $Option -replace "`"", "'"

                    # Wrap all semi-colons used in a single option in single-quotes.
                    $Option = $Option -replace ";", "';'"

                    $Count++
                    if ( $Count -eq 1 )
                    {
                        $Parsed = "Options parsed from file ${WrapperFileName}:`n"
                    }
                    $Parsed += "   $Count $Option`n"

                    if ( $Count -ne 1 )
                    {
                        # Add an option separator (semi-colon) to the end of the previous
                        # option value, prior to adding the new option.  Adding the separator
                        # prior to appending a new option, rather than adding the separator as
                        # part of the previous option prevents a troublesome trailing separator
                        # appearing at the end of the option string.
                        $OptionList += ( ";" )
                    }
                    $OptionList += $Option
                }
            }
        }
        

        if ( $Count -eq 0 )
        {
            Write-Warning "No options were parsed from $WrapperFileName"
        }

        elseif ( -not ( $Parsed.IsNullOrEmpty ) )
        {
            Write-Verbose "$Parsed`n"
        }
        
    }
    
    <#
    # Check for JMX port and add JMX options if we find one.
    # TODO: Before this will work, we need to setup file permissions correctly on conf\jmxremote.password, otherwise tomcat will not start.
    [String] $JmxPortFileName = "${CatalinaBase}\conf\jmxremote.port" 
    if ( (Test-Path -Path $JmxPortFileName -PathType Leaf) -eq $true )
    {
        $JmxPort = Get-Content $JmxPortFileName
        
        if ( $JmxPort -ne "-1" )
        {
            $JmxOptions = "-Dcom.sun.management.jmxremote.port=${JmxPort};-Dcom.sun.management.jmxremote.ssl=false;-Dcom.sun.management.jmxremote.authenticate=true;-Dcom.sun.management.jmxremote.password.file=${CatalinaBase}\conf\jmxremote.password;-Dcom.sun.management.jmxremote.access.file=${CatalinaBase}\conf\jmxremote.access"
            
            if ( $Count -ne 1 )
            {
                $OptionList += ( ";" )
            }
            $OptionList += $JmxOptions
        }    
    }
    #>

    return $OptionList
}


function GetClassPath ( [String] $WrapperFileName )
{
    [String] $ClassPath = ""

    if ( (Test-Path -Path $WrapperFileName -PathType Leaf) -eq $false )
    {
        Write-Error -Message "ERROR: File not found: $WrapperFileName" `
                    -Category ObjectNotFound
    }

    else
    {
        $Count = 0
        $Parsed = ""

        foreach ( $Line in ( Get-Content $WrapperFileName ) )
        {
            if ( $Line -match "^\s*wrapper.java.classpath.\d+\s*=\s*.*" )
            {
                [String] $Path = ( $Line -split "^\s*wrapper.java.classpath.\d+\s*=\s*" )
                $Path = $Path.Trim()
                if ( $Path.Length -gt 0 )
                {
                    $Count++
                    if ( $Count -eq 1 )
                    {
                        $Parsed = "Class paths parsed from file ${WrapperFileName}:`n"
                    }
                    $Parsed += "   $Count $Path`n"

                    if ( $Count -ne 1 )
                    {
                        $ClassPath += ( ";" )
                    }
                    $ClassPath += $Path
                }
            }
        }

        if ( $Count -eq 0 )
        {
            Write-Warning "No class paths were parsed from $WrapperFileName"
        }

        elseif ( -not ( $Parsed.IsNullOrEmpty ) )
        {
            Write-Verbose "$Parsed`n"
        }
    }

    return $ClassPath
}


function GetServiceProperties ( [String] $WrapperFileName, [String] $Name )
{
    $ServiceHash = @{}

    if ( (Test-Path -Path $WrapperFileName -PathType Leaf) -eq $false )
    {
        Write-Error -Message "ERROR: File not found: $WrapperFileName" `
                    -Category ObjectNotFound
    }

    else
    {
        Write-Debug 'Initializing default service properties'
        $ServiceHash["name"] = $Name
        $ServiceHash["display"] = "SAS $Name - WebAppServer"
        $ServiceHash["description"] = "SAS WebAppServer - Apache Tomcat web application server"
        
        foreach ( $Line in ( Get-Content $WrapperFileName ) )
        {
            if ( $Line -match "^\s*wrapper.ntservice.\w+\s*=\s*.*" )
            {
                [String] $WrapperName = ( $Line -split "^\s*wrapper.ntservice.\w+\s*=\s*" )
                $WrapperName = $WrapperName.Trim()
                $WrapperName = $WrapperName -replace '%wrapper.ntservice.id%', $Name
                
                Write-Debug "Considering Line: $Line"
                Write-Debug "Considering WrapperName: $WrapperName"
                
                if ( $WrapperName.Length -gt 0 )
                {
                    if ( $Line -match "^\s*wrapper.ntservice.name\s*=\s*.*" )
                    {
                        Write-Debug 'Found name in wrapper.conf'
                        $ServiceHash["name"] = $WrapperName
                    }
                    elseif ( $Line -match "^\s*wrapper.ntservice.displayname\s*=\s*.*" )
                    {
                        Write-Debug 'Found displayname in wrapper.conf'
                        $ServiceHash["display"] = $WrapperName
                    }
                    elseif ( $Line -match "^\s*wrapper.ntservice.description\s*=\s*.*" )
                    {
                        Write-Debug 'Found description in wrapper.conf'
                        $ServiceHash["description"] = $WrapperName
                    }
                }
            }
            
        }
    }

    return $ServiceHash
}


function GetHeapProperties ( [String] $WrapperFileName )
{
    $HeapHash = @{}

    if ( (Test-Path -Path $WrapperFileName -PathType Leaf) -eq $false )
    {
        Write-Error -Message "ERROR: File not found: $WrapperFileName" `
                    -Category ObjectNotFound
    }

    else
    {
        Write-Debug 'Initializing default heap properties'
        $HeapHash["JvmMs"] = "1024"
        $HeapHash["JvmMx"] = "4096"
        
        foreach ( $Line in ( Get-Content $WrapperFileName ) )
        {
            if ( $Line -match "^\s*wrapper.java.additional.\d+\s*=\s*.*" )
            {
                [String] $Option = ( $Line -split "^\s*wrapper.java.additional.\d+\s*=\s*" )
                $Option = $Option.Trim()
                if ( ($Option.Length -gt 0) -and ($Option.StartsWith('"-X')) )
                {
                    # Replace double-quotes around the entire option with
                    # single-quotes around just the option value.
                    if ( $Option -match "^`".*`"$" )
                    {
                        $Option = $Option -replace '"', ''
                    }

                    if ( $Option.StartsWith("-Xmx" ) )
                    {
                        $MemSize = $Option -replace "-Xmx", ""
                        $HeapHash["JvmMx"] = FormattedSizeInBytes $MemSize $HeapHash["JvmMx"]
                    }
                    elseif ( $Option.StartsWith("-Xss" ) )
                    {
                        $MemSize = $Option -replace "-Xss", ""
                        $HeapHash["JvmMs"] = FormattedSizeInBytes $MemSize $HeapHash["JvmMs"]
                    }
                }
            }
        }
    }

    return $HeapHash
}

function Format-Bytes ( [Float] $Value )
{
   $MemSizes = 'KB','MB','GB','TB','PB'

   for ( $i = 0; $i -lt $MemSizes.count; $i++ )
   {
      if ( $Value -lt "1$($MemSizes[$i])" )
      {
         if ( $i -eq 0 )
         {
            return "$Value B"
         }

         else
         {
            $Value = $Value / "1$($MemSizes[$i-1])"
            $Value = "{0:n2}" -f $Value
            return "$Value $($MemSizes[$i-1])"
         }
      }
   }
}


function FormattedSizeInBytes( [String] $FormattedSize, [String] $DefaultSizeInMB )
{
    # Tomcat wants the memory size in MB's. So the goal is to convert to MB's.
    [String] $SizeInMB = $DefaultSizeInMB

    # <size>[g|G|m|M|k|K]
    if ( $FormattedSize  -match "^\d+[gGmMkK]?")
    {
        if ( $FormattedSize -match "[gG]" )
        {
            $RawSize = $FormattedSize -replace "[gG]", ""
            $NumericSize = [int] $RawSize
            $NumericSize = $NumericSize * 1024
            $SizeInMB = $NumericSize.toString()
        }
        elseif ( $FormattedSize -match "[mM]" )
        {
            # Lucky us, this is what we wanted.
            $SizeInMB = $FormattedSize -replace "[mM]", ""
        }
        elseif ( $FormattedSize -match "[kK]" )
        {
            $RawSize = $FormattedSize -replace "[kK]", ""
            $NumericSize = [int] $RawSize
            $NumericSize = $NumericSize / 1024
            $NumericSize = [math]::ceiling($NumericSize)
            if ( $NumericSize -le 0 )
            {
                $NumericSize = 1
            }
            $SizeInMB = $NumericSize.toString()
        }
        else
        {
            # Size is interpreted as bytes.
            $NumericSize = [int] $FormattedSize
            $NumericSize = $NumericSize / (1024 * 1024)
            $NumericSize = [math]::ceiling($NumericSize)
            if ( $NumericSize -le 0 )
            {
                $NumericSize = 1
            }
            $SizeInMB = $NumericSize.toString()
        }
    }

    return $SizeInMB
}

function GetInstanceProperties ( [String] $PropertiesFileName )
{
    $PropertiesHash = @{}

    if ( (Test-Path -Path $PropertiesFileName -PathType Leaf) -eq $false )
    {
        Write-Error -Message "ERROR: File not found: $PropertiesFileName" `
                    -Category ObjectNotFound
    }

    else
    {
        Write-Debug 'Initializing default instance properties'
        $PropertiesHash["base.shutdown.port"] = "-1"
        $PropertiesHash["nio.http.port"] = "8080"
        $PropertiesHash["base.jmx.port"] = "6969"
        
        foreach ( $Line in ( Get-Content $PropertiesFileName ) )
        {
            if ( $Line -match "^\s*[\w\.]+\s*=\s*.*" )
            {
                [String] $PropertyValue = ( $Line -split "^\s*[\w\.]+\s*=\s*" )
                $PropertyValue = $PropertyValue.Trim()
                
                Write-Debug "Considering Line: $Line"
                Write-Debug "Considering PropertyValue: $PropertyValue"
                
                if ( $PropertyValue.Length -gt 0 )
                {
                    if ( $Line -match "^\s*base.shutdown.port\s*=\s*.*" )
                    {
                        Write-Debug 'Found base.shutdown.port in properties file.'
                        $PropertiesHash["base.shutdown.port"] = $PropertyValue
                    }
                    elseif ( $Line -match "^\s*nio.http.port\s*=\s*.*" )
                    {
                        Write-Debug 'Found nio.http.port in properties file.'
                        $PropertiesHash["nio.http.port"] = $PropertyValue
                    }
                    elseif ( $Line -match "^\s*base.jmx.port\s*=\s*.*" )
                    {
                        Write-Debug 'Found base.jmx.port in properties file.'
                        $PropertiesHash["base.jmx.port"] = $PropertyValue
                    }
                }
            }
        }
    }

    return $PropertiesHash
}

function MakeBase( [String] $CatalinaHome, [String] $CatalinaBase )
{
    # Create the main catalina base directory.
    New-Item -Force -Path $CatalinaBase -ItemType "directory"
    
    # Create the catalina base subdirectories.
    New-Item -Force -Path $CatalinaBase -ItemType "directory" -Name "bin"
    New-Item -Force -Path $CatalinaBase -ItemType "directory" -Name "conf"
    New-Item -Force -Path $CatalinaBase -ItemType "directory" -Name "lib"
    New-Item -Force -Path $CatalinaBase -ItemType "directory" -Name "logs"
    New-Item -Force -Path $CatalinaBase -ItemType "directory" -Name "temp"
    New-Item -Force -Path $CatalinaBase -ItemType "directory" -Name "webapps"
    New-Item -Force -Path $CatalinaBase -ItemType "directory" -Name "work"
    
    # Copy all the conf directory changes over.
    #
    # Note: There is a bug in makebase.bat where it doesn't handle spaces in
    # the source directory when we copy these files over. So that's why we just
    # avoid using makebase.bat.
    $sourcConfFiles = Join-Path -Path $CatalinaHome -ChildPath 'conf\*'
    $destConfDir = Join-Path -Path $CatalinaBase -ChildPath 'conf'
    Copy-Item -Force -Path $sourcConfFiles -Destination $destConfDir -Recurse
}

function UpdateHttpPort( [String] $CatalinaBase, [String] $HttpPort )
{
    $ServerXmlFilePath = Join-Path -Path $CatalinaBase -ChildPath 'conf\server.xml'
    Write-Debug "Updating http port to $HttpPort in $ServerXmlFilePath."
    $Content = Get-Content $ServerXmlFilePath
    $Content = $Content -replace '8080', $HttpPort
    Set-Content -Path $ServerXmlFilePath -Value $Content
}

function UpdateShutdownPort( [String] $CatalinaBase, [String] $ShutdownPort )
{
    $ServerXmlFilePath = Join-Path -Path $CatalinaBase -ChildPath 'conf\server.xml'
    Write-Debug "Updating shutdown port to $ShutdownPort in $ServerXmlFilePath."
    $Content = Get-Content $ServerXmlFilePath
    $Content = $Content -replace '8005', $ShutdownPort
    Set-Content -Path $ServerXmlFilePath -Value $Content
}

function MakeContextsReloadableByDefault( [String] $CatalinaBase )
{
    $ContextXmlFilePath = Join-Path -Path $CatalinaBase -ChildPath 'conf\context.xml'
    Write-Debug "Making default context reloadable in $ContextXmlFilePath."
    $Content = Get-Content $ContextXmlFilePath
    $Content = $Content -replace '<Context>', '<Context reloadable="true">'
    Set-Content -Path $ContextXmlFilePath -Value $Content
}

function CreateRuntimeCtl( [String] $CatalinaHome, [String] $CatalinaBase, [String] $JavaHomeDir )
{
    Write-Debug "Creating tcruntime-ctl.bat..."
    $runtimeCtlPsFilePath = Join-Path -Path $PSScriptRoot -ChildPath "tcshim.ps1"
    [String] $Contents = -join(
        "@echo off`r`n",
        "`r`n",
        "rem variables for the configuration.`r`n",
        "set CATALINA_HOME=$CatalinaHome`r`n",
        "set CATALINA_BASE=$CatalinaBase`r`n",
        "set JAVA_HOME=$JavaHomeDir`r`n",
        "`r`n",
        "rem Use the shim script to translate over to tomcat.`r`n",
        "powershell -f ""$runtimeCtlPsFilePath"" -action %1"
    )
    
    Write-Debug "Contents: $Contents"
    
    $runtimeCtlBatFilePath = Join-Path -Path $CatalinaBase -ChildPath 'bin\tcruntime-ctl.bat'
    Write-Debug "tcruntime-ctl.bat: $runtimeCtlBatFilePath"
    Set-Content -Path $runtimeCtlBatFilePath -Value $Contents
}

function CreateWrapperConf( [String] $CatalinaHome, [String] $CatalinaBase )
{
    Write-Debug "Creating wrapper.conf..."
    [String] $Contents = -join(
        "#minimal wrapper.conf`r`n",
        "wrapper.java.classpath.1=${CatalinaHome}\bin\bootstrap.jar`r`n",
        "wrapper.java.classpath.2=${CatalinaHome}\bin\tomcat-juli.jar`r`n",
        "wrapper.java.additional.1=""-Djava.endorsed.dirs=%CATALINA_HOME%\common\endorsed""`r`n",
        "wrapper.java.additional.2=""-Dcatalina.base=%CATALINA_BASE%""`r`n",
        "wrapper.java.additional.3=""-Dcatalina.home=%CATALINA_HOME%""`r`n",
        "wrapper.java.additional.4=""-Djava.io.tmpdir=%CATALINA_BASE%\temp""`r`n",
        "wrapper.ntservice.name=%wrapper.ntservice.id%`r`n",
        "wrapper.ntservice.displayname=SASWebAppServer`r`n",
        "wrapper.ntservice.description=SAS Web Application Server at %wrapper.ntservice.id%`r`n"
    )
    
    Write-Debug "Contents: $Contents"
    
    $wrapperConfFilePath = Join-Path -Path $CatalinaBase -ChildPath 'conf\wrapper.conf'
    Write-Debug "wrapper.conf: $wrapperConfFilePath"
    Set-Content -Path $wrapperConfFilePath -Value $Contents
}

function CreateJmxFiles( [String] $CatalinaBase, [String] $JmxPort )
{
    $jmxAccessPath = Join-Path -Path $CatalinaBase -ChildPath 'conf\jmxremote.access'
    $jmxPasswordPath = Join-Path -Path $CatalinaBase -ChildPath 'conf\jmxremote.password'
    $jmxPortPath = Join-Path -Path $CatalinaBase -ChildPath 'conf\jmxremote.port'
    
    # Generate a random JMX password of mixed case letters and numbers
    $jmxPass = -join ((65..90) + (97..122) + (48..57) | Get-Random -Count 32 | % {[char]$_})
    Set-Content -Path $jmxAccessPath -Value "monitorRole readonly"
    Set-Content -Path $jmxPasswordPath -Value "monitorRole ${jmxPass}"
    Set-Content -Path $jmxPortPath -Value $JmxPort

    <#
    $Acl = Get-Acl $jmxPasswordPath

    # Set the owner to the local Administrators group.
    $User = New-Object System.Security.Principal.Ntaccount("NT AUTHORITY\LocalService")
    $Acl.SetOwner($User)

    # Disable inheritance and remove inherited access rules.
    $Acl.SetAccessRuleProtection($true,$false)

    # Allow the LocalService account to read.
    $fileSystemAccessRuleArgumentList = "NT AUTHORITY\LocalService", "Read", "Allow"
    $fileSystemAccessRule = New-Object -TypeName System.Security.AccessControl.FileSystemAccessRule -ArgumentList $fileSystemAccessRuleArgumentList
    $Acl.SetAccessRule($fileSystemAccessRule)

    # Allow local admins full control
    $fileSystemAccessRuleArgumentList = "BUILTIN\Administrators", "FullControl", "Allow"
    $fileSystemAccessRule = New-Object -TypeName System.Security.AccessControl.FileSystemAccessRule -ArgumentList $fileSystemAccessRuleArgumentList
    $Acl.SetAccessRule($fileSystemAccessRule)
 
    # Set the ACL back onto the file.
    Set-Acl -Path $jmxPasswordPath -AclObject $Acl
    #>
}

function UpdateJVMOptions( [String] $TomcatExe, [String] $CatalinaHome, [String] $CatalinaBase, [String] $JavaHomeDir, [String] $Name, [Switch] $NoExecMode )
{
    # TODO: Update options, we could make this smarter and only update if needed.
    [String] $OptionList = GetJVMOptions( $CatalinaBase )
    $OptionList = $OptionList -replace '%CATALINA_HOME%', $CatalinaHome
    $OptionList = $OptionList -replace '%CATALINA_BASE%', $CatalinaBase
    $OptionList = $OptionList -replace '%JAVA_HOME%', $JavaHomeDir
    $OptionList = $OptionList -replace '%JRE_HOME%', $JavaHomeDir

    Write-Output "NOTE: The options are: $OptionList"

    #
    # Read heap properties from wrapper.conf
    #
    $HeapHash = GetHeapProperties ( "${CatalinaBase}\conf\wrapper.conf" )
    $JvmMs = $HeapHash["JvmMs"]
    $JvmMx = $HeapHash["JvmMx"]
        
    [String] $UpdateCommandArgs = "--JvmOptions `"$OptionList`" " +
        "--JvmMs $JvmMs " +
        "--JvmMx $JvmMx "
    $UpdateCommand = "& `"$TomcatExe`" `"//US//$Name`" $UpdateCommandArgs"
    
    if ( $NoExecMode -eq $true )
    {
        Write-Output "Simulated execution of: $UpdateCommand"
    }
    else
    {
        Write-Output "NOTE: UpdateCommand: $UpdateCommand"
        Invoke-Expression -command $UpdateCommand
        $Status = $LastExitCode
        if ( $Status -ne 0 )
        {
            Write-Output "UPDATE returned code $Status"
            Write-Error "ERROR: Failed to update JVM options for the '$Name' service"
        }
        else
        {
            Write-Output "NOTE: The JVM options for the '$Name' service has been updated."
        }
    }
}

function ActionCreate( [String] $CatalinaHome, [String] $CatalinaBase, [String] $JavaHomeDir, [String] $PropertiesFile )
{
    # TODO: Validate that we have a properties file.
            
    # Read instance properties
    $PropertiesHash = GetInstanceProperties( $PropertiesFile )
            
    # Do what makebase.bat would do, but without the bug when there is a space in %CATALINA_HOME%.
    MakeBase $CatalinaHome $CatalinaBase
            
    # Update the HTTP port in server.xml
    UpdateHttpPort $CatalinaBase $PropertiesHash["nio.http.port"]
    
    # Update shutdown port in server.xml
    UpdateShutdownPort $CatalinaBase $PropertiesHash["base.shutdown.port"]
    
    # Make all contexts reloadable by default.
    # Note: This doesn't seem to make all contexts reloadable when a jar changes.
    #MakeContextsReloadableByDefault $CatalinaBase
            
    # Create JMX configuration files.
    CreateJmxFiles $CatalinaBase $PropertiesHash["base.jmx.port"]
            
    # Create a tcruntime-ctl.bat
    CreateRuntimeCtl $CatalinaHome $CatalinaBase $JavaHomeDir
            
    # Create a minimal wrapper.conf
    CreateWrapperConf $CatalinaHome $CatalinaBase
}

function ActionInstall( [String] $CatalinaHome, [String] $CatalinaBase, [String] $JavaHomeDir, [Switch] $NoExecMode )
{
    [String] $OptionList = GetJVMOptions( $CatalinaBase )
    $OptionList = $OptionList -replace '%CATALINA_HOME%', $CatalinaHome
    $OptionList = $OptionList -replace '%CATALINA_BASE%', $CatalinaBase
    $OptionList = $OptionList -replace '%JAVA_HOME%', $JavaHomeDir
    $OptionList = $OptionList -replace '%JRE_HOME%', $JavaHomeDir

    Write-Verbose "Final JVM option string:`n$OptionList`n`n"

    [String] $ClassPath = GetClassPath ( "${CatalinaBase}\conf\wrapper.conf" )
    $ClassPath = $ClassPath -replace '%CATALINA_HOME%', $CatalinaHome
    $ClassPath = $ClassPath -replace '%CATALINA_BASE%', $CatalinaBase
    $ClassPath = $ClassPath -replace '%JAVA_HOME%', $JavaHomeDir
    $ClassPath = $ClassPath -replace '%JRE_HOME%', $JavaHomeDir

    Write-Verbose "Final class path string:`n$ClassPath`n`n"

    #
    # Read service properties from wrapper.conf
    #
    $ServiceHash = GetServiceProperties "${CatalinaBase}\conf\wrapper.conf" $Name
    $Display = $ServiceHash["display"]
    $Description = $ServiceHash["description"]

    #
    # Read heap properties from wrapper.conf
    #
    $HeapHash = GetHeapProperties ( "${CatalinaBase}\conf\wrapper.conf" )
    $JvmMs = $HeapHash["JvmMs"]
    $JvmMx = $HeapHash["JvmMx"]
    
    Write-Debug "Display = $Display"
    Write-Debug "Description = $Description"

    #
    # First check to see if there is an existing service, and remove it if there is.
    #
    $ExistingServices = Get-Service -Name $Name -ErrorAction SilentlyContinue
    if ( $ExistingServices.Length -gt 0 )
    {
        ActionUninstall $TomcatExe $Name $NoExecMode
    }

    #
    # Actually install the service.
    #
    [String] $CommandArgs = "--Install `"$TomcatExe`" " +
                            "--DisplayName `"$Display`" " +
                            "--Description `"$Description`" " +
                            "--LogLevel WARN " +
                            "--LogPath `"$CatalinaBase\logs`" " +
                            "--PidFile tcserver.pid " +
                            "--Environment `"CATALINA_HOME=$CatalinaHome;CATALINA_BASE=$CatalinaBase`" " +
                            "--JavaHome `"$JavaHomeDir`" " +
                            "--Jvm `"$JavaHomeDir\bin\server\jvm.dll`" " +
                            "--StartMode jvm " +
                            "--StopMode jvm " +
                            "--StdOutput auto " +
                            "--StdError auto " +
                            "--Startup auto " +
                            "--StartPath `"$CatalinaHome`" " +
                            "--StopPath `"$CatalinaHome`" " +
                            "--StartClass org.apache.catalina.startup.Bootstrap " +
                            "--StopClass org.apache.catalina.startup.Bootstrap " +
                            "--StartParams start " +
                            "--StopParams stop " +
                            "--Classpath `"$ClassPath`" " +
                            "--JvmOptions `"$OptionList`" " +
                            "--JvmOptions9 `"--add-opens=java.base/java.lang=ALL-UNNAMED#--add-opens=java.base/java.io=ALL-UNNAMED#--add-opens=java.rmi/sun.rmi.transport=ALL-UNNAMED`" "  +
                            "--JvmMs $JvmMs " +
                            "--JvmMx $JvmMx "

    $Command = "& `"$TomcatExe`" `"//IS//$Name`" $CommandArgs"
    if ( $NoExecMode -eq $true )
    {
        Write-Output "Simulated execution of:"
        ( $Command -split " --" ) -join "`r`n--"
    }

    else
    {
        Invoke-Expression -command $Command
        $Status = $LastExitCode
        if ( $Status -ne 0 )
        {
            Write-Output "INSTALL returned code $Status"
            Write-Error "ERROR: Failed to install the '$Name' service"
        }

        else
        {
            Write-Output "NOTE: The '$Name' service has been installed."
        }
    }
}

function ActionUninstall( [String] $TomcatExe, [String] $Name, [Switch] $NoExecMode )
{
    $Command = "& `"$TomcatExe`" `"//DS//$Name`""
    if ( $NoExecMode -eq $true )
    {
        Write-Output "Simulated execution of: $Command"
    }

    else
    {
        Invoke-Expression -command $Command
        $Status = $LastExitCode
        if ( $Status -ne 0 )
        {
            Write-Output "UNINSTALL returned code $Status"
            Write-Error "ERROR: Failed to uninstall the '$Name' service"
        }

        else
        {
            Write-Output "NOTE: The '$Name' service has been uninstalled."
        }
    }
}

function ActionStart( [String] $TomcatExe, [String] $CatalinaHome, [String] $CatalinaBase, [String] $JavaHomeDir, [String] $Name, [Switch] $NoExecMode )
{    
    $Command = "Start-Service -Name $Name -ErrorAction `"Continue`" -ErrorVariable ErrorStatus"
    if ( $NoExecMode -eq $true )
    {
        Write-Output "Simulated execution of: $Command"
    }

    else
    {
        # wrapper.conf may have been updated, so check for changes and update the service definition.
        UpdateJVMOptions $TomcatExe $CatalinaHome $CatalinaBase $JavaHomeDir $Name $NoExecMode
        
        # Now actually start the service.
        Invoke-Expression -command $Command

        if ( $ErrorStatus.Count -ne 0 )
        {
            Write-Error "ERROR: Failed to start the '$Name' service"
        }

        else
        {
            Write-Output "NOTE: The '$Name' service has been started."
        }
    }
}

function ActionStop( [String] $Name, [Switch] $NoExecMode )
{
    $Command = "Stop-Service -Name $Name -ErrorAction `"Continue`" -ErrorVariable ErrorStatus"
    if ( $NoExecMode -eq $true )
    {
        Write-Output "Simulated execution of: $Command"
    }

    else
    {
        Invoke-Expression -command $Command

        if ( $ErrorStatus.Count -ne 0 )
        {
            Write-Error "ERROR: Failed to stop the '$Name' service"
        }

        else
        {
            Write-Output "NOTE: The '$Name' service has been stopped."
        }
    }
}

function Main()
{
    $InitialDebugPreference = $DebugPreference
    $InitialVerbosePreference = $VerbosePreference
    $InitialErrorActionPreference = $ErrorActionPreference

    $ErrorActionPreference = "Stop"

    $NoExecMode = $false
    [String] $JavaHomeDir = ""
    [String] $TomcatExe = ""
    $Status = 0
    [String] $Display = ""
    [String] $Description = ""
    


    #
    # Report message and execution options if they are used.
    #
    if ( $PSCmdlet.MyInvocation.BoundParameters['Debug'].IsPresent )
    {
        $DebugPreference = 'Continue'
        Write-Debug 'Debug mode is enabled'
    }

    if ( $PSCmdlet.MyInvocation.BoundParameters['Verbose'].IsPresent )
    {
        $VerbosePreference = 'Continue'
        Write-Verbose 'Verbose mode is enabled'
    }

    if ( $NoExec.IsPresent )
    {
        Write-Verbose 'NoExec mode is enabled'
        $NoExecMode = $true
    }

    #
    # Standardize the action command to upper case.
    #
    $Action = $Action.ToUpper()
    
    #
    # Ensure CATALINA_BASE is defined and exists.
    #
    [String] $CatalinaBase = ""
    if ( ( $Action -eq "CREATE" ) )
    {
        $CatalinaBase = GetEnvPath( "CATALINA_BASE" )
    }
    else
    {
        $CatalinaBase = ValidateEnvPath( "CATALINA_BASE" )
    }
    Write-Debug "CATALINA_BASE = $CatalinaBase"
    
    #
    # Generate the name from catalina base like tc server used to.
    #
    [String] $GenName = $CatalinaBase -replace '\\', '-'
    $GenName = $GenName -replace '/', '-'
    $GenName = $GenName -replace ' ', '-'
    $GenName = $GenName -replace ':', ''
    $GenName = "tcruntime-$GenName"
    [String] $Name = $GenName

    Write-Debug "Execution directory = $PSScriptRoot"
    Write-Debug "Action = $Action"
    Write-Debug "Name = $Name"


    if ( ( $Action -eq "CREATE" ) -or ( $Action -eq "INSTALL" ) -or ( $ACTION -eq "UNINSTALL") -or ( $Action -eq "START" ) )
    {
        #
        # Ensure CATALINA_HOME is defined and exists.
        #
        [String] $CatalinaHome = ValidateEnvPath( "CATALINA_HOME" )
        Write-Debug "CATALINA_HOME = $CatalinaHome"

        #
        # Ensure the Tomcat utility for Windows Services exists.
        #
        $TomcatExe = "$CatalinaHome\bin\$TOMCAT_EXECUTABLE_NAME"
        if ( (Test-Path -Path $TomcatExe -PathType Leaf) -eq $false )
        {
            Write-Error -Message "ERROR: File not found: $TomcatExe" `
                        -Category ObjectNotFound
        }
        Write-Debug "Tomcat executable = $TomcatExe"
    }

    if ( ($Action -eq "CREATE") -or ($Action -eq "INSTALL") -or ($Action -eq "START") )
    {
        #
        # Ensure JAVA is defined and exists.
        #
        [String] $JavaHomeEnv = $Env:JAVA_HOME
        if ( $JavaHomeEnv.Length -eq 0 )
        {
            Write-Error -Message "ERROR: The JAVA_HOME environment variable is not defined and is required." `
            -Category ObjectNotFound
        }

        else
        {
            Write-Debug "JAVA_HOME = $JavaHomeEnv"
            $JavaHomeDir = $JavaHomeEnv
        }

        $FilePath = "$JavaHomeDir\bin\java.exe"
        if ( (Test-Path -Path $FilePath -PathType Leaf) -eq $false )
        {
            Write-Error -Message "ERROR: File not found: $FilePath" `
                        -Category ObjectNotFound
        }

        $FilePath = "$JavaHomeDir\bin\server\jvm.dll"
        if ( (Test-Path -Path $FilePath -PathType Leaf) -eq $false )
        {
            Write-Error -Message "ERROR: File not found: $FilePath" `
                        -Category ObjectNotFound
        }
    }


    #
    # Perform the specified ACTION.
    #
    switch ( $Action )
    {
        "CREATE"
        {
            ActionCreate $CatalinaHome $CatalinaBase $JavaHomeDir $PropertiesFile
        }
        
        "INSTALL"
        {
            ActionInstall $CatalinaHome $CatalinaBase $JavaHomeDir $NoExecMode
        }

        "UNINSTALL"
        {
            ActionUninstall $TomcatExe $Name $NoExecMode
        }

        "START"
        {
            ActionStart $TomcatExe $CatalinaHome $CatalinaBase $JavaHomeDir $Name $NoExecMode
        }

        "STOP"
        {
            ActionStop $Name $NoExecMode
        }

        # Should never happen
        Default { Write-Error "Invalid action specified: $Action" }
    }

    $DebugPreference = $InitialDebugPreference
    $VerbosePreference = $InitialVerbosePreference
    $ErrorActionPreference = $InitialErrorActionPreference

    exit $Status
}

#
# Invoke the main funciton for the script.
#
Main
