/*
 Copyright (C) 2020-2023 SAS Institute Inc. Cary, NC, USA
*/




/**
   \file 
\anchor irm_rest_get_ticket

   \brief   Authenticate and retrieve IRM REST API ticket using Central Authentication Service (CAS)

   \param [in] url Full url to the REST resource
   \param [in] LogonPort (Optional) port where the /SASLogon web application is listening. If blank, it is assumed that the /SASLogon is on the same port as /SASIRMServer (Default: blank)
   \param [in] username Username credentials.
   \param [in] password Password credentials: it can be plain text or SAS-Encoded (it will be masked during execution). If both username and password are left blank, then the system assumes single sign-on is configured
   \param [in] ticketType Controls which type of ticket is returned: Service Ticket (ST) or Ticket Granting Ticket (TGT). (Default: ST)
   \param [in] tgt_ticket Ticket granting ticket. If ticketType = ST and this parameter is not blank, this TGT ticket is used to obtain a Service ticket for the requested REST resource
   \param [in] debug True/False. If True, debugging informations are printed to the log (Default: false)
   \param [in] logOptions Logging options (i.e. mprint mlogic symbolgen ...)
   \param [in] server <b><i>DEPRECATED</i></b> Name of the Web Application Server that provides the REST service (Default: SASIRMServer)
   \param [out] outVarTicket Name of the output macro variable which will contain the Service Ticket (Default: service_ticket)


   \details
   The Service Ticket is an authorizing ticket which allows a user to login into the application having \<Service URL\> as a target destination.
   This is a ticket obtained by explicitly providing user credentials in some form (REST API/Java Service Registry - AuthenticationClient & AuthenticationClientHolder)
   \n

   This macro performs the following operations:
      1. Initially use the SASLogon web application to acquire a generic Ticket Granting Ticket (TGT): this step is skipped if <i>ticketType = ST</i> and a TGT ticket has been provided with parameter <i>tgt_ticket</i>.
      2.1 If <i>ticketType = TGT</i>, end execution and return the TGT ticket.
      2.2 If <i>ticketType = ST</i>, use the TGT to request a Service Ticket (ST) in order to access a specific SAS REST service API.
   \n \n

   Once a Service Ticket for a given <i>\<URL\></i> has been acquired, the target resource can be obtained by sending a http request to <i>\<URL\><b>?&amp;ticket=\<Service Ticket\></b></i>.

   <b>Example:</b>

   1) Set up the environment (set SASAUTOS and required LUA libraries)
   \code
      %let source_path = <Path to the root folder of the Federated Content Area (root folder, excluding the Federated Content folder)>;
      %let fa_id = <Name of the Federated Area Content folder>;
      %include "&source_path./&fa_id./source/sas/ucmacros/irm_setup.sas";
      %irm_setup(source_path = &source_path.
                , fa_id = &fa_id.
                );
   \endcode

   2) Get the service ticket for a specific resource
   \code
      %let service_ticket =;
      %let url = <protocol>://<host>:<port>/SASIRMServer/rest/basedates;
      %irm_rest_get_ticket(url = &url.
                           , username = <userid>
                           , password = <pwd>
                           , outVarTicket = service_ticket
                           );
      %put service_ticket = &service_ticket.;
   \endcode

   3) then use the ticket to request the resource and print the result to the log
   \code
      filename out temp;
      proc http
         url = "&url.?%str(&)ticket=&service_ticket."
         out = out
         method = 'GET'
         ct = 'application/json';
      run;

      data _null_;
         infile out;
         input;
         put _infile_;
      run;

      filename out clear;
   \endcode

   <b>Sample output:</b>
   \code
      [
        {
          "current": true,
          "id": "03312015",
          "version": 1,
          "name": "03312015",
          "key": 1
        },
        {
          "current": false,
          "id": "03312014",
          "version": 1,
          "name": "03312014",
          "key": 2
        }
      ]
   \endcode

   \note The client needs to acquire a ST for each different service URL (not simply for each service context or resource path). \n
          Normally, CAS service tickets are <b>one-time-use only</b>, but for the SAS implementation, a ST can be reused when accessing the same URL.


   \ingroup macroUtils

   \author  SAS Institute Inc.
   \date    2015
*/

%macro irm_rest_get_ticket(url =
                           , LogonPort =
                           , username = %nrstr()
                           , password = %nrstr()
                           , ticketType = ST
                           , tgt_ticket =
                           , outVarTicket = service_ticket
                           , debug = false
                           , logOptions =
                           , server = SASIRMServer /* Deprecated: here for backward compatibility */
                           );

   %local
      auth_negotiate_flg
      root_url
      oldnotes
      oldsgen
      oldmprint
      oldmlogic
      i
      content_type
      service_url
      regex_url
      protocol
      port
      ;

   /* Set the required log options */
   %if(%length(&logOptions.)) %then
      options &logOptions.;
   ;

   %let oldnotes = %sysfunc(getoption(notes));
   %let oldsgen = %sysfunc(getoption(symbolgen));
   %let oldmprint = %sysfunc(getoption(mprint));
   %let oldmlogic = %sysfunc(getoption(mlogic));

   /* Check the ticketType parameter and assign default value if missing */
   %if(%sysevalf(%superq(ticketType)=, boolean)) %then
      %let ticketType = ST;
   %else
      %let ticketType = %upcase(&ticketType.);

   /* Check if username and password have been provided. If not, then single sign-on is assumed */
   %let auth_negotiate_flg = N;
   %if(%sysevalf(%superq(username)=, boolean) or "%superq(password)" = "") %then
      %let auth_negotiate_flg = Y;

   /* Use regex to remove any double forward slash */
   %let url = %qsysfunc(prxchange(s/(?<!:)\/+/\//, -1, %superq(url)));

   /* Use regex to extract the protocol and host from the url parameter */
   %if(%length(&LogonPort.) > 0) %then
      %let root_url = %sysfunc(prxchange(s/(https?:\/\/[^:\/]+)((:\d+)|\/|$)(.*)/$1:&LogonPort./i, -1, %superq(url)));
   %else
      /* &LogonPort is missing: assume that the /SASLogon web application is on the same port of /SASIRMServer */
      %let root_url = %sysfunc(prxchange(s/(https?:\/\/[^:\/]+)((:\d+)|\/|$)(.*)/$1$3/i, -1, %superq(url)));

   %if (%upcase(&DEBUG.) EQ TRUE) %then %do;
      %put Server URL: &root_url.;
   %end;

   /* Declare the output variable as global if it does not exist */
   %if not %symexist(&outVarTicket.) %then
      %global &outVarTicket.;
   ;

   /* Declare temporary filerefs */
   filename hin  temp;
   filename hout temp;
   filename in   temp;
   filename out  temp;

   /* Use regex to remove any double forward slash */
   %let service_url = %qsysfunc(prxchange(s/(?<!:)\/+/\//, -1, %superq(url)));

   %let content_type = application/x-www-form-urlencoded;
   %if %substr(&sysvlong.,%index(&sysvlong., M)+1,1) > 7 %then  %do;
      /* Set the regular expression to extract protocol, host and port from the url */
      %let regex_url = ((http[s]?):\/\/)?([^:\/]+)((:(\d+))|\/|$)(.*);
      /* Extract the protocol */
      %let protocol = %sysfunc(prxchange(s/&regex_url./\L$2/i, -1, %superq(service_url)));
      /* Extract the port */
      %let port = %sysfunc(prxchange(s/&regex_url./$6/i, -1, %superq(service_url)));
      /* M8 ret api request are suppressing the default ports */
      %if (&port. = 80 and &protocol. = http) or (&port. = 443 and &protocol. = https) %then %do;
         %let service_url = %qsysfunc(prxchange(s/(?<=)(:80)(?=\/)|(?<=)(:443)(?=\/)//, 1, %superq(service_url)));
      %end;  
   %end;

   /* Request Header */
   data _null_;
      file hin;
      put "Accept:&content_type.";
   run;

   /* If the TGT has been provided as an input then jump to the ST section of this macro */
   %if (&ticketType. = ST and %sysevalf(%superq(tgt_ticket) ne, boolean)) %then %do;
      %let &outVarTicket. = &tgt_ticket.;
      %goto ST_TICKET;
   %end;

   /* ************************************************************** */
   /*               Get Ticket Granting Ticket                       */
   /* ************************************************************** */

   %if(&auth_negotiate_flg. = N) %then %do;
      /* Suppressing temporarily the printing options to the log */
      option nonotes nosgen nomprint nomlogic;
   %end;

   /* Get the TGT */
   proc http
         headerin  = hin
         headerout = hout
         method    = 'post'
         ct        = "&content_type."
         %if(&auth_negotiate_flg. = Y) %then %do;
            /* Negotiate authentication (single sign-on) */
            url       = %str("&root_url./SASLogon/login")
            auth_negotiate
         %end;
         %else %do;
            /* Use username and password to login */
            url    = %str("&root_url./SASLogon/v1/tickets")
            in     = "username=%bquote(&username.)%nrstr(&)password=%sysfunc(urlencode(%bquote(&password.)))"
         %end;
         /* Clear both the shared connection and cookie caches before the HTTP request is executed */
         clear_cache
         ;
   run;

   /* Reactivate the log options */
   option &oldnotes. &oldsgen. &oldmprint. &oldmlogic.;

   /* Parse the response for the TGT ticket */
   data _null_;
      infile hout;
      input;
      length tgt $2048.;
      retain rx;
      if _N_ = 1 then
         %if(&auth_negotiate_flg. = Y) %then %do;
            /* Extract the TGT ticket from the cookie */
            rx = prxparse("s/^Set-Cookie: CASTGC=(.*?);.*\s*/$1/i");
         %end;
         %else %do;
            /* Extract the TGT ticket from the Location */
            rx = prxparse("s/^Location:\s*(.*)\/SASLogon\/v1\/tickets\///i");
         %end;

      if prxmatch(rx, _infile_) then do;
         /* Extract the TGT ticket */
         tgt = prxchange(rx, -1, _infile_);
         call symput("&outVarTicket.", strip(tgt));
         stop;
      end;
   run;

   %if (%upcase(&DEBUG.) EQ TRUE) %then %do;
      %put -----------------------------------------------;
      %put Ticket Granting Ticket:;
      %put  --> &outVarTicket = &&&outVarTicket..;
      %put -----------------------------------------------;
   %end;

   /* Check if we got the TGT ticket */
   %if %index(&&&outVarTicket., %str(TGT-)) EQ 0 %then %do;
      %put ERROR: Failed to get ticket grant ticket.;
      %let &outVarTicket.=;
      %return;
   %end;

   /* If ticketType = TGT we are done. Skipping the rest of the execution */
   %if(&ticketType. = TGT) %then
      %goto EXIT;

%ST_TICKET:

   /* ************************************************************** */
   /*                  Get Service Ticket                            */
   /* ************************************************************** */
   
   /* Request body: the service for which the ticket is required */
   data _null_;
      file in recfm = f lrecl = 1;
      put "service=%sysfunc(urlencode(&service_url.))";
   run;

   /* Get the service ticket */
   proc http
         url       = %str("%trim(&root_url./SASLogon/v1/tickets/&&&outVarTicket.)")
         headerin  = hin
         headerout = hout
         in        = in
         out       = out
         method    = 'post'
         ct        = "&content_type."
         ;
   run;

   /* Read the Service Ticket from the response */
   data _null_;
      infile out;
      input;
      call symput("&outVarTicket.", _infile_);
   run;

   %if (%upcase(&DEBUG.) EQ TRUE) %then %do;
      %put -----------------------------------------------;
      %put Service Ticket:;
      %put  --> &outVarTicket = &&&outVarTicket..;
      %put -----------------------------------------------;
   %end;

   /* Check if we got the ST ticket */
   %if %index(&&&outVarTicket.., %str(ST-)) NE 1 %then %do;
      %put ERROR: Failed to get service ticket.;
      %return;
   %end;

%EXIT:

   /* Cleanup */
   filename in clear;
   filename out clear;
   filename hin clear;
   filename hout clear;

%mend;
