#
# Copyright © 2017 Pivotal Software, Inc. All rights reserved.
#
# httpdctl.ps1  This script takes care of installing, starting and stopping
#               Pivotal Web Server services on the Windows platform, following
#               the same syntax as the unix httpdctl script.
#
# Be absolutely certain to save this file in Encoding 'UTF-8'.


function IsFIPSEnabled ( [String] $SSLConfigFileName )
{
   [Boolean] $Enabled = $false

   if ( (Test-Path -Path $SSLConfigFileName -PathType Leaf) -eq $false )
   {
       Write-Error -Message "ERROR: File not found: $ConfigFileName" `
                   -Category ObjectNotFound
   }

   else
   {
      Write-Debug "SSLConfigFileName = $SSLConfigFileName"

      foreach ( $Line in ( Get-Content $SSLConfigFileName ) )
      {
         if ( $Line -match "^\s*SSLFIPS\s+ON" )
         {
            $Enabled = $true
            break
         }
      }
   }

   return $Enabled
}


$name = "httpdctl"
$OutputEncoding = [System.Text.Encoding]::UTF8

$server_root="@ServerRoot@"
$apache_root="@exp_httpddir@"
$apache_bin="$apache_root/bin/httpd.exe"
$apache_pid="$server_root/logs/httpd.pid"
$service_name="Pivotal httpd @ServerInstance@"
$action="help"
$start_flags=""

Write-Host "httpdctl.ps1 - manage the $service_name server instance"
Write-Host "Copyright © 2017 Pivotal Software, Inc. All rights reserved."
Write-Host ""

if ($args.Count -gt 0) {
    $action=$args[0]
    if ($args.Count -gt 1) {
        $start_flags = $args[1..($args.Count - 1)]
        for ($i = 0; $i -lt $start_flags.Count; ++$i) {
            if ($start_flags[$i].IndexOf(" ") -ge 0) {
                $start_flags[$i] = '"' + $start_flags[$i] + '"'
            }
        }
        $start_flags = [string]$start_flags
    }
}

if (-not (Test-Path "$server_root/conf/httpd.conf")) {
    Write-Host "FATAL: $server_root/conf/httpd.conf not found"
    exit 1
}

if ($action -eq "start") {
    Write-Host "Starting $service_name"
    & $apache_bin -k start -n "$service_name" -d "$server_root" $start_flags
    $rv = $LastExitCode
    if ($?) {
        Write-Host "The $service_name service is running."
    }
    exit $rv
}
elseif ($action -eq "stop") {
    Write-Host "Stopping $service_name"
    & $apache_bin -k stop -n "$service_name"
    exit $LastExitCode
}
elseif ($action -eq "restart") {
    Write-Host "Restarting $service_name"
    & $apache_bin -k stop -n "$service_name"
    & $apache_bin -k start -n "$service_name" -d "$server_root" $start_flags
    $rv = $LastExitCode
    if ($?) {
        Write-Host "The $service_name service is running."
    }
    exit $rv
}
elseif ($action -eq "graceful") {
    Write-Host "Gracefully restarting $service_name"
    & $apache_bin -k restart -n "$service_name"
    exit $LastExitCode
}
elseif ($action -eq "status") {
    $env:RUNNING = 0
    if (Test-Path $apache_pid) {
        $env:PID = [Int64](get-content $apache_pid)
        $apid = "pid $env:PID"
    }
    else {
        $apid = "no pid file"
    }

    # KLUDGE: Variable 'service_name' contains the display-name for the service.
    # The actual service-name as used by Windows is the display name with all of
    # the whitespace removed.  In addition, the Get-Service cmdlet treats square
    # brackets as wildcard syntax and they cannot be escaped or the behavior
    # overridden, so to work the characters have to be replaced with ? wildcards.
    $internal_service_name = $service_name -replace "(\[|\])","?" -replace ' ', ''
    $scmstatus = (Get-Service $internal_service_name).Status.ToString().ToUpper()
    if ($scmstatus -eq "STOPPED") { $scmstatus = "NOT RUNNING" }
    if ($scmstatus -eq "RUNNING") { $env:RUNNING = 1 }
    $env:STATUS = "$service_name ($apid) $scmstatus"
    Write-Host $env:STATUS
}
elseif ($action -eq "configtest") {
    Write-Host "Configuration test results for $service_name"
    & $apache_bin -t -n "$service_name" -d "$server_root" $start_flags
    exit $LastExitCode
}
elseif ($action -eq "install") {
    Write-Host "Installing Windows service $service_name"
    & $apache_bin -k install -n "$service_name" -d "$server_root" $start_flags
    $rv = $LastExitCode
    if (! $?) {
        Write-Host ""
        Write-Host "FATAL: The $service_name service failed to install."
        exit $rv
    }

    #
    # Add environment variables if FIPS is enabled
    #
    $ssl_config_file = "$server_root/conf/extra/httpd-ssl.conf"
    if ( -not ( Test-Path $ssl_config_file ) ) {
        Write-Error -Message "FATAL: $ssl_config_file not found" `
                    -Category ObjectNotFound
        exit 1
    }
    else {
        if ( IsFIPSEnabled ( $ssl_config_file ) == $true ) {
            Write-Host "FIPS is enabled.  Setting service environment variables."
            # KLUDGE: Variable 'service_name' contains the display-name for the service.
            # The actual service-name as used by Windows is the display name with all of
            # the whitespace removed.
            $internal_service_name = $service_name -replace '\s'
            $ScriptPath = [System.IO.Path]::GetDirectoryName($myInvocation.MyCommand.Definition)
            . $ScriptPath/OpenSSLEnv.ps1 -SASServiceName "$internal_service_name" -OpenSSLConfigFile $server_root/conf/extra/openssl.cnf -OpenSSLProviderPath $apache_root/bin
            $rv = $LastExitCode
            if (! $?) {
                Write-Host ""
                Write-Host "FATAL: Failed to define environment variables for $service_name."
                exit $rv
            }
        }
    }

    Write-Host ""
    Write-Host "You should consider changing the 'Log On' system account property for"
    Write-Host "this $service_name service to a unique, unprivileged user account"
    Write-Host "to finish this configuration, before starting the service."
    Write-Host "The unprivileged user account should have write access only to the logs"
    Write-Host "and var directories within $server_root"
    Write-Host ""
}
elseif ($action -eq "uninstall") {
    Write-Host "Uninstalling Windows service $service_name"
    & $apache_bin -k stop -n "$service_name"
    & $apache_bin -k uninstall -n "$service_name"
    exit $LastExitCode
}
else {
    Write-Host "Usage: httpdctl.ps1 {command} {optional arguments}"
    Write-Host ""
    Write-Host "where {command} is one of:"
    Write-Host "    start"
    Write-Host "    stop"
    Write-Host "    restart"
    Write-Host "    graceful"
    Write-Host "    status"
    Write-Host "    configtest"
    Write-Host "    install"
    Write-Host "    uninstall"
    Write-Host ""
    if ($action -ne "help") {
        Write-Host "FATAL: Command $action was not recognized!"
        Write-Host ""
        exit 1
    }
}
