# ------------------------------------------------------------------------------
#         Copyright (c) 2022 by SAS Institute Inc., Cary, NC USA 27513
# ------------------------------------------------------------------------------
#
# PUROPSE:  Add an "Environment" property to the Windows Service definition for
#           the SAS Web Server, and populate that property with values that
#           define the location of modules and configuration files needed for
#           OpenSSL-3 to support FIPS.
#
# NOTES:    WARNING: This script should only be invoked when the web server is
#           configured to use SSL/TLS and FIPS certified encryption algorithms.
#
# ------------------------------------------------------------------------------

<#
.SYNOPSIS

Add an "Environment" property to the Windows Service definition for the SAS Web
Server, and populate that property with values that define the location of
modules and configuration files needed for OpenSSL-3 to support FIPS.

WARNING: This script should only be invoked when the web server is configured to
use SSL/TLS and FIPS certified encryption algorithms.


.DESCRIPTION
The SAS Web Server can be executed as a Windows service, allowing it to run
without anyone having to login to the machine.  After the Windows service that
hosts the web server has been created, this utility is used to define a set of
environment variables that exist within the service's run-time environment.

From the OpenSSL documentation for openssl-env found at:
https://www.openssl.org/docs/manmaster/man7/openssl-env.html

   OPENSSL_CONF, OPENSSL_CONF_INCLUDE
      Specifies the path to a configuration file and the directory
      for included files.

   OPENSSL_MODULES
      Specifies the directory from which cryptographic providers
       are loaded.

A web server configured to use OpenSSL-3 for FIPS support requires that the
standard OpenSSL configuration file (openssl.cnf) be updated.  However,
modifying files in the SASHome area is not allowed, so the file is copied to the
'conf' directory for the SAS Web Server instance in SASConfig, updated, and used
from there. The OPENSSL_CONF environment variable defines the location of the
configuration file that OpenSSL is to use when not using the default one in
SASHome.

When OpenSSL is configured to load "provider" modules like the one that provides
the FIPS capabilities, it must be able to locate the required modules.  The
OPENSSL_MODULES environment variable defines the location of those provider
modules.

.PARAMETER SASServiceName
REQUIRED - The name of the Windows service to operate on.

WARNING: This argument takes the actual "Service Name" as shown on the
Properties dialog for the service in the Windows Services control panel.  This
is not the same as the "Display Name" shown on the service-list pane of the
Services control panel.

.PARAMETER OpenSSLConfigFile
REQUIRED - Path and file name of the OpenSSL configuration file.

.PARAMETER OpenSSLProviderPath
REQUIRED - Path to the OpenSSL provider modules.

.PARAMETER Debug
OPTIONAL - Output additional messages showing run-time parameters and values.

.PARAMETER Verbose
OPTIONAL - Output additional messages showing more in-depth information that would
normally be shown.

.EXAMPLE
OpenSSLEnv.ps1 -SASServiceName SAS[Config-Lev1]httpd-WebServer -OpenSSLConfigFile E:\SASConfig\Lev1\Web\WebServer\conf\extra\openssl.cnf -OpenSSLProviderPath E:\SASHome\SASWebServer\9.4\httpd-2.4.54\bin

.EXAMPLE
OpenSSLEnv.ps1 -SASServiceName "SAS [Config-Lev2] httpd-WebServer" -OpenSSLConfigFile "C:\Program Files\SASConfig\Lev2\Web\WebServer\conf\extra\openssl.cnf" -OpenSSLProviderPath "C:\Program Files\SASHome\SASWebServer\9.4\httpd-2.4.54\bin"
#>



[ CmdletBinding ( PositionalBinding = $false ) ]

Param ( [Parameter ( Mandatory = $true,
                     HelpMessage = "(Required) Name of the service to operate on."
                   )
        ]
        [ValidateNotNullOrEmpty()]
        [String] $SASServiceName,
        # ----------------------------------------------------------------------
        [Parameter ( Mandatory = $true,
                     HelpMessage = "(Required) Path and file name of the OpenSSL configuration file."
                   )
        ]
        [String] $OpenSSLConfigFile,
        # ----------------------------------------------------------------------
        [Parameter ( Mandatory = $true,
                     HelpMessage = "(Required) Path to the OpenSSL provider modules"
                   )
        ]
        [String] $OpenSSLProviderPath
      )



function ValidatePath ( [String] $DirPath )
{
    if ( ( Test-Path -Path $DirPath -PathType Container ) -eq $false )
    {
        Write-Error -Message "ERROR: The specified directory was not found: $DirPath" `
                    -Category ObjectNotFound
    }
}



function ValidateFile ( [String] $FilePath )
{
    if ( ( Test-Path -Path $FilePath -PathType leaf ) -eq $false )
    {
        Write-Error -Message "ERROR: The specified file was not found: $FilePath" `
                    -Category ObjectNotFound
    }
}



function Main()
{
   $InitialDebugPreference = $DebugPreference
   $InitialVerbosePreference = $VerbosePreference
   $InitialErrorActionPreference = $ErrorActionPreference

   if ( $PSCmdlet.MyInvocation.BoundParameters['Debug'].IsPresent )
   {
      $DebugPreference = 'Continue'
   }

   if ( $PSCmdlet.MyInvocation.BoundParameters['Verbose'].IsPresent )
   {
      $VerbosePreference = 'Continue'
   }

   $ErrorActionPreference = "Stop"

   # NOTE: Registry key path must use back-slash not forward-slash.
   $RegistryPath = "Registry::HKEY_LOCAL_MACHINE\SYSTEM\ControlSet001\Services\$SASServiceName"
   $KeyName = "Environment"
   $KeyValue = "OPENSSL_CONF=$OpenSSLConfigFile",
               "OPENSSL_MODULES=$OpenSSLProviderPath"

   Write-Debug "Execution directory = $PSScriptRoot"
   Write-Debug "SASServiceName = $SASServiceName"
   Write-Debug "RegistryPath = $RegistryPath"
   Write-Debug "KeyName = $KeyName"
   Write-Debug "KeyValue = $KeyValue"


   ValidateFile ( $OpenSSLConfigFile )
   ValidatePath ( $OpenSSLProviderPath )


   if ( ! ( Test-Path -LiteralPath $RegistryPath ) )
   {
      Write-Error -Message "ERROR: Service registry key not found: $RegistryPath" `
                  -Category ObjectNotFound

   }

   else
   {
      Write-Debug "Found registry path entry: $RegistryPath"
      New-ItemProperty -LiteralPath $RegistryPath -Name $KeyName -Value $KeyValue `
                       -PropertyType MultiString `
                       -Force | Out-Null
      Write-Verbose "Property `"$KeyName`"`nin $RegistryPath`nset to $KeyValue"
   }


   $DebugPreference = $InitialDebugPreference
   $VerbosePreference = $InitialVerbosePreference
   $ErrorActionPreference = $InitialErrorActionPreference
}


#
# Invoke the main funciton for the script.
#
Main