sap.ui.define([
    "sap/m/InputType",
    "sap/ui/core/ValueState",
    "sas/hc/m/Input",
    "sas/hc/ui/core/ListItem",
    "sas/ltjs/BIRD/models/common/constraints/NumericConstraint",
    "sas/vaviewer/views/prompts/PromptElementView",
    "sas/ltjs/commons/util/TextUtil",
    "sas/vaviewer/views/prompts/TextFieldFormatter",
    "sas/hc/m/RangeValidationAdapter",
    "sas/hc/m/InputNumberFormatter"
], function (
    InputType,
    ValueState,
    Input,
    ListItem,
    NumericConstraint,
    PromptElementView,
    TextUtil,
    TextFieldFormatter,
    RangeValidationAdapter,
    InputNumberFormatter
) {
    "use strict";
    var rb = sap.ui.getCore().getLibraryResourceBundle("sas.vaviewer");
    var formatFloatInstance = sap.ui.core.format.NumberFormat.getFloatInstance();

    return PromptElementView.extend("sas.vaviewer.views.prompts.TextFieldTypeAhead", {
        metadata: {
            properties: {
                autoSize: {type: "boolean", defaultValue:false}
            }
        },

        renderer: {},

        init: function () {
            PromptElementView.prototype.init.apply(this, arguments);
            var input = new Input({
                // our inputChangeHandler should only fire on input that passes validation
                fireChangeOnError: false,
                change: [this._inputChangeHandler, this],
                // Let input text suggestion popup extend wider than the input.
                wrapSuggestionPopupContent: false
            });
            input.addEventDelegate({
                onAfterRendering: function() {
                    if(this.getAutoSize()) {
                        var data = this.getControllerData().rowData;
                        var inputRef = input.getDomRef();
                        var innerRef = input.getDomRef("inner");
                        if(inputRef && innerRef) {
                            var innerStyle = getComputedStyle(innerRef);
                            var supplementalWidth = 0;
                            if(innerStyle.boxSizing === "border-box") {
                                supplementalWidth = parseFloat(innerStyle.borderLeftWidth) + parseFloat(innerStyle.borderLeftWidth) + parseFloat(innerStyle.paddingLeft) + parseFloat(innerStyle.paddingRight);
                            }
                            var maxWidth = parseFloat(getComputedStyle(inputRef).maxWidth) - supplementalWidth;
                            var width = 0;
                            if(data && data.length) {
                                width = TextUtil.getTextWidthFromComputedStyle(data, innerStyle, maxWidth, function(dataItem){
                                    return dataItem.value;
                                });
                            }
                            width = Math.max(width, TextUtil.getTextWidthFromComputedStyle(input.getPlaceholder(), innerStyle, maxWidth));
                            innerRef.style.width = Math.ceil(width) + supplementalWidth + "px"; //Round up. Generously accounting for floating point errors.
                        }
                    }
                }
            }, this);
            input.setWidth('100%');
            this.addChild(input);
            this.addStyleClass('TextInput');

            var oldSetShowSuggestion = input.setShowSuggestion;
            var self = this;
            input.setShowSuggestion = function(bValue){
                var wasSuggestionPopupCreated = bValue && !this._oSuggestionPopup;

                oldSetShowSuggestion.call(this, bValue);

                if (wasSuggestionPopupCreated) {
                    self.applyPropagatingStyleClassesToControl(this._oSuggestionPopup);
                }
            };
            input.setShowSuggestion(true); //this needs to go after the function override

            // Coppied from van
            // use a custom filter function so that suggestions work correctly when
            // the data has leading whitespace
            input.setFilterFunction(function (userText, item) {
                var searchText = userText.toLowerCase();
                var itemText = item.getText().toLowerCase();
                return itemText.indexOf(searchText) !== -1;
            });

            this.input = input;
            this._inputHeight = 0;
            this._isNumeric = false;
        },

        // Override PropmtElementView
        _getClearSelectionMenuEnabled: function() {
            // If context menu is not needed or prompt has a required value, return false
            if (!this.getContextMenuEnabled() || this._isRequired) {
                return false;
            }
            var value = this.input.getValue();
            return value && value.length;
        },

        _inputChangeHandler: function(event) {
            var value = event.getParameter("value");
            if (value === undefined || value === null) {
                return;
            }

            // S1425724: Don't assign empty string to controller with numeric constraint
            if (!this._isNumeric || value !== "") {
                var controller = this.getController();
                if(controller) {
                    var dataValue = controller.getDataValueForUserInput(value);
                    controller.setCurrentValue(dataValue);
                    this.input.setTooltip(this.updateTooltip());
                }
            }
        },

        // Overridden from parent class
        clearSelection: function() {
            PromptElementView.prototype.clearSelection.apply(this);
            // Clear text input content
            this.input.setValue("");
        },

        propagatePropagatingStyleClasses: function () {
            PromptElementView.prototype.propagatePropagatingStyleClasses.apply(this, arguments);

            if (this.input && this.input._oSuggestionPopup) {
                // apply report theme class to the popover
                this.applyPropagatingStyleClassesToControl(this.input._oSuggestionPopup);
            }
        },

        initUI: function () {
            PromptElementView.prototype.initUI.apply(this, arguments);
            var input = this.input;
            var controllerData = this.getControllerData();
            var selected = this.getControllerSelectedValues(false)[0] || "";
            var rowData = controllerData.rowData;
            var items = input.getSuggestionItems() || [];
            var item;

            for (var i = 0; i < rowData.length; i++) {
                item = items[i];
                var row = rowData[i];
                var textValue = row.value === null ? "" : row.value;

                if(item) {
                    item.setText(textValue);
                    item.setAdditionalText(row.frequency);
                } else {
                    input.addSuggestionItem(new ListItem({text: textValue, additionalText: row.frequency}));
                }
            }
            for(var j = rowData.length; j < items.length; j++) {
                item = items[j];
                if(item) {
                    input.removeSuggestionItem(item);
                    item.destroy();
                }
            }

            input.setDisplaySecondaryValues(controllerData.hasFrequencies);
            input.setValue(selected);

            var categoryLabel = this.getLabel();
            if (categoryLabel) {
                input.setPlaceholder(rb.getText("TextFieldTypeAhead.placeholder.fmt", categoryLabel));
            }
            input.setTooltip(this.updateTooltip());
        },

        getControllerSelectedValues: function(forTooltip) {
            var selections = [];
            var controller = this.getController();
            if (!controller) {
                return null;
            }
            var currentValue = controller.getCurrentValue();
            var value;
            if (currentValue && this._isNumeric && !forTooltip) {
                value = currentValue.getNumericValue();
            } else {
                value = this.getFormattedValue(currentValue);
                value = value || "";
            }
            selections.push(value);
            return selections;
        },

        updateTooltip: function() {
            var controller = this.getController();
            if (controller) {
                var categoryLabel = controller.getLabel();
                if (categoryLabel) {
                    var measureLabel = controller.getMeasureVariableLabel();
                    var categoryValue = null;
                    // For validation purpose, we need raw string for category value (esp. missing values)
                    var currentDataValue = controller.getCurrentValue();
                    if (currentDataValue) {
                        categoryValue = currentDataValue.getStringValue();
                    }
                    var dataValue = controller.getDataValueForUserInput(categoryValue);
                    var isValidCategoryValue = controller.validateValue(dataValue) ? true : false;
                    // Once validation is done, we need formatted string.
                    categoryValue = this.getControllerSelectedValues(true)[0];
                    if (isValidCategoryValue && categoryValue) {
                        // If it has measure assigned, append the label and value
                        var measureFormattedValue = controller.getCurrentFormattedMeasure();
                        if (measureLabel && measureFormattedValue) {
                            return rb.getText("PromptElement.categoryLabelValueAndMeasureLabelValueTooltip.fmt", [categoryLabel, categoryValue, measureLabel, measureFormattedValue]);
                        }
                        return rb.getText("PromptElement.categoryLabelValueTooltip.fmt", [categoryLabel, categoryValue]);
                    } else if (measureLabel) {
                        return rb.getText("TextFieldTypeAhead.categoryAndMeasureTooltip.fmt", [categoryLabel, measureLabel]);
                    } else {
                        return rb.getText("TextFieldTypeAhead.categoryTooltip.fmt", categoryLabel);
                    }
                }
            }
            return "";
        },

        /**
         * @override
         */
        getUISelectionIndices: function () {
            //super override
            var input = this.input;
            var items = input.getSuggestionItems();
            var value = input.getValue();
            if (this._isNumeric) {
                value = formatFloatInstance.parse(value);
            }
            for (var i = 0; i < items.length; i++) {
                if (items[i].getText() === value) {
                    return [i];
                }
            }
            return [];
        },

        setController: function (controller) {
            PromptElementView.prototype.setController.apply(this, arguments); 
        },

        updateConstraints: function(constraint, isParameter) {
            PromptElementView.prototype.updateConstraints.apply(this, arguments);
            // The only time we have numeric constraints is when an Input Prompt
            // is used with a numeric parameter.  In that case there is always a min/max.
            if(isParameter && constraint && constraint instanceof NumericConstraint) {
                var input = this.input,
                    controller = this.getController();

                // Force input to numeric type
                this._isNumeric = true;
                // Force to integer
                input.setAllowIntegerOnly(!!constraint.getIntegerOnly());

                input.setType(InputType.Number);
                input.setFormatValue(true);

                var textFieldFormatter = new TextFieldFormatter();
                textFieldFormatter.setFormatName(controller.getFormatName());
                input.setNumericFormatter(textFieldFormatter);

                var rangeValidator = new RangeValidationAdapter({
                    minValue: parseFloat(constraint.getMin()),
                    maxValue: parseFloat(constraint.getMax()),
                    inclusive: constraint.getMinInclusive() && constraint.getMaxInclusive()
                });
                input.setValidator(rangeValidator);
                //
                // input.setValidator sets rangeValidator's numeric formatter to the
                // input's numeric validator (our TextFieldFormatter). We need error
                // text to have default locale formatting, so we override it after setValidator
                //
                rangeValidator.setNumberFormatter(new InputNumberFormatter());
            }
        },

        applyStyleOverrides: function () {
            var styles = this.getPromptStyleOverrides();
            this.input.setCustomStyles(styles);
        },

        measureContent: function() {
            // TODO: this should never get smaller
            // in some cases sizes were coming in later that were too small
            // we need to investigate why.
            this._inputHeight = Math.max(this.input.$().outerHeight(true), this._inputHeight);
            return {
                minWidth: PromptElementView.DEFAULT_MINIMUM_WIDTH,
                minHeight: this._inputHeight,
                height: this._inputHeight
            };
        },

         destroy: function () {
            PromptElementView.prototype.destroy.apply(this, arguments);
            if (this.input) {
                this.input.destroy();
                this.input = null;
            }
        }
    });
}, true);
