sap.ui.define([
    "sas/vaviewer/views/VisualElementView",
    "sas/ltjs/BIRD/util/hud/DataStateChangeEvent",
    "sas/vaviewer/views/VisualEvent",
    "sas/ltjs/BIRD/util/RichTextStringUtil"
], function(
    VisualElementView,
    DataStateChangeEvent,
    VisualEvent,
    RichTextStringUtil
) {
    "use strict";
    return VisualElementView.extend("sas.vaviewer.views.TextElementView", {
        metadata: {
            properties: {
                "text": {type: "string", defaultValue: ''}
            }
        },

        renderer: {},

        init: function () {
            VisualElementView.prototype.init.apply(this, arguments);
            this.addStyleClass("vavRichText");
        },

        setController: function (controller) {
            this._removeEventListeners();
            VisualElementView.prototype.setController.apply(this, arguments);
            this._addEventListeners();
            this._createText();
        },

        _addEventListeners: function () {
            this._handles = this._handles || [];
            this._handles.push(this._controller.attachEvent(DataStateChangeEvent.DATACHANGE, this._onDataChange, this));
        },

        _removeEventListeners: function () {
            if (this._handles) {
                for (var i = this._handles.length - 1; i >= 0; i--) {
                    this._handles[i].detach();
                }
                this._handles = null;
            }
        },

        _onDataChange: function() {
            this._createText();
        },

        _createText: function () {
            var text = RichTextStringUtil.parseHTMLString(this._controller.getRichTextAsHTML());
            if (!text) {
                this.setText('');
                return;
            }
            var regex = new RegExp("<a href=\"ac\\d+\"", "g");
            var matches = text.match(regex);
            if (matches) {
                matches.forEach(function(match) {
                    // Get the interaction id after the first 9 characters: <a href="
                    var interactionId = match.substring(9);
                    var newString = "<a data-interaction-id=\"" + interactionId + "\" href=\"javascript: void(0);\"";
                    text = text.replace(match, newString);
                });
            }
            this.setText(text);
        },

        setText: function (value) {
            if(value !== this.getText()) {
                this.setProperty("text", value, true);
                var wrapper = this.getDomRef("wrapper");
                if (wrapper) {
                    wrapper.innerHTML = value;
                }
                this.fireEvent(VisualEvent.LAYOUT_CHANGED, {}, false, true);
            }
        },

        onclick: function(event) {
            // trigger interactions when a NavigationAction link is clicked.
            var element = event.target;
            // data-interaction-id is accessible through dataset.interactionId
            if(element.dataset && element.dataset.interactionId) {
                this._controller.triggerInteraction(element.dataset.interactionId);
            }
        },

        _renderChildren: function (rm) {
            VisualElementView.prototype._renderChildren.apply(this, arguments);
            rm.write("<div id='" + this.getId() + "-wrapper' class='vavRichTextWrapper'>");
            rm.write(this.getText());
            rm.write("</div>");
        },

        _getCustomCSSRules: function () {
            return this.getCustomStyles().getStyles();
        },

        getMinimumHeight: function() {
            return this._calculateHeight() || 0;
        },

        getPreferredHeight: function(constrainedWidth) {
            if(!constrainedWidth) {
                return NaN; //no need to set initialy since we rely on the callback.
            }
            return this._calculateHeight(constrainedWidth);
        },

        _calculateHeight: function(constrainedWidth) {
            var $element = this.$("wrapper");
            if (!$element.length) {
                return NaN;
            }
            
            // don't set NaN, negatives, undefined or null
            var height = 0;
            if (constrainedWidth || constrainedWidth === 0) {
                $element.width(constrainedWidth);
                height = $element.height();
                //clear the width
                $element.width('');
            } else {
                height = this._calculateSingleLineHeight();
            }
            return height;
        },

        _calculateSingleLineHeight: function() {
            var height = 0;
            if (this._controller) {
                height = this._controller.getMaxFontHeight();
            }
            return height;
        },

        canExportData: function() {
            return false;
        },

        destroy: function () {
            this._removeEventListeners();
            VisualElementView.prototype.destroy.apply(this, arguments);
        }
    });
}, true);
