sap.ui.define([
    "sas/hc/ui/core/Styles",
    "sas/ltjs/BIRD/controllers/view/ContainerController",
    "sas/vaviewer/views/BIRDViewFactory",
    "sas/vaviewer/views/ContainerUtil",
    "sas/vaviewer/views/Header",
    "sas/vaviewer/views/MediaContainer",
    "sas/vaviewer/views/VisualElementView",
    "sas/vaviewer/views/VisualEvent",
    "sas/vaviewer/views/VisualizationElementView",
    "sas/ltjs/BIRD/ui/transitions/SlideTransition",
    "sas/ltjs/commons/styles/AttributeNames",
    "sas/ltjs/commons/util/DimensionUnit",
    "sas/ltjs/commons/util/StyleUtils",
    "sas/ltjs/commons/views/UIFlexBox",
    "sas/ltjs/commons/views/UISizing",
    "sas/vaviewer/SupplementalVisualSplitter",
    "sas/ltjs/commons/util/ScrollBarUtil",
    "sas/ltjs/BIRD/controllers/view/VisualController",
    "sas/ltjs/BIRD/controllers/view/PromptContainerController"
], function(
    Styles,
    ContainerController,
    BIRDViewFactory,
    ContainerUtil,
    Header,
    MediaContainer,
    VisualElementView,
    VisualEvent,
    VisualizationElementView,
    SlideTransition,
    AttributeNames,
    DimensionUnit,
    StyleUtils,
    UIFlexBox,
    UISizing,
    SupplementalVisualSplitter,
    ScrollBarUtil,
    VisualController,
    PromptContainerController
) {
    "use strict";

    return VisualElementView.extend("sas.vaviewer.views.Section", {

        metadata: {
            // to supress the assertion failure from commons 2.5 sdk
            specialSettings: {
                "vaViewerConfig": {},
                "shareEnabled": {}
            }
        },

        init: function () {
            this._bodyView = null;
            this._headerView = null;
            this._type = "Section";
            this._expandedMode = false;
            this._expandedVisual = null;
            this._expandedVisualIndex = -1;
            this._futureVisual = null;
            this._futureVisualIndex = -1;
            this._headerVisible = true;
            this.setFlexBox(UIFlexBox.FLEXCOLUMN);
            this._addEventListeners();
            this._vaViewerConfig = null;
            this._shareEnabled = true;
            this._objectViewOnlyContent = null;

            // Report page padding is added on the section
            this.addStyleClass("ReportPagePadding");
            this.addStyleClass("vavSection");

            this._slideTransition = new SlideTransition({
                visible: false
            });
            this._slideTransition.attachTransitionComplete(this._onTransitionComplete, this);
            this._allExpandedVisuals = null;
        },

        applySettings: function (settings, scope) {
            if(VisualElementView.prototype.applySettings) {
                VisualElementView.prototype.applySettings.call(this, settings, scope);
            }
            this._vaViewerConfig = settings.vaViewerConfig;
            this._shareEnabled = settings.shareEnabled;
        },

        renderer: {},

        setController: function (controller) {
            VisualElementView.prototype.setController.apply(this, arguments);
            var mediaContainerController;
            var headerController;
            var bodyController;
            this.removeAllChildren();
            this._bodyView = null;
            this._headerView = null;

            if(controller) {
                // For object view only mode, just add a single visual to the section and return
                if (this._vaViewerConfig.getObjectViewOnly()) {
                    var content = this._getObjectViewOnlyContent(controller);
                    if (content) {
                        this._objectViewOnlyContent = content;
                        this.addChild(content);
                        this.addChild(this._slideTransition);
                    }
                    return;
                }

                var model = controller.getModel();
                var showSelectionControl = model.getShowSelectionControl();

                // Create Header, add as child
                headerController = controller.getHeaderController();
                
                if (headerController || showSelectionControl) {
                    this._headerView = new Header();
                    this._headerView.addStyleClass('PagePromptArea');
                    this._headerView.setController(headerController);

                    if (showSelectionControl) {
                        this._headerView.setAutoInteractionMode(model.getAutoInteractionMode());
                        this._headerView.setAutoInteractionController(controller.getAutoInteractionController());
                    }

                    // The section header should match the Report body
                    var reportStyles = this._getReportStyles();
                    if (reportStyles) {
                        this._headerView.setCustomStyles(reportStyles);
                    }

                    this.addChild(this._headerView);
                }

                // Create Body, add as child
                bodyController = controller.getBodyController();
                if (bodyController) {
                    mediaContainerController = bodyController.getMediaContainerController();
                    if (mediaContainerController) {
                        this._bodyView = new MediaContainer();
                        this._bodyView.addStyleClass("SectionBody");
                        this._bodyView.setSizing(UISizing.FILL);
                        this._bodyView.setController(mediaContainerController);
                        
                        // Disable expanding visualElements if only one visual or one composite visual and not precision layout.
                        var allBIRDTitledContents = this._bodyView.getFocusableContent();
                        var allBIRDTitledContentsCount = allBIRDTitledContents.length;
                        var content;
                        // set vaViewerConfig property on BIRDTitledContent
                        for (var i = 0; i < allBIRDTitledContentsCount; i++) {
                            content = allBIRDTitledContents[i];
                            content.setVaViewerConfig(this._vaViewerConfig);
                            content.setShareEnabled(this._shareEnabled);
                        }

                        var hasSupplementalVisual = true;
                        // If a supplemental visual is present, allow expanding the visual even if it there is a single one.
                        if (allBIRDTitledContentsCount === 1) {
                            var birdTitledContent = allBIRDTitledContents[0];
                            hasSupplementalVisual = this._isSupplementalVisualPresent(birdTitledContent);
                        }
                        if (allBIRDTitledContentsCount <= 1 && !hasSupplementalVisual &&
                            this._bodyView.getLayoutType() !== ContainerUtil.LayoutType.RELATIVE) {
                            this._bodyView.setExpandable(false);
                        }

                        this.addChild(this._bodyView);
                    }
                }
                this.addChild(this._slideTransition);
            }
        },

        _onTransitionComplete: function(e) {
            var oldSplitter = e.getParameter("oldContent");

            // cleanup oldSplitter
            if(oldSplitter) {
                oldSplitter.fireEvent(VisualEvent.VISUAL_UNEXPANDED, {
                    id: oldSplitter.sId
                }, true, true);
            }

            if(this._futureVisual) {
                this._transitionVisuals(this._futureVisual, this._futureVisualIndex);
                this._futureVisual = null;
                this._futureVisualIndex = -1;
            }
        },

        _getObjectViewOnlyContent: function(sectionController) {
            var bodyController = sectionController.getBodyController();
            if (!bodyController) {
                return null;
            }
            var mediaContainerController = bodyController.getMediaContainerController();
            if (!mediaContainerController) {
                return null;
            }
            var allVisualAndCompositeContainerControllers = this._getAllVisualAndCompositeContainerControllers(mediaContainerController);
            var contentController = this._getObjectViewOnlyContentController(allVisualAndCompositeContainerControllers);
            if (!contentController) {
                return null;
            }
            var content = BIRDViewFactory.createBIRDTitledContent(contentController, false);
            content.setSizing(UISizing.FILL);
            // Single object need not be expandable.
            content.setExpandable(false);
            content.setVaViewerConfig(this._vaViewerConfig);
            // We need to tell the visualization that we are in 
            // object view only mode to disable the links from the content
            // This would apply only for a normal visual and not a composite
            // visual since they don't have links from them.
            if (!(contentController instanceof ContainerController)) {
                var visualElementView = content.getVisualElementView();
                if (visualElementView && visualElementView instanceof VisualizationElementView) {
                    var visualization = visualElementView.getVisualization();
                    if (visualization) {
                        visualization.setObjectViewOnly(this._vaViewerConfig.getObjectViewOnly());
                    }
                }
            }
            return content;
        },

        _getAllVisualAndCompositeContainerControllers: function(modelController) {
            if (!modelController) {
                return [];
            }
            var vacControllers = modelController.getOrderedVisualAndContainerControllers();
            if (!vacControllers || !vacControllers.length) {
                return [];
            }
            var allVacControllers = [];
            vacControllers.forEach(function(vacController) {
                if (vacController instanceof PromptContainerController) {
                    return;
                } else if (vacController instanceof VisualController) {
                    allVacControllers.push(vacController);
                } else if (vacController instanceof ContainerController) {
                    if (vacController.isCompositeContainer()) {
                        allVacControllers.push(vacController);
                    } else {
                        allVacControllers = allVacControllers.concat(this._getAllVisualAndCompositeContainerControllers(vacController));
                    }
                }
            }.bind(this));
            return allVacControllers;
        },

        _getObjectViewOnlyContentController: function(allVisualAndCompositeContainerControllers) {
            var veName = this._vaViewerConfig.getObjectName();
            var visualElementController, model;
            for (var i = 0, len = allVisualAndCompositeContainerControllers.length; i < len; ++i) {
                var positionedModelController = allVisualAndCompositeContainerControllers[i];
                if (positionedModelController instanceof ContainerController) {
                    visualElementController = positionedModelController.getVisualContainerElementController();
                } else {
                    visualElementController = positionedModelController.getVisualElementController();
                }
                if (visualElementController) {
                    model = visualElementController.getModel();
                    if (model && model.getName() === veName) {
                        return positionedModelController;
                    }
                }
            }
            return null;
        },

        _isSupplementalVisualPresent: function (birdTitledContent) {
            var veController = birdTitledContent.getVisualElementController();
            if (veController) {
                var suppVisualControllers = veController.getSupplementalVisualElementControllers();
                // If we have at least one supplemental visual element controller, then we have a supplemental visual.
                if (suppVisualControllers && suppVisualControllers.length > 0) {
                    return true;
                }
            }
            return false;
        },

        _getReportStyles: function () {
            var styleChain = this._controller.getReportController().getReportStyleChain();
            var bgColor = StyleUtils.getBackgroundColorString(styleChain);
            if (bgColor) {         
                return new Styles().setStyles({'background-color': bgColor});
            }

            return null;
        },

        applyStyleOverrides: function () {
            var padding = this._controller.getPagePadding();
            if (padding) {
                this.setPadding(padding.getScalarValueForUnit(DimensionUnit.PIXELS) + 'px');
            }
        },

        getExpandedMode: function () {
            return this._expandedMode;
        },

        setExpandedMode: function(value) {
            if(this._headerView) {
                //if we are in expended mode then we do not want to display the header.
                if (value) {
                    this._headerView.setVisible(false);
                } else if (this._headerVisible) {
                    this._headerView.setVisible(true);
                }
            }
            if(this._bodyView) {
                this._bodyView.setExpandedMode(value);
            }
            this._expandedMode = value;
        },

        layoutSubviews: function () {
            if (!this.getController() || this._expandedVisual) {
                return;
            }
            if (this._objectViewOnlyContent) {
                var contentController = this._objectViewOnlyContent.getController();
                var isContainer = contentController instanceof ContainerController;
                // set up constraints for containers
                if (isContainer) {
                    // make sure the content has set its min and preferred sizes before trying to layout
                    this._objectViewOnlyContent.refreshConstraints();
                    // apply constraints from BIRD, considering active mediaTarget
                    contentController.refreshLayoutConstraints();
                }
                this._objectViewOnlyContent.layoutSubviews();
                // add scrollbars for containers
                if(isContainer) {
                    var container = this._objectViewOnlyContent.getContainerContent();
                    var children = container ? container.getChildren() : null;
                    // If the avalable space has changes we need to re-layout.
                    var containerBoundsChanged = ContainerUtil.updateScrollbars(this, children);
                    if(containerBoundsChanged) {
                        this.layoutSubviews();
                    }
                }
            } else {
                var bodyView = this.getBodyView();
                if (bodyView) {
                    bodyView.layoutSubviews();
                }
            }
        },

        getHeaderView: function () {
            return this._headerView;
        },

        getBodyView: function () {
            return this._bodyView;
        },

        getAllVisualViews: function() {
            if(this._bodyView) {
                return this._bodyView.getAllVisualViews();
            } else {
                return [];
            }
        },

        getObjectViewOnlyContent: function() {
            return this._objectViewOnlyContent;
        },

        setVisible: function(value) {
            VisualElementView.prototype.setVisible.apply(this, arguments);
            // NOTE: this is to work around Slider issues. slider._runPostOnAfterRendering() 
            // should be called immediately as opposed to on a timeout to avoid layout issues/jank.
            if(value && this._headerView) {
                this._headerView.layoutSubviews();
            }
        },

        setHeaderVisible: function(value) {
            if(value === this._headerVisible) {
                return;
            }
            if(this._headerView) {
                this._headerVisible = value;
                this._headerView.setVisible(value);
            }
        },

        setOverlaysEnabled: function(value) {
            this.getAllVisualViews().forEach(function(v) {
                v.setOverlaysEnabled(value);
            });
        },

        _addEventListeners: function() {
            this.attachEvent(VisualEvent.VISUAL_EXPANDED, this._onVisualExpanded);
            this.attachEvent(VisualEvent.VISUAL_UNEXPANDED, this._onVisualUnexpanded);
        },

        _removeEventListeners: function() {
            this.detachEvent(VisualEvent.VISUAL_EXPANDED, this._onVisualExpanded);
            this.detachEvent(VisualEvent.VISUAL_UNEXPANDED, this._onVisualUnexpanded);
        },

        _onVisualExpanded: function(event) {
            var visual = event.getParameter("visual");
            if (!visual || this._expandedVisual === visual) {
                return;
            }
            var visuals = this._bodyView.getFocusableContent();
            var newVisualIndex = visuals.indexOf(visual);
            if(this._slideTransition.isAnimating()) {
                this._futureVisual = visual;
                this._futureVisualIndex = newVisualIndex;
                return;
            }
            this._transitionVisuals(visual, newVisualIndex);
            if (!this._allExpandedVisuals) {
                this._allExpandedVisuals = [];
            }
            this._allExpandedVisuals.push(visual);
        },

        _transitionVisuals: function(visual, visualIndex) {
            visual.setHidden(false);
            visual.setVisible(true);

            var newSplitter = this._createSupplementalVisual(visual);

            if (this._expandedVisualIndex > visualIndex) {
                this._slideTransition.transitionContentRight(newSplitter);
            } else {
                this._slideTransition.transitionContentLeft(newSplitter);
            }
            this._expandedVisual = visual;
            this._expandedVisualIndex = visualIndex; 
        },

        _createSupplementalVisual: function(visual) {
            if(!visual) {
                return null;
            }
            // Create a custom Splitter extended from the HTML commons Splitter component
            // which would contain the primary and supplemental visual.
            var suppVisualSplitter = new SupplementalVisualSplitter({ persistPaneSizes: false });
            suppVisualSplitter.addContent(visual);

            // Apply report override styles to the splitter
            var reportStyles = this._getReportStyles();
            if (reportStyles) {
                suppVisualSplitter.setCustomStyles(reportStyles);
            }
            var bodyView = this.getBodyView();
            if(!bodyView) {
                return null;
            }
            // Hide the rest of the visuals in the section
            bodyView.setVisible(false);
            this._slideTransition.setVisible(true);
            return suppVisualSplitter;
        },

        _onVisualUnexpanded: function() {
            this.clearMaximizedView();
        },

        clearMaximizedView: function () {
            var bodyView = this.getBodyView();
            if(!bodyView) {
                return;
            }
            this._slideTransition.destroyContent();
            this._slideTransition.setVisible(false);
            bodyView.setVisible(true);
            this.unexpandExpandedVisual();
            // close supplemental visual and add the visual/container back to the section
            this._expandedVisual = null;
            this._expandedVisualIndex = -1;
            this.layoutSubviews();
        },

        unexpandExpandedVisual: function () {
            if (!this._expandedVisual) {
                return;
            }

            var decorator = this._expandedVisual.getReportElementDecorator();
            if (decorator) {
                decorator.setExpanded(false);
            }
            if (this._allExpandedVisuals) {
                this._allExpandedVisuals.forEach(function(eachVisual) {
                    decorator = eachVisual && eachVisual.getReportElementDecorator();
                    if (decorator) {
                        decorator.setExpanded(false);
                    }
                });
                this._allExpandedVisuals = null;
            }
        },

        setFixedBounds: function(fixedSectionBounds, actualSectionBounds) {
            var bodyView = this.getBodyView();
            if(!bodyView) {
                return;
            }
            var bodyViewContent = bodyView.getContainerContent();
            if (!bodyViewContent) {
                return;
            }

            var pageHeaderHeight = 0;
            var pageHeader = this.getHeaderView();
            if (pageHeader && pageHeader.getVisible()) {
               pageHeaderHeight = pageHeader.$().height();
            }
            // Account for the padding on all 4 sides of the page.
            var sidePading = 0;
            if (this._controller) {
                var padding = this._controller.getPagePadding();
                if (padding) {
                    sidePading = 2 * padding.getScalarValueForUnit(DimensionUnit.PIXELS);
                }
            }
            var fixedBodyBounds = {
                width: fixedSectionBounds.width - sidePading,
                height: fixedSectionBounds.height - pageHeaderHeight - sidePading
            };
            // These are the available bounds to the section body but it is supposed to
            // take fixedBodyBounds. So adjust the bounds to accommodate the scrollbars, if needed.
            var availableBounds = {
                width: actualSectionBounds.width - sidePading,
                height: actualSectionBounds.height - pageHeaderHeight - sidePading
            };
            var horizontalScrollbarVisible = fixedBodyBounds.width > availableBounds.width;
            var verticalScrollbarVisible = fixedBodyBounds.height > availableBounds.height;
            // If both the scrollbars are going to show up, do not reduce the fixed body bounds.
            if (horizontalScrollbarVisible && !verticalScrollbarVisible) {
                fixedBodyBounds.height -= ScrollBarUtil.getHeight();
            }
            if (verticalScrollbarVisible && !horizontalScrollbarVisible) {
                fixedBodyBounds.width -= ScrollBarUtil.getWidth();
            }
            bodyView.setFlexBox(UIFlexBox.NONE);
            bodyView.addStyleClass("overflowScroll");
            bodyViewContent.setSizing(UISizing.CSSSIZE);
            
            bodyViewContent.setBounds(fixedBodyBounds);
            // Remove any scroll styles added to body content during re-layout.
            bodyViewContent.removeStyleClass("vavHorizontalScrollContainer");
            bodyViewContent.removeStyleClass("vavVerticalScrollContainer");

            // S1505580: Hack to avoid infinite loop during layout
            bodyView.setShouldUpdateScrollbars(false);
        },

        destroy: function () {
            this._removeEventListeners();
            this._slideTransition.destroy();
            if (this._bodyView) {
                this._bodyView.destroy();
                this._bodyView = null;
            }
            if (this._headerView) {
                this._headerView.destroy();
                this._headerView = null;
            }
            if (this._allExpandedVisuals) {
                this._allExpandedVisuals.length = 0;
                this._allExpandedVisuals = null;
            }
            VisualElementView.prototype.destroy.apply(this, arguments);
        }
    });
}, true);
