sap.ui.define([
    "sas/ltjs/BIRD/models/view/layoutElements/LayoutElementOrientation",
    "sas/ltjs/BIRD/models/view/layoutElements/RelativeLayout",
    "sas/ltjs/BIRD/models/view/layoutElements/ResponsiveLayout",
    "sas/ltjs/BIRD/models/view/layoutElements/SlideLayout",
    "sas/ltjs/BIRD/models/view/layoutElements/StackLayout",
    "sas/ltjs/BIRD/util/layout/ResponsiveLayoutScrollDirection",
    "sas/ltjs/commons/layout/ResponsiveLayoutOverflowType",
    "sas/ltjs/commons/util/ScrollBarUtil",
    "sas/ltjs/commons/views/UISizing"
], function(
    LayoutElementOrientation,
    RelativeLayout,
    ResponsiveLayout,
    SlideLayout,
    StackLayout,
    ResponsiveLayoutScrollDirection,
    ResponsiveLayoutOverflowType,
    ScrollBarUtil,
    UISizing
) {
    "use strict";

    var ContainerUtil = {
        LayoutType: {
            RELATIVE: 0,
            RESPONSIVE: 1,
            STACK: 2,
            SLIDE_VERTICAL: 3,
            SLIDE_HORIZONTAL: 4
        },
        getLayoutTypeForController: function (controller) {
            var model = controller.getModel();
            if (!model.getLayout) {
                return null;
            }

            var containerLayout = model.getLayout();
            var layoutType;

            if (containerLayout instanceof RelativeLayout) {
                layoutType = ContainerUtil.LayoutType.RELATIVE;
            } else if (containerLayout instanceof StackLayout) {
                layoutType = ContainerUtil.LayoutType.STACK;
            } else if (containerLayout instanceof ResponsiveLayout) {
                var overflow = containerLayout.getOverflow();
                if (overflow === ResponsiveLayoutOverflowType.STACK) {
                    layoutType = ContainerUtil.LayoutType.STACK;
                } else if (controller.getResponsiveLayoutScrollDirection() === ResponsiveLayoutScrollDirection.HORIZONTAL) {
                    layoutType = ContainerUtil.LayoutType.SLIDE_HORIZONTAL;
                } else if(controller.getResponsiveLayoutScrollDirection() === ResponsiveLayoutScrollDirection.VERTICAL){
                    layoutType = ContainerUtil.LayoutType.SLIDE_VERTICAL;
                } else {
                    layoutType = ContainerUtil.LayoutType.RESPONSIVE;
                }
            } else {
                var slideOrientation = containerLayout instanceof SlideLayout ? containerLayout.getOrientation() : null;

                if (slideOrientation === LayoutElementOrientation.HORIZONTAL) {
                    layoutType = ContainerUtil.LayoutType.SLIDE_HORIZONTAL;
                } else {
                    layoutType = ContainerUtil.LayoutType.SLIDE_VERTICAL;
                }
            }

            return layoutType;
        },
        /**
        * Will add and remove scrollbars base on the DOM positioning of `children` relative to the `container`
        * @param {SizedContent} container the container that will conditionally get the scrollbars
        * @param {Control} children the list of Controls to check the positioning of.
        * @returns {boolean} if scrollbars were added or removed to `container`
        */
        updateScrollbars: function(container, children) {
            var verticalScrollbarNeeded = false;
            var horizontalScrollbarNeeded = false;
            var boundsChanged = false;
            if(children) {
                var containerBounds = container.getBounds();

                // S1577126: getBounds uses clientWidth and clientHeight. In Chrome, these values
                // sometimes get rounded down when the decimal is greater than .5, causing
                // scrollbars to be added where they aren't needed and sometimes starting an
                // infinite loop. As a workaround, use the more precise values returned by getWidth
                // and getHeight when available.
                if (container.getSizing() === UISizing.CSSSIZE) {
                    containerBounds = {
                        height: container.getHeight() || containerBounds.height,
                        width: container.getWidth() || containerBounds.width,
                    }
                }

                var containerHeight = parseFloat(containerBounds.height);
                var containerWidth = parseFloat(containerBounds.width);
                // Some browsers (Safari) can get into an infinite loop without this check. With zero sized containers.
                //   If scrollbars won't fit, then don't try and add them.
                var scrollableWidth = containerWidth > ScrollBarUtil.getWidth();
                var scrollableHeight = containerHeight > ScrollBarUtil.getHeight();
                if(scrollableWidth || scrollableHeight) {
                    for(var i = 0, len = children.length; (!verticalScrollbarNeeded || !horizontalScrollbarNeeded) && i < len; ++i){
                        var child = children[i];
                        var node = child.getDomRef();
                        if (!node) {
                            continue;
                        }
                        var bottom = parseFloat(node.style.top) + parseFloat(node.style.height);
                        // Account for rounding errors.
                        if(scrollableHeight && (bottom - containerHeight > .5)) {
                            verticalScrollbarNeeded = true;
                        }
                        var left = parseFloat(node.style.left);
                        if (left < -0.5) {
                            //
                            // assume right-to-left with space inversion if left is negative
                            // if left is negative, then it is more than a containerWidth away from
                            // the right edge of the container.
                            //
                            horizontalScrollbarNeeded = true;
                            continue;
                        }
                        var right =  left + parseFloat(node.style.width);
                        // Account for rounding errors.
                        if(scrollableWidth && (right - containerWidth > .5)) {
                            horizontalScrollbarNeeded = true;
                        }
                    }
                }
            }

            if(verticalScrollbarNeeded !== container.getVerticalScroll()) {
                container.setVerticalScroll(verticalScrollbarNeeded);
                if(ScrollBarUtil.getWidth()) {
                    boundsChanged = true;
                }
            }

            if(horizontalScrollbarNeeded !== container.getHorizontalScroll()) {
                container.setHorizontalScroll(horizontalScrollbarNeeded);
                if(ScrollBarUtil.getHeight()) {
                    boundsChanged = true;
                }
            }

            return boundsChanged;
        }
    };

    return ContainerUtil;
}, true);
