sap.ui.define([
    "sas/hc/ui/core/Styles",
    "sas/ltjs/BIRD/controllers/view/ContainerController",
    "sas/ltjs/BIRD/controllers/view/VisualController",
    "sas/ltjs/BIRD/models/view/layoutElements/ResponsiveLayout",
    "sas/ltjs/BIRD/util/layout/ResponsiveLayoutScrollDirection",
    "sas/ltjs/BIRD/util/mediaTarget/WindowSize",
    "sas/vaviewer/views/BIRDTitledContent",
    "sas/vaviewer/views/Visual",
    "sas/vaviewer/views/StackContainer",
    "sas/ltjs/commons/layout/LayoutContext",
    "sas/ltjs/commons/layout/PreferredSizeRequestEvent",
    "sas/ltjs/commons/styles/AttributeNames",
    "sas/ltjs/commons/styles/ButtonType",
    "sas/ltjs/commons/styles/HorizontalPosition",
    "sas/ltjs/commons/styles/VerticalPosition",
    "sas/ltjs/commons/util/Dimension",
    "sas/ltjs/commons/util/DimensionUnit",
    "sas/ltjs/commons/util/ScrollBarUtil",
    "sas/ltjs/commons/util/StyleUtils",
    "sas/ltjs/commons/views/UI",
    "sas/ltjs/commons/views/UIFlexBox",
    "sas/ltjs/commons/views/UISizing",
    "sas/vaviewer/views/BIRDViewFactory",
    "sas/vaviewer/views/ContainerUtil",
    "sas/ltjs/BIRD/util/LocaleUtil"
], function(
    Styles,
    ContainerController,
    VisualController,
    ResponsiveLayout,
    ResponsiveLayoutScrollDirection,
    WindowSize,
    BIRDTitledContent,
    Visual,
    StackContainer,
    LayoutContext,
    PreferredSizeRequestEvent,
    AttributeNames,
    ButtonType,
    HorizontalPosition,
    VerticalPosition,
    Dimension,
    DimensionUnit,
    ScrollBarUtil,
    StyleUtils,
    UI,
    UIFlexBox,
    UISizing,
    BIRDViewFactory,
    ContainerUtil,
    LocaleUtil
) {
    "use strict";

    var TILE_GAP = 10;
    var CONTAINER_INSET = 5;
    var PADDING = 3;

    var MEDIA_MAP = {
        Desktop: WindowSize.LARGE,
        Tablet: WindowSize.MEDIUM,
        Phone: WindowSize.SMALL
    };

    var BaseContainer = BIRDTitledContent.extend('sas.vaviewer.views.BaseContainer', {
        metadata: {
            properties: {
                'contentMode': { type: 'string', defaultValue: 'left' }
            }
        },

        renderer: {},

        init: function () {
            BIRDTitledContent.prototype.init.apply(this, arguments);
            this._hiddenOverflow = false;
            this._layoutType = null;
            this._onScroll = null;
            this.setFlexBox(UIFlexBox.FLEXCOLUMN);
            this._handles = [];
            this._totalHorizontalPercentage = 0;
            this._totalVerticalPercentage = 0;
            this._precision = false;
            this._shouldUpdateScrollbars = true;
            this._recursionCount = 0;

            // create contents
            this._containerContent = new UI({
                id: this.getId() + '-content',
                sizing: UISizing.FILL
            });

            this._containerContent.addDelegate({
                onAfterRendering: function() {
                    var content = this._containerContent;
                    if(!content) {
                        return;
                    }
                    this._onScroll = function (event) {
                        var vertOffset, horzOffset;
                        var element = event.target;
                        if(this.getHorizontalScroll()) {
                            if(element.scrollLeft === 0) {
                                horzOffset = new Dimension("0%");
                            } else {
                                horzOffset = new Dimension((element.scrollLeft / element.clientWidth * 100) + '%');
                            }
                        } 
                        if(this.getVerticalScroll()) {
                            if(element.scrollTop === 0) {
                                vertOffset = new Dimension("0%");
                            } else {
                                vertOffset = new Dimension((element.scrollTop / element.clientHeight * 100) + '%');
                            }
                        }
                        if(this._precision) {
                            this.getController().setRelativeHorizontalOffset(horzOffset);
                            this.getController().setRelativeVerticalOffset(vertOffset);
                        } else if(vertOffset || horzOffset) {
                            this.getController().setSlideOffset(vertOffset || horzOffset);
                        }
                    }.bind(this);
                    content.attachBrowserEvent('scroll', this._onScroll);
                }.bind(this)
            });
        },

        setController: function (controller) {
            BIRDTitledContent.prototype.setController.apply(this, arguments);

            this.destroyChildren();
            if (controller) {
                this._addEventListeners();

                var layoutType = this.getLayoutType();
                var childElements = this._createChildVisualElements(controller.getOrderedVisualAndContainerControllers());

                this.applyContainerStyles();

                var stackContainer = null;
                for (var plane = 0; plane < childElements.length; plane++) {
                    var currentPlane = childElements[plane];
                    if (currentPlane) {
                        for (var i = 0; i < currentPlane.length; i++) {
                            if (layoutType === ContainerUtil.LayoutType.STACK) {
                                if (!stackContainer) {
                                    stackContainer = new StackContainer();
                                    stackContainer.attachIndexChanged(this._onStackContainerIndexChanged.bind(this));
                                    
                                    this._containerContent.addChild(stackContainer);
                                }

                                var label = "";
                                var visualElementController = currentPlane[i].getVisualElementController();
                                if(visualElementController) {
                                    label = visualElementController.getModel().getLabel();
                                }
                                stackContainer.addPanel(currentPlane[i], label);
                            } else {
                                this._containerContent.addChild(currentPlane[i]);
                            }
                        }
                    }
                }

                if (stackContainer) {
                    stackContainer.buildModel();
                    var stackController = this.getVisualElementController();
                    if (stackController) {
                        var model = stackController.getModel();
                        if (model) {
                            var stackNavigationDetails = model.getStackNavigationControl();
                            if (stackNavigationDetails) {
                                stackContainer.setNavigationDetails(stackNavigationDetails);
                            }
                        }
                    }
                    //
                    // Initialize the selected/active stack from the value on the controller.
                    //
                    stackContainer.setSelectedIndex(controller.getStackActiveVisualIndex());
                }

                // this._containerContent.addStyleClass('vavFlexBlocker');
                this.addChild(this._containerContent);

                this.applyPaddingFromController(null);
            }
        },

        getVisualElementController: function() {
            var controller = this.getController();
            if (controller instanceof ContainerController) {
                return controller.getVisualContainerElementController();
            }
            return null;
        },

        getContainerContent: function() {
            return this._containerContent;
        },

        _onStackContainerIndexChanged: function(event) {
            var newIndex = event.getParameter("newIndex");
            this._controller.setStackActiveVisualIndex(newIndex);
        },
        
        _renderChildren: function( rm ) {
            BIRDTitledContent.prototype._renderChildren.apply(this, arguments);
            rm.renderControl(this.getReportElementDecorator());
        },

        /**
         * can be saved as an image or not
         * @returns {boolean} true or false
         */
        canSaveAsImage: function() {
            // Default would be false. Child classes would decide the actual return value.
            return false;
        },

        /**
         * can data be exported or not
         * @returns {boolean} true or false
         */
        canExportData: function() {
            return false;
        },

        _createChildVisualElements: function (visualAndContainerControllers) {
            var childElements = [];
            var layoutType = this.getLayoutType();
            var width, height, proxy, constraint;
            this._totalHorizontalPercentage = this._totalVerticalPercentage = 0;

            if (visualAndContainerControllers && visualAndContainerControllers.length > 0) {
                var controllerCount = visualAndContainerControllers.length;
                for (var i = 0; i < controllerCount; i++) {
                    var visualAndContainerController = visualAndContainerControllers[i];
                    var childLayout = ContainerUtil.getLayoutTypeForController(visualAndContainerController);
                    var visualModel = visualAndContainerController.getModel();
                    var relativeConstraint = visualModel.getRelativeConstraint();

                    var childView = this.createBIRDTitledContent(visualAndContainerController);
                    //childView can be a Visual, Container, CompositeContainer or PromptContainer
                    childView.setPosition("absolute");
                    if (layoutType === ContainerUtil.LayoutType.SLIDE_HORIZONTAL && visualModel.getWidth()) {
                        this._totalHorizontalPercentage += parseFloat(visualModel.getWidth());
                    } else if (layoutType === ContainerUtil.LayoutType.SLIDE_VERTICAL && visualModel.getHeight()) {
                        this._totalVerticalPercentage += parseFloat(visualModel.getHeight());
                    } else if (layoutType === ContainerUtil.LayoutType.RELATIVE) {
                        //precision layout sometimes takes up more space then the container.
                        var xPercent = parseFloat(relativeConstraint.getHorizontalConstraint().getOffset());
                        var yPercent = parseFloat(relativeConstraint.getVerticalConstraint().getOffset());
                        var widthPercent = parseFloat(relativeConstraint.getWidthConstraint().getWidth());
                        var heightPercent = parseFloat(relativeConstraint.getHeightConstraint().getHeight());
                        this._totalHorizontalPercentage = Math.max(this._totalHorizontalPercentage, xPercent + widthPercent);
                        this._totalVerticalPercentage =  Math.max(this._totalVerticalPercentage, yPercent + heightPercent);
                        this._precision = true;

                        // BIRD layout doesn't add this for us with precision layouts
                        if (childView instanceof Visual) {
                            childView.setPadding((TILE_GAP / 2) + PADDING + "px");
                        }
                    }
                    
                    //promoted r9 -> r10 scroll container.
                    proxy = visualAndContainerController.getLayoutProxy();
                    constraint = proxy.getDefaultConstraint();
                    var scrollDirection = this.getController().getResponsiveLayoutScrollDirection();
                    if(scrollDirection) {
                        if(proxy) {
                            if(constraint) {
                                height = constraint.getHeight();
                                width = constraint.getWidth();
                                if(scrollDirection  === ResponsiveLayoutScrollDirection.HORIZONTAL && width && width.getUnit() === DimensionUnit.PERCENTAGE) {
                                        this._totalHorizontalPercentage += width.getValue();
                                } else if(scrollDirection  === ResponsiveLayoutScrollDirection.VERTICAL && height && height.getUnit() === DimensionUnit.PERCENTAGE) {
                                        this._totalVerticalPercentage += height.getValue();
                                }
                            }
                        }
                    }

                    //If some SizedContent controller has a horizontal offset and a vertical offset, then it is part of a precision container
                    // promoted r9 -> r10
                    if(proxy) {
                        if(constraint) {
                            var hozOffset = constraint.getHorizontalOffset();
                            var vertOffset = constraint.getVerticalOffset();
                            height = constraint.getHeight();
                            width = constraint.getWidth();
                            //We currenlty only handle the case where the offest and sizes are percentage values.
                            if( hozOffset && hozOffset.getUnit() === DimensionUnit.PERCENTAGE &&
                                vertOffset && vertOffset.getUnit() === DimensionUnit.PERCENTAGE &&
                                height && height.getUnit() === DimensionUnit.PERCENTAGE && 
                                width && width.getUnit() === DimensionUnit.PERCENTAGE) {
                                this._precision = true;
                                this._totalHorizontalPercentage = Math.max(this._totalHorizontalPercentage, hozOffset.getValue() + width.getValue());
                                this._totalVerticalPercentage =  Math.max(this._totalVerticalPercentage, vertOffset.getValue() + height.getValue());
                            }
                        }
                    }

                    var index = childElements.length;
                    var zIndex;
                    if (layoutType === ContainerUtil.LayoutType.RELATIVE) {
                        zIndex = relativeConstraint.getZIndex();
                        if (zIndex) {
                            index = parseInt(zIndex);
                        } else {
                            index = 0;
                        }
                    } else if(layoutType !== ContainerUtil.LayoutType.STACK) {
                        var responsiveConstraint = visualModel.getResponsiveConstraint();
                        if(responsiveConstraint) {
                            zIndex = responsiveConstraint.getZIndex();
                            if (zIndex) {
                                index = parseInt(zIndex);
                            } else {
                                index = 0;
                            }
                        }
                    }
                    if (!childElements[index]) {
                        childElements[index] = [];
                    }
                    childElements[index].push(childView);
                } // end looping through visualAndContainerControllers

                this.setHorizontalScroll(this._totalHorizontalPercentage > 100);
                this.setVerticalScroll(this._totalVerticalPercentage > 100);
            }
            return childElements;
        },

        /**
         * Create an object of the type BIRDTitledContent depending on the controller.
         * @param {sas.ltjs.BIRD.controllers.visualElements.VisualElementController} visualAndContainerController - portable controller
         * @returns {sas.vaviewer.views.BIRDTitledContent} - Object of type BIRDTitledContent
         */
        createBIRDTitledContent: function(visualAndContainerController) {
            return BIRDViewFactory.createBIRDTitledContent(visualAndContainerController, this.getIsCompositeMember());
        },

        setHorizontalScroll: function (value) {
            this.setProperty("horizontalScroll", value, true);
            if(value) {
                this._containerContent.addStyleClass("vavHorizontalScrollContainer");
            } else {
                this._containerContent.removeStyleClass("vavHorizontalScrollContainer");
            }
        },

        setVerticalScroll: function (value) {
            this.setProperty("verticalScroll", value, true);
            if(value) {
                this._containerContent.addStyleClass("vavVerticalScrollContainer");
            } else {
                this._containerContent.removeStyleClass("vavVerticalScrollContainer");
            }
        },

        setExpandable: function (isExpandable) {
            BIRDTitledContent.prototype.setExpandable.apply(this, arguments);
            var children = this._containerContent.getChildren();
            for (var i = children.length - 1; i >= 0; --i) {
                var visualView = children[i];
                // Need this instanceof check in case of StackContainer which doesn't have expandable property. 
                if (visualView instanceof BIRDTitledContent) {
                    visualView.setExpandable(isExpandable);
                }
            }
            if(this.getReportElementDecorator()) {
                this.getReportElementDecorator().setExpandable(isExpandable);
            }
        },

        // virtual -- focus on the first child that says it wants focus
        focus: function () {
            var children = this._containerContent.getChildren();
            if (children) {
                for (var i = 0, length = children.length; i < length; i++) {
                    if (children[i].focus()) {
                        return true;
                    }
                }
            }
            return false;
        },

        layoutSubviews: function () {
            // There are cases where layoutSubviews gets stuck in an infinite loop of adding and
            // removing scrollbars. If it looks like we're in that situation, stop recursion and
            // print an error to the console.
            if (this._recursionCount >= 4) {
                var controller = this.getController();
                var model = controller && controller.getModel && controller.getModel();
                var name = model && model.getName && model.getName();
                console.error("Infinite loop detected while running layoutSubviews for model " + name + ". Recursion stopped.");
                return;
            }

            var controller = this.getController();
            var content = this._containerContent;

            if (!controller || !content) {
                return;
            }

            // var isRightAligned = this.getContentMode() === 'right';
            var visualView;
            var children = this._containerContent.getChildren();
            var i;

            var contentBounds = content.getBounds();
            var childBounds = {
                width: parseFloat(contentBounds.width),
                height: parseFloat(contentBounds.height)
            };

            this.doBIRDLayout(controller, childBounds);

            for (i = children.length - 1; i >= 0; --i) {
                visualView = children[i];
                
                //if this is true then the StackContainer will be the only child
                if (visualView instanceof StackContainer) {
                    childBounds = {top: 0, left: 0, width: childBounds.width, height: childBounds.height};
                } else {
                    childBounds = visualView.getController().getRelativeBounds();
                    childBounds = {left: childBounds[0], top: childBounds[1], width: childBounds[2], height: childBounds[3]};
                }

                // if (isRightAligned) {
                //     childBounds.left = parentBounds.width - childBounds.left - childBounds.width;
                // }

                visualView.setBounds(childBounds);
            }
            if (this.shouldUpdateScrollbars()) {
                // If the available space has changes we need to re-layout.
                var containerBoundsChanged = ContainerUtil.updateScrollbars(this, children);
                if(containerBoundsChanged) {
                    // PERFOPT: If we waited until we new the layout was settled before we called _baseView.setFrame(bounds); for the tables and graphs
                    //     we could save a few cycles. For this we would need an additional function call,
                    //     something like this.updateFrameSize (terrible name),  that would be called  down the report hierarchy to the Visuals once the layout 
                    //     has settled.
                    this._recursionCount++;
                    this.layoutSubviews();
                    this._recursionCount--;
                }
            }
        },

        doBIRDLayout: function(controller, bounds) {
            var reportController = controller.getReportController();
            var layout = controller.getModel().getLayout();
            var i;
            var layoutContext = new LayoutContext();

            if (layout instanceof ResponsiveLayout) {
                var currentRange = sap.ui.Device.media.getCurrentRange(sap.ui.Device.media.RANGESETS.SAP_STANDARD);
                var mediaName = currentRange ? currentRange.name : 'Desktop';
                var windowSize = MEDIA_MAP[mediaName] || WindowSize.LARGE;
                reportController.setWindowSize(windowSize);
                //Set up layout specific context properties
                layoutContext.setTileGap(0);
                layoutContext.setContainerInset(0);
            }  else {
                layoutContext.setTileGap(TILE_GAP);
                layoutContext.setContainerInset(CONTAINER_INSET);
            }

            var layoutProxy = controller.getLayoutProxy();
            layoutProxy.setX(new Dimension(0, DimensionUnit.PIXELS));
            layoutProxy.setY(new Dimension(0, DimensionUnit.PIXELS));
            layoutProxy.setW(new Dimension(bounds.width ? bounds.width : 0, DimensionUnit.PIXELS));
            layoutProxy.setH(new Dimension(bounds.height ? bounds.height : 0, DimensionUnit.PIXELS));

            layoutContext.setPositionTiledContentRelativeToRoot(true);
            layoutContext.setEnableTileReflow(false);
            layoutContext.setDefaultMinimumHeight(reportController.getSASReportConfiguration().getSectionHeaderHeight());
            layoutContext.setEnableTileMinimums(true);
            layoutContext.setDisableRecursion(true);
            layoutContext.setIgnoreContainerPadding(true);
            if (LocaleUtil.isRTL()) {
                layoutContext.setInvertXSpace(true);
            }

            layoutProxy.layout(layoutContext);
        },

        refreshConstraints: function() {
            var children = this._containerContent.getChildren();
            var numChildren = children.length;
            for (var i = 0; i < numChildren; i++) {
                // if someone passes in true, we only want the event fired once
                var child = children[i];
                if (child instanceof BaseContainer) {
                    child.refreshConstraints();  
                } else if (child instanceof StackContainer) {
                    var views = child.getAllVisualViews();
                    for(var j = 0, len2 = views.length; j < len2; ++j) {
                        var view = views[j];
                        if (view instanceof BaseContainer) {
                            view.refreshConstraints();  
                        } else if (view instanceof BIRDTitledContent) {
                            view.updateSizeConstraints(false);
                        }
                    }
                } else if (child instanceof BIRDTitledContent) {
                    child.updateSizeConstraints(false);
                }
            }

            // event fired last, so do other things before super
            this.updateSizeConstraints(false);
        },

        updateSizeConstraints: function() {
            BIRDTitledContent.prototype.updateSizeConstraints.apply(this, arguments);
            var measuredContent = this.getMeasuredContent();
            var layoutConstraint = this.getController().getLayoutProxy().getConstraint();
            layoutConstraint.setAdditionalContentHeight(measuredContent ? measuredContent.additionalContentHeight : 0);
            layoutConstraint.setAdditionalContentWidth(measuredContent ? measuredContent.additionalContentWidth : 0);
        },

        measureContent: function(constrainedWidth) {
            var contentSize = BIRDTitledContent.prototype.measureContent.apply(this, arguments);
            //Account for additionalContentHeight, the mechanism for Container to affect its min height.
            var additionalContentHeight = this.getTitleSize(constrainedWidth).height;
            var additionalContentWidth = 0;
            if(this.getLayoutType() === ContainerUtil.LayoutType.STACK) {
               var content = this._containerContent.getChildren();
                if(content) {
                    var stack = content[0];
                    if(stack instanceof StackContainer) {
                        var measuredContent = stack.getMeasuredContent();
                        if (measuredContent) {
                            additionalContentHeight += measuredContent.height;
                            additionalContentWidth += measuredContent.width;
                        }
                    }
                }
            }
            contentSize.additionalContentHeight = additionalContentHeight;
            contentSize.additionalContentWidth = additionalContentWidth;
            return contentSize;
        },

        /**
         * Returns a list of UI child views that are focusable. Contains things
         *   like CompositeContainer and Visual
         * @returns {Array} array of objects - visuals and composite visual containers
         */
        getFocusableContent: function() {
            var allBIRDTitledContents = [];
            var children = this._containerContent.getChildren();
            if (children) {
                for (var i = 0, length = children.length; i < length; i++) {
                    var child = children[i];
                    if (child) {
                        allBIRDTitledContents = allBIRDTitledContents.concat(child.getFocusableContent());
                    }
                }
            }
            return allBIRDTitledContents;
        },

        /**
         * Returns all of the child visuals.
         * @returns {Visual[]} all child visuals
         */
        getAllVisualViews: function() {
            var allVisualViews = [];
            var visualView;
            var children = this._containerContent.getChildren();
            if (children) {
                for (var i = 0, length = children.length; i < length; i++) {
                    visualView = children[i];
                    if (visualView) {
                        allVisualViews = allVisualViews.concat(visualView.getAllVisualViews());
                    }
                }
            }
            return allVisualViews;
        },

        propagateHidden: function(value) {
            var children = this._containerContent.getChildren();
            if (children) {
                for (var i = 0, length = children.length; i < length; i++) {
                    var child = children[i];
                    if(child instanceof BIRDTitledContent) {
                        child.propagateHidden(value);
                    }
                }
            }
        },

        getLayoutType: function () {
            if (!this._layoutType) {
                this._layoutType = ContainerUtil.getLayoutTypeForController(this.getController());
            }
            return this._layoutType;
        },

        isHiddenOverlflow: function () {
            return this._hiddenOverflow;
        },

        setShouldUpdateScrollbars: function(shouldUpdate) {
            this._shouldUpdateScrollbars = shouldUpdate;
        },

        // S1505580: Hack to avoid infinite loop during layout
        shouldUpdateScrollbars: function() {
            return this._shouldUpdateScrollbars;
        },

        applyContainerStyles: function () {
            if (this._controller instanceof sas.ltjs.BIRD.controllers.view.MediaContainerController) {
                return;
            }
            var styleChain = this._controller.getContainerBackgroundStyleChain();
            var css;

            var bgColor = StyleUtils.getBackgroundColorString(styleChain);
            if(bgColor) {
                css = css || {};
                css['background-color'] = bgColor;
            }
            css = StyleUtils.getAllBorderStyles(styleChain, css);

            if (css) {
                this.setCustomStyles(new Styles().setStyles(css));
            }
        },

        _addEventListeners: function () {
            this._handles.push(this._controller.getLayoutProxy().attachEvent(PreferredSizeRequestEvent.PREFERRED_HEIGHT_REQUEST, this._onPreferredHeightRequest, this));
        },

        _onPreferredHeightRequest: function(preferredSizeRequestEvent) {
            this.measureContent(preferredSizeRequestEvent.getConstrainedSize());
            this.updateSizeConstraints();
        },

        _removeEventListeners: function () {
            if (this._handles) {
                for (var i = this._handles.length - 1; i >= 0; i--) {
                    this._handles[i].detach();
                }
            }
            this._handles = null;
        },

        destroy: function () {
            if (this._onScroll && this._containerContent) {
                this._containerContent.detachBrowserEvent('scroll', this._onScroll);
                this._onScroll = null;
            }
            this._removeEventListeners();
            
            BIRDTitledContent.prototype.destroy.apply(this, arguments);
        }
    });
    return BaseContainer;
}, true);
