sap.ui.define([
    "sas/vaviewer/views/VisualElementView",
    "sas/ltjs/BIRD/controllers/visualElements/WebContentElementController",
    "sas/hc/ui/core/HTML",
    "sas/ltjs/BIRD/util/hud/DataStateChangeEvent",
    "sas/vaviewer/transport/TransportConnection"
], function(
    VisualElementView,
    WebContentElementController,
    HTML,
    DataStateChangeEvent,
    TransportConnection
) {
    "use strict";
    return VisualElementView.extend("sas.vaviewer.views.WebContentElementView", {
        
        renderer: sas.vaviewer.views.VisualElementViewRenderer,

        init: function() {
            VisualElementView.prototype.init.apply(this, arguments);
            this._iframeId = this.getId() + "-iframe";
            this._handles = null;
            this._url = null;
            this._boundIframeDataHandler = this._iframeDataHandler.bind(this);
            this._boundOnDataChange = this._onDataChange.bind(this);
        },

        setController: function (controller) {
            VisualElementView.prototype.setController.apply(this, arguments);
            if (!controller) {
                return;
            }   
            // If we already have any children, destroy them
            this.destroyChildren();
            var webContentElement = controller.getModel();
            if (!webContentElement) {
                return;
            }
            var url = webContentElement.getUrl();
            if (!url) {
                return;
            }
            this._url = url;
            this._addEventListeners();
            var html = new HTML({
                content: this._getContentHTML(url)
            });
            this.addChild(html);
        },

        _getContentHTML: function(url) {
            return '<iframe id = ' + this._iframeId + ' class="stpFrame" src="' + url + '" sandbox="' + this._getIframeSandbox() + '"></iframe>';
        },

        _getIframeSandbox: function() {
            var reportController = this._controller.getReportController();
            if (reportController) {
                var sasReportConfiguration = reportController.getSASReportConfiguration();
                if (sasReportConfiguration) {
                    var iframeSandbox = TransportConnection.getIframeSandboxForSASReportConfiguration(sasReportConfiguration);
                    if (iframeSandbox) {
                        return iframeSandbox;
                    }
                }
            }

            return "allow-presentation allow-scripts allow-same-origin allow-forms";
        },

        _addEventListeners: function () {
            // We need to listen to data change only if the controller has data
            if (this._controller.hasData()) {
                this._handles = this._handles || [];
                this._handles.push(this._controller.attachEvent(DataStateChangeEvent.DATACHANGE, this._boundOnDataChange, this));
                window.addEventListener("message", this._boundIframeDataHandler, false);
            }
        },

        _removeEventListeners: function () {
            if (this._handles) {
                for (var i = 0, len = this._handles.length; i < len; ++i) {
                    this._handles[i].detach();
                }
                this._handles = null;
            }
            window.removeEventListener("message", this._boundIframeDataHandler);
        },

        _onDataChange: function(event) {
            if (!this._controller) {
                return;
            }
            var jsonString = this._controller.getJSONXData();
            if (!jsonString) {
                return;
            }
            var jsonData = {};
            try {
                jsonData = JSON.parse(jsonString);
            } catch (e) {
                console.error("Error parsing the JSON string: " + jsonString, e);
                return;
            }
            var iframe = document.getElementById(this._iframeId);
            if (iframe && iframe.contentWindow) {
                iframe.contentWindow.postMessage(jsonData, this._url);
            }
        },

        /**
         * Event handler for the postMessage call from Data Driven Content iframes
         * @param {Object} event - event object passed by the postMessage call
         * @returns {void}
         * @private
         */
        _iframeDataHandler: function(event) {
            if (!event) {
                return;
            }
            if (!event.data || !event.origin) {
                return;
            }
            // Since this post message listener is on window, we might end up hitting here from several places.
            // Check if the message is being sent from same url as the iframe with which we set it up.
            // It is possible that more than one data driven content have the same url being used, in which case,
            // this check would not suffice. So if user interacts with one web content, we would end up hitting 
            // this listener for other web content having same url. However, portable controller has check to 
            // see that the controller is getting the JSON data with the right resutlName, else it gets ignored.
            //
            if (this._url.indexOf(event.origin) === -1) {
                return;
            }
            // Extra checks to confirm that we are getting the expected objects in the JSON data.
            if (!event.data.resultName || !event.data.selections) {
                return;
            }
            var jsonString = JSON.stringify(event.data);
            this._controller.selectFromJSONX(jsonString);
        },

        onBeforeRendering: function() {
            VisualElementView.prototype.onBeforeRendering.apply(this, arguments);
            var iframe = document.getElementById(this._iframeId);
            if (iframe) {
                iframe.removeEventListener("load", this._boundOnDataChange);
            }
        },

        onAfterRendering: function() {
            VisualElementView.prototype.onAfterRendering.apply(this, arguments);
            // Add an event listener on the load event of the iframe. The iframe gets reset and loads 
            // again when we maximize / minimize it since we move it within out DOM structure. 
            // So if the iframe gets reset, we have to again post data to it.
            var iframe = document.getElementById(this._iframeId);
            if (iframe) {
                iframe.addEventListener("load", this._boundOnDataChange, false);
            }
        },

        getMinimumWidth: function() {
            return 100;
        },

        getMinimumHeight: function() {
            return 100;
        },


        canSaveAsImage: function() {
            return false;
        },

        canExportData: function() {
            return false;
        },

        destroy: function () {
            this._removeEventListeners();
            VisualElementView.prototype.destroy.apply(this, arguments);
        }
    });
}, true);
