sap.ui.define([
    "sas/hc/m/Button",
    "sas/hc/ui/core/Styles",
    "sas/ltjs/BIRD/controllers/view/VisualController",
    "sas/ltjs/BIRD/controllers/visualElements/DataVisualElementController",
    "sas/ltjs/BIRD/controllers/visualElements/PromptElementController",
    "sas/ltjs/BIRD/controllers/visualElements/VisualElementController",
    "sas/ltjs/BIRD/models/interactions/InteractionReason",
    "sas/ltjs/BIRD/util/hud/DataStateChangeEvent",
    "sas/ltjs/BIRD/util/hud/VisualSelectionEvent",
    "sas/vaviewer/views/BIRDTitledContent",
    "sas/vaviewer/views/BIRDViewFactory",
    "sas/vaviewer/views/Failure",
    "sas/vaviewer/views/VisualEvent",
    "sas/vaviewer/views/visualBreadcrumb/Breadcrumb",
    "sas/ltjs/commons/layout/PreferredSizeRequestEvent",
    "sas/ltjs/commons/models/warnings/Warning",
    "sas/ltjs/commons/models/warnings/WarningEvent",
    "sas/ltjs/commons/styles/AttributeNames",
    "sas/ltjs/commons/styles/PredefinedAttributeBundles",
    "sas/ltjs/commons/util/Dimension",
    "sas/ltjs/commons/util/DimensionUnit",
    "sas/ltjs/commons/util/StyleUtils",
    "sas/ltjs/commons/views/UISizing",
    "sas/vaviewer/views/ViewerEvents",
    "sas/hc/ui/commons/windowshade/WindowShade",
    "sas/hc/ui/commons/windowshade/WindowShadeSection",
    "sas/hc/ui/commons/FormattedTextView",
    "sap/ui/core/ResizeHandler",
    "sas/vacommon/controls/pill/VisualElementStatusEnum",
    "sas/vaviewer/views/ReportElementStatusDecorator",
    "sas/vacommon/controls/VAToast",
    "sas/ltjs/BIRD/controllers/visualElements/GraphElementController",
    "sas/ltjs/BIRD/util/RichTextStringUtil",
    "sas/vaviewer/views/prompts/Slider"
], function(
    Button,
    Styles,
    VisualController,
    DataVisualElementController,
    PromptElementController,
    VisualElementController,
    InteractionReason,
    DataStateChangeEvent,
    VisualSelectionEvent,
    BIRDTitledContent,
    BIRDViewFactory,
    Failure,
    VisualEvent,
    Breadcrumb,
    PreferredSizeRequestEvent,
    Warning,
    WarningEvent,
    AttributeNames,
    PredefinedAttributeBundles,
    Dimension,
    DimensionUnit,
    StyleUtils,
    UISizing,
    ViewerEvents,
    WindowShade,
    WindowShadeSection,
    FormattedTextView,
    ResizeHandler,
    VisualElementStatusEnum,
    ReportElementStatusDecorator,
    VAToast,
    GraphElementController,
    RichTextStringUtil,
    Slider
) {
    "use strict";

    var rb = sap.ui.getCore().getLibraryResourceBundle("sas.vaviewer");

    var BUSY_DELAY = 350; //Set the busy indicator delay to be 350 milliseconds (same as iOS)
    
    //
    // duration to show the message toast indicating
    // the bottom of the drill hierarchy.
    //
    var TOAST_DURATION = 4000;

    return BIRDTitledContent.extend("sas.vaviewer.views.Visual", {

        metadata: {
            aggregations: {
                "statusDecorator": {type: "sas.vaviewer.views.ReportElementStatusDecorator", multiple: false}
            }
        },

        renderer: {
        },

        init: function () {
            BIRDTitledContent.prototype.init.call(this);
            this.addStyleClass("Container");
            this.setIcon(null);
            this._failureMessage = null;
            this._hasSevereWarning = false;
            this._breadcrumbs = [];
            this._expandButton = null;
            this._visualElementController = null;
            this._alertTooltip = null;
            this._visualElementView = null;
            this._warningMessage = null;
            this._warningSeverity = 0;
            this._aboutInfoWindowShade = null;
            this._aboutInfoResizeListenerId = null;
            this._busyDelayTimer = null;
            this._toast = null;
            
            var statusDecorator = new ReportElementStatusDecorator({
                reportElement: this,
                message: rb.getText("Visual.loading.txt")
            });
            statusDecorator.addEventDelegate({
                ontap: this._onStatusTap.bind(this)
            });
            this.setStatusDecorator(statusDecorator);    

            // PromptContainers get their padding from the report theme
            this.addStyleClass("ReportElementContainer");
        },

        _setBusy: function(isBusy, isAutoRefresh) {
            if(!isBusy || isAutoRefresh) {
                this._setBusyImpl(isBusy, isAutoRefresh);
            } else {
                //
                // Delay the busy state by BUSY_DELAY.
                // If there is already a timer running then let it
                // run to completion.
                //
                if(!this._busyDelayTimer) {
                    this._busyDelayTimer = setTimeout(this._setBusyImpl.bind(this, true), BUSY_DELAY);
                }
            }
        },

        _setBusyImpl: function (isBusy, isAutoRefresh) {
            //
            // If there is a busy timer then stop it.
            //
            if(this._busyDelayTimer) {
                clearTimeout(this._busyDelayTimer);
                this._busyDelayTimer = null;
            }
            //
            // We should have a status decorator
            //
            var statDecorator = this.getStatusDecorator();
            if(!statDecorator) {
                return;
            }
            //
            // Add/Remove busy state.
            //
            if(isBusy) {
                this.addStyleClass("vavBusy");
                statDecorator.setStatus(VisualElementStatusEnum.BUSY);
            } else {
                this.removeStyleClass("vavBusy");
                statDecorator.setStatus(VisualElementStatusEnum.NORMAL);
            }
            //
            // If this is busy as a result of an auto refresh, then hide the busy indicator
            // until the user clicks on the visual to try to interact with it.
            // This reduces the visual clutter with a rapid refresh rate.
            //
            if (isBusy && isAutoRefresh) {
                this.addStyleClass("vavStatusHidden");
                statDecorator.addStyleClass("vavStatusHidden");
            } else {
                this.removeStyleClass("vavStatusHidden");
                statDecorator.removeStyleClass("vavStatusHidden");
            }
        },

        _onStatusTap: function(event) {
            //
            // Clicking on the busy status will unhide it
            //
            this.removeStyleClass("vavStatusHidden");
            var statDecorator = this.getStatusDecorator();
            if(statDecorator) {
                statDecorator.removeStyleClass("vavStatusHidden");
            }
        },

        onAfterRendering: function() {
            BIRDTitledContent.prototype.onAfterRendering.apply(this, arguments);
            this.layoutSubviews();
            // HACK: to fire event after visual is rendered to enable bubbling of event to parent
            this._hideObjectToolbarOnSevereWarning();
        },

        onBeforeRendering: function() {
            BIRDTitledContent.prototype.onBeforeRendering.apply(this, arguments);
            // S1385221: Having breadcrumbs in the visual causes invalidation and the status 
            // decorator gets rendered outside of the visual. So checking here if it is already
            // rendered and removing it so that it gets rendered through _renderWrapper.
            // This works because we don't attach event listeners to `statusDecorator`
            // HACK: This is a hack because the rendering of `statusDecorator` subverts the natural
            //     flow of things and renders outside of it's parent.
            var statusDecorator = this.getStatusDecorator();
            if (statusDecorator) {
                var statusDecoratorDom = statusDecorator.getDomRef();
                if (statusDecoratorDom) {
                    statusDecoratorDom.parentNode.removeChild(statusDecoratorDom);
                }
            }
        },

        setController: function (controller) {
            this._removeListeners();
            BIRDTitledContent.prototype.setController.apply(this, arguments);

            if (controller) {
                this.removeAllChildren();
                this._handles = this._handles || [];

                this._visualElementController = controller.getVisualElementController().retain();

                // This must be called after setting our visualElementController, because
                // it uses that to resolve padding.
                this.applyPaddingFromController(BIRDTitledContent.DEFAULT_PADDING);

                // set role="complementary" for a11y landmark
                this.setRole("complementary");

                // Only composite visual members are not selectable
                if (!this.getIsCompositeMember()) {
                    this.setSelectable(true);
                }
                if(this._visualElementController.isBusy()){
                    this._setBusy(true);
                } else {
                    this._updateBreadcrumbs();
                }
                this.applyVisualElementContainerStyles();
                
                if (!this.getIsCompositeMember()) {
                    // Initialize the aria labels.  Also listen for update events from the visual element
                    // in order to make aria label changes when needed.
                    this.updateAriaLabels();

                    if(this._visualElementView) {
                        this._visualElementView.detachEvent(VisualEvent.UPDATE_ARIA_LABELS, this.updateAriaLabels);
                    }
                }

                // This line is needed because BIRDViewFactory somehow does not get pulled in via sap defines (circular dependency?)
                BIRDViewFactory = sas.vaviewer.views.BIRDViewFactory;

                this._visualElementView = BIRDViewFactory.createVisualElementView(this._visualElementController);
                if(this._visualElementView) {
                    this.addChild(this._visualElementView);
                    if (!this.getIsCompositeMember()) {
                        this._visualElementView.attachEvent(VisualEvent.UPDATE_ARIA_LABELS, this.updateAriaLabels, this);
                    }
                }
                // Slider hack for overflowing bounds. Trying to minimize code impact.
                if(this._visualElementView instanceof Slider){
                    this.addStyleClass("vavOverflowHidden");
                }

                this._addListeners();

                this._visualElementController.updateWarningStatus();
                this._visualElementView.setSizing(UISizing.FILL);
            }
            // Update the about info, if present. We do not get DATACHANGE event on switching pages.
            this._updateAboutInfo();
        },

        setSelected: function (isSelected, playbackAnimation) {
            BIRDTitledContent.prototype.setSelected.apply(this, arguments);

            if (playbackAnimation && this._visualElementView) {
                if (isSelected) {
                    this._visualElementView.playAnimation();
                } else {
                    this._visualElementView.pauseAnimation();
                }
            }
        },

        // virtual
        focus: function () {
            return this._visualElementView.focus();
        },

        propagateHidden: function(value) {
            //If the content is hidden don't upate the graph and remove it from layout calculations.
            this._visualElementView.propagateHidden(value);
        },

        // Override this function to give Visual specific implementation.
        getVisualSelectionDisabled: function() {
            var isParentSelectionDisabled = BIRDTitledContent.prototype.getVisualSelectionDisabled.call(this);
            if (isParentSelectionDisabled) {
                return true;
            }
            if (this._visualElementController) {
                if (this._visualElementController instanceof DataVisualElementController) {
                    return (this._visualElementController.isPromptFilter() || this._visualElementController.isSectionFilter());
                }
            }
            return false;
        },

        getVisualElementController: function() {
            return this._visualElementController;
        },

        _setFailure: function(value){
            if (value) {
                if (!this._failureMessage) {
                    this._failureMessage = new Failure({
                        messageText: value
                    });
                    this.addChild(this._failureMessage);
                } else {
                    this._failureMessage.setMessageText(value);
                }
            }
        },

        _getFailure: function() {
            return this._failureMessage;
        },

        _hideObjectToolbarOnSevereWarning: function() {
            if(this._hasSevereWarning) {
                // for the case when warnings for visuals are generated at a later time after rendering
                this.setOverlaysEnabled(false);
                if(this.getIsCompositeMember()) {
                    // fire an event for warning generated to handle in composite container
                    this.fireEvent(VisualEvent.WARNING_GENERATED, {}, true, true);
                }
            }
        },

        /**
         * Event handler for all the portable dispatched warnings
         * @param {sas.ltjs.commons.models.warnings.WarningEvent} warnings portable warnings object
         * @returns {void}
         * @private
         */
        _onWarning: function(warnings) {
            if (!warnings) {
                return;
            }
            var allWarnings = warnings.getWarnings();

            //
            // Filter out all warnings with severity < 40.0.
            // Defect S1327500.
            //
            if(allWarnings) {
                allWarnings = allWarnings.filter(function(warning) {
                    return warning.getSeverity() >= 40;
                });
            }

            if (!allWarnings || !allWarnings.length) {
                this.setIcon(null);     //Remove any previously set warning
                if (this._failureMessage) {
                    this._removeSevereWarning();
                }
                return;
            }
            // allWarnings is a sorted array returned by portable in descending order of the severity.
            // If there is a severe warning, just show the top most one. Do not combine all other messages.
            if (allWarnings[0].getSeverity() >= 90) {
                this._hasSevereWarning = true;
                this._handleSevereWarning(allWarnings[0].getMessage());
            } else {
                var allWarningMessages = "";
                for (var i = 0, length = allWarnings.length; i < length; ++i) {
                    allWarningMessages += allWarnings[i].getMessage();
                }
                this._handleRegularWarning(allWarningMessages);
            }
        },

        /**
         * Handles severe warning messages
         * @param {string} severeWarningMessage severe warning message
         * @private
         */
        _handleSevereWarning: function(severeWarningMessage) {
            this.setTitleEnabled(false);
            this.setIcon(null);
            var children = this.getChildren();
            if (children) {
                for (var i = 0, length = children.length; i < length; ++i) {
                    var child = children[i];
                    if (child) {
                        // Do not make the existing failure message invisible.
                        if (this._failureMessage && child === this._failureMessage) {
                            continue;
                        }
                        child.addStyleClass("vaInvisible");
                    }
                }
            }
            this._hideObjectToolbarOnSevereWarning();
            this._setFailure(severeWarningMessage);
            // remove any override styles
            this.setCustomStyles(undefined);
            if (this._visualElementView) {
                this._visualElementView.setBaseViewSize({width: 0, height: 0});
            }
        },

        /**
         * handles regular warning messages
         * @param {string} allWarningMessages All warning messages combined
         * @private
         */
        _handleRegularWarning: function (allWarningMessages) {
            this.setIcon(sas.icons.HC.ALERT);
            var iconCtrl = this.getIconButton();
            if(iconCtrl) {
                iconCtrl.addStyleClass("alertIcon");
                iconCtrl.setTooltip(allWarningMessages);
            }
        },

        _removeSevereWarning: function() {
            this.setTitleEnabled(true);
            var children = this.getChildren();
            if (children) {
                for (var i = 0, length = children.length; i < length; ++i) {
                    var child = children[i];
                    if (child) {
                        if (this._failureMessage && child === this._failureMessage) {
                            this.removeChild(child);
                            this._failureMessage.destroy();
                            this._failureMessage = null;
                        } else {
                            child.removeStyleClass("vaInvisible");
                        }
                    }
                }
            }
            this._hasSevereWarning = false;
            var decorator = this.getReportElementDecorator();
            if(decorator) {
                decorator.getObjectToolbar().setVisible(true);
            }
        },

        layoutSubviews: function () {
            BIRDTitledContent.prototype.layoutSubviews.apply(this, arguments);
            var view = this._visualElementView;
            if(view) {
                view.layoutSubviews();
            }
            if(this._failureMessage) {
                this._failureMessage.layoutSubviews();
            }
            this.layoutDecorator();
        },

        _onVisualToast: function (event) {
            var visualDom = this.getDomRef();
            var message = event.mParameters.message;
            this._toast = new VAToast({
                message: message,
                undoable: false
            });
            this._toast.show(TOAST_DURATION, visualDom);
        },

        _addListeners: function () {
            //S1137887: Removed the conditional adding of event handler for generic warnings. - Shrikant 02/18/2015
            this._handles.push(this._controller.attachEvent(WarningEvent.GENERIC, this._onWarning, this));
            this._handles.push(this._visualElementController.attachEvent(DataStateChangeEvent.DATACHANGE, this._onDataChange, this));
            this._handles.push(this._controller.attachEvent(VisualElementController.READY, this._onVisualElementControllerReady, this));
            this._handles.push(this._controller.attachEvent(VisualElementController.BUSY, this._onVisualElementControllerBusy, this));
            this._handles.push(this._controller.getLayoutProxy().attachEvent(PreferredSizeRequestEvent.PREFERRED_HEIGHT_REQUEST, this._onPreferredHeightRequest, this));
            var visualElementController = this.getVisualElementController();
            // This event is dispatched only by GraphElementController
            if (visualElementController instanceof GraphElementController) {
                this._handles.push(visualElementController.attachEvent(GraphElementController.TITLE_CHANGED, this._onTitleChanged, this));
            }
            this.attachEvent(VisualEvent.VISUAL_TOAST, this._onVisualToast);
            this.attachEvent(ViewerEvents.UPDATE_CONTEXT_MENU, this.onUpdateContextMenu);
        },

        _removeListeners: function () {
            if (this._handles) {
                for (var i = this._handles.length - 1; i >= 0; i--) {
                    this._handles[i].detach();
                }
            }
            this._handles = null;
            this.detachEvent(VisualEvent.VISUAL_TOAST, this._onVisualToast);
            this.detachEvent(ViewerEvents.UPDATE_CONTEXT_MENU, this.onUpdateContextMenu);
        },

        _onDataChange: function (/* Event.js */ event) {
            this.fireEvent(VisualEvent.VISUAL_DATA_CHANGED, {
                id: this.getId(),
                visual: this
            }, true, true);
            this._updateBreadcrumbs();
            if(event.getStructureChanged()) {
                this._updateAboutInfo();
            }
        },

        _onVisualElementControllerReady: function(/* Event.js */ event){
            this._setBusy(false);
        },

        _onVisualElementControllerBusy: function(event){
            var isAutoRefresh = event.getInteractionReason() === InteractionReason.REFRESH;
            this._setBusy(true, isAutoRefresh);
        },

        _updateBreadcrumbs: function() {
            if (!(this._visualElementController instanceof DataVisualElementController)) {
                return;
            }
            var breadcrumbsExist = !!this._breadcrumbs.length;
            var breadcrumb;
            var i, length;

            var drillControllers = this._visualElementController.getAllDrillControllers();
            if(drillControllers && drillControllers.length) {
                //iterate backwards so that insertChild(item) adds breadcrumbs in the correct order.
                for(i = drillControllers.length - 1; i >= 0; --i) {
                    var drillController = drillControllers[i];
                    if(breadcrumbsExist && this._breadcrumbs[i]) {
                        this._breadcrumbs[i].setDrillController(drillController);
                    } else {
                        breadcrumb = new Breadcrumb({
                            visualElementController: this._visualElementController
                        });
                        //Hack/order dependency issue. breadcrumb sets up on setDrillController.
                        breadcrumb.setDrillController(drillController);
                        this._breadcrumbs[i] = breadcrumb;
                        this.insertChild(breadcrumb);
                    }
                }
                if(breadcrumbsExist) {
                    this.layoutSubviews(); //in case number of visible breadcrumbs has changed
                }
            } else {
                // Special case: VAE created hierarchy.
                var results = this._visualElementController.getResultControllers();
                if(!results) {
                    return;
                }
                for (i = 0, length = results.length; i < length; ++i) {
                    var result = results[i];
                    var hierarchyDataset = result.getHierarchyDataSet();
                    if(!hierarchyDataset) {
                        continue;
                    }

                    var hierarchies = hierarchyDataset.getHierarchies();
                    if(!hierarchies || !hierarchies.length){
                        continue;
                    }

                    for(var j = hierarchies.length - 1; j >= 0; --j) {
                        var hierarchy = hierarchies[j];
                        var drillable = hierarchy.canDrillUp();
                        if(!drillable) {

                            var levels = hierarchies[j].getLevels();
                            if(levels.length > 1) {
                                breadcrumb = new Breadcrumb();
                                breadcrumb.createDisabledItems(hierarchy);
                                this._breadcrumbs.push(breadcrumb);
                                this.insertChild(breadcrumb);
                            }
                        }
                    }
                }
            }
        },

        _updateAboutInfo: function() {
            var section = null;

            // Clean up any old resize listener
            if(this._aboutInfoResizeListenerId) {
                ResizeHandler.deregister(this._aboutInfoResizeListenerId);
                this._aboutInfoResizeListenerId = null;
            }

            // Get the AboutInfo from the controller and create the new WindowShadeSection.
            // We use an HTML list inside a FormattedTextView to match what VAN does.
            if(this._visualElementController) {
                var aboutInfo = this._visualElementController.getAboutInfo();
                if(aboutInfo) {
                    var messages = aboutInfo.getMessages();
                    if(messages && messages.length > 0) {
                        var messageHTMLText = "<ul>";
                        for(var i = 0; i < messages.length; i++) {
                            messageHTMLText = messageHTMLText + "<li>" + jQuery.sap.encodeHTML(messages[i]) + "</li>";
                        }
                        messageHTMLText += "</ul>";
                        var messageText = new FormattedTextView({
                            htmlText: messageHTMLText
                        });
                        messageText.addStyleClass("vavAboutForecast");
                        section = new WindowShadeSection({
                            title: aboutInfo.getLabel(),
                            content: [messageText]
                        });
                    }
                }
            }

            if(section) {
                // Create a new WindowShade and add it as a child. If one already
                // exists then destroy the existing sections, because then new one
                // will be added.
                if(!this._aboutInfoWindowShade) {
                    this._aboutInfoWindowShade = new WindowShade({
                        width: "100%",
                        height: "auto"
                    });
                    this.addChild(this._aboutInfoWindowShade);
                    this._bindResizeDelegates(this._aboutInfoWindowShade);
                } else {
                    this._aboutInfoWindowShade.destroySections();
                }
                this._aboutInfoWindowShade.addSection(section);
            } else if(this._aboutInfoWindowShade) {
                // No about info, but we have an old window shade that
                // needs cleanup.
                this.removeChild(this._aboutInfoWindowShade);
                this._aboutInfoWindowShade.destroy();
                this._aboutInfoWindowShade = null;
            }
        },

        _bindResizeDelegates: function(aboutInfoWindowShade) {
            aboutInfoWindowShade.addDelegate({
                onBeforeRendering: function() {
                    if(this._aboutInfoResizeListenerId) {
                        ResizeHandler.deregister(this._aboutInfoResizeListenerId);
                        this._aboutInfoResizeListenerId = null;
                    }
                }.bind(this),
                onAfterRendering: function() {
                    this._aboutInfoResizeListenerId = ResizeHandler.register(aboutInfoWindowShade.getDomRef(), this._onWindowShadeResize.bind(this));
                }.bind(this)
            });
        },

        _onWindowShadeResize: function() {
            this.fireEvent(VisualEvent.LAYOUT_CHANGED, { }, true, true);
        },

        _renderChildren: function( rm ) {
            BIRDTitledContent.prototype._renderChildren.call(this, rm);
            rm.renderControl(this.getReportElementDecorator());
        },

        _renderWrapper: function(rm, callback) {
            callback(rm, this);
            rm.renderControl(this.getStatusDecorator());
        },

        getVisualElementView: function() {
            return this._visualElementView;
        },

        applyVisualElementContainerStyles: function () {
            var styleChain = this._visualElementController.getBackgroundStyleChain();
            if(styleChain) {
                var css;

                var bgColor = StyleUtils.getBackgroundColorString(styleChain);
                if (bgColor) {
                    css = css || {};
                    css['background-color'] = bgColor;
                }
                css = StyleUtils.getAllBorderStyles(styleChain, css);

                if (css) {
                    this.setCustomStyles(new Styles().setStyles(css));
                }
            }  
        },

        getMinimumWidth: function() {
            return this._visualElementView.getMinimumWidth();
        },

        getMinimumHeight: function() {
            return this._calculateHeight(this._visualElementView.getMinimumHeight());
        },

        getPreferredHeight: function(constrainedWidth) {
            return this._calculateHeight(this._visualElementView.getPreferredHeight(constrainedWidth), constrainedWidth);
        },

        getPreferredWidth: function(constrainedHeight) {
            return this._visualElementView.getPreferredWidth(constrainedHeight);
        },

        _calculateHeight: function(visualElementHeight, constrainedWidth) {
            // only return a height if the view has one
            if (!isNaN(visualElementHeight) && visualElementHeight !== undefined && visualElementHeight !== null) {
                var titleHeight = this.getTitleSize(constrainedWidth).height;
                var breadCrumbHeight = 0;
                if(this._breadcrumbs) {
                    for (var i = 0; i < this._breadcrumbs.length; i++) {
                        var breadcrumb = this._breadcrumbs[i];
                        if(!breadcrumb.hasStyleClass("vaInvisible")) {
                            breadCrumbHeight += breadcrumb.$().outerHeight(true);
                        }
                    }
                }
                var aboutInfoHeight = 0;
                if(this._aboutInfoWindowShade) {
                    aboutInfoHeight = this._aboutInfoWindowShade.$().outerHeight(true);
                }

                var toolBarHeight = 0;
                // if(this.getReportElementDecorator()) {
                //     var $toolBar = this.getReportElementDecorator().getObjectToolbar().$();
                //     if ($toolBar.length > 0) {
                //         toolBarHeight = $toolBar.outerHeight(true) + $toolBar.position().top;
                //     }
                // }
                
                return Math.max(toolBarHeight, visualElementHeight + titleHeight + breadCrumbHeight + aboutInfoHeight);
            }
            return NaN;
        },

        _onPreferredHeightRequest: function(preferredSizeRequestEvent) {
            if (this._controller) {
                var layoutConstraint = this.getLayoutConstraint();
                var constrainedWidth = preferredSizeRequestEvent.getConstrainedSize();

                // Padding and border is included in the constrained size for visuals in relative layouts only.
                // The if condition was inspired by ResponsiveLayoutHelper::isRelativeLayout()
                var layoutProxy = this._controller.getLayoutProxy();
                var constraint = layoutProxy && layoutProxy.getConstraint();
                if (constraint && constraint.getHorizontalOffset() && constraint.getVerticalOffset()) {
                    var paddingDimension = layoutConstraint.getPadding();
                    var padding = paddingDimension ? paddingDimension.getScalarValueForUnit(DimensionUnit.PIXELS, -1.0, 1.0) : 0;
                    var border = layoutConstraint.getBorderThickness();
                    constrainedWidth -= padding * 2 + border * 2;
                }

                var preferredHeight = this.getPreferredHeight(constrainedWidth);
                // if we don't have a preferredHeight, one may have been set by the BIRD so don't override it
                if (preferredHeight) {
                    layoutConstraint.setHeight(new Dimension(preferredHeight, DimensionUnit.PIXELS));
                }
                // S1580423: width affects the height of the title, so minimumHeight needs to be recalculated
                var minimumHeight = this._calculateHeight(this._visualElementView.getMinimumHeight(), constrainedWidth);
                if (minimumHeight) {
                    layoutConstraint.setMinimumHeight(minimumHeight);
                }
            }
        },

        shouldCreateDecorator: function() {
            if(BIRDTitledContent.prototype.shouldCreateDecorator.apply(this, arguments) && !this._hasSevereWarning) {
                return true;
            }
            return false;
        },

        /**
         * can be saved as an image or not
         * @returns {boolean} true or false
         */
        canSaveAsImage: function() {
            return true;
        },

        /**
         * can data be exported or not
         * @returns {boolean} true or false
         */
        canExportData: function() { 
            return true;
        },

        getFocusableContent: function() {
            // We are collecting all the child views polymorphically. Just return itself since this is the leaf node.
            return [this];
        },

        getAllVisualViews: function() {
            return [this];
        },

        _onTitleChanged: function() {
            var visualElementController = this.getVisualElementController();
            if (!this._titleContainer || !this.getTitleEnabled() || !visualElementController) {
                return;
            }
            var titleText = RichTextStringUtil.parseHTMLString(visualElementController.getTitleAsHTML());
            this._titleContainer.setText(titleText);
        },

        destroy: function () {
            this._removeListeners();

            if(this._visualElementController) {
                this._visualElementController.release();
                this._visualElementController = null;
            }

            if(this._failureMessage) {
                this._failureMessage.destroy();
                this._failureMessage = null;
            }

            if(this._alertTooltip){
                this._alertTooltip.destroy();
                this._alertTooltip = null;
            }

            if(this._breadcrumbs) {
                for(var i = this._breadcrumbs.length - 1; i >= 0; --i) {
                    this._breadcrumbs[i].destroy();
                }
                this._breadcrumbs = null;
            }

            if(this._aboutInfoResizeListenerId) {
                ResizeHandler.deregister(this._aboutInfoResizeListenerId);
                this._aboutInfoResizeListenerId = null;
            }

            if(this._aboutInfoWindowShade) {
                this._aboutInfoWindowShade.destroy();
                this._aboutInfoWindowShade = null;
            }

            if(this._visualElementView) {
                this._visualElementView.detachEvent(VisualEvent.UPDATE_ARIA_LABELS, this.updateAriaLabels);
                this._visualElementView.destroy();
                this._visualElementView = null;
            }

            if(this._busyDelayTimer) {
                clearTimeout(this._busyDelayTimer);
                this._busyDelayTimer = null;
            }

            if(this._toast) {
                this._toast.destroy();
                this._toast = null;
            }

            BIRDTitledContent.prototype.destroy.apply(this, arguments);
        }

    });

}, true);
