sap.ui.define([
    "sas/ltjs/BIRD/util/RichTextStringUtil",
    "sas/vaviewer/views/reportDecorator/ReportElementDecoratorFactory",
    "sas/vaviewer/views/SizedContent",
    "sas/vaviewer/views/VisualEvent",
    "sas/ltjs/commons/util/Dimension",
    "sas/ltjs/commons/util/DimensionUnit",
    "sas/ltjs/commons/views/UIFlexBox",
    "sas/ltjs/commons/views/UIText",
    "sas/ltjs/commons/layout/constraints/LayoutConstraint",
    "sas/ltjs/commons/styles/PredefinedAttributeBundles",
    "sas/ltjs/commons/styles/AttributeNames"
], function(
    RichTextStringUtil,
    ReportElementDecoratorFactory,
    SizedContent,
    VisualEvent,
    Dimension,
    DimensionUnit,
    UIFlexBox,
    UIText,
    LayoutConstraint,
    PredefinedAttributeBundles,
    AttributeNames
) {
    "use strict";

    var BIRDTitledContent = SizedContent.extend("sas.vaviewer.views.BIRDTitledContent", {
        metadata: {
            properties: { 
                'titleEnabled': {type: "boolean", defaultValue: true},
                'expandable': {type: "boolean", defaultValue: true},
                'isCompositeMember': {type: "boolean", defaultValue: false},
                'vaViewerConfig': {type: "object", defaultValue: null},
                'shareEnabled': {type: "boolean", defaultValue: true}
            }
        },

        renderer: {},

        init: function () {
            SizedContent.prototype.init.apply(this, arguments);
            this._titleContainer = null;
            this.setFlexBox(UIFlexBox.FLEXCOLUMN);
            this._reportElementDecorator = null;
            this._mouseDown = false;
            this._controller = null;
            this._overlaysEnabled = true;
        },

        setBounds: function(){
            SizedContent.prototype.setBounds.apply(this, arguments);
            if(this._titleContainer && this._titleContainer.getText()) {
                this.updateSizeConstraints(true);
            }
        },

        setHidden: function(value) {
            var oldHidden = this.getHidden();
            SizedContent.prototype.setHidden.apply(this, arguments);
            if(oldHidden !== value) {
                this.propagateHidden(value);
            }
        },

        _updateOverlays: function() {
            var icon = this.getIconButton();
            if (icon) {
                icon.setVisible(this._overlaysEnabled);
            }

            var decorator = this.getReportElementDecorator();
            if (decorator) {
                decorator.setVisible(this._overlaysEnabled);
            }
        },

        setOverlaysEnabled: function (value) {
            if (this._overlaysEnabled !== value) {
                this._overlaysEnabled = value;
                this._updateOverlays();
            }
        },

        //overriden
        propagateHidden: function() {
        },

        setController: function (controller) {
            if(controller && controller.equals(this._controller)) {
                return;
            }
            if(this._controller) {
                this._controller.release();
            }
            this._controller = controller ? controller.retain() : null;
        },

        applyPaddingFromController: function (defaultPadding) {
            if(!this._controller) {
                return;
            }

            if(this._controller.isPaddingAvailable()) {
                //
                // If there is a VisualElementController, then get any override
                // padding applied to the element container.
                // If there is no override then this padding from the report theme css is used.
                //
                var vec = this.getVisualElementController();
                var paddingDimension;
                if(vec) {
                    var styleChain = vec.getElementContainerStyleChain();
                    if(styleChain && styleChain.hasAttributeAboveBase(AttributeNames.PADDING)) {
                        paddingDimension = styleChain.getDimensionAttribute(AttributeNames.PADDING);
                        if(paddingDimension) {
                            this.setPadding(paddingDimension.getScalarValueForUnit(DimensionUnit.PIXELS, -1.0, 1.0) + 'px');
                        }
                    }
                } else {
                    //
                    // If there is no VisualElementContainer, then get the padding
                    // directly from the controller.
                    //
                    paddingDimension = this._controller.getPadding();
                    var padding = paddingDimension ? paddingDimension.getScalarValueForUnit(DimensionUnit.PIXELS, -1.0, 1.0) : 0;
                    this.setPadding(padding + 'px');
                }
            } else if(typeof defaultPadding === "string") {
                this.setPadding(defaultPadding);
            }
        },

        getLayoutConstraint: function () {
            if (!this._controller) {
                return null;
            }

            var layoutProxy = this._controller.getLayoutProxy();
            var layoutConstraint = layoutProxy.getConstraint();
            if (!layoutConstraint) {
                layoutConstraint = new LayoutConstraint();
                layoutProxy.setDefaultConstraint(layoutConstraint);
            }
            return layoutConstraint;
        },

        getReportElementDecorator: function() {
            return this._reportElementDecorator;
        },

        getController: function () {
            return this._controller;
        },

        ontap: function(oEvent) {
            if(this.getSelectable()) {
                this.setSelected(true);
                // If we clicked on a non-focuable element, default to this.focus()
                if(!document.activeElement || document.activeElement === window.document.body) {
                        this.focus();
                }
                //so the Application.view does not get this event and clear out the selected item.
                oEvent.stopPropagation();
            }
        },

        onmousedown: function() {
            this._mouseDown = true;
        },

        //event handler for sas.hc.ui components
        onclick: function() {
            if (this.getSelectable()) {
                this.setSelected(true);
                // If we clicked on a non-focuable element, default to this.focus()
                if(!document.activeElement || document.activeElement === window.document.body) {
                    this.focus();
                }
            }
        },

        setExpandable: function(value) {
            this.setProperty("expandable", value, true);
        },

        onfocusin: function() {
            if (this.getSelectable()) {
                if (!this._mouseDown) {
                    this.setSelected(true);
                }
                this._mouseDown = false;
            }
        },
        
        // virtual
        getVisualElementController: function() {
            return null;
        },

        setSelected: function (isSelected, playbackAnimation) {
            SizedContent.prototype.setSelected.apply(this, arguments);
            if (!this.getVisualSelectionDisabled()) {
                if (isSelected) {
                    this.addStyleClass("ReportElementSelected");
                    if(this._reportElementDecorator) {
                        this._reportElementDecorator.setDocked(true);
                    }

                    if (playbackAnimation) {
                        var showMaximized = playbackAnimation.isMaximizedMode();
                        if (showMaximized) {
                            if(this._reportElementDecorator) {
                                this._reportElementDecorator.setExpanded(true, showMaximized);
                            } else {
                                this.fireEvent(VisualEvent.VISUAL_EXPANDED, {
                                    id: this.sId,
                                    visual: this
                                }, true, true);
                            }
                        }
                    }
                } else {
                    this.removeStyleClass("ReportElementSelected");
                    if(this._reportElementDecorator) {
                        this._reportElementDecorator.setDocked(false);
                    }
                }
            }
        },

        getTitleSize: function(constrainedWidth) {
            if(this._titleContainer && this._titleContainer.getText()) {
                var width;
                var height;
                var $wrapper = this._titleContainer.$();
                width = $wrapper.outerWidth(true);
                if (constrainedWidth) {
                    $wrapper.width(constrainedWidth);
                }
                height = $wrapper.outerHeight(true);
                if (constrainedWidth) {
                    // if we were given a constrained width, we can safely clear out the width because
                    // it means we are in the middle of the portable layout and we will get a real size later
                    $wrapper.width(null);
                }
                return {width: width, height: height};
            }
            return {width: 0, height: 0};
        },

        // This function is overridden in Visual.js to check if there is a severe warning
        shouldCreateDecorator: function() {
            return !this.getVisualSelectionDisabled() || this.getIsCompositeMember();
        },

        onBeforeRendering: function() {
            SizedContent.prototype.onBeforeRendering.apply(this, arguments);

            if(!this._reportElementDecorator && this.shouldCreateDecorator()) {
                var decorator = ReportElementDecoratorFactory.createDecorator({
                    BIRDTitledContent: this
                }, this.getIsCompositeMember());
                if(decorator) {
                    decorator.oParent = this;
                    decorator.setPropagatingStyleClasses(this._propagatingStyleClasses); 
                    this._reportElementDecorator = decorator;
                }
            }
            // This style is used for hover effect
            if (!this.getVisualSelectionDisabled()) {
                this.addStyleClass("ReportElementSelectable");
            }
            if(this.getTitleEnabled()) {
                this._titleContainer = BIRDTitledContent.createTitleContainer(this.getVisualElementController(), this._titleContainer);
                if(this._titleContainer.getParent() !== this) {
                    // Calling setParent again, after it is already set to this causes
                    // Openui to assert, because it is trying to remove the child from the 
                    // previous parent aggregation (which is null). So only set the parent
                    // the first time.
                    this._titleContainer.setParent(this);
                }
            } else if(this._titleContainer) {
                this._titleContainer.destroy();
                this._titleContainer = null;
            }

            this._updateOverlays();
        },

        layoutSubviews: function() {
            SizedContent.prototype.layoutSubviews.apply(this, arguments);
            this.layoutDecorator();
        },

        layoutDecorator: function() {
            if(this._reportElementDecorator) {
                var bounds = this.getBounds();
                // The toolbar should not overlap the alert icon.
                if(this.getIcon()) {
                    var $icon = this.getIconButton().$();
                    if($icon.length) {
                        bounds.height -= $icon.outerHeight(true);
                    }
                }
                this._reportElementDecorator.setBounds(bounds);
            }
        },

        updateSizeConstraints: function(shouldFireEvent) {
            SizedContent.prototype.updateSizeConstraints.apply(this, arguments);
            var controller = this.getController();
            if (controller) {
                var layoutProxy = controller.getLayoutProxy();
                var layoutConstraint = layoutProxy.getConstraint();
                if (!layoutConstraint) {
                    layoutConstraint = new LayoutConstraint();
                    layoutProxy.setDefaultConstraint(layoutConstraint);
                }

                var height = this.getPreferredHeight(NaN);
                if(!isNaN(height)) {
                    layoutConstraint.setHeight(new Dimension(height, DimensionUnit.PIXELS));
                }
                var width = this.getPreferredWidth(NaN);
                if(!isNaN(width)) {
                    layoutConstraint.setWidth(new Dimension(width, DimensionUnit.PIXELS));
                }

                layoutConstraint.setMinimumHeight(this.getMinimumHeight());
                layoutConstraint.setMinimumWidth(this.getMinimumWidth());
            }
        },

        // Overridden from UI
        _renderChildren: function (rm) {            
            if(this._titleContainer && this._titleContainer.getText()) {
                rm.renderControl(this._titleContainer);
            }
            SizedContent.prototype._renderChildren.apply(this, arguments);
        },

        getVisualSelectionDisabled: function() {
            // Visual selection should be disabled for composite visual members
            if (!this.getSelectable() || this.getIsCompositeMember()) {
                return true;
            }
            var visualElementController = this.getVisualElementController();
            if (visualElementController) {
                return visualElementController.getVisualElementModel().getSelectionDisabled();
            }
            return false;
        },

        updateAriaLabels: function() {
            var visualElementController = this.getVisualElementController();
            if (visualElementController) {
                // In the future we may set other things based on controller values for accessibilityValue and accessibilityHint.
                var label = visualElementController.getAccessibilityLabel();
                this.setAriaLabel(label);
                this.setDataLandmarkLabel(label);
            }
        },

        setFocusable: function(value) {
            SizedContent.prototype.setFocusable.apply(this, arguments);
            if(value) {
                this.addStyleClass("ReportElementFocusable");
            } else {
                this.removeStyleClass("ReportElementFocusable");
            }
        },

        populateDecoratorMenuActions: function(oEvent) {
            if (!this._reportElementDecorator) {
                return;
            }
            this._reportElementDecorator.populateMenuActions(oEvent);
        },

        onUpdateContextMenu: function(oEvent) {
            if (!this._reportElementDecorator) {
                return;
            }
            this.populateDecoratorMenuActions(oEvent);
        },

        destroy: function() {
            if(this._titleContainer) {
                this._titleContainer.destroy();
                this._titleContainer = null;
            }
            if(this._reportElementDecorator) {
                this._reportElementDecorator.destroy();
                this._reportElementDecorator = null;
            }
            if(this._controller) {
                this._controller.release();
                this._controller = null;
            }
            SizedContent.prototype.destroy.call(this);
        }
    });

    /**
     * Returns a UIText node using the title from a VisualElementController
     * @param {sas.ltjs.BIRD.controllers.visualElements.VisualElementController} visualElementController with a title.
     * @param {UIText=} titleContainer UIText control to populate with a title (optional)
     * @returns {UIText} UIText control with the VisualElementController's HTML title.
     * @static
     */
    BIRDTitledContent.createTitleContainer = function (visualElementController, titleContainer) {
        if (!titleContainer) {
            titleContainer = new UIText();
            titleContainer.addStyleClass('vavTitleStyle'); // adding class for rendering title to the saved image
            titleContainer.setEnableUnsafeText(true); //portable will sanitize for us.
        }

        if(visualElementController) {
            var titleText = RichTextStringUtil.parseHTMLString(visualElementController.getTitleAsHTML());
            if(titleText) {
                titleContainer.setText(titleText);
            }
        } else {
            titleContainer.setText("");
        }

        return titleContainer;
    };

    BIRDTitledContent.DEFAULT_PADDING = "3px";

    return BIRDTitledContent;
}, true);
