sap.ui.define([
    "sap/ui/base/ManagedObject",
    "sas/hc/m/MessageDialog",
    "sas/ltjs/commons/ltjsUtil",
    "sas/ltjs/commons/models/warnings/WarningLevel",
    "sas/ltjs/transport/depot/Depot",
    "sas/ltjs/transport/depot/DepotConfiguration",
    "sas/ltjs/transport/depot/DeviceInfo",
    "sas/ltjs/transport/depot/ReportManager",
    "sas/ltjs/transport/depot/Voucher",
    "sas/ltjs/transport/depot/warnings/TransportWarning",
    "sas/ltjs/transport/models/local/Connection",
    "sas/ltjs/transport/models/local/Report",
    "sas/ltjs/transport/models/local/ReportCollectionOwner",
    "sas/ltjs/transport/models/local/Server",
    "sas/ltjs/transport/models/media/ServerConfig",
    "sas/ltjs/transport/models/media/particles/Capability",
    "sas/ltjs/transport/models/media/particles/ReportImageSize",
    "sas/vacommon/settings/EsriSettingsUtil",
    "sas/vaviewer/app/VAViewerSettings",
    "sas/vaviewer/app/VAViewerSettingsHelper",
    "sas/vaviewer/transport/TransportVoucher"
], function(
    ManagedObject,
    MessageDialog,
    ltjsUtil,
    WarningLevel,
    Depot,
    DepotConfiguration,
    DeviceInfo,
    ReportManager,
    Voucher,
    TransportWarning,
    Connection,
    Report,
    ReportCollectionOwner,
    Server,
    ServerConfig,
    Capability,
    ReportImageSize,
    EsriSettingsUtil,
    VAViewerSettings,
    VAViewerSettingsHelper,
    TransportVoucher
) {
    "use strict";
    var rb = sap.ui.getCore().getLibraryResourceBundle("sas.vaviewer");

    //
    // A map of Connection to TransportConnection.
    // Each instance of the ReportViewer maintains its own instnace of a
    // TransportConnection, and through that its own depot. Maintaining
    // a map from Connection to TransportConnection allows us to provide
    // report based static api's, keeping TransportConnection instances
    // from being passed all throughout the codebase.
    //
    var _instances = new Map();
    //
    // Each instance of the depot gets a different path in the virtual
    // file system for its cache, and that is done by keeping a count and
    // using that count in the cache path.
    //
    var _depotCount = 0;
    var _depotCachePathRoot = "/webDepot/caches/";
    
    /**
     * Common error handling code for all Transport warnings.
     * @param {sas.ltjs.commons.models.warnings.Warning} warning
     * @private
     */
    var _handleTransportWarning = function(warning) {
        if(warning) {
            var level = warning.getLevel();
            if(level === WarningLevel.ERROR_LEVEL || level === WarningLevel.FATAL_LEVEL) {
                var message = warning.getMessage() || rb.getText("ErrorDialog.error.txt");
                MessageDialog.error({text: message, title: sas.config.getAppName()});
            } else {
                ltjsUtil.warn(warning.getMessage());
            }
        }
    };
    
    /**
     * Handler function for TransportVoucher warnings
     * @param {sas.vaviewer.transport.TransportVoucher} transportVoucher
     * @private
     */
    var _handleVoucherWarning = function(transportVoucher) {
        _handleTransportWarning(transportVoucher.getWarning());
    };
    
    /**
     * Creates a new TransportVoucher with default warning handling.
     * @param {sas.ltjs.transport.depot.Voucher} voucher the raw voucher
     * @returns {sas.vaviewer.transport.TransportVoucher} the TransportVoucher
     * @private
     */
    var _createTransportVoucher = function(voucher) {
        return TransportVoucher.create(voucher).warning(_handleVoucherWarning);
    };

    /**
     * Creates a new TransportVoucher that issues an INVALID_ARGUMENTS
     * warning.
     * @returns {sas.vaviewer.transport.TransportVoucher} the TransportVoucher
     * @private
     */
    var _createInvalidArgsVoucher = function() {
        var voucher = TransportVoucher.createWarning(TransportWarning.INVALID_ARGUMENTS, WarningLevel.ERROR_LEVEL);
        return voucher.warning(_handleVoucherWarning);
    };
    
    var TransportConnection = ManagedObject.extend('sas.vaviewer.transport.TransportConnection', {
        init: function () {
            this._depot = null;
            this._connection = null;
            this._connectionEstablished = false;
            this._connectionEstablishedCallbacks = [];
            this._mapConnectionInfo = null;
            this._updateTimersId = null;
            this._allowGuest = false;
            this._iframeSandbox = null;
            this._iframeSandboxMap = new Map();
        },

        /**
         * Transport requests that require a connection should call this to ensure readiness.
         * @param {Function} callback
         * @private
         */
        _runWhenConnectionEstablished: function(callback) {
            if (this._connectionEstablished) {
                callback();
            } else {
                this._connectionEstablishedCallbacks.push(callback);
            }
        },

        _getConnectionManager: function() {
            return this._depot ? this._depot.getConnectionManager(this._connection) : null;
        },

        _getReportManager: function() {
            return this._depot ? this._depot.getReportManager(this._connection) : null;
        },

        _getHistoryManager: function() {
            return this._depot ? this._depot.getHistoryManager(this._connection) : null;
        },

        _getThumbnailManager: function() {
            return this._depot ? this._depot.getThumbnailManager(this._connection) : null;
        },

        /**
         * Gets the MapConnectionInfo for this connection.
         * This is needed on the ReportConfiguration when creating a new
         * ReportController.
         * @returns {sas.ltjs.commons.maps.MapConnectionInfo} the MapConnectionInfo
         */
        getMapConnectionInfo: function() {
            if (!this._mapConnectionInfo) {
                var connectionManager = this._getConnectionManager();
                var mapInfo = connectionManager ? connectionManager.getMapConnectionInfo() : null;
                this._mapConnectionInfo = mapInfo ? mapInfo.retain() : null;
            }
            return this._mapConnectionInfo;
        },

        /**
         * Sets up the depot and creates the connection to transport.
         * @param {sas.ltjs.transport.models.media.particles.ReportDataLevel} dataLevel
         * @param {string} host
         * @param {string} port
         * @param {boolean} debug enables debug mode on transport
         */
        initializeDepot: function (dataLevel, host, port, debug) {
            if(!this._depot) {
                //
                // The path to the depot changes for each instance of
                // TransportConnection.
                // /webDepot/caches/0, /webDepot/caches/1, etc
                //
                var depotPath = _depotCachePathRoot + _depotCount;
                _depotCount++;
                //
                // Create the Depot
                //
                var configuration = this._createDepotConfiguration(dataLevel, depotPath);
                this._depot = new Depot(configuration).retain();
                this._depot.setUseHighContrastReportStyle(VAViewerSettingsHelper.overrideWithHighContrastTheme);
                this._depot.setDebug(debug);
                //
                // Create the Connection
                //
                this._connection = new Connection().retain();
                this._connection.setServer(this._createServer(host, port));
                this._connection.setUseGuestCredentials(false);
                //
                // Establish the connection by making getServerConfig/initializeSession
                // calls to transport.
                //
                this._establishConnectionVoucher = this._establishConnection().always(function() {
                    this._establishConnectionVoucher = null;
                }.bind(this));
                //
                // Add an entry into the static instances map, which
                // maps this connection to the instance of TransportConnection.
                //
                _instances.set(this._connection, this);
                //
                // Update the depot timers once every second
                //
                this._updateTimersId = setInterval(function() {
                    this._depot.updateTimers();
                }.bind(this), 1000);
                // Set the theme version to be the HTML commons version (7.0, 8.0 etc.) on the Transport ReportManager
                this._runWhenConnectionEstablished(function() {
                    var reportManager = this._getReportManager();
                    reportManager.setThemeVersion(sas.config.get("commonsVersion"));
                }.bind(this));
            }
        },

        _createDepotConfiguration: function(dataLevel, depotPath) {
            var conf = new DepotConfiguration();
            conf.setCacheDirectoryPath(depotPath);
            conf.setShouldAutoConnect(false);
            conf.setShouldAutoUpdateSubscribedReports(false);
            conf.setShouldAutoStartReportAssetDownloads(true);
            conf.setSupportsTransportServerLogging(false);
            conf.setGenerateReportDataLevel(dataLevel);
            conf.setSupportsTethered(false);
            conf.setShouldPerformPeriodicDeviceChecks(false);
            conf.setMaximumNumberOfConcurrentReportDownloads(1);
            conf.setShouldAutoDownloadPrintedReports(false);
            conf.setGenerateReportTimeout(55);  //S1139556
            conf.setAutoUpdateFrequency(VAViewerSettingsHelper.refreshTime);
            conf.setGenerateReportImagesTimeout(0.5);
            conf.setAllowFutureServers(true);
            conf.setMaximumHistoryEntries(18);  //S1390596

            var deviceInfo = new DeviceInfo();
            deviceInfo.setDeviceModel(navigator.platform);
            deviceInfo.setDeviceType('WEB');
            deviceInfo.setOsVersion(navigator.appVersion);
            deviceInfo.setMobileAppVersion(sas.config.get('softwareComponentName') + ' HTML5');
            conf.setDeviceInfo(deviceInfo);

            return conf;
        },

        _createServer: function(host, port) {
            var server = new Server();
            var isSecure = location.protocol === 'https:';
            server.setScheme(isSecure ? 'https' : 'http');
            server.setHostName(host);
            server.setUseSSL(isSecure);
            server.setPort(+port || (isSecure ? 443 : 80));
            return server;
        },

        _establishConnection: function() {
            //
            // Call getServerConfig
            //
            return TransportVoucher.create(this._getConnectionManager().getServerConfig()).then(function(voucher) {
                //
                // Then call initializeSession.
                // Return the voucher to create a chained TransportVoucher.
                //
                return this._getConnectionManager().initializeSession("false");
            }.bind(this)).fulfilled(function(voucher) {
                //
                // Add the connection to the depot.
                //
                var transportWarning = this._depot.addConnection(this._connection);
                if(transportWarning) {
                    _handleTransportWarning(transportWarning);
                } else {
                    //
                    // Mark the connectionEstablished and allowGuest flags and process
                    // all runWhenConnectionEstablished callbacks.
                    //
                    this._connectionEstablished = true;
                    var meta = this._connection.getServer().getMeta();
                    if (meta) {
                        this._allowGuest = meta.getAllowGuest();
                        this._iframeSandbox = meta.getIframeSandbox();
                    }

                    this._connectionEstablishedCallbacks.forEach(function(callback) {
                        callback();
                    });
                    this._connectionEstablishedCallbacks.length = 0;
                    //
                    // Handle USER_ACTIVE and KEEP_ALIVE
                    // 
                    if(sas.timeoutManager) {
                        sas.eventBus.subscribe(sas.timeoutManager.events.USER_ACTIVE, this._keepAlive, this);
                        sas.eventBus.subscribe(sas.timeoutManager.events.KEEP_ALIVE, this._keepAlive, this);
                    }
                    //
                    // Listen for changes to ESRI credentials
                    //
                    sas.eventBus.subscribe(sas.hc.ui.core.SETTINGS_CHANNEL, sas.hc.ui.core.SETTINGS_CHANGE, this._onEsriSettingsChange, this);
                }
            }.bind(this)).warning(_handleVoucherWarning);
        },

        _keepAlive: function () {
            // S1189005: Keep VATS HTTP session alive
            var connectionManager = this._getConnectionManager();
            if(connectionManager) {
                connectionManager.keepAlive();
            }
        },

        _onEsriSettingsChange: function (channelId, eventId, data) {
            if (data && data.length) {
                var connectionManager = this._getConnectionManager();
                data.forEach(function (setting) {
                    if (setting.id === EsriSettingsUtil.SETTING_ESRI_OPT_IN &&
                        setting.newValue !== this._connection.getUseEsriPremiumServices()) {
                        // Whenever ESRI_EULA_ACCEPTED changes, the opt-in value
                        // also changes so we wait for that event before
                        // updating transport
                        this._connection.setUseEsriPremiumServices(setting.newValue);
                    } else if (this._connection.getUseEsriPremiumServices() && connectionManager) {
                        // Only listen to user id and password events when
                        // premium services are enabled.
                        if (setting.id === EsriSettingsUtil.SETTING_ESRI_USER_ID ||
                            setting.id === EsriSettingsUtil.SETTING_ESRI_USER_PASSWORD) {
                            connectionManager.requestEsriPremiumToken();
                        } else if (setting.id === EsriSettingsUtil.SETTING_LOCAL_ESRI_USER_ID ||
                                   setting.id === EsriSettingsUtil.SETTING_LOCAL_ESRI_USER_PASSWORD) {
                            connectionManager.requestEsriPortalToken();
                        }
                    }
                }.bind(this));
            }
        },

        /**
         * Downloads the report and all of its asset files from
         * the connected transport services.
         * @param {string} reportUri
         * @param {boolean} withUserState true generates a report package with user state
         * @returns {sas.vaviewer.transport.TransportVoucher} report generate
         * voucher
         */
        generateReport: function (reportUri, withUserState) {
            var generateReportVoucher = _createTransportVoucher();

            this._runWhenConnectionEstablished(function() {
                var beginGenerate = window.performance.now();
                var reportManager = this._getReportManager();
                var rawVoucher = withUserState ? reportManager.getReportWithUserState(reportUri) : reportManager.getReport(reportUri);
                generateReportVoucher.setVoucher(rawVoucher).always(function() {
                    var endGenerate = window.performance.now();
                    console.log("PERFORMANCE: Total time to generate report:", endGenerate - beginGenerate);
                });
            }.bind(this));

            return generateReportVoucher;
        },

        getCapabilities: function() {
            return new Promise(function(resolve, reject) {
                this._runWhenConnectionEstablished(function() {
                    resolve(this._getCapabilitiesImpl());
                }.bind(this));
            }.bind(this));
        },
        
        _getCapabilitiesImpl: function() {
            var connectionManager = this._getConnectionManager();
            var userCapabilities = null;

            if (connectionManager) {
                userCapabilities = {
                    createReport: connectionManager.hasCapability(Capability.CREATE_REPORT),
                    email: connectionManager.hasCapability(Capability.EMAIL),
                    viewComments: connectionManager.hasCapability(Capability.VIEW_COMMENTS),
                    addComments: connectionManager.hasCapability(Capability.ADD_COMMENTS),
                    exportData: connectionManager.hasCapability(Capability.EXPORT_DATA),
                    exportDetailData: connectionManager.hasCapability(Capability.EXPORT_DETAIL_DATA),
                    exportImage : connectionManager.hasCapability(Capability.EXPORT_IMAGE),
                    exportPdf: connectionManager.hasCapability(Capability.EXPORT_PDF),
                    subscribeToReportAlerts : connectionManager.hasCapability(Capability.SUBSCRIBE_TO_REPORT_ALERTS),
                    renderReportImages: connectionManager.hasCapability(Capability.RENDER_REPORT_IMAGES),
                    accessEsriPremiumServices: connectionManager.hasCapability(Capability.ACCESS_ESRI_PREMIUM_SERVICES),
                    accessRecents: connectionManager.hasCapability(Capability.ACCESS_RECENTS),
                    personalization: connectionManager.hasCapability(Capability.PERSONALIZATION)
                };
            }

            return userCapabilities;
        },

        /**
         * Prints the report with the given setup and report state.
         * @param {sas.ltjs.transport.models.media.PrintSetup} printSetup
         * @param {sas.ltjs.transport.models.local.Report} report
         * @param {string} reportStateString
         *  @param {string} fileName - fileName for report pdf; can be null: default "report.pdf" name will be used
         * @returns {sas.vaviewer.transport.TransportVoucher} print voucher
         */
        printReport: function(printSetup, report, reportStateString, fileName) {
            var printVoucher = _createTransportVoucher();
            this._runWhenConnectionEstablished(function() {
                if(reportStateString) {
                    printVoucher.setVoucher(this._getReportManager().printReport(printSetup, report, reportStateString, fileName));
                } else {
                    printVoucher.setVoucher(this._getReportManager().printReport(printSetup, report, "<SASReportState></SASReportState>", fileName));
                }
            }.bind(this));
            return printVoucher;
        },

        /**
         * Gets the thumbnail image URL for a report
         * @param {sas.ltjs.transport.models.local.Report} report
         * @returns {sas.vaviewer.transport.TransportVoucher} voucher
         */
        getThumbnailUrl: function(report) {
            return TransportVoucher.create(this._getThumbnailManager().getReportImageLink(report, ReportImageSize.MEDIUM_CINEMA));
        },

        /**
         * Tests if the user has the specified capability.
         * @param {sas.ltjs.transport.models.media.particles.Capability} capability
         * @returns {boolean} support for the capability
         */
        hasCapability: function(capability) {
            var connectionManager = this._getConnectionManager();
            return connectionManager ? connectionManager.hasCapability(capability) : null;
        },

        /**
         * Gets the Depot associated with this TransportConnection
         * @returns {sas.ltjs.transport.depot.Depot} the transport Depot
         */
        getDepot: function() {
            return this._depot;
        },

        /**
         * Returns the path in the virtual filesystem where the assets
         * are being stored for the report.
         * @param {sas.ltjs.transport.models.local.Report} report
         * @returns {string} report file system path
         */
        getReportPath: function(report) {
            var reportManager = this._getReportManager();
            return reportManager ? reportManager.getReportSubdirectoryPath(report) : null;
        },

        /**
         * Marks the report as opened and adds it to the history manager.
         * @param {sas.ltjs.transport.models.local.Report} report
         */
        reportOpened: function(report) {
            this.addToHistory(report);
        },

        /**
         * Marks the report as closed and cleans up any report resources.
         * @param {sas.ltjs.transport.models.local.Report} report
         * @param {boolean} [saveReport=false] if true the report will not be deleted from the depot (and virtual file system)
         */
        reportClosed: function(report, saveReport) {
            if(!saveReport) {
                this._depot.removeReport(report);
            }
        },

        /**
         * Gets the recent report history.
         * When fulfilled, the returned voucher's requestedResource is a 
         * ReportCollectionOwner containing the recent reports.
         * @returns {sas.vaviewer.transport.TransportVoucher} history voucher
         */
        getHistory: function() {
            var getHistoryVoucher = _createTransportVoucher();
            this._runWhenConnectionEstablished(function() {
                if(this.hasCapability(Capability.ACCESS_RECENTS)) {
                    getHistoryVoucher.setVoucher(this._getHistoryManager().getHistory());
                } else {
                    getHistoryVoucher.cancel();
                }
            }.bind(this));

            return getHistoryVoucher;
        },

        /**
         * Adds the report to the history manager.
         * @param {sas.ltjs.transport.models.local.Report} report
         * @returns {sas.vaviewer.transport.TransportVoucher} voucher
         */
        addToHistory: function(report) {
            var addHistoryVoucher = _createTransportVoucher();
            this._runWhenConnectionEstablished(function() {
                if(this.hasCapability(Capability.ACCESS_RECENTS)) {
                    addHistoryVoucher.setVoucher(this._getHistoryManager().addToHistory(report));
                } else {
                    addHistoryVoucher.cancel();
                }
            }.bind(this));

            return addHistoryVoucher;
        },

        /**
         * Fetches the alertable items for a report from transport.
         * When fulfilled, the returned voucher's requesteResource is the
         * Report, which has getAlertableItems api that contain the now populated
         * list of alertable items.
         * @param {sas.ltjs.transport.models.local.Report} report
         * @returns {sas.vaviewer.transport.TransportVoucher} voucher
         */
        getAlertableItems: function(report) {
            return _createTransportVoucher(this._getReportManager().getAlertableItems(report));
        },

        /**
         * Subscribes the report to the alert with the specified name.
         * @param {sas.ltjs.transport.models.local.Report} report
         * @param {string} alertName
         * @returns {sas.vaviewer.transport.TransportVoucher} voucher
         */
        subscribeToAlert: function(report, alertName) {
            return _createTransportVoucher(this._getReportManager().subscribeToAlert(report, alertName));
        },

        /**
         * Unsubscribes the report to the alert with the specified name.
         * @param {sas.ltjs.transport.models.local.Report} report
         * @param {string} alertName
         * @returns {sas.vaviewer.transport.TransportVoucher} voucher
         */
        unsubscribeFromAlert: function(report, alertName) {
            return _createTransportVoucher(this._getReportManager().unsubscribeFromAlert(report, alertName));
        },

        /**
         * Gets the share url for the specified report.  If the sectionName
         * is supplied then the url is for opening the report directly to
         * that section.
         * @param {sas.ltjs.transport.models.local.Report} report
         * @param {string} sectionName
         * @returns {string} report url
         */
        getShareUrl: function(report, sectionName) {
            var reportManager = this._getReportManager();
            if(report && reportManager) {
                if(sectionName) {
                    return reportManager.getShareURLWithSection(report, sectionName);
                } else {
                    return reportManager.getShareURL(report);
                }
            }
            return null;
        },

        /**
         * Returns true if this server supports guest mode access
         * @returns {boolean} allowGuest
         */
        getAllowGuest: function() {
            return this._allowGuest;
        },

        destroy: function () {
            ManagedObject.prototype.destroy.apply(this, arguments);

            if (this._iframeSandboxMap) {
                this._iframeSandboxMap.clear();
                this._iframeSandboxMap = null; 
            }

            if (this._updateTimersId) {
                clearTimeout(this._updateTimersId);
                this._updateTimersId = null;
            }
            
            if (this._connection) {
                _instances.delete(this._connection);
                this._connection.release();
                this._connection = null;
            }

            if(this._establishConnectionVoucher) {
                this._establishConnectionVoucher.cancel();
                this._establishConnectionVoucher = null;
            }

            if (this._depot) {
                this._depot.shutdown();
                this._depot.release();
                this._depot = null;
            }

            if (this._mapConnectionInfo) {
                this._mapConnectionInfo.release();
                this._mapConnectionInfo = null;
            }

            if (this._connectionEstablishedCallbacks && this._connectionEstablishedCallbacks.length) {
                this._connectionEstablishedCallbacks.length = 0;
            }

            if(sas.timeoutManager) {
                sas.eventBus.unsubscribe(sas.timeoutManager.events.USER_ACTIVE, this._keepAlive, this);
                sas.eventBus.unsubscribe(sas.timeoutManager.events.KEEP_ALIVE, this._keepAlive, this);
            }

            sas.eventBus.unsubscribe(sas.hc.ui.core.SETTINGS_CHANNEL, sas.hc.ui.core.SETTINGS_CHANGE, this._onEsriSettingsChange, this);
        }
    });

    /**
     * Gets the instance of TransportConnection that is associated
     * with the specified connection.
     * @param {sas.ltjs.transport.models.local.Connection} connection 
     * @returns {sas.vaviewer.transport.TransportConnection} the instance
     */
    TransportConnection.getInstance = function(connection) {
        return _instances.get(connection);
    };

    /**
     * Fetches the alertable items for a report from transport.
     * When fulfilled, the returned voucher's requesteResource is the
     * Report, which has getAlertableItems api that contain the now populated
     * list of alertable items.
     * @param {sas.ltjs.transport.models.local.Report} report
     * @returns {sas.vaviewer.transport.TransportVoucher} voucher
     */
    TransportConnection.getAlertableItems = function(report) {
        if(!report) {
            return _createInvalidArgsVoucher();
        }
        var tc = TransportConnection.getInstance(report.getConnection());
        return tc.getAlertableItems(report);
    };

    /**
     * Subscribes the report to the alert with the specified name.
     * @param {sas.ltjs.transport.models.local.Report} report
     * @param {string} alertName
     * @returns {sas.vaviewer.transport.TransportVoucher} voucher
     */
    TransportConnection.subscribeToAlert = function(report, alertName) {
        if(!report) {
            return _createInvalidArgsVoucher();
        }
        var tc = TransportConnection.getInstance(report.getConnection());
        return tc.subscribeToAlert(report, alertName);
    };

    /**
     * Unsubscribes the report to the alert with the specified name.
     * @param {sas.ltjs.transport.models.local.Report} report
     * @param {string} alertName
     * @returns {sas.vaviewer.transport.TransportVoucher} voucher
     */
    TransportConnection.unsubscribeFromAlert = function(report, alertName) {
        if(!report) {
            return _createInvalidArgsVoucher();
        }
        var tc = TransportConnection.getInstance(report.getConnection());
        return tc.unsubscribeFromAlert(report, alertName);
    };

    /**
     * Prints the report with the given setup and report state.
     * @param {sas.ltjs.transport.models.media.PrintSetup} printSetup
     * @param {sas.ltjs.transport.models.local.Report} report
     * @param {string} reportStateString
     * @param {string} fileName - fileName for report pdf; can be null: default "report.pdf" name will be used
     * @returns {sas.vaviewer.transport.TransportVoucher} print voucher
     */
    TransportConnection.printReport = function(printSetup, report, reportStateString, fileName) {
        if(!report || !report.getConnection()) {
            return _createInvalidArgsVoucher();
        }
        var tc = TransportConnection.getInstance(report.getConnection());
        return tc.printReport(printSetup, report, reportStateString, fileName);
    };

    /**
     * Gets the thumbnail image URL for a report
     * @param {sas.ltjs.transport.models.local.Report} report
     * @returns {sas.vaviewer.transport.TransportVoucher} voucher
     */
    TransportConnection.getThumbnailUrl = function(report) {
        if(!report || !report.getConnection()) {
            return _createInvalidArgsVoucher();
        }
        var tc = TransportConnection.getInstance(report.getConnection());
        return tc.getThumbnailUrl(report);
    };

    /**
     * Gets the share url for the specified report.  If the sectionName
     * is suplied then the url is for opening the report directly to
     * that section.
     * @param {sas.ltjs.transport.models.local.Report} report
     * @param {string} sectionName
     * @returns {string} report url
     */
    TransportConnection.getShareUrl = function(report, sectionName) {
        if(!report || !report.getConnection()) {
            return null;
        }
        var tc = TransportConnection.getInstance(report.getConnection());
        return tc.getShareUrl(report, sectionName);
    };

    /**
     * Returns the path in the virtual filesystem where the assets
     * are being stored for the report.
     * @param {sas.ltjs.transport.models.local.Report} report
     * @returns {string} report file system path
     */
    TransportConnection.getReportPath = function(report) {
        if(!report || !report.getConnection()) {
            return null;
        }
        var tc = TransportConnection.getInstance(report.getConnection());
        return tc.getReportPath(report);
    };

    /**
     * Returns the ReportManager for the given report.
     * @param {sas.ltjs.transport.models.local.Report} report
     * @returns {sas.ltjs.transport.depot.ReportManager} reportManager
     */
    TransportConnection.getReportManager = function(report) {
        if(!report || !report.getConnection()) {
            return null;
        }
        var tc = TransportConnection.getInstance(report.getConnection());
        return tc._getReportManager();
    };

    /**
     * Closes the given report (deletes it from the Depot)
     * @param {sas.ltjs.transport.models.local.Report} report
     */
    TransportConnection.reportClosed = function(report) {
        if(!report || !report.getConnection()) {
            return null;
        }
        var tc = TransportConnection.getInstance(report.getConnection());
        tc.reportClosed(report, false);
    };

    /**
     * Set map entry for iframeSandbox.
     * @param {sas.ltjs.BIRD.util.SASReportConfiguration} sasReportConfiguration
     * @param {sas.ltjs.transport.models.local.Report} report
     */
    TransportConnection.setIframeSandboxForSASReportConfiguration = function(sasReportConfiguration, report) {
        if (!sasReportConfiguration || !report || !sasReportConfiguration.getContentKey() || !report.getConnection()) {
            return;
        }  
        var key = sasReportConfiguration.getContentKey();
        var tc = TransportConnection.getInstance(report.getConnection());
        if (tc._iframeSandbox)  {
            tc._iframeSandboxMap.set(key, tc._iframeSandbox);
        }
    };

    /**
     * Get iframeSandbox from transport map entry.
     * @param {sas.ltjs.BIRD.util.SASReportConfiguration} sasReportConfiguration
     * @returns {string} iframeSandbox
     */
    TransportConnection.getIframeSandboxForSASReportConfiguration = function(sasReportConfiguration) {
        if (!sasReportConfiguration || !sasReportConfiguration.getContentKey()) {
            return null;
        }
        var key = sasReportConfiguration.getContentKey();
        var iframeSandbox = null;
        _instances.forEach(function(instance) {
            if (instance._iframeSandboxMap.get(key)) {
                iframeSandbox = instance._iframeSandboxMap.get(key);
            }
        });
        return iframeSandbox;
    };

    return TransportConnection;
}, true);
