sap.ui.define([
    "sas/hc/ui/layout/Splitter",
    "sas/vaviewer/views/BIRDViewFactory",
    "sas/hc/m/IconTabBar",
    "sas/hc/m/IconTabFilter",
    "sap/ui/core/Orientation",
    "sas/ltjs/BIRD/controllers/visualElements/TableElementController",
    "sas/ltjs/commons/views/UISizing",
    "sas/hc/ui/layout/SplitterLayoutData",
    "sas/ltjs/commons/views/UI",
    "sap/ui/core/ResizeHandler",
    "sas/vaviewer/views/ViewerEvents",
    "sas/vaviewer/views/VisualEvent"
], function(
    Splitter,
    BIRDViewFactory,
    IconTabBar,
    IconTabFilter,
    Orientation,
    TableElementController,
    UISizing,
    SplitterLayoutData,
    UI,
    ResizeHandler,
    ViewerEvents,
    VisualEvent
) {
    "use strict";

    // S1589846: Disable pane size persistence for all splitters in the app
    Splitter.setMasterOverrideToPersistPaneSizes(Splitter.MasterOverrideToPersistPaneSizes.MASTER_OVERRIDE_OFF);

    // Extend the HTML Commons Splitter component to add custom funcitonality to it.
    return Splitter.extend("sas.vaviewer.SupplementalVisualSplitter", {

        renderer: "sas.vaviewer.SupplementalVisualSplitterRenderer",

        init: function () {
            Splitter.prototype.init.apply(this, arguments);
            // Private variables
            this._primaryVisual = null;
            this._suppVisual = null;
            this._oldPrimaryVisualParent = null;
            this._oldPrimaryVisualParentId = null;
            this._oldParentIndex = 0;
            this._suppViewBar = null;
            this._splitterResizeHandlerId = null;
            this.setOrientation(Orientation.Vertical);
            this.addStyleClass("vaViewerSuppVisualSplitter");
            this.attachResize(this._onSplitterResize, this);
            this.attachEvent(ViewerEvents.UPDATE_CONTEXT_MENU, this._onUpdateContextMenu);
            this.attachEvent(VisualEvent.VISUAL_UNEXPANDED, this._onVisualUnexpanded);
        },

        // adding the primary and supplemental visuals in the Splitter
        addContent: function (primaryVisual) {
            if (!primaryVisual) {
                return;
            }
            this._primaryVisual = primaryVisual;
            var primaryVisualController = primaryVisual.getVisualElementController();
            if (!primaryVisualController) {
                return;
            }
            var primaryVisualContainerLayout = new SplitterLayoutData({
                size: "75%",
                minSize: 100
            });
            // Wrapper container for the primary visual
            var primaryVisualContainer = new UI({
                layoutData: primaryVisualContainerLayout,
                sizing: UISizing.FILL,
                position: "relative"
            });
            this.addAggregation("contentAreas", primaryVisualContainer, true);
            primaryVisualContainer.addStyleClass("overflowHidden");
            var suppVisualControllers = primaryVisualController.getSupplementalVisualElementControllers();
            if (suppVisualControllers && suppVisualControllers.length > 0) {
                var numOfSuppVisualControllers = suppVisualControllers.length;
                // If the number of supplemental visuals are more than 1, we would need a tabbed view
                if (numOfSuppVisualControllers > 1) {
                    this._suppViewBar = new IconTabBar({
                        firstTabClosable: false,
                        showOverflowPopOver: true,
                        select: [this._onSuppViewBarSelect, this]
                    });
                    this._suppViewBar.addStyleClass("vaViewerSuppVisualBar");
                    this._suppViewBar.addStyleClass("sasVaFlexColumn");
                    for(var i = 0; i < numOfSuppVisualControllers; ++i) {
                        var eachSuppVisualController = suppVisualControllers[i];
                        if (eachSuppVisualController) {
                            var eachSuppView = this._createSupplementalVisual(eachSuppVisualController);
                            //track the active supplemental visual
                            if(!this._suppVisual) {
                                this._suppVisual = eachSuppView;
                            }
                            var eachSuppViewTitle = eachSuppVisualController.getPresentationLabel();
                            var eachSuppViewFilter = new IconTabFilter({
                                text: eachSuppViewTitle,
                                content: eachSuppView
                            });
                            this._suppViewBar.addItem(eachSuppViewFilter);
                        }
                    }
                } else {
                    var singleSuppVisualController = suppVisualControllers[0];
                    if (singleSuppVisualController) {
                        this._suppVisual = this._createSupplementalVisual(singleSuppVisualController);
                    }
                }
            }
            // We would either have an icon tab bar as supplemental visual or a simple list table
            var supplementalVisual = this._suppViewBar || this._suppVisual;
            if (supplementalVisual) {
                var supplementalVisualLayout = new SplitterLayoutData({
                    size: "25%",
                    minSize: 20
                });
                supplementalVisual.setLayoutData(supplementalVisualLayout);
                supplementalVisual.addStyleClass("overflowHidden");
                this.addAggregation("contentAreas", supplementalVisual, true);
            } else {
                // if there is no supplemental visual, let the primary visual take 100% of the size
                primaryVisualContainerLayout.setSize("100%");
            }
        },

        // Set the supplemental visual from the selected tab and resize the contents. 
        _onSuppViewBarSelect: function (event) {
            var tabFilter = event.getParameter("selectedItem");
            this._suppVisual = tabFilter.getContent()[0];
            this._onSplitterResize();
        },

        _onUpdateContextMenu: function(oEvent) {
            this._primaryVisual.populateDecoratorMenuActions(oEvent);
        },

        // ensure that the unexpand event bubbles up through the same
        // hierarchy it would if the element were not expanded
        _onVisualUnexpanded: function (event) {
            if (this._oldPrimaryVisualParent) {
                event.cancelBubble();
                this._oldPrimaryVisualParent.fireEvent(event.getId(), event.getParameters(), true, true);
            }
        },

        _createSupplementalVisual: function(controller) {
            controller.updateViewData();
            // Sorting has to be disabled on supplementary table to avoid sorting of graph since they share the same query
            // It is safer to have an instanceof check since setSortEnabled is a TableElementController specific api
            if (controller instanceof TableElementController) {
                controller.setSortEnabled(false);
            }
            var suppView = BIRDViewFactory.createVisualElementView(controller);
            suppView.setSizing(UISizing.FILL);
            suppView.addStyleClass("suppVisual");
            return suppView;
        },

        // Override the parent Splitter's resize function to add our own logic.
        _onSplitterResize: function (event) {
            // Get the DOM width and height of content areas and set the bounds on the primary and supplemental visuals.
            var contentAreas = this.getContentAreas();
            if (!contentAreas) {
                return;
            }
            var primaryVisualArea = contentAreas[0];
            if (!primaryVisualArea) {
                return;
            }
            var primaryVisualAreaDom = primaryVisualArea.getDomRef();
            if (this._primaryVisual && primaryVisualAreaDom) {
                this._primaryVisual.setBounds({
                    top: 0,
                    left: 0,
                    width: primaryVisualAreaDom.clientWidth,
                    height: primaryVisualAreaDom.clientHeight
                });
            }
            
            // We need to set the bounds of supplemental visual only if it has been created
            if (!this._suppVisual) {
                return;
            }
            var suppVisualArea = contentAreas[1];
            if (!suppVisualArea) {
                return;
            }
            var suppVisualAreaDom = suppVisualArea.getDomRef();
            if (!suppVisualAreaDom) {
                return;
            }
            var container = suppVisualAreaDom;
            if(this._suppViewBar) {
                container = this._suppViewBar.getDomRef("content");
            }
            if (!container) {
                return;
            }
            //Use jQuery width/height so that padding is not included.
            var $container = $(container);
            this._suppVisual.setBounds({
                top: 0,
                left: 0,
                width: $container.width(),
                height: $container.height()
            });
        },

        onBeforeRendering: function() {
            if (Splitter.prototype.onBeforeRendering) {
                Splitter.prototype.onBeforeRendering.apply(this, arguments);
            }
            // If Splitter or its contents change things before rendering, 
            // restore the state just before the Splitter gets rendered.
            this._restoreVisual();
            if(this._splitterResizeHandlerId) {
                ResizeHandler.deregister(this._splitterResizeHandlerId);
                this._splitterResizeHandlerId = null;
            }
        },

        // Override this to add some styles to the IconTabBar contents and resize the contents
        onAfterRendering: function () {
            if (Splitter.prototype.onAfterRendering) {
                Splitter.prototype.onAfterRendering.apply(this, arguments);
            }
            // Protective null checks before we proceed
            if (!this._primaryVisual) {
                return;
            }
            var primaryVisualDom = this._primaryVisual.getDomRef();
            if (!primaryVisualDom) {
                return;
            }
            var primaryVisualParentDom = primaryVisualDom.parentNode;
            if (!primaryVisualParentDom) {
                return;
            }
            this._oldPrimaryVisualParentId = primaryVisualParentDom.id;
            // Save the parent to be used later while restoring the visual
            this._oldPrimaryVisualParent = this._primaryVisual.getParent();
            
            // Save the DOM node index of the previous parent of the primary visual
            var oldDomChildren = primaryVisualParentDom.children;
            if (!oldDomChildren) {
                return;
            }
            for(var i = 0, len = oldDomChildren.length; i < len; ++i) {
                if(oldDomChildren[i] === primaryVisualDom) {
                    this._oldParentIndex = i;
                    break;
                }
            }
            // Append the primary visual dom directly to the splitter content dom.
            var contentAreas = this.getContentAreas();
            if (contentAreas) {
                var primaryVisualArea = contentAreas[0];
                if (primaryVisualArea) {
                    var primaryVisualAreaDom = primaryVisualArea.getDomRef();
                    if (primaryVisualAreaDom) {
                        primaryVisualAreaDom.appendChild(primaryVisualDom);
                    }
                }
            }
            this._primaryVisual.oParent = this;
            // In order to get the correct sizing for the iconTabBar content add classes to sub Elements.
            if(this._suppViewBar) {
                this._suppViewBar.$("containerContent").addClass( "sasVaFlexColumn" );
                this._suppViewBar.$("containerContent").addClass( "sasVaFlexFillSpace" );
                this._suppViewBar.$("content").addClass( "sasVaFlexFillSpace" );
            }
            this._splitterResizeHandlerId = ResizeHandler.register(this.getDomRef(), this._onSplitterResize.bind(this));
            this._onSplitterResize();
        },

        _restoreVisual: function() {
            if(!this._oldPrimaryVisualParent) {
                return;
            }
            // Put the primary visual back in the original dom structure before maximize
            var oldDomRef = document.getElementById(this._oldPrimaryVisualParentId);
            if(oldDomRef) {
                var oldDomParentChildren = oldDomRef.children;
                var visualRef = this._primaryVisual.getDomRef();
                if (visualRef) {
                    if(oldDomParentChildren && oldDomParentChildren.length) {
                        // S1251461: If it is the last node in DOM children list, then we should pass null to the insertBefore api.
                        var nextChild = oldDomParentChildren[this._oldParentIndex] || null;
                        // if they are equal then the parent has re-rendered at some point, so hop out.
                        if(nextChild !== visualRef) {
                            oldDomRef.insertBefore(visualRef, nextChild);
                        }
                    } else {
                        oldDomRef.appendChild(visualRef);
                    }
                }
            }
            // Restore the parent of the primary visual to what it was before adding it to the splitter
            this._primaryVisual.oParent = this._oldPrimaryVisualParent;
        },

        _removeSplitter: function() {
            var dom = this.getDomRef();
            if(dom && dom.parentNode) {
                dom.parentNode.removeChild(dom);
            }
        },

        destroy: function () {
            this._restoreVisual();
            this._removeSplitter();
            if(this._splitterResizeHandlerId) {
                ResizeHandler.deregister(this._splitterResizeHandlerId);
                this._splitterResizeHandlerId = null;
            }
            this._oldPrimaryVisualParent = null;
            if (this._suppVisual) {
                this._suppVisual.destroy();
                this._suppVisual = null;
            }
            if (this._suppViewBar) {
                this._suppViewBar.destroy();
                this._suppViewBar = null;
            }
            this.detachEvent(ViewerEvents.UPDATE_CONTEXT_MENU, this._onUpdateContextMenu);
            Splitter.prototype.destroy.apply(this, arguments);
        }
    });
}, true);
