sap.ui.define([
    "sap/ui/core/Control",
    "sap/ui/core/ResizeHandler",
    "sas/hc/m/Button",
    "sas/hc/m/IconTabBar",
    "sas/hc/m/IconTabFilter",
    "sas/hc/m/IconTabHeader",
    "sas/hc/m/MessageDialog",
    "sas/hc/ui/core/Styles",
    "sas/hc/ui/core/Narrator",
    "sas/ltjs/BIRD/controllers/interactions/InteractionEvent",
    "sas/ltjs/BIRD/controllers/ReportController",
    "sas/ltjs/BIRD/models/transport/cachedReport/CachedReportURL",
    "sas/ltjs/BIRD/models/transport/cachedReport/CachedReportURLType",
    "sas/ltjs/BIRD/models/visualElements/actions/NavigationAction",
    "sas/ltjs/BIRD/models/visualElements/actions/NavigationActionType",
    "sas/ltjs/BIRD/ui/transitions/SlideTransition",
    "sas/ltjs/BIRD/util/ReportParameterValue",
    "sas/ltjs/BIRD/util/SASReportConfiguration",
    "sas/vaviewer/views/Header",
    "sas/vaviewer/views/Section",
    "sas/vaviewer/views/Visual",
    "sas/vaviewer/views/VisualEvent",
    "sas/ltjs/commons/ClosetObject",
    "sas/ltjs/commons/ltjsEnvironment",
    "sas/ltjs/commons/models/warnings/WarningLevel",
    "sas/ltjs/commons/styles/AttributeNames",
    "sas/ltjs/commons/ui/tooltip/TooltipManager",
    "sas/ltjs/commons/util/AnimateUtil",
    "sas/ltjs/commons/util/StyleUtils",
    "sas/ltjs/commons/views/UI",
    "sas/ltjs/commons/views/UISizing",
    "sas/ltjs/transport/models/media/particles/FileType",
    "sas/ltjs/vav/report/ReportSession",
    "sas/ltjs/transport/models/media/particles/ReportAssetURLType",
    "sas/vacommon/controls/panels/PanelContainer",
    "sas/vaviewer/app/VAViewerSettingsHelper",
    "sas/vaviewer/events/ViewerEventType",
    "sas/vaviewer/ui/HiddenSectionDialog",
    "sas/vaviewer/ui/InfoPanel",
    "sas/vaviewer/util/FullscreenUtil",
    "sas/ltjs/commons/ltjsUtil",
    "sas/vaviewer/views/BIRDTitledContent",
    "sas/ltjs/BIRD/controllers/view/VisualController",
    "sas/vaviewer/ui/playback/PlaybackTimer",
    "sas/vaviewer/ui/playback/PlaybackAnimation",
    "sas/vaviewer/ui/playback/PlaybackNavigation",
    "sas/vaviewer/transport/TransportConnection" 
], function(
    Control,
    ResizeHandler,
    Button,
    IconTabBar,
    IconTabFilter,
    IconTabHeader,
    MessageDialog,
    Styles,
    Narrator,
    InteractionEvent,
    ReportController,
    CachedReportURL,
    CachedReportURLType,
    NavigationAction,
    NavigationActionType,
    SlideTransition,
    ReportParameterValue,
    SASReportConfiguration,
    Header,
    Section,
    Visual,
    VisualEvent,
    ClosetObject,
    ltjsEnvironment,
    WarningLevel,
    AttributeNames,
    TooltipManager,
    AnimateUtil,
    StyleUtils,
    UI,
    UISizing,
    FileType,
    ReportSession,
    ReportAssetURLType,
    PanelContainer,
    VAViewerSettingsHelper,
    ViewerEventType,
    HiddenSectionDialog,
    InfoPanel,
    FullscreenUtil,
    ltjsUtil,
    BIRDTitledContent,
    VisualController,
    PlaybackTimer,
    PlaybackAnimation,
    PlaybackNavigation,
    TransportConnection
) {
    "use strict";

    var rb = sap.ui.getCore().getLibraryResourceBundle("sas.vaviewer");
    
    /**
     * ReportContainer constructor.
     *
     * @param {string} [sId] id for the new control, generated automatically if no id is given
     * @param {object} [mSettings] initial settings for the new control
     *
     * @class
     *
     * ReportContainer is the API driven control of the report view.
     *
     * @extends sap.ui.core.Control
     *
     * @constructor
     * @public
     * @name sas.vaviewer.ReportContainer
     */
    return Control.extend("sas.vaviewer.ReportContainer", {

        metadata: {
            properties: {
                // TODO: why is this a metadata property?  Should this be removed?
                reportController: {
                    type: "object"
                },
                promptBarVisible: {
                    type: "boolean",
                    defaultValue: true
                }
            },
            aggregations: {
                "header": {
                    type: "sas.vaviewer.views.Header",
                    multiple: false
                },
                exitFullscreenButton: {
                    type: "sas.hc.m.Button",
                    multiple: false,
                    visibility: "hidden"
                }
            },
            events: {
                selectedObjectChange: {
                     parameters: {
                        object: {type: "sap.ui.core.Control"}
                    }
                }
            },
            // to suppress the assertion failure from commons 2.5 sdk
            specialSettings: {
                "vaViewerConfig": {},
                "sectionViewOnly": false
            }
        },

        init: function () {
            this.addStyleClass("Report");
            this.addStyleClass("reportContainer");
            this.addStyleClass("titledElement");
            this.addStyleClass("sasVaFlexFillSpace");
            this._reportTheme = null;
            this._urlParametersInitialized = false;
            this._minSizeTimeoutID = null;
            this._maxLayoutLoop = 0;
            this._currentRenderedViewIndex = -1;
            this._sectionWrapper = new SlideTransition();
            // Setup tap on empty area to clear selection
            this._sectionWrapper.addEventDelegate({
                ontap: function() {
                    this.setSelectedObject(this);
                }
            }, this);
            this._renderManager = sap.ui.getCore().createRenderManager();
            this._lastIndexSaved = false;
            this._lastIndex = -1;
            this._reportViewSize = null;
            this._resizeListenerId = null;

            this._handles = null;
            this._infoPanel = null;
            this._autoFocusFirstVisualFlag = false;
            this._sectionViewOnly = false;
            this._selectedObject = this;
            this._overlaysEnabled = true;
            this._playbackMode = false;
            this._collapsedSidePanelWidth = 0;
            this._pageTabsHeaderHeight = 0;
            this._isFixedSizeReport = false;
            this._fixedReportWidth = 0;
            this._fixedReportHeight = 0;
            //
            // Create the button to exit fullscreen mode.
            //
            var exitFullscreenButton = new Button({
                icon: sas.icons.HC.NORMALVIEW,
                press: function () {
                    FullscreenUtil.exitFullscreen();
                    Narrator.addTextNow(rb.getText("ReportContainer.restoreButtonAriaAnnounce.txt"));
                },
                tooltip: rb.getText("ReportContainer.restoreButton.tip.txt")
            });
            exitFullscreenButton.addStyleClass("exitFullscreen");
            this.setAggregation("exitFullscreenButton", exitFullscreenButton);
            this._playbackTimer = new PlaybackTimer({
                visible: false
            });
            // There are layout issues if the timer is added later during playback,
            // so add it upfront and then make it visible later.
            this._sectionWrapper.addChild(this._playbackTimer);
            this._playbackNavigation = new PlaybackNavigation({
                visible: false
            });
            this._sectionWrapper.addChild(this._playbackNavigation);
            this._playbackAnimation = null;
        },

        applySettings: function(settings, scope) {
            if(Control.prototype.applySettings) {
                Control.prototype.applySettings.call(this, settings, scope);
            }
            this._vaViewerConfig = settings.vaViewerConfig;

            this.attachEvent(VisualEvent.UPDATE_ARIA_LIVE, this.updateAriaLiveHandler);
            
            // Set this flag to true, when we want to render only the section part of the report container 
            // and not the extra chrome around it having the info panel and the page tabs
            this._sectionViewOnly = !!settings.sectionViewOnly || this._vaViewerConfig.getObjectViewOnly();
            if (!this._sectionViewOnly) {
                //
                // The PanelContainer contains the ReportContainer
                // and side panel. The SidePanel isn't created and added to the
                // PanelContainer until the report is loaded.
                //
                this._panelContainer = new PanelContainer({
                    content: this._sectionWrapper
                });
                
                this._panelContainer.addStyleClass("vavReportPanelContainer");
                //
                // The main page content an IconTabBar that contains the
                // PanelContainer as its content.  The tabs are not set
                // on the IconTabBar until the report is loaded.
                // The default busyIndicatorDelay on a Control is 1000ms, but that
                // is too long.  The IconTabBar and its content is set to busy
                // while the report is loading, and that state needs to be shown
                // immediately, so busyIndicatorDelay is set to 0
                //
                this._sectionTabBar = new IconTabBar({
                    enableDynamicStyle: true,
                    applyContentPadding: false,
                    stretchContentHeight: true,
                    enableDarkTheme: true,
                    content: this._panelContainer,
                    busyIndicatorDelay: 0,
                    select: [this._onSectionTabSelect, this]
                });
                this._sectionTabBar.oParent = this;
                this._sectionTabBar.addEventDelegate({
                    ontap: function(event) {
                        // The IconTabHeader consists of IconTabFilter and left and right arrow Icons.
                        // If the empty area along the IconTabHeader is clicked, only then the source 
                        // of the event is IconTabHeader else it is either IconTabFilter or Icon.
                        // If the empty area is clicked, we want the report to be selected.
                        if (event && event.srcControl instanceof IconTabHeader) {
                            this.setSelectedObject(this);
                        }
                    }
                }, this);
                this._sectionTabBar.addStyleClass("sectionIconTabBar");
                this._sectionTabBar.addStyleClass("sasVaFlexColumn");
                this._sectionTabBar.addStyleClass("sasVaFlexFillSpace");
                this._addSectionTabBarSelectionHandler();
                this._sectionWrapper.addStyleClass("sasVaFlexFillSpace");
            } else {
                // In section view only mode, the section wrapper would be the 
                // outermost UI control whose parent need to be set.
                this._sectionWrapper.oParent = this;
            }
        },

        updateAriaLiveHandler: function (evt) {
            var text = evt.getParameter("text");
            // S1252766 - hack for this release until a common aria-live div solution is ready (JIRA HTMLCOMMONS-5988).
            // Setting the text immediately sometimes has no effect, so introducing a delay seems to work to give
            // screen readers a chance to flush/queue their buffers. A timeout of 0 did not work and 100 worked intermittently.
            // 500 seems to work well, and synchronizes nicely with the graph transition animations.
            this.$("AriaLiveDiv").text("");
            setTimeout(function (newText) {
                this.$("AriaLiveDiv").text(newText);
            }.bind(this, text), 500);
        },

        getReportController: function() {
            // TODO: this should be done by property or don't have property
            return this.reportController;
        },

        setPromptBarVisible: function(value) {
            this.setProperty("promptBarVisible", value, true);
            this._setHeaderVisibility(value);
            var sectionView = this.getCurrentSectionView();
            if(sectionView) {
                sectionView.setHeaderVisible(value);
            }
            this._updateReportSize(true);
        },

        setOverlaysEnabled: function(value) {
            if (this._overlaysEnabled !== value) {
                this._overlaysEnabled = value;
                var section = this.getCurrentSectionView();
                if (section) {
                    section.setOverlaysEnabled(value);
                }
            }
        },

        /**
         * Adds a override handler for section tab bar when the currently selected tab is selected again.
         * We set the current section to be selected when the same tab is tapped.
         * @private
         */
        _addSectionTabBarSelectionHandler: function() {
            if (this._sectionTabBar) {
                var sectionTabHeader = this._sectionTabBar._getIconTabHeader();
                if (sectionTabHeader) {
                    // The IconTabBar doesn't fire any event if the selected item is selected again.
                    // So override the function and set the current section to be selected.
                    sectionTabHeader._handleTabChange = function(oControl) {
                        IconTabHeader.prototype._handleTabChange.apply(sectionTabHeader, arguments);
                        if (oControl === sectionTabHeader.getSelectedItem()) {
                            this.setSelectedObject(this.getCurrentSectionView());
                        }
                    }.bind(this);
                }
            }
        },

        /**
         * sets the visibility for the ReportContainer header.
         * @param {boolean} value
         * @private
         */
        _setHeaderVisibility: function(value) {
            var headerView = this.getAggregation("header");
            if (headerView) {
                headerView.setVisible(value);
            }
        },

        /**
         * Set the current section index and change the current section view.
         * @param {number} index
         * @returns {sas.vaviewer.ReportContainer} Returns <code>this</code> to allow method chaining.
         */
        setCurrentSectionIndex: function (index) {
            // We want to set the selected item on the page tabs even if index is of current section.
            // This is needed during report refresh to the maintain the selected page tab.
            this._setSectionTabBarSelectedItem(index);
            // If there's no change in section index, no need to switch sections.
            if (index === this.getCurrentSectionIndex()) {
                return this;
            }
            // Don't go for a section switch if the previous one is still transitioning
            if (!this._sectionWrapper.isAnimating()) {
                if (this.reportController) {
                    this.reportController.setCurrentSectionIndex(index);
                }
            } else {
                // Hold on to the last clicked section index
                this._lastIndex = index;
                this._lastIndexSaved = true;
            }
            return this;
        },

        _setSectionTabBarSelectedItem: function(index) {
            if(this._sectionTabBar) {
                var items = this._sectionTabBar.getItems();
                if (items && index >= 0 && index < items.length) {
                    var item = items[index];
                    // Initially selectedItem could be undefined
                    var selectedItem = this._sectionTabBar.getSelectedItem();
                    // We want to set the selected item only if it is not already set,
                    // else it causes the select handler to be invoked again.
                    if (item !== selectedItem) {
                        this._sectionTabBar.setSelectedItem(item);
                    }
                }
            }
        },
        
        /**
         * @returns {sas.ltjs.BIRD.views.Section} Current section.
         */
        getCurrentSectionView: function () {
            return this._sectionWrapper.getActiveContent();
        },

        /**
         * @returns {number} Current section's index.
         */
        getCurrentSectionIndex: function () {
            if (this.reportController) {
                return this.reportController.getCurrentSectionIndex();
            }
            return null;
        },

        /**
         * @returns {number} Number of sections.
         */
        getNumberOfSections: function () {
            return this.reportController.getNumberOfSections();
        },

        renderer: {
            render: function (rm, control) {
                rm.addClass('Report');
                rm.write("<div");

                rm.writeControlData(control);
                rm.writeStyles();
                rm.writeClasses();
                rm.write(">");

                rm.renderControl(control.getAggregation("header"));
                if (control._sectionViewOnly) {
                    rm.renderControl(control._sectionWrapper);
                } else {
                    // Render the section tab bar which contains the panel container which contains the section wrapper
                    rm.renderControl(control._sectionTabBar);
                }

                var exitFullscreenButton = control.getAggregation("exitFullscreenButton");
                if (exitFullscreenButton) {
                    rm.renderControl(exitFullscreenButton);
                }

                rm.write("<div id='" + control.getId() + "-AriaLiveDiv' aria-live='polite' style='opacity:0;height:0;'></div>");

                rm.write("</div>");
            },

            getCustomCSSRules: function (control) {
                var css;
                var customStyles = control.getCustomStyles().getStyles();
                if(customStyles) {
                    css = {};
                    var id = "#" + control.getId();
                    css[id] = customStyles;
                }
         
                return css;
            }
        },

        _updateReportSize: function (forceUpdate) {
            if(!this._sectionWrapper) {
                return;
            }

            var sectionWrapperDiv = this._sectionWrapper.getDomRef();
            if (!sectionWrapperDiv) {
                this._reportViewSize = {
                    width: 400,
                    height: 400
                };
            } else {
                if (!forceUpdate && this._reportViewSize && (sectionWrapperDiv.clientWidth === this._reportViewSize.width) &&
                    (sectionWrapperDiv.clientHeight === this._reportViewSize.height)) {
                    return;
                }

                this._reportViewSize = {
                    width: sectionWrapperDiv.clientWidth,
                    height: sectionWrapperDiv.clientHeight
                };
            }
            var sectionView = this.getCurrentSectionView();
            if (!sectionView) {
                return;
            }
            
            // Check for invalid bounds.
            var bounds = this._sectionWrapper.getBounds();
            if(!bounds || bounds.width <= 0 || bounds.height <= 0) {
                return;
            }
            if (this._isFixedSizeReport) {
                var fixedBounds = {top: 0, left: 0, width: this._fixedReportWidth, height: this._fixedReportHeight};
                sectionView.setFixedBounds(fixedBounds, bounds);
            }
            sectionView.setBounds(bounds);

            
            // S1339990: Safari flex box and 100% height bug: https://bugs.webkit.org/show_bug.cgi?id=137730
            // Although this bug is fixed in Safari 11, our browser matrix currently supports
            // Safari 10 which has this bug. So use calc to set the height of the icon tab bar to be:
            // (100% - the height of the report header)
            if(!this._isFixedSizeReport && (ltjsUtil.getBrowserName() === ltjsUtil.BROWSER.SAFARI) && this.hasReportPrompts()) {
                $(".sectionIconTabBar").css({'height': 'calc(100% - ' + $(".vavHeaderScroll").height() + 'px)'});
            }
        },

        _setSectionTabBarFixedBounds: function() {
            if (!this._sectionTabBar) {
                return;
            }
            var fixedBounds = {top: 0, left: 0, width: this._fixedReportWidth, height: this._fixedReportHeight};
            this._sectionTabBar.removeStyleClass("sasVaFlexFillSpace");
            // Fixed sizing of the _sectionTabBar
            // S1490422: With HC 9.0, there is a 1 px border for the IconTabBar content on 
            // three sides - left, right and bottom. So, we have to account for that.
            // VAN doesn't have this problem since they do no use the IconTabBar content, they
            // have their own container in place of that content.
            var tabWidth = fixedBounds.width + this._collapsedSidePanelWidth + 2;
            var tabHeight = fixedBounds.height + this._pageTabsHeaderHeight + 1;
            this._sectionTabBar.addStyleClass("vavCenterContent");
            var reportContainerDom = this.getDomRef();
            if (!reportContainerDom) {
                return;
            }
            var reportHeaderHeight = 0;
            var headerView = this.getAggregation("header");
            if (headerView && headerView.getVisible()) {
                reportHeaderHeight = headerView.$().height();
            }
            var maxAvailableTabHeight = reportContainerDom.clientHeight - reportHeaderHeight;
            var maxAvailableTabWidth = reportContainerDom.clientWidth;
            
            // Find the min bounds of what the section tab bar is supposed to take and 
            // what is available if the window is resized.
            tabWidth = Math.min(tabWidth, maxAvailableTabWidth);
            tabHeight = Math.min(tabHeight, maxAvailableTabHeight);
            var $sectionTabBar = this._sectionTabBar.$();
            var currentTabWidth = $sectionTabBar.width();
            var currentTabHeight = $sectionTabBar.height();
            if (currentTabWidth !== tabWidth) {
                $sectionTabBar.width(tabWidth + "px");
            }
            if (currentTabHeight !== tabHeight) {
                $sectionTabBar.height(tabHeight + "px");
            }
        },


        /**
         * The id used to remove a listener from the {@link ResizeHandler}.
         * @type {string}
         * @private
         */
        _addResizeListeners: function () {
            this._resizeListenerId = ResizeHandler.register(this._sectionWrapper.getDomRef(), jQuery.proxy(this._resizeHandler, this));
            if (this._isFixedSizeReport) {
                this._reportContainerResizeId = ResizeHandler.register(this.getDomRef(), jQuery.proxy(this._setSectionTabBarFixedBounds, this));
            }
        },
        _removeResizeListeners: function () {
            ResizeHandler.deregister(this._resizeListenerId);
            ResizeHandler.deregister(this._reportContainerResizeId);
        },
        _resizeHandler: function() {
            this._updateReportSize();
            TooltipManager.instance().hideTooltips();
        },

        _addReportListeners: function () {
            //  While refreshing the report, event listeners get added again. So first remove the previously added event listeners, if any.
            this._removeReportListeners();
            this._handles = this._handles || [];
            this._handles.push(this.reportController.attachEvent(ReportController.CURRENT_SECTION_CHANGED, this._handleReportControllerEvent, this));
            // This is fired only once when the section is created/switched-to for first time.
            this._handles.push(this.reportController.attachEvent(ReportController.SECTION_CONTROLLERS_CHANGED, this._handleReportControllerEvent, this));
            this._handles.push(this.reportController.attachEvent(InteractionEvent.ACTION, this._handleInteractionControllerEvent, this));

            sas.eventBus.subscribe(sas.themes.THEMES, sas.themes.REPORT_THEME_LOAD_ERROR, this._handleReportThemeLoadError, this);
            sas.eventBus.subscribe(sas.themes.THEMES, sas.themes.REPORT_THEME_LOADED, this._handleReportThemeLoaded, this);
            this.attachEvent(VisualEvent.OBJECT_SELECTED, this._onObjectSelected);
            this.attachEvent(VisualEvent.LAYOUT_CHANGED, this._onLayoutChanged);

            this._sectionWrapper.attachTransitionComplete(this._onTransitionComplete, this);
            this._sectionWrapper.attachEvent(VisualEvent.LAYOUT_CHANGED, this._onLayoutChanged, this);
        },

        _removeReportListeners: function () {
            if (this._handles) {
                for (var i = this._handles.length - 1; i >= 0; i--) {
                    this._handles[i].detach();
                }
            }
            this._handles = null;
            sas.eventBus.unsubscribe(sas.themes.THEMES, sas.themes.REPORT_THEME_LOAD_ERROR, this._handleReportThemeLoadError, this);
            sas.eventBus.unsubscribe(sas.themes.THEMES, sas.themes.REPORT_THEME_LOADED, this._handleReportThemeLoaded, this);
            
            this.detachEvent(VisualEvent.OBJECT_SELECTED, this._onObjectSelected);
            this.detachEvent(VisualEvent.LAYOUT_CHANGED, this._onLayoutChanged);

            this._sectionWrapper.detachTransitionComplete(this._onTransitionComplete);
            this._sectionWrapper.detachEvent(VisualEvent.LAYOUT_CHANGED, this._onLayoutChanged, this);
        },

        _createSectionTabs: function (sectionLabels) {
            //
            // Remove/destroy the existing tabs and create new ones.
            //
            if (this._sectionTabBar) {
                this._sectionTabBar.destroyItems();
                if(sectionLabels) {
                    $.each(sectionLabels, function(index, label) {
                        var sectionTabFilter = new IconTabFilter({
                            text: label
                        });
                        this._sectionTabBar.addItem(sectionTabFilter);
                    }.bind(this));
                }
            }
            //
            // Set the initial section selected on initial report load
            // to get the correct style on the section tab.
            // 
            this.setCurrentSectionIndex(this.getCurrentSectionIndex());
        },

        setReportLoadPending: function (loadPending) {
            if(this._sectionTabBar) {
                this._sectionTabBar.setBusy(loadPending);
            }
        },

        _updateSidePanel: function() {
            //
            // Delayed instantiation of the InfoPanel
            //
            if(!this._infoPanel && this._vaViewerConfig.getShowInfoPanel()) {
                this._infoPanel =  new InfoPanel({
                    vaViewerConfig: this._vaViewerConfig
                });
                if (this._panelContainer) {
                    this._panelContainer.addSidePanel(this._infoPanel);
                }
            }

            //
            // Update the side panel 
            //
            if(this._infoPanel) {
                this._infoPanel.setReportContainer(this);
                if (this._transportReport) {
                    this._infoPanel.setTransportReport(this._transportReport);
                }
                this._updateSidePanelDisplay();
            }
        },

        _updateSidePanelDisplay: function() {
            if (this._infoPanel) {
                this._infoPanel.updateDisplay();
            }
        },

        setInfoPanelOverflowItems: function(overflowButtons) {
            // Delayed addition of the overflow items to the info panel in component case
            // or overflow menu button in application case.
            if (this._infoPanel) {
                // The overflow button is never shown on the header when using 
                // ReportViewer component. Instead, it is shown as part of the side panel.
                this._infoPanel.setOverflowItems(overflowButtons);
            }
        },

        hasInfoPanelOverflowItems: function() {
            if (this._infoPanel) {
                var overflowItems = this._infoPanel.getOverflowItems();
                if (overflowItems && overflowItems.length > 0) {
                    return true;
                }
            }
            return false;
        },

        _onSectionTabSelect: function (evt) {
            var source = evt.getSource();
            var items = source.getItems();
            var currItem = evt.getParameter("item");
            var index = items.indexOf(currItem);
            this.setCurrentSectionIndex(index);
        },

        _switchSection: function (isBackwards) {
            var reportController = this.getReportController();
            if (reportController && this._sectionTabBar) {
                var items = this._sectionTabBar.getItems();
                if (items.length > 1) {
                    var index;
                    if (isBackwards) {
                        index = reportController.getCurrentSectionIndex() - 1;
                        if (index < 0) {
                            index = items.length - 1;
                        }
                    } else { 
                        index = reportController.getCurrentSectionIndex() + 1;
                        if (index >= items.length) {
                            index = 0;
                        }
                    }
                    this.setCurrentSectionIndex(index);
                }
            }
        },

        getCurrentSectionFocusableContent: function () {
            var sectionView = this.getCurrentSectionView();
            if (sectionView) {
                var bodyView = sectionView.getBodyView();
                if (bodyView) {
                    return bodyView.getFocusableContent();
                }
            }
            return null;
        },

        _setFocus: function() {
            if (this._autoFocusFirstVisualFlag) {
                var focusableContent = this.getCurrentSectionFocusableContent();
                if (focusableContent && focusableContent.length > 0) {
                    focusableContent[0].focus();
                }
                this._autoFocusFirstVisualFlag = false;
            }
        },

        getPlaybackNavigation: function() {
            return this._playbackNavigation;
        },

        startPlayback: function (animation) {
            this.addStyleClass("vavPlayback");
            this._playbackMode = true;
            this._playbackAnimation = animation;
            
            // We never want to see the side panel when in playback mode
            if (this._infoPanel) {
                this._infoPanel.setVisible(false);
            }

            // Display only the report content area
            if (animation.getShowCanvasOnly()) {
                this.addStyleClass("vavReportAreaOnly");
            }

            // unexpand any visuals that were expanded by the user
            var sectionView = this.getCurrentSectionView();
            if (sectionView) {
                sectionView.unexpandExpandedVisual();
                // Update the timer count even if timer is not enabled.
                // It is used in pause time calculations.
                this._updateTimerCount(this._getFocusableContentForSection(sectionView));
                // Make the timer visible, if enabled.
                if (animation.getShowTimer()) {
                    this._playbackTimer.setVisible(true);
                }
            }
            this._playbackNavigation.addKeyboardListeners();
            if (animation.getShowNavigationControls()) {
                this._playbackNavigation.setControlsTooltip(animation.isMaximizedMode());
                this._playbackNavigation.setVisible(true);
            }
            this._addPlaybackTransitionListeners(sectionView);
            
            this.setSelectedObject(null, animation);

            // TODO turn off auto save on report session

            this._enableOverlaysAfterPlayback = this._overlaysEnabled;
            this.setOverlaysEnabled(false);
            // If we were already fullscreen, the exit fullscreen button would be visible
            // which should be hidden since user would not be able to click it.
            var exitFullscreenButton = this.getAggregation("exitFullscreenButton");
            if (exitFullscreenButton) {
                exitFullscreenButton.setVisible(false);
            }
        },

        stopPlayback: function () {
            this._playbackMode = false;
            var animation = this._playbackAnimation;
            if (animation) {
                this._playbackTimer.clearTimer();
                if (animation.getShowTimer()) {
                    this._playbackTimer.setVisible(false);
                }
                this._playbackNavigation.removeKeyboardListeners();
                if (animation.getShowNavigationControls()) {
                    this._playbackNavigation.setVisible(false);
                    this._removePlaybackTransitionListeners(this.getCurrentSectionView());
                }
            }
            this._playbackAnimation = null;

            if (this._infoPanel) {
                this._infoPanel.setVisible(true);
            }
            this.removeStyleClass("vavReportAreaOnly");
            this.removeStyleClass("vavPlayback");

            // unexpand any visuals that were expanded by playback
            var sectionView = this.getCurrentSectionView();
            if (sectionView) {
                sectionView.clearMaximizedView();
            }

            // reset selection
            this.setSelectedObject(this);

            // TODO re-enable auto save on report session

            this.setOverlaysEnabled(this._enableOverlaysAfterPlayback);
        },

        // pauseAnimations: function () {
        //     // TODO, fire event, iterate through all visuals/containers?
            
        // },

        _getFocusableContentForSection: function(sectionView) {
            if (!sectionView) {
                return null;
            }
            var bodyView = sectionView.getBodyView();
            if (!bodyView) {
                return null;
            }
            return bodyView.getFocusableContent();
        },

        getPlaybackTimer: function() {
            return this._playbackTimer;
        },

        _updateTimerCount: function(allVisualViews) {
            var timerCount = this._playbackAnimation.getSecondsPerUnit();
            if (this._playbackAnimation.getTransitionUnit() === PlaybackAnimation.OBJECT 
                && !this._playbackAnimation.getShowMaximized()) {
                if (allVisualViews && allVisualViews.length) {
                    timerCount *= allVisualViews.length;
                }
            }
            this._playbackAnimation.setTimerCount(timerCount);
        },

        switchToNextSection: function() {
            this._autoFocusFirstVisualFlag = true;
            this._switchSection(false);
        },

        switchToPrevSection: function() {
            this._autoFocusFirstVisualFlag = true;
            this._switchSection(true);
        },

        _handleReportControllerEvent: function (e) {
            if (e.getType() === ReportController.CURRENT_SECTION_CHANGED) {
                this._updateLoadedViews();
                // Firing the SECTION_CHANGE event for ViewerContainer to fetch updated navigation stack info
                // to update the page title, incoming filters and prompt toggle button state
                this.fireEvent(ViewerEventType.SECTION_CHANGE, {
                    sectionIndex: this.getCurrentSectionIndex()
                }, true, true);
                this._setFocus();
            } else if (e.getType() === ReportController.SECTION_CONTROLLERS_CHANGED) {
                this._updateLoadedViews();
            }
        },

        _handleInteractionControllerEvent: function (e) {
            if (e.getType() === InteractionEvent.ACTION && e.getAction() instanceof NavigationAction) {
                var navigationAction = e.getAction();
                switch(navigationAction.getType()) {
                    case NavigationActionType.SECTION: {
                        var reportController = this.reportController;
                        if(reportController) {
                            var sectionName = navigationAction.getValue();
                            if(reportController.isHiddenSection(sectionName)) {
                                //Put the hidden section in a dialog
                                var hiddenSec = new HiddenSectionDialog();
                                hiddenSec.addPropagatingStyleClass(this._reportTheme);
                                // This event would get removed when the dialog is destroyed on closing it.
                                hiddenSec.attachEvent(VisualEvent.OBJECT_SELECTED, this._onHiddenSectionObjectSelected, this);
                                hiddenSec.show(this._transportReport, reportController, this._reportViewSize, sectionName, e, this._vaViewerConfig, this._reportTheme, this._overlaysEnabled);
                            } else {
                                sas.eventBus.publish(this._vaViewerConfig.getChannelId(), ViewerEventType.SECTION_LINK, {
                                    interactionEvent: e
                                });
                            }
                        }
                        break;
                    }
                    case NavigationActionType.URL: {
                        var extURL = e.getResolvedURL();
                        if(extURL) {
                            window.open(this._replacePound(this._stripLocalHost(extURL)), '_blank');
                            window.focus();
                        }
                        break;
                    }
                    case NavigationActionType.REPORT: {
                        sas.eventBus.publish(this._vaViewerConfig.getChannelId(), ViewerEventType.REPORT_LINK, {
                            interactionEvent: e
                        });
                        break;
                    }
                }
            }
        },

        _stripLocalHost: function(url) {
            var localHostRegex = /(http|https):\/\/localhost(:\d*)?(\/.*)/;
            var matches = url.match(localHostRegex);
            if(matches && matches.length) {
                url = matches.pop();
            }
            return url;
        },

        _replacePound: function(url) {
            return url.replace(/%23/g, "#");
        },

        _onHiddenSectionObjectSelected: function(event) {
            this.setSelectedObject(event.getParameter("object"));
        },

        _handleReportThemeLoadError: function () {
            this.setBusy(false);
            MessageDialog.error({text: rb.getText("ReportContainer.themeLoadFailed.txt"), title: sas.config.getAppName()});
        },

        _handleReportThemeLoaded: function () {
            this.setBusy(false);
            this._updateReportSize(true);
        },

        _onObjectSelected: function (event) {
            this.setSelectedObject(event.getParameter("object"));
        },

        _updateLoadedViews: function () {
            this._changeSection();
        },

        // S1424134: play animation after transition ends, animation isn't loaded at playback selection time the first
        // time an animation is viewed
        _playSelectedVisualAnimation: function () {
            if (!this._playbackMode || !this._selectedObject || !this._selectedObject.getVisualElementView) {
                return;
            }

            var visualElementView = this._selectedObject.getVisualElementView();
            if (visualElementView) {
                visualElementView.playAnimation();
            }
        },

        // Should be called only when playback controls are enabled
        _addPlaybackTransitionListeners: function(sectionView) {
            if (!this._playbackAnimation) {
                return;
            }
            if (this._playbackAnimation.isMaximizedMode()) {
                sectionView._slideTransition.attachTransitionStart(this._onPlaybackTransitionStart, this);
                sectionView._slideTransition.attachTransitionComplete(this._onPlaybackTransitionComplete, this);
            }
            this._addPlaybackSectionWrapperTransitionListeners();
        },

        _addPlaybackSectionWrapperTransitionListeners: function() {
            this._sectionWrapper.attachTransitionStart(this._onPlaybackTransitionStart, this);
            this._sectionWrapper.attachTransitionComplete(this._onPlaybackTransitionComplete, this);
        },

        _removePlaybackTransitionListeners: function(sectionView) {
            if (!this._playbackAnimation) {
                return;
            }
            if (this._playbackAnimation.isMaximizedMode()) {
                sectionView._slideTransition.detachTransitionStart(this._onPlaybackTransitionStart, this);
                sectionView._slideTransition.detachTransitionComplete(this._onPlaybackTransitionComplete, this);
            }
            this._sectionWrapper.detachTransitionStart(this._onPlaybackTransitionStart, this);
            this._sectionWrapper.detachTransitionComplete(this._onPlaybackTransitionComplete, this);
        },

        _onPlaybackTransitionStart: function() {
            this._playbackNavigation.setIsTransitionComplete(false);
        },

        _onPlaybackTransitionComplete: function() {
            this._playbackNavigation.setIsTransitionComplete(true);
            // Once the transition is complete, it could have been complete either from
            // section switch or from section's object transition in maximized mode.
            // When we switch sections, the listners would be only on the section wrapper.
            // But now, we need to add the listeners again on the current Section.
            // So, we first remove all the listeners and add them back to avoid multiple listeners.
            this._removePlaybackTransitionListeners(this.getCurrentSectionView());
            this._addPlaybackTransitionListeners(this.getCurrentSectionView());
        },

        _createSection: function () {
            if (this.reportController) {
                var currentIndex = this.getCurrentSectionIndex();
                if(currentIndex === this._currentRenderedViewIndex) {
                    return;
                }
                var sectionView = new Section({vaViewerConfig: this._vaViewerConfig});
                sectionView.setController(this.reportController.getCurrentSectionController());
                sectionView.setHeaderVisible(this.getPromptBarVisible());

                if (this._reportTheme) {
                    sectionView.addPropagatingStyleClass(this._reportTheme);
                }
                sectionView.setSelectable(true);
                sectionView.setOverlaysEnabled(this._overlaysEnabled);

                sectionView._slideTransition.attachTransitionComplete(this._playSelectedVisualAnimation, this);

                if (this._playbackMode && this._playbackAnimation) {
                    // Update the timer count for each new section
                    this._updateTimerCount(this._getFocusableContentForSection(sectionView));
                    if (this._playbackAnimation.getShowNavigationControls()) {
                        // Since we are changing sections, we need transition listeners
                        // only on the section wrapper. The Section's slide transition also
                        // fires start and complete events in object maximized mode which we don't 
                        // want to listen to during section switch.
                        this._removePlaybackTransitionListeners(this.getCurrentSectionView());
                        this._addPlaybackSectionWrapperTransitionListeners();
                    }
                }

                if (this._currentRenderedViewIndex < currentIndex) {
                    this._sectionWrapper.transitionContentLeft(sectionView);
                } else {
                    this._sectionWrapper.transitionContentRight(sectionView);
                }
                this._currentRenderedViewIndex = currentIndex;
            }
        },

        _changeSection: function () {
            // Create the section first and then start the transition
            this._createSection();
            // The model in the side panel object selector should update right away
            // to contain the new section and its visuals before we set the selection on it.
            this._updateSidePanelDisplay();
            // Set the new section as the selected object.
            var newSection = this.getCurrentSectionView();
            if (newSection) {
                this.setSelectedObject(newSection);
            }

        },

         _onTransitionComplete: function () {
             // We could be in middle of transition when sections were switched rapidly.
             // So now, we can switch to the last clicked section.
             if(this._lastIndexSaved) {
                 this._lastIndexSaved = false;
                 this.setCurrentSectionIndex(this._lastIndex);
             } else if (this.reportController) {
                 // update aria-live
                 var currentPageLabel = this.reportController.getCurrentSectionController().getModel().getLabel();
                 this.fireEvent(VisualEvent.UPDATE_ARIA_LIVE, {text: currentPageLabel}, true, true);
             }
             this._playSelectedVisualAnimation();
        },

        _createHeader: function () {
            var headerController = this.reportController.getHeaderController();
            if (!headerController) {
                return;
            }
            var headerView = new Header();
            headerView.addStyleClass("ReportPromptArea");
            headerView.setController(headerController);
            headerView.setVisible(this.getPromptBarVisible());
            headerView.addStyleClass("vavReportHeader");
            headerView.addEventDelegate({
                ontap: function() {
                    this.setSelectedObject(this);
                }
            }, this);

            // Apply report prompt header overrides.
            // This is only applicable for reports created in VAD. 
            var reportPromptStyleChain = this.reportController.getReportPromptStyleChain();
            var bgColor = StyleUtils.getBackgroundColorString(reportPromptStyleChain);
            if (bgColor) {         
                headerView.setCustomStyles(new Styles().setStyles({'background-color': bgColor}));
            }

            this.setAggregation("header", headerView);
        },

        _onLayoutChanged: function () {
            if (this._minSizeTimeoutID === null) {
                this._minSizeTimeoutID = setTimeout(function() {
                    // break the loop
                    // Some defensive programing since this is a patch fix for legacy code.
                    if(this._maxLayoutLoop >= 10) {
                        this._updateReportSize(true);
                        // clear timeoutID second so any new events won't trigger another timer
                        this._minSizeTimeoutID = null;
                        this._maxLayoutLoop = 0;
                    } else {
                        this._maxLayoutLoop++;
                        this._minSizeTimeoutID = null;
                        this._updateReportSize(true);
                        if(this._minSizeTimeoutID === null) {
                            this._maxLayoutLoop = 0;
                        }
                    }
                }.bind(this), 0);
            }
        },

        setSelectedObject: function(value, playbackAnimation) {
            // During playbackMode, selection is driven by the PlaybackAnimationController. So short circuit other forms of selection.
            if(this._playbackMode && !playbackAnimation) {
                return;
            }
            // If value is null, set the report to be the selected object.
            if (!value) {
                if(this._selectedObject && this._selectedObject !== this) {
                    this._selectedObject.setSelected(false);
                }
                this._selectedObject = this;
                return;
            }
            // If the current selected object is same as the new one, just return.
            if (this._selectedObject === value) {
                return;
            }
            // Set the selection of previously selected object to false
            if(this._selectedObject && this._selectedObject !== this) {
                this._selectedObject.setSelected(false, playbackAnimation);
            }
            // Set the selection of the new object to true
            this._selectedObject = value;
            if(this._selectedObject !== this) {
                this._selectedObject.setSelected(true, playbackAnimation);
            }
            // Fire the selection event for the passed in object
            this.fireSelectedObjectChange({object: value});
            // set current selected controller for auto interactions to work
            // this is only applicable to Visual objects, e.g. not composites.
            if(this.reportController) {
                var currentController = null;
                if (this._selectedObject instanceof BIRDTitledContent) {
                    var visualController = this._selectedObject.getController();
                    if (visualController instanceof VisualController) {
                        currentController = visualController.getVisualElementController();
                    } else {
                        currentController = visualController;
                    }
                } else if(this._selectedObject === this) {
                    currentController = this.reportController;
                } else if (this._selectedObject instanceof Section) {
                    currentController = this.reportController.getCurrentSectionController();
                }
                this.reportController.setCurrentSelectedController(currentController);
            }
        },

        getSelectedObject: function() {
            return this._selectedObject;
        },

        hasIncomingLink: function() {
            var controller = this.reportController;
            if(controller) {
                // If either reportLinkInfo or linkInfoForCurrentSection are non-null
                // then there are incoming filters
                return (!!controller.getReportLinkInfo() || !!controller.getLinkInfoForCurrentSection());
            }

            return false;
        },

        getIncomingReportFilterDescriptions: function() {
            var filterExpressions = null;
            if(this.reportController) {
                var interactionEvent = this.reportController.getReportLinkInfo();
                if(interactionEvent) {
                    filterExpressions = interactionEvent.getFilterExpressionDescriptions(this.reportController);
                }
            }
            return filterExpressions;
        },

        getIncomingSectionFilterDescriptions: function() {
            var filterExpressions = null;
            if(this.reportController) {
                var sectionLinkEvents = this.reportController.getSectionLinkEvents();
                if(sectionLinkEvents) {
                    var length = sectionLinkEvents.length;
                    // get the most recent section link event
                    var latestSectionLinkEvent = sectionLinkEvents[length - 1];
                    if(latestSectionLinkEvent) {
                        filterExpressions = latestSectionLinkEvent.getFilterExpressionDescriptions(this.reportController);
                    }
                }
            }
            return filterExpressions;
        },

        /**
         * Returns if the report has any report level prompts
         * @returns {Boolean} true if report level prompts are present, false otherwise
         */
        hasReportPrompts: function() {
            if(this.getHeader()) {
                return true;
            }
            return false;
        },

        /**
         * Returns if the report has any page level prompts
         * @returns {Boolean} true if page level prompts are present, false otherwise
         */
        hasPagePrompts: function() {
            if(this.reportController) {
                var reportModel = this.reportController.getModel();
                if(reportModel) {
                    var view = reportModel.getView();
                    if(view) {
                        var sections = view.getSections();
                        if(sections) {
                            for(var sectionIndex = 0; sectionIndex < sections.length; ++sectionIndex) {
                                var header = sections[sectionIndex].getHeader();
                                if(header) {
                                    return true;
                                }
                            }
                        }
                    }
                }
            }
            return false;
        },

        /**
         * Returns if the current page has page level prompts
         * @returns {Boolean} true if page level prompts are present, false otherwise
         */
        hasCurrentPagePrompts: function() {
            var sectionView = this.getCurrentSectionView();
            if(sectionView) {
                if(sectionView.getHeaderView()) {
                    return true;
                }
            }
            return false;
        },

        /**
         * Returns if the report has any auto interactions on any page
         * @returns {Boolean} true if auto interactions are present, false otherwise
         */
        hasAutoInteractions: function() {
            if(this.reportController) {
                var reportModel = this.reportController.getModel();
                if(reportModel) {
                    var view = reportModel.getView();
                    if(view) {
                        var sections = view.getSections();
                        if(sections) {
                            for(var sectionIndex = 0; sectionIndex < sections.length; ++sectionIndex) {
                                var section = sections[sectionIndex];
                                if (section.getShowSelectionControl()) {
                                    return true;
                                }
                            }
                        }
                    }
                }
            }
            return false;
        },

        /**
         * Returns if the current page has auto interactions
         * @returns {Boolean} true if page auto interactions are present, false otherwise
         */
        hasCurrentAutoInteractions: function() {
            if(this.reportController) {
                var reportModel = this.reportController.getModel();
                if(reportModel) {
                    var view = reportModel.getView();
                    if(view) {
                        var sections = view.getSections();
                        if (sections) {
                            var currentSection = sections[this.getCurrentSectionIndex()];
                            if (currentSection) {
                                return currentSection.getShowSelectionControl();
                            }
                        }
                    }
                }
            }
            return false;
        },

        _getComparisonEpsilon: function(report) {
            //
            // Get the comparisonEpsilon from the ServerConfig. Default
            // to 0 if for some reason we can't get it (shouldn't happen).
            //
            if(report) {
                var connection = report.getConnection();
                if(connection) {
                    var server = connection.getServer();
                    if(server) {
                        var serverConfig = server.getMeta();
                        if(serverConfig) {
                            return serverConfig.getComparisonEpsilon();
                        }
                    }
                }
            }
            return 0;
        },

        /**
         * Updates the ReportContainer with the given report.
         * @param {sas.vaviewer.reportitem.ReportDataItem} reportItem - the report item to open
         */
        setReport: function (reportItem) {
            this.destroyAggregation("header");

            var beginReportParse = window.performance.now();
            var transportReport = reportItem.getReport();
            // only release the old this._transportReport if transportReport has really changed
            if (!ClosetObject.equals(this._transportReport, transportReport)) {
                if(this._transportReport) {
                    this._transportReport.release();
                    this._transportReport = null;
                }
                if(transportReport) {
                    this._transportReport = transportReport.retain();
                }
            }
            if(this.reportController) {
                this.reportController.release();
            }

            // apply custom css override
            this._applyCSSOverride(reportItem.getReportPath());

            //
            // Create ReportSession
            //
            var reportSession = new ReportSession(reportItem.getReportManager(), reportItem.getReport());

            //
            // Initialize configuration options
            //
            reportItem.setReportSession(reportSession);
            reportSession.setAutoSavePrimaryState(true);
            reportSession.setAutoSaveIntervalInSeconds(10.0);
            this._initializeReportConfiguration(reportItem);

            //
            // Open report
            //
            reportSession.openReport();
            this.reportController = reportSession.getReportController().retain();
            var endReportParse = window.performance.now();
            console.log("PERFORMANCE: Time to create a ReportController and parse a report:", endReportParse - beginReportParse);

            var warnings = this.reportController.getWarnings();
            if (warnings) {
                var l = warnings.length;
                var warning;
                for (var i = 0; i < l; i++) {
                    warning = warnings[i];
                    if (warning.getLevel() === WarningLevel.FATAL_LEVEL) {
                        MessageDialog.error({text: warning.getMessage(), title: rb.getText("ReportContainer.reportCorrupted.txt")});
                        this.setBusy(false);
                        return;
                    }
                }
            }

            this._addReportListeners();

            // loads report theme if it's not already loaded and refreshes styles
            this.loadReportTheme();

            this._createHeader();
            // Set properties for section wrapper only when report is created
            this._sectionWrapper.addStyleClass("overflowHidden");
            this._sectionWrapper.setSizing(UISizing.FILL);
            this._sectionWrapper._setIsParentFlexbox(true);
            // Remove any of the section view child, if present. esp. during app refresh.
            this._sectionWrapper.destroyContent();
            this._currentRenderedViewIndex = -1;

            //
            // At this point we should have a valid section controller.  But if we don't then
            // that is because we had an invalid navigation context (bad section name or index).
            // Recover from that by setting the current section index to 0.
            //
            if(this.reportController.getCurrentSectionIndex() < 0) {
                this.reportController.setCurrentSectionIndex(0);
            }
            this._createSection();
            // S1358437: when the report is refreshed, we need to set the report as the selected object.
            this.setSelectedObject(this);
            //Ensure the size of the wrapper matches the report size. We need to invalidate since Container.layoutSubviews depends on knowing the DOM element size.
            //  We could alternatively attach an afterRendering event listener,  but this seems more strait forward at the moment. ~chedwa
            this.invalidate();
            // We want the side panel to listen to the object selected event fired by this class,
            // so create the side panel first and then the section tabs.
            this._updateSidePanel();

            var reportModel = this.reportController.getModel();
            var view = reportModel.getView();
            var sections = view.getSections();
            var sectionLabels = [];
            for (var j = 0, numSections = sections.length; j < numSections; j++) {
                sectionLabels.push(sections[j].getLabel());
            }
            this._createSectionTabs(sectionLabels);
            this.fireEvent(VisualEvent.UPDATE_ARIA_LIVE, {text: rb.getText("AriaLive.reportOpened.txt")}, true, true);
            this._checkForFixedSizeReport();
        },

        _checkForFixedSizeReport: function() {
            if (!this.reportController) {
                return;
            }
            if (this._sectionViewOnly || !this._sectionTabBar) {
                return;
            }
            var reportController = this.reportController;
            var fixedWidth = reportController.getFixedBodyWidth();
            var fixedHeight = reportController.getFixedBodyHeight();
            if (fixedWidth && fixedHeight) {
                var fixedWidthScalar = fixedWidth.getScalarValueForUnit();
                var fixedHeightScalar = fixedHeight.getScalarValueForUnit();
                if (fixedWidthScalar && fixedHeightScalar) {
                    this._isFixedSizeReport = true;
                    this._fixedReportWidth = fixedWidthScalar;
                    this._fixedReportHeight = fixedHeightScalar;
                    this.addStyleClass("overflowHidden");
                }
            } else {
                // On refresh, the ReportContainer is not initialized again.
                // So reset the properties, if fixed sizing of report is removed.
                this._isFixedSizeReport = false;
                this._fixedReportWidth = 0;
                this._fixedReportHeight = 0;
                this.removeStyleClass("overflowHidden");
            }
        },

        _applyCSSOverride: function (reportPath) {
            if (this._vaViewerConfig.getCustomReportCSSOverride()) {
                // parse report.xml and find the BaseStyleSheet file
                var reportContent = ltjsEnvironment._ltjsModule.getStringContentsOfFile((reportPath + "/report.xml"));
                if (reportContent) {
                    var match = reportContent.match(/<BaseStylesheetResource file="(.*?)"/);
                    if (match && match.length === 2) {
                        var baseCssFilePath = reportPath + "/" + match[1];
                        // overwrite the base stylesheet file with the custom css
                        ltjsEnvironment._ltjsModule.exports.FS.writeFile(baseCssFilePath, this._vaViewerConfig.getCustomReportCSSOverride());
                    }
                }
            }
        },

        /**
         * Updates the custom css on an already open report
         * @param {sas.vaviewer.reportitem.ReportDataItem} reportItem - the report item to open
         */
        updateReportCustomCss: function(reportItem) {
            if (this._transportReport && this.reportController) {
                var config = this.reportController.getSASReportConfiguration();
                var navContext = config.getNavigationContext();
                if (navContext) {
                    reportItem.setNavigationContext(navContext);
                }
                this.setReport(reportItem);
            }
        },

        /**
         * Loads the reportTheme specified or loads the report theme by querying the report.
         * @param {string=} reportTheme Report theme to load, e.g., "light".
         * @see sas.hc.ui.core.theme.ThemeUtil
         */
        loadReportTheme: function (reportTheme) {
            var sectionView;
            // Remove previous report theme
            if (this._reportTheme) {
                this._sectionWrapper.removeStyleClass(this._reportTheme);
                if (this._sectionTabBar) {
                    this._sectionTabBar._getIconTabHeader().removeStyleClass(this._reportTheme);
                }
                sectionView = this.getCurrentSectionView();
                if (sectionView) {
                    sectionView.removePropagatingStyleClass(this._reportTheme);
                }
            }

            if (!reportTheme) {
                // Determine report theme from report.
                var styleChain = this.reportController.getReportStyleChain();
                reportTheme = styleChain.getStringAttribute(AttributeNames.THEME, "umstead");
            }
            this._reportTheme = reportTheme;

            // Kick off download of report theme if it hasn't already been loaded
            sas.themes.loadReportTheme(reportTheme);

            // Apply the report theme to the section wrapper
            this._sectionWrapper.addStyleClass(this._reportTheme);
            // Apply the report theme to the icon tab bar header
            if (this._sectionTabBar) {
                this._sectionTabBar._getIconTabHeader().addStyleClass(this._reportTheme);
            }

            // Pass the report theme on to children who might render content
            // outside the sectionWrapper, e.g. popovers in static area
            sectionView = this.getCurrentSectionView();
            if (sectionView) {
                sectionView.addPropagatingStyleClass(this._reportTheme);
            }

            // Apply report style overrides. sectionWrapper is our root for 
            // report content
            var reportStyleChain = this.reportController.getReportStyleChain();
            var reportCss = StyleUtils.getAllTextStyles(reportStyleChain);
            if (reportCss) {
                this._sectionWrapper.setCustomStyles(new Styles().setStyles(reportCss));
            }
        },

        getTransportReport: function() {
            return this._transportReport;
        },
        
        /**
         * Gets the ReportConfiguration for a given report if one does not already exist.
         * @param {sas.vaviewer.reportitem.ReportDataItem} reportItem - the report item to open
         * @returns {sas.ltjs.BIRD.util.SASReportConfiguration} SASReportConfiguration.
         */
        _initializeReportConfiguration: function (reportItem) {
            var configuration = reportItem.getReportSession().getSASReportConfiguration();
            configuration.setFullyQualifiedPathToBIRDReportDirectory(reportItem.getReportPath());
            configuration.setUseSingleThreadMode(false);
            configuration.setUseDefaultGTLDataTips(false);
            configuration.setUseReadOnlyMode(false);
            configuration.setMaximumNumberOfSectionsInMemory(-1);
            if (VAViewerSettingsHelper.overrideWithHighContrastTheme) {
                configuration.setDoVisualStyleOverride(true);
            }
            configuration.setNavigationContext(reportItem.getNavigationContext());
            configuration.setMapConnectionInfo(reportItem.getMapConnectionInfo());
            configuration.setComparisonEpsilon(this._getComparisonEpsilon(this._transportReport));
            // Only honor URL Parameters on the first go. (This is what VAD does)
            var parameters = this._getReportParameters(reportItem.getReportParameters());
            if (parameters) {
                configuration.setParameterValues(parameters);
            }
            
            if (this._transportReport) {
                var midHost = this._vaViewerConfig.getMidHost();
                if (midHost) {
                    configuration.setTransportURL("http://" + midHost + ":" + this._vaViewerConfig.getMidHostPort());
                } else {
                    var isSecure = location.protocol === 'https:';
                    configuration.setTransportURL((isSecure ? 'https' : 'http') + "://" +
                        location.hostname + ":" + (location.port || (isSecure ? 443 : 80)));
                }
                var manifest = this._transportReport.getManifest();
                if (manifest) {
                    configuration.setCachedReportURLs(this._getCachedReportURLsFromManifest(manifest));
                    var contentKey = this._getContentKeyFromManifest(manifest);
                    if(contentKey) {
                        configuration.setContentKey(contentKey);
                    }
                }
            }
            TransportConnection.setIframeSandboxForSASReportConfiguration(configuration, this._transportReport); 
        },

        _getReportParameters: function(reportParameters) {
            var parameters = null, param, key, values;
            // For VAViewer application case, read the report parameters from the URL
            if (!this._vaViewerConfig.getComponentMode()) {
                var vavConfig = sas.config.get("vaviewer");
                if(vavConfig && !this._urlParametersInitialized) {
                    this._urlParametersInitialized = true;
                    // S1483322: Use top level url to override appSwitcher change impacting key encoding
                    var uri = sas.applicationSwitcher ? top.jQuery.sap.getUriParameters() : jQuery.sap.getUriParameters();
                    for (key in uri.mParams) {
                        values = uri.mParams[key];
                        if(values && !vavConfig[key]) { // Filter out known vav parms
                            param = new ReportParameterValue();
                            param.setLabel(key);
                            param.setValues(values);
                            if(!parameters) {
                                parameters = [];
                            }
                            parameters.push(param);
                        }
                    }
                }
            } else if (reportParameters) {
                // for the component case read the report parameters set on the ReportViewer, if any.    
                parameters = [];
                for (key in reportParameters) {
                    values = reportParameters[key];
                    if (!Array.isArray(values)) {
                        values = [values];
                    }
                    param = new ReportParameterValue();
                    param.setLabel(key);
                    param.setValues(values);
                    parameters.push(param);
                }
            }
            return parameters;
        },

        _getCachedReportURLsFromManifest: function(manifest) {
            var urls = manifest.getUrls();
            var cachedReportURLVector = [];
            if(urls) {
                for (var reportURLIndex = urls.length - 1; reportURLIndex >= 0; --reportURLIndex) {
                    var reportAssetURL = urls[reportURLIndex];
                    var assetType = reportAssetURL.getType();
                    if (assetType !== ReportAssetURLType.STP && assetType !== ReportAssetURLType.IMAGE) {
                        continue;
                    }

                    var cachedReportURL = new CachedReportURL();
                    cachedReportURL.setName(reportAssetURL.getName());
                    cachedReportURL.setType(CachedReportURLType.STP);
                    cachedReportURL.setContentKey(reportAssetURL.getContentKey());
                    cachedReportURL.setServerFile(reportAssetURL.getServerFile());
                    cachedReportURL.setContent(reportAssetURL.getUrlString());

                    cachedReportURLVector.push(cachedReportURL);
                }
            }

            return cachedReportURLVector;
        },

        _getContentKeyFromManifest: function(manifest) {
            //
            // R9 and earlier we simply search through manifest's urls
            // 
            var urls = manifest.getUrls();
            if(urls) {
                for(var urlIndex = 0; urlIndex < urls.length; ++urlIndex) {
                    var reportAssetURL = urls[urlIndex];
                    if(reportAssetURL && reportAssetURL.getType() === ReportAssetURLType.REPORT) {
                        return reportAssetURL.getContentKey();
                    }
                }
            }

            //
            // R10 and later we check the manifest's files
            // 
            var files = manifest.getFiles();
            if(files) {
                for(var fileIndex = 0; fileIndex < files.length; ++fileIndex) {
                    var file = files[fileIndex];
                    if(file && file.getType() === FileType.REPORT) {
                        return file.getContentKey();
                    }
                }
            }

            return null;
        },

        // Utility function that is useful for debugging and may be used by theme designer. It is currently not called anywhere, but I am
        // leaving it in as a private method until we determine how exactly theme designer will be using the ReportViewer component   
        _getReportCss: function () {
            var css = "";

            var reportController = this.reportController;
            if (reportController) {
                var configuration = reportController.getSASReportConfiguration();
                if (configuration) {
                    var reportDir = configuration.getFullyQualifiedPathToBIRDReportDirectory() + '/';
                    var mediaSchemes = reportController.getModel().getMediaSchemes();
                    if (mediaSchemes && mediaSchemes.length > 0) {
                        var baseStylesheetResource = mediaSchemes[0].getBaseStylesheetResource();
                        if (baseStylesheetResource) {
                            var file = baseStylesheetResource.getFile();
                            if (file) {
                                var fullFilePath = reportDir + file;
                                css = sas.ltjslib.ltjs.getStringContentsOfFile(fullFilePath);
                            }
                        }
                    }
                }
            }

            return css;
        },

        onBeforeRendering: function () {
            this._removeResizeListeners();
        },

        onAfterRendering: function () {
            if (this._isFixedSizeReport) {
                if (this._infoPanel) {
                    this._collapsedSidePanelWidth = this._infoPanel.$().width();
                }
                if (this._sectionTabBar) {
                    this._pageTabsHeaderHeight = this._sectionTabBar._getIconTabHeader().$().height();
                }
                this._setSectionTabBarFixedBounds();
            }
            this._addResizeListeners();
            this._updateReportSize(true);
        },

        // Add a setter and getter api for the transition flag to avoid any kind of weird object state since 
        // we wait for portable section change event before we actually switch sections.
        setPerformTransition: function(performTransition) {
            this._sectionWrapper.setAnimationsEnabled(performTransition);
        },

        getPerformTransition: function() {
            return this._sectionWrapper.getAnimationsEnabled();
        },

        cancelPageTransition: function () {
            this._sectionWrapper.cancelTransition();
        },

        destroy: function () {
            Control.prototype.destroy.apply(this, arguments);

            this._removeResizeListeners();
            this._removeReportListeners();
            this._reportViewSize = null;

            if (this._currSectionView) {
                this._currSectionView.destroy();
                this._currSectionView = null;
            }
            if (this._sectionWrapper) {
                this._sectionWrapper.destroy();
                this._sectionWrapper = null;
            }
            if(this.reportController) {
                this.reportController.release();
                this.reportController = null;
            }
            if(this._transportReport) {
                this._transportReport.release();
                this._transportReport = null;
            }
            if (this._panelContainer) {
                this._panelContainer.destroy();
                this._panelContainer = null;
            }
            if (this._sectionTabBar) {
                this._sectionTabBar.destroy();
                this._sectionTabBar = null;
            }
            if (this._infoPanel) {
                this._infoPanel.destroy();
                this._infoPanel = null;
            }
        }
    });
}, true);
