/**
 * @copyright
*  This software is protected by copyright laws and international treaties.
*  U.S. GOVERNMENT RESTRICTED RIGHTS Use, duplication, or disclosure of this
*  software and related documentation by the U.S. government is subject to the
*  Agreement with SAS Institute and the restrictions set forth in FAR 52.227-19,
*  Commercial Computer Software - Restricted Rights (June 1987).
*  Copyright (c) 2012 by SAS Institute Inc., Cary, NC, USA   All rights reserved.
*  
*  @file This file includes the APIs of HTML5 Enhanced Editor.
*  @author <a href="mailto:crdmdgh@sas.com">Mobile Development Group</a>
*  @version 1.9
*/
// Export as a dojo component.
define([
    "./_codeEditor/jquery/jquery-3.6.1.min",
    "./_codeEditor/EditorCore",
    "./_codeEditor/Basic",
    "./_codeEditor/Config",
    "./_codeEditor/EditorController",
    "./_codeEditor/Settings",
    "./_codeEditor/FindAndReplaceDialog",
    "./_codeEditor/GoToLineDialog",
    "./_codeEditor/CodeChecker"      // jpnjfk
], function(declare, EditorCore){   
/**
 * @namespace SAS
 * 
 */
SAS = window.SAS || {}; //Define SAS namespace.
/**
 *  @author <a href="mailto:crdmdgh@sas.com">Mobile Development Group</a>
 *  @version 1.0
 *  @module SAS.Editor
 *  @class SAS.Editor
 *  @classdesc This object defines HTML5 Enhanced Editor. A user can create a new instance by its constructor, 
 *  then operate the instance by its methods. HTML5 Enhanced Editor is one of HTML common components 
 *  that brings the ability of editing SAS code to the Web. There are several code editors in various SAS products,
 *  such as Enhanced Editor and Flex Code Editor. HTML5 Enhanced Editor will provide experience similar to them,
 *  and meet SAS' i18n and a11y requirements. 
 *  @constructor
 *  @param
 *  {string}id - This is the id of the container containing the new editor object, 
 *  it can be the id of an HTML element like div.
 *  @param 
 *  {string}content - The initial text to be put into the editor.
 *  @param 
 *  {integer}langMode - The language mode for the editor object that determines the syntax highlighting.
 *  @see SAS.Editor.LanguageMode
 */
SAS.Editor = function (id, content, langMode) {
    this.settings_ = new SAS.Settings();
    this._core = new EditorCore(id, content, langMode, this.settings_);
    this.ctrl_ = new SAS.EditorController(this._core);
};
SAS.Editor.prototype = {
    /**
     * This method frees the resources hold by the editor object.
     * 
     */
    dispose : function () {
        //this.ctrl_.dispose();
        //this.settings__.settingChanged.dispose();
    },
    /**
     * This method makes the editor instance active, then the user can do find,replace or switch tab correctly. 
     */
    activate : function () {
        this.ctrl_.activate();
    },
    /**
     * This method cancels the active status of the editor object.
     */
    deactivate : function () {
        this.ctrl_.deactivate();
    },
    /**
     * This method lets the editor object be in focus.
     *
     */
    focus : function () {
        this.ctrl_.focus();
    },
    /**
     * This method is used to register shortcut keys to the editor object.
     * All the handlers registered with the same shortcut keys will be executed by the sequence they were registered.
     * @param
     * {Array}config - It's an array object. 
     * Each element of it is an object contains property ctrlKey, shiftKey, altKey, keyCode and handle, 
     * which describes what keys to be registers and what handler will be invoked when these keys are pressed.
     * @example 
     * An example of config:
     * [{ctrlKey:false, altKey:false, shiftKey:false, keyCode: 9,  handle:function(e){..}},
     *  {ctrlKey:false, altKey:false, shiftKey:false, keyCode: 13,  handle:onReturn}].
     */
    regShortcuts: function (config) {
        this.ctrl_.regCommands(config);
    },
    /**
     * This method is used to cut the selected content that will be put on Clipboard.
     * 
     */
    cut : function () {        
        //this.ctrl_.focus();
        this.ctrl_.cut();
    },
    /**
     * This method will redo just action if the redo queue is not empty.
     * 
     */
    redo : function () {
        // this.ctrl_.focus();
        this.ctrl_.redo();
    },
    /**
     * This method will undo just action if the undo queue is not empty.
     * 
     */
    undo : function () {
        // this.ctrl_.focus();
        this.ctrl_.undo();
    },
    /**
     * This method returns if the user can do redo action.
     */
    canRedo : function () {
        return this.ctrl_.canRedo();
    },
    /**
     * This method returns if the user can do undo action.
     */
    canUndo : function () {
        return this.ctrl_.canUndo();
    },
    /**
     * This method returns if the user can do paste action.
     */
    canPaste : function () {
        return this.ctrl_.canPaste();
    },
    /**
     * This method will copy the selected content to Clipboard.
     * 
     */
    copy : function () {
        //this.ctrl_.focus();
        this.ctrl_.copy();
    },
    /**
     * This method will paste the content on Clipboard to the editor object at cursor.
     * 
     */
    paste : function () {
        // this.ctrl_.focus();
        this.ctrl_.paste();
    },
    /**
     * This method selects all the text in the editor object.
     * 
     */
    selectAll : function () {
        //this.ctrl_.focus();
        this.ctrl_.selectAll();
    },
    /**
     * This method queries the total line count in the editor object.   
     * 
     */    
    lineCount : function() {
        return this.ctrl_.lineCount();
    },
    /**
     * This method gets all content in the editor object. 
     *
     */
    getText : function () {
        return this.ctrl_.getText('all');
    },
    /**
     * This method gets the selected content in the editor object.
     */
    getSelectedText : function () {
        return this.ctrl_.getText("selected");
    },
    /**
     * This method is used to set/get the font size for the editor object.
     * 
     * @param 
     * {integer}size - If no size, the function only returns the current font size.
     * @return
     * The current font size.
     */
    fontSize : function (size) {
        return this.settings_.fontSize(size);
    },
    /**
     * This method is used to get tab size option or set it.
     * If calling without tabSize parameter, the method only returns option value.
     * If calling with tabSize parameter, it sets the option to a new value.
     * @param
     * {integer}tabSize - The new tab size to set.
     * @return
     * The tab size.
     * @since 2.0 or 3.0
     * @attention: Not available currently.
     */
    tabSize : function (tabSize) {
        //this.ctrl_.focus();
        return this.settings_.tabSize.apply(this.settings_, arguments);
    },
    /**
     * This method is used to enable the line number feature or disable it, 
     * and it can be used to get this option value.
     *  @param
     *  {boolean}enable - If calling with true or false, the method will enable line number or disable it.
     *           If calling without this parameter, the method only returns the option value.
     *  @param
     *  {boolean}temporary - If true, this setting will only affect current instance.
     * @return
     * If the line number feature is enabled.
     */
    lineNumber : function (enable, temporary) {
        //this.ctrl_.focus();
        return this.settings_.lineNumber.apply(this.settings_, arguments);
    },
    /**
     * This method is used to enable the auto completion feature or disable it,
     * and it can be used to get this option value.
     * @param
     * {boolean}enable - If calling with true or false, the method will enable auto completion or disable it.
     *                  If calling without this parameter, the method only returns the option value.
     * @return
     * Whether the auto completion feature is enabled or not.
     */
    autoComplete : function (enable) {
        return this.settings_.autoCompleteEnabled.apply(this.settings_, arguments);
    },
    /**
     * This method is used to enable the hint feature or disable it,
     * and it can be used to get this option value.
     * @param
     * {boolean}enable - If calling with true or false, the method will enable hint or disable it.
     *                  If calling without this parameter, the method only returns the option value.
     * @return
     * Whether the hint feature is enabled or not.
     */
    enableHint : function (enable) {
        return this.settings_.hintEnabled.apply(this.settings_, arguments);
    },
    /**
     * This method is used to enable the syntax highlighting feature or disable it,
     * and it can be used to get this option value.
     * @param
     * {boolean}enable - If calling with true or false, this method will enable syntax highlighting feature or disable it.
     *          If calling without this parameter, it only returns the option value. 
     * @return
     * If the syntax highlighting feature is enabled.
     */
    syntaxHighlighting : function (enable) {
        //this.ctrl_.focus();
        return this.settings_.syntaxHighlighting.apply(this.settings_, arguments);
    },
    /**
     * This method is used to enable the tab as spaces feature or disable it,
     * and it can be used to get this option value.
     * @param
     * {boolean}enable - If calling with true or false, the method will enable tab as spaces feature or disable it.
     *          If calling without this parameter, it only returns the option value.
     * @return
     * If the tab as spaces feature is enabled. 
     * @since 2.0 or 3.0
     * @attention: Not available currently.
     */
    tabAsSpaces : function (enable) {
        //this.ctrl_.focus();
        return this.settings_.tabAsSpaces.apply(this.settings_, arguments);
    },

    /**
     * This method is used to enable the line wrapped feature or disable it,
     * and it can be used to query this option value.
     * @param
     * {boolean}enable - If calling with true or false, the method will enable line wrapped feature or disable it.
     *          If calling without this parameter, it only returns the options value.
     * @return
     * If the editor object is read only.
     * @since 2.0 or 3.0
     * @attention: Not available currently.
     */
    lineWrapped : function (enable) {
        //this.ctrl_.focus();
        return this.settings_.lineWrapped.apply(this.settings_, arguments);
    },

    /**
     * The method queries if the editor object is read only.
     *
     */
    readOnly : function () {
        return this.settings_.readOnly.apply(this.settings_, arguments);
    },

    /**
     * The method moves the cursor to a specified line.
     * @param
     * {integer}line - The valid value scope is 1-n.
     */
    gotoLine : function (line) {
        this.ctrl_.gotoline(line);
    },
    /**
     * The method will replace all the content of the editor object using new content, 
     * and the internal undo/redo manager will be reset. 
     * @param
     *  {string}content - The new content to set to the editor object. 
     */
    setText : function (content) {
        //this.ctrl_.focus();
        this.ctrl_.setText(content);
    },
    /**
     *  The method inserts the specified content to the editor object. 
     *  If no selection, it will inserts the content to the cursor position.
     *  If there are some content selected, it will replace these content using new content.
     *  The user can use the undo interface to cancel this action.
     *  @param
     *  {string}content - The content to be inserted the editor object.  
     */
    insert : function (content) {
        this.ctrl_.insertText(content);
        window.setTimeout(this.focus.bind(this), 400);
    },
    /**
     * The method allows a user to listen to the events of the editor object.
     *  @param
     *  {string}evt - The event name to be listened, and it is a string containing one or more JavaScript event types.
     *  The allowed events include :<br>
     *     <li><strong>textChanged</strong> - raised when the Text property changes(occurs every time a character is typed).</li>
     *     <li><strong>selectionChanged</strong> - raised when the Selection property changes.</li>
     *     <li><strong>caretMoved</strong> - raised when the caret position changes.</li>
     *     <li><strong>editAttempted</strong> - raised when the user attempts to edit the text.</li>
     *  @param
     *  {function}listener - A function that will be called when the specified event occurs.
     *  The prototype of a listener is :
     *        void listener(event);<br><br>
     *  The event parameter is an object like:
     *        {type: 'textChanged', data: undefined};<br><br>
     *  and <br>
     *  (1)if type is textChanged or editAttempted, data is undefined,<br><br>
     *  (2)if type is selectionChanged, data is an object like 
     *         {start: {line:1, column:2}, end: {line:2, column:4}}<br><br>
     *  (3)if type is caretMoved, data is an object like
     *          {line:3, column:4}.<br>
     */
    bind : function (evt, listener) {
        this.ctrl_.bind(evt, listener);
    },
    /**
     * The method removes a previously-attached event listener from the editor object.
     * @see bind 
     * @param 
     * {string}evt - A string containing one or more JavaScript event types.
     * @param 
     * {function}lisenter - The function to be detached.  
     */
    unbind : function (evt, listener) {
        this.ctrl_.unbind(evt, listener);
    },
    /**
     * The method clears the content of the editor object, then the user can use undo method to cancel this action.
     * 
     */
    clear : function () {
        this.ctrl_.clear();
        //this.ctrl_.focus();
    },
    /**
     * The method does search in editor by the parameters
     * @param 
     * {string}key - The parameter is a text string to be searched.
     * @param
     * {object}config - The parameter specifies the searching rules, and it is an object with key/value pairs, as following:
     * @property
     *{string}direction - specifies the searching direction, the value may be "forward" or "backward".
     *@property
     *{boolean}caseIgnored - specifies if the searching is case sensitive, the value may be "false" or "true".
     *@property
     *{boolean}wrapped - specifies if the searching will begin from the start position of the document when reaching the end, 
     *the value may be "false" or "true",
     *@property
     *{boolean}all - specified either searching one or searching all, the value may be "false" or "true".
     *@example 
     * An example of parameter config :
     *   config = {
     *      direction : "forward",
     *      caseIgnored: true,
     *      wrapped: false,
     *      all: false
     * }
     */
    search : function (key, config) {
        // this.ctrl_.focus();
        return this.ctrl_.search(key, config);
        // this.ctrl_.focusOut();
    },
    /**
     * The method does replace the specified text with new text.
     * @param 
     * {string}key - A text string to be replaced.
     * @param
     * {string}value - A new text string to replace the specified text(key).
     * @param 
     * {object}config - The parameter specifies the replacing rules, and it is an object with key/value pairs, as following:
     * @property
     * {string}direction - specifies the searching direction, the value may be "forward" or "backward".
     * @property
     * {boolean}caseIgnored - specifies if the searching is case sensitive, the value may be "false" or "true".
     * @property
     * {boolean}wrapped - specifies if the searching will begin from the start position of the document when reaching the end, 
     * the value may be "false" or "true".
     * @property
     * {boolean}all - specified either replacing one or replacing all, the value may be "false" or "true".
     * @example
     * An example of parameter config :
     *   config = {
     *      direction : "forward",
     *      caseIgnored: true,
     *      wrapped: false,
     *      all: false
     * }
     */
    replace : function (key, value, config) {
        this.ctrl_.replace(key, value, config);
        this.ctrl_.focus();
    },
    /**
     * The method collapses one or all code block(s).
     * @param
     * all - If calling without this parameter, the method will collapse the code block with cursor.
     *       If calling with this parameter, it will collapse all code blocks.
     * @since 2.0 or 3.0
     * @attention: Not available currently.
     */
    /*collapse : function (all) {
        this.ctrl_.focus();
        if (!all) {
            this.ctrl_.collapse();
        } else {
            this.ctrl_.collapseAll();
        }
    },*/
    /** The method will expand one or all code block(s).
     * @param
     * all - If calling without this parameter, the method will expand the code block with cursor.
     *        If calling with this parameter, it will expand all code blocks.
     * @since 2.0 or 3.0
     * @attention: Not available currently.
     */
    /*expand : function (all) {        
        this.ctrl_.focus();
        if (!all) {
            this.ctrl_.expand();
        } else {
            this.ctrl_.expandAll();
        }
    },*/
    /**
     * The method sets the size of the editor object.
     * @param 
     * {integer}width - The new width of the editor object.
     * @param
     * {integer}height- The new height of the editor object.
     * 
     */
    resize : function (width, height) {
        // this.ctrl_.focus();
        this.ctrl_.resize.apply(this.ctrl_, arguments);
    },
    /**
     * The method is used to set folding icon or get folding icon.
     * If icons is not null, the method will set this options.
     * @param
     * {object}icons - This parameter is an object like {expand: <path>, collapse: <path>}.
     * @return
     * The folding icons in the editor object.
     * @since 2.0 or 3.0
     * @attention: Not available currently.
     */
    /*foldingIcons : function (icons) {        
        return SAS.settings_.getInstance().foldingIcons(icons);
    },*/
    /** 
     * The method is used to display the GoToLineDialog.
     */
    showGoToLineDialog : function () {
        this.ctrl_.showGoToLineDialog();
    },
    /**
     * The method is used to display the FindAndReplaceDialog.
     * 
     */
    showFindReplaceDialog : function () {
        this.ctrl_.showFindReplaceDialog();
    },
    /**
     * The method is used to hide the FindAndReplaceDialog
     */
    hideFindReplaceDialog : function () {
        this.ctrl_.hideFindReplaceDialog();
    },
	/**
     * The method is used to hide the GoToLineDialog
     */
    hideGoToLineDialog : function () {
        this.ctrl_.hideGoToLineDialog();
    },
    /**
     * Set the next focus handler so the consumer can control the tabbing.
     */
    setNextFocusHandler : function (nextFunction) {
        this.ctrl_.nextFunction = nextFunction;
    },
    /**
     * Set the previous focus handler so the consumer can control the tabbing.
     */
    setPreviousFocusHandler : function (prevFunction) {
        this.ctrl_.prevFunction = prevFunction;
    },
    /**
     * This method gets all or part HTML color-coded version of the SAS code in the editor object.
     * 
     * @param
     * {object}option - The parameter option is an object like 
     * {
     *     lineNumber: true,
     *     onlyVisible: false
     * }.
     * That lineNumber equals true indicates the returned HTML includes line numbers.
     * That onlyVisible equals true indicates the returned HTML only includes the visible part.
     * @return
     * The HTML string that is color-coded SAS code.
     */
    getHTML : function (option) {
        return this.ctrl_.getHTML(option);
    },
    /**
     * Change the prompt text of Editor,
     * and it can be used to get the prompt text.
     * @param
     * {string}text - If calling with this parameter, the function will change the prompt text to text and return new value.
     *        If calling without this parameter, it returns the current text.
     * @return
     * Return the prompt text.
     */
    promptText : function (text) {
        // this.ctrl_.focus();
        return this.settings_.promptText.apply(this.settings_, arguments);
    },
    /**
     * Get the context menu object owned by the editor object.
     * @return
     * The context menu object which the user can customize. 
     */
    getContextMenu: function() {
        return this.ctrl_.getContextMenu();
    },
    
    setLibService: function(fn) {
        this.ctrl_.setLibService(fn);
    }
};
/**
 * @class SAS.Editor.LanguageMode
 * The literal object defines the language mode supported by Editor, 
 * but Editor only supports SasCode now.
 */
SAS.Editor.LanguageMode = {
    Unset: 'none',
    Text: 'text',
    Html: 'html',
    SclCode: 'scl',
    SasCode: 'sas',
    Xml: 'xml',
    Log: 'log',
    Lst: 'lst',
    Mdx: 'mdx',
    R: 'r'
};

   return SAS.Editor;
});
