/*! Copyright (c) 2005-2008 SAS Institute, Inc.  All rights reserved. */

/* ------------------------------------------------------------------------ *
 * Purpose: Contains javascript functionality for some common ajax features *
 *                                                                          *
 * Author:                                                                  *
 * Version:  1.0                                                            *
 *                                                                          *
 * ------------------------------------------------------------------------ */

sas.provides("sas_ajax");
dojo.require("dojo.io");

if (typeof sas == "undefined") 
{
	sas={};	
}

var csrfHeader = null;
var csrfToken = null;

if (typeof sas.ajax == "undefined") 
{
	sas.ajax = {
			inuse: false,
			/**
			 * when more than one ajax request is in que, wait until the first one finish loading.
			 */
			requestPool: new Array,

			/**
			 * Makes initial request to set the csrf header variables
			 */
			getCsrfToken: function(xhr) {
				if (xhr && xhr.getResponseHeader) {
					var CSRFHeader = xhr.getResponseHeader("X-CSRF-HEADER");
					var CSRFToken = xhr.getResponseHeader("X-CSRF-TOKEN");
					if (CSRFHeader) {
						csrfHeader = CSRFHeader;
					}
					if (CSRFToken) {
						csrfToken = CSRFToken;
					}
				}
			},

			/**
			 * Makes an XMLHttpRequest using dojo's io.bind() function.
			 * @param {Object} url
			 * @param {Object} requestParms
			 * @param {Object} callback
			 * @param {Object} errorHandler
			 * @param {Object} sync
			 * @param {Object} userData
			 */
			send: function(url, requestParms, callback, errorHandler, sync, userData) {
				url = sas_addRequestParams(url, requestParms);
				var qmIndex = (url!=null) ? url.indexOf("?") : -1;
				
				var reqOptions = {
					url:  (qmIndex<0) ? url : url.substr(0,qmIndex),
					mimetype:  "text/html",
					transport: "XMLHTTPTransport",
					method: "post",
					postContent: (qmIndex<0) ? null : url.substr(qmIndex+1),
					error:     errorHandler || sas.ajax.handleRequestErrors,
					load: callback || sas.ajax.defaultHandler,
					userData: userData,
					sync: sync || false
				};

				// add csrf token header if set
				if (csrfHeader && csrfToken) {
					reqOptions.headers = {
							"X-CSRF-HEADER":csrfHeader,
							"X-CSRF-TOKEN":csrfToken
					}
				}

				if (typeof window.sas_framework_updateTimeout != "undefined")
					window.sas_framework_updateTimeout();
				dojo.io.bind(reqOptions);
			},
			handleRequestErrors: function(type, data, xhr) {
				var httpStatus = xhr.status;
				var httpMsg    = xhr.statusText;
				if (httpStatus != 200) {
					if (!csrfToken && httpStatus == 403 && xhr.responseText.includes("invalid CSRF Token")) {
						sas.ajax.getCsrfToken(xhr);
						//retry call
						sas.ajax.send(this.postContent?this.url+"?"+this.postContent:this.url, null, this.load, this.error, this.sync, this.userData);
					}
					else {
						alert(httpStatus + " " + httpMsg);
					}
				}
				else {
					alert("An error has occured.");
				}
				//sas_Log.error(xhr.responseText);
				sas.ajax.inuse = false;
				if (window['fastClickManager']) window['fastClickManager'].unprotectPage();
			},
			/**
			 * The default AJAX handler for updating the DOM and evaluating the scripts.
			 * @param {Object} str
			 * @param {Object} parms
			 */
			defaultHandler: function(str, parms) {
				var element = document.getElementById(parms.containerId);
				if ( element )
					sas.ajax.replaceInnerHTML(element, str);
			},
			/**
			 * The default AJAX handler for updating the DOM and evaluating the scripts
			 * and then closing the current window.
			 * @param {Object} str
			 * @param {Object} parms
			 */
			defaultCloseWindowHandler: function(str, parms) {
				sas.ajax.defaultHandler(str, parms);
    			if( window.selectorDialog )
    				window.selectorDialog.close();
			},			
			/**
			 * Replaces the contents of a given element with a new string.
			 * @param {Object} element
			 * @param {Object} str
			 */
			replaceInnerHTML: function(element, str) {
				str = "<span>Loading</span>" + str;	
    			element.innerHTML = str;
    			element.removeChild(element.firstChild);
    			sas.ajax.evalScripts(element);
				if (window.sas_init_dialog){
					sas_init_dialog();
				}
			},
			/**
			 * Evaluate the scripts
			 * @param {Object} element
			 */
			evalScripts: function(element) {
				sas_Log.debug("Eval Scripts start");
				sas.ajax.includeScripts(element);
				sas.ajax.evalScriptContent(element);
			},
			
			/**
			 * For a given element take care of evaluating all of its
			 * inline javascript blocks.
			 * @param {Object} element
			 */
			evalScriptContent: function(element) {
				sas_Log.debug("Eval Script Content");
				var scriptContent = sas.ajax.getScriptContent(element);
				if(scriptContent!=null && scriptContent!="")
				{
            		try 
            		{
                		with (window) // to handle global variables
                   			eval(scriptContent);
            		}
           		 	catch (e) 
            		{
            			sas_Log.warning("failed evaling script content: " + e.message ); // + " > " + scriptContent); 
            			alert(e.message);
            		}
    			}
    			sas_Log.debug("Eval Script Content complete");
			},
			/**
			 * 
			 * @param {Object} url
			 * @param {Object} requestParms
			 * @param {Object} handler
			 * @param {Object} postprocess
			 */
			process: function(url, requestParms, handler, postprocess) {
	   			sas_Log.debug( "starting ajax processing" );
	   			if (sas.ajax.inuse) {
					sas.ajax.requestPool.push({url: url, requestParms: requestParms, handler: handler});
					return false;
				}
    			var delegateToHandler = function(type, data, xhr) {
    					sas.ajax.inuse = false;
        				sas.ajax.handleResponse(handler, postprocess, type, data, xhr);
    			};
				sas.ajax.inuse = true;
				
				sas.ajax.send(url, requestParms, delegateToHandler);
			}, 
			/**
			 *  Handle the response to a XMLHttpRequest.
			 *  This will eval the reponseText and then call sas.ajax.updateViewers().
			 *  The responseText should define sas.ajax.updateViewers().
			 * 	@param {Object} handler
			 * 	@param {Object} postprocess
			 */
			handleResponse: function(handler, postprocess, type, data, xhr) {
				var sas_AJAX_update = handler;
				var str = xhr.responseText;
				if (sas_checkSessionExpired(str)) return;
				eval(str);
        		sas.ajax.updateViewers(); // should be defined in the responseText
        		if (postprocess) postprocess();
				if (sas.ajax.requestPool.length > 0) {
					var poolObj = sas.ajax.requestPool.shift();
					sas.ajax.process(poolObj.url, poolObj.requestParms, poolObj.handler);
				}
				sas_Log.debug( "done ajax processing" );
			},
			/**
			 * Default implementation of function called by sas_AJAX_handleResponse.
			 * Eval on the responseText will normally define a more useful method.
			 */
			updateViewers: function() {
			},
			/**
			 * Swaps a node
			 * @param {Object} node
			 * @param {Object} str
			 */
			swapTableCell: function(node, str) {
    			if (node==null)
        			return;
                // create a table and stick in the new html
    			var div = document.createElement("div");
    			div.innerHTML = "<table><tr>" + str + "</tr></table>";
                // get the generated cell
    			var row = div.firstChild.rows[0];
                return this._replaceNode( node, row.firstChild );
            },
            _replaceNode: function( node, generatedNode )
            {
                // clone the node (without children)
                // On Firefox, replacing a node may cause the scripts to
                // get evaluated, so we'll copy the contents seperately
                var newNode = generatedNode.cloneNode(false);
                node.parentNode.replaceChild(newNode, node);
                // and copy over the html
                newNode.innerHTML = generatedNode.innerHTML;
                // on IE the innerHTML on the newNode may be missing some
                // leading non visual nodes (eg scripts), so use original
                sas.ajax.evalScripts(generatedNode);
                return newNode;
			},
			swapNode: function(node, str) {
    			if (node==null)
        			return;
				
    			var div = document.createElement("div");
    			div.innerHTML = str;
                return this._replaceNode( node, div.firstChild );
			},
			/**
			 * Include the scripts contained within the specified element.
			 * @param {Object} element
			 */
			includeScripts: function(element) {
				if (!element) return;
    			var scripts = element.getElementsByTagName("script");
    			for (var i=0; i<scripts.length; i++)
    			{
        			var script = scripts[i];
        			if (script.innerHTML.indexOf("sas.require") > -1) {
				        with(window) {
    	    				eval(script.innerHTML);
        				}
        			}
        			else {
         				if ( script.innerHTML.indexOf("sas_includeDojo") > 0)
    	     				with (window)
        	 			eval(script.innerHTML);
        			}
    			}
			},
			
			/**
			 * For a given element find all inline javascript block
			 * and aggregate them together.
			 * This should not include script src= type script blocks
			 * @param {Object} element
			 */
			getScriptContent: function(element) {
				if (!element) return;
				var scriptsForEval = [];
    			// get script tags
    			var scripts = element.getElementsByTagName("script");
    			for (var i=0; i<scripts.length; i++)
    			{
        			var script = scripts[i];
        			if (script.innerHTML.indexOf("sas.require") == -1) {
	        			if (script.innerHTML && script.innerHTML!="")
    	    			{
    						//scriptContent=scriptContent+script.innerHTML;
	        			    scriptsForEval.push(script.innerHTML);
        				}
        			}
    			}
				//alert(scriptContent);
                var scriptContent=scriptsForEval.join("\n");
	    		return scriptContent;
			},
			
			doAction: function(url, closeWindow) {
				if ( closeWindow )
					sas.ajax.process(url,null,sas.ajax.defaultCloseWindowHandler);
				else
				sas.ajax.process(url,null,sas.ajax.defaultHandler);
			},
            
            cleanupListeners: {},
            
            addCleanupListener: function(id, listener)
            {
                var cleanup = sas.ajax.cleanupListeners;
                var info = cleanup[id];
                if (!info)
                {
                    info = cleanup[id] = {id:id, listeners: []};
                }
                info.listeners.push(listener);
            },
            cleanup: function(id)
            {
                sas.ajax._cleanup(id);
                var cleanup = sas.ajax.cleanupListeners;
                var el = dojo.byId(id);
                for (var cid in cleanup)
                {
                    var cEl = document.getElementById(cid);
                    if (sas_DOMContains(el, cEl))
                        sas.ajax._cleanup(cid);
                }
            },
            _cleanup: function(id)
            {
                if (typeof id != "string") return;
                var cleanup = sas.ajax.cleanupListeners;
                if (cleanup[id])
                {
                    var info = cleanup[id];
                    var listeners = info.listeners;
                    for (var i=0; i<listeners.length; i++)
                    {
                        var f = listeners[i];
                        f();
                    }
                    delete cleanup[id];
                }
            }
	}
}

