/*
 Copyright (C) 2015 SAS Institute Inc. Cary, NC, USA
*/

/**
   \file
   
   \brief   Upload a file inside a jobflow instance
   
   \param [in] host Host url, including the protocol
   \param [in] LogonPort (Optional) port where the /SASLogon web application is listening. If blank, it is assumed that the /SASLogon is on the same port as /SASIRMServer (Default: blank)
   \param [in] server Name of the Web Application Server that provides the REST service (Default: SASIRMServer)
   \param [in] port Server port (if blank, default port 80 will be used). Default: 7980
   \param [in] username Username credentials
   \param [in] password Password credentials: it can be plain text or SAS-Encoded (it will be masked during execution). 
   \param [in] authMethod: Authentication method (accepted values: TOKEN/SSO). Used when Username/Password credentials are left blank to determine the authentication method. (Default: TOKEN). 
   \param [in] topInstanceKey Instance key of the top flow
   \param [in] dataKey Data object unique key
   \param [in] filePath Full path of the SAS dataset file on server to be loaded
   \param [in] comment Comment attached to the file
   \param [in] debug True/False. If True, debugging informations are printed to the log (Default: false)
   \param [in] logOptions Logging options (i.e. mprint mlogic symbolgen ...)
   \param[in]  http_status_code_var_name     the name of global macro variable whose value will be set to the HTTP status code
   \param[in]  server_msg_var_name     the name of global macro variable whose value will be set to the server message of data upload
   
      
   \details 
   This macro sends a POST request to <b><i>\<host\>:\<port\>/SASIRMServer/rest/jobflow/instance/&topInstanceKey./import/local/data/&dataKey.</i></b> in order to upload a file into the specified instance.
   
   \author  SAS Institute Inc.
   \date    2018
*/
%macro irm_rest_load_local_sasdataset(host =
                            , LogonPort =
                            , server = SASIRMServer
                            , port = 7980
                            , username =
                            , password =
                            , authMethod = token /* token/sso */
                            , topInstanceKey =
                            , dataKey     =
                            , filePath     =
                            , comment     = %str(Input was modified by user)
                            , debug       = false
                            , logOptions =
                            , http_status_var_name =http_status_var_nm
                            , server_msg_var_name =server_msg_var_nm
                            );
   

   /* Set the required log options */
   %if(%length(&logOptions.)) %then
      options &logOptions.;
   ;

   /* Check if username and password have been provided. */
   %local without_credentials_flg;
   %let without_credentials_flg = N;
   %if(%sysevalf("%superq(username)"="", boolean) or %sysevalf("%superq(password)"="", boolean)) %then
      %let without_credentials_flg = Y;

   /* Get the current value of mlogic and symbolgen options */
   %local oldLogOptions;
   %let oldLogOptions = %sysfunc(getoption(mlogic)) %sysfunc(getoption(symbolgen));

   %if(%length(&port.) = 0) %then
      %let port = 80;
   
   %let url = %str(&host.:&port./&server./rest/jobflow/instance/&topInstanceKey./import/local/data/&dataKey./);

   /* Get a Service ticket in the following cases:
      - Credentials have been provided => authenticate with the credentials
      - The system is configured to use Single Sign-On (authMethod = sso) => use proc http AUTO_NEGOTIATE option
   */
   %if(&without_credentials_flg. = N or %upcase(&authMethod.) = SSO) %then %do;
      /* Temporary disable mlogic and symbolgen options to avoid printing of userid/pwd to the log */
      option nomlogic nosymbolgen;
      
      /* Get the Service Ticket. Returns service ticket in the macro variable __irm_rest_service_ticket__ */
      %irm_rest_service_ticket(url     = &url.
                        , appcontext   = &server.
                        , irm_username = &username.
                        , irm_password = &password.
                        , debug        = &debug.
                        , logOptions   = &oldLogOptions.
                           );
   %end;

   %put NOTE: service ticket is &__irm_rest_service_ticket__.;
   
   /* Build the JSON body for the POST request */
   filename json_run temp;
   %put NOTE: Creating JSON payload for a loading local file;
   proc json out = json_run pretty;
      write open o ; 
         write values "localDataPath" "&filePath.";
         write values "comment" "&comment.";
      write close;
   run;
   
   filename out_run temp;
   filename headers temp;

   %put NOTE: Load sasdata set from server for instance via PROC HTTP;
   proc http
      in = json_run
      headerout=headers
      out = out_run
      method = 'post'
      ct = 'application/json'
      %if(&without_credentials_flg. = Y and %upcase(&authMethod.) = TOKEN) %then %do;
         /* No credentials -> use Token authentication (this is only allowed if running the code from an authenticated SAS Session) */
         url = "%str(&url.)"
         http_tokenauth
      %end;
      %else %do;
         /* Use the ticket */
         url = "%str(&url.)%nrstr(?&ticket=)&__irm_rest_service_ticket__."
      %end;
      /* Clear connection cache */
      clear_cache
      HEADEROUT_OVERWRITE
      ;
   run;

   %global HTTP_STATUS_CODE;
   %irm_parse_http_header(
                                header_ref                  =headers,
                                http_status_code_var_name =HTTP_STATUS_CODE);

    %global &http_status_var_name.;
    %global &server_msg_var_name.;
   
    data _null_;
        call symput("&http_status_var_name", trim("&HTTP_STATUS_CODE"));
    run;

      data _null_;
         infile out_run;
         input;
         if _N_ = 1 then
            put "Response from the server:";
         put _infile_;
         call symput("&server_msg_var_name",   _infile_);
      run;
   
   
   /* Clear filenames */
   filename json_run clear;
   filename headers clear; 
   filename out_run clear; 
 
   
%mend;                            
