/*
 Copyright (C) 2018 SAS Institute Inc. Cary, NC, USA
*/

/**
   \file
   
   \brief   Send HTTP request to the specified REST API endpoint and parse the result
   
   \param [in] url Full url to the REST resource 
   \param [in] method The HTTP request method: GET/PUT/POST/DELETE. (Default: GET)
   \param [in] LogonPort (Optional) port where the /SASLogon web application is listening. If blank, it is assumed that the /SASLogon is on the same port as /SASIRMServer (Default: blank)
   \param [in] server Name of the Web Application Server that provides the REST service (Default: SASIRMServer)
   \param [in] username Username credentials
   \param [in] password Password credentials: it can be plain text or SAS-Encoded (it will be masked during execution). 
   \param [in] authMethod: Authentication method (accepted values: TOKEN/SSO). Used when Username/Password credentials are left blank to determine the authentication method. (Default: TOKEN). 
   \param [in] tgt_ticket Ticket granting ticket. If provided, this TGT ticket is used to obtain a Service ticket for the requested REST resource
   \param [in] headerIn Optional. Request header: this can be either a fileref or a string.
   \param [in] headerOut Optional. Response header fileref. (Default: __hout_)
   \param [in] body Optional. Request body: this can be either a fileref or a string.
   \param [in] fout Optional. Fileref for the reponse body. A temporary fileref is created is missing
   \param [in] contentType Optional. Request content type. (Default: application/json)
   \param [in] parser Name of the LUA parser function responsible for converting the JSON response into the output SAS table (Default: irmRestPlain -> plain one-level JSON structure). See \link irm_rest_parser.lua \endlink for details).
   \param [in] debug True/False. If True, debugging informations are printed to the log (Default: false)
   \param [in] logOptions Logging options (i.e. mprint mlogic symbolgen ...)
   \param [in] restartLUA. Flag (Y/N). Resets the state of Lua code submission for a SAS session if set to Y (Default: Y)
   \param [in] clearCache Flag (Y/N). Controls whether the connection cache is cleared across multiple proc http calls. (Default: Y)
   \param [in] printResponse Flag (Y/N). Controls whether the Reponse body is printed to the log when no loa parser is provided (Default: Y)
   \param [in] arg1 Additional parameter passed to the LUA parser function
   \param [in] arg2 Additional parameter passed to the LUA parser function
   \param [in] arg3 Additional parameter passed to the LUA parser function
   \param [in] arg4 Additional parameter passed to the LUA parser function
   \param [in] arg5 Additional parameter passed to the LUA parser function
   \param [in] arg6 Additional parameter passed to the LUA parser function
   \param [in] arg7 Additional parameter passed to the LUA parser function
   \param [in] arg8 Additional parameter passed to the LUA parser function
   \param [out] outds Name of the output table
   \param [out] outVarTicket Name of the output macro variable which will contain the Service Ticket (Default: ticket)
   \param [out] outSuccess Name of the output macro variable that indicates if the request was successful (&outSuccess = 1) or not (&outSuccess = 0). (Default: httpSuccess)
   \param [out] outResponseStatus Name of the output macro variable containing the HTTP response header status: i.e. HTTP/1.1 200 OK. (Default: responseStatus)
   
   \details 
   This macro performs the following operations:
      1. Get a Service Ticket for the requested URL (see \link irm_rest_get_ticket.sas \endlink for details)
         1.a the ticket is created only if the macro variable referenced by outVarTicket is blank (i.e. -> <i> if &&&outVarTicket = %str() </i>)
      2. Send Http request (GET/POST/PUT/...) to the specified URL
      3. Parse the JSON response using the specified parser function (only if the parser is provided)
      4. Return the parsed response into a SAS table. 
      

   <b>Example:</b>
   
   1) Set up the environment (set SASAUTOS and required LUA libraries)
   \code
      %let source_path = <Path to the root folder of the Federated Content Area (root folder, excluding the Federated Content folder)>;
      %let fa_id = <Name of the Federated Area Content folder>;
      %include "&source_path./&fa_id./source/sas/ucmacros/irm_setup.sas";
      %irm_setup(source_path = &source_path.
                , fa_id = &fa_id.
                );
   \endcode
   
   2) Send a Http GET request and parse the JSON response into the output table WORK.result using the default parser irmRestPlain
   \code
      %let ticket =;
      %irm_rest_request(url = <host>:<port>/SASIRMServer/<resource>
                        , method = GET
                        , server = SASIRMServer
                        , username = <userid>
                        , password = <pwd>
                        , headerIn = Accept:application/json
                        , body = {"param1":"value1", "param2":"value2", ...}
                        , parser = irmRestPlain
                        , outds = WORK.result
                        , outVarTicket = ticket
                        , outSuccess = httpSuccess
                        , outResponseStatus = responseStatus
                        );
      %put &=ticket;                        
      %put &=httpSuccess;                        
      %put &=responseStatus;
   \endcode
         
   \note The structure of the output table depends on the JSON content and the parser function
         
   
   \ingroup irmRestUtils
   
   \author  SAS Institute Inc.
   \date    2018
*/

%macro irm_rest_request(url =
                        , method = GET
                        , LogonPort =
                        , server = SASIRMServer
                        , username =
                        , password =
                        , authMethod = token /* token/sso */
                        , tgt_ticket =
                        , headerIn =
                        , headerOut = __hout_
                        , body =
                        , fout =
                        , contentType = application/json
                        , parser = irmRestPlain
                        , outds = 
                        , outVarTicket = ticket
                        , outSuccess = httpSuccess
                        , outResponseStatus = responseStatus
                        , debug = false
                        , logOptions =
                        , restartLUA = Y
                        , clearCache = Y
                        , printResponse = Y
                        , logSeverity = ERROR
                        , arg1 =
                        , arg2 =
                        , arg3 =
                        , arg4 =
                        , arg5 =
                        , arg6 =
                        , arg7 =
                        , arg8 =
                        );

   /* Set the required log options */
   %if(%length(&logOptions.)) %then
      options &logOptions.;
   ;

   /* Check if username and password have been provided. */
   %local without_credentials_flg;
   %let without_credentials_flg = N;
   %if(%sysevalf("%superq(username)"="", boolean) or "%superq(password)" = "") %then
      %let without_credentials_flg = Y;

   /* Get the current value of mlogic and symbolgen options */
   %local oldLogOptions;
   %let oldLogOptions = %sysfunc(getoption(mlogic)) %sysfunc(getoption(symbolgen));

   /* Delete output table if it exists */
   %if %sysevalf(%superq(outds)^=,boolean) %then %do;
       %if (%rsk_dsexist(&outds.)) %then %do;
          proc sql;
             drop table &outds.;
          quit;
       %end;
   %end;

   /* ************************************************ */
   /* Process the outSuccess parameter                 */
   /* ************************************************ */

   /* OutSuccess cannot be missing. Set a default value */
   %if(%sysevalf(%superq(outSuccess) =, boolean)) %then
      %let outSuccess = httpSuccess;
   
   /* Declare the output variable as global if it does not exist */
   %if(not %symexist(&outSuccess.)) %then
      %global &outSuccess.;


   /* ************************************************ */
   /* Process the OutResponseStatus parameter          */
   /* ************************************************ */

   /* OutResponseStatus cannot be missing. Set a default value */
   %if(%sysevalf(%superq(outResponseStatus) =, boolean)) %then
      %let outResponseStatus = responseStatus;
   
   /* Declare the output variable as global if it does not exist */
   %if(not %symexist(&outResponseStatus.)) %then
      %global &outResponseStatus.;


   /* ************************************************ */
   /* Process the OutVarTicket parameter               */
   /* ************************************************ */
   
   /* OutVarTicket cannot be missing. Set a default value */
   %if(%sysevalf(%superq(outVarTicket) =, boolean)) %then
      %let outVarTicket = ticket;
   
   /* Declare the output variable as global if it does not exist */
   %if(not %symexist(&outVarTicket.)) %then
      %global &outVarTicket.;


   /* ************************************************ */
   /* Get a Service Ticket                             */
   /* ************************************************ */
      
   /* Get a Service ticket in the following cases:
      - a ticket is not available (outVarTicket is blank) AND credentials have been provided => authenticate with the credentials
      - a ticket is not available (outVarTicket is blank) AND the system is configured to use Single Sign-On (authMethod = sso) => use proc http AUTO_NEGOTIATE option
      - a ticket is not available (outVarTicket is blank) AND the a TGT ticket has been provided => Use the TGT ticket request a service ticket
   */
   %if(%length(&&&outVarTicket.) = 0 and (&without_credentials_flg. = N or %upcase(&authMethod.) = SSO or %sysevalf(%superq(tgt_ticket) ne, boolean))) %then %do;
      /* Temporary disable mlogic and symbolgen options to avoid printing of userid/pwd to the log */
      option nomlogic nosymbolgen;
      /* Get the Service Ticket */
      %irm_rest_get_ticket(url = &url.
                           , LogonPort = &LogonPort.
                           , server = &server.
                           , username = &username.
                           , password = &password.
                           , ticketType = ST
                           , tgt_ticket = &tgt_ticket.
                           , outVarTicket = &outVarTicket.
                           , debug = &debug.
                           , logOptions = &oldLogOptions.
                           );
   %end;

   
   /* ************************************************ */
   /* Process the headerIn parameter                   */
   /* ************************************************ */
   %local hin_fref_flg;
   %let hin_fref_flg = N;
   %if(%sysevalf(%superq(headerIn) ne, boolean)) %then %do;
      /* Check if the headerIn is a fileref */
      %if(%length(%superq(headerIn)) <= 8) %then %do;
         %if(%sysfunc(fileref(&headerIn.)) ne 0) %then
            /* It is not a fileref. Will need to create a temporary file */
            %let hin_fref_flg = Y;
      %end;
      %else
         %let hin_fref_flg = Y;
   %end;
   
   /* Create a temporary fileref and write the header content in it */
   %if (&hin_fref_flg. = Y) %then %do;
      filename __hin__ temp;
      data _null_;
         file __hin__;
         put "%sysfunc(prxchange(s/[""]/""/i, -1, %superq(headerIn)))";
      run;

      %let headerIn = __hin__;
   %end;
   
   
   /* ************************************************ */
   /* Process the body parameter                       */
   /* ************************************************ */
   %if(%sysevalf(%superq(body) ne, boolean)) %then %do;
      /* Check if the body is a fileref */
      %if(%length(%superq(body)) <= 8) %then %do;
         %if(%sysfunc(fileref(&body.)) ne 0) %then %do;
            /* It is not a fileref. Although short, it must be a string. Convert " to "" and enclose the entire string within double quotes.*/
            %let body = "%sysfunc(prxchange(s/[""]/""/i, -1, %superq(body)))";
         %end;
      %end;
      %else
         /* Convert " to "" and enclose the entire string within double quotes. */
         %let body = "%sysfunc(prxchange(s/[""]/""/i, -1, %superq(body)))";
   %end;  

   
   /* ************************************************ */
   /* Process the headerOut parameter                  */
   /* ************************************************ */
   %if(%sysevalf(%superq(headerOut) =, boolean)) %then
      %let headerOut = __hout_;
   
   /* Assign the HeaderOut fileref if it does not exist */
   %if(%sysfunc(fileref(&headerOut.)) ne 0) %then %do;
      filename &headerOut. temp;
   %end;
   
   /* Make sure the fout parameter is set */
   %let out_fref_flg = N;
   %if(%sysevalf(%superq(fout) =, boolean)) %then %do;
      %let fout = __out__;
      %let out_fref_flg = Y;
   %end;
   
   /* Check the fout parameter */
   %if(%sysevalf(%superq(fout) ne, boolean)) %then %do;
      /* Assign the fout fileref if it does not exist */
      %if(%sysfunc(fileref(&fout.)) ne 0) %then %do;
         filename &fout. temp;
      %end;
   %end;

   /* Check if the url contains any question mark */
   %local question_mark;
   %if(%index(&url.,?) = 0) %then
      %let question_mark = ?;
   
   /* Print the request URL to the log */
   %if (%upcase(&debug.) eq TRUE) %then %do;
      %put ------------------------------------------------;
      %if(&without_credentials_flg. = Y and %upcase(&authMethod.) = TOKEN) %then 
         %put Request URL: %str(&url.);
      %else
         %put Request URL: %str(&url.)&question_mark.%nrstr(&ticket=)&&&outVarTicket.;
      %put ------------------------------------------------;
   %end;
   
   
   /* ************************************************ */
   /* Send HTTP request                                */
   /* ************************************************ */
   proc http
      /* Request header */
      %if(%sysevalf(%superq(headerIn) ne, boolean)) %then %do;
         headerin = &headerIn.
      %end;
      /* Request body */
      %if(%sysevalf(%superq(body) ne, boolean)) %then %do;
         in = &body.
      %end;
      /* Response header */
      headerout = &headerOut.
      /* Response body */
      %if(%sysevalf(%superq(fout) ne, boolean)) %then %do;
         out = &fout.
      %end;
      method = "%upcase(&method.)"
      %if(%sysevalf(%superq(contentType) ne, boolean)) %then %do;
         ct = "&contentType."
      %end;
      %if(&without_credentials_flg. = Y and %upcase(&authMethod.) = TOKEN and %sysevalf(%superq(&outVarTicket) =, boolean)) %then %do;
         /* No credentials -> use Token authentication (this is only allowed if running the code from an authenticated SAS Session) */
         url = "%str(&url.)"
         http_tokenauth
      %end;
      %else %do;
         /* Use the ticket */
         url = "%str(&url.)&question_mark.%nrstr(&ticket=)&&&outVarTicket."
      %end;
      %if(%upcase(&clearCache.) = Y) %then %do;
         /* Clear connection cache */
         clear_cache
      %end;
      ;
   run;

   
   /* Check the response header for errors */
   %let &outSuccess. = 0;
   data _null_;
      infile &headerOut.;
      /* Read the response header */
      input;
      if _N_ = 1;
      /* Extract the response code */
      response_code = input(scan(_infile_, 2, " "), 8.);
      /* Check whether the HTTP return code is in the 200's (200 OK, 201 CREATED, etc) */
      success = response_code >= 200 and response_code < 300;
      call symput("&outSuccess.", strip(success));
      call symput("&outResponseStatus.", strip(_infile_));
      stop;
   run;
   
   /* Stop macro execution if there were any errors */
   %if(&&&outSuccess.. = 0) %then %do;
      %put &logSeverity.: The server response returned &&&outResponseStatus...;
      /* Avoid parsing the response. We will just return the response body in the output table as-is. */
      %let parser =;
   %end;
   
   /* Parse the request if a JSON parser is provided */
   %if(%sysevalf(%superq(parser) ne, boolean)) %then %do;
      /* Declare the PRODUCT macro variable, required in sas.risk.rmx.rsk_init @ line # 24 */
      %local 
         product
         entrypoint_module
         entrypoint_function
      ;
      %let product = IRM;
      %let entrypoint_module = %sysfunc(prxchange(s/(^|(((\.?\w+)+)\.))(\w+)$/$3/i, -1, &parser.));
      %let entrypoint_function = %sysfunc(prxchange(s/(^|(((\.?\w+)+)\.))(\w+)$/$5/i, -1, &parser.));
      /* Assign default parser module sas.risk.irm.irm_rest_parser if not specifically provided */
      %if(%sysevalf(%superq(entrypoint_module) =, boolean)) %then
         %let entrypoint_module = sas.risk.irm.irm_rest_parser;
      
      %put NOTE: Parsing response using Lua module &entrypoint_module..&entrypoint_function.;
      
      /* Parse the JSON file and write the result in the output table */
      %irm_call_riskexec(entrypoint_module         =   &entrypoint_module.
                         , entrypoint_function     =   &entrypoint_function.
                         , restartLUA              =   &restartLUA.
                         , arg1                    =   __out__
                         , arg2                    =   &outds.
                         , arg3                    =   &arg1.
                         , arg4                    =   &arg2.
                         , arg5                    =   &arg3.
                         , arg6                    =   &arg4.
                         , arg7                    =   &arg5.
                         , arg8                    =   &arg6.
                         , arg9                    =   &arg7.
                         , arg10                   =   &arg8.
                         );
   %end;
   %else %do;
      /* Check if we got any response */
      %if(%sysfunc(fileref(&fout.)) = 0) %then %do;
         /* Read the server response AS-IS and return it into a table */
         data 
            %if (%sysevalf(%superq(outds)^=, boolean)) %then
               &outds.;
            %else
               _null_;
            ;
            length response $32000.;
            infile &fout. lrecl = 32000;
            input;
            response = strip(_infile_);
            %if(&&&outSuccess.. = 0 or (%sysevalf(%superq(outds)=, boolean) and &printResponse. = Y)) %then %do;
               if _N_ = 1 then
                  put "Response from the Server:"; 
               put response;
            %end;
         run;
      %end;
   %end;
   
   /* Cleanup */
   %if(%upcase(&debug.) ne TRUE) %then %do;
      %if (&hin_fref_flg. = Y) %then %do;
         filename __hin__ clear;
      %end;
      %if(&out_fref_flg. = Y) %then %do;
         filename __out__ clear;
      %end;
   %end;
   
%mend;
