/*
 Copyright (C) 2018 SAS Institute Inc. Cary, NC, USA
*/

/**
   \file
   
   \brief  This macro installs (adds) a Federated Area (FA) on the SAS Server and makes the new FA available for use without needing a server restart.
           The macro may be executed in an interactive session such as SAS Studio or a background process.

   <b> Identified Inputs </b> 

   \param [in] meta_host                  Complete name of the server running the SAS Metadata Server. 
   \param [in] meta_port                  Port on which the SAS Metadata Server is listening. Defaults to 8561.
   \param [in] meta_repos                 Name of the metadata repository. Defaults to Foundation.
   \param [in] username                   Username credentials for logging into IRM (not required if logged-in to SAS Studio).
   \param [in] passwd                     Password credentials: it can be plain text or SAS-Encoded (will be masked during execution). Not required if logged-in to SAS Studio.
   \param [in] fa_src_path                Complete path (including zip file name) of the source location where the FA zip exists.
   \param [in] fa_tgt_path                Complete path to the target location where the FA will be installed. Should not be the same as the source location.
   \param [in] fa_id       [OPTIONAL]     User provided name for the FA ID. If provided, becomes the new FA ID. Example: fa_id = com.sas.solutions.risk.irm.fa.machine_learning
   \param [in] force_flg   [OPTIONAL]     Flag (True/False) forcing/canceling install respectively if dependencies are not met or FA is invalid. Also, forces use of fa_id, if provided. 
                                          Defaults to false.
   \param [in] debug       [OPTIONAL]     Amount of execution information in the log. Defaults to FALSE.
   \param [in] logOptions  [OPTIONAL]     Amount of macro execution information in the log. Defaults to blank. Accepts a space separated list of supported SAS options.
                                          Example: logOptions = source mprint mlogic symbolgen
   \param [in] connection_type [OPTIONAL] Type of connection: INTERNAL or EXTERNAL (in case the connection is redirected). Default: INTERNAL.  
   
   \details 
            PURPOSE: The macro provides an interface to add a new Federated Area on the server and make the FA available for immediate use (no server restart).
            
            PRE-REQUISITES:
                     1. The FA to be installed is available at an accessible location on the server.
 
            USAGE:   Sample values have been provided for some parameters to illustrate usage.
                     %irm_install_fa(
                              meta_host=<host; metadata server>
                           ,  meta_port=<8561; optional for default metadata server port>
                           ,  meta_repos=<Foundation>
                           ,  username=<userid; optional in SAS Studio>
                           ,  passwd=<pwd; optional in SAS Studio>
                           ,  fa_src_path=<FA Src Path>
                           ,  fa_tgt_path=<FA Dest Path>
                           ,  fa_id=<machine_learning; optional>
                           ,  force_flg=<true; optional>
                           ,  debug=<true; optional>
                           ,  logOptions=<mprint symbolgen; optional>
                           ,  connection_type=<INTERNAL or EXTERNAL; optional>
                     );

            OUTPUT:
                     1. Success: New Federated Area added and available for use
                     2. Failure: Appropriate messages in the execution log
                           
   \ingroup api
   
   \author  SAS Institute Inc.
   \date    2018
*/

%macro irm_install_fa(
         meta_host      =
      ,  meta_port      = 8561
      ,  meta_repos     = Foundation
      ,  username       =
      ,  passwd         =
      ,  fa_src_path    = 
      ,  fa_tgt_path    = 
      ,  fa_id          = 
      ,  force_flg      = false
      ,  debug          = false
      ,  logOptions     =
      ,  connection_type= INTERNAL
)/minoperator;
                                   
   %put ENTERING &sysmacroname.; 
   
   /* Declare LOCAL macro variables for REST call */
   %local
      SWCNAME
      server
      oldLogOptions
      suppressLogOptions
      _fa_id
      __fa_id
      first_char
      pos
      fa_dir_path
   ;  
   
   /* Set required log options */
   %if(%length(&logOptions.)) %then
      options &logOptions.;
      
   /* Get current values of source, mlogic, mprint and symbolgen options */
   %let oldLogOptions = %sysfunc(getoption(source)) %sysfunc(getoption(mlogic)) %sysfunc(getoption(mprint)) %sysfunc(getoption(symbolgen));
   
   /* Set macro variable for suppressing log options */
   %let suppressLogOptions = nosource nomlogic nomprint nosymbolgen;

   /* Initialize */
   %let SWCNAME=IRM Mid-Tier Server;
   %let server = SASIRMServer;
   
   /* Check for valid inputs */
   /* Host must be provided */
   %if (%bquote(&meta_host.) eq %str()) %then %do;
      %rsk_terminate(key=rsk_meta_host_missing);
      %return;
   %end;
   
   /* Port must be provided */
   %if (&meta_port. eq %str()) %then
      %let meta_port=8561;
   
   /* Repository is optional */
   %if (%bquote(&meta_repos.) eq %str()) %then
      %let meta_repos=Foundation;

   /* Source path of FA zip file must be provided */
   %if (%bquote(&fa_src_path.) eq %str()) %then %do;
      %rsk_terminate(key=rsk_fa_src_path_missing);
      %return;
   %end;

   /* Target path where FA will be installed must be provided */   
   %if (%bquote(&fa_tgt_path.) eq %str()) %then %do;
      %rsk_terminate(key=rsk_fa_tgt_path_missing);
      %return;
   %end;

   /* Target location cannot be the same as the source location of the FA zip file */
   %let pos = %sysfunc(findc(&fa_src_path., '/\', b));
   %let fa_dir_path=%qksubstr(&fa_src_path., 1, %eval(&pos.-1));

   %if ((%bquote(&fa_tgt_path.) eq %bquote(&fa_src_path.)) or (%bquote(&fa_tgt_path.) eq %bquote(&fa_dir_path.)))%then %do;
      %rsk_terminate(key=rsk_fa_same_tgt_src_path, s1=%nrbquote(&fa_tgt_path.), s2=%nrbquote(&fa_src_path.), s3=%nrbquote(&fa_dir_path.));
      %return;
   %end;
   
   /* Check for valid FA ID, if provided. Take no actions if fa_id is empty or force_flg is false. */
   %let _fa_id = %nrbquote(&fa_id.);
   
   %if ((&force_flg eq %lowcase(true)) and ("&_fa_id." ne %str(""))) %then %do;
      /* Throw error if FA ID contains characters other than 0-9, A-z, a-z or '.' */
      %let __fa_id = %sysfunc(compress(&_fa_id., '1Aa.', daU));

      %if (&__fa_id. ne %str()) %then %do;
         %rsk_terminate(key=rsk_invalid_fa_id, s1="&_fa_id.");
         %return;
      %end;   
   
      /* Throw error if FA ID does not start with 1-9, A-Z or a-y */
      %let first_char = %sysfunc(compress(%qksubstr("&_fa_id.",2,1), '1Aa', kdaU));

      %if ((&first_char. eq %str()) or (&first_char. eq 0) or (&first_char. eq z))  %then %do;
         %rsk_terminate(key=rsk_invalid_first_char_fa_id, s1="&_fa_id.");
         %return;
      %end;
   %end;
   
   /* Check for username. If empty, could be a SAS Studio session. */
   %if (%bquote(&username.) eq %str()) %then %do;
      /* Assume a SAS Studio session and try to extract username. */
      %let username=&_CLIENTUSERNAME.;
      
      /* Still empty, then exit. */
      %if (%bquote(&username.) eq %str()) %then %do;
         %rsk_terminate(key=rsk_username_missing);
         %return;
      %end;
   %end;
   
   /* Query metadata to return mid-tier information. This will be used to construct REST URL:
      --host name
      --port number
      --protocol (http/https)
   */
   
   %global irm_host_name;
   %global irm_host_port;
   %global irm_host_proto;
   
   /* Suppress printing of passwords. */
   options &suppressLogOptions.;
   
   %irm_server_meta_query(
               i_meta_host          = &meta_host.
            ,  i_meta_port          = &meta_port.
            ,  i_meta_repos         = &meta_repos.
            ,  i_meta_user          = &username.
            ,  i_meta_passwd        = &passwd.
            ,  i_irm_server         = &server.
            ,  i_connection_type    = &connection_type.
            ,  o_host_name          = irm_host_name
            ,  o_host_port          = irm_host_port
            ,  o_host_proto         = irm_host_proto
   );
   
   /* Restore log options. */
   options &oldLogOptions.;
   
   /* Get Service ticket to POST new FA information through REST call */
   /* Constant String */
   %local install_fa_url;
   %let install_fa_url = %str(/rest/federatedareas/install/);

   /* Define server URL */
   %local server_url;
   %let server_url = %str(&irm_host_proto.://&irm_host_name.:&irm_host_port./&server.);
   
   /* Define REST URL */
   %local rest_url;
   %let rest_url = %str(&server_url.&install_fa_url.);

   /* Check whether session is already authenticated. */
   %local auth_session_flg;
   %let auth_session_flg = N;
   %if %sysevalf("%superq(passwd)"="", boolean) %then
        %let auth_session_flg = Y;

   /* Get a Service ticket if credentials have been provided */
   %if(&auth_session_flg. eq N) %then %do;
      /* Temporarily disable logging options to avoid printing of userid/pwd to the log */
      options &suppressLogOptions.;
      
      /* Get the Service Ticket. Returns service ticket in the macro variable __irm_rest_service_ticket__ */
      %irm_rest_service_ticket(
                url          = &rest_url.
              , appcontext   = &server.
              , irm_username = &username.
              , irm_password = &passwd.
              , debug        = &debug.
              , logOptions   = &oldLogOptions.
       );
   %end;
   
   /* Restore original log options. */
   options &oldLogOptions.;
   
   /* Make the REST call. */
   filename json_in temp;
   filename json_out temp;
   filename hdrout temp;

   /* Build the JSON body for the POST request. Need to pass as payload:
      sourcepath, targetpath, forceFlag, faName */
   filename json_in temp;
   %put NOTE: Creating JSON payload for a new FA;
   proc json out = json_in pretty;
      write open o ; 
         write values "sourcePath" "&fa_src_path.";
         write values "targetPath" "&fa_tgt_path.";
         write values "forceFlag" "&force_flg.";
         write values "faName" "&_fa_id.";
      write close;
   run;  

   /* Send the request */
   proc http
      in        = json_in
      out       = json_out
      headerout = hdrout
      method    = 'post'
      ct        = 'application/json'
      %if(&auth_session_flg. eq Y) %then %do;
         /* No credentials -> use Token authentication (this is only allowed if running the code from an authenticated SAS Session) */
         url = "%str(&rest_url.)"
         http_tokenauth
      %end;
      %else %do;
         /* Use the ticket */
         url = "&rest_url?ticket=&__irm_rest_service_ticket__." 
      %end;
      /* Clear connection cache */
      clear_cache
      ;
      headers
         'X-Requested-With' = 'XMLHttpRequest'
      ;
   run;

   %local
      json_msg
      hdr_msg
      status_code
   ;
   
   %global _status_code;
   %let i_status_code=_status_code;

   /* Collect server response */
   /* JSON */
   data _null_;
      length tmp $32767;
      retain tmp '';
      infile json_out flowover end=last;
      input;
      tmp=cats(tmp, _infile_);
      if last then 
         call symputx('json_msg', htmldecode(tmp));
   run;
   
   /* Header - fetch HTTP status code */
   %irm_parse_http_header(
            header_ref                  =hdrout,
            http_status_code_var_name   =&i_status_code);
            
   /********************************************************************************************
    HTTP STATUS CODE CATEGORIES           
    ============================================================================================
    |#| CATEGORY      | DESCRIPTION                       | CODES                              |
    ============================================================================================
    |1| INFORMATIONAL | Issued on a provisional basis     | 100-103                            | 
    --------------------------------------------------------------------------------------------
    |2| SUCCESS       | Normally a success condition      | 200-208,226                        | 
    --------------------------------------------------------------------------------------------
    |3| REDIRECTION   | Mostly used in URL redirection    | 300-308                            |
    --------------------------------------------------------------------------------------------
    |4| CLIENT ERRORS | Mostly a client error             | 400-418,421-424,426,428,429,431,451| 
    --------------------------------------------------------------------------------------------
    |5| SERVER ERRORS | Server failed to fulfil a request | 500-508,510,511                    |
    --------------------------------------------------------------------------------------------
    ********************************************************************************************/
    
   /*  Print appropriate message to the log. Status codes starting with 1 and 2 are treated as success.  */
   %if (%substr(&&&i_status_code,1,1) in (1 2)) %then
      %let msg=%rsk_get_msg(key=rsk_install_fa_success_msg, s1=%nrbquote(&fa_src_path.), s2=%nrbquote(&fa_tgt_path.));
   %else
      %let msg=%rsk_get_msg(key=rsk_install_fa_err_msg, s1=%nrbquote(&fa_src_path.), s2=%nrbquote(&fa_tgt_path.));      
   
   %put &msg.;
   %put %qksubstr(%nrbquote(&json_msg.),1);
   
   /* Clear filerefs */
   filename json_out clear;
   filename json_in clear;
   filename hdrout clear;

%mend;
